/********************************************************************************************
 * Little File Compressor for *nix systems.
 *
 * Copyright (C) 2002 Claudio Scordino , Linda Martorini, Francesco Lelli
 *
 * ******************************************************************************************
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ********************************************************************************************/

/*albero.c*/

#include <stdlib.h>
#include <stdio.h>


#include "bitIO.h"
#include "albero.h"

struct nodo
{
	int frequenza;
	int padre;
	int sin,des;
	int lunghezza;
	char codifica[N/8+ 1];
};


/**********************************************************************************************/

void codifica_print (struct nodo app)
/*Funzione utilizzata durante il debug*/

{
	int i;
	for (i=0; i<app.lunghezza && app.lunghezza!=-1;i++)
	{
		char byte = app.codifica[i/8];
		byte=(byte>>(i%8))&1;
		if (byte==0)
		 	printf ("0");
		else
		 	printf ("1");
	}
}


/**********************************************************************************************/

void stampa_tabella(struct nodo* tab)
/*Funzione utilizzata durante il debug*/

{
	int i;
	for (i=0;i<2*N+1;i++)
	{
		printf ("Nodo %d-esimo(%c): fr:%d , padre:%d , sx:%d , dx:%d , codifica:",i,(char)i,
		tab[i].frequenza,tab[i].padre,tab[i].sin,tab[i].des);
		codifica_print(tab[i]);
		printf ("\n");
	}
	fflush(NULL);
}


/**********************************************************************************************/

void calcola_frequenza(struct nodo* tab, FILE* fdes)
{
	unsigned char byte;
	int ritorno;
	int i;
	for (i=0;i<2*N+1;i++)				/*Inizializzazione della tabella*/
	{
		tab[i].padre=0;
		tab[i].frequenza=0;
		tab[i].lunghezza=-1;
	}

	while(1)
	{
		ritorno=fread(&byte,sizeof (char),1,fdes);
		if (ritorno<=0)
			break;				/*abbiamo scorso tutto il file*/
		tab[(unsigned)byte].frequenza++;	/*usiamo direttamente la codifica ASCII come entry nella
							tabella perche' supponiamo siano possibili tutti i valori ASCII*/
	}
	tab[N].frequenza=1;				/*carattere di fine file*/
}


/**********************************************************************************************/

void codifica (struct nodo* tab, int indice_nodo)
/* da chiamare sulla radice*/

{
	int padre=tab[indice_nodo].padre;
	if (padre==0)
		tab[indice_nodo].lunghezza=0;
	else
	{
		int i;
		char byte;
		int lunghezza=tab[padre].lunghezza;
		for (i=0;i<lunghezza/8+1;i++)
				tab[indice_nodo].codifica[i]=tab[padre].codifica[i];
		byte = tab[indice_nodo].codifica[lunghezza/8];
		if (tab[padre].sin==indice_nodo)
			/*Sono figlio sinistro, devo aggiungere 0 alla codifica*/
			byte &=~(1<<lunghezza%8);
		else
			/*Sono figlio destro, devo aggiungere 1 alla codifica*/
			byte |= (1<<lunghezza%8);
		tab[indice_nodo].codifica[lunghezza/8]=byte;
		tab[indice_nodo].lunghezza=lunghezza+1;
	}

	if (indice_nodo > N)
	{
		/*Non sono una foglia, quindi mi richiamo sui figli*/
		codifica(tab, tab[indice_nodo].sin);
		codifica(tab, tab[indice_nodo].des);
	}
}


/**********************************************************************************************/

void ricerca (struct nodo *tab,int libera,int* figliod, int* figlios)
/*ricerca dei due nodi rimanenti a frequenza minore*/
/*Commento: questa funzione  O(N); se avessimo usato uno heap sarebbe stata O(logN)*/
/*Questa funzione  invocata da inizializzazione, che viene chiamata all'inizio della 
compressione e della decompressione per costruire l'albero a partire dalle frequenze*/
{
	int i;
	*figliod=-1;
	*figlios=-1;
	for (i=0;i<libera;i++)
	{
		if (tab[i].padre==0)
		{
			if ((*figliod)==-1)
				(*figliod)=i;
			else if ((*figlios)==-1)
				(*figlios)=i;
			else if ( ((tab[(*figliod)].frequenza) > (tab[i].frequenza)) &&
			((tab[(*figliod)].frequenza) > (tab[(*figlios)].frequenza)))
				(*figliod)=i;
			else if ((tab[(*figlios)].frequenza) > (tab[i].frequenza))
				(*figlios)=i;
		}
	}
}


/**********************************************************************************************/

void inizializzazione(struct nodo* tab)
/*le prime N entry della tabella sono inizializzate (in ordine alfabetico) con
la frequenza di apparizione dei simboli*/

{
	int libera=N+1;
	int figliod;
	int figlios;

	/*ricerca dei primi due nodi a frequenza minore*/
	int i;
	for (i=N+1;i<2*N+1;i++)
	{
		ricerca(tab,libera,&figliod,&figlios);
		tab[i].sin=figlios;
		tab[i].des=figliod;
		tab[i].frequenza=tab[figliod].frequenza + tab[figlios].frequenza;
		tab[figlios].padre=i;
		tab[figliod].padre=i;
		libera++;
	}
}


/**********************************************************************************************/

void visita_dec(struct nodo* tab, unsigned int nod, char byte, unsigned int* successivo)
/*Serve per la decodifica; in ingresso ha il valore del bit e se non e' una foglia modifica
successivo per dove poter ripartire
Il parametro byte puo' essere 0 o 1.
Nel caso in cui sia una foglia il paramentro successivo contiene l'indice della foglia */

{
	if ((unsigned)byte==0) 			/*devo andare a sinistra*/
		(*successivo)=(unsigned int) ((tab[nod]).sin);
	else
		(*successivo)=(unsigned int)((tab[nod]).des);
}


/**********************************************************************************************/

int leggi_frequenze_da_file(struct nodo* tab,struct bitIO* file)
/*leggo dal file compresso e mi ricostruisco l'albero*/

{
	int i;
	int posizione=lseek(descrittore(file),0,SEEK_SET); /*mi sposto a inizio file*/
	if (posizione!=0)
		goto errore;
	for (i=0;i<=N;i++) /*leggo le frequenze di apparizione dei simboli*/
		bit_read(file,&(tab[i].frequenza),8*sizeof (int));
	for (i=0;i<2*N+1;i++)  /*inizializzazione della tabella*/
	{
		tab[i].padre=0;
		tab[i].lunghezza=-1;
	}
	return 1;

errore:
	printf("errore nella funzione leggi_frequenze_da_file\n");
	fflush(NULL);
	return (int) NULL;
}

/**********************************************************************************************/

struct nodo* crea_tabella(int num_elem)
{
	struct nodo *b;
	b= (struct nodo* ) malloc (sizeof (struct nodo)*num_elem);
	return b;
}


/**********************************************************************************************/

int ritorno_freq(struct nodo *tab, int i)
{
	return tab[i].frequenza;
}

/**********************************************************************************************/

int ritorno_lung(struct nodo *tab, int i)
{
	return tab[i].lunghezza;
}

/**********************************************************************************************/

char* ritorno_cod(struct nodo *tab, int i)
{
	return tab[i].codifica;
}

/**********************************************************************************************/

void distruggi_tabella(struct nodo *tab)
{
	free(tab);
}





