! PURPOSE
!
! This module contains Fortran datatypes and flags.
!
! USES
!  H5FORTRAN_TYPES - H5FORTRAN_TYPES is generated at run time by H5match_types.c
!
! COPYRIGHT
! * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
!   Copyright by The HDF Group.                                               *
!   All rights reserved.                                                      *
!                                                                             *
!   This file is part of HDF5.  The full HDF5 copyright notice, including     *
!   terms governing use, modification, and redistribution, is contained in    *
!   the COPYING file, which can be found at the root of the source code       *
!   distribution tree, or in https://www.hdfgroup.org/licenses.               *
!   If you do not have access to either file, you may request a copy from     *
!   help@hdfgroup.org.                                                        *
! * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
!
! AUTHOR
!  Elena Pourmal
!
!*****

MODULE H5GLOBAL
  USE H5FORTRAN_TYPES

  IMPLICIT NONE

  ! Enumerate data type that is interoperable with C.
  ENUM, BIND(C)
    ENUMERATOR :: enum_dtype
  END ENUM

!> \addtogroup FH5
!> @{
  INTEGER, PARAMETER :: ENUM_T = KIND(enum_dtype)  !< Enumerate data type that is interoperable with C.

  ! Parameters used in the function 'h5kind_to_type' located in H5_ff.F90.
  ! The flag is used to tell the function whether the kind input variable
  ! is for a REAL or INTEGER data type.
  INTEGER, PARAMETER :: H5_REAL_KIND = 1    !< Used in the function h5kind_to_type()
  INTEGER, PARAMETER :: H5_INTEGER_KIND = 0 !< Used in the function h5kind_to_type()
!> @}

  ! Definitions for reference datatypes.
  ! If you change the value of these parameters, do not forget to change corresponding
  ! values in the H5f90.h file.
  INTEGER, PARAMETER :: REF_REG_BUF_LEN = 3
  INTEGER, PARAMETER :: H5O_TOKEN_BUF_LEN = 16 ! Matches C defined value in H5public.h

!> \addtogroup FH5R
!> @{
!> @brief hobj_ref_t_f derived type
  TYPE :: hobj_ref_t_f
     INTEGER(HADDR_T) ref
  END TYPE hobj_ref_t_f

!> @brief hdset_reg_ref_t_f derived type
  TYPE :: hdset_reg_ref_t_f
     INTEGER, DIMENSION(1:REF_REG_BUF_LEN) :: ref
  END TYPE hdset_reg_ref_t_f
!> @}

  ! Do not forget to change these values when new predefined
  ! datatypes are added
  INTEGER, PARAMETER :: PREDEF_TYPES_LEN = 19
  INTEGER, PARAMETER :: FLOATING_TYPES_LEN = 4
  INTEGER, PARAMETER :: INTEGER_TYPES_LEN = 27

  ! These arrays need to be global because they are used in
  ! both h5open_f and in h5close_f; initialize to fix linking issues
  ! on OSX and Intel compilers.
  INTEGER(HID_T), DIMENSION(1:PREDEF_TYPES_LEN)   :: predef_types = -1
  INTEGER(HID_T), DIMENSION(1:FLOATING_TYPES_LEN) :: floating_types = -1
  INTEGER(HID_T), DIMENSION(1:INTEGER_TYPES_LEN)  :: integer_types = -1

  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_REAL_C_FLOAT
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_REAL_C_DOUBLE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_REAL_C_LONG_DOUBLE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_INTEGER
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_REAL
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_DOUBLE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_CHARACTER
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_REF_OBJ
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_REF_DSETREG
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_IEEE_F32BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_IEEE_F32LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_IEEE_F64BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_IEEE_F64LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I8BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I8LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I16BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I16LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I32BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I32LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I64BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_I64LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U8BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U8LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U16BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U16LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U32BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U32LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U64BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_U64LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STRING
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B8BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B8LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B16BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B16LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B32BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B32LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B64BE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STD_B64LE
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_B8
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_B16
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_B32
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_B64
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_FORTRAN_S1
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_C_S1
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_INTEGER_KIND
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NATIVE_FLOAT_128
  !DEC$endif
!> \addtogroup FH5T
!> @{
  INTEGER(HID_T) :: H5T_NATIVE_REAL_C_FLOAT       !< H5T_NATIVE_REAL_C_FLOAT
  INTEGER(HID_T) :: H5T_NATIVE_REAL_C_DOUBLE      !< H5T_NATIVE_REAL_C_DOUBLE
  INTEGER(HID_T) :: H5T_NATIVE_REAL_C_LONG_DOUBLE !< H5T_NATIVE_REAL_C_LONG_DOUBLE
  INTEGER(HID_T) :: H5T_NATIVE_INTEGER            !< H5T_NATIVE_INTEGER
  INTEGER(HID_T) :: H5T_NATIVE_REAL      !< H5T_NATIVE_REAL
  INTEGER(HID_T) :: H5T_NATIVE_DOUBLE    !< H5T_NATIVE_DOUBLE
  INTEGER(HID_T) :: H5T_NATIVE_CHARACTER !< H5T_NATIVE_CHARACTER
  INTEGER(HID_T) :: H5T_STD_REF_OBJ      !< H5T_STD_REF_OBJ
  INTEGER(HID_T) :: H5T_STD_REF_DSETREG  !< H5T_STD_REF_DSETREG
  INTEGER(HID_T) :: H5T_IEEE_F32BE  !< H5T_IEEE_F32BE
  INTEGER(HID_T) :: H5T_IEEE_F32LE  !< H5T_IEEE_F32LE
  INTEGER(HID_T) :: H5T_IEEE_F64BE  !< H5T_IEEE_F64BE
  INTEGER(HID_T) :: H5T_IEEE_F64LE  !< H5T_IEEE_F64LE
  INTEGER(HID_T) :: H5T_STD_I8BE  !< H5T_STD_I8BE
  INTEGER(HID_T) :: H5T_STD_I8LE  !< H5T_STD_I8LE
  INTEGER(HID_T) :: H5T_STD_I16BE  !< H5T_STD_I16BE
  INTEGER(HID_T) :: H5T_STD_I16LE  !< H5T_STD_I16LE
  INTEGER(HID_T) :: H5T_STD_I32BE  !< H5T_STD_I32BE
  INTEGER(HID_T) :: H5T_STD_I32LE  !< H5T_STD_I32LE
  INTEGER(HID_T) :: H5T_STD_I64BE  !< H5T_STD_I64BE
  INTEGER(HID_T) :: H5T_STD_I64LE  !< H5T_STD_I64LE
  INTEGER(HID_T) :: H5T_STD_U8BE  !< H5T_STD_U8BE
  INTEGER(HID_T) :: H5T_STD_U8LE  !< H5T_STD_U8LE
  INTEGER(HID_T) :: H5T_STD_U16BE  !< H5T_STD_U16BE
  INTEGER(HID_T) :: H5T_STD_U16LE  !< H5T_STD_U16LE
  INTEGER(HID_T) :: H5T_STD_U32BE  !< H5T_STD_U32BE
  INTEGER(HID_T) :: H5T_STD_U32LE  !< H5T_STD_U32LE
  INTEGER(HID_T) :: H5T_STD_U64BE  !< H5T_STD_U64BE
  INTEGER(HID_T) :: H5T_STD_U64LE  !< H5T_STD_U64LE
  INTEGER(HID_T) :: H5T_STRING  !< H5T_STRING
  INTEGER(HID_T) :: H5T_STD_B8BE  !< H5T_STD_B8BE
  INTEGER(HID_T) :: H5T_STD_B8LE  !< H5T_STD_B8LE
  INTEGER(HID_T) :: H5T_STD_B16BE  !< H5T_STD_B16BE
  INTEGER(HID_T) :: H5T_STD_B16LE  !< H5T_STD_B16LE
  INTEGER(HID_T) :: H5T_STD_B32BE  !< H5T_STD_B32BE
  INTEGER(HID_T) :: H5T_STD_B32LE  !< H5T_STD_B32LE
  INTEGER(HID_T) :: H5T_STD_B64BE  !< H5T_STD_B64BE
  INTEGER(HID_T) :: H5T_STD_B64LE  !< H5T_STD_B64LE
  INTEGER(HID_T) :: H5T_NATIVE_B8  !< H5T_NATIVE_B8
  INTEGER(HID_T) :: H5T_NATIVE_B16  !< H5T_NATIVE_B16
  INTEGER(HID_T) :: H5T_NATIVE_B32  !< H5T_NATIVE_B32
  INTEGER(HID_T) :: H5T_NATIVE_B64  !< H5T_NATIVE_B64
  INTEGER(HID_T) :: H5T_FORTRAN_S1  !< H5T_FORTRAN_S1
  INTEGER(HID_T) :: H5T_C_S1     !< H5T_C_S1

#ifndef H5_DOXYGEN
  INTEGER, PARAMETER :: NUM_NATIVE_INTEGER_KIND = 5
  ! INTEGER*1, INTEGER*2, INTEGER*4, INTEGER*8, INTEGER*16
  INTEGER(HID_T), DIMENSION(1:NUM_NATIVE_INTEGER_KIND) :: H5T_NATIVE_INTEGER_KIND
#endif
  ! NEED IFDEF -MSB-
  INTEGER(HID_T) :: H5T_NATIVE_FLOAT_128 !< Quadruple-precision floating-point
!> @}
  !
  ! H5generic flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5_INDEX_UNKNOWN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_INDEX_NAME_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_INDEX_CRT_ORDER_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_INDEX_N_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_ITER_UNKNOWN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_ITER_INC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_ITER_DEC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_ITER_NATIVE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_ITER_N_F
  !DEC$ATTRIBUTES DLLEXPORT :: HADDR_UNDEF_F
  !DEC$endif
!> \addtogroup FH5
!> @{
  INTEGER :: H5_INDEX_UNKNOWN_F   !< H5_INDEX_UNKNOWN
  INTEGER :: H5_INDEX_NAME_F      !< H5_INDEX_NAME
  INTEGER :: H5_INDEX_CRT_ORDER_F !< H5_INDEX_CRT_ORDER
  INTEGER :: H5_INDEX_N_F         !< H5_INDEX_N
  INTEGER :: H5_ITER_UNKNOWN_F    !< H5_ITER_UNKNOWN
  INTEGER :: H5_ITER_INC_F        !< H5_ITER_INC
  INTEGER :: H5_ITER_DEC_F        !< H5_ITER_DEC
  INTEGER :: H5_ITER_NATIVE_F     !< H5_ITER_NATIVE
  INTEGER :: H5_ITER_N_F          !< H5_ITER_N

  INTEGER(HADDR_T) :: HADDR_UNDEF_F  !< HADDR_UNDEF
!> @}
  !
  ! H5F flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_ACC_RDWR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_ACC_RDONLY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_ACC_TRUNC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_ACC_EXCL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_ACC_DEBUG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_SCOPE_LOCAL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_SCOPE_GLOBAL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_CLOSE_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_CLOSE_WEAK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_CLOSE_SEMI_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_CLOSE_STRONG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_OBJ_FILE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_OBJ_DATASET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_OBJ_GROUP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_OBJ_DATATYPE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_OBJ_ALL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_LIBVER_EARLIEST_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_LIBVER_LATEST_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_UNLIMITED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_LIBVER_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_LIBVER_NBOUNDS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_LIBVER_V18_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_LIBVER_V110_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_LIBVER_V112_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5F_LIBVER_V114_F
  !DEC$endif
!> \addtogroup FH5F
!> @{
  INTEGER :: H5F_ACC_RDWR_F        !< H5F_ACC_RDWR
  INTEGER :: H5F_ACC_RDONLY_F      !< H5F_ACC_RDONLY
  INTEGER :: H5F_ACC_TRUNC_F       !< H5F_ACC_TRUNC
  INTEGER :: H5F_ACC_EXCL_F        !< H5F_ACC_EXCL
  INTEGER :: H5F_ACC_DEBUG_F       !< H5F_ACC_DEBUG
  INTEGER :: H5F_SCOPE_LOCAL_F     !< H5F_SCOPE_LOCAL
  INTEGER :: H5F_SCOPE_GLOBAL_F    !< H5F_SCOPE_GLOBAL
  INTEGER :: H5F_CLOSE_DEFAULT_F   !< H5F_CLOSE_DEFAULT
  INTEGER :: H5F_CLOSE_WEAK_F      !< H5F_CLOSE_WEAK
  INTEGER :: H5F_CLOSE_SEMI_F      !< H5F_CLOSE_SEMI
  INTEGER :: H5F_CLOSE_STRONG_F    !< H5F_CLOSE_STRONG
  INTEGER :: H5F_OBJ_FILE_F        !< H5F_OBJ_FILE
  INTEGER :: H5F_OBJ_DATASET_F     !< H5F_OBJ_DATASET
  INTEGER :: H5F_OBJ_GROUP_F       !< H5F_OBJ_GROUP
  INTEGER :: H5F_OBJ_DATATYPE_F    !< H5F_OBJ_DATATYPE
  INTEGER :: H5F_OBJ_ALL_F         !< H5F_OBJ_ALL
  INTEGER :: H5F_LIBVER_EARLIEST_F !< H5F_LIBVER_EARLIEST
  INTEGER :: H5F_LIBVER_LATEST_F   !< H5F_LIBVER_LATEST
  INTEGER :: H5F_UNLIMITED_F       !< H5F_UNLIMITED
  INTEGER :: H5F_LIBVER_ERROR_F    !< H5F_LIBVER_ERROR
  INTEGER :: H5F_LIBVER_NBOUNDS_F  !< H5F_LIBVER_NBOUNDS
  INTEGER :: H5F_LIBVER_V18_F      !< H5F_LIBVER_V18
  INTEGER :: H5F_LIBVER_V110_F     !< H5F_LIBVER_V110
  INTEGER :: H5F_LIBVER_V112_F     !< H5F_LIBVER_V112
  INTEGER :: H5F_LIBVER_V114_F     !< H5F_LIBVER_V114
!> @}
  !
  ! H5G flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_UNKNOWN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_GROUP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_DATASET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_TYPE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_LINK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_UDLINK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_SAME_LOC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_LINK_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_LINK_HARD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_LINK_SOFT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_STORAGE_TYPE_UNKNOWN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_STORAGE_TYPE_SYMBOL_TABLE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_STORAGE_TYPE_COMPACT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5G_STORAGE_TYPE_DENSE_F
  !DEC$endif
!> \addtogroup FH5G
!> @{
  INTEGER :: H5G_UNKNOWN_F      !< H5G_UNKNOWN
  INTEGER :: H5G_GROUP_F        !< H5G_GROUP
  INTEGER :: H5G_DATASET_F      !< H5G_DATASET
  INTEGER :: H5G_TYPE_F         !< H5G_TYPE
  INTEGER :: H5G_LINK_F         !< H5G_LINK
  INTEGER :: H5G_UDLINK_F       !< H5G_UDLINK
  INTEGER :: H5G_SAME_LOC_F     !< H5G_SAME_LOC
  INTEGER :: H5G_LINK_ERROR_F   !< H5G_LINK_ERROR
  INTEGER :: H5G_LINK_HARD_F    !< H5G_LINK_HARD
  INTEGER :: H5G_LINK_SOFT_F    !< H5G_LINK_SOFT
  INTEGER :: H5G_STORAGE_TYPE_UNKNOWN_F      !< H5G_STORAGE_TYPE_UNKNOWN
  INTEGER :: H5G_STORAGE_TYPE_SYMBOL_TABLE_F !< H5G_STORAGE_TYPE_SYMBOL_TABLE
  INTEGER :: H5G_STORAGE_TYPE_COMPACT_F      !< H5G_STORAGE_TYPE_COMPACT
  INTEGER :: H5G_STORAGE_TYPE_DENSE_F        !< H5G_STORAGE_TYPE_DENSE
!> @}
  !
  ! H5D flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_COMPACT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_CONTIGUOUS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_CHUNKED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_ALLOC_TIME_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_ALLOC_TIME_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_ALLOC_TIME_EARLY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_ALLOC_TIME_LATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_ALLOC_TIME_INCR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_SPACE_STS_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_SPACE_STS_NOT_ALLOCATED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_SPACE_STS_PART_ALLOCATED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_SPACE_STS_ALLOCATED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_TIME_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_TIME_ALLOC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_TIME_NEVER_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_VALUE_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_VALUE_UNDEFINED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_VALUE_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_FILL_VALUE_USER_DEFINED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_CHUNK_CACHE_NSLOTS_DFLT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_CHUNK_CACHE_NBYTES_DFLT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_CHUNK_CACHE_W0_DFLT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_MPIO_NO_COLLECTIVE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_MPIO_CHUNK_INDEPENDENT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_MPIO_CHUNK_COLLECTIVE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_MPIO_CHUNK_MIXED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_MPIO_CONTIG_COLLECTIVE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_VDS_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_VDS_FIRST_MISSING_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_VDS_LAST_AVAILABLE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5D_VIRTUAL_F
  !DEC$endif
!> \addtogroup FH5D
!> @{
  INTEGER :: H5D_COMPACT_F  !< H5D_COMPACT
  INTEGER :: H5D_CONTIGUOUS_F  !< H5D_CONTIGUOUS
  INTEGER :: H5D_CHUNKED_F  !< H5D_CHUNKED

  INTEGER :: H5D_ALLOC_TIME_ERROR_F  !< H5D_ALLOC_TIME_ERROR
  INTEGER :: H5D_ALLOC_TIME_DEFAULT_F  !< H5D_ALLOC_TIME_DEFAULT
  INTEGER :: H5D_ALLOC_TIME_EARLY_F  !< H5D_ALLOC_TIME_EARLY
  INTEGER :: H5D_ALLOC_TIME_LATE_F  !< H5D_ALLOC_TIME_LATE
  INTEGER :: H5D_ALLOC_TIME_INCR_F  !< H5D_ALLOC_TIME_INCR

  INTEGER :: H5D_SPACE_STS_ERROR_F  !< H5D_SPACE_STS_ERROR
  INTEGER :: H5D_SPACE_STS_NOT_ALLOCATED_F  !< H5D_SPACE_STS_NOT_ALLOCATED
  INTEGER :: H5D_SPACE_STS_PART_ALLOCATED_F  !< H5D_SPACE_STS_PART_ALLOCATED
  INTEGER :: H5D_SPACE_STS_ALLOCATED_F  !< H5D_SPACE_STS_ALLOCATED

  INTEGER :: H5D_FILL_TIME_ERROR_F  !< H5D_FILL_TIME_ERROR
  INTEGER :: H5D_FILL_TIME_ALLOC_F  !< H5D_FILL_TIME_ALLOC
  INTEGER :: H5D_FILL_TIME_NEVER_F  !< H5D_FILL_TIME_NEVER

  INTEGER :: H5D_FILL_VALUE_ERROR_F  !< H5D_FILL_VALUE_ERROR
  INTEGER :: H5D_FILL_VALUE_UNDEFINED_F  !< H5D_FILL_VALUE_UNDEFINED
  INTEGER :: H5D_FILL_VALUE_DEFAULT_F  !< H5D_FILL_VALUE_DEFAULT
  INTEGER :: H5D_FILL_VALUE_USER_DEFINED_F  !< H5D_FILL_VALUE_USER_DEFINED

  ! shortened "_DEFAULT" to "_DFLT" to satisfy the limit of 31 (F95)
  ! characters for variable names in Fortran.
  ! shortened "_CONTIGUOUS" to "_CONTIG" to satisfy the limit of 31 (F95)
  ! characters for variable names in Fortran.

  INTEGER(SIZE_T) :: H5D_CHUNK_CACHE_NSLOTS_DFLT_F  !< H5D_CHUNK_CACHE_NSLOTS_DEFAULT
  INTEGER(SIZE_T) :: H5D_CHUNK_CACHE_NBYTES_DFLT_F  !< H5D_CHUNK_CACHE_NBYTES_DEFAULT

  INTEGER :: H5D_CHUNK_CACHE_W0_DFLT_F    !< H5D_CHUNK_CACHE_W0_DEFAULT
  INTEGER :: H5D_MPIO_NO_COLLECTIVE_F     !< H5D_MPIO_NO_COLLECTIVE
  INTEGER :: H5D_MPIO_CHUNK_INDEPENDENT_F !< H5D_MPIO_CHUNK_INDEPENDENT
  INTEGER :: H5D_MPIO_CHUNK_COLLECTIVE_F  !< H5D_MPIO_CHUNK_COLLECTIVE
  INTEGER :: H5D_MPIO_CHUNK_MIXED_F       !< H5D_MPIO_CHUNK_MIXED
  INTEGER :: H5D_MPIO_CONTIG_COLLECTIVE_F !< H5D_MPIO_CONTIGUOUS_COLLECTIVE
  INTEGER :: H5D_VDS_ERROR_F              !< H5D_VDS_ERROR
  INTEGER :: H5D_VDS_FIRST_MISSING_F      !< H5D_VDS_FIRST_MISSING
  INTEGER :: H5D_VDS_LAST_AVAILABLE_F     !< H5D_VDS_LAST_AVAILABLE
  INTEGER :: H5D_VIRTUAL_F                !< H5D_VIRTUAL
!> @}
  !
  ! H5E flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5E_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5E_MAJOR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5E_MINOR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5E_WALK_UPWARD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5E_WALK_DOWNWARD_F
  !DEC$endif
!> \addtogroup FH5E
!> @{
  INTEGER(HID_T) :: H5E_DEFAULT_F !< H5E_DEFAULT
  INTEGER :: H5E_MAJOR_F          !< H5E_MAJOR
  INTEGER :: H5E_MINOR_F          !< H5E_MINOR
  INTEGER :: H5E_WALK_UPWARD_F    !< H5E_WALK_UPWARD
  INTEGER :: H5E_WALK_DOWNWARD_F  !< H5E_WALK_DOWNWARD
!> @}
  !
  ! H5FD file drivers flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MPIO_INDEPENDENT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MPIO_COLLECTIVE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_NOLIST_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_SUPER_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_BTREE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_DRAW_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_GHEAP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_LHEAP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_OHDR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MEM_NTYPES_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_CORE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_FAMILY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_LOG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MPIO_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_MULTI_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_SEC2_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5FD_STDIO_F
  !DEC$endif
!> \addtogroup FH5P
!> @{
  INTEGER :: H5FD_MPIO_INDEPENDENT_F !< H5FD_MPIO_INDEPENDENT
  INTEGER :: H5FD_MPIO_COLLECTIVE_F  !< H5FD_MPIO_COLLECTIVE
  INTEGER :: H5FD_MEM_NOLIST_F       !< H5FD_MEM_NOLIST
  INTEGER :: H5FD_MEM_DEFAULT_F      !< H5FD_MEM_DEFAULT
  INTEGER :: H5FD_MEM_SUPER_F        !< H5FD_MEM_SUPER
  INTEGER :: H5FD_MEM_BTREE_F        !< H5FD_MEM_BTREE
  INTEGER :: H5FD_MEM_DRAW_F         !< H5FD_MEM_DRAW
  INTEGER :: H5FD_MEM_GHEAP_F        !< H5FD_MEM_GHEAP
  INTEGER :: H5FD_MEM_LHEAP_F        !< H5FD_MEM_LHEAP
  INTEGER :: H5FD_MEM_OHDR_F         !< H5FD_MEM_OHDR
  INTEGER :: H5FD_MEM_NTYPES_F       !< H5FD_MEM_NTYPES
  INTEGER :: H5FD_SUBFILING_CURR_FAPL_VERSION_F    !< H5FD_SUBFILING_CURR_FAPL_VERSION
  INTEGER :: H5FD_SUBFILING_FAPL_MAGIC_F           !< H5FD_SUBFILING_FAPL_MAGIC
  INTEGER :: H5FD_SUBFILING_DEFAULT_STRIPE_COUNT_F !< H5FD_SUBFILING_DEFAULT_STRIPE_COUNT
  INTEGER :: H5FD_IOC_CURR_FAPL_VERSION_F          !< H5FD_IOC_CURR_FAPL_VERSION
  INTEGER :: H5FD_IOC_FAPL_MAGIC_F                 !< H5FD_IOC_FAPL_MAGIC
  INTEGER :: H5FD_IOC_DEFAULT_THREAD_POOL_SIZE_F   !< H5FD_IOC_DEFAULT_THREAD_POOL_SIZE
  INTEGER :: SELECT_IOC_ONE_PER_NODE_F    !< Default, SELECT_IOC_ONE_PER_NODE
  INTEGER :: SELECT_IOC_EVERY_NTH_RANK_F  !< Starting at rank 0, select-next += N, SELECT_IOC_EVERY_NTH_RANK
  INTEGER :: SELECT_IOC_WITH_CONFIG_F     !< NOT IMPLEMENTED: Read-from-file, SELECT_IOC_WITH_CONFIG
  INTEGER :: SELECT_IOC_TOTAL_F           !< Starting at rank 0, mpi_size / total, SELECT_IOC_TOTAL
  INTEGER :: IOC_SELECTION_OPTIONS_F      !< Sentinel value, IOC_SELECTION_OPTIONS

  INTEGER(HID_T) :: H5FD_CORE_F      !< H5FD_CORE
  INTEGER(HID_T) :: H5FD_FAMILY_F    !< H5FD_FAMILY
  INTEGER(HID_T) :: H5FD_LOG_F       !< H5FD_LOG
  INTEGER(HID_T) :: H5FD_MPIO_F      !< H5FD_MPIO
  INTEGER(HID_T) :: H5FD_MULTI_F     !< H5FD_MULTI
  INTEGER(HID_T) :: H5FD_SEC2_F      !< H5FD_SEC2
  INTEGER(HID_T) :: H5FD_STDIO_F     !< H5FD_STDIO
  INTEGER(HID_T) :: H5FD_SUBFILING_F !< H5FD_SUBFILING
  INTEGER(HID_T) :: H5FD_SUBFILING_DEFAULT_STRIPE_SIZE_F !< H5FD_SUBFILING_DEFAULT_STRIPE_SIZE

!> @}
  !
  ! H5I flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_FILE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_GROUP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_DATATYPE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_DATASPACE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_DATASET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_ATTR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_BADID_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_UNINIT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_VFL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_VOL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_GENPROP_CLS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_GENPROP_LST_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_ERROR_CLASS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_ERROR_MSG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_ERROR_STACK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_NTYPES_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5I_INVALID_HID_F
  !DEC$endif
!> \addtogroup FH5I
!> @{
  INTEGER :: H5I_FILE_F  !< H5I_FILE
  INTEGER :: H5I_GROUP_F  !< H5I_GROUP
  INTEGER :: H5I_DATATYPE_F  !< H5I_DATATYPE
  INTEGER :: H5I_DATASPACE_F  !< H5I_DATASPACE
  INTEGER :: H5I_DATASET_F  !< H5I_DATASET
  INTEGER :: H5I_ATTR_F  !< H5I_ATTR
  INTEGER :: H5I_BADID_F  !< H5I_BADID
  INTEGER :: H5I_UNINIT_F  !< H5I_UNINIT
  INTEGER :: H5I_VFL_F  !< H5I_VFL
  INTEGER :: H5I_VOL_F  !< H5I_VOL
  INTEGER :: H5I_GENPROP_CLS_F  !< H5I_GENPROP_CLS
  INTEGER :: H5I_GENPROP_LST_F  !< H5I_GENPROP_LST
  INTEGER :: H5I_ERROR_CLASS_F  !< H5I_ERROR_CLASS
  INTEGER :: H5I_ERROR_MSG_F  !< H5I_ERROR_MSG
  INTEGER :: H5I_ERROR_STACK_F  !< H5I_ERROR_STACK
  INTEGER :: H5I_NTYPES_F  !< H5I_NTYPES
  INTEGER :: H5I_INVALID_HID_F  !< H5I_INVALID_HID
!> @}

  !
  ! H5L flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_TYPE_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_TYPE_HARD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_TYPE_SOFT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_TYPE_EXTERNAL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_SAME_LOC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5L_LINK_CLASS_T_VERS_F
  !DEC$endif
!> \addtogroup FH5L
!> @{
  INTEGER :: H5L_TYPE_ERROR_F  !< H5L_TYPE_ERROR
  INTEGER :: H5L_TYPE_HARD_F  !< H5L_TYPE_HARD
  INTEGER :: H5L_TYPE_SOFT_F  !< H5L_TYPE_SOFT
  INTEGER :: H5L_TYPE_EXTERNAL_F  !< H5L_TYPE_EXTERNAL
  INTEGER :: H5L_SAME_LOC_F  !< H5L_SAME_LOC
  INTEGER :: H5L_LINK_CLASS_T_VERS_F  !< H5L_LINK_CLASS_T_VERS
!> @}
  !
  ! H5O flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_SHALLOW_HIERARCHY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_EXPAND_SOFT_LINK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_EXPAND_EXT_LINK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_EXPAND_REFERENCE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_WITHOUT_ATTR_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_PRESERVE_NULL_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_COPY_ALL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_NONE_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_SDSPACE_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_DTYPE_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_FILL_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_PLINE_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_ATTR_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_ALL_FLAG_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_CHUNK0_SIZE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_ATTR_CRT_ORDER_TRACK_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_ATTR_CRT_ORDER_INDEX_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_ATTR_STORE_PHASE_CHA_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_STORE_TIMES_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_HDR_ALL_FLAGS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_MAX_NINDEXES_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_SHMESG_MAX_LIST_SIZE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_TYPE_UNKNOWN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_TYPE_GROUP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_TYPE_DATASET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_TYPE_NAMED_DATATYPE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_TYPE_NTYPES_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_ALL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_BASIC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_TIME_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_NUM_ATTRS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_HDR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5O_INFO_META_SIZE_F
  !
  !DEC$endif

!> \addtogroup FH5O
!> @{
!> @brief h5o_token_t_f derived type
  TYPE, BIND(C) :: h5o_token_t_f
     INTEGER(C_INT8_T), DIMENSION(1:H5O_TOKEN_BUF_LEN) :: token
  END TYPE h5o_token_t_f

  ! *** THESE VARIABLES DO NOT MATCH THE C VARIABLE IN ORDER TO STAY UNDER THE 31 Character Limit (F95) ***
  INTEGER :: H5O_COPY_SHALLOW_HIERARCHY_F !< H5O_COPY_SHALLOW_HIERARCHY_FLAG
  INTEGER :: H5O_COPY_EXPAND_SOFT_LINK_F  !< H5O_COPY_EXPAND_SOFT_LINK_FLAG
  INTEGER :: H5O_COPY_EXPAND_EXT_LINK_F   !< H5O_COPY_EXPAND_EXT_LINK_FLAG
  INTEGER :: H5O_COPY_EXPAND_REFERENCE_F  !< H5O_COPY_EXPAND_REFERENCE_FLAG
  !                             ***
  INTEGER :: H5O_COPY_WITHOUT_ATTR_FLAG_F   !< H5O_COPY_WITHOUT_ATTR_FLAG
  INTEGER :: H5O_COPY_PRESERVE_NULL_FLAG_F  !< H5O_COPY_PRESERVE_NULL_FLAG
  INTEGER :: H5O_COPY_ALL_F !< H5O_COPY_ALL
  INTEGER :: H5O_SHMESG_NONE_FLAG_F  !< H5O_SHMESG_NONE_FLAG
  INTEGER :: H5O_SHMESG_SDSPACE_FLAG_F  !< H5O_SHMESG_SDSPACE_FLAG
  INTEGER :: H5O_SHMESG_DTYPE_FLAG_F  !< H5O_SHMESG_DTYPE_FLAG
  INTEGER :: H5O_SHMESG_FILL_FLAG_F  !< H5O_SHMESG_FILL_FLAG
  INTEGER :: H5O_SHMESG_PLINE_FLAG_F  !< H5O_SHMESG_PLINE_FLAG
  INTEGER :: H5O_SHMESG_ATTR_FLAG_F  !< H5O_SHMESG_ATTR_FLAG
  INTEGER :: H5O_SHMESG_ALL_FLAG_F  !< H5O_SHMESG_ALL_FLAG
  INTEGER :: H5O_HDR_CHUNK0_SIZE_F  !< H5O_HDR_CHUNK0_SIZE
  INTEGER :: H5O_HDR_ATTR_CRT_ORDER_TRACK_F ! 32 CHARACTER
  INTEGER :: H5O_HDR_ATTR_CRT_ORDER_INDEX_F ! VARIABLE
  INTEGER :: H5O_HDR_ATTR_STORE_PHASE_CHA_F ! LENGTH ***
  INTEGER :: H5O_HDR_STORE_TIMES_F  !< H5O_HDR_STORE_TIMES
  INTEGER :: H5O_HDR_ALL_FLAGS_F  !< H5O_HDR_ALL_FLAGS
  INTEGER :: H5O_SHMESG_MAX_NINDEXES_F  !< H5O_SHMESG_MAX_NINDEXES
  INTEGER :: H5O_SHMESG_MAX_LIST_SIZE_F  !< H5O_SHMESG_MAX_LIST_SIZE
  INTEGER :: H5O_TYPE_UNKNOWN_F  !< H5O_TYPE_UNKNOWN
  INTEGER :: H5O_TYPE_GROUP_F  !< H5O_TYPE_GROUP
  INTEGER :: H5O_TYPE_DATASET_F  !< H5O_TYPE_DATASET
  INTEGER :: H5O_TYPE_NAMED_DATATYPE_F  !< H5O_TYPE_NAMED_DATATYPE
  INTEGER :: H5O_TYPE_NTYPES_F  !< H5O_TYPE_NTYPES
  INTEGER :: H5O_INFO_ALL_F  !< H5O_INFO_ALL
  INTEGER :: H5O_INFO_BASIC_F  !< H5O_INFO_BASIC
  INTEGER :: H5O_INFO_TIME_F  !< H5O_INFO_TIME
  INTEGER :: H5O_INFO_NUM_ATTRS_F  !< H5O_INFO_NUM_ATTRS
  INTEGER :: H5O_INFO_HDR_F  !< H5O_INFO_HDR
  INTEGER :: H5O_INFO_META_SIZE_F  !< H5O_INFO_META_SIZE
!> @}
  !
  ! H5P flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_FILE_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_FILE_ACCESS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DATASET_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DATASET_XFER_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_FILE_MOUNT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DEFAULT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_ROOT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_OBJECT_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DATASET_ACCESS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_GROUP_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_GROUP_ACCESS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DATATYPE_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_DATATYPE_ACCESS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_STRING_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_ATTRIBUTE_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_OBJECT_COPY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_LINK_CREATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_LINK_ACCESS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_CRT_ORDER_INDEXED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5P_CRT_ORDER_TRACKED_F
  !DEC$endif
!> \addtogroup FH5P
!> @{
  INTEGER(HID_T) :: H5P_FILE_CREATE_F  !< H5P_FILE_CREATE
  INTEGER(HID_T) :: H5P_FILE_ACCESS_F  !< H5P_FILE_ACCESS
  INTEGER(HID_T) :: H5P_DATASET_CREATE_F  !< H5P_DATASET_CREATE
  INTEGER(HID_T) :: H5P_DATASET_XFER_F  !< H5P_DATASET_XFER
  INTEGER(HID_T) :: H5P_FILE_MOUNT_F  !< H5P_FILE_MOUNT
  INTEGER(HID_T) :: H5P_DEFAULT_F  !< H5P_DEFAULT
  INTEGER(HID_T) :: H5P_ROOT_F  !< H5P_ROOT
  INTEGER(HID_T) :: H5P_OBJECT_CREATE_F  !< H5P_OBJECT_CREATE
  INTEGER(HID_T) :: H5P_DATASET_ACCESS_F  !< H5P_DATASET_ACCESS
  INTEGER(HID_T) :: H5P_GROUP_CREATE_F  !< H5P_GROUP_CREATE
  INTEGER(HID_T) :: H5P_GROUP_ACCESS_F  !< H5P_GROUP_ACCESS
  INTEGER(HID_T) :: H5P_DATATYPE_CREATE_F  !< H5P_DATATYPE_CREATE
  INTEGER(HID_T) :: H5P_DATATYPE_ACCESS_F  !< H5P_DATATYPE_ACCESS
  INTEGER(HID_T) :: H5P_STRING_CREATE_F  !< H5P_STRING_CREATE
  INTEGER(HID_T) :: H5P_ATTRIBUTE_CREATE_F  !< H5P_ATTRIBUTE_CREATE
  INTEGER(HID_T) :: H5P_OBJECT_COPY_F  !< H5P_OBJECT_COPY
  INTEGER(HID_T) :: H5P_LINK_CREATE_F  !< H5P_LINK_CREATE
  INTEGER(HID_T) :: H5P_LINK_ACCESS_F  !< H5P_LINK_ACCESS

  INTEGER :: H5P_CRT_ORDER_INDEXED_F  !< H5P_CRT_ORDER_INDEXED
  INTEGER :: H5P_CRT_ORDER_TRACKED_F  !< H5P_CRT_ORDER_TRACKED
!> @}
  !
  ! H5R flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5R_OBJECT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5R_DATASET_REGION_F
  !DEC$endif
!> \addtogroup FH5R
!> @{
  INTEGER :: H5R_OBJECT_F  !< H5R_OBJECT
  INTEGER :: H5R_DATASET_REGION_F  !< H5R_DATASET_REGION
!> @}
  !
  ! H5S flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_UNLIMITED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_ALL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SCALAR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SIMPLE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_NULL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_NOOP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_SET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_OR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_AND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_XOR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_NOTB_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_NOTA_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_APPEND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_PREPEND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SELECT_INVALID_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SEL_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SEL_NONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SEL_POINTS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SEL_HYPERSLABS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5S_SEL_ALL_F
  !DEC$endif
!> \addtogroup FH5S
!> @{
  INTEGER(HSIZE_T) :: H5S_UNLIMITED_F  !< H5S_UNLIMITED

  INTEGER(HID_T) :: H5S_ALL_F  !< H5S_ALL

  INTEGER :: H5S_SCALAR_F  !< H5S_SCALAR
  INTEGER :: H5S_SIMPLE_F  !< H5S_SIMPLE
  INTEGER :: H5S_NULL_F  !< H5S_NULL

  INTEGER :: H5S_SELECT_NOOP_F  !< H5S_SELECT_NOOP
  INTEGER :: H5S_SELECT_SET_F  !< H5S_SELECT_SET
  INTEGER :: H5S_SELECT_OR_F  !< H5S_SELECT_OR
  INTEGER :: H5S_SELECT_AND_F  !< H5S_SELECT_AND
  INTEGER :: H5S_SELECT_XOR_F  !< H5S_SELECT_XOR
  INTEGER :: H5S_SELECT_NOTB_F  !< H5S_SELECT_NOTB
  INTEGER :: H5S_SELECT_NOTA_F  !< H5S_SELECT_NOTA
  INTEGER :: H5S_SELECT_APPEND_F  !< H5S_SELECT_APPEND
  INTEGER :: H5S_SELECT_PREPEND_F  !< H5S_SELECT_PREPEND
  INTEGER :: H5S_SELECT_INVALID_F  !< H5S_SELECT_INVALID

  INTEGER :: H5S_SEL_ERROR_F  !< H5S_SEL_ERROR
  INTEGER :: H5S_SEL_NONE_F  !< H5S_SEL_NONE
  INTEGER :: H5S_SEL_POINTS_F  !< H5S_SEL_POINTS
  INTEGER :: H5S_SEL_HYPERSLABS_F  !< H5S_SEL_HYPERSLABS
  INTEGER :: H5S_SEL_ALL_F  !< H5S_SEL_ALL
!> @}
  !
  ! H5T flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NO_CLASS_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_INTEGER_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_FLOAT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_TIME_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STRING_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_BITFIELD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_OPAQUE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_COMPOUND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_REFERENCE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ENUM_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_VLEN_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ARRAY_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ORDER_LE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ORDER_BE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ORDER_VAX_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ORDER_NONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_ORDER_MIXED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_PAD_ZERO_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_PAD_ONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_PAD_BACKGROUND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_PAD_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_SGN_NONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_SGN_2_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_SGN_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NORM_IMPLIED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NORM_MSBSET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_NORM_NONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_CSET_ASCII_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_CSET_UTF8_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STR_NULLTERM_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STR_NULLPAD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STR_SPACEPAD_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_STR_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_DIR_ASCEND_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5T_DIR_DESCEND_F
  !DEC$endif
!> \addtogroup FH5T
!> @{
  INTEGER :: H5T_NO_CLASS_F  !< H5T_NO_CLASS
  INTEGER :: H5T_INTEGER_F  !< H5T_INTEGER
  INTEGER :: H5T_FLOAT_F  !< H5T_FLOAT
  INTEGER :: H5T_TIME_F  !< H5T_TIME
  INTEGER :: H5T_STRING_F  !< H5T_STRING
  INTEGER :: H5T_BITFIELD_F  !< H5T_BITFIELD
  INTEGER :: H5T_OPAQUE_F  !< H5T_OPAQUE
  INTEGER :: H5T_COMPOUND_F  !< H5T_COMPOUND
  INTEGER :: H5T_REFERENCE_F  !< H5T_REFERENCE
  INTEGER :: H5T_ENUM_F  !< H5T_ENUM
  INTEGER :: H5T_VLEN_F  !< H5T_VLEN
  INTEGER :: H5T_ARRAY_F  !< H5T_ARRAY
  INTEGER :: H5T_ORDER_LE_F  !< H5T_ORDER_LE
  INTEGER :: H5T_ORDER_BE_F  !< H5T_ORDER_BE
  INTEGER :: H5T_ORDER_VAX_F  !< H5T_ORDER_VAX
  INTEGER :: H5T_ORDER_NONE_F  !< H5T_ORDER_NONE
  INTEGER :: H5T_ORDER_MIXED_F  !< H5T_ORDER_MIXED
  INTEGER :: H5T_PAD_ZERO_F  !< H5T_PAD_ZERO
  INTEGER :: H5T_PAD_ONE_F  !< H5T_PAD_ONE
  INTEGER :: H5T_PAD_BACKGROUND_F  !< H5T_PAD_BACKGROUND
  INTEGER :: H5T_PAD_ERROR_F  !< H5T_PAD_ERROR
  INTEGER :: H5T_SGN_NONE_F  !< H5T_SGN_NONE
  INTEGER :: H5T_SGN_2_F  !< H5T_SGN_2
  INTEGER :: H5T_SGN_ERROR_F  !< H5T_SGN_ERROR
  INTEGER :: H5T_NORM_IMPLIED_F  !< H5T_NORM_IMPLIED
  INTEGER :: H5T_NORM_MSBSET_F  !< H5T_NORM_MSBSET
  INTEGER :: H5T_NORM_NONE_F  !< H5T_NORM_NONE
  INTEGER :: H5T_CSET_ASCII_F  !< H5T_CSET_ASCII
  INTEGER :: H5T_CSET_UTF8_F  !< H5T_CSET_UTF8
  INTEGER :: H5T_STR_NULLTERM_F  !< H5T_STR_NULLTERM
  INTEGER :: H5T_STR_NULLPAD_F  !< H5T_STR_NULLPAD
  INTEGER :: H5T_STR_SPACEPAD_F  !< H5T_STR_SPACEPAD
  INTEGER :: H5T_STR_ERROR_F  !< H5T_STR_ERROR
  INTEGER :: H5T_DIR_ASCEND_F  !< H5T_DIR_ASCEND
  INTEGER :: H5T_DIR_DESCEND_F  !< H5T_DIR_DESCEND
!> @}
  !
  ! H5Z flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_ERROR_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_NONE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_ALL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_DEFLATE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_SHUFFLE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_FLETCHER32_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_SZIP_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_ERROR_EDC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_DISABLE_EDC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_ENABLE_EDC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_NO_EDC_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FLAG_OPTIONAL_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_ENCODE_ENABLED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_DECODE_ENABLED_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_NBIT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_FILTER_SCALEOFFSET_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_SO_FLOAT_DSCALE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_SO_FLOAT_ESCALE_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_SO_INT_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5Z_SO_INT_MINBITS_DEFAULT_F
  !DEC$endif
!> \addtogroup FH5Z
!> @{
  INTEGER :: H5Z_FILTER_ERROR_F  !< H5Z_FILTER_ERROR
  INTEGER :: H5Z_FILTER_NONE_F  !< H5Z_FILTER_NONE
  INTEGER :: H5Z_FILTER_ALL_F  !< H5Z_FILTER_ALL
  INTEGER :: H5Z_FILTER_DEFLATE_F  !< H5Z_FILTER_DEFLATE
  INTEGER :: H5Z_FILTER_SHUFFLE_F  !< H5Z_FILTER_SHUFFLE
  INTEGER :: H5Z_FILTER_FLETCHER32_F  !< H5Z_FILTER_FLETCHER32
  INTEGER :: H5Z_FILTER_SZIP_F  !< H5Z_FILTER_SZIP
  INTEGER :: H5Z_ERROR_EDC_F  !< H5Z_ERROR_EDC
  INTEGER :: H5Z_DISABLE_EDC_F  !< H5Z_DISABLE_EDC
  INTEGER :: H5Z_ENABLE_EDC_F  !< H5Z_ENABLE_EDC
  INTEGER :: H5Z_NO_EDC_F  !< H5Z_NO_EDC
  INTEGER :: H5Z_FLAG_OPTIONAL_F  !< H5Z_FLAG_OPTIONAL
  INTEGER :: H5Z_FILTER_ENCODE_ENABLED_F  !< H5Z_FILTER_ENCODE_ENABLED
  INTEGER :: H5Z_FILTER_DECODE_ENABLED_F  !< H5Z_FILTER_DECODE_ENABLED
  INTEGER :: H5Z_FILTER_NBIT_F  !< H5Z_FILTER_NBIT
  INTEGER :: H5Z_FILTER_SCALEOFFSET_F  !< H5Z_FILTER_SCALEOFFSET
  INTEGER :: H5Z_SO_FLOAT_DSCALE_F  !< H5Z_SO_FLOAT_DSCALE
  INTEGER :: H5Z_SO_FLOAT_ESCALE_F  !< H5Z_SO_FLOAT_ESCALE
  INTEGER :: H5Z_SO_INT_F  !< H5Z_SO_INT
  INTEGER :: H5Z_SO_INT_MINBITS_DEFAULT_F  !< H5Z_SO_INT_MINBITS_DEFAULT
!> @}
  !
  ! H5 Library flags declaration
  !
  !DEC$if defined(BUILD_HDF5_DLL)
  !DEC$ATTRIBUTES DLLEXPORT :: H5_SZIP_EC_OM_F
  !DEC$ATTRIBUTES DLLEXPORT :: H5_SZIP_NN_OM_F
  !DEC$endif
!> \addtogroup FH5
!> @{
  INTEGER :: H5_SZIP_EC_OM_F  !< H5_SZIP_EC_OM
  INTEGER :: H5_SZIP_NN_OM_F  !< H5_SZIP_NN_OM
!> @}

END MODULE H5GLOBAL

