/* --------------------------------------------------------------------------
 * core of haserl.cgi - a poor-man's php for embedded/lightweight environments
 * $Id: haserl.c,v 1.13 2004/11/10 17:59:35 nangel Exp $ 
 * Copyright (c) 2003,2004    Nathan Angelacos (nangel@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * -----
 * The x2c() and unescape_url() routines were taken from  
 *  http://www.jmarshall.com/easy/cgi/getcgi.c.txt 
 * 
 * The comments in that text file state:
 *
 ***  Written in 1996 by James Marshall, james@jmarshall.com, except 
 ***  that the x2c() and unescape_url() routines were lifted directly 
 ***  from NCSA's sample program util.c, packaged with their HTTPD. 
 ***     For the latest, see http://www.jmarshall.com/easy/cgi/ 
 * -----
 *
 ------------------------------------------------------------------------- */
/*Splint info follows:  */
/* splint +posixlib webconf.c */
/*@-mustfreeonly -branchstate@*/

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <time.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <stdlib.h>
#include <string.h>

#if HAVE_SIGNAL_H
#include <signal.h>
#endif 

#include "haserl.h"

const char *REVTAG="$Id: haserl.c,v 1.13 2004/11/10 17:59:35 nangel Exp $";

#ifndef TEMPDIR
#define TEMPDIR "/tmp"
#endif

#ifndef MAX_UPLOAD_KB
#define MAX_UPLOAD_KB 2048
#endif

/* Refuse to disable the subshell */
#ifndef SUBSHELL_CMD
#define SUBSHELL_CMD "/bin/sh"
#endif

buflist_t	/*@null@*/ *env_list = NULL; 
token_t 	/*@null@*/ *token_list = NULL;

char	global_variable_prefix[] = HASERL_VAR_PREFIX;
int	global_subshell_pipe[4];
int	global_subshell_pid;
int	global_subshell_died = 0;
long unsigned   global_upload_size = 0;

/*
 * Convert 2 char hex string into char it represents
 * (from http://www.jmarshall.com/easy/cgi)
 */
char x2c (char *what) {
	char digit;
	
	/*@+charint@*/
	digit=(what[0] >= 'A' ? ((what[0] & 0xdf) - 'A')+10 : (what[0] - '0'));
	digit *=16;
	digit+=(what[1] >= 'A' ? ((what[1] & 0xdf) - 'A')+10 : (what[1] - '0'));
	return(digit);
	/*@-charint@*/
	}

/*
 * unsescape %xx to the characters they represent 
 */

void unescape_url (char *url) {
	int  i,j;
	/*@-predboolothers@*/
	for (i=0, j=0; url[j]; ++i, ++j) {
		if ((url[i] = url[j]) == '%') {
			url[i] = x2c(&url[j+1]);
			j+=2;	
			}
		}
	url[i]='\0';
	/*@+predboolothers@*/
	}

/*
 * myputenv (char *newbuf, char *prefix )
 *   Searches for an existing "key=" value in the chain;
 *   if one exists, it removes it from the environment, deletes the 
 *   existing link, then adds the new varaiable in
 *   Returns -1 on error, 0 on success
 *
 *   prefix is appended to the environment string (e.g. US_blah=foo )
 */ 
int myputenv (char *newbuf, char *prefix ) {
	buflist_t *cur = env_list;
	buflist_t *prev = NULL;
	size_t 	keylen;
	char	*entry = NULL;
	
	entry=calloc (sizeof(char), strlen(newbuf)+strlen(prefix)+1);
	if (entry == NULL ) { return (-1); }
	if (strlen(prefix)) { memcpy (entry, prefix, strlen(prefix)); }
	memcpy ((char *) (entry + strlen(prefix)), newbuf, strlen(newbuf));

	keylen = (size_t) (index (entry, '=') - entry);
	
	if ( keylen <= 0 ) {
			free (entry);
			return (-1);
			}
	
	/* does the value already exist? */
	while ( cur != NULL ) {
		if (memcmp (cur->buf, entry, keylen+1) == 0) {
			entry[keylen] = '\0';
			/*@-unrecog@*/ /*splint doesn't know about unsetenv */
			unsetenv (entry);
			/*@+unrecog@*/ /*splint doesn't know about unsetenv */
			entry[keylen] = '=';
			free(cur->buf);
			if (prev != NULL) prev->next=cur->next;
			free(cur);
			cur=prev;
			} /* end if found a matching key */
		prev=cur;
		cur = (buflist_t *) cur->next; 
		} /* end if matching key */

	/* add the value to the end of the chain  */
	if (cur == NULL) { /* paranoia... cur should always be null */
		cur=malloc(sizeof(buflist_t));
		cur->buf=entry;
		putenv(cur->buf);
		if (prev != NULL) prev->next=cur; 
		} else {
		return(-1);
		}

return(0);
}


/* readenv 
 * reads the current environment an popluates our environment chain 
 */

void readenv () {
	extern char **environ;
	int count = 0;

	while (environ[count] != NULL) {
		myputenv(environ[count], "");
		count++;
		}
	}

 
 /* freeenvchain
  * Misnamed, but basically goes through the linked list of 
  * buflist and clears the entire chain, but does not clear 
  * the environment
  */

void freeenvchain () {
	buflist_t *cur = env_list;
	buflist_t *next;
	while ( cur != NULL ) {
			free(cur->buf);
			next=(buflist_t *) cur->next;
			free(cur);
			cur=next;
			}

}

/* CookieVars ()
 * if HTTP_COOKIE is passed as an environment variable,
 * attempt to parse its values into environment variables
 */
void CookieVars () {
	char *qs;
	char *token;

	if (getenv("HTTP_COOKIE") != NULL) {
		qs=strdup(getenv("HTTP_COOKIE"));
		} else {
		return;
		}
	
	/** split on; to extract name value pairs */
	token=strtok(qs, ";");
	while (token) {
		// skip leading spaces 
		while ( token[0] == ' ' ) { token++; }
		myputenv(token, global_variable_prefix);
		token=strtok(NULL, ";");
		}
	free (qs);
	}


/* FreeTokenChain () 
 * goes through and frees the token chain 
 */

void FreeTokenChain () {
	token_t *cur= token_list;
	token_t *next;
	while ( cur != NULL ) {
			next=(token_t *) cur->next;
			free (cur);
			cur=next;
			}
	}


/* SessionID 
 *  Makes a uniqe SESSIONID environment variable for this script 
 */

void sessionid () {
	char	session[29];
		
	sprintf (session, "SESSIONID=%x%x", getpid(), (int) time( NULL ));
 	myputenv(session, "" );
	} 

void wcversion () {
	char  version[200];
	sprintf (version, "HASERLVER=%s", PACKAGE_VERSION);
	myputenv(version, "" );
	}

/* 
 * Read cgi variables from query string, and put in environment
 */

int ReadCGIQueryString () {
	char *qs;
	char *token;
	int i;

	if (getenv("QUERY_STRING") != NULL) {
		qs=strdup(getenv("QUERY_STRING"));
		} else {
		return (0);
		}
	
	/* change plusses into spaces */
	for (i=0; qs[i]; i++ ) { if (qs[i] == '+' ) { qs[i] = ' ';} };

	/** split on & and ; to extract name value pairs */
	
	token=strtok(qs, "&;");
	while (token) {
		unescape_url(token);
		myputenv(token, global_variable_prefix);
		token=strtok(NULL, "&;");
		}
	free (qs);
	return(0);
	}


/* 
 * Read cgi variables from stdin (for POST queries)
 * (oh... and if its mime-encoded file upload, we save the
 * file to /tmp; and return the name of the tmp file
 * the cgi script is responsible for disposing of the tmp file
 */

int ReadCGIPOSTValues () {
	char *qs;
	int content_length;
	int i;
	char *token;


 	if (getenv("CONTENT_LENGTH") == NULL) {	
		/* no content length?  Trash this request! */
		PrintParseError("HTTP POST request did not specify a Content Length.", 0);
		return(-1);
		} else {
		content_length = atoi(getenv("CONTENT_LENGTH"));
		}
	
	/* protect ourselves from 20GB file uploads */
	if (content_length > MAX_UPLOAD_KB * 1024 ) {
		PrintParseError("Attempted to send content larger than allowed limits.", 0);
		/* But we need to finish reading the content */
	 	while (	fread( &i, sizeof(int), 1, stdin) == 1 );
		return(-1);
		}
 
 	if (!(qs=malloc(content_length+1))) {
		PrintParseError("Unable to Allocate memory for POST content.", 0 );
		return(-1);
		}

	/* set the buffer to null, so that a browser messing with less 
  	   data than content_length won't buffer underrun us */
	memset(qs, 0 ,content_length+1);

	if ( (!fread(qs,content_length,1,stdin) && (content_length > 0) 
		&& !feof(stdin))) {
		PrintParseError( "Unable to read from stdin", 0);
		printf ("Content Length is %d ", content_length);
		free(qs);
		return(-1);
		}
	if (getenv("CONTENT_TYPE")) {
		if (strncasecmp(getenv("CONTENT_TYPE"), "multipart/form-data", 19) == 0) {
			/* This is a mime request, we need to go to the mime handler */
			i=ReadMimeEncodedInput(qs);
			free(qs);
			return(i);
			}
		}

	/* change plusses into spaces */
	for (i=0; qs[i]; i++ ) { if (qs[i] == '+' ) { qs[i] = ' ';} };

	/** split on & and ; to extract name value pairs */
	
	token=strtok(qs, "&;");
	while (token) {
		unescape_url(token);
		myputenv(token, global_variable_prefix);
		token=strtok(NULL, "&;");
		}
	free(qs);
	return(0);	
	}
/*
 *  LineToStr - Scans char and replaces the first "\n" with a "\0";
 *  If it finds a "\r", it does that to; (fix DOS brokennes) returns
 *  the length of the string;
 */
int LineToStr (char *string, size_t max) {
	size_t offset=0;

	while ((offset < max) && (string[offset] != '\n') && (string[offset] != '\r')) {
		offset++;
		}
	if (string[offset] == '\r') {
		string[offset]='\0';
		offset++;
		}
	if (string[offset] == '\n') {
		string[offset]='\0';
		offset++;
		}
	return (offset);
	}


/*
 * ReadMimeEncodedInput - handles things that are mime encoded
 * takes a pointer to the input; returns 0 on success
 */

int ReadMimeEncodedInput(char *qs) {
	char 	*boundary;
	char	*ct;
	int	i;
	int	datastart;
 	size_t	cl;
	size_t	offset;
	char    *envname;
	char	*filename;
	char 	*ptr;
	int	line;
	char 	tmpname[] = TEMPDIR "/XXXXXX";
	int	fd;
	/* we should only get here if the content type was set. Segfaults happen
	   if Content_Type is null */

	if (getenv("CONTENT_LENGTH") == NULL) {
		/* No content length?! */
		PrintParseError("No Content Length in HTTP Header from client", 0);
		return(-1);
		}

	cl=atoi(getenv("CONTENT_LENGTH"));
	
	/* we do this 'cause we can't mess with the real env. variable - it would
	 * overwrite the environment - I tried.
	 */
	i=strlen(getenv("CONTENT_TYPE"))+1;
	ct=malloc(i);
	if (ct) {
		memcpy(ct, getenv("CONTENT_TYPE"), i);
		} else {
		return(-1);
		}
	
	i=(int) NULL;
 	if (ct != NULL) {
		while (i < strlen(ct) && (strncmp("boundary=", &ct[i], 9) != 0)) { i++; }
		}
	if (i == strlen(ct)) {
		/* no boundary informaiton found */
		PrintParseError("No Mime Boundary Information Found", 0);
		free(ct);
		return (-1);
		}
	boundary=&ct[i+7];
	/* add two leading -- to the boundary */
	boundary[0]='-';
	boundary[1]='-';
	
	/* begin the big loop.  Look for:
		--boundary
		Content-Disposition: form-data;  name="......." 
		....
		<blank line>
		content
		--boundary
		Content-Disposition: form-data; name="....." filename="....."
		...
		<blank line>
		--boundary--
		eof
	*/

	offset=0;
	while (offset < cl) {
		/* first look for boundary */
		while ((offset < cl) && (memcmp(&qs[offset], boundary, strlen(boundary)))) {
			offset++;
			}
		/* if we got here and we ran off the end, its an error 		*/
		if (offset >= cl) { 
			PrintParseError ("Malformed MIME Encoding", 0);
			free(ct);
			return(-1);
			}
		/* if the two characters following the boundary are --, 	*/ 
		/* then we are at the end, exit 				*/
		if (memcmp(&qs[offset+strlen(boundary)], "--", 2) == 0) {
			offset+=2;
			break;
			}
		/* find where the offset should be */
		line=LineToStr(&qs[offset], cl-offset);
		offset+=line;
				
		/* Now we're going to look for content-disposition		*/ 
		line=LineToStr(&qs[offset], cl-offset);
		if (strncasecmp(&qs[offset], "Content-Disposition", 19) != 0) {
			/* hmm... content disposition was not where we expected it */
			PrintParseError ("Content-Disposition Missing", 0);
			free(ct);
			return(-1);
			}
		/* Found it, so let's go find "name="				*/
	 	if (!(envname=strstr(&qs[offset], "name="))) {
			/* now name= is missing?!  				*/
			PrintParseError ("Content-Disposition missing name tag", 0);
			free(ct);
			return(-1);
			} else {
			envname+=6;
			}
		/* is there a filename tag? 					*/
		if ((filename=strstr(&qs[offset], "filename="))!= NULL) {
			filename+=10;
			} else {
			filename=NULL;
			}
		/* make envname and filename ASCIIZ				*/
		i=0;
		while ((envname[i] != '"') && (envname[i] != '\0')) { i++; }
		envname[i] = '\0';
		if (filename) {
			i=0;
			while ((filename[i] != '"') && (filename[i] != '\0')) { i++; }
			filename[i] = '\0';
			}
		offset+=line;
		/* Ok, by some miracle, we have the name; let's skip till we	*/
		/* come to a blank line						*/
		line=LineToStr(&qs[offset], cl-offset);
		while (strlen(&qs[offset]) > 1) {
			offset+=line;
			line=LineToStr(&qs[offset], cl-offset);
			}
		offset+=line;
		datastart=offset;
		/* And we go back to looking for a boundary */
		while ((offset < cl) && (memcmp(&qs[offset], boundary, strlen(boundary)))) {
			offset++;
			}
		/* strip [cr] lf */
		if ((qs[offset-1] == '\n') && (qs[offset-2] == '\r')) {
			offset-=2; 
			} else {
			offset-=1;
			}
		qs[offset]=0;
		/* ok, at this point, we know where the name is, and we know    */
		/* where the content is... we have to do one of two things      */
		/* based on whether its a file or not				*/
		if (filename==NULL) { /* its not a file, so its easy		*/
				/* just jam the content after the name		*/
			memcpy(&envname[strlen(envname)+1], &qs[datastart], offset-datastart+1);
			envname[strlen(envname)]='=';
			myputenv(envname, global_variable_prefix);
			}
			else {	/* handle the fileupload case		*/
				if (offset-datastart) {  /* only if they uploaded */
					if ( global_upload_size == 0 ) {
						PrintParseError ("File uploads not allowed here.", 0);
						return ( -1 );
						}
					/*  stuff in the filename */
					ptr= calloc ( sizeof (char), strlen(envname)+strlen(filename)+2+5 );
					sprintf (ptr, "%s_name=%s", envname, filename);
					myputenv(ptr, global_variable_prefix);
					free(ptr);
						
					fd=mkstemp(tmpname);
				
					if (fd == -1) {
						PrintParseError ("Unable to open temp file", 0);
						return(-1);
						}
					write(fd, &qs[datastart], offset-datastart);
					close(fd);
					ptr= calloc (sizeof(char), strlen(envname)+strlen(tmpname)+2);
					sprintf (ptr, "%s=%s", envname, tmpname);
					myputenv(ptr, global_variable_prefix);
					free(ptr);
					}
			}
		}
	free(ct);
	return(0);
	}


/*
 * Loadscript attempts to open the script and put it in the script buffer
 * Returns 0 on success, -1 on error 
 */
int loadscript ( char *filename, script_t *scriptbuf) {
	int scriptfp;
	struct stat filestat;
	char *here;

	scriptfp = open (filename,O_NONBLOCK+O_RDONLY);
	if (scriptfp == -1) { /* open failed */
		return -1;
		}

	fstat (scriptfp, &filestat); 
	scriptbuf->name = (char *)calloc(sizeof(char), strlen(filename)+1);
	if (scriptbuf->name) {
		memcpy (scriptbuf->name, filename, sizeof(filename));
		}
	
	scriptbuf->size=filestat.st_size;
	scriptbuf->mtime=filestat.st_mtime;
	scriptbuf->uid=filestat.st_uid;
	scriptbuf->gid=filestat.st_gid;

	scriptbuf->buffer= (char *) calloc(sizeof(char), filestat.st_size+1);
	read (scriptfp, scriptbuf->buffer, filestat.st_size);
	scriptbuf->curpos=0;


	
	/* skip over the first line, if it starts #! 
 	 * This means that offset will start at the beginning of 
	 * the second line in most cases.  
	 */

	here=scriptbuf->buffer;
	if ( memcmp (scriptbuf->buffer , "#!", 2) == 0) {
		while ((scriptbuf->curpos < scriptbuf->size) &&
		 ((char)(here[scriptbuf->curpos]) !=  '\n' )) {
			(scriptbuf->curpos)++; 
			}
		(scriptbuf->curpos)++;
		}

	close (scriptfp);
	return (0);
	}


/*
 * Freescript frees the memory used by a script structure 
 * Returns 0 on success, -1 on error 
 */
int freescript (script_t *scriptbuf) {
	free(scriptbuf->name);
	free(scriptbuf->buffer);
	return(0);
	}

/*
 * PrintParseError ( char * error, int line number )
 *
 *
 */

void PrintParseError ( char *error, int linenum  ) {
	printf ("HTTP/1.0 500 Server Error\n" \
		"Content-Type: text/html\n\n" \
		"<html><body><b><font color=#CC0000>" PACKAGE_NAME " CGI Error</font></b><br>\n" \
		"Error: %s", error);
	if (linenum > 0) {
		printf (" near line %i\n", linenum);
		}
	printf ("</body></html>\n");
	}

/*
 *
 * scanline ( strip_t *script, int offset)
 *  given a pointer to a script type, count the newline characters from
 *  the beginning to the offset, and return the result
 */

int scanline ( script_t *script, int endcount) {
	int count=0;
	int offset=0;
	char *here;
	
	here=script->buffer;
	while ((offset <= endcount) && 
		(offset < script->size - script->curpos)) {
		if (here[offset] == '\n') {
			count++;
			}
		offset++;
		}

	return (count+1);
	}


/*
 * PushTokenChain - Adds a token to the end of the token chain 
   returun TRUE on succes; FALSE on failure.
 */

int PushTokenChain ( enum tag_t tag, char *buf, size_t len, int depth ) {
	token_t *cur = token_list;	
	token_t /*@null@*/ *prev =  NULL;


	/* Exception:  We don't store null sized html */
	if (( tag == HTML ) && ( len == 0 )) { return ( TRUE ); }
	
        /* walk the chain */
	while ( cur != NULL ) {
		prev=cur;
		cur=( token_t *) cur->next;
		}

         cur=malloc(sizeof(token_t));
	 if ( cur ) {
		if (token_list == NULL ) { token_list=cur; }
		if ( prev) { prev->next=(void *)cur; }
		cur->tag=tag;
		cur->buf=buf;
		cur->len=len;
		cur->depth=depth;
		cur->next=NULL;
		return (TRUE);
                } else {
		return (FALSE);
		}
	
	}


/* 
 * FindTag - given a char buffer, len, and tag to find return the offset
 * for the tag, or -1 if not found 
 */

int FindTag ( char *buf, size_t len, char *tag ) {
	size_t offset = 0;

	while ((memcmp (buf+offset, tag, 2) != 0) &&
                (offset <= len )) {
                offset++;
                }

	if ( offset > len ) { 
		return -1;
		} else {
		return offset;
		} 
	}


/* 
 * BuildTokenChain - given a pointer to the script buffer
 *  build a linked list of token_t structures.  Return TRUE
 *  if the token chain was built without errors, FALSE if
 *  there were errors.     
 */

int BuildTokenChain ( script_t *script ) {
	int 	offset, tagstart, tagend, taglen;
	char	*cp;
	int	depth = 0;
	int 	seenelse = 0;

	/* set cp so that we can reference cp as an 
	   array instead of a pointer to a pointer. */
	cp=(char *)script->buffer+script->curpos;
	offset=script->curpos;

	while ( offset <= script->size ) {
		tagstart = FindTag ( cp, script->size - offset, "<?" );
		tagend 	 = FindTag ( cp, script->size - offset, "?>" );
		taglen   = tagend - tagstart - 2; 

		/* check ?> before <? 	*/
		if (( tagend < tagstart ) || 
			(( tagstart == -1 ) && ( tagend >= 0 ))) {
			PrintParseError ( "Found ?&gt; tag before &lt;? tag",
				scanline ( script, tagend+offset ));
			return ( FALSE );
			}
		
		/* check <? within <? */
		if (( tagstart > -1 ) && 
			FindTag ( cp + tagstart + 2, taglen , "<?" )
			 > -1 ) {
			PrintParseError ( "Found &lt;? within a &lt;? ",
				scanline ( script, tagstart+offset ) );
			return ( FALSE );
			}

		/* check <? without a matching ?> */
		if (( tagend == -1 ) && ( tagstart > -1 )) {
			PrintParseError ("Found &lt;? without matching ?&gt;",
				scanline ( script, tagstart+offset) );
			return ( FALSE );
			}

		/* we have bare html before the tag, save it first */
		if ( tagstart > -1 ) {
			PushTokenChain ( HTML, cp, tagstart, depth );
			}

		/* trailing html */
		if (( tagstart == -1 ) && ( tagend == -1 )) {
			/* if its a \n then we don't save it.  That way when the user has
			 * a SCRIPT that ends the cgi, we don't tack on an extra linefeed. */
			if (!(( script->size - offset < 3) && ( cp[0] == '\n' ))) {
					PushTokenChain ( HTML, cp, script->size - offset , 0 );
				}
			offset=script->size +1;
			} else {

			/* We cheat and set the <? and ?> tags to nulls; this
			 * helps us out later on when we print the buffers -
			 * they will be ASCIIZ strings 
			 */
			memset (cp+tagstart, 0, 2 );
			memset (cp+tagend, 0, 2);

			tagstart+=2;	/* skip <? */
			taglen = tagend-tagstart;

			/* check for "abort/abend" cmd */
			if ( memcmp (cp+tagstart, "ab", 2 ) == 0 ) {
				PushTokenChain ( ABORT, cp+tagstart+1,
					taglen - 1, depth );
				}
			
			/* check for "run" cmd */
			if ( strchr ( "\n\r\t ", (int) *(cp+tagstart)) ) {
				PushTokenChain ( RUN, cp+tagstart+1, 
					taglen - 1, depth );
				}

			/* check for "if" cmd */
			if (memcmp ( cp+tagstart, "if", 2 ) == 0) {
				PushTokenChain ( IF, cp+tagstart+2,
					taglen-2, depth );
				depth++;
				seenelse=FALSE;
				}

			/* check for "fi" cmd */
			if (memcmp ( cp+tagstart, "fi", 2 ) == 0) {
				if ( depth <= 0 ) {
					PrintParseError ("Found &lt;?fi before &lt;?if",
						scanline ( script, tagstart+offset) );
					return ( FALSE );
					} else { 
					seenelse=FALSE;
					depth--;
					PushTokenChain ( FI, cp+tagstart+2,
					taglen-2, depth );
					}
				}

			/* check for "el" cmd */
			if (memcmp ( cp+tagstart, "el", 2 ) == 0) {
				if ( depth <= 0 ) {
					PrintParseError ("Found &lt;?el before &lt;?if",
						scanline ( script, tagstart+offset) );
					return ( FALSE );
					}
				if ( seenelse == TRUE ) {
					PrintParseError ("Found &lt;?el following a &lt;?el",
						scanline ( script, tagstart+offset ) );
					return ( FALSE );
					} else { 
					PushTokenChain ( ELSE, cp+tagstart+2,
					taglen-2, depth-1 );
					seenelse = TRUE;
					}
				}
			}			
				
			if ( tagend > -1 ) {
				offset += tagend + 2;
				} else {
				offset = script->size+1;
				}
			cp=script->buffer + offset;
 
		}	/* end of while loop for entire script */

	/* we finished without errors */
	return ( TRUE );
	}
	

/* 
 * FindHeadBlock - resizes the script so that it only contains
 *  the header block
 */
int FindHeadBlock ( script_t *script ) {
	char *offset	= NULL;
	char *cp	= script->buffer;
	size_t size;


	if (!(offset=strstr(cp, "\n\n"))) {
		offset=strstr(cp, "\r\n\r\n");
		}

	if (!offset) {
		return (-1);	/* there was no head block */
		} else {
		size=offset-cp+2;
		script->buffer=realloc(script->buffer, size);
		script->size=size;	
		return(0);	
		}	
	}


/* ProcessTokenChain ( token_t *chain )
 * Processes the token chain, starting at the token passed
 * exits when the next token is of a lower depth than the
 * token it was given.
 */

int ProcessTokenChain ( token_t *token ) {
	int depth;
	int retval;
	
	depth=token->depth;
	while (( token != NULL ) && ( token->depth >= depth )) {
		/* html */
		if ( token->tag == HTML ) {
			write ( STDOUT, token->buf, token->len );
			}
			
		/* abort */
		if ( token-> tag == ABORT ) {
			/* set all the tokens below me to No-op */
			while ( token != NULL ) {
				token->tag = NOOP;
				token=(void *)token->next;
				}
			return (TRUE);
			}
		
		/* run */
		if ( token->tag == RUN ) {
			ExecSubshell(token-> buf);
			}
 
		/* Note - We use 0 to mean TRUE, and !0 to mean FALSE 	
		 * That's backwards from coding logic, but the way that
		 * most programs work.
		 */
		
		/* if  */
		if ( token->tag == IF ) {
				retval = ExecSubshell( token->buf );
				if (!retval) {  /* TRUE */
					ProcessTokenChain ( (token_t *)token->next );
					token=(void *)token->next;
					while (( token != NULL ) && ( token->depth > depth )) {
						token=(void *)token->next;
						if (( token->tag == ELSE ) && ( token->depth == depth)) {
							token=(void *)token->next;
							}
						}
					} /* this will move us to the fi */
				if ( retval ) { /* FALSE */
					token=(void *)token->next;
					while (( token != NULL ) && ( token->depth > depth )) {
						token=(void *)token->next;
						if (( token->tag == ELSE ) && (token->depth == depth )){
							ProcessTokenChain ( (token_t *)token->next );
							token=(void *)token->next;
							}
						}	
					} /* this moves us to the fi */
				} /* end if */
		token=(void *)token->next;
		}		

	return (TRUE);
	}

#ifdef DEBUG 

int PrintTokenChain ( token_t *token ) {
	while ( token != NULL ) {
		printf( "Token: type %1d, size %3d at depth %2d--%s--\n",
			token->tag, token->len, token->depth, token->buf);
		token=(void *)token->next;
		}		
		
	return 0;
	}
#endif

void sig_handler ( int signo )	{
	switch (signo) {
		case SIGCHLD:	exit(-1);
				break;
		default:	break;
		}
	}

/* 
 * Attempt to open a subshell, return TRUE if it worked 
 */

int OpenSubshell ( void ) {
	int retcode = 0;
	/* Pipe fd's are defined in haserl.h */

	retcode= pipe ( &global_subshell_pipe[PARENT_IN] );
	if ( retcode == 0 ) { retcode= pipe ( &global_subshell_pipe[PARENT_CTRLIN]); }
	if ( retcode == 0 ) {
		signal ( SIGCHLD, sig_handler ); /* die if the child dies early */
		global_subshell_pid=fork();
		if ( global_subshell_pid == -1 ) { return FALSE; }

		if ( global_subshell_pid == 0 ) {
			/* I'm the child */
			signal(SIGCHLD, SIG_IGN);
			dup2 ( global_subshell_pipe[PARENT_IN], STDIN );
			close ( global_subshell_pipe[PARENT_IN]);
			close ( global_subshell_pipe[PARENT_OUT]); /* we won't write on this pipe */
			if ( global_subshell_pipe[CHILD_CTRLOUT] != 5 ) {
				/* force the outbound control pipe to be fd 5 */
				dup2 ( global_subshell_pipe[CHILD_CTRLOUT], 5 ); 
				close ( global_subshell_pipe[CHILD_CTRLOUT] );
				}
			//system ( SUBSHELL_CMD );
			execlp ( SUBSHELL_CMD, SUBSHELL_CMD, NULL);
			/* if we get here, we had a failure, so close down all the pipes */
			PrintParseError("Unable to start subshell - critical error.", 0 );
			/* loop stdin in to stdout so we don't hang */
			// dup2 ( global_subshell_pipe[PARENT_IN], STDOUT );
			exit (-1);
			} else {
			/* I'm parent, move along please */
			}
		}

	/* control should get to this point only in the parent.
 	 * send a dummy command just to make sure the shell is there
	 */
	return ( 0 );
	}


int CloseSubshell ( void ) {
	int x;

	signal ( SIGCHLD, SIG_IGN );

	/* closing the pipe should close the subshell if it is a bash-like shell */
	for ( x=0; x < 4; x++ ) {
		close (global_subshell_pipe[x]);
		}

	/* but kill it just to make sure */
	kill ( global_subshell_pid, 9 );

	return TRUE;
	}


int ExecSubshell ( char *buf ) {
	char *mybuf;
	const char postfix[]="\necho $? >&5\n";
	int count = 0;
	int retval;

	mybuf=calloc (sizeof(char), strlen(buf)+strlen(postfix)+1);
	if (mybuf == NULL ) { return (-1); }
	memcpy (mybuf, buf, strlen(buf));
	memcpy (mybuf+strlen(buf), postfix, strlen(postfix));
	write (global_subshell_pipe[PARENT_OUT], mybuf, strlen(mybuf));

	/* wait for the result to come back */
	/* clear mybuf to start */
	memset ( mybuf, 0, sizeof(mybuf));
	while ( mybuf[count] != '\n' )	 {
		read ( global_subshell_pipe[PARENT_CTRLIN], &mybuf[count], 1 );
		if ( mybuf[count]  != '\n' ) { count++; }
		}

	mybuf[count]=0;
	retval=atoi(mybuf);
	free(mybuf);
	return (retval);
	}


/* get the command line arguments, and the filestats on the script,
 * if you can find them
 */

char *parse_args ( int argc, char **argv ) {
	struct stat filestat;
	char *filename;
	
	switch ( argc ) {
		case 2:	/* script is in argv[1] */
			filename=argv[1];
			break;
		case 3: /* cmdline parms in argv[1], script in argv[2] */
			if ( strchr(argv[1], 'u')) {
				global_upload_size= MAX_UPLOAD_KB;
				} 
			filename=argv[2];
			break;
		default:
			return (NULL);
		}

	if ( stat ( filename, &filestat ) ) {
		/* can't open the script?!! */
		return (NULL);
		}

	return (filename);
	}


int BecomeUser( uid_t uid, gid_t gid )  {
	/* This silently fails if it doesn't work */
	setgid (gid);
	setuid (uid);
	return (0);	
}


/*-------------------------------------------------------------------------
 *
 * Main 
 *
 *------------------------------------------------------------------------*/

int main ( int argc, char *argv[]) {

script_t script;
int	retval = 0;
char	*filename;

/*  attempt to open ourself and read our script */
filename=parse_args(argc, argv);

	if ( filename == NULL ) {
	/* we were run, instead of called as a shell script */
	puts ("This is " PACKAGE_NAME " version " PACKAGE_VERSION "\n"
	"This program runs as a cgi interpeter, not interactively.\n"
	"Bug reports to: " PACKAGE_BUGREPORT "\n");
	return (0);
	}

/* set REVTAG to NULL to prevent compiler warning */
REVTAG=NULL;

if (loadscript(filename, &script )) {
	return(-1);
	}


/* drop permissions */
BecomeUser( script.uid, script.gid );

/* Read the current environment into our chain */
readenv();
sessionid();
wcversion();
CookieVars();


if (getenv("REQUEST_METHOD")) {
	if (strcasecmp(getenv("REQUEST_METHOD"), "GET") == 0) {
		retval = ReadCGIQueryString();
		}

	if (strcasecmp(getenv("REQUEST_METHOD"), "POST") == 0) {
		retval = ReadCGIPOSTValues();
		}

	if (strcasecmp(getenv("REQUEST_METHOD"), "HEAD") == 0) {
	 	retval = FindHeadBlock(&script);
		}
	}

if ( retval == 0) { retval=OpenSubshell();  }

if (( retval == 0) && ( BuildTokenChain ( &script ) == TRUE )) {
	#ifdef DEBUG
	PrintTokenChain ( (void *)token_list );
	#else
	ProcessTokenChain ( (void *)token_list );
	#endif
	}

if ( retval == 0 ) { CloseSubshell(); }
freescript(&script);
freeenvchain();
FreeTokenChain();
return (0);
} 
