//////////////////////////////////////////////////////////////////////////////
//    Copyright 2004-2014, SenseGraphics AB
//
//    This file is part of MedX3D.
//
//    MedX3D is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    MedX3D is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with MedX3D; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//    A commercial license is also available. Please contact us at 
//    www.sensegraphics.com for more information.
//
//
/// \file MarchingCubes.h
/// \brief Header file for MarchingCubes, MedX3D scene graph node.
///
//
//////////////////////////////////////////////////////////////////////////////
#ifndef __MARCHINGCUBES_H__
#define __MARCHINGCUBES_H__

#include <H3D/MedX3D/MedX3D.h>
#include <H3D/X3DGeometryNode.h>
#include <H3D/X3DTexture3DNode.h>
#include <H3D/SFInt32.h>
#include <H3D/MFString.h>
#include <H3D/Scene.h>
#include <H3D/TextureProperties.h>

#define GRADIENTS_ON_THE_FLY

namespace H3D {

  /// \ingroup Geometries
  /// \class MarchingCubes
  /// \brief The MarchingCubes geometry is a node for generating an 
  /// triangle-mesh iso-surface from a volume data set and an iso-value.
  ///
  /// This node currently only works for LUMINANCE textures as
  /// input to the voxels field. MarchingCubes will not work for textures in
  /// which any of the texture dimensions contains less than 2 pixels.
  class MarchingCubes : public X3DGeometryNode {  
  public:

    /// The OctTreeNode class defines a tree structure for voxel space 
    /// partitioning where each node is has 8 children dividing the 
    /// space it contains.
    /// Each node contains information about which voxels it contains,
    /// what triangles has been generated for the node( if leaf), etc.
    /// The structure is used in order to only regenerate local triangles
    /// when doing dynamic updates to the volume data instead of having to 
    /// regenerate triangles for the entire dataset.
    struct MEDX3D_API OctTreeNode : public RefCountedClass {
      /// Constructor.
      OctTreeNode():
        cache(0),
        value_min(0), // todo: proper vbalues
        value_max( 1000 ),
        parent( NULL ),
        next( NULL ) {
        for( int i=0; i<8; ++i )
          children[i] = NULL;
      }
      
      /// Returns true if this node is a leaf node.
      inline bool isLeaf() { return children[0] == NULL; }

      /// The parent of the node.
      OctTreeNode *parent;
      
      /// If this is a leaf node next points to the next node in the leaf.
      OctTreeNode *next;
      
      /// The children nodes of the OctTreeNode.
      OctTreeNode *children[8];

      /// The minimum voxel value of any voxel within this node.
      H3DFloat value_min;

      /// The maximum voxel value of any voxel within this node.
      H3DFloat value_max;

      /// The minimum voxel index value in the x-direction of this node.
      int x_min;
      /// The maximum voxel index value in the x-direction of this node.
      int x_max;
      /// The minimum voxel index value in the y-direction of this node.
      int y_min;
      /// The maximum voxel index value in the y-direction of this node.
      int y_max;
      /// The minimum voxel index value in the z-direction of this node.
      int z_min;
      /// The maximum voxel index value in the z-direction of this node.
      int z_max;

      /// The OpenGL display list used for rendering the triangles generated by
      /// the marching cubes algorithm for this node. 0 if no such display list
      /// has been generated.
      GLuint cache;

      /// The vertices generated by the marching cubes algorithm for this node.
      /// Empty if not leaf node.
      vector<Vec3f> vvertex;

      /// The normals generated by the marching cubes algorithm for this node.
      /// Empty if not leaf node.
      vector<Vec3f> vnormal;

      /// Render this node using OpenGL. If a leaf node a display list will
      /// be generated to render the triangles for this node the next time.
      void render();

      /// Subdivide the current node depth number of times. All previous
      /// children will be removed.
      void subdivide( int depth );
    };
  

    /// The SFOctTree class is a field class that updates its internal 
    /// oct tree and marching cubes triangles according to changes in
    /// volume data and iso value.
    ///
    /// routes_in[0] is the isovalue   (SFFloat)
    /// routes_in[1] is the voxel data (SFTexture3DNode)
    class MEDX3D_API SFOctTree: public  RefCountSField< OctTreeNode > {
    public:
      /// Constructor.
      SFOctTree() : 
        data_matrix( 0 ),
        x_points( 0 ),
        y_points( 0 ),
        z_points( 0 ),
        function_count( 0 ),
        update_thread( new PeriodicThread ) {
    update_thread->setThreadName( "MarchingCubes update thread" );
  }
    
      /// Structure for transfering data to separate thread to calculate
      /// new triangles if thread mode is used.
      struct UpdateData {
        float iso_value;
        unsigned int x_min, y_min, z_min, x_max, y_max, z_max;
        AutoRef< MarchingCubes > mc;
        OctTreeNode *oct_tree_leaf;
      };
      
      /// Structure for transferring triangles back when new triangles
      /// have been calculated in separate thread.
      struct TransferData {
        auto_ptr< vector<Vec3f > > vertices;
        auto_ptr< vector<Vec3f > > normals;
        AutoRef< MarchingCubes > mc;
        OctTreeNode *oct_tree_leaf;
      };
      
      /// Update the oct tree value_min and value_max values according
      /// to the current values of the voxel values in the data_matrix
      /// member.
      void updateMinMaxValues( OctTreeNode *t);
      
      /// Recalculates all the triangles in the oct tree using the 
      /// marching cubes algorithm using the voxel values in the data_matrix
      /// member.
      /// 
      /// \param tree The oct tree which we should update with new triangle
      /// info.
      /// \param iso_value The iso-value of the iso-surface of which to
      /// extract triangles.
      /// \param update_data If the update_data paramater is given the 
      /// function will not update the triangle data directly. Instead it will
      /// collect a set of functions and parameters that have to be called in
      /// order for the update to occur. This can be used to e.g. call these
      /// functions in different threads in order to speed up computation.
      void updateMCTriangles( OctTreeNode *tree,
                              H3DFloat iso_value, 
                              vector< pair< PeriodicThread::CallbackFunc, 
                              UpdateData * > >  *update_data = NULL );
      
      
      /// Recalculates all the triangles within a subvolume in the oct tree
      /// using the marching cubes algorithm using the voxel values in the
      /// data_matrix member.
      /// 
      /// \param tree The oct tree which we should update with new triangle
      /// info.
      /// \param iso_value The iso-value of the iso-surface of which to extract
      /// triangles.
      /// \param volume_min_x The voxel with the lowest index in x direction of
      /// the subvolume.
      /// \param volume_max_x The voxel with the highest index in x direction
      /// of the subvolume.
      /// \param volume_min_y The voxel with the lowest index in y direction of
      /// the subvolume.
      /// \param volume_max_y The voxel with the highest index in y direction
      /// of the subvolume.
      /// \param volume_min_z The voxel with the lowest index in z direction of
      /// the subvolume.
      /// \param volume_max_z The voxel with the highest index in z direction
      /// of the subvolume.
      /// \param update_data If the update_data paramater is given the function
      /// will not update the triangle data directly. Instead it will collect a
      /// set of functions and parameters that have to be called in order for
      /// the update to occur. This can be used to e.g. call these functions in
      /// different threads in order to speed up computation.
      void updateMCTrianglesInVolume(  OctTreeNode *tree,
                                       H3DFloat iso_value,  
                                       int volume_min_x,
                                       int volume_min_y,
                                       int volume_min_z,
                                       int volume_max_x,
                                       int volume_max_y,
                                       int volume_max_z,
                                    vector< pair< PeriodicThread::CallbackFunc,
                                       UpdateData * > > *update_data = NULL );

      /// This function generates triangles using the marching cubes algorithm 
      /// for a specified subvolume of data.
      /// 
      /// \param iso_value The iso-value of the iso-surface of which to extract
      /// triangles.
      /// \param x_min The voxel with the lowest index in x direction of the
      /// subvolume.
      /// \param x_max The voxel with the highest index in x direction of the
      /// subvolume.
      /// \param y_min The voxel with the lowest index in y direction of the
      /// subvolume.
      /// \param y_max The voxel with the highest index in y direction of the
      /// subvolume.
      /// \param z_min The voxel with the lowest index in z direction of the
      /// subvolume.
      /// \param z_max The voxel with the highest index in z direction of the
      /// subvolume.
      /// \param vertices Generated vertices of the triangles.
      /// \param normals Generated normals of the triangles.
      void vMarchingCubes( H3DFloat iso_value, 
                           unsigned int x_min, unsigned int x_max, 
                           unsigned int y_min, unsigned int y_max, 
                           unsigned int z_min, unsigned int z_max, 
                           vector< Vec3f > &vertices,
                           vector< Vec3f > &normals );

      /// Create the local copy of the data that is maintained for both speed
      /// and for thread consistency when using different threads. It will
      /// update the data_matrix, voxel_size and ?_points members.
      void buildDataMatrix( Image *i, TextureProperties *tp );

      /// Update the local copy of the data that is maintained for both speed
      /// and for thread consistency when using different threads. It will
      /// update the data_matrix, voxel_size and ?_points members in the
      /// subvolume defined.
      void updateDataMatrix( Image *i,
                             int volume_min_x,
                             int volume_min_y,
                             int volume_min_z,
                             int volume_max_x,
                             int volume_max_y,
                             int volume_max_z,
                             TextureProperties *tp );
      
      /// Update the gradients in the subvolume specified by the function
      /// parameters. Updates will be done directly in the gradients parameter
      /// and it is assumed that the vector is of the correct size before a
      /// call to this function.
      virtual  void updateGradients(int x_min, int x_max, 
                                   int y_min, int y_max, 
                                   int z_min, int z_max,
                                   vector< Vec3f > &gradients );
      
      /// Returns the value of the voxel at the specified index.
      inline H3DFloat getVoxelValue( unsigned int x, 
                                     unsigned int y, 
                                     unsigned int z ) {
        //MarchingCubes *mc = static_cast< MarchingCubes * >( getOwner() );
        //return mc->voxels->getValue()->
        //         image->getValue()->getPixel( x, y, z ).r;
        return data_matrix[ ( z*y_points + y ) * x_points +x ];
      }

      /// Returns the gradient of the voxel at the specified index.
      inline Vec3f getGradient( unsigned int x, 
                                unsigned int y, 
                                unsigned int z ) {
#ifdef GRADIENTS_ON_THE_FLY
        int index = ( z*y_points + y ) * x_points +x;
        Vec3f gradient;

        if (x==0)  {
          gradient.x=(getVoxelValue(x+1,y,z)-getVoxelValue(x,y,z))
            /voxel_size.x;          
        } else if (x==x_points-1) {
          gradient.x=(getVoxelValue(x,y,z)-getVoxelValue(x-1,y,z))
            /voxel_size.x;
        } else {
          gradient.x=(getVoxelValue(x+1,y,z)-getVoxelValue(x-1,y,z))
            /(2*voxel_size.x);
        }
        
        if (y==0)  {
          gradient.y=(getVoxelValue(x,y+1,z)-getVoxelValue(x,y,z))
            /voxel_size.y;
        } else if (y==y_points-1) {
          gradient.y=(getVoxelValue(x,y,z)-getVoxelValue(x,y-1,z))
            /voxel_size.y;
        } else {
          gradient.y=(getVoxelValue(x,y+1,z)-getVoxelValue(x,y-1,z))
            /(2*voxel_size.y);
        }
        
        if (z==0) {
          gradient.z=(getVoxelValue(x,y,z+1)-getVoxelValue(x,y,z))
            /voxel_size.z;
        } else if (z==z_points-1) {
          gradient.z=(getVoxelValue(x,y,z)-getVoxelValue(x,y,z-1))
            /voxel_size.z;
        } else{
          gradient.z=(getVoxelValue(x,y,z+1)-getVoxelValue(x,y,z-1))
            /(2*voxel_size.z);
        }
        
        return gradient;
#else
        return gradients[ ( z*y_points + y ) * x_points +x ];
#endif
      }
      
      /// The field update function to update the SFOctTree field.
      virtual void update(); 

    protected:

      /// Finds the approximate point of intersection of the surface
      /// between two points with the values fValue1 and fValue2
      float fGetOffset(float fValue1, float fValue2, float fValueDesired);

      /// Stores the last iso_value.
      H3DFloat old_value;

      /// The volume data. Each value is a value between 0 and 1. Dimensions of
      /// the data can be found in x_points, y_points and z_points,
      float *data_matrix;

      /// The size in metres of each voxel of the data in data_matrix.
      Vec3f voxel_size;

      /// The x-dimension of the data in data_matrix.
      unsigned int x_points;
      /// The y-dimension of the data in data_matrix.
      unsigned int y_points;
      /// The z-dimension of the data in data_matrix.
      unsigned int z_points;

      /// The gradients in each point in data_matrix
      /// ( when calculated by grad() )
      vector< Vec3f > gradients;

      /// the number of functions currently executing updates in separate
      /// threads.
      unsigned int function_count;
      
      /// Callback function for calculating new triangles in separate threads.
      static PeriodicThread::CallbackCode calculateNewTriangles( void * data );
      
      /// Callback function for transfering triangle updates after done in
      /// separate thread.
      static Scene::CallbackCode transferUpdatedTriangles( void *data );
      
      // thread used for updating the marching cubes triangles.
      auto_ptr< PeriodicThread > update_thread;
      
    }; //end of  class SFOctTree

    typedef DependentSFNode<X3DTexture3DNode, 
                            FieldRef<H3DDisplayListObject,
                                     H3DDisplayListObject::DisplayList,
                                     &H3D::H3DDisplayListObject::displayList>, 
                            true > SFTexture3DNode;


    /// The SFBound class is specialized to update its Bound from the 
    /// voxel data routed to it.
    ///
    /// routes_in[0] is the voxels (SFTexture3DNode)
    class SFBound: public TypedField< X3DGeometryNode::SFBound,
                   SFTexture3DNode > {
      virtual void update();
    };
  
    /// Constructor.
    MarchingCubes( Inst< SFNode  > _metadata        = 0,
                   Inst< SFBound > _bound           = 0,
                   Inst< DisplayList > _displayList = 0,
                   Inst< SFFloat  >  _isovalue      = 0,
                   Inst< SFOctTree > _octTree       = 0,
                   Inst< SFTexture3DNode > _voxels  = 0  );

    /// Render the node in OpenGL.
    virtual void render();

    /// The iso-value of the iso-surface for which we should extract triangles.
    /// 
    /// <b>Access type:</b> inputOutput \n
    /// <b>Default value:</b> 10
    auto_ptr< SFFloat >  isovalue;

    /// The voxel data to extract iso surfaces in.
    /// 
    /// <b>Access type:</b> inputOutput
    auto_ptr< SFTexture3DNode > voxels;

    /// The internal octTree structure used for dynamic updates of the iso
    /// surface.
    auto_ptr< SFOctTree> octTree; 

    /// The H3DNodeDatabase for this node.
    static H3DNodeDatabase database;

  protected:
    /// a2iVertexOffset lists the positions (int), relative to vertex0,
    /// of each of the 8 vertices of a cube
    static const int a2iVertexOffset[8][3];
        
    /// a2fVertexOffset lists the positions (float), relative to vertex0,
    /// of each of the 8 vertices of a cube
    static const float a2fVertexOffset[8][3];
    
    /// a2iEdgeConnection lists the index of the endpoint vertices for each of
    /// the 12 edges of the cube
    static const int a2iEdgeConnection[12][2];
    
    /// a2fEdgeDirection lists the direction vector (vertex1-vertex0) for each
    /// edge in the cube
    static const float a2fEdgeDirection[12][3];

    /// For each of the possible vertex states listed in aiCubeEdgeFlags
    /// there is a specific triangulation of the edge intersection points.
    /// a2iTriangleConnectionTable lists all of them in the form of
    /// 0-5 edge triples with the list terminated by the invalid value -1.
    /// For example: a2iTriangleConnectionTable[3] list the 2 triangles 
    /// formed when corner[0] and corner[1] are inside of the surface, but
    /// the rest of the cube is not.
    static const int a2iTriangleConnectionTable[256][16];

    /// For any edge, if one vertex is inside of the surface and the other
    /// is outside of the surface then the edge intersects the surface.
    /// For each of the 8 vertices of the cube can be two possible states :
    /// either inside or outside of the surface.
    /// For any cube the are 2^8=256 possible sets of vertex states.
    /// This table lists the edges intersected by the surface for all 256
    /// possible vertex states. There are 12 edges. For each entry in the 
    /// table, if edge bit n is intersected, then bit n is set to 1
    static const int aiCubeEdgeFlags[256];

  }; 
}

#endif
