/*
 *  $Id: quazarnpic.c 23869 2021-06-23 11:52:16Z yeti-dn $
 *  Copyright (C) 2021 David Necas (Yeti).
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <stdlib.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwyutils.h>
#include <libprocess/stats.h>
#include <libgwymodule/gwymodule-file.h>
#include <app/gwymoduleutils-file.h>
#include <app/data-browser.h>

#include "err.h"

/* The eight zeros is a region we do not compare.  Not sure the FRAME content is stable. */
#define MAGIC "\x80\x04\x95\x00\x00\x00\x00\x00\x00\x00\x00\x8c\x15numpy.core.multiarray"
#define MAGIC_SIZE (sizeof(MAGIC)-1)

#define EXTENSION ".npic"

typedef enum {
/* {{{ */
    OPCODE_MARK = '(',
    OPCODE_EMPTY_TUPLE = ')',
    OPCODE_EMPTY_LIST = ']',
    OPCODE_EMPTY_DICT = '}',
    OPCODE_STOP = '.',
    OPCODE_POP = '0',
    OPCODE_POP_MARK = '1',
    OPCODE_DUP = '2',
    OPCODE_APPEND = 'a',
    OPCODE_BINBYTES = 'B',
    OPCODE_BUILD = 'b',
    OPCODE_SHORT_BINBYTES = 'C',
    OPCODE_GLOBAL = 'c',
    OPCODE_DICT = 'd',
    OPCODE_APPENDS = 'e',
    OPCODE_FLOAT = 'F',
    OPCODE_BINFLOAT = 'G',
    OPCODE_GET = 'g',
    OPCODE_BINGET = 'h',
    OPCODE_INT = 'I',
    OPCODE_INST = 'i',
    OPCODE_BININT = 'J',
    OPCODE_LONG_BINGET = 'j',
    OPCODE_BININT1 = 'K',
    OPCODE_BININT2 = 'M',
    OPCODE_LONG = 'L',
    OPCODE_LIST = 'l',
    OPCODE_NONE = 'N',
    OPCODE_OBJ = 'o',
    OPCODE_PERSID = 'P',
    OPCODE_PUT = 'p',
    OPCODE_BINPERSID = 'Q',
    OPCODE_BINPUT = 'q',
    OPCODE_REDUCE = 'R',
    OPCODE_LONG_BINPUT = 'r',
    OPCODE_STRING = 'S',
    OPCODE_SETITEM = 's',
    OPCODE_BINSTRING = 'T',
    OPCODE_TUPLE = 't',
    OPCODE_SHORT_BINSTRING = 'U',
    OPCODE_SETITEMS = 'u',
    OPCODE_BINUNICODE = 'X',
    OPCODE_UNICODE = 'V',
    OPCODE_PROTO = 0x80,
    OPCODE_NEWOBJ = 0x81,
    OPCODE_EXT1 = 0x82,
    OPCODE_EXT2 = 0x83,
    OPCODE_EXT4 = 0x84,
    OPCODE_TUPLE1 = 0x85,
    OPCODE_TUPLE2 = 0x86,
    OPCODE_TUPLE3 = 0x87,
    OPCODE_LONG1 = 0x8a,
    OPCODE_LONG4 = 0x8b,
    OPCODE_FRAME = 0x95,
    OPCODE_NEWTRUE = 0x88,
    OPCODE_NEWFALSE = 0x89,
    OPCODE_SHORT_BINUNICODE = 0x8c,
    OPCODE_BINUNICODE8 = 0x8d,
    OPCODE_BINBYTES8 = 0x8e,
    OPCODE_EMPTY_SET = 0x8f,
    OPCODE_ADDITEMS = 0x90,
    OPCODE_FROZENSET = 0x91,
    OPCODE_NEWOBJ_EX = 0x92,
    OPCODE_STACK_GLOBAL = 0x93,
    OPCODE_MEMOIZE = 0x94,
    /* }}} */
} PickleOpcodeType;

typedef enum {
    /* {{{ */
    ARG_NONE,
    ARG_BYTES1,
    ARG_BYTES4,
    ARG_BYTES8,
    ARG_DECIMALNL_LONG,
    ARG_DECIMALNL_SHORT,
    ARG_FLOAT8,
    ARG_FLOATNL,
    ARG_INT4,
    ARG_LONG1,
    ARG_LONG4,
    ARG_STRING1,
    ARG_STRING4,
    ARG_STRINGNL,
    ARG_STRINGNL_NOESCAPE,
    ARG_STRINGNL_NOESCAPE_PAIR,
    ARG_UINT1,
    ARG_UINT2,
    ARG_UINT4,
    ARG_UINT8,
    ARG_UNICODESTRING1,
    ARG_UNICODESTRING4,
    ARG_UNICODESTRING8,
    ARG_UNICODESTRINGNL,
    /* }}} */
} PickleArgType;

typedef enum {
    /* {{{ */
    STACK_VOID,
    STACK_ANYOBJECT,
    STACK_ANYOBJECT2,
    STACK_ANYOBJECT3,
    STACK_MARKOBJECT,
    STACK_MARKOBJECT_ANYOBJECT_STACKSLICE,
    STACK_MARKOBJECT_STACKSLICE,
    STACK_PYBOOL,
    STACK_PYBYTES,
    STACK_PYBYTES_OR_STR,
    STACK_PYDICT,
    STACK_PYDICT_ANYOBJECT2,
    STACK_PYDICT_MARKOBJECT_STACKSLICE,
    STACK_PYFLOAT,
    STACK_PYFROZENSET,
    STACK_PYINT,
    STACK_PYINTEGER_OR_BOOL,
    STACK_PYLIST,
    STACK_PYLIST_ANYOBJECT,
    STACK_PYLIST_MARKOBJECT_STACKSLICE,
    STACK_PYNONE,
    STACK_PYSET,
    STACK_PYSET_MARKOBJECT_STACKSLICE,
    STACK_PYTUPLE,
    STACK_PYUNICODE,
    STACK_PYUNICODE_PYUNICODE,
    /* }}} */
} PickleStackType;

typedef struct {
    guchar opcode;
    guchar protocol : 4;
    PickleArgType argtype : 6;
    PickleStackType stack_before : 6;
    PickleStackType stack_after : 6;
    const gchar *name;
} PickleOpcode;

typedef struct {
    union {
        guint64 u;
        gint64 i;
        gdouble d;
        gchar *s;
    } v;
    gsize len;
    PickleArgType type;
    gboolean free_s;
} PickleArg;

/* {{{ */
static const PickleOpcode opcodes[] = {
    { OPCODE_ADDITEMS,         4, ARG_NONE,                   STACK_PYSET_MARKOBJECT_STACKSLICE,     STACK_PYSET,             "ADDITEMS",         },
    { OPCODE_APPEND,           0, ARG_NONE,                   STACK_PYLIST_ANYOBJECT,                STACK_PYLIST,            "APPEND",           },
    { OPCODE_APPENDS,          1, ARG_NONE,                   STACK_PYLIST_MARKOBJECT_STACKSLICE,    STACK_PYLIST,            "APPENDS",          },
    { OPCODE_BINBYTES,         3, ARG_BYTES4,                 STACK_VOID,                            STACK_PYBYTES,           "BINBYTES",         },
    { OPCODE_BINBYTES8,        4, ARG_BYTES8,                 STACK_VOID,                            STACK_PYBYTES,           "BINBYTES8",        },
    { OPCODE_BINFLOAT,         1, ARG_FLOAT8,                 STACK_VOID,                            STACK_PYFLOAT,           "BINFLOAT",         },
    { OPCODE_BINGET,           1, ARG_UINT1,                  STACK_VOID,                            STACK_ANYOBJECT,         "BINGET",           },
    { OPCODE_BININT1,          1, ARG_UINT1,                  STACK_VOID,                            STACK_PYINT,             "BININT1",          },
    { OPCODE_BININT,           1, ARG_INT4,                   STACK_VOID,                            STACK_PYINT,             "BININT",           },
    { OPCODE_BININT2,          1, ARG_UINT2,                  STACK_VOID,                            STACK_PYINT,             "BININT2",          },
    { OPCODE_BINPERSID,        1, ARG_NONE,                   STACK_ANYOBJECT,                       STACK_ANYOBJECT,         "BINPERSID",        },
    { OPCODE_BINPUT,           1, ARG_UINT1,                  STACK_VOID,                            STACK_VOID,              "BINPUT",           },
    { OPCODE_BINSTRING,        1, ARG_STRING4,                STACK_VOID,                            STACK_PYBYTES_OR_STR,    "BINSTRING",        },
    { OPCODE_BINUNICODE,       1, ARG_UNICODESTRING4,         STACK_VOID,                            STACK_PYUNICODE,         "BINUNICODE",       },
    { OPCODE_BINUNICODE8,      4, ARG_UNICODESTRING8,         STACK_VOID,                            STACK_PYUNICODE,         "BINUNICODE8",      },
    { OPCODE_BUILD,            0, ARG_NONE,                   STACK_ANYOBJECT2,                      STACK_ANYOBJECT,         "BUILD",            },
    { OPCODE_DICT,             0, ARG_NONE,                   STACK_MARKOBJECT_STACKSLICE,           STACK_PYDICT,            "DICT",             },
    { OPCODE_DUP,              0, ARG_NONE,                   STACK_ANYOBJECT,                       STACK_ANYOBJECT2,        "DUP",              },
    { OPCODE_EMPTY_DICT,       1, ARG_NONE,                   STACK_VOID,                            STACK_PYDICT,            "EMPTY_DICT",       },
    { OPCODE_EMPTY_LIST,       1, ARG_NONE,                   STACK_VOID,                            STACK_PYLIST,            "EMPTY_LIST",       },
    { OPCODE_EMPTY_SET,        4, ARG_NONE,                   STACK_VOID,                            STACK_PYSET,             "EMPTY_SET",        },
    { OPCODE_EMPTY_TUPLE,      1, ARG_NONE,                   STACK_VOID,                            STACK_PYTUPLE,           "EMPTY_TUPLE",      },
    { OPCODE_EXT1,             2, ARG_UINT1,                  STACK_VOID,                            STACK_ANYOBJECT,         "EXT1",             },
    { OPCODE_EXT2,             2, ARG_UINT2,                  STACK_VOID,                            STACK_ANYOBJECT,         "EXT2",             },
    { OPCODE_EXT4,             2, ARG_INT4,                   STACK_VOID,                            STACK_ANYOBJECT,         "EXT4",             },
    { OPCODE_FLOAT,            0, ARG_FLOATNL,                STACK_VOID,                            STACK_PYFLOAT,           "FLOAT",            },
    { OPCODE_FRAME,            4, ARG_UINT8,                  STACK_VOID,                            STACK_VOID,              "FRAME",            },
    { OPCODE_FROZENSET,        4, ARG_NONE,                   STACK_MARKOBJECT_STACKSLICE,           STACK_PYFROZENSET,       "FROZENSET",        },
    { OPCODE_GET,              0, ARG_DECIMALNL_SHORT,        STACK_VOID,                            STACK_ANYOBJECT,         "GET",              },
    { OPCODE_GLOBAL,           0, ARG_STRINGNL_NOESCAPE_PAIR, STACK_VOID,                            STACK_ANYOBJECT,         "GLOBAL",           },
    { OPCODE_INST,             0, ARG_STRINGNL_NOESCAPE_PAIR, STACK_MARKOBJECT_STACKSLICE,           STACK_ANYOBJECT,         "INST",             },
    { OPCODE_INT,              0, ARG_DECIMALNL_SHORT,        STACK_VOID,                            STACK_PYINTEGER_OR_BOOL, "INT",              },
    { OPCODE_LIST,             0, ARG_NONE,                   STACK_MARKOBJECT_STACKSLICE,           STACK_PYLIST,            "LIST",             },
    { OPCODE_LONG,             0, ARG_DECIMALNL_LONG,         STACK_VOID,                            STACK_PYINT,             "LONG",             },
    { OPCODE_LONG1,            2, ARG_LONG1,                  STACK_VOID,                            STACK_PYINT,             "LONG1",            },
    { OPCODE_LONG4,            2, ARG_LONG4,                  STACK_VOID,                            STACK_PYINT,             "LONG4",            },
    { OPCODE_LONG_BINGET,      1, ARG_UINT4,                  STACK_VOID,                            STACK_ANYOBJECT,         "LONG_BINGET",      },
    { OPCODE_LONG_BINPUT,      1, ARG_UINT4,                  STACK_VOID,                            STACK_VOID,              "LONG_BINPUT",      },
    { OPCODE_MARK,             0, ARG_NONE,                   STACK_VOID,                            STACK_MARKOBJECT,        "MARK",             },
    { OPCODE_MEMOIZE,          4, ARG_NONE,                   STACK_ANYOBJECT,                       STACK_ANYOBJECT,         "MEMOIZE",          },
    { OPCODE_NEWFALSE,         2, ARG_NONE,                   STACK_VOID,                            STACK_PYBOOL,            "NEWFALSE",         },
    { OPCODE_NEWOBJ,           2, ARG_NONE,                   STACK_ANYOBJECT2,                      STACK_ANYOBJECT,         "NEWOBJ",           },
    { OPCODE_NEWOBJ_EX,        4, ARG_NONE,                   STACK_ANYOBJECT3,                      STACK_ANYOBJECT,         "NEWOBJ_EX",        },
    { OPCODE_NEWTRUE,          2, ARG_NONE,                   STACK_VOID,                            STACK_PYBOOL,            "NEWTRUE",          },
    { OPCODE_NONE,             0, ARG_NONE,                   STACK_VOID,                            STACK_PYNONE,            "NONE",             },
    { OPCODE_OBJ,              1, ARG_NONE,                   STACK_MARKOBJECT_ANYOBJECT_STACKSLICE, STACK_ANYOBJECT,         "OBJ",              },
    { OPCODE_PERSID,           0, ARG_STRINGNL_NOESCAPE,      STACK_VOID,                            STACK_ANYOBJECT,         "PERSID",           },
    { OPCODE_POP,              0, ARG_NONE,                   STACK_ANYOBJECT,                       STACK_VOID,              "POP",              },
    { OPCODE_POP_MARK,         1, ARG_NONE,                   STACK_MARKOBJECT_STACKSLICE,           STACK_VOID,              "POP_MARK",         },
    { OPCODE_PROTO,            2, ARG_UINT1,                  STACK_VOID,                            STACK_VOID,              "PROTO",            },
    { OPCODE_PUT,              0, ARG_DECIMALNL_SHORT,        STACK_VOID,                            STACK_VOID,              "PUT",              },
    { OPCODE_REDUCE,           0, ARG_NONE,                   STACK_ANYOBJECT2,                      STACK_ANYOBJECT,         "REDUCE",           },
    { OPCODE_SETITEM,          0, ARG_NONE,                   STACK_PYDICT_ANYOBJECT2,               STACK_PYDICT,            "SETITEM",          },
    { OPCODE_SETITEMS,         1, ARG_NONE,                   STACK_PYDICT_MARKOBJECT_STACKSLICE,    STACK_PYDICT,            "SETITEMS",         },
    { OPCODE_SHORT_BINBYTES,   3, ARG_BYTES1,                 STACK_VOID,                            STACK_PYBYTES,           "SHORT_BINBYTES",   },
    { OPCODE_SHORT_BINSTRING,  1, ARG_STRING1,                STACK_VOID,                            STACK_PYBYTES_OR_STR,    "SHORT_BINSTRING",  },
    { OPCODE_SHORT_BINUNICODE, 4, ARG_UNICODESTRING1,         STACK_VOID,                            STACK_PYUNICODE,         "SHORT_BINUNICODE", },
    { OPCODE_STACK_GLOBAL,     4, ARG_NONE,                   STACK_PYUNICODE_PYUNICODE,             STACK_ANYOBJECT,         "STACK_GLOBAL",     },
    { OPCODE_STOP,             0, ARG_NONE,                   STACK_ANYOBJECT,                       STACK_VOID,              "STOP",             },
    { OPCODE_STRING,           0, ARG_STRINGNL,               STACK_VOID,                            STACK_PYBYTES_OR_STR,    "STRING",           },
    { OPCODE_TUPLE,            0, ARG_NONE,                   STACK_MARKOBJECT_STACKSLICE,           STACK_PYTUPLE,           "TUPLE",            },
    { OPCODE_TUPLE1,           2, ARG_NONE,                   STACK_ANYOBJECT,                       STACK_PYTUPLE,           "TUPLE1",           },
    { OPCODE_TUPLE2,           2, ARG_NONE,                   STACK_ANYOBJECT2,                      STACK_PYTUPLE,           "TUPLE2",           },
    { OPCODE_TUPLE3,           2, ARG_NONE,                   STACK_ANYOBJECT3,                      STACK_PYTUPLE,           "TUPLE3",           },
    { OPCODE_UNICODE,          0, ARG_UNICODESTRINGNL,        STACK_VOID,                            STACK_PYUNICODE,         "UNICODE",          },
};
/* }}} */

static gboolean      module_register(void);
static gint          npic_detect    (const GwyFileDetectInfo *fileinfo,
                                     gboolean only_name);
static GwyContainer* npic_load      (const gchar *filename,
                                     GwyRunType mode,
                                     GError **error);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Imports Quazar data files stored as Python pickles v4."),
    "Yeti <yeti@gwyddion.net>",
    "0.1",
    "David Nečas (Yeti)",
    "2021",
};

GWY_MODULE_QUERY2(module_info, quazarnpic)

static gboolean
module_register(void)
{
    gwy_file_func_register("quazarnpic",
                           N_("Quazar Python-pickled data (.npic)"),
                           (GwyFileDetectFunc)&npic_detect,
                           (GwyFileLoadFunc)&npic_load,
                           NULL,
                           NULL);

    return TRUE;
}

static gint
npic_detect(const GwyFileDetectInfo *fileinfo,
            gboolean only_name)
{
    const gchar *p = fileinfo->head;

    return 0;

    if (only_name)
        return g_str_has_suffix(fileinfo->name_lowercase, EXTENSION) ? 10 : 0;

    if (fileinfo->buffer_len < MAGIC_SIZE)
        return 0;

    if (memcmp(p, MAGIC, 3) || memcmp(p + 3 + 8, MAGIC + 3 + 8, MAGIC_SIZE - 3 - 8))
        return 0;

    /* TODO: Try to find some familiar strings in the tail part which contains a Python dictionary with measurement
     * settings. */

    return 80;
}

static gboolean
read_arg(const guchar **p, gsize *size, PickleArg *arg)
{
    if (arg->type == ARG_NONE)
        return TRUE;
    else if (arg->type == ARG_UINT1) {
        if (*size < 1)
            return FALSE;
        arg->v.u = **p;
        gwy_debug("UINT1 value %lu", (gulong)arg->v.u);
        (*p)++;
        (*size)--;
        return TRUE;
    }
    else if (arg->type == ARG_UINT2) {
        if (*size < 2)
            return FALSE;
        arg->v.u = gwy_get_guint16_le(p);
        gwy_debug("UINT2 value %lu", (gulong)arg->v.u);
        *size -= 2;
        return TRUE;
    }
    else if (arg->type == ARG_UINT4) {
        if (*size < 4)
            return FALSE;
        arg->v.u = gwy_get_guint32_le(p);
        gwy_debug("UINT4 value %lu", (gulong)arg->v.u);
        *size -= 4;
        return TRUE;
    }
    else if (arg->type == ARG_INT4) {
        if (*size < 4)
            return FALSE;
        arg->v.i = gwy_get_gint32_le(p);
        gwy_debug("INT4 value %ld", (glong)arg->v.i);
        *size -= 4;
        return TRUE;
    }
    else if (arg->type == ARG_UINT8) {
        if (*size < 8)
            return FALSE;
        arg->v.u = gwy_get_gint64_le(p);
        gwy_debug("UINT8 value %lu", (gulong)arg->v.u);
        *size -= 8;
        return TRUE;
    }
    else if (arg->type == ARG_FLOAT8) {
        if (*size < 8)
            return FALSE;
        arg->v.d = gwy_get_gdouble_le(p);
        gwy_debug("FLOAT8 value %g", arg->v.d);
        *size -= 8;
        return TRUE;
    }
    else if (arg->type == ARG_UNICODESTRING1 || arg->type == ARG_UNICODESTRING4 || arg->type == ARG_UNICODESTRING8
             || arg->type == ARG_BYTES1 || arg->type == ARG_BYTES4 || arg->type == ARG_BYTES8) {
        if (arg->type == ARG_UNICODESTRING1 || arg->type == ARG_BYTES1) {
            if (*size < 1)
                return FALSE;
            arg->len = **p;
            (*p)++;
            (*size)--;
        }
        else if (arg->type == ARG_UNICODESTRING4 || arg->type == ARG_BYTES4) {
            if (*size < 4)
                return FALSE;
            arg->len = gwy_get_guint32_le(p);
            *size -= 4;
        }
        else {
            if (*size < 8)
                return FALSE;
            arg->len = gwy_get_guint64_le(p);
            *size -= 8;
        }
        if (*size < arg->len)
            return FALSE;
        arg->free_s = TRUE;
        arg->v.s = g_new(gchar, arg->len+1);
        memcpy(arg->v.s, *p, arg->len);
        arg->v.s[arg->len] = '\0';
        gwy_debug("BUFFER-LIKE value of length %lu", (gulong)arg->len);
        *p += arg->len;
        *size -= arg->len;
        return TRUE;
    }
    /* This should only be the newline formats which are silly and long which we have no way of representing anyway
     * (unless they are used for normal-sized integers). */
    g_warning("Argument type %u not implemented.", arg->type);
    return FALSE;
}

static gboolean
read_one_object(const guchar **p, gsize size, GError **error)
{
    guint protocol = 0;

    gwy_debug("starting to read a new object");
    while (TRUE) {
        const PickleOpcode *opcode;
        PickleArg arg;
        guint op, i;

        gwy_clear(&arg, 1);
        if (size < 1) {
            err_TRUNCATED_PART(error, "object");
            return FALSE;
        }
        op = **p;
        (*p)++;
        size--;
        for (i = 0; i < G_N_ELEMENTS(opcodes); i++) {
            opcode = opcodes + i;
            if (opcode->opcode == op)
                break;
        }
        if (i == G_N_ELEMENTS(opcodes)) {
            g_set_error(error, GWY_MODULE_FILE_ERROR, GWY_MODULE_FILE_ERROR_DATA,
                        _("Uknown opcode 0x%02x encountered"), op);
            return FALSE;
        }

        gwy_debug("opcode %s (proto=%u, args=%d, before=%d after=%d)",
                  opcode->name, opcode->protocol, opcode->argtype, opcode->stack_before, opcode->stack_after);
        arg.type = opcode->argtype;
        if (!read_arg(p, &size, &arg)) {
            err_TRUNCATED_PART(error, opcode->name);
            return FALSE;
        }
        if (op == OPCODE_STOP) {
            /* TODO: Check stack.  Pop the object. */
            return TRUE;
        }
        else if (op == OPCODE_PROTO) {
            protocol = MAX(arg.v.u, protocol);
            gwy_debug("protocol %u (->%u)", (guint)arg.v.u, protocol);
        }
        else if (op == OPCODE_FRAME) {
            gwy_debug("frame %lu", (gulong)arg.v.u);
        }

        if (arg.free_s)
            g_free(arg.v.s);
    }
    return FALSE;
}

static GwyContainer*
npic_load(const gchar *filename,
          G_GNUC_UNUSED GwyRunType mode,
          GError **error)
{
    GwyContainer *container = NULL;
    guchar *buffer = NULL;
    const guchar *p;
    gsize size;
    GError *err = NULL;

    if (!gwy_file_get_contents(filename, &buffer, &size, &err)) {
        err_GET_FILE_CONTENTS(error, &err);
        goto fail;
    }

    p = buffer;
    while (p - buffer < size) {
        if (!read_one_object(&p, size - (p - buffer), error))
            goto fail;
    }

    err_NO_DATA(error);

fail:
    gwy_file_abandon_contents(buffer, size, NULL);

    return container;
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
