/*
 *  $Id: relate.c 21914 2019-02-26 09:49:26Z yeti-dn $
 *  Copyright (C) 2018 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwynlfit.h>
#include <libprocess/arithmetic.h>
#include <libprocess/stats.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwygraph.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>

#define RELATE_RUN_MODES (GWY_RUN_INTERACTIVE)

enum {
    MAX_PARAMS = 3,
    MAX_PLOT_DATA = 16384,
};

typedef enum {
    RELATE_FUNCTION_PROP      = 0,
    RELATE_FUNCTION_OFFSET    = 1,
    RELATE_FUNCTION_LINEAR    = 2,
    RELATE_FUNCTION_SQUARE    = 3,
    RELATE_FUNCTION_PARABOLIC = 4,
    RELATE_FUNCTION_UPOWER    = 5,
    RELATE_FUNCTION_LOG       = 6,
} RelateFunction;

typedef gdouble (*RelateEvalFunc)(gdouble z1, const gdouble *params);
typedef void (*RelateMakeLSMFunc)(const gdouble *z1, const gdouble *z2,
                                  guint n,
                                  gdouble *matrix, gdouble *rhs);

typedef struct {
    const char *name;
    gint power_x;
    gint power_y;
} GwyNLFitParam;

typedef struct {
    RelateFunction id;
    const gchar *name;
    const gchar *formula;
    const GwyNLFitParam *paraminfo;
    guint nparams;
    RelateEvalFunc func;
    RelateMakeLSMFunc make_lsm;
} RelateFuncInfo;

typedef struct {
    const gdouble *xdata;
    const gdouble *ydata;
    RelateEvalFunc func;
} RelateNLFittingData;

typedef struct {
    GwyAppDataId other_image;
    GwyAppDataId target_graph;
    GwyMaskingType masking;
    RelateFunction func;
    GwyResultsReportType report_style;
} RelateArgs;

typedef struct {
    RelateArgs *args;
    GtkDialog *dialog;
    GtkWidget *graph;
    GtkWidget *other_image;
    GtkWidget *target_graph;
    GtkWidget *func;
    GtkWidget *formula;
    GtkWidget *param_table;
    GtkWidget *param_name[MAX_PARAMS];
    GtkWidget *param_equal[MAX_PARAMS];
    GtkWidget *param_value[MAX_PARAMS];
    GtkWidget *param_pm[MAX_PARAMS];
    GtkWidget *param_error[MAX_PARAMS];
    GtkWidget *rss_label;
    GSList *masking;
    GtkWidget *rexport;

    GwyDataField *dfield1;
    GwyDataField *dfield2;
    GwyDataField *mask;
    GwyResults *results;
    GwyAppDataId first_image;

    gdouble param[MAX_PARAMS];
    gdouble error[MAX_PARAMS];
    gdouble *xdata;
    gdouble *ydata;
    gdouble rss;
    guint ndata;
} RelateControls;

#define DECLARE_FIT_FUNC(name) \
    static gdouble relate_func_##name(gdouble z1, const gdouble *param); \
    static void relate_lsm_##name(const gdouble *z1, const gdouble *z2, \
                                  guint n, \
                                  gdouble *matrix, gdouble *rhs)

DECLARE_FIT_FUNC(prop);
DECLARE_FIT_FUNC(offset);
DECLARE_FIT_FUNC(linear);
DECLARE_FIT_FUNC(square);
DECLARE_FIT_FUNC(parabolic);
DECLARE_FIT_FUNC(upower);
DECLARE_FIT_FUNC(log);

static gboolean              module_register     (void);
static void                  relate              (GwyContainer *data,
                                                  GwyRunType run);
static gboolean              relate_dialog       (RelateArgs *args,
                                                  GwyContainer *data,
                                                  GwyDataField *dfield,
                                                  GwyDataField *mask,
                                                  gint id);
static GtkWidget*            create_func_chooser (RelateControls *controls);
static void                  update_target_graphs(RelateControls *controls);
static gboolean              filter_target_graphs(GwyContainer *data,
                                                  gint id,
                                                  gpointer user_data);
static void                  target_graph_changed(RelateControls *controls);
static gboolean              filter_other_image  (GwyContainer *data,
                                                  gint id,
                                                  gpointer user_data);
static void                  other_image_changed (GwyDataChooser *chooser,
                                                  RelateControls *controls);
static void                  func_changed        (GtkComboBox *combo,
                                                  RelateControls *controls);
static void                  masking_changed     (GtkToggleButton *toggle,
                                                  RelateControls *controls);
static void                  report_style_changed(RelateControls *controls,
                                                  GwyResultsExport *rexport);
static const RelateFuncInfo* find_relate_func    (RelateFunction id);
static void                  replot_data         (RelateControls *controls);
static void                  recalculate         (RelateControls *controls);
static void                  plot_fit            (RelateControls *controls,
                                                  const RelateFuncInfo *finfo);
static gdouble               nlfitter_fit_func   (gdouble x,
                                                  gint nparam,
                                                  const gdouble *param,
                                                  gpointer user_data,
                                                  gboolean *success);
static void                  shuffle_array_stable(gdouble *a,
                                                  guint n,
                                                  guint nhead);
static void                  fill_results        (RelateControls *controls);
static void                  create_results      (RelateControls *controls);
static void                  update_param_table  (RelateControls *controls);
static void                  relate_load_args    (GwyContainer *container,
                                                  RelateArgs *args);
static void                  relate_save_args    (GwyContainer *container,
                                                  RelateArgs *args);

static const RelateArgs relate_defaults = {
    GWY_APP_DATA_ID_NONE, GWY_APP_DATA_ID_NONE,
    GWY_MASK_IGNORE, RELATE_FUNCTION_PROP,
    GWY_RESULTS_REPORT_COLON,
};

static const GwyNLFitParam params_prop[] = {
    { "a", -1, 1 },
};

static const GwyNLFitParam params_offset[] = {
    { "b", 0, 1 },
};

static const GwyNLFitParam params_linear[] = {
    { "a", -1, 1 },
    { "b",  0, 1 },
};

static const GwyNLFitParam params_square[] = {
    { "a", -2, 1 },
};

static const GwyNLFitParam params_parabolic[] = {
    { "a", -2, 1 },
    { "b", -1, 1 },
    { "c",  0, 1 },
};

static const GwyNLFitParam params_upower[] = {
    { "p", 0, 0 },
    { "q", 0, 0 },
};

static const GwyNLFitParam params_log[] = {
    { "p", 0, 0 },
    { "q", 0, 0 },
};

static const RelateFuncInfo func_info[] = {
    {
        RELATE_FUNCTION_PROP, N_("Proportion"),
        "<i>z</i><sub>2</sub> = <i>az</i><sub>1</sub>",
        params_prop, G_N_ELEMENTS(params_prop),
        relate_func_prop, relate_lsm_prop,
    },
    {
        RELATE_FUNCTION_OFFSET, N_("Offset"),
        "<i>z</i><sub>2</sub> = <i>z</i><sub>1</sub> + <i>b</i>",
        params_offset, G_N_ELEMENTS(params_offset),
        relate_func_offset, relate_lsm_offset,
    },
    {
        RELATE_FUNCTION_LINEAR, N_("Linear"),
        "<i>z</i><sub>2</sub> = <i>az</i><sub>1</sub> + <i>b</i>",
        params_linear, G_N_ELEMENTS(params_linear),
        relate_func_linear, relate_lsm_linear,
    },
    {
        RELATE_FUNCTION_SQUARE, N_("Square"),
        "<i>z</i><sub>2</sub> = <i>az</i><sub>1</sub><sup>2</sup>",
        params_square, G_N_ELEMENTS(params_square),
        relate_func_square, relate_lsm_square,
    },
    {
        RELATE_FUNCTION_PARABOLIC, N_("Parabolic"),
        "<i>z</i><sub>2</sub> = <i>az</i><sub>1</sub><sup>2</sup> + <i>bz</i><sub>1</sub> + <i>c</i>",
        params_parabolic, G_N_ELEMENTS(params_parabolic),
        relate_func_parabolic, relate_lsm_parabolic,
    },
    {
        RELATE_FUNCTION_UPOWER, N_("Power"),
        "ln <i>z</i><sub>2</sub> = <i>p</i>ln <i>z</i><sub>1</sub> + <i>q</i>",
        params_upower, G_N_ELEMENTS(params_upower),
        relate_func_upower, relate_lsm_upower,
    },
    {
        RELATE_FUNCTION_LOG, N_("Logarithm"),
        "<i>z</i><sub>2</sub> = <i>p</i>ln |<i>z</i><sub>1</sub>| + <i>q</i>",
        params_log, G_N_ELEMENTS(params_log),
        relate_func_log, relate_lsm_log,
    },
};

static GwyAppDataId other_id  = GWY_APP_DATA_ID_NONE;
static GwyAppDataId target_id = GWY_APP_DATA_ID_NONE;

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Plots one image data as a function of another and finds relations."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas",
    "2018",
};

GWY_MODULE_QUERY2(module_info, relate)

static gboolean
module_register(void)
{
    gwy_process_func_register("relate",
                              (GwyProcessFunc)&relate,
                              N_("/_Multidata/_Relation..."),
                              NULL,
                              RELATE_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Find simple relations between data"));

    return TRUE;
}

static void
relate(GwyContainer *data, GwyRunType run)
{
    RelateArgs args;
    GwyDataField *dfield, *mfield;
    gint id;

    g_return_if_fail(run & RELATE_RUN_MODES);
    relate_load_args(gwy_app_settings_get(), &args);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_MASK_FIELD, &mfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(dfield);

    relate_dialog(&args, data, dfield, mfield, id);
    relate_save_args(gwy_app_settings_get(), &args);
}

static gboolean
relate_dialog(RelateArgs *args, GwyContainer *data,
              GwyDataField *dfield, GwyDataField *mask, gint id)
{
    RelateControls controls;
    GtkWidget *dialog, *hbox, *label, *table;
    GwyDataChooser *chooser;
    GwyGraphModel *gmodel;
    GwyResultsExport *rexport;
    gint response, row;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.dfield1 = dfield;
    controls.mask = mask;
    controls.first_image.datano = gwy_app_data_browser_get_number(data);
    controls.first_image.id = id;

    dialog = gtk_dialog_new_with_buttons(_("Relate"), NULL, 0,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    controls.dialog = GTK_DIALOG(dialog);
    gtk_dialog_set_default_response(controls.dialog, GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);

    hbox = gtk_hbox_new(FALSE, 12);
    gtk_box_pack_start(GTK_BOX(controls.dialog->vbox), hbox, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);

    gmodel = gwy_graph_model_new();
    g_object_set(gmodel,
                 "axis-label-bottom", "z<sub>1</sub>",
                 "axis-label-left", "z<sub>2</sub>",
                 NULL);
    controls.graph = gwy_graph_new(gmodel);
    gtk_widget_set_size_request(controls.graph, 480, 480);
    gwy_graph_enable_user_input(GWY_GRAPH(controls.graph), FALSE);
    g_object_unref(gmodel);
    gtk_box_pack_start(GTK_BOX(hbox), controls.graph, TRUE, TRUE, 0);

    table = gtk_table_new(9 + (mask ? 4 : 0), 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_box_pack_start(GTK_BOX(hbox), table, FALSE, FALSE, 0);
    row = 0;

    controls.other_image = gwy_data_chooser_new_channels();
    chooser = GWY_DATA_CHOOSER(controls.other_image);
    gwy_data_chooser_set_active(chooser, NULL, -1);
    gwy_data_chooser_set_filter(chooser, filter_other_image, dfield, NULL);
    gwy_data_chooser_set_active_id(chooser, &args->other_image);
    gwy_data_chooser_get_active_id(chooser, &args->other_image);
    gwy_table_attach_adjbar(table, row++, _("Second _image:"), NULL,
                            GTK_OBJECT(controls.other_image),
                            GWY_HSCALE_WIDGET);
    g_signal_connect(controls.other_image, "changed",
                     G_CALLBACK(other_image_changed), &controls);

    controls.target_graph = gwy_data_chooser_new_graphs();
    chooser = GWY_DATA_CHOOSER(controls.target_graph);
    gwy_data_chooser_set_none(chooser, _("New graph"));
    gwy_data_chooser_set_active(chooser, NULL, -1);
    gwy_data_chooser_set_filter(chooser, filter_target_graphs, &controls, NULL);
    gwy_data_chooser_set_active_id(chooser, &args->target_graph);
    gwy_data_chooser_get_active_id(chooser, &args->target_graph);
    gwy_table_attach_adjbar(table, row++, _("Target _graph:"), NULL,
                            GTK_OBJECT(controls.target_graph),
                            GWY_HSCALE_WIDGET);
    g_signal_connect_swapped(controls.target_graph, "changed",
                             G_CALLBACK(target_graph_changed), &controls);

    if (mask) {
        gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
        gtk_table_attach(GTK_TABLE(table),
                         gwy_label_new_header(_("Masking Mode")),
                         0, 2, row, row+1, GTK_FILL, 0, 0, 0);
        row++;

        controls.masking
            = gwy_radio_buttons_create(gwy_masking_type_get_enum(), -1,
                                       G_CALLBACK(masking_changed),
                                       &controls, args->masking);
        row = gwy_radio_buttons_attach_to_table(controls.masking,
                                                GTK_TABLE(table), 2, row);
    }

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table), gwy_label_new_header(_("Function")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.func = create_func_chooser(&controls);
    gwy_table_attach_adjbar(table, row++, _("_Function type:"), NULL,
                            GTK_OBJECT(controls.func), GWY_HSCALE_WIDGET);

    controls.formula = label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 0, 3, row, row+1,
                     GTK_FILL, 0, 0, 0);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table), gwy_label_new_header(_("Fit Results")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.param_table = gtk_table_new(1, 5, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(controls.param_table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(controls.param_table), 8);
    gtk_table_attach(GTK_TABLE(table), controls.param_table,
                     0, 3, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    label = gtk_label_new(_("Mean square difference:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
                     GTK_FILL, 0, 0, 0);

    controls.rss_label = label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 1, 2, row, row+1,
                     GTK_FILL, 0, 0, 0);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.rexport = gwy_results_export_new(args->report_style);
    rexport = GWY_RESULTS_EXPORT(controls.rexport);
    gwy_results_export_set_title(rexport, _("Save Parameters"));
    gtk_table_attach(GTK_TABLE(table), controls.rexport, 0, 2, row, row+1,
                     GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(rexport, "format-changed",
                             G_CALLBACK(report_style_changed), &controls);

    gtk_widget_show_all(dialog);
    func_changed(GTK_COMBO_BOX(controls.func), &controls);
    other_image_changed(GWY_DATA_CHOOSER(controls.other_image), &controls);

    do {
        response = gtk_dialog_run(controls.dialog);
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            g_free(controls.xdata);
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            gwy_app_add_graph_or_curves(gmodel, data, &args->target_graph, 1);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    g_free(controls.xdata);
    gtk_widget_destroy(dialog);
    return TRUE;
}

static GtkWidget*
create_func_chooser(RelateControls *controls)
{
    GwyEnum *model;
    guint i, n = G_N_ELEMENTS(func_info);
    GtkWidget *combo;

    model = g_new(GwyEnum, n);
    for (i = 0; i < n; i++) {
        model[i].name = func_info[i].name;
        model[i].value = func_info[i].id;
    }

    combo = gwy_enum_combo_box_new(model, n,
                                   G_CALLBACK(func_changed), controls,
                                   controls->args->func, TRUE);
    g_object_weak_ref(G_OBJECT(combo), (GWeakNotify)g_free, model);

    return combo;
}

static void
update_target_graphs(RelateControls *controls)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(controls->target_graph);
    gwy_data_chooser_refilter(chooser);
}

static gboolean
filter_target_graphs(GwyContainer *data, gint id, gpointer user_data)
{
    RelateControls *controls = (RelateControls*)user_data;
    GwyGraphModel *gmodel, *targetgmodel;
    GQuark quark = gwy_app_get_graph_key_for_id(id);

    gmodel = gwy_graph_get_model(GWY_GRAPH(controls->graph));
    g_return_val_if_fail(GWY_IS_GRAPH_MODEL(gmodel), FALSE);
    return (gwy_container_gis_object(data, quark, (GObject**)&targetgmodel)
            && gwy_graph_model_units_are_compatible(gmodel, targetgmodel));
}

static void
target_graph_changed(RelateControls *controls)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(controls->target_graph);

    gwy_data_chooser_get_active_id(chooser, &controls->args->target_graph);
}

static gboolean
filter_other_image(GwyContainer *data, gint id, gpointer user_data)
{
    GwyDataField *dfield = (GwyDataField*)user_data, *otherfield;
    GQuark quark = gwy_app_get_data_key_for_id(id);

    if (!gwy_container_gis_object(data, quark, (GObject**)&otherfield)
        || otherfield == dfield)
        return FALSE;

    if (gwy_data_field_check_compatibility(dfield, otherfield,
                                           GWY_DATA_COMPATIBILITY_RES
                                           | GWY_DATA_COMPATIBILITY_REAL
                                           | GWY_DATA_COMPATIBILITY_LATERAL))
        return FALSE;

    return TRUE;
}

static void
other_image_changed(GwyDataChooser *chooser, RelateControls *controls)
{
    RelateArgs *args = controls->args;
    GwyContainer *data;
    GQuark quark;

    if (gwy_data_chooser_get_active_id(chooser, &args->other_image)) {
        data = gwy_app_data_browser_get(args->other_image.datano);
        quark = gwy_app_get_data_key_for_id(args->other_image.id);
        controls->dfield2 = gwy_container_get_object(data, quark);

        replot_data(controls);
        recalculate(controls);
        update_target_graphs(controls);
    }

    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialog),
                                      GTK_RESPONSE_OK, !!controls->dfield2);
}

static void
func_changed(GtkComboBox *combo, RelateControls *controls)
{
    const RelateFuncInfo *finfo;
    GtkTable *table;
    GtkWidget *label;
    guint i, nparam;

    finfo = find_relate_func(controls->args->func);
    nparam = finfo->nparams;
    for (i = 0; i < nparam; i++) {
        if (controls->param_name[i]) {
            gtk_widget_destroy(controls->param_name[i]);
            controls->param_name[i] = NULL;
        }
        if (controls->param_equal[i]) {
            gtk_widget_destroy(controls->param_equal[i]);
            controls->param_equal[i] = NULL;
        }
        if (controls->param_value[i]) {
            gtk_widget_destroy(controls->param_value[i]);
            controls->param_value[i] = NULL;
        }
        if (controls->param_pm[i]) {
            gtk_widget_destroy(controls->param_pm[i]);
            controls->param_pm[i] = NULL;
        }
        if (controls->param_error[i]) {
            gtk_widget_destroy(controls->param_error[i]);
            controls->param_error[i] = NULL;
        }
    }

    controls->args->func = gwy_enum_combo_box_get_active(combo);
    finfo = find_relate_func(controls->args->func);
    g_return_if_fail(finfo);
    nparam = finfo->nparams;

    gtk_label_set_markup(GTK_LABEL(controls->formula), finfo->formula);

    table = GTK_TABLE(controls->param_table);
    gtk_table_resize(table, nparam, 5);

    for (i = 0; i < nparam; i++) {
        label = controls->param_name[i] = gtk_label_new(NULL);
        gtk_label_set_markup(GTK_LABEL(label), finfo->paraminfo[i].name);
        gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
        gtk_table_attach(table, label, 0, 1, i, i+1,
                         GTK_EXPAND | GTK_FILL, 0, 0, 0);

        label = controls->param_equal[i] = gtk_label_new("=");
        gtk_table_attach(table, label, 1, 2, i, i+1, GTK_FILL, 0, 0, 0);

        label = controls->param_value[i] = gtk_label_new(NULL);
        gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
        gtk_table_attach(table, label, 2, 3, i, i+1, GTK_FILL, 0, 0, 0);

        label = controls->param_pm[i] = gtk_label_new("±");
        gtk_table_attach(table, label, 3, 4, i, i+1, GTK_FILL, 0, 0, 0);

        label = controls->param_error[i] = gtk_label_new(NULL);
        gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
        gtk_table_attach(table, label, 4, 5, i, i+1, GTK_FILL, 0, 0, 0);
    }
    gtk_widget_show_all(controls->param_table);

    create_results(controls);
    recalculate(controls);
}

static void
masking_changed(GtkToggleButton *toggle, RelateControls *controls)
{
    if (!gtk_toggle_button_get_active(toggle))
        return;

    controls->args->masking = gwy_radio_buttons_get_current(controls->masking);
    replot_data(controls);
    recalculate(controls);
}

static void
report_style_changed(RelateControls *controls, GwyResultsExport *rexport)
{
    controls->args->report_style = gwy_results_export_get_format(rexport);
}

static const RelateFuncInfo*
find_relate_func(RelateFunction id)
{
    guint i;

    for (i = 0; i < G_N_ELEMENTS(func_info); i++) {
        if (func_info[i].id == id)
            return func_info + i;
    }
    return NULL;
}

static void
replot_data(RelateControls *controls)
{
    GwyGraphModel *gmodel;
    GwyGraphCurveModel *gcmodel;
    const gdouble *d1, *d2, *m;
    GwyMaskingType masking;
    GwySIUnit *unit;
    gint nc, xres, yres, n, i, ndata;

    gmodel = gwy_graph_get_model(GWY_GRAPH(controls->graph));
    nc = gwy_graph_model_get_n_curves(gmodel);
    if (nc > 0)
        gcmodel = gwy_graph_model_get_curve(gmodel, 0);
    else {
        gcmodel = gwy_graph_curve_model_new();
        g_object_set(gcmodel,
                     "mode", GWY_GRAPH_CURVE_POINTS,
                     "point-type", GWY_GRAPH_POINT_SQUARE,
                     "point-size", 1,
                     "color", gwy_graph_get_preset_color(0),
                     "description", _("Data"),
                     NULL);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
    }

    if (!controls->dfield2)
        return;

    unit = gwy_data_field_get_si_unit_z(controls->dfield1);
    g_object_set(gmodel, "si-unit-x", unit, NULL);
    unit = gwy_data_field_get_si_unit_z(controls->dfield2);
    g_object_set(gmodel, "si-unit-y", unit, NULL);

    xres = gwy_data_field_get_xres(controls->dfield1);
    yres = gwy_data_field_get_yres(controls->dfield1);
    n = xres*yres;
    d1 = gwy_data_field_get_data_const(controls->dfield1);
    d2 = gwy_data_field_get_data_const(controls->dfield2);
    masking = controls->args->masking;

    if (!controls->xdata) {
        controls->xdata = g_new(gdouble, 2*n);
        controls->ydata = controls->xdata + n;
    }

    if (!controls->mask || masking == GWY_MASK_IGNORE) {
        gwy_assign(controls->xdata, d1, n);
        gwy_assign(controls->ydata, d2, n);
        ndata = n;
    }
    else {
        m = gwy_data_field_get_data_const(controls->mask);
        ndata = 0;
        for (i = 0; i < n; i++) {
            if ((masking == GWY_MASK_INCLUDE && m[i] >= 1.0)
                || (masking == GWY_MASK_EXCLUDE && m[i] <= 0.0)) {
                controls->xdata[ndata] = d1[i];
                controls->ydata[ndata] = d2[i];
                ndata++;
            }
        }
    }

    controls->ndata = ndata;
    if (ndata > MAX_PLOT_DATA) {
        shuffle_array_stable(controls->xdata, ndata, MAX_PLOT_DATA);
        shuffle_array_stable(controls->ydata, ndata, MAX_PLOT_DATA);
        ndata = MAX_PLOT_DATA;
    }
    gwy_graph_curve_model_set_data(gcmodel,
                                   controls->xdata, controls->ydata, ndata);
}

static void
recalculate(RelateControls *controls)
{
    const RelateFuncInfo *finfo;
    RelateArgs *args = controls->args;
    GwyNLFitter *fitter;
    gdouble *matrix;
    guint nparam, i;
    gboolean ok = TRUE;
    gdouble rss;

    if (!controls->xdata)
        return;

    /* Linear fitting.  For simple relations this is the same as the final
     * fit but for transformed we take it just as an estimate. */
    finfo = find_relate_func(args->func);
    g_return_if_fail(finfo);
    nparam = finfo->nparams;

    matrix = g_new0(gdouble, nparam*(nparam+1)/2);
    gwy_clear(controls->param, MAX_PARAMS);
    finfo->make_lsm(controls->xdata, controls->ydata, controls->ndata,
                    matrix, controls->param);
    if (gwy_math_choleski_decompose(nparam, matrix))
        gwy_math_choleski_solve(nparam, matrix, controls->param);
    else
        ok = FALSE;
    g_free(matrix);

    /* Non-linear fitting. */
    if (nparam >= controls->ndata)
        ok = FALSE;

    if (ok) {
        fitter = gwy_math_nlfit_new(nlfitter_fit_func, NULL);
        rss = gwy_math_nlfit_fit(fitter, controls->ndata,
                                 controls->xdata, controls->ydata,
                                 nparam, controls->param,
                                 finfo->func);
        if (rss > 0.0) {
            controls->rss = sqrt(rss/(controls->ndata - nparam));
            for (i = 0; i < nparam; i++)
                controls->error[i] = gwy_math_nlfit_get_sigma(fitter, i);
        }
        else
            ok = FALSE;
        gwy_math_nlfit_free(fitter);
    }

    if (!ok) {
        gwy_clear(controls->param, nparam);
        gwy_clear(controls->error, nparam);
        controls->rss = 0.0;
    }

    fill_results(controls);
    update_param_table(controls);
    plot_fit(controls, finfo);

    if (!ok)
        g_warning("Fit failed!");
}

static void
update_param_table(RelateControls *controls)
{
    const GwySIUnitFormatStyle style = GWY_SI_UNIT_FORMAT_VFMARKUP;
    GwySIUnit *unit, *xunit, *yunit;
    GwySIValueFormat *vf = NULL;
    const RelateFuncInfo *finfo;
    gchar buf[80];
    guint i;

    xunit = gwy_data_field_get_si_unit_xy(controls->dfield1);
    yunit = gwy_data_field_get_si_unit_z(controls->dfield2);
    unit = gwy_si_unit_new(NULL);
    finfo = find_relate_func(controls->args->func);
    for (i = 0; i < finfo->nparams; i++) {
        gint power_x = finfo->paraminfo[i].power_x;
        gint power_y = finfo->paraminfo[i].power_y;

        gwy_si_unit_set_from_string(unit, NULL);
        gwy_si_unit_power_multiply(xunit, power_x, yunit, power_y, unit);
        vf = gwy_si_unit_get_format(unit, style, controls->param[i], vf);
        vf->precision += 3;
        g_snprintf(buf, sizeof(buf), "%.*f%s%s",
                   vf->precision, controls->param[i]/vf->magnitude,
                   *vf->units ? " " : "", vf->units);
        gtk_label_set_markup(GTK_LABEL(controls->param_value[i]), buf);

        vf = gwy_si_unit_get_format(unit, style, controls->error[i], vf);
        g_snprintf(buf, sizeof(buf), "%.*f%s%s",
                   vf->precision, controls->error[i]/vf->magnitude,
                   *vf->units ? " " : "", vf->units);
        gtk_label_set_markup(GTK_LABEL(controls->param_error[i]), buf);
    }

    vf = gwy_si_unit_get_format(yunit, style, controls->rss, vf);
    g_snprintf(buf, sizeof(buf), "%.*f %s",
               vf->precision, controls->rss/vf->magnitude, vf->units);
    gtk_label_set_markup(GTK_LABEL(controls->rss_label), buf);

    gwy_si_unit_value_format_free(vf);
    g_object_unref(unit);
}

static void
plot_fit(RelateControls *controls, const RelateFuncInfo *finfo)
{
    GwyGraphModel *gmodel;
    GwyGraphCurveModel *gcmodel;
    gdouble min, max;
    gdouble *xdata, *ydata;
    RelateEvalFunc evalfunc;
    gint i, ndata, nc, xres, yres;

    gmodel = gwy_graph_get_model(GWY_GRAPH(controls->graph));
    nc = gwy_graph_model_get_n_curves(gmodel);
    if (nc < 2) {
        gcmodel = gwy_graph_curve_model_new();
        g_object_set(gcmodel,
                     "mode", GWY_GRAPH_CURVE_LINE,
                     "color", gwy_graph_get_preset_color(1),
                     "description", _("Fit"),
                     NULL);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
    }
    else
        gcmodel = gwy_graph_model_get_curve(gmodel, 1);

    ndata = 241;
    xdata = g_new(gdouble, 2*ndata);
    ydata = xdata + ndata;
    xres = gwy_data_field_get_xres(controls->dfield1);
    yres = gwy_data_field_get_yres(controls->dfield1);
    gwy_data_field_area_get_min_max_mask(controls->dfield1, controls->mask,
                                         controls->args->masking,
                                         0, 0, xres, yres,
                                         &min, &max);

    evalfunc = finfo->func;
    for (i = 0; i < ndata; i++) {
        gdouble t = i/(ndata - 1.0);

        xdata[i] = t*max + (1.0 - t)*min;
        ydata[i] = evalfunc(xdata[i], controls->param);
    }
    gwy_graph_curve_model_set_data(gcmodel, xdata, ydata, ndata);
    g_free(xdata);
}

/* Deterministically shuffle the beginning an array.  Arrays of the same size
 * is always shuffled the same way. */
static void
shuffle_array_stable(gdouble *a, guint n, guint nhead)
{
    guint i, nshuf = MIN(n, nhead);
    GRand *rng = g_rand_new_with_seed(42);

    for (i = 0; i < nshuf; i++) {
        guint j = g_rand_int_range(rng, 0, n);

        GWY_SWAP(gdouble, a[i], a[j]);
    }

    g_rand_free(rng);
}

static gdouble
nlfitter_fit_func(gdouble x,
                  G_GNUC_UNUSED gint nparam,
                  const gdouble *param,
                  gpointer user_data,
                  gboolean *success)
{
    RelateEvalFunc func = (RelateEvalFunc)user_data;

    *success = TRUE;
    return func(x, param);
}

static gdouble
relate_func_prop(gdouble z1, const gdouble *param)
{
    return z1*param[0];
}

static void
relate_lsm_prop(const gdouble *z1, const gdouble *z2, guint n,
                gdouble *matrix, gdouble *rhs)
{
    guint i;

    for (i = 0; i < n; i++) {
        gdouble x = z1[i], y = z2[i];

        matrix[0] += x*x;
        rhs[0] += y*x;
    }
}

static gdouble
relate_func_offset(gdouble z1, const gdouble *param)
{
    return z1 + param[0];
}

static void
relate_lsm_offset(const gdouble *z1, const gdouble *z2, guint n,
                  gdouble *matrix, gdouble *rhs)
{
    guint i;

    for (i = 0; i < n; i++) {
        gdouble x = z1[i], y = z2[i];

        matrix[0] += 1.0;
        rhs[0] += y - x;
    }
}

static gdouble
relate_func_linear(gdouble z1, const gdouble *param)
{
    return z1*param[0] + param[1];
}

static void
relate_lsm_linear(const gdouble *z1, const gdouble *z2, guint n,
                  gdouble *matrix, gdouble *rhs)
{
    guint i;

    for (i = 0; i < n; i++) {
        gdouble x = z1[i], y = z2[i];

        matrix[0] += x*x;
        matrix[1] += x;
        matrix[2] += 1.0;
        rhs[0] += y*x;
        rhs[1] += y;
    }
}

static gdouble
relate_func_square(gdouble z1, const gdouble *param)
{
    return z1*z1*param[0];
}

static void
relate_lsm_square(const gdouble *z1, const gdouble *z2, guint n,
                  gdouble *matrix, gdouble *rhs)
{
    guint i;

    for (i = 0; i < n; i++) {
        gdouble x = z1[i], y = z2[i];
        gdouble xx = x*x;

        matrix[0] += xx*xx;
        rhs[0] += y*xx;
    }
}

static gdouble
relate_func_parabolic(gdouble z1, const gdouble *param)
{
    return z1*(z1*param[0] + param[1]) + param[2];
}

static void
relate_lsm_parabolic(const gdouble *z1, const gdouble *z2, guint n,
                     gdouble *matrix, gdouble *rhs)
{
    guint i;

    for (i = 0; i < n; i++) {
        gdouble x = z1[i], y = z2[i];
        gdouble xx = x*x;

        matrix[0] += xx*xx;
        matrix[1] += xx*x;
        matrix[2] += xx;
        matrix[3] += xx;
        matrix[4] += x;
        matrix[5] += 1.0;
        rhs[0] += y*xx;
        rhs[1] += y*x;
        rhs[2] += y;
    }
}

static gdouble
relate_func_upower(gdouble z1, const gdouble *param)
{
    if (z1 == 0.0)
        return 0.0;
    if (z1 < 0.0)
        return -pow(fabs(z1), param[0]) * exp(param[1]);
    return pow(fabs(z1), param[0]) * exp(param[1]);
}

static void
relate_lsm_upower(const gdouble *z1, const gdouble *z2, guint n,
                  gdouble *matrix, gdouble *rhs)
{
    guint i;

    for (i = 0; i < n; i++) {
        gdouble x = z1[i], y = z2[i], w, lx, ly;

        if (x == 0.0 || y == 0.0)
            continue;

        w = fabs(x) + fabs(y);
        lx = log(fabs(x));
        ly = log(fabs(y));
        matrix[0] += lx*lx*w;
        matrix[1] += lx*w;
        matrix[2] += w;
        rhs[0] += ly*lx*w;
        rhs[1] += ly*w;
    }
}

static gdouble
relate_func_log(gdouble z1, const gdouble *param)
{
    if (z1 == 0.0)
        return 0.0;
    return param[0]*log(fabs(z1)) + param[1];
}

static void
relate_lsm_log(const gdouble *z1, const gdouble *z2, guint n,
               gdouble *matrix, gdouble *rhs)
{
    guint i;

    for (i = 0; i < n; i++) {
        gdouble x = z1[i], y = z2[i], lx;

        if (x == 0.0)
            continue;

        lx = log(fabs(x));
        matrix[0] += lx*lx;
        matrix[1] += lx;
        matrix[2] += 1.0;
        rhs[0] += y*lx;
        rhs[1] += y;
    }
}

static void
create_results(RelateControls *controls)
{
    RelateArgs *args = controls->args;
    GwyResults *results;
    guint i;
    const RelateFuncInfo *finfo;

    GWY_OBJECT_UNREF(controls->results);
    results = controls->results = gwy_results_new();
    gwy_results_add_header(results, N_("Fit Results"));
    gwy_results_add_value_str(results, "file", N_("File"));
    gwy_results_add_value_str(results, "channel1", N_("First image"));
    gwy_results_add_value_str(results, "channel2", N_("Second image"));
    gwy_results_add_format(results, "npts", N_("Number of points"), TRUE,
    /* TRANSLATORS: %{n}i and %{total}i are ids, do NOT translate them. */
                           N_("%{n}i of %{ntotal}i"), NULL);
    gwy_results_add_value_str(results, "func", N_("Fitted function"));
    gwy_results_add_value_z(results, "rss", N_("Mean square difference"));

    gwy_results_add_separator(results);
    gwy_results_add_header(results, N_("Parameters"));

    finfo = find_relate_func(args->func);
    for (i = 0; i < finfo->nparams; i++) {
        const gchar *name = finfo->paraminfo[i].name;
        gint power_x = finfo->paraminfo[i].power_x;
        gint power_y = finfo->paraminfo[i].power_y;

        gwy_results_add_value(results, name, "",
                              "symbol", name, "is-fitting-param", TRUE,
                              "power-x", power_x, "power-y", power_y,
                              NULL);

    }

    gwy_results_export_set_results(GWY_RESULTS_EXPORT(controls->rexport),
                                   controls->results);
}

static void
fill_results(RelateControls *controls)
{
    RelateArgs *args = controls->args;
    GwyResults *results = controls->results;
    GwyDataField *dfield1 = controls->dfield1;
    GwyDataField *dfield2 = controls->dfield2;
    const GwyAppDataId *id;
    GwySIUnit *xunit, *yunit;
    guint n, i;
    const RelateFuncInfo *finfo;

    finfo = find_relate_func(args->func);
    xunit = gwy_data_field_get_si_unit_z(dfield1);
    yunit = gwy_data_field_get_si_unit_z(dfield2);
    n = gwy_data_field_get_xres(dfield1) * gwy_data_field_get_yres(dfield1);
    id = &controls->first_image;
    gwy_results_fill_channel(results, "channel1",
                             gwy_app_data_browser_get(id->datano), id->id);
    id = &controls->args->other_image;
    gwy_results_fill_channel(results, "channel2",
                             gwy_app_data_browser_get(id->datano), id->id);
    gwy_results_set_unit(results, "x", xunit);
    gwy_results_set_unit(results, "y", yunit);

    id = &controls->first_image;
    gwy_results_fill_filename(results, "file",
                              gwy_app_data_browser_get(id->datano));
    gwy_results_fill_values(results,
                            "func", finfo->name,
                            "rss", controls->rss,
                            NULL);
    gwy_results_fill_format(results, "npts",
                            "n", controls->ndata,
                            "ntotal", n,
                            NULL);

    for (i = 0; i < finfo->nparams; i++) {
        gwy_results_fill_values_with_errors(results,
                                            finfo->paraminfo[i].name,
                                            controls->param[i],
                                            controls->error[i],
                                            NULL);
    }
}

static const gchar func_key[]         = "/module/relate/func";
static const gchar masking_key[]      = "/module/relate/masking";
static const gchar report_style_key[] = "/module/relate/report_style";

static void
relate_sanitize_args(RelateArgs *args)
{
    gwy_app_data_id_verify_channel(&args->other_image);
    gwy_app_data_id_verify_graph(&args->target_graph);
}

static void
relate_load_args(GwyContainer *container, RelateArgs *args)
{
    *args = relate_defaults;

    gwy_container_gis_enum_by_name(container, masking_key, &args->masking);
    gwy_container_gis_enum_by_name(container, func_key, &args->func);
    gwy_container_gis_enum_by_name(container, report_style_key,
                                   &args->report_style);
    args->target_graph = target_id;
    args->other_image = other_id;
    relate_sanitize_args(args);
}

static void
relate_save_args(GwyContainer *container, RelateArgs *args)
{
    target_id = args->target_graph;
    other_id = args->other_image;
    gwy_container_set_enum_by_name(container, masking_key, args->masking);
    gwy_container_set_enum_by_name(container, func_key, args->func);
    gwy_container_set_enum_by_name(container, report_style_key,
                                   args->report_style);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
