/*
 * support.c
 * This file is part of gtk-nodoka-engine
 *
 * Copyright (C) 2007, 2008 - Martin Sourada, Daniel Geiger
 *
 * gtk-nodoka-engine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * gtk-nodoka-engine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gtk-nodoka-engine; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */ 

#include "support.h"

GtkTextDirection
nodoka_get_direction (GtkWidget * widget)
{
	GtkTextDirection dir;

	if (widget)
		dir = gtk_widget_get_direction (widget);
	else
		dir = GTK_TEXT_DIR_LTR;

	return dir;
}

G_GNUC_INTERNAL void
nodoka_rgb_to_hls (gdouble * r, gdouble * g, gdouble * b)
{
	gdouble min;
	gdouble max;
	gdouble red;
	gdouble green;
	gdouble blue;
	gdouble h, l, s;
	gdouble delta;

	red = *r;
	green = *g;
	blue = *b;

	if (red > green)
	{
		if (red > blue)
			max = red;
		else
			max = blue;

		if (green < blue)
			min = green;
		else
			min = blue;
	}
	else
	{
		if (green > blue)
			max = green;
		else
			max = blue;

		if (red < blue)
			min = red;
		else
			min = blue;
	}

	l = (max + min) / 2;
	s = 0;
	h = 0;

	if (max != min)
	{
		if (l <= 0.5)
			s = (max - min) / (max + min);
		else
			s = (max - min) / (2 - max - min);

		delta = max - min;
		if (red == max)
			h = (green - blue) / delta;
		else if (green == max)
			h = 2 + (blue - red) / delta;
		else if (blue == max)
			h = 4 + (red - green) / delta;

		h *= 60;
		if (h < 0.0)
			h += 360;
	}

	*r = h;
	*g = l;
	*b = s;
}

G_GNUC_INTERNAL void
nodoka_hls_to_rgb (gdouble * h, gdouble * l, gdouble * s)
{
	gdouble hue;
	gdouble lightness;
	gdouble saturation;
	gdouble m1, m2;
	gdouble r, g, b;

	lightness = *l;
	saturation = *s;

	if (lightness <= 0.5)
		m2 = lightness * (1 + saturation);
	else
		m2 = lightness + saturation - lightness * saturation;

	m1 = 2 * lightness - m2;

	if (saturation == 0)
	{
		*h = lightness;
		*l = lightness;
		*s = lightness;
	}
	else
	{
		hue = *h + 120;
		while (hue > 360)
			hue -= 360;
		while (hue < 0)
			hue += 360;

		if (hue < 60)
			r = m1 + (m2 - m1) * hue / 60;
		else if (hue < 180)
			r = m2;
		else if (hue < 240)
			r = m1 + (m2 - m1) * (240 - hue) / 60;
		else
			r = m1;

		hue = *h;
		while (hue > 360)
			hue -= 360;
		while (hue < 0)
			hue += 360;

		if (hue < 60)
			g = m1 + (m2 - m1) * hue / 60;
		else if (hue < 180)
			g = m2;
		else if (hue < 240)
			g = m1 + (m2 - m1) * (240 - hue) / 60;
		else
			g = m1;

		hue = *h - 120;
		while (hue > 360)
			hue -= 360;
		while (hue < 0)
			hue += 360;

		if (hue < 60)
			b = m1 + (m2 - m1) * hue / 60;
		else if (hue < 180)
			b = m2;
		else if (hue < 240)
			b = m1 + (m2 - m1) * (240 - hue) / 60;
		else
			b = m1;

		*h = r;
		*l = g;
		*s = b;
	}
}

void
nodoka_shade (const NodokaRGB * a, NodokaRGB * b, float k)
{
	double red;
	double green;
	double blue;

	red = a->r;
	green = a->g;
	blue = a->b;

	nodoka_rgb_to_hls (&red, &green, &blue);

	green *= k;
	if (green > 1.0)
		green = 1.0;
	else if (green < 0.0)
		green = 0.0;

	blue *= k;
	if (blue > 1.0)
		blue = 1.0;
	else if (blue < 0.0)
		blue = 0.0;

	nodoka_hls_to_rgb (&red, &green, &blue);

	b->r = red;
	b->g = green;
	b->b = blue;
}


void
nodoka_gtk_treeview_get_header_index (GtkTreeView * tv, GtkWidget * header,
									  gint * column_index, gint * columns,
									  gboolean * resizable)
{
	GList *list, *list_start;
	*column_index = *columns = 0;
	list_start = list = gtk_tree_view_get_columns (tv);

	do
	{
		GtkTreeViewColumn *column = GTK_TREE_VIEW_COLUMN (list->data);
		if (gtk_tree_view_column_get_widget (column) == header)
		{
			*column_index = *columns;
			*resizable = gtk_tree_view_column_get_resizable (column);
		}
		if (gtk_tree_view_column_get_visible (column))
			(*columns)++;
	}
	while ((list = g_list_next (list)));

	g_list_free (list_start);
}

#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION < 90 && !defined(GTK_DISABLE_DEPRECATED)
void
nodoka_gtk_clist_get_header_index (GtkCList * clist, GtkWidget * button,
								   gint * column_index, gint * columns)
{
	int i;
	*columns = clist->columns;

	for (i = 0; i < *columns; i++)
	{
		if (clist->column[i].button == button)
		{
			*column_index = i;
			break;
		}
	}
}
#endif

gboolean
nodoka_sanitize_size (GdkWindow * window, gint * width, gint * height)
{
	gboolean set_bg = FALSE;

	if ((*width == -1) && (*height == -1))
	{
		set_bg = GDK_IS_WINDOW (window);
		gdk_drawable_get_size (window, width, height);
	}
	else if (*width == -1)
		gdk_drawable_get_size (window, width, NULL);
	else if (*height == -1)
		gdk_drawable_get_size (window, NULL, height);

	return set_bg;
}

static GtkRequisition default_option_indicator_size = { 7, 13 };
static GtkBorder default_option_indicator_spacing = { 7, 5, 2, 2 };

void
nodoka_option_menu_get_props (GtkWidget * widget,
							  GtkRequisition * indicator_size,
							  GtkBorder * indicator_spacing)
{
	GtkRequisition *tmp_size = NULL;
	GtkBorder *tmp_spacing = NULL;

	if (widget)
		gtk_widget_style_get (widget, "indicator_size", &tmp_size,
							  "indicator_spacing", &tmp_spacing, NULL);

	if (tmp_size)
	{
		*indicator_size = *tmp_size;
		g_free (tmp_size);
	}
	else
		*indicator_size = default_option_indicator_size;

	if (tmp_spacing)
	{
		*indicator_spacing = *tmp_spacing;
		gtk_border_free (tmp_spacing);
	}
	else
		*indicator_spacing = default_option_indicator_spacing;
}

GtkWidget *
nodoka_special_get_ancestor (GtkWidget * widget, GType widget_type)
{
	g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);

	while (widget && gtk_widget_get_parent (widget)
		   && !g_type_is_a (G_OBJECT_TYPE (gtk_widget_get_parent (widget)), widget_type))
		widget = gtk_widget_get_parent (widget);

	if (!
		(widget && gtk_widget_get_parent (widget)
		 && g_type_is_a (G_OBJECT_TYPE (gtk_widget_get_parent (widget)), widget_type)))
		return NULL;

	return widget;
}

GtkWidget *
nodoka_get_parent_window (GtkWidget * widget)
{
	GtkWidget *parent = gtk_widget_get_parent (widget);

	while (parent && !gtk_widget_get_has_window (parent))
		parent = gtk_widget_get_parent (parent);

	return parent;
}

GdkColor *
nodoka_get_parent_bgcolor (GtkWidget * widget)
{
	GtkWidget *parent = nodoka_get_parent_window (widget);

	if (parent && gtk_widget_get_style (parent))
		return &gtk_widget_get_style (parent)->bg[GTK_STATE_NORMAL];

	return NULL;
}

GtkWidget *
ndk_find_combo_box_widget (GtkWidget * widget)
{
	GtkWidget *result = NULL;

	if (widget && !GTK_IS_COMBO_BOX_ENTRY (widget))
	{
		if (GTK_IS_COMBO_BOX (widget))
			result = widget;
		else
			result = ndk_find_combo_box_widget (gtk_widget_get_parent (widget));
	}

	return result;
}

gboolean
ndk_is_combo_box (GtkWidget * widget)
{
	return (ndk_find_combo_box_widget (widget) != NULL);
}

void
nodoka_gdk_color_to_rgb (GdkColor * c, double *r, double *g, double *b)
{
	*r = (double) c->red / (double) 65535;
	*g = (double) c->green / (double) 65535;
	*b = (double) c->blue / (double) 65535;
}

void
nodoka_get_parent_bg (const GtkWidget * widget, NodokaRGB * color)
{
	GtkStateType state_type;
	const GtkWidget *parent;
	GdkColor *gcolor;

	if (widget == NULL)
	{
		color->r = color->g = color->b = 255;
		return;
	}

	parent = gtk_widget_get_parent ((GtkWidget *) widget);

	while (parent && !gtk_widget_get_has_window ((GtkWidget *) parent))
		parent = gtk_widget_get_parent ((GtkWidget *) parent);

	if (parent == NULL)
		parent = widget;

	state_type = gtk_widget_get_state ((GtkWidget *) parent);

	gcolor = &gtk_widget_get_style ((GtkWidget *) parent)->bg[state_type];

	nodoka_gdk_color_to_rgb (gcolor, &color->r, &color->g, &color->b);
}

NodokaStepper
nodoka_scrollbar_get_stepper (GtkWidget * widget, GdkRectangle * stepper)
{
	NodokaStepper value = NDK_STEPPER_UNKNOWN;
	GdkRectangle tmp;
	GdkRectangle check_rectangle;
	GtkOrientation orientation;
	GtkAllocation allocation;

	g_return_val_if_fail (GTK_IS_RANGE (widget), NDK_STEPPER_UNKNOWN);
	gtk_widget_get_allocation (widget, &allocation);

	check_rectangle.x = allocation.x;
	check_rectangle.y = allocation.y;
	check_rectangle.width = stepper->width;
	check_rectangle.height = stepper->height;

	if (GTK_IS_HSCROLLBAR (widget))
		orientation = GTK_ORIENTATION_HORIZONTAL;
	else
		orientation = GTK_ORIENTATION_VERTICAL;

	if (allocation.x == -1 && allocation.y == -1)
		return NDK_STEPPER_UNKNOWN;

	if (gdk_rectangle_intersect (stepper, &check_rectangle, &tmp))
		value = NDK_STEPPER_A;

	if (value == NDK_STEPPER_UNKNOWN)	/* Haven't found a match */
	{
		if (orientation == GTK_ORIENTATION_HORIZONTAL)
			check_rectangle.x = allocation.x + stepper->width;
		else
			check_rectangle.y = allocation.y + stepper->height;

		if (gdk_rectangle_intersect (stepper, &check_rectangle, &tmp))
			value = NDK_STEPPER_B;
	}

	if (value == NDK_STEPPER_UNKNOWN)	/* Still haven't found a match */
	{
		if (orientation == GTK_ORIENTATION_HORIZONTAL)
			check_rectangle.x =
				allocation.x + allocation.width -
				(stepper->width * 2);
		else
			check_rectangle.y =
				allocation.y + allocation.height -
				(stepper->height * 2);

		if (gdk_rectangle_intersect (stepper, &check_rectangle, &tmp))
			value = NDK_STEPPER_C;
	}

	if (value == NDK_STEPPER_UNKNOWN)	/* STILL haven't found a match */
	{
		if (orientation == GTK_ORIENTATION_HORIZONTAL)
			check_rectangle.x =
				allocation.x + allocation.width -
				stepper->width;
		else
			check_rectangle.y =
				allocation.y + allocation.height -
				stepper->height;

		if (gdk_rectangle_intersect (stepper, &check_rectangle, &tmp))
			value = NDK_STEPPER_D;
	}

	return value;
}

NodokaStepper
nodoka_scrollbar_visible_steppers (GtkWidget * widget)
{
	NodokaStepper steppers = 0;

	g_return_val_if_fail (GTK_IS_SCROLLBAR (widget), NDK_STEPPER_UNKNOWN);

	gboolean has_stepper_a;
	gboolean has_stepper_b;
	gboolean has_stepper_c;
	gboolean has_stepper_d;

	gtk_widget_style_get (widget, "has-backward-stepper", &has_stepper_a,
                                      "has-secondary-forward-stepper", &has_stepper_b,
                                      "has-secondary-backward-stepper", &has_stepper_c,
                                      "has-forward-stepper", &has_stepper_d, NULL);

	if (has_stepper_a)
		steppers |= NDK_STEPPER_A;

	if (has_stepper_b)
		steppers |= NDK_STEPPER_B;

	if (has_stepper_c)
		steppers |= NDK_STEPPER_C;

	if (has_stepper_d)
		steppers |= NDK_STEPPER_D;
		
	return (steppers);
}

NodokaJunction
nodoka_scrollbar_get_junction (GtkWidget * widget)
{
	GtkAdjustment *adj;
	NodokaJunction junction = NDK_JUNCTION_NONE;

	g_return_val_if_fail (GTK_IS_SCROLLBAR (widget), NDK_JUNCTION_NONE);

	adj = gtk_range_get_adjustment (GTK_RANGE (widget));

	gboolean has_stepper_a;
	gboolean has_stepper_b;
	gboolean has_stepper_c;
	gboolean has_stepper_d;

	gtk_widget_style_get (widget, "has-backward-stepper", &has_stepper_a,
                                      "has-secondary-forward-stepper", &has_stepper_b,
                                      "has-secondary-backward-stepper", &has_stepper_c,
                                      "has-forward-stepper", &has_stepper_d, NULL);

	if (gtk_adjustment_get_value (adj) <= gtk_adjustment_get_lower (adj) &&
		(has_stepper_a
		 || has_stepper_b))
	{
		junction |= NDK_JUNCTION_BEGIN;
	}

	if (gtk_adjustment_get_value (adj) >= gtk_adjustment_get_upper (adj) - gtk_adjustment_get_page_size (adj) &&
		(has_stepper_c
		 || has_stepper_d))
	{
		junction |= NDK_JUNCTION_END;
	}

	return junction;
}

gboolean
ndk_is_panel_widget (GtkWidget * widget)
{
	return widget && (strcmp (G_OBJECT_TYPE_NAME (widget), "PanelApplet") == 0
					  || strcmp (G_OBJECT_TYPE_NAME (widget),
								 "PanelWidget") == 0);
}

gboolean
ndk_object_is_a (const GObject * object, const gchar * type_name)
{
  gboolean result = FALSE;
 
  if ((object))
    {
      GType tmp = g_type_from_name (type_name);

      if (tmp)
	result = g_type_check_instance_is_a ((GTypeInstance *) object, tmp);
    }
 
  return result;
}

