/*
 * nodoka_style.c
 * This file is part of gtk-nodoka-engine
 *
 * Copyright (C) 2007, 2008 - Martin Sourada, Daniel Geiger
 *
 * gtk-nodoka-engine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * gtk-nodoka-engine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gtk-nodoka-engine; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */ 

#include <gtk/gtk.h>
#include <cairo.h>
#include <math.h>
#include <string.h>
#include <sys/time.h>

#include "nodoka_style.h"
#include "nodoka_rc_style.h"
#include "nodoka_draw.h"
#include "support.h"

/* #define DEBUG 1 */

#define SCALE_SIZE 5

#define DETAIL(xx)   ((detail) && (!strcmp(xx, detail)))
#define COMPARE_COLORS(a,b) (a.red == b.red && a.green == b.green \
                                            && a.blue == b.blue)

#define DRAW_ARGS    GtkStyle       *style, \
                     GdkWindow      *window, \
                     GtkStateType    state_type, \
                     GtkShadowType   shadow_type, \
                     GdkRectangle   *area, \
                     GtkWidget      *widget, \
                     const gchar    *detail, \
                     gint            x, \
                     gint            y, \
                     gint            width, \
                     gint            height

#ifdef HAVE_ANIMATION
#include "animation.h"
#endif

static GtkStyleClass *nodoka_parent_class;

static cairo_t *
nodoka_begin_paint (GdkDrawable * window, GdkRectangle * area)
{
	cairo_t *cr;

	g_return_val_if_fail (window != NULL, NULL);

	cr = (cairo_t *) gdk_cairo_create (window);
	cairo_set_line_width (cr, 1.0);

	if (area)
	{
		cairo_rectangle (cr, area->x, area->y, area->width,
						 area->height);
		cairo_clip_preserve (cr);
		cairo_new_path (cr);
	}

	return cr;
}

static void
nodoka_set_widget_parameters (const GtkWidget * widget,
							  const GtkStyle * style, GtkStateType state_type,
							  WidgetParameters * params)
{
	if (state_type == GTK_STATE_NORMAL && widget && GTK_IS_ENTRY (widget))
		state_type = gtk_widget_get_state ((GtkWidget *) widget);

	params->active = (state_type == GTK_STATE_ACTIVE);
	params->prelight = (state_type == GTK_STATE_PRELIGHT);
	params->disabled = (state_type == GTK_STATE_INSENSITIVE);
	params->state_type = (NodokaStateType) state_type;
	params->corners = NDK_CORNER_ALL;
	params->roundness = NODOKA_STYLE (style)->roundness;
	params->hilight_ratio = NODOKA_STYLE (style)->hilight_ratio;
	params->gradients = NODOKA_STYLE (style)->gradients;
	if (widget)
		params->ltr = !(nodoka_get_direction ((GtkWidget *) widget) == GTK_TEXT_DIR_RTL);
	else
		params->ltr = TRUE;

	params->focus = widget && gtk_widget_has_focus ((GtkWidget *) widget);
	params->is_default = widget && gtk_widget_has_default ((GtkWidget *) widget);

	if (!params->active && widget && GTK_IS_TOGGLE_BUTTON (widget))
		params->active =
			gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));

	params->xthickness = style->xthickness;
	params->ythickness = style->ythickness;

	/* I want to avoid to have to do this. I need it for GtkEntry, unless I
	 * find out why it doesn't behave the way I expect it to. */
	if (widget)
		nodoka_get_parent_bg (widget, &params->parentbg);
}

static void
nodoka_style_draw_flat_box (DRAW_ARGS)
{
	if (detail && state_type == GTK_STATE_SELECTED
		&& (!strncmp ("cell_even", detail, 9)
			|| !strncmp ("cell_odd", detail, 8)))
	{

		NodokaStyle *nodoka_style = NODOKA_STYLE (style);
		NodokaColors *colors = &nodoka_style->colors;
		cairo_t *cr = nodoka_begin_paint (window, area);

		nodoka_sanitize_size (window, &width, &height);

		WidgetParameters params;
		nodoka_set_widget_parameters (widget, style, state_type, &params);

		nodoka_draw_selected_cell (cr, colors, &params, x, y, width, height);
		cairo_destroy (cr);
	}
	else if (DETAIL ("tooltip"))
	{
		WidgetParameters params;
		nodoka_set_widget_parameters (widget, style, state_type, &params);

		NodokaStyle *nodoka_style = NODOKA_STYLE (style);
		NodokaColors *colors = &nodoka_style->colors;
		cairo_t *cr = nodoka_begin_paint (window, area);

		nodoka_sanitize_size (window, &width, &height);

		nodoka_draw_tooltip (cr, colors, &params, x, y, width, height);

		cairo_destroy (cr);
	}
	else if (DETAIL ("entry_bg") && !(widget && gtk_widget_get_parent (widget) && GTK_IS_TREE_VIEW (gtk_widget_get_parent (widget))))
	{
		/* This is a workaround to allow big roundness in entries  *
		 * which are drown twice and so parts of rounded edges get *
		 * cut.                                                    */
		
		NodokaStyle *nodoka_style = NODOKA_STYLE (style);
		cairo_t *cr = nodoka_begin_paint (window, area);

		nodoka_sanitize_size (window, &width, &height);
		
		WidgetParameters params;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		/* the drawing area is padded inside the widget, so we need *
		 * to extent it                                             */
		x -= style->xthickness;	
		y -= style->ythickness;	
		width += style->xthickness*2;
		height += style->ythickness*2;

		if (widget && gtk_widget_get_parent (widget)
			&& (NDK_IS_COMBO (gtk_widget_get_parent (widget))
				|| GTK_IS_COMBO_BOX_ENTRY (gtk_widget_get_parent (widget))
				|| GTK_IS_SPIN_BUTTON (widget)))
		{
			width += style->xthickness;

			if (!params.ltr)
			{
				x -= style->xthickness;
				params.corners = NDK_CORNER_TOPRIGHT | NDK_CORNER_BOTTOMRIGHT;
			}
			else
				params.corners = NDK_CORNER_TOPLEFT | NDK_CORNER_BOTTOMLEFT;

			if (GTK_IS_COMBO_BOX_ENTRY (gtk_widget_get_parent (widget)))
			{
				height += 1;
				/* GTKComboBoxEntry workaround */
				if (style->ythickness < 4)
				{
					y += style->ythickness;	
					height -= style->ythickness*2;		
					y -= 4;
					height += 8;
				}
			}
		}

		EntryParameters entry;

		entry.focus.inner = nodoka_style->focus_inner;
		entry.focus.fill = FALSE;

		if (params.roundness <= 0)
			params.corners = NDK_CORNER_NONE;

		nodoka_draw_entry (cr, &nodoka_style->colors, &params, &entry, x, y, width,
						   height);
		
		cairo_destroy (cr);
	}
	else if (DETAIL("checkbutton") || DETAIL("radiobutton"))
	{
		/* Don't draw background for checkbutton and radiobutton */
	}

	else
	{
		nodoka_parent_class->draw_flat_box (style, window, state_type,
											shadow_type, area, widget, detail,
											x, y, width, height);
	}
	/* Dotted listview */
	if (detail
		&& (!strncmp ("cell_even", detail, 9)
			|| !strncmp ("cell_odd", detail, 8)))
	{
		NodokaStyle *nodoka_style = NODOKA_STYLE (style);
		if (nodoka_style->listviewstyle > 0)
		{
			NodokaColors *colors = &nodoka_style->colors;
			cairo_t *cr = nodoka_begin_paint (window, area);

			cairo_translate (cr, x, y);

			int i;
			int pos = 1;
			if (nodoka_style->listviewheaderstyle != 1)
				pos = 2;

			cairo_set_source_rgba (cr, colors->text[GTK_STATE_NORMAL].r,
								   colors->text[GTK_STATE_NORMAL].g,
								   colors->text[GTK_STATE_NORMAL].b, 0.42);
			for (i = 2; i < height; i += 4)
			{
				cairo_rectangle (cr, -pos, i, 1, 1);
				cairo_fill (cr);
			}

			cairo_destroy (cr);
		}
	}
}

static void
nodoka_style_draw_shadow (DRAW_ARGS)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;
	cairo_t *cr = nodoka_begin_paint (window, area);

	nodoka_sanitize_size (window, &width, &height);

	if (DETAIL ("entry")
		&& !(widget && gtk_widget_get_parent (widget) && GTK_IS_TREE_VIEW (gtk_widget_get_parent (widget))))
	{
		WidgetParameters params;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		if (widget && gtk_widget_get_parent (widget)
			&& (NDK_IS_COMBO (gtk_widget_get_parent (widget))
				|| GTK_IS_COMBO_BOX_ENTRY (gtk_widget_get_parent (widget))
				|| GTK_IS_SPIN_BUTTON (widget)))
		{
			width += style->xthickness;

			if (!params.ltr)
			{
				x -= style->xthickness;
				params.corners = NDK_CORNER_TOPRIGHT | NDK_CORNER_BOTTOMRIGHT;
			}
			else
				params.corners = NDK_CORNER_TOPLEFT | NDK_CORNER_BOTTOMLEFT;
		}

		EntryParameters entry;

		entry.focus.inner = nodoka_style->focus_inner;
		entry.focus.fill = FALSE;

		if (params.roundness <= 0)
			params.corners = NDK_CORNER_NONE;

                if (!widget || !g_object_get_data (G_OBJECT (widget), "transparent-bg-hint"))
                {
                	cairo_rectangle(cr, 0, 0, width, height);
                	cairo_set_source_rgb(cr, params.parentbg.r, params.parentbg.g, params.parentbg.b);
            	        cairo_fill(cr);
                }
                
		nodoka_draw_entry (cr, &nodoka_style->colors, &params, &entry, x, y, width,
						   height);
	}
	else if (DETAIL ("frame") && widget && gtk_widget_get_parent (widget) && GTK_IS_STATUSBAR (gtk_widget_get_parent (widget)))
	{
		WidgetParameters params;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		gtk_style_apply_default_background (style, window, TRUE, state_type,
											area, x, y, width, height);

		nodoka_draw_statusbar (cr, colors, &params, x, y, width, height);
	}
	else if (DETAIL ("frame"))
	{
		WidgetParameters params;
		FrameParameters frame;
		frame.shadow = shadow_type;
		frame.gap_x = -1;		/* No gap will be drawn */
		frame.border = &colors->shade[4];
		frame.draw_fill = FALSE;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		params.corners = NDK_CORNER_NONE;

		if (widget
			&& !g_str_equal ("XfcePanelWindow",
							 gtk_widget_get_name (gtk_widget_get_toplevel
												  (widget))))
			nodoka_draw_frame (cr, colors, &params, &frame, x, y, width,
							   height);
	}
	else if (DETAIL ("scrolled_window") || DETAIL ("viewport")
			 || detail == NULL || shadow_type != GTK_SHADOW_NONE)
	{
		NodokaRGB *border = (NodokaRGB *) & colors->shade[5];
		cairo_rectangle (cr, x + 0.5, y + 0.5, width - 1, height - 1);
		cairo_set_source_rgb (cr, border->r, border->g, border->b);
		cairo_set_line_width (cr, 1);
		cairo_stroke (cr);
	}
	cairo_destroy (cr);
}

static void
nodoka_style_draw_box_gap (DRAW_ARGS, GtkPositionType gap_side, gint gap_x,
						   gint gap_width)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;
	cairo_t *cr = nodoka_begin_paint (window, area);

	if (DETAIL ("notebook"))
	{
		WidgetParameters params;
		FrameParameters frame;

		frame.shadow = NDK_SHADOW_FLAT;
		frame.gap_side = gap_side;
		frame.gap_x = gap_x;
		frame.gap_width = gap_width;
		frame.border = &colors->shade[5];
		frame.draw_fill = TRUE;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		/* Somehow needed to fix full-width notebook problem */
		gint window_width;
		gdk_drawable_get_size (window, &window_width, NULL);
		if ((nodoka_style->roundness == 0) || window_width == width)
		{
			params.corners = NDK_CORNER_NONE;
		}
		else
		{
			if (frame.gap_side == NDK_GAP_TOP)
			{
				params.corners =
					NDK_CORNER_BOTTOMRIGHT | NDK_CORNER_BOTTOMLEFT;

				if (frame.gap_width != width)
				{
					if (params.ltr) 
						params.corners |= NDK_CORNER_TOPRIGHT;
					else
						params.corners |= NDK_CORNER_TOPLEFT;
				}
			}
			else if (frame.gap_side == NDK_GAP_BOTTOM)
			{
				params.corners =
					NDK_CORNER_TOPRIGHT | NDK_CORNER_TOPLEFT;

				if (frame.gap_width != width)
				{
					if (params.ltr) 
						params.corners |= NDK_CORNER_BOTTOMRIGHT;
					else
						params.corners |= NDK_CORNER_BOTTOMLEFT;
				}
			}
			else if (frame.gap_side == NDK_GAP_LEFT)
			{
				params.corners =
					NDK_CORNER_BOTTOMRIGHT | NDK_CORNER_TOPRIGHT;

				if (frame.gap_width != height)
					params.corners |= NDK_CORNER_BOTTOMLEFT;
			}
			else
			{
				params.corners =
					NDK_CORNER_BOTTOMLEFT | NDK_CORNER_TOPLEFT;

				if (frame.gap_width != height)
					params.corners |= NDK_CORNER_BOTTOMRIGHT;
			}
		}

		nodoka_draw_frame (cr, colors, &params, &frame, x, y, width, height);
	}
	else
	{
		nodoka_parent_class->draw_box_gap (style, window, state_type,
										   shadow_type, area, widget, detail, x,
										   y, width, height, gap_side, gap_x,
										   gap_width);
	}

	cairo_destroy (cr);
}

static void
nodoka_style_draw_extension (DRAW_ARGS, GtkPositionType gap_side)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;
	cairo_t *cr = nodoka_begin_paint (window, area);

	if (DETAIL ("tab"))
	{
		WidgetParameters params;
		TabParameters tab;

		nodoka_set_widget_parameters (widget, style, state_type, &params);
		params.corners = NDK_CORNER_NONE;

		tab.gap_side = (NodokaGapSide) gap_side;

		if (nodoka_style->roundness > 0)
		{
			if (gap_side == GTK_POS_BOTTOM)
				params.corners = NDK_CORNER_TOPLEFT | NDK_CORNER_TOPRIGHT;
			else if (gap_side == GTK_POS_TOP)
				params.corners = NDK_CORNER_BOTTOMLEFT | NDK_CORNER_BOTTOMRIGHT;
			else if (gap_side == GTK_POS_LEFT)
				params.corners = NDK_CORNER_TOPRIGHT | NDK_CORNER_BOTTOMRIGHT;
			else if (gap_side == GTK_POS_RIGHT)
				params.corners = NDK_CORNER_TOPLEFT | NDK_CORNER_BOTTOMLEFT;
		}
		else
			params.corners = NDK_CORNER_NONE;

		nodoka_draw_tab (cr, colors, &params, &tab, x, y, width, height);
	}
	else
	{
		nodoka_parent_class->draw_extension (style, window, state_type,
											 shadow_type, area, widget, detail,
											 x, y, width, height, gap_side);
	}

	cairo_destroy (cr);
}

static void
nodoka_style_draw_handle (DRAW_ARGS, GtkOrientation orientation)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;
	cairo_t *cr = nodoka_begin_paint (window, area);
	gboolean is_horizontal;

	nodoka_sanitize_size (window, &width, &height);

	/* Evil hack to work around broken orientation for toolbars */
	is_horizontal = (width > height);

	if (DETAIL ("handlebox"))
	{
		WidgetParameters params;
		HandleParameters handle;

		nodoka_set_widget_parameters (widget, style, state_type, &params);
		handle.type = NDK_HANDLE_TOOLBAR;
		handle.horizontal = is_horizontal;

		if (widget && GTK_IS_TOOLBAR (widget) && shadow_type != GTK_SHADOW_NONE)
		{
			ToolbarParameters toolbar;
			toolbar.style = nodoka_style->toolbarstyle;
			toolbar.horizontal = is_horizontal;

			cairo_save (cr);
			nodoka_draw_toolbar (cr, colors, &params, &toolbar, x, y, width, height);
			cairo_restore (cr);
		}

		nodoka_draw_handle (cr, colors, &params, &handle, x, y, width, height);
	}
	else if (DETAIL ("paned"))
	{
		WidgetParameters params;
		HandleParameters handle;

		nodoka_set_widget_parameters (widget, style, state_type, &params);
		handle.type = NDK_HANDLE_SPLITTER;
		handle.horizontal = orientation == GTK_ORIENTATION_HORIZONTAL;

		nodoka_draw_handle (cr, colors, &params, &handle, x, y, width, height);
	}
	else
	{
		WidgetParameters params;
		HandleParameters handle;

		nodoka_set_widget_parameters (widget, style, state_type, &params);
		handle.type = NDK_HANDLE_TOOLBAR;
		handle.horizontal = is_horizontal;

		if (widget && GTK_IS_TOOLBAR (widget) && shadow_type != GTK_SHADOW_NONE)
		{
			ToolbarParameters toolbar;
			toolbar.style = nodoka_style->toolbarstyle;
			toolbar.horizontal = is_horizontal;

			cairo_save (cr);
			nodoka_draw_toolbar (cr, colors, &params, &toolbar, x, y, width, height);
			cairo_restore (cr);
		}

		nodoka_draw_handle (cr, colors, &params, &handle, x, y, width, height);
	}

	cairo_destroy (cr);
}

static void
nodoka_style_draw_box (DRAW_ARGS)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	const NodokaColors *colors = &nodoka_style->colors;
        if (area && (DETAIL ("hscrollbar") || DETAIL ("vscrollbar") || DETAIL ("stepper")))
        {
		area->x -= 5;
		area->y -= 5;
		area->height += 10;
		area->width += 10;
	}
	cairo_t *cr = nodoka_begin_paint (window, area);

	if ((width == -1) && (height == -1))
		gdk_drawable_get_size (window, &width, &height);
	else if (width == -1)
		gdk_drawable_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_drawable_get_size (window, NULL, &height);

	if (DETAIL ("menubar")
		&& !(widget && gtk_widget_get_parent (widget) && (ndk_is_panel_widget (gtk_widget_get_parent (widget)))))
	{
		WidgetParameters params;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		MenuBarParameters menubar;
		menubar.style = nodoka_style->menubarstyle;

		nodoka_draw_menubar (cr, colors, &params, &menubar, x, y, width, height);
	}
	else if (DETAIL ("button") && widget && gtk_widget_get_parent (widget)
			 && (GTK_IS_TREE_VIEW (gtk_widget_get_parent (widget))
			 	 || NDK_IS_CLIST (gtk_widget_get_parent (widget))
				 || NDK_IS_ETREE (gtk_widget_get_parent (widget))
				 || NDK_IS_ETABLE (gtk_widget_get_parent (widget))))
	{
		WidgetParameters params;
		ListViewHeaderParameters header;
		header.style = nodoka_style->listviewheaderstyle;

		gint columns, column_index;
		gboolean resizable = TRUE;

		nodoka_set_widget_parameters (widget, style, state_type, &params);
		params.corners = NDK_CORNER_NONE;

		if (widget && gtk_widget_get_parent (widget) && GTK_IS_TREE_VIEW (gtk_widget_get_parent (widget)))
		{
			nodoka_gtk_treeview_get_header_index (GTK_TREE_VIEW
						  (gtk_widget_get_parent (widget)), widget,
						  &column_index, &columns,
						  &resizable);
		}
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION < 90 && !defined(GTK_DISABLE_DEPRECATED)
		else if (widget && gtk_widget_get_parent (widget) && NDK_IS_CLIST (gtk_widget_get_parent (widget)))
		{
			nodoka_gtk_clist_get_header_index (GTK_CLIST (gtk_widget_get_parent (widget)),
						   widget, &column_index, &columns);
		}
#endif
 
		header.resizable = resizable;

		if (column_index == 0)
			header.order = params.ltr ? NDK_ORDER_FIRST : NDK_ORDER_LAST;
		else if (column_index == columns - 1)
			header.order = params.ltr ? NDK_ORDER_LAST : NDK_ORDER_FIRST;
		else
			header.order = NDK_ORDER_MIDDLE;

		gtk_style_apply_default_background (style, window, FALSE, state_type,
						area, x, y, width, height);

		nodoka_draw_list_view_header (cr, colors, &params, &header, x, y, width,
									  height);
	}
        else if (DETAIL ("buttondefault"))
        {
	}
	else if (DETAIL ("button"))
	{
		WidgetParameters params;
		nodoka_set_widget_parameters (widget, style, state_type, &params);

		boolean horizontal = TRUE;
		if (((float) width / height < 0.5))
			horizontal = FALSE;

		boolean shadows = TRUE;
		if ((widget && gtk_widget_get_parent (widget)
			 && (NDK_IS_COMBO (gtk_widget_get_parent (widget))
				|| GTK_IS_COMBO_BOX_ENTRY (gtk_widget_get_parent (widget)))))
		{

			if (nodoka_style->roundness > 0)
			{
				if (params.ltr)
					params.corners = NDK_CORNER_TOPRIGHT | NDK_CORNER_BOTTOMRIGHT;
				else
					params.corners = NDK_CORNER_TOPLEFT | NDK_CORNER_BOTTOMLEFT;
			}
			else
				params.corners = NDK_CORNER_NONE;

			/* Seriously, why can't non-gtk-apps at least try to be decent citizens?
			 * Take OpenOffice.org 1.9 for example. The morons responsible
			 * for this utter piece of crap gave the clip size wrong values! :'(  */
			if (params.xthickness > 2)
			{
				if (params.ltr)
					x--;
				width++;
			}
			if (!params.active)
				shadows = FALSE;
		}
		else if (nodoka_style->roundness > 0)
			params.corners = NDK_CORNER_ALL;
		else
			params.corners = NDK_CORNER_NONE;

		if (widget && GTK_IS_TOGGLE_BUTTON (widget)
			&& gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
			params.active = TRUE;

		ButtonParameters button;
		button.shadows = shadows;
		button.horizontal = horizontal;
		button.focus.inner = nodoka_style->focus_inner;
		button.focus.fill = nodoka_style->focus_fill;

		nodoka_draw_button (cr, &nodoka_style->colors, &params, &button, x, y, width, height);
	}
	else if (DETAIL ("spinbutton_up") || DETAIL ("spinbutton_down"))
	{
		WidgetParameters params;
		
		nodoka_set_widget_parameters (widget, style, state_type, &params);
		params.state_type = (NodokaStateType) state_type;
		params.disabled = (state_type == GTK_STATE_INSENSITIVE);
		params.focus = FALSE;


	  	/* Only draw button as disabled if entire widget is disabled */
	  	if (params.disabled && !(gtk_widget_get_state ((GtkWidget *) widget) == GTK_STATE_INSENSITIVE)) 
		{
			params.disabled = FALSE;
			params.state_type = GTK_STATE_NORMAL;
		}

		if (nodoka_style->roundness > 0)
		{
			if (params.ltr)
				params.corners = NDK_CORNER_TOPRIGHT | NDK_CORNER_BOTTOMRIGHT;
			else
				params.corners = NDK_CORNER_TOPLEFT | NDK_CORNER_BOTTOMLEFT;
		}
		else
			params.corners = NDK_CORNER_NONE;

		if (style->xthickness >= 3)
		{
			if (params.ltr)
				x--;
			width++;
		}
		
		/* Fake real transparancy */
		cairo_set_source_rgb (cr, params.parentbg.r, params.parentbg.g, params.parentbg.b);
		cairo_rectangle (cr, x, y, width, height);
		cairo_fill_preserve (cr);
		cairo_clip (cr);

		ButtonParameters button;
		button.shadows = FALSE;
		button.horizontal = TRUE;
		button.focus.inner = nodoka_style->focus_inner;
		button.focus.fill = nodoka_style->focus_fill;

		if (DETAIL ("spinbutton_up"))
			nodoka_draw_button (cr, &nodoka_style->colors, &params, &button, x,
								 y, width, height * 2);
		else
			nodoka_draw_button (cr, &nodoka_style->colors, &params, &button, x,
								 y - height, width, height * 2);
			
	}
	else if (DETAIL ("spinbutton"))	{ }
	else if (detail && g_str_has_prefix (detail, "trough") && widget && GTK_IS_SCALE (widget))
	{
		WidgetParameters params;
		SliderParameters slider;

		nodoka_set_widget_parameters (widget, style, state_type, &params);
		params.corners = NDK_CORNER_NONE;

		slider.lower = DETAIL ("trough-lower");
		slider.fill_level = DETAIL ("trough-fill-level") || DETAIL ("trough-fill-level-full");

		slider.horizontal =
			(GTK_IS_HSCALE (widget));

		slider.focus.inner = nodoka_style->focus_inner;
		slider.focus.fill = nodoka_style->focus_fill;

		nodoka_draw_scale_trough (cr, &nodoka_style->colors, &params, &slider,
								  x, y, width, height);
	}
	else if (DETAIL ("trough") && widget && GTK_IS_PROGRESS_BAR (widget))
	{
		WidgetParameters params;

		ProgressBarParameters progressbar;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		progressbar.orientation = gtk_progress_bar_get_orientation (GTK_PROGRESS_BAR (widget));

		if (params.roundness > 0)
 			params.corners = NDK_CORNER_ALL;
		else
			params.corners = NDK_CORNER_NONE;

                if (!widget || !g_object_get_data (G_OBJECT (widget), "transparent-bg-hint"))
                {
                	cairo_rectangle(cr, 0, 0, width, height);
                	cairo_set_source_rgb(cr, params.parentbg.r, params.parentbg.g, params.parentbg.b);
            	        cairo_fill(cr);
                }

		nodoka_draw_progressbar_trough (cr, colors, &params, &progressbar, x, y, width,
										height);
	}
	else if (DETAIL ("trough") && widget
			 && (GTK_IS_VSCROLLBAR (widget) || GTK_IS_HSCROLLBAR (widget)))
	{
		WidgetParameters params;
		ScrollBarParameters scrollbar;
		GtkAllocation allocation;

		gtk_widget_get_allocation (widget, &allocation);

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		/* Workaround some gecko < 1.9 crap */
		if ((allocation.x == -1) || (allocation.y == -1))
			params.roundness = 0;

		if (params.roundness > 0)
			params.roundness = 6;

		scrollbar.horizontal = GTK_IS_HSCROLLBAR (widget);
		scrollbar.junction = nodoka_scrollbar_get_junction (widget);
		scrollbar.steppers = nodoka_scrollbar_visible_steppers (widget);

		params.corners = NDK_CORNER_NONE;
		if (params.roundness > 0)
		{
			if (scrollbar.horizontal && width < 14)
				params.roundness = width / 2 - 1;
			else if (height < 14)
				params.roundness = height / 2 - 1;
			
			if (!(scrollbar.steppers & (NDK_STEPPER_A | NDK_STEPPER_B)))
				params.corners |= (NDK_CORNER_TOPLEFT | NDK_CORNER_TOPRIGHT);
			if (!(scrollbar.steppers & (NDK_STEPPER_C | NDK_STEPPER_D)))
				params.corners |= (NDK_CORNER_BOTTOMLEFT | NDK_CORNER_BOTTOMRIGHT);
		}


		nodoka_draw_scrollbar_trough (cr, colors, &params, &scrollbar, x, y,
									  width, height);

		/* Need to draw steppers background due to steppers design */
		if ((GTK_IS_RANGE(widget)) && (params.roundness > 0))
		{
			ScrollBarStepperParameters stepper;

			params.state_type = GTK_STATE_NORMAL;

			int stepper_size;

			gtk_widget_style_get (widget, "stepper-size", &stepper_size,
					      NULL);

			int sx = x;
			int sy = y;
			int sw0 = (scrollbar.horizontal ? stepper_size : width);
			int sh0 = (scrollbar.horizontal ? height : stepper_size);

			/*Draw steppers A and B*/
			int sw = (scrollbar.horizontal ? 0 : sw0);
			int sh = (scrollbar.horizontal ? sh0 : 0);

			if (scrollbar.steppers & (NDK_STEPPER_A | NDK_STEPPER_B))
			{
				params.corners = NDK_CORNER_BOTTOMLEFT | NDK_CORNER_BOTTOMRIGHT;
			}	

			if (scrollbar.steppers & NDK_STEPPER_A)
			{
				sw = sw0;
				sh = sh0;
			}
			if (scrollbar.steppers & NDK_STEPPER_B)
			{
				if (scrollbar.horizontal)
					sw += sw0;
				else
					sh += sh0;
			}
			stepper.stepper = NDK_STEPPER_B;

			nodoka_draw_scrollbar_stepper (cr, colors, &params, &scrollbar,
										   &stepper, sx, sy, sw, sh);

			/*Draw steppers C and D*/
			if (scrollbar.steppers & (NDK_STEPPER_C | NDK_STEPPER_D))
			{
				params.corners = NDK_CORNER_TOPLEFT | NDK_CORNER_TOPRIGHT;
			}	

			sx = x + width - (scrollbar.horizontal ? 0 : sw0);
			sy = y + height - (scrollbar.horizontal ? sh0 : 0);
			sw = (scrollbar.horizontal ? 0 : sw0);
			sh = (scrollbar.horizontal ? sh0 : 0);
			if (scrollbar.steppers & NDK_STEPPER_D)
			{
				sw = sw0;
				sh = sh0;
				if (scrollbar.horizontal)
					sx -= sw0;
				else
					sy -= sh0;
			}
			if (scrollbar.steppers & NDK_STEPPER_C)
			{
				if (scrollbar.horizontal)
				{
					sx -= sw0;
					sw += sw0;
				}
				else
				{
					sy -= sh0;
					sh += sh0;
				}
			}
			stepper.stepper = NDK_STEPPER_C;

			nodoka_draw_scrollbar_stepper (cr, colors, &params, &scrollbar,
										   &stepper, sx, sy, sw, sh);
		}

	}
	else if (DETAIL ("bar"))
	{
		WidgetParameters params;
		ProgressBarParameters progressbar;
		gdouble elapsed = 0.0;

#ifdef HAVE_ANIMATION
		if (nodoka_style->animation && NDK_IS_PROGRESS_BAR (widget))
		{
			gboolean activity_mode = GTK_PROGRESS (widget)->activity_mode;

			if (!activity_mode)
				nodoka_animation_progressbar_add ((gpointer) widget);
		}

		elapsed = nodoka_animation_elapsed (widget);
#endif

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		if (widget && GTK_IS_PROGRESS_BAR (widget))
			progressbar.orientation =
				gtk_progress_bar_get_orientation (GTK_PROGRESS_BAR (widget));
		else
			progressbar.orientation = NDK_ORIENTATION_LEFT_TO_RIGHT;

		if (!params.ltr)
		{
			if (progressbar.orientation == NDK_ORIENTATION_LEFT_TO_RIGHT)
				progressbar.orientation = NDK_ORIENTATION_RIGHT_TO_LEFT;
			else if (progressbar.orientation == NDK_ORIENTATION_RIGHT_TO_LEFT)
				progressbar.orientation = NDK_ORIENTATION_LEFT_TO_RIGHT;
		}


		progressbar.stripes = nodoka_style->stripes;

		/* The x-1 and width+2 are to make the fill cover the left and
		 * right-hand sides of the trough box */

#ifndef HAVE_ANIMATIONTOLEFT
		progressbar.offset = 10 - (int) (elapsed * 10.0) % 10;
#else
		progressbar.offset = 10 + (int) (elapsed * 10.0) % 10;
#endif
		nodoka_draw_progressbar_fill (cr, colors, &params, &progressbar, x - 1, y, width + 2, height);
	}
	else if (DETAIL ("entry-progress"))
	{
		WidgetParameters params;
		EntryProgressParameters progress;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		progress.max_size_known = FALSE;
		progress.max_size.x = 0;
		progress.max_size.y = 0;
		progress.max_size.width = 0;
		progress.max_size.height = 0;
		progress.border.left = style->xthickness;
		progress.border.right = style->xthickness;
		progress.border.top = style->ythickness;
		progress.border.bottom = style->ythickness;

		if (widget && GTK_IS_ENTRY (widget))
		{
			GtkBorder *border;
			/* Try to retrieve the style property. */
			gtk_widget_style_get (widget,
			                      "progress-border", &border,
			                      NULL);

			if (border)
			{
				progress.border = *border;
				gtk_border_free (border);
			}

			/* We got an entry, but well, we may not be drawing to
			 * this particular widget ... it may not even be realized.
			 * Also, we need to be drawing on a window obviously ... */
#if (GTK_MAJOR_VERSION >= 3) || (GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION >= 20)
			if (gtk_widget_get_realized (widget) &&
#else
			if (GTK_WIDGET_REALIZED (widget) &&
#endif
			    window && GDK_IS_WINDOW (window) &&
			    gdk_window_is_visible (gtk_widget_get_window (widget)))
			{
				/* Assumptions done by this code:
				 *  - GtkEntry has some nested windows.
				 *  - widget->window is the entries window
				 *  - widget->window is the size of the entry part
				 *    (and not larger)
				 *  - only one layer of subwindows
				 * These should be true with any GTK+ 2.x version.
				 */

				if (gtk_widget_get_window (widget) == window)
				{
					progress.max_size_known = TRUE;
					gdk_drawable_get_size (gtk_widget_get_window (widget),
					                       &progress.max_size.width,
					                       &progress.max_size.height);

				}
				else
				{
					GdkWindow *parent;
					parent = gdk_window_get_parent (window);
					if (gtk_widget_get_window (widget) == parent)
					{
						gint pos_x, pos_y;
						/* widget->window is the parent window
						 * of the current one. This means we can
						 * calculate the correct offsets. */
						gdk_window_get_position (window, &pos_x, &pos_y);
						progress.max_size.x = -pos_x;
						progress.max_size.y = -pos_y;

						progress.max_size_known = TRUE;
						gdk_drawable_get_size (gtk_widget_get_window (widget),
						                       &progress.max_size.width,
						                       &progress.max_size.height);
					} /* Nothing we can do in this case ... */
				}

				/* Now, one more thing needs to be done. If interior-focus
				 * is off, then the entry may be a bit smaller. */
				if (progress.max_size_known && gtk_widget_has_focus (widget))
				{
					gboolean interior_focus = TRUE;
					gint focus_line_width = 1;

					gtk_widget_style_get (widget,
					                      "interior-focus", &interior_focus,
					                      "focus-line-width", &focus_line_width,
					                      NULL);

					if (!interior_focus)
					{
						progress.max_size.x += focus_line_width;
						progress.max_size.y += focus_line_width;
						progress.max_size.width -= 2*focus_line_width;
						progress.max_size.height -= 2*focus_line_width;
					}
				}

				if (progress.max_size_known)
				{
					progress.max_size.x += progress.border.left;
					progress.max_size.y += progress.border.top;
					progress.max_size.width -= progress.border.left + progress.border.right;
					progress.max_size.height -= progress.border.top + progress.border.bottom;

					/* Now test that max_size.height == height, if that
					 * fails, something has gone wrong ... so then throw away
					 * the max_size information. */
					if (progress.max_size.height != height)
					{
						progress.max_size_known = FALSE;
						progress.max_size.x = 0;
						progress.max_size.y = 0;
						progress.max_size.width = 0;
						progress.max_size.height = 0;
					}
				}
			}
		}
		
		nodoka_draw_entry_progress (cr, colors, &params, &progress, x - 1, y, width + 2, height);

	}
	else if (DETAIL ("hscale") || DETAIL ("vscale"))
	{
		WidgetParameters params;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		boolean horizontal = TRUE;
		if (DETAIL ("vscale"))
			horizontal = FALSE;

		if (nodoka_style->roundness > 0)
			params.corners = NDK_CORNER_ALL;
		else
			params.corners = NDK_CORNER_NONE;

		SliderParameters slider;
		//slider.inverted = FALSE;
		//slider.fill_size = 0;
		slider.horizontal = horizontal;
		slider.focus.inner = nodoka_style->focus_inner;
		slider.focus.fill = nodoka_style->focus_fill;


		nodoka_draw_scale_slider (cr, &nodoka_style->colors, &params, &slider, x, y, width, height);
	}
	else if (DETAIL ("optionmenu"))
	{
		WidgetParameters params;
		OptionMenuParameters optionmenu;

		GtkRequisition indicator_size;
		GtkBorder indicator_spacing;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		optionmenu.focus.inner=nodoka_style->focus_inner;
		optionmenu.focus.fill=nodoka_style->focus_fill;

		if (nodoka_style->roundness > 0)
			params.corners = NDK_CORNER_ALL;
		else
			params.corners = NDK_CORNER_NONE;

		nodoka_option_menu_get_props (widget, &indicator_size,
									  &indicator_spacing);

		if (widget && nodoka_get_direction (widget) == GTK_TEXT_DIR_RTL)
			optionmenu.linepos =
				(indicator_size.width + indicator_spacing.left +
				 indicator_spacing.right) + style->xthickness;
		else
			optionmenu.linepos =
				width - (indicator_size.width + indicator_spacing.left +
						 indicator_spacing.right) - style->xthickness;

		nodoka_draw_optionmenu (cr, &nodoka_style->colors, &params, &optionmenu, x, y, width,
								height);
	}
	else if (DETAIL ("menuitem"))
	{
		WidgetParameters params;
		nodoka_set_widget_parameters (widget, style, state_type, &params);

		params.corners = NDK_CORNER_NONE;

		if (widget && gtk_widget_get_parent (widget) && !(GTK_IS_MENU_BAR (gtk_widget_get_parent (widget))))
		{
			nodoka_draw_menuitem (cr, colors, &params, x, y, width, height);
		}

		if (widget && gtk_widget_get_parent (widget) && GTK_IS_MENU_BAR (gtk_widget_get_parent (widget)))
		{
			params.active = TRUE;
			params.prelight = TRUE;
			params.focus = TRUE;
			params.state_type = NDK_STATE_SELECTED;
			params.xthickness = 2;
			params.ythickness = 2;

			if (nodoka_style->roundness > 0)
				params.corners = NDK_CORNER_TOPRIGHT | NDK_CORNER_TOPLEFT;
			else
				params.corners = NDK_CORNER_NONE;
			/* Tab-like menubaritem */
			nodoka_draw_menuitem (cr, colors, &params, x, y, width, height + 1);
		}
	}
	else if (DETAIL ("hscrollbar") || DETAIL ("vscrollbar") || DETAIL ("slider")
			 || DETAIL ("stepper"))
	{
		WidgetParameters params;
		ScrollBarParameters scrollbar;

		GtkAllocation allocation;
		gtk_widget_get_allocation (widget, &allocation);

		nodoka_set_widget_parameters (widget, style, state_type, &params);
		
		/* Workaround some gecko < 1.9 crap */
		if ((allocation.x == -1) || (allocation.y == -1))
			params.roundness = 0;
		
		if (params.roundness > 0)
			params.roundness = 6;
		params.corners = NDK_CORNER_NONE;
		scrollbar.has_color = FALSE;
		scrollbar.horizontal = GTK_IS_HSCROLLBAR (widget);
		scrollbar.junction = nodoka_scrollbar_get_junction (widget);
		scrollbar.steppers = nodoka_scrollbar_visible_steppers (widget);
		scrollbar.style = nodoka_style->scrollbarstyle;
	
		if (params.roundness > 0)
		{
			if (scrollbar.horizontal)
			{
				if (height < 14) 
					params.roundness = height / 2 - 1;
			}
			else
			{
				if (width < 14)
					params.roundness = width / 2 - 1;
			}
		}

		if (nodoka_style->has_scrollbar_color)
		{
			nodoka_gdk_color_to_rgb (&nodoka_style->scrollbar_color,
									 &scrollbar.color.r, &scrollbar.color.g,
									 &scrollbar.color.b);
			scrollbar.has_color = TRUE;
		}

		if (DETAIL ("slider"))
		{
			if (params.roundness >= 1)
				params.corners = NDK_CORNER_ALL;
			else
				params.corners = NDK_CORNER_NONE;

			nodoka_draw_scrollbar_slider (cr, colors, &params, &scrollbar, x, y,
										  width, height);
		}
		else
		{

			ScrollBarStepperParameters stepper;
			GdkRectangle this_rectangle = {x, y, width, height};

			stepper.stepper =
				nodoka_scrollbar_get_stepper (widget, &this_rectangle);

			
			/* Set inner corners to be rounded, vscrollbar is base */
			if (params.roundness > 0)
			{
				if (((stepper.stepper == NDK_STEPPER_A) && (!(scrollbar.steppers & NDK_STEPPER_B))) || (stepper.stepper == NDK_STEPPER_B))
				{
					params.corners = NDK_CORNER_BOTTOMLEFT | NDK_CORNER_BOTTOMRIGHT;
				}	
				else if  (((stepper.stepper == NDK_STEPPER_D) && (!(scrollbar.steppers & NDK_STEPPER_C))) || (stepper.stepper == NDK_STEPPER_C))
				{
					params.corners = NDK_CORNER_TOPLEFT | NDK_CORNER_TOPRIGHT;
				}	
			}
			else
				params.corners = NDK_CORNER_NONE;

			/* When secondary, blend border with primary */			
			if (stepper.stepper == NDK_STEPPER_B)
			{
				if (scrollbar.horizontal)
				{
					x -= 1;
					width += 1;
				}
				else
				{
					y -= 1;
					height += 1;
				}
			}
			else if (stepper.stepper == NDK_STEPPER_C)
			{
				if (scrollbar.horizontal)
					width += 1;
				else
					height += 1;
			}

			nodoka_draw_scrollbar_stepper (cr, colors, &params, &scrollbar,
										   &stepper, x, y, width, height);
		}
	}
	else if (DETAIL ("toolbar") || DETAIL ("handlebox_bin")
			 || DETAIL ("dockitem_bin"))
	{
		WidgetParameters params;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		/* Workaround broken toolbar orientation */
		boolean is_horizontal = (width*2 > height);

		ToolbarParameters toolbar;
		toolbar.horizontal = is_horizontal;
		toolbar.style = nodoka_style->toolbarstyle;

		if (shadow_type != GTK_SHADOW_NONE)
			nodoka_draw_toolbar (cr, colors, &params, &toolbar, x, y, width, height);
	}
	else if (DETAIL ("trough"))
	{
	}
	else if (DETAIL ("menu"))
	{
		WidgetParameters params;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		nodoka_draw_menu_frame (cr, colors, &params, x, y, width, height);
	}
	else
	{
		nodoka_parent_class->draw_box (style, window, state_type, shadow_type,
									   area, widget, detail, x, y, width,
									   height);
	}

	cairo_destroy (cr);
}

static void
nodoka_style_draw_slider (DRAW_ARGS, GtkOrientation orientation)
{
	if (DETAIL ("hscale") || DETAIL ("vscale"))
	{
		nodoka_style_draw_box (style, window, state_type, shadow_type, area,
							   widget, detail, x, y, width, height);
	}
	else
		nodoka_parent_class->draw_slider (style, window, state_type,
										  shadow_type, area, widget, detail, x,
										  y, width, height, orientation);
}

static void
nodoka_style_draw_option (DRAW_ARGS)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;
	cairo_t *cr;
	WidgetParameters params;

	cr = nodoka_begin_paint (window, area);

	nodoka_sanitize_size (window, &width, &height);

	/* Treeview Workarounds */	
	nodoka_set_widget_parameters (widget, style, state_type, &params);
	if (!(GTK_IS_CHECK_BUTTON (widget)))
		params.focus = FALSE;

	if (DETAIL ("cellradio") && !params.disabled && widget && gtk_widget_get_parent (widget))
	{
		params.disabled = (gtk_widget_get_state (gtk_widget_get_parent (widget)) == GTK_STATE_INSENSITIVE);
		params.state_type = gtk_widget_get_state (gtk_widget_get_parent (widget));
	}
	/* --- */


	OptionParameters option;

	option.draw_bullet = ((shadow_type == GTK_SHADOW_IN)
						  || (shadow_type == GTK_SHADOW_ETCHED_IN));
	option.inconsistent = (shadow_type == GTK_SHADOW_ETCHED_IN);
	nodoka_gdk_color_to_rgb (&nodoka_style->bullet_color, &option.bullet_color.r, &option.bullet_color.g, &option.bullet_color.b);
	double trans = 1.0;

#ifdef HAVE_ANIMATION
	if (nodoka_style->animation)
		nodoka_animation_connect_checkbox (widget);

	if (nodoka_style->animation && widget && GTK_IS_CHECK_BUTTON (widget)
		&& nodoka_animation_is_animated (widget)
		&& !gtk_toggle_button_get_inconsistent (GTK_TOGGLE_BUTTON (widget)))
	{
		gfloat elapsed = nodoka_animation_elapsed (widget);
		trans = sqrt (sqrt (MIN (elapsed / CHECK_ANIMATION_TIME, 1.0)));
	}
#endif

	nodoka_draw_radiobutton (cr, colors, &params, &option, x, y, width, height,
							 trans);

	cairo_destroy (cr);
}

static void
nodoka_style_draw_check (DRAW_ARGS)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;
	cairo_t *cr;
	WidgetParameters params;

	cr = nodoka_begin_paint (window, area);

	nodoka_sanitize_size (window, &width, &height);

	nodoka_set_widget_parameters (widget, style, state_type, &params);

	/* Treeview Workarounds */	
	if (!(GTK_IS_CHECK_BUTTON (widget)))
		params.focus = FALSE;

	if (DETAIL ("cellcheck") && !params.disabled && widget && gtk_widget_get_parent (widget))
	{
		params.disabled = (gtk_widget_get_state (gtk_widget_get_parent (widget)) == GTK_STATE_INSENSITIVE);
		params.state_type = gtk_widget_get_state (gtk_widget_get_parent (widget));
	}
	/* --- */

	OptionParameters option;

	option.draw_bullet = ((shadow_type == GTK_SHADOW_IN)
						  || (shadow_type == GTK_SHADOW_ETCHED_IN));
	option.inconsistent = (shadow_type == GTK_SHADOW_ETCHED_IN);
	nodoka_gdk_color_to_rgb (&nodoka_style->bullet_color, &option.bullet_color.r, &option.bullet_color.g, &option.bullet_color.b);
	double trans = 1.0;

#ifdef HAVE_ANIMATION
	if (nodoka_style->animation)
		nodoka_animation_connect_checkbox (widget);

	if (nodoka_style->animation && widget && GTK_IS_CHECK_BUTTON (widget)
		&& nodoka_animation_is_animated (widget)
		&& !gtk_toggle_button_get_inconsistent (GTK_TOGGLE_BUTTON (widget)))
	{
		gfloat elapsed = nodoka_animation_elapsed (widget);
		trans = sqrt (sqrt (MIN (elapsed / CHECK_ANIMATION_TIME, 1.0)));
	}
#endif

	nodoka_draw_checkbutton (cr, colors, &params, &option, x, y, width, height,
							 trans);

	cairo_destroy (cr);
}

static void
nodoka_style_draw_tab (DRAW_ARGS)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;
	WidgetParameters params;
	ArrowParameters arrow;

	cairo_t *cr = nodoka_begin_paint (window, area);

	nodoka_set_widget_parameters (widget, style, state_type, &params);
	arrow.type = NDK_ARROW_COMBO;
	arrow.direction = NDK_DIRECTION_DOWN;

	nodoka_draw_arrow (cr, colors, &params, &arrow, x, y, width, height);

	cairo_destroy (cr);
}

static void
nodoka_style_draw_vline (GtkStyle * style, GdkWindow * window,
						 GtkStateType state_type, GdkRectangle * area,
						 GtkWidget * widget, const gchar * detail, gint y1,
						 gint y2, gint x)
{
	SeparatorParameters separator = { FALSE };
	cairo_t *cr = nodoka_begin_paint (window, area);
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;

	if (!
		(widget && gtk_widget_get_parent (widget) && GTK_IS_HBOX (gtk_widget_get_parent (widget)) && gtk_widget_get_parent (gtk_widget_get_parent (widget))
		 && GTK_IS_TOGGLE_BUTTON (gtk_widget_get_parent (gtk_widget_get_parent (widget))) && gtk_widget_get_parent (gtk_widget_get_parent (gtk_widget_get_parent (widget)))
		 && GTK_IS_COMBO_BOX (gtk_widget_get_parent (gtk_widget_get_parent (gtk_widget_get_parent (widget))))))
	{
		nodoka_draw_separator (cr, colors, NULL, &separator, x, y1, 2, y2 - y1);
	}
	else
		nodoka_draw_combo_separator (cr, colors, NULL, x, y1, 2, y2 - y1);
	cairo_destroy (cr);
}

static void
nodoka_style_draw_hline (GtkStyle * style, GdkWindow * window,
						 GtkStateType state_type, GdkRectangle * area,
						 GtkWidget * widget, const gchar * detail, gint x1,
						 gint x2, gint y)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;
	cairo_t *cr = nodoka_begin_paint (window, area);

	SeparatorParameters separator;
	separator.horizontal = TRUE;

	nodoka_draw_separator (cr, colors, NULL, &separator, x1, y, x2 - x1, 2);

	cairo_destroy (cr);
}

static void
nodoka_style_draw_shadow_gap (DRAW_ARGS, GtkPositionType gap_side, gint gap_x,
							  gint gap_width)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	const NodokaColors *colors = &nodoka_style->colors;
	cairo_t *cr = nodoka_begin_paint (window, area);

	if (DETAIL ("frame"))
	{
		WidgetParameters params;
		FrameParameters frame;

		frame.shadow = shadow_type;
		frame.gap_side = gap_side;
		frame.gap_x = gap_x;
		frame.gap_width = gap_width;
		frame.border = (NodokaRGB *) & colors->shade[4];
		frame.draw_fill = TRUE;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		if (params.roundness > 0)
			params.corners = NDK_CORNER_ALL;
		else
			params.corners = NDK_CORNER_NONE;

		nodoka_draw_frame (cr, colors, &params, &frame, x, y, width, height);
	}
	else
	{
		nodoka_parent_class->draw_shadow_gap (style, window, state_type,
											  shadow_type, area, widget, detail,
											  x, y, width, height, gap_side,
											  gap_x, gap_width);
	}

	cairo_destroy (cr);
}

static void
nodoka_style_draw_resize_grip (GtkStyle * style, GdkWindow * window,
							   GtkStateType state_type, GdkRectangle * area,
							   GtkWidget * widget, const gchar * detail,
							   GdkWindowEdge edge, gint x, gint y, gint width,
							   gint height)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;

	cairo_t *cr;

	WidgetParameters params;
	ResizeGripParameters grip;
	grip.edge = (NodokaWindowEdge) edge;

	g_return_if_fail (window != NULL);

	cr = nodoka_begin_paint (window, area);

	nodoka_set_widget_parameters (widget, style, state_type, &params);

	nodoka_draw_resize_grip (cr, colors, &params, &grip, x, y, width, height);

	cairo_destroy (cr);

	return;
}

static void
nodoka_style_draw_arrow (GtkStyle * style, GdkWindow * window,
						 GtkStateType state_type, GtkShadowType shadow,
						 GdkRectangle * area, GtkWidget * widget,
						 const gchar * detail, GtkArrowType arrow_type,
						 gboolean fill, gint x, gint y, gint width, gint height)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	NodokaColors *colors = &nodoka_style->colors;
	cairo_t *cr = nodoka_begin_paint (window, area);

	nodoka_sanitize_size (window, &width, &height);

	WidgetParameters params;
	ArrowParameters arrow;

	nodoka_set_widget_parameters (widget, style, state_type, &params);
	params.state_type = (NodokaStateType) state_type;
	arrow.type = NDK_ARROW_NORMAL;
	arrow.direction = (NodokaDirection) arrow_type;


	if (arrow_type == (GtkArrowType) 4)
	{
		cairo_destroy (cr);
		return;
	}

	if (widget && gtk_widget_get_parent (widget) && gtk_widget_get_parent (gtk_widget_get_parent (widget)) && gtk_widget_get_parent (gtk_widget_get_parent (gtk_widget_get_parent (widget))) 
	    && GTK_IS_COMBO_BOX (gtk_widget_get_parent (gtk_widget_get_parent (gtk_widget_get_parent (widget)))) && 
	    !(GTK_IS_COMBO_BOX_ENTRY (gtk_widget_get_parent (gtk_widget_get_parent (gtk_widget_get_parent (widget))))))
		x += 1;


	if (DETAIL ("arrow"))
	{
		if (widget && GTK_IS_COMBO_BOX (widget))
		{
			arrow.type = NDK_ARROW_COMBO;
			y -= 2;
			height += 4;
			x += 1;
		}

	}
	else if (DETAIL ("hscrollbar") || DETAIL ("vscrollbar"))
	{
		arrow.type = NDK_ARROW_SCROLL;
		if (arrow.direction == NDK_DIRECTION_RIGHT)
			x--;
	}
	else if (DETAIL ("spinbutton"))
	{
		arrow.type = NDK_ARROW_SPINBUTTON;
	}
	
	nodoka_draw_arrow (cr, colors, &params, &arrow, x, y, width, height);

	cairo_destroy (cr);
}

static void
nodoka_style_draw_layout (GtkStyle * style, GdkWindow * window,
						  GtkStateType state_type, gboolean use_text,
						  GdkRectangle * area, GtkWidget * widget,
						  const gchar * detail, gint x, gint y,
						  PangoLayout * layout)
{
	GdkGC *gc;

	g_return_if_fail (GTK_IS_STYLE (style));
	g_return_if_fail (window != NULL);

	gc = use_text ? style->text_gc[state_type] : style->fg_gc[state_type];

	if (area)
		gdk_gc_set_clip_rectangle (gc, area);

	if (state_type == GTK_STATE_INSENSITIVE)
	{
		NodokaStyle *nodoka_style = NODOKA_STYLE (style);
		NodokaColors *colors = &nodoka_style->colors;

		WidgetParameters params;

		nodoka_set_widget_parameters (widget, style, state_type, &params);

		GdkColor etched;
		NodokaRGB temp;

		if (widget && !gtk_widget_get_has_window (widget))
			nodoka_shade (&params.parentbg, &temp, 1.2);
		else
			nodoka_shade (&colors->bg[gtk_widget_get_state (widget)], &temp, 1.2);
		etched.red = (int) (temp.r * 65535);
		etched.green = (int) (temp.g * 65535);
		etched.blue = (int) (temp.b * 65535);

		gdk_draw_layout_with_colors (window, style->text_gc[state_type], x + 1,
									 y + 1, layout, &etched, NULL);
		gdk_draw_layout (window, style->text_gc[state_type], x, y, layout);
	}
	else
		gdk_draw_layout (window, gc, x, y, layout);

	if (area)
		gdk_gc_set_clip_rectangle (gc, NULL);
}

static void
nodoka_style_init_from_rc (GtkStyle * style, GtkRcStyle * rc_style)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	double shades[] = { 1.15, 0.95, 0.896, 0.82, 0.7, 0.665, 0.5, 0.45, 0.4 };
	NodokaRGB spot_color;
	NodokaRGB bg_normal;
	double contrast;
	int i;

	nodoka_parent_class->init_from_rc (style, rc_style);

	bg_normal.r = style->bg[0].red / 65535.0;
	bg_normal.g = style->bg[0].green / 65535.0;
	bg_normal.b = style->bg[0].blue / 65535.0;

	contrast = NODOKA_RC_STYLE (rc_style)->contrast;
	nodoka_style->hilight_ratio = NODOKA_RC_STYLE (rc_style)->hilight_ratio;

	nodoka_style->roundness = NODOKA_RC_STYLE (rc_style)->roundness;
	nodoka_style->menubarstyle = NODOKA_RC_STYLE (rc_style)->menubarstyle;
	nodoka_style->toolbarstyle = NODOKA_RC_STYLE (rc_style)->toolbarstyle;
	nodoka_style->listviewheaderstyle =
		NODOKA_RC_STYLE (rc_style)->listviewheaderstyle;
	nodoka_style->listviewstyle = NODOKA_RC_STYLE (rc_style)->listviewstyle;
	nodoka_style->scrollbarstyle =
		NODOKA_RC_STYLE (rc_style)->scrollbarstyle % 3;
	nodoka_style->has_scrollbar_color =
		NODOKA_RC_STYLE (rc_style)->has_scrollbar_color;
	nodoka_style->animation = NODOKA_RC_STYLE (rc_style)->animation;
	nodoka_style->gradients = NODOKA_RC_STYLE (rc_style)->gradients;
	nodoka_style->stripes = NODOKA_RC_STYLE (rc_style)->stripes;
	nodoka_style->bullet_color = NODOKA_RC_STYLE (rc_style)->bullet_color;
	nodoka_style->focus_inner = NODOKA_RC_STYLE (rc_style)->focus_inner;
	nodoka_style->focus_fill = NODOKA_RC_STYLE (rc_style)->focus_fill;

	if (nodoka_style->has_scrollbar_color)
		nodoka_style->scrollbar_color =
			NODOKA_RC_STYLE (rc_style)->scrollbar_color;

	/* Lighter to darker */
	for (i = 0; i < 9; i++)
	{
		nodoka_shade (&bg_normal, &nodoka_style->colors.shade[i],
					  (shades[i] - 0.7) * contrast + 0.7);
	}

	spot_color.r = style->bg[GTK_STATE_SELECTED].red / 65535.0;
	spot_color.g = style->bg[GTK_STATE_SELECTED].green / 65535.0;
	spot_color.b = style->bg[GTK_STATE_SELECTED].blue / 65535.0;

	nodoka_shade (&spot_color, &nodoka_style->colors.spot[0], 1.42);
	nodoka_style->colors.spot[1] = spot_color;
	nodoka_shade (&spot_color, &nodoka_style->colors.spot[2], 0.65);

	for (i = 0; i < 5; i++)
	{
		nodoka_gdk_color_to_rgb (&style->bg[i], &nodoka_style->colors.bg[i].r,
								 &nodoka_style->colors.bg[i].g,
								 &nodoka_style->colors.bg[i].b);

		nodoka_gdk_color_to_rgb (&style->base[i],
								 &nodoka_style->colors.base[i].r,
								 &nodoka_style->colors.base[i].g,
								 &nodoka_style->colors.base[i].b);

		nodoka_gdk_color_to_rgb (&style->text[i],
								 &nodoka_style->colors.text[i].r,
								 &nodoka_style->colors.text[i].g,
								 &nodoka_style->colors.text[i].b);
	}
}

static void
nodoka_style_draw_focus (GtkStyle * style, GdkWindow * window,
						 GtkStateType state_type, GdkRectangle * area,
						 GtkWidget * widget, const gchar * detail, gint x,
						 gint y, gint width, gint height)
{
	NodokaStyle *nodoka_style = NODOKA_STYLE (style);
	cairo_t *cr = nodoka_begin_paint (window, area);
	gboolean free_dash_list = FALSE;
	gint line_width = 1;
	gchar *dash_list = "\1\1";    

	FocusParameters focus;
	focus.inner = nodoka_style->focus_inner;
	focus.fill = nodoka_style->focus_fill;

	gint padding = 0;

	if (widget)
	{
		gtk_widget_style_get (widget, "focus-line-width", &line_width,
					      "focus-line-pattern", (gchar *) & dash_list,
					      "focus-padding", &padding,
					      NULL);

		free_dash_list = TRUE;
	}

	if (detail && !strcmp (detail, "add-mode"))
	{
		if (free_dash_list)
			g_free (dash_list);

		dash_list = "\4\4";
		free_dash_list = FALSE;
	}

	nodoka_sanitize_size (window, &width, &height);

	/* Handle nautilus' icon view */
	if (DETAIL ("icon-container"))
	{
		WidgetParameters params;
		nodoka_set_widget_parameters (widget, style, state_type, &params);
		params.roundness = 5;		
		
		nodoka_draw_simple_focus (cr, &nodoka_style->colors, &params, &focus, x, y, width, height);
	}
	/* Handle calendar and icon view */
	else if (DETAIL ("calendar-day") || DETAIL ("icon_view"))
	{ 
		WidgetParameters params;
		nodoka_set_widget_parameters (widget, style, state_type, &params);
		params.roundness = 1;		
		
		nodoka_draw_simple_focus (cr, &nodoka_style->colors, &params, &focus, x, y, width, height);
	}
	/* Handle buttons with Relief set to NONE */
	else if (widget && GTK_IS_BUTTON (widget) && (gtk_button_get_relief (GTK_BUTTON (widget)) == GTK_RELIEF_NONE))
	{
		if (state_type == GTK_STATE_NORMAL)
		{
			WidgetParameters params;
			nodoka_set_widget_parameters (widget, style, state_type, &params);

			if ((widget && gtk_widget_get_parent (widget)
				 && (GTK_IS_COMBO_BOX_ENTRY (gtk_widget_get_parent (widget)))))
			{
	
				if (nodoka_style->roundness > 0)
				{
					if (params.ltr)
						params.corners = NDK_CORNER_TOPRIGHT | NDK_CORNER_BOTTOMRIGHT;
					else
						params.corners = NDK_CORNER_TOPLEFT | NDK_CORNER_BOTTOMLEFT;
				}
				else
					params.corners = NDK_CORNER_NONE;
	
				/* Seriously, why can't non-gtk-apps at least try to be decent citizens?
				 * Take OpenOffice.org 1.9 for example. The morons responsible
				 * for this utter piece of crap gave the clip size wrong values! :'(  */
				if (params.xthickness > 2)
				{
					if (params.ltr)
						x--;
					width++;
				}
			}
			else if (nodoka_style->roundness > 0)
				params.corners = NDK_CORNER_ALL;
			else
				params.corners = NDK_CORNER_NONE;

			params.active = FALSE;

			padding += 2;

			nodoka_draw_focus (cr, &nodoka_style->colors, &params, &focus, x - padding, y - padding, width + padding*2, height + padding*2);
		}
	}
	/* Handle Tree View */
	else if (widget && (GTK_IS_TREE_VIEW (widget) || NDK_IS_CLIST (widget)))
	{
		cairo_rectangle (cr, x + 0.5, y + 0.5, width - 1, height - 1);
		cairo_set_line_width (cr, 1);
		cairo_set_source_rgb (cr, style->bg[GTK_STATE_SELECTED].red / 65535.0 * 0.755,
					  style->bg[GTK_STATE_SELECTED].green / 65535.0 * 0.755,
					  style->bg[GTK_STATE_SELECTED].blue / 65535.0 * 0.755);
		cairo_stroke (cr);
	}
	/* Handle Expander */
	else if (widget && GTK_IS_EXPANDER (widget))
	{
		WidgetParameters params;
		nodoka_set_widget_parameters (widget, style, state_type, &params);
		if (focus.inner)
			padding -= 1;
		else
			padding -= 2;

		nodoka_draw_focus (cr, &nodoka_style->colors, &params, &focus, x - padding, y - padding, width + padding*2, height + padding*2);
	}
	/* And now handle the rest */
	else if (widget && !((GTK_IS_BUTTON (widget) || GTK_IS_RANGE (widget) || GTK_IS_CHECK_BUTTON (widget) 
                              || GTK_IS_NOTEBOOK (widget)) && gtk_widget_has_focus ((GtkWidget *) widget)))
	{
		WidgetParameters params;
		nodoka_set_widget_parameters (widget, style, state_type, &params);

		if (nodoka_style->roundness > 0)
			params.corners = NDK_CORNER_ALL;
		else
			params.corners = NDK_CORNER_NONE;
		if (widget && GTK_IS_BUTTON (widget))
		{
			padding += 2;
			if ((widget && gtk_widget_get_parent (widget)
				 && (NDK_IS_COMBO (gtk_widget_get_parent (widget))
					|| GTK_IS_COMBO_BOX_ENTRY (gtk_widget_get_parent (widget)))))
			{
		
				if (nodoka_style->roundness > 0)
				{
					if (params.ltr)
						params.corners = NDK_CORNER_TOPRIGHT | NDK_CORNER_BOTTOMRIGHT;
					else
						params.corners = NDK_CORNER_TOPLEFT | NDK_CORNER_BOTTOMLEFT;
				}
				else
					params.corners = NDK_CORNER_NONE;
	
				/* Seriously, why can't non-gtk-apps at least try to be decent citizens?
				 * Take OpenOffice.org 1.9 for example. The morons responsible
				 * for this utter piece of crap gave the clip size wrong values! :'(  */
				if (params.xthickness > 2)
				{
					if (params.ltr)
						x--;
					width++;
				}
			}

			else if (nodoka_style->roundness > 0)
				params.corners = NDK_CORNER_ALL;
			else
				params.corners = NDK_CORNER_NONE;
		}
		if (widget && (GTK_IS_CHECK_BUTTON (widget)))
			padding -= 4;

		nodoka_draw_focus (cr, &nodoka_style->colors, &params, &focus, x - padding, y - padding, width + padding*2, height + padding*2);
	}

	cairo_destroy (cr);

	if (free_dash_list)
		g_free (dash_list);
}

static void
nodoka_style_copy (GtkStyle * style, GtkStyle * src)
{
	NodokaStyle *ndk_style = NODOKA_STYLE (style);
	NodokaStyle *ndk_src = NODOKA_STYLE (src);

	ndk_style->colors = ndk_src->colors;
	ndk_style->hilight_ratio = ndk_src->hilight_ratio;
	ndk_style->roundness = ndk_src->roundness;
	ndk_style->progressbarstyle = ndk_src->progressbarstyle;
	ndk_style->menubarstyle = ndk_src->menubarstyle;
	ndk_style->toolbarstyle = ndk_src->toolbarstyle;
	ndk_style->listviewheaderstyle = ndk_src->listviewheaderstyle;
	ndk_style->listviewstyle = ndk_src->listviewstyle;
	ndk_style->scrollbarstyle = ndk_src->scrollbarstyle;
	ndk_style->scrollbar_color = ndk_src->scrollbar_color;
	ndk_style->has_scrollbar_color = ndk_src->has_scrollbar_color;
	ndk_style->animation = ndk_src->animation;
	ndk_style->gradients = ndk_src->gradients;
	ndk_style->stripes = ndk_src->stripes;
	ndk_style->bullet_color = ndk_src->bullet_color;
	ndk_style->focus_inner = ndk_src->focus_inner;
	ndk_style->focus_fill = ndk_src->focus_fill;

	nodoka_parent_class->copy (style, src);
}

static void
nodoka_style_unrealize (GtkStyle * style)
{
	nodoka_parent_class->unrealize (style);
}

static GdkPixbuf *
set_transparency (const GdkPixbuf * pixbuf, gdouble alpha_percent)
{
	GdkPixbuf *target;
	guchar *data, *current;
	guint x, y, rowstride, height, width;

	g_return_val_if_fail (pixbuf != NULL, NULL);
	g_return_val_if_fail (GDK_IS_PIXBUF (pixbuf), NULL);

	/* Returns a copy of pixbuf with it's non-completely-transparent pixels to
	 * have an alpha level "alpha_percent" of their original value. */

	target = gdk_pixbuf_add_alpha (pixbuf, FALSE, 0, 0, 0);

	if (alpha_percent == 1.0)
		return target;
	width = gdk_pixbuf_get_width (target);
	height = gdk_pixbuf_get_height (target);
	rowstride = gdk_pixbuf_get_rowstride (target);
	data = gdk_pixbuf_get_pixels (target);

	for (y = 0; y < height; y++)
	{
		for (x = 0; x < width; x++)
		{
			/* The "4" is the number of chars per pixel, in this case, RGBA,
			 * the 3 means "skip to the alpha" */
			current = data + (y * rowstride) + (x * 4) + 3;
			*(current) = (guchar) (*(current) * alpha_percent);
		}
	}

	return target;
}

static GdkPixbuf *
scale_or_ref (GdkPixbuf * src, int width, int height)
{
	if (width == gdk_pixbuf_get_width (src)
		&& height == gdk_pixbuf_get_height (src))
	{
		return g_object_ref (src);
	}
	else
	{
		return gdk_pixbuf_scale_simple (src, width, height,
										GDK_INTERP_BILINEAR);
	}
}

static GdkPixbuf *
nodoka_style_draw_render_icon (GtkStyle * style, const GtkIconSource * source,
							   GtkTextDirection direction, GtkStateType state,
							   GtkIconSize size, GtkWidget * widget,
							   const char *detail)
{
	int width = 1;
	int height = 1;
	GdkPixbuf *scaled;
	GdkPixbuf *stated;
	GdkPixbuf *base_pixbuf;
	GdkScreen *screen;
	GtkSettings *settings;

	/* Oddly, style can be NULL in this function, because
	 * GtkIconSet can be used without a style and if so
	 * it uses this function.
	 */

	base_pixbuf = gtk_icon_source_get_pixbuf (source);

	g_return_val_if_fail (base_pixbuf != NULL, NULL);

	if (widget && gtk_widget_has_screen (widget))
	{
		screen = gtk_widget_get_screen (widget);
		settings = gtk_settings_get_for_screen (screen);
	}
	else if (style->colormap)
	{
		screen = gdk_colormap_get_screen (style->colormap);
		settings = gtk_settings_get_for_screen (screen);
	}
	else
	{
		settings = gtk_settings_get_default ();
		GTK_NOTE (MULTIHEAD,
				  g_warning
				  ("Using the default screen for"
				   " gtk_default_nodoka_style_draw_render_icon()"));
	}

	if (size != (GtkIconSize) - 1
		&& !gtk_icon_size_lookup_for_settings (settings, size, &width, &height))
	{
		g_warning (G_STRLOC ": invalid icon size '%d'", size);
		return NULL;
	}

	/* If the size was wildcarded, and we're allowed to scale, then scale; otherwise,
	 * leave it alone.
	 */
	if (size != (GtkIconSize) - 1
		&& gtk_icon_source_get_size_wildcarded (source))
		scaled = scale_or_ref (base_pixbuf, width, height);
	else
		scaled = g_object_ref (base_pixbuf);

	/* If the state was wildcarded, then generate a state. */
	if (gtk_icon_source_get_state_wildcarded (source))
	{
		if (state == GTK_STATE_INSENSITIVE)
		{
			stated = set_transparency (scaled, 0.3);
			gdk_pixbuf_saturate_and_pixelate (stated, stated, 0.1, FALSE);

			g_object_unref (scaled);
		}
		else if (state == GTK_STATE_PRELIGHT)
		{
			stated = gdk_pixbuf_copy (scaled);

			gdk_pixbuf_saturate_and_pixelate (scaled, stated, 1.2, FALSE);

			g_object_unref (scaled);
		}
		else
		{
			stated = scaled;
		}
	}
	else
		stated = scaled;

	return stated;
}

static void
nodoka_style_init (NodokaStyle * style)
{
}

static void
nodoka_style_class_init (NodokaStyleClass * klass)
{
	GtkStyleClass *style_class = GTK_STYLE_CLASS (klass);

	nodoka_parent_class = g_type_class_peek_parent (klass);

	style_class->copy = nodoka_style_copy;
	style_class->unrealize = nodoka_style_unrealize;
	style_class->init_from_rc = nodoka_style_init_from_rc;
	style_class->draw_handle = nodoka_style_draw_handle;
	style_class->draw_slider = nodoka_style_draw_slider;
	style_class->draw_shadow_gap = nodoka_style_draw_shadow_gap;
	style_class->draw_focus = nodoka_style_draw_focus;
	style_class->draw_box = nodoka_style_draw_box;
	style_class->draw_shadow = nodoka_style_draw_shadow;
	style_class->draw_box_gap = nodoka_style_draw_box_gap;
	style_class->draw_extension = nodoka_style_draw_extension;
	style_class->draw_option = nodoka_style_draw_option;
	style_class->draw_check = nodoka_style_draw_check;
	style_class->draw_flat_box = nodoka_style_draw_flat_box;
	style_class->draw_tab = nodoka_style_draw_tab;
	style_class->draw_vline = nodoka_style_draw_vline;
	style_class->draw_hline = nodoka_style_draw_hline;
	style_class->draw_resize_grip = nodoka_style_draw_resize_grip;
	style_class->draw_arrow = nodoka_style_draw_arrow;
	style_class->draw_layout = nodoka_style_draw_layout;
	style_class->render_icon = nodoka_style_draw_render_icon;
}

GType nodoka_type_style = 0;

void
nodoka_style_register_type (GTypeModule * module)
{
	static const GTypeInfo object_info = {
		sizeof (NodokaStyleClass),
		(GBaseInitFunc) NULL,
		(GBaseFinalizeFunc) NULL,
		(GClassInitFunc) nodoka_style_class_init,
		NULL,					/* class_finalize */
		NULL,					/* class_data */
		sizeof (NodokaStyle),
		0,					/* n_preallocs */
		(GInstanceInitFunc) nodoka_style_init,
		NULL
	};

	nodoka_type_style =
		g_type_module_register_type (module, GTK_TYPE_STYLE, "NodokaStyle",
									 &object_info, 0);
}
