/*
 * nodoka_draw.c
 * This file is part of gtk-nodoka-engine
 *
 * Copyright (C) 2007, 2008 - Martin Sourada, Daniel Geiger
 *
 * gtk-nodoka-engine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * gtk-nodoka-engine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gtk-nodoka-engine; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */ 


#include "nodoka_draw.h"
#include "nodoka_style.h"
#include "nodoka_types.h"

#include "support.h"

#include <cairo.h>

#define M_PI 3.14159265358979323846
#define GLASS_HILIGHT 1.1
#define BULGING_TOP_HILIGHT 1.105
#define BULGING_BOTTOM_HILIGHT 1.06
#define HOLLOW_TOP_HILIGHT 1.147
#define HOLLOW_BOTTOM_HILIGHT 1
#define DARK_TOP_HILIGHT 1.487
#define DARK_BOTTOM_HILIGHT 1
#define GRADIENT_CENTER 0.7

/* Static Nodoka Functions */

/* Rounded rectangle with "negative" roundness */
static void
nodoka_rounded_rectangle_inverted (cairo_t * cr, double x, double y, double w, double h, double r, uint8 corners)
{
	cairo_translate (cr, x, y);

	if (corners & NDK_CORNER_TOPLEFT)
		cairo_move_to (cr, 0, -r);
	else
		cairo_move_to (cr, 0, 0);

	if (corners & NDK_CORNER_BOTTOMLEFT)
		cairo_arc (cr, r, h + r, r, M_PI * 1.0, M_PI * 1.5);
	else
		cairo_line_to (cr, 0, h);

	if (corners & NDK_CORNER_BOTTOMRIGHT)
		cairo_arc (cr, w - r, h + r, r, M_PI * 1.5, M_PI * 2.0);
	else
		cairo_line_to (cr, w, h);

	if (corners & NDK_CORNER_TOPRIGHT)
		cairo_arc (cr, w - r, -r, r, M_PI * 0.0, M_PI * 0.5);
	else
		cairo_line_to (cr, w, 0);

	if (corners & NDK_CORNER_TOPLEFT)
		cairo_arc (cr, r, -r, r, M_PI * 0.5, M_PI * 1.0);
	else
		cairo_line_to (cr, 0, 0);

	cairo_translate (cr, -x, -y);

}


/* Standard rounded rectangle */
static void
nodoka_rounded_rectangle (cairo_t * cr,
							  double x, double y, double w, double h,
							  int radius, uint8 corners)
{
	/* Sanity check */
	if (radius <= 0)
	{
		cairo_rectangle (cr, x, y, w, h);
		return;
	}

	int r = radius;
	if (w / 2 <= r)
		r = w / 2;
	if (h / 2 <= r)
		r = h / 2;
	if (r < 0)
		r = 0;


	/* And begin the drawign */
	if (corners & NDK_CORNER_TOPLEFT)
		cairo_move_to (cr, x + r, y);
	else
		cairo_move_to (cr, x, y);

	if (corners & NDK_CORNER_TOPRIGHT)
		cairo_arc (cr, x + w - r, y + r, r, M_PI * 1.5,
				   M_PI * 2);
	else
		cairo_line_to (cr, x + w, y);

	if (corners & NDK_CORNER_BOTTOMRIGHT)
		cairo_arc (cr, x + w - r, y + h - r, r, 0, M_PI * 0.5);
	else
		cairo_line_to (cr, x + w, y + h);

	if (corners & NDK_CORNER_BOTTOMLEFT)
		cairo_arc (cr, x + r, y + h - r, r, M_PI * 0.5, M_PI);
	else
		cairo_line_to (cr, x, y + h);

	if (corners & NDK_CORNER_TOPLEFT)
		cairo_arc (cr, x + r, y + r, r, M_PI, M_PI * 1.5);
	else
		cairo_line_to (cr, x, y);
}

/* Fast version of the rounded rectangle, to be used only for outlines */
static void
nodoka_rounded_rectangle_fast (cairo_t * cr, double x, double y, double w,
					double h, int radius, uint8 corners)
{
	if (radius >= 1)
	{
		nodoka_rounded_rectangle (cr, x, y, w, h, radius, corners);
		return;
	}
	else if (radius <= 0)
	{
		cairo_rectangle (cr, x, y, w, h);
		return;
	}

	const float RADIUS_CORNERS = 0.36;

	if (corners & NDK_CORNER_TOPLEFT)
		cairo_move_to (cr, x + RADIUS_CORNERS, y);
	else
		cairo_move_to (cr, x, y);

	if (corners & NDK_CORNER_TOPRIGHT)
	{
		cairo_line_to (cr, x + w - RADIUS_CORNERS, y);
		cairo_move_to (cr, w + x, y + RADIUS_CORNERS);
	}
	else
		cairo_line_to (cr, x + w, y);

	if (corners & NDK_CORNER_BOTTOMRIGHT)
	{
		cairo_line_to (cr, w + x, h + y - RADIUS_CORNERS);
		cairo_move_to (cr, w + x - RADIUS_CORNERS, h + y);
	}
	else
		cairo_line_to (cr, w + x, h + y);

	if (corners & NDK_CORNER_BOTTOMLEFT)
	{
		cairo_line_to (cr, x + RADIUS_CORNERS, h + y);
		cairo_move_to (cr, x, h + y - RADIUS_CORNERS);
	}
	else
		cairo_line_to (cr, x, h + y);

	if (corners & NDK_CORNER_TOPLEFT)
		cairo_line_to (cr, x, y + RADIUS_CORNERS);
	else
	{
		if (corners == NDK_CORNER_NONE)
			cairo_close_path (cr);
		else
			cairo_line_to (cr, x, y);
	}
}

/* Basic cairo transforms */
static void
rotate_mirror_translate (cairo_t * cr, double radius, double x, double y,
						 boolean mirror_horizontally, boolean mirror_vertically)
{
	cairo_matrix_t matrix_rotate;
	cairo_matrix_t matrix_mirror;
	cairo_matrix_t matrix_result;

	double r_cos = cos (radius);
	double r_sin = sin (radius);

	cairo_matrix_init (&matrix_rotate, r_cos, r_sin, r_sin, r_cos, x, y);

	cairo_matrix_init (&matrix_mirror, mirror_horizontally ? -1 : 1, 0, 0,
					   mirror_vertically ? -1 : 1, 0, 0);

	cairo_matrix_multiply (&matrix_result, &matrix_mirror, &matrix_rotate);

	cairo_set_matrix (cr, &matrix_result);
}

/* Nodoka gradient */
/* Use this as close to drawing function as possible, on some systems these
   settings get removed if cairo_clip(_preserve) () is called */
static void
nodoka_set_gradient (cairo_t * cr, const NodokaRGB * color, double top_hilight,
					  double bottom_hilight, double gradient_center, int width,
					  int height, boolean gradients, boolean transparent,
					  double alpha)
{
	/* If we are not transparent, don't draw transparently */
	if (!(transparent))
		alpha = 1.0;
		
	if (gradients)
	{
		cairo_pattern_t *pattern;

		NodokaRGB top_shade;
		NodokaRGB bottom_shade;

		nodoka_shade (color, &top_shade, top_hilight);
		nodoka_shade (color, &bottom_shade, bottom_hilight);

		pattern = cairo_pattern_create_linear (0, 0, width, height);
		cairo_pattern_add_color_stop_rgba (pattern, 0, top_shade.r, top_shade.g,
										  top_shade.b, alpha);
		cairo_pattern_add_color_stop_rgba (pattern, gradient_center, color->r,
										  color->g, color->b, alpha);
		cairo_pattern_add_color_stop_rgba (pattern, 1, bottom_shade.r,
										  bottom_shade.g, bottom_shade.b, alpha);

		cairo_set_source (cr, pattern);
		cairo_pattern_destroy (pattern);
	}
	else
	{
		cairo_set_source_rgba (cr, color->r, color->g, color->b, alpha);
	}
}

/* Nodoka scale draw gradient */
static void
nodoka_scale_draw_gradient (cairo_t * cr,
							const NodokaRGB * c1,
							const NodokaRGB * c2,
							int x, int y, int width, int height,
							boolean alpha, boolean horizontal)
{
	if (alpha)
		cairo_set_source_rgba (cr, c1->r, c1->g, c1->b, 0.4);
	else
		cairo_set_source_rgb (cr, c1->r, c1->g, c1->b);

	/* Draw gradient for background */
	nodoka_set_gradient (cr, c1, HOLLOW_BOTTOM_HILIGHT, HOLLOW_TOP_HILIGHT,
						  GRADIENT_CENTER, (horizontal ? 0 : width),
						  (horizontal ? height : 0), TRUE, FALSE, 1.0);

	cairo_rectangle (cr, x, y, width, height);
	cairo_fill (cr);

	cairo_rectangle (cr, x, y, width, height);
	cairo_set_source_rgba (cr, c2->r, c2->g, c2->b, 0.8);
	cairo_stroke (cr);
}



/* Widget Functions */
void
nodoka_draw_button (cairo_t * cr,
					const NodokaColors * colors,
					const WidgetParameters * widget, const ButtonParameters * button,
					int x, int y, int width, int height)
{
	double xoffset = 0, yoffset = 0;
	NodokaRGB fill;
	NodokaRGB border_disabled = colors->shade[4];
	NodokaRGB border_normal;
	nodoka_shade (&colors->shade[6], &border_normal, 0.95);
        nodoka_shade (&colors->bg[widget->state_type], &fill, 1.0);
	double alpha = 1;

	cairo_translate (cr, x, y);
	cairo_set_line_width (cr, 1.0);

	if (widget->xthickness >= 3)
		xoffset = 1;
	if (widget->ythickness >= 3)
		yoffset = 1;

	if (widget->disabled)
	{
		border_disabled.r = border_disabled.r * (0.6) + fill.r * 0.4;
		border_disabled.g = border_disabled.g * (0.6) + fill.g * 0.4;
		border_disabled.b = border_disabled.b * (0.6) + fill.b * 0.4;
		alpha = 0.35;
	}
	else
	{
		border_normal.r = border_normal.r * (0.6) + fill.r * 0.4;
		border_normal.g = border_normal.g * (0.6) + fill.g * 0.4;
		border_normal.b = border_normal.b * (0.6) + fill.b * 0.4;
		if (widget->is_default)
                {
			border_normal.r = border_normal.r * 0.3 + colors->spot[1].r * 0.7;
			border_normal.g = border_normal.g * 0.3 + colors->spot[1].g * 0.7;
			border_normal.b = border_normal.b * 0.3 + colors->spot[1].b * 0.7;
                        fill.r = fill.r * 0.9 + colors->spot[0].r * 0.1;
                        fill.g = fill.g * 0.9 + colors->spot[0].g * 0.1;
                        fill.b = fill.b * 0.9 + colors->spot[0].b * 0.1;

                }
	}

	/* Draw background */
	nodoka_rounded_rectangle (cr, xoffset + 0.5, yoffset + 0.5,
								  width - (xoffset * 2) - 1,
								  height - (yoffset * 2) - 1,
								  widget->roundness, widget->corners);

	nodoka_set_gradient (cr, &fill, (widget->active ? HOLLOW_TOP_HILIGHT
									 : BULGING_TOP_HILIGHT),
						  (widget->active ? HOLLOW_BOTTOM_HILIGHT
						   : BULGING_BOTTOM_HILIGHT),
						  GRADIENT_CENTER, (button->horizontal ? 0 : width),
						  (button->horizontal ? height : 0), widget->gradients,
						  FALSE, alpha);

	cairo_save (cr);
	if (widget->roundness > 1)
		cairo_clip_preserve (cr);

	int curve_pos = 1;
	if (widget->roundness < 2)
		curve_pos = 2;

	cairo_fill (cr);
	cairo_restore (cr);

	/* Draw shadow */
	if (!(widget->disabled) && button->shadows)
	{
		cairo_set_source_rgba (cr, 0.2, 0.2, 0.2, 0.1);
		if (widget->active)
			nodoka_rounded_rectangle_fast (cr, xoffset + 1.5, yoffset + 1.5,
									  width - (xoffset * 2) - 2,
									  height - (yoffset * 2) - 2,
									  widget->roundness - 1, widget->corners);
		else
			nodoka_rounded_rectangle_fast (cr, xoffset + 0.5, yoffset + 0.5,
									  width - (xoffset * 2),
									  height - (yoffset * 2),
									  widget->roundness + 1, widget->corners);
		cairo_stroke (cr);
	}

	/* Draw border */
	if (widget->disabled)
		cairo_set_source_rgb (cr, border_disabled.r, border_disabled.g,
							  border_disabled.b);
	else
		cairo_set_source_rgb (cr, border_normal.r, border_normal.g,
							  border_normal.b);
	nodoka_rounded_rectangle_fast (cr, xoffset + 0.5, yoffset + 0.5,
							  width - (xoffset * 2) - 1,
							  height - (yoffset * 2) - 1, widget->roundness,
							  widget->corners);
	cairo_stroke (cr);

	/* Draw focus */	
	if (widget->focus)
	{
		nodoka_draw_focus (cr, colors, widget, &(button->focus), 0 + xoffset, 0 + yoffset, width - xoffset*2, height - yoffset*2);
	}
}

void
nodoka_draw_entry (cairo_t * cr,
				   const NodokaColors * colors,
				   const WidgetParameters * widget, const EntryParameters * entry,
				   int x, int y, int width, int height)
{
	NodokaRGB *base = (NodokaRGB *) & colors->base[widget->state_type];
	NodokaRGB border;

	if (widget->focus)
	{
		border.r = colors->spot[1].r * 0.3 + colors->spot[2].r * 0.7;
		border.g = colors->spot[1].g * 0.3 + colors->spot[2].g * 0.7;
		border.b = colors->spot[1].b * 0.3 + colors->spot[2].b * 0.7;
	}
	else
		border = colors->shade[widget->disabled ? 3 : 5];

	cairo_translate (cr, x + 0.5, y + 0.5);
	cairo_set_line_width (cr, 1.0);

	/* Fill the entry's base color */
	nodoka_rounded_rectangle (cr, 0.5, 0.5, width - 2, height - 2, widget->roundness, widget->corners);
	cairo_set_source_rgb (cr, base->r, base->g, base->b);
	cairo_fill (cr);

	/* Draw the border */
	cairo_set_source_rgb (cr, border.r, border.g, border.b);
	nodoka_rounded_rectangle_fast (cr, 1, 1, width - 3, height - 3, widget->roundness, widget->corners);
	cairo_stroke (cr);

	/* Draw shadow */
	cairo_save (cr);
	nodoka_rounded_rectangle (cr, 1.5, 1.5, width - 4, height - 4, widget->roundness - 1, widget->corners);
	cairo_clip (cr);

	cairo_set_source_rgba (cr, 0.2, 0.2, 0.2, widget->disabled ? 0 : 0.15);	
	nodoka_rounded_rectangle_fast (cr, 2, 2, width - 3, height - 3, widget->roundness - 1, widget->corners);
	cairo_stroke (cr);

	cairo_restore (cr);
	
	/* Draw the focus */
	if (widget->focus)
	{
		/* Is it in... */
		if (entry->focus.inner)
		{
			nodoka_rounded_rectangle_fast (cr, 2, 2, width - 5, height - 5, widget->roundness - 1,
											  widget->corners);
			cairo_set_source_rgba (cr, border.r, border.g, border.b, 0.5);
		}
		/* ...or out? */
		else
		{
			nodoka_rounded_rectangle_fast (cr, 0, 0, width - 1, height - 1, widget->roundness + 1,
											  widget->corners);
			cairo_set_source_rgba (cr, border.r, border.g, border.b, 0.35);
		}
		cairo_stroke (cr);
	}
}

#define TROUGH_SIZE 6
void
nodoka_draw_scale_trough (cairo_t * cr,
						  const NodokaColors * colors,
						  const WidgetParameters * widget,
						  const SliderParameters * slider,
						  int x, int y, int width, int height)
{
	int     trough_width, trough_height;
	double  translate_x, translate_y;

	cairo_save (cr);

	if (slider->horizontal)
	{
		trough_width  = width;
		trough_height = TROUGH_SIZE;
		
		translate_x   = x;
		translate_y   = y + (height/2) - (TROUGH_SIZE/2);
	}
	else
	{
		trough_width  = TROUGH_SIZE;
		trough_height = height;
		
		translate_x   = x + (width/2) - (TROUGH_SIZE/2);
		translate_y  = y;
	}

	cairo_set_line_width (cr, 1.0);
	cairo_translate (cr, translate_x, translate_y);

	/* Drawing Fill Level */
	if (slider->fill_level)
	{
		NodokaRGB *fill = (NodokaRGB *) & colors->spot[1];
		
		cairo_rectangle (cr, 1.5, 1.5, trough_width - 3, trough_height - 3);
		
		nodoka_set_gradient (cr, fill, HOLLOW_BOTTOM_HILIGHT, 
							HOLLOW_TOP_HILIGHT, GRADIENT_CENTER, 
							(slider->horizontal ? 0 : trough_width),
							(slider->horizontal ? trough_height : 0),
							widget->gradients, TRUE, 0.35);
		cairo_fill_preserve (cr);
		
		cairo_set_source_rgba (cr, colors->spot[2].r, colors->spot[2].g, 
								colors->spot[2].b, 0.28);
		cairo_stroke (cr);
	}
	/* Drawing Fill */
	else if (slider->lower)
	{
		NodokaRGB *fill = (NodokaRGB *) & colors->spot[1];
		
		cairo_rectangle (cr, 1.5, 1.5, trough_width - 3, trough_height - 3);
		
		nodoka_set_gradient (cr, fill, HOLLOW_BOTTOM_HILIGHT, 
							HOLLOW_TOP_HILIGHT, GRADIENT_CENTER, 
							(slider->horizontal ? 0 : trough_width),
							(slider->horizontal ? trough_height : 0),
							widget->gradients, FALSE, 1.0);
		cairo_fill_preserve (cr);
		
		cairo_set_source_rgba (cr, colors->spot[2].r, colors->spot[2].g, 
								colors->spot[2].b, 0.8);
		cairo_stroke (cr);
	}
	/* Drawing Trough */
	else
	{
		NodokaRGB *bg = (NodokaRGB *) & colors->shade[3];
		
		cairo_rectangle (cr, 1.5, 1.5, trough_width - 3, trough_height - 3);
		
		nodoka_set_gradient (cr, bg, HOLLOW_BOTTOM_HILIGHT, 
							HOLLOW_TOP_HILIGHT, GRADIENT_CENTER, 
							(slider->horizontal ? 0 : trough_width),
							(slider->horizontal ? trough_height : 0),
							widget->gradients, FALSE, 1.0);
		cairo_fill_preserve (cr);
		
		cairo_set_source_rgba (cr, colors->shade[5].r, colors->shade[5].g, 
								colors->shade[5].b, 0.8);
		cairo_stroke (cr);
		
		
		/* Draw shadow  */
		cairo_move_to (cr, 2.5, trough_height - 2.5);
		cairo_line_to (cr, 2.5, 2.5);
		cairo_line_to (cr, trough_width - 2.5, 2.5);
		cairo_set_source_rgba (cr, 0.2, 0.2, 0.2, 0.1);
		cairo_stroke (cr);
	}

	cairo_restore (cr);
}

void
nodoka_draw_scale_slider (cairo_t * cr,
						  const NodokaColors * colors,
						  const WidgetParameters * widget, const SliderParameters * slider,
						  int x, int y, int width, int height)
{
	/* Draw button */
	ButtonParameters button;
	button.shadows = TRUE;
	button.horizontal = slider->horizontal;
	button.focus.inner = slider->focus.inner;
	button.focus.fill = slider->focus.fill;

	nodoka_draw_button (cr, colors, widget, &button, x, y, width, height);

	/* Set colors */
	NodokaRGB fill;
	NodokaRGB fill2;

	fill = colors->bg[0];
	fill2 = colors->bg[widget->state_type];

	if (widget->prelight)
		nodoka_shade (&fill2, &fill, 1.06);

	NodokaRGB style;

	nodoka_shade (&fill, &style, 0.55);

	/* Rotate if not horizontal  */
	if (!(slider->horizontal))
	{
		int tmp = height;

		rotate_mirror_translate (cr, M_PI / 2, x, y, FALSE, FALSE);
		height = width;
		width = tmp;
	}

	/* Draw scale slider decoration */
	int circ_radius = 1;
	int circ_space = 5;
	int i;

	int circ_y = height / 2.0 - 2.5;
	int circ_x = width / 2.0 - (circ_space / 2.0);

	cairo_translate (cr, 0.5, 0.5);

	for (i = 0; i < 2; i++)
	{
		cairo_move_to (cr, circ_x, circ_y);

		cairo_arc (cr, circ_x, circ_y, circ_radius + 0.5, 0, M_PI * 2);
		cairo_close_path (cr);

		cairo_arc (cr, circ_x, circ_y + circ_space, circ_radius + 0.5, 0,
				   M_PI * 2);
		cairo_close_path (cr);

		cairo_set_source_rgba (cr, style.r, style.g, style.b, 0.2);
		cairo_fill (cr);

		cairo_arc (cr, circ_x, circ_y, circ_radius, 0, M_PI * 2);
		cairo_close_path (cr);

		cairo_arc (cr, circ_x, circ_y + circ_space, circ_radius, 0, M_PI * 2);
		cairo_close_path (cr);

		cairo_set_source_rgba (cr, style.r, style.g, style.b, 0.1);
		cairo_fill (cr);
		circ_x += circ_space;
	}
}

void
nodoka_draw_progressbar_trough (cairo_t * cr, const NodokaColors * colors,
								const WidgetParameters * widget,
							  	const ProgressBarParameters * progressbar, 
								int x, int y,
								int width, int height)
{
	boolean is_horizontal = progressbar->orientation < 2;
	NodokaRGB *border = (NodokaRGB *) & colors->shade[3];
	NodokaRGB *fill = (NodokaRGB *) & colors->bg[widget->state_type];
	int r = 0;

	if (widget->roundness > 0)
		r = 1;

	cairo_translate (cr, x, y);

	cairo_set_line_width (cr, 1.0);


	/* Fill with bg color */
	nodoka_set_gradient (cr, fill, HOLLOW_TOP_HILIGHT, HOLLOW_BOTTOM_HILIGHT,
						  GRADIENT_CENTER, is_horizontal ? 0 : width, is_horizontal ? height : 0, widget->gradients, FALSE,
						  1.0);
	nodoka_rounded_rectangle (cr, 0.5, 0.5, width - 1, height - 1, r, widget->corners);
	cairo_fill (cr);

	/* Draw border */
	nodoka_rounded_rectangle_fast (cr,0.5, 0.5, width - 1, height - 1, r, widget->corners);
	cairo_set_source_rgba (cr, border->r, border->g, border->b, 0.8);
	cairo_stroke (cr);

	/* Draw shadow */
	cairo_move_to (cr, 1.5, height - 1.5);
	cairo_line_to (cr, 1.5,  1.5);
	cairo_line_to (cr, width - 1.5, 1.5);
	
	cairo_set_source_rgba (cr, 0.2, 0.2, 0.2, 0.1);
	cairo_stroke (cr);
}

void
nodoka_draw_progressbar_fill (cairo_t * cr, const NodokaColors * colors,
							  const WidgetParameters * widget,
							  const ProgressBarParameters * progressbar, int x,
							  int y, int width, int height)
{
	boolean is_horizontal = progressbar->orientation < 2;
	double tile_pos = 0;
	double stroke_width;
	int x_step;
	NodokaRGB *fill = (NodokaRGB *) & colors->spot[1];
	NodokaRGB border;

	border.r = colors->spot[2].r * 0.5 + colors->spot[1].r * 0.5;
	border.g = colors->spot[2].g * 0.5 + colors->spot[1].g * 0.5;
	border.b = colors->spot[2].b * 0.5 + colors->spot[1].b * 0.5;

	if (is_horizontal)
	{
		if (progressbar->orientation == NDK_ORIENTATION_LEFT_TO_RIGHT)
			rotate_mirror_translate (cr, 0, x, y, FALSE, FALSE);
		else
			rotate_mirror_translate (cr, 0, x + width, y, TRUE, FALSE);
	}
	else
	{
		int tmp = height;

		height = width;
		width = tmp;

		x = x + 1;
		y = y - 1;
		width = width + 2;
		height = height - 2;

		if (progressbar->orientation == NDK_ORIENTATION_TOP_TO_BOTTOM)
			rotate_mirror_translate (cr, M_PI / 2, x, y, FALSE, FALSE);
		else
			rotate_mirror_translate (cr, M_PI / 2, x, y + width, TRUE, FALSE);
	}

	cairo_rectangle (cr, 1, 0, width - 2, height);

	cairo_save (cr);
	cairo_clip (cr);

	stroke_width = height * 2;
	x_step = (((float) stroke_width / 10) * progressbar->offset);
	cairo_set_line_width (cr, 1.0);
	cairo_save (cr);

	/* Draw fill */
	cairo_rectangle (cr, 2, 1, width - 4, height - 2);
	nodoka_set_gradient (cr, fill, DARK_TOP_HILIGHT, DARK_BOTTOM_HILIGHT,
						  GRADIENT_CENTER, 0, height, widget->gradients, FALSE,
						  1.0);
	cairo_fill (cr);

	/* Draw strokes */
	if (progressbar->stripes)
	{
		while (tile_pos <= width + x_step - 2)
		{
			cairo_move_to (cr, stroke_width / 2 - x_step, 0);
			cairo_line_to (cr, stroke_width - x_step, 0);
			cairo_line_to (cr, stroke_width / 2 - x_step, height);
			cairo_line_to (cr, -x_step, height);

			cairo_translate (cr, stroke_width, 0);
			tile_pos += stroke_width;
		}

		cairo_pattern_t *pattern;

		pattern = cairo_pattern_create_linear (0, 0, 0, height);
		cairo_pattern_add_color_stop_rgba (pattern, 0, border.r, border.g, border.b, 0.25);
		cairo_pattern_add_color_stop_rgba (pattern, 1, border.r, border.g, border.b, 0.15);

		cairo_set_source (cr, pattern);
		cairo_pattern_destroy (pattern);

		cairo_fill (cr);
		cairo_restore (cr);
	}

	/* Draw border */
	cairo_set_source_rgba (cr, border.r, border.g, border.b, 0.8);
	cairo_rectangle (cr, 1.5, 0.5, width - 3, height - 1);
	cairo_stroke (cr);
}

void nodoka_draw_entry_progress (cairo_t * cr,
						const NodokaColors * colors,
						const WidgetParameters * widget,
						const EntryProgressParameters * progress, 
						int x, int y, int width, int height)
{
	NodokaRGB border;
	NodokaRGB fill;
	gint entry_width, entry_height;
	double entry_radius;
	double radius;
	
	cairo_save (cr);

	fill = colors->bg[widget->state_type];
	nodoka_shade (&fill, &border, 0.9);

	if (progress->max_size_known)
	{
		entry_width = progress->max_size.width + progress->border.left + progress->border.right;
		entry_height = progress->max_size.height + progress->border.top + progress->border.bottom;
		entry_radius = MIN (widget->roundness, MIN ((entry_width - 4.0) / 2.0, (entry_height - 4.0) / 2.0));
	}
	else
	{
		entry_radius = widget->roundness;
	}

	radius = MAX (0, entry_radius + 1.0 - MAX (MAX (progress->border.left, progress->border.right),
	                                           MAX (progress->border.top, progress->border.bottom)));

	if (progress->max_size_known)
	{
		/* Clip to the max size, and then draw a (larger) rectangle ... */
		nodoka_rounded_rectangle (cr, progress->max_size.x,
		                              progress->max_size.y,
		                              progress->max_size.width,
		                              progress->max_size.height,
		                              radius,
		                              NDK_CORNER_ALL);
		cairo_clip (cr);

		/* We just draw wider by one pixel ... */
		cairo_set_source_rgb (cr, fill.r, fill.g, fill.b);
		cairo_rectangle (cr, x, y + 1, width, height - 2);
		cairo_fill (cr);

		cairo_set_line_width (cr, 1.0);
		cairo_set_source_rgb (cr, border.r, border.g, border.b);
		cairo_rectangle (cr, x - 0.5, y + 0.5, width + 1, height - 1);
		cairo_stroke (cr);
	}
	else
	{
		nodoka_rounded_rectangle (cr, x, y, width + 10, height + 10, radius, NDK_CORNER_ALL);
		cairo_clip (cr);
		nodoka_rounded_rectangle (cr, x - 10, y - 10, width + 10, height + 10, radius, NDK_CORNER_ALL);
		cairo_clip (cr);

		cairo_set_source_rgb (cr, fill.r, fill.g, fill.b);
		nodoka_rounded_rectangle (cr, x + 1, y + 1, width - 2, height - 2, radius, NDK_CORNER_ALL);
		cairo_fill (cr);

		cairo_set_line_width (cr, 1.0);
		cairo_set_source_rgb (cr, border.r, border.g, border.b);
		nodoka_rounded_rectangle (cr, x + 0.5, y + 0.5, width - 1.0, height - 1.0, radius, NDK_CORNER_ALL);
		cairo_stroke (cr);
	}

	cairo_restore (cr);	
}							

void
nodoka_draw_optionmenu (cairo_t * cr,
						const NodokaColors * colors,
						const WidgetParameters * widget,
						const OptionMenuParameters * optionmenu,
						int x, int y, int width, int height)
{
	boolean horizontal = TRUE;
	if (((float) width / height < 0.5))
		horizontal = FALSE;

	ButtonParameters button;
	button.horizontal = horizontal;
	button.shadows = TRUE;
	button.focus.inner = optionmenu->focus.inner;
	button.focus.fill = optionmenu->focus.fill;
	
	nodoka_draw_button (cr, colors, widget, &button, x, y, width, height);
}

void
nodoka_draw_menubar (cairo_t * cr, const NodokaColors * colors,
					const WidgetParameters * widget, const MenuBarParameters * menubar, 
					int x, int y, int width, int height)
{
	NodokaRGB *fill = (NodokaRGB *) & colors->bg[0];
	NodokaRGB *dark = (NodokaRGB *) & colors->shade[3];

	cairo_translate (cr, x, y);
	cairo_rectangle (cr, 0, 0, width, height);

	/* Raised menubar */
	if (menubar->style == 1)
	{
		double alpha = 1;
		nodoka_set_gradient (cr, fill, BULGING_TOP_HILIGHT, BULGING_BOTTOM_HILIGHT,
							  GRADIENT_CENTER, 0, height, widget->gradients,
							  FALSE, alpha);

		cairo_fill (cr);
	}
	/* Lowered menubar */
	else if (menubar->style == 2)
	{
		double alpha = 1;
		nodoka_set_gradient (cr, fill, HOLLOW_TOP_HILIGHT, HOLLOW_BOTTOM_HILIGHT,
							  GRADIENT_CENTER, 0, height, widget->gradients,
							  FALSE, alpha);

		cairo_fill (cr);
	}
	/* Gradient menubar */
	else if (menubar->style == 3)
	{
		cairo_pattern_t *pattern;
		NodokaRGB lower;
		nodoka_shade (fill, &lower, 0.93);
		pattern = cairo_pattern_create_linear (0, 0, 0, height);
		cairo_pattern_add_color_stop_rgb (pattern, 0.0, fill->r, fill->g,
										  fill->b);
		cairo_pattern_add_color_stop_rgb (pattern, 1.0, lower.r, lower.g,
										  lower.b);
		cairo_set_source (cr, pattern);
		cairo_fill (cr);
		cairo_pattern_destroy (pattern);
	}
	/* Flat menubar */
	else
	{
		cairo_set_source_rgb (cr, fill->r, fill->g, fill->b);
		cairo_fill (cr);
	}

	cairo_move_to (cr, 0, height - 0.5);
	cairo_line_to (cr, width, height - 0.5);
	cairo_set_source_rgb (cr, dark->r, dark->g, dark->b);
	cairo_stroke (cr);
}

static void
nodoka_get_frame_gap_clip (int x, int y, int width, int height,
						   FrameParameters * frame,
						   NodokaRectangle * bevel, NodokaRectangle * border)
{
	if (frame->gap_side == NDK_GAP_TOP)
	{
		NODOKA_RECTANGLE_SET ((*bevel), 1.5 + frame->gap_x, -0.5,
							  frame->gap_width - 3, 2.0);
		NODOKA_RECTANGLE_SET ((*border), 0.5 + frame->gap_x, -0.5,
							  frame->gap_width - 2, 2.0);
	}
	else if (frame->gap_side == NDK_GAP_BOTTOM)
	{
		NODOKA_RECTANGLE_SET ((*bevel), 1.5 + frame->gap_x, height - 2.5,
							  frame->gap_width - 3, 2.0);
		NODOKA_RECTANGLE_SET ((*border), 0.5 + frame->gap_x, height - 1.5,
							  frame->gap_width - 2, 2.0);
	}
	else if (frame->gap_side == NDK_GAP_LEFT)
	{
		NODOKA_RECTANGLE_SET ((*bevel), -0.5, 1.5 + frame->gap_x,
							  2.0, frame->gap_width - 3);
		NODOKA_RECTANGLE_SET ((*border), -0.5, 0.5 + frame->gap_x,
							  1.0, frame->gap_width - 2);
	}
	else if (frame->gap_side == NDK_GAP_RIGHT)
	{
		NODOKA_RECTANGLE_SET ((*bevel), width - 2.5, 1.5 + frame->gap_x,
							  2.0, frame->gap_width - 3);
		NODOKA_RECTANGLE_SET ((*border), width - 1.5, 0.5 + frame->gap_x,
							  1.0, frame->gap_width - 2);
	}
}

void
nodoka_draw_frame (cairo_t * cr,
				   const NodokaColors * colors,
				   const WidgetParameters * widget,
				   const FrameParameters * frame,
				   int x, int y, int width, int height)
{
	NodokaRGB *border = frame->border;
	NodokaRectangle bevel_clip = { 0, 0, 0, 0 };
	NodokaRectangle frame_clip = { 0, 0, 0, 0 };

	NodokaRGB *dark = (NodokaRGB *) & colors->shade[3];
	NodokaRGB hilight;
	nodoka_shade (dark, &hilight, 1.3);

	if (frame->gap_x != -1)
		nodoka_get_frame_gap_clip (x, y, width, height,
								   (FrameParameters *) frame,
								   &bevel_clip, &frame_clip);

	cairo_set_line_width (cr, 1.0);
	cairo_translate (cr, x + 0.5, y + 0.5);

	/* Draw the background */
	if (frame->shadow == NDK_SHADOW_NONE)
		return;

	if ((frame->draw_fill))
	{
		const NodokaRGB * fill1 = (NodokaRGB *) & colors->bg[0];
		NodokaRGB fill = colors->bg[0];
		if (frame->shadow == NDK_SHADOW_OUT || frame->shadow == NDK_SHADOW_ETCHED_IN)
		{
			nodoka_shade (fill1, &fill, 1.0346);
		}
		else if (frame->shadow == NDK_SHADOW_IN || frame->shadow == NDK_SHADOW_ETCHED_OUT)
		{
			nodoka_shade (fill1, &fill, 0.9666);
		}
		if (frame->shadow == NDK_SHADOW_FLAT)
			nodoka_rounded_rectangle (cr, -0.5, -0.5, width, height,
							  widget->roundness, widget->corners);
		else
			nodoka_rounded_rectangle (cr, -0.5, -0.5, width - 1, height - 1,
							  widget->roundness, widget->corners);

		cairo_set_source_rgb (cr, fill.r, fill.g, fill.b);
		cairo_fill (cr);
	}


	cairo_save (cr);

	/* Set clip for the bevel */
	if (frame->gap_x != -1)
	{
		/* Set clip for gap */
		cairo_set_fill_rule (cr, CAIRO_FILL_RULE_EVEN_ODD);
		cairo_rectangle (cr, -0.5, -0.5, width, height);
		cairo_rectangle (cr, bevel_clip.x, bevel_clip.y, bevel_clip.width,
						 bevel_clip.height);
		cairo_clip (cr);
	}

	/* Draw the bevel and shadow */
	if (frame->shadow == NDK_SHADOW_ETCHED_IN
		|| frame->shadow == NDK_SHADOW_ETCHED_OUT)
	{
		cairo_set_source_rgb (cr, hilight.r, hilight.g, hilight.b);
		if (frame->shadow == NDK_SHADOW_ETCHED_IN)
			nodoka_rounded_rectangle (cr, 1, 1, width - 2, height - 2,
									  widget->roundness, widget->corners);
		else
			nodoka_rounded_rectangle (cr, 0, 0, width - 1, height - 1,
									  widget->roundness, widget->corners);
		cairo_stroke (cr);
	}
	if (frame->shadow != NDK_SHADOW_NONE
			 && frame->shadow != NDK_SHADOW_FLAT)
	{
		ShadowParameters shadow;
		shadow.corners = widget->corners;
		shadow.shadow = frame->shadow;
		if (frame->shadow == NDK_SHADOW_IN || frame->shadow == NDK_SHADOW_ETCHED_OUT)
		{
			int d = 0;
			cairo_save (cr);
			cairo_set_fill_rule (cr, CAIRO_FILL_RULE_EVEN_ODD);
			if (frame->shadow == NDK_SHADOW_ETCHED_OUT)
			{	
				d = 1;
				cairo_rectangle (cr, bevel_clip.x - 1, bevel_clip.y - 1, bevel_clip.width + 2,
						 bevel_clip.height + 2);
			}
			nodoka_rounded_rectangle (cr, 0.5 + d, 0.5 + d, width - 2 - d, height - 2 - d,
									  widget->roundness - 1, widget->corners);
			cairo_clip (cr);
			
			nodoka_rounded_rectangle (cr, 1 + d, 1 + d, width - 2 - d, height - 2 - d,
									  widget->roundness - 1, widget->corners);
			cairo_set_source_rgba (cr, 0.2, 0.2, 0.2, 0.15);
			cairo_stroke (cr);

			cairo_restore (cr);
		}
		else
		{
			cairo_set_source_rgba (cr, 0.2, 0.2, 0.2, 0.15);
			nodoka_rounded_rectangle (cr, 0, 0, width - 1, height - 1,
									  widget->roundness + 1, widget->corners);
				
			cairo_stroke (cr);
		}


	}

	/* restore the previous clip region */
	cairo_restore (cr);
	cairo_save (cr);
	if (frame->gap_x != -1)
	{
		/* Set clip for gap */
		cairo_set_fill_rule (cr, CAIRO_FILL_RULE_EVEN_ODD);
		cairo_rectangle (cr, -0.5, -0.5, width, height);
		cairo_rectangle (cr, frame_clip.x, frame_clip.y, frame_clip.width,
						 frame_clip.height);
		cairo_clip (cr);
	}

	/* Draw frame */
	if (frame->shadow == NDK_SHADOW_ETCHED_IN
		|| frame->shadow == NDK_SHADOW_ETCHED_OUT)
	{
		cairo_set_source_rgb (cr, dark->r, dark->g, dark->b);
		if (frame->shadow == NDK_SHADOW_ETCHED_IN)
			nodoka_rounded_rectangle (cr, 0, 0, width - 2, height - 2,
									  widget->roundness, widget->corners);
		else
			nodoka_rounded_rectangle (cr, 1, 1, width - 2, height - 2,
									  widget->roundness, widget->corners);
	}
	else
	{
		cairo_set_source_rgb (cr, border->r, border->g, border->b);
		if (frame->shadow & NDK_SHADOW_OUT)
			nodoka_rounded_rectangle (cr, 0, 0, width - 2, height - 2,
									  widget->roundness, widget->corners);
		else
			nodoka_rounded_rectangle (cr, 0, 0, width - 1, height - 1,
									  widget->roundness, widget->corners);
	}
	cairo_stroke (cr);
	cairo_restore (cr);
}

void
nodoka_draw_tab (cairo_t * cr,
				 const NodokaColors * colors,
				 const WidgetParameters * widget,
				 const TabParameters * tab, int x, int y, int width, int height)
{
	const float RADIUS = 3.0;
	int x1, x2, y1, y2;
	cairo_pattern_t *pattern;

	NodokaRGB fill1;
	NodokaRGB fill2;
	NodokaRGB border1;
	NodokaRGB border2;

	/* Set clip */
	cairo_rectangle (cr, x, y, width, height);
	cairo_clip (cr);
	cairo_new_path (cr);
	cairo_set_line_width (cr, 1.0);

	cairo_translate (cr, x, y);

	/* Prepare colors for border and fill */
	if (widget->active)
	{
		fill2 = colors->bg[widget->state_type];
		nodoka_shade (&fill2, &fill1, HOLLOW_TOP_HILIGHT);
		border2 = colors->shade[4];
		border1 = colors->shade[4];
	}
	else
	{
		fill2 = colors->bg[0];
		fill1.r = fill2.r * 0.8 + colors->spot[1].r * 0.2;
		fill1.g = fill2.g * 0.8 + colors->spot[1].g * 0.2;
		fill1.b = fill2.b * 0.8 + colors->spot[1].b * 0.2;
		border2 = border1 = colors->shade[5];
		border1 = colors->spot[2];
	}

	/* Make the tabs slightly bigger than they should be, to create a gap */
	if (tab->gap_side == NDK_GAP_TOP || tab->gap_side == NDK_GAP_BOTTOM)
	{
		height += RADIUS;
		x1 = 0;
		x2 = 0;


		if (tab->gap_side == NDK_GAP_TOP)
		{
			cairo_translate (cr, 0.0, -4.0);	/* gap at the other side */
			y1 = height;
			y2 = RADIUS;
		}
		else
		{
			y1 = 0;
			y2 = height - RADIUS;
		}


	}
	else
	{
		width += RADIUS;
		y1 = 0;
		y2 = 0;

		if (tab->gap_side == NDK_GAP_LEFT)
		{
			cairo_translate (cr, -4.0, 0.0);	/* gap at the other side */
			x1 = width;
			x2 = RADIUS;
		}
		else
		{
			x1 = 0;
			x2 = width - RADIUS;
		}
	}
	
	/* Draw fill */
	nodoka_rounded_rectangle (cr, 1, 1, width - 2, height - 2,
								  widget->roundness, widget->corners);
	pattern = cairo_pattern_create_linear (x1, y1, x2, y2);
	
	cairo_pattern_add_color_stop_rgb (pattern, 0.0, fill1.r, fill1.g, fill1.b);
	cairo_pattern_add_color_stop_rgb (pattern, GRADIENT_CENTER, fill2.r, fill2.g, fill2.b);
	cairo_pattern_add_color_stop_rgb (pattern, 1.0, fill2.r, fill2.g, fill2.b);
	cairo_set_source (cr, pattern);
	cairo_fill (cr);

	cairo_pattern_destroy (pattern);

	/* Handle focus */
	/* XXX: need to support outer focus as well */
	if (widget->focus && !(widget->active))
	{
		border1.r = colors->spot[1].r * 0.3 + colors->spot[2].r * 0.7;
		border1.g = colors->spot[1].g * 0.3 + colors->spot[2].g * 0.7;
		border1.b = colors->spot[1].b * 0.3 + colors->spot[2].b * 0.7;
		
		nodoka_rounded_rectangle_fast (cr, 1.5, 1.5, width - 3, height - 3,
							  widget->roundness - 1, widget->corners);
		
		pattern = cairo_pattern_create_linear (x1, y1, x2, y2);
		
		cairo_pattern_add_color_stop_rgba (pattern, 0.0, border1.r, border1.g, border1.b, 0.5);
		cairo_pattern_add_color_stop_rgb (pattern, GRADIENT_CENTER, fill2.r, fill2.g, fill2.b);
		cairo_pattern_add_color_stop_rgb (pattern, 1.0, fill2.r, fill2.g, fill2.b);
		cairo_set_source (cr, pattern);
		cairo_stroke (cr);
		cairo_pattern_destroy (pattern);
	}

	/* Draw border */
	nodoka_rounded_rectangle_fast (cr, 0.5, 0.5, width - 1, height - 1,
								  widget->roundness , widget->corners);
	
	pattern = cairo_pattern_create_linear (x1, y1, x2, y2);
	

	cairo_pattern_add_color_stop_rgb (pattern, 0.0, border1.r, border1.g, border1.b);
	cairo_pattern_add_color_stop_rgb (pattern, GRADIENT_CENTER, border2.r, border2.g, border2.b);
	cairo_pattern_add_color_stop_rgb (pattern, 1.0, border2.r, border2.g, border2.b);
	cairo_set_source (cr, pattern);
	cairo_stroke (cr);
	cairo_pattern_destroy (pattern);
}

void
nodoka_draw_separator (cairo_t * cr, const NodokaColors * colors,
					   const WidgetParameters * widget,
					   const SeparatorParameters * separator, int x, int y,
					   int width, int height)
{
	NodokaRGB *dark = (NodokaRGB *) & colors->shade[6];

	if (separator->horizontal)
	{
		cairo_set_line_width (cr, 1.0);
		cairo_translate (cr, x, y + 0.5);

		cairo_move_to (cr, 0.0, 0.0);
		cairo_line_to (cr, width + 1, 0.0);
		cairo_set_source_rgba (cr, dark->r, dark->g, dark->b, 0.4);
		cairo_stroke (cr);
	}
	else
	{
		cairo_set_line_width (cr, 1.0);
		cairo_translate (cr, x + 0.5, y);

		cairo_move_to (cr, 0.0, 0.0);
		cairo_line_to (cr, 0.0, height);
		cairo_set_source_rgba (cr, dark->r, dark->g, dark->b, 0.4);
		cairo_stroke (cr);
	}
}

void
nodoka_draw_combo_separator (cairo_t * cr,
							 const NodokaColors * colors,
							 const WidgetParameters * widget,
							 int x, int y, int width, int height)
{
	NodokaRGB *dark = (NodokaRGB *) & colors->shade[6];

	cairo_set_line_width (cr, 1.0);
	cairo_translate (cr, x + 0.5, y);

	cairo_move_to (cr, 0.0, 0.0);
	cairo_line_to (cr, 0.0, height + 1);
	cairo_set_source_rgba (cr, dark->r, dark->g, dark->b, 0.4);
	cairo_stroke (cr);
}

void
nodoka_draw_list_view_header (cairo_t * cr,
							  const NodokaColors * colors,
							  const WidgetParameters * widget,
							  const ListViewHeaderParameters * header,
							  int x, int y, int width, int height)
{
	const NodokaRGB *fill = &colors->bg[widget->state_type];
	NodokaRGB border = colors->shade[3];
	NodokaRGB hilight;
	nodoka_shade (&border, &hilight, 1.3);

	cairo_translate (cr, x, y);
	cairo_set_line_width (cr, 1.0);

	if (header->order == NDK_ORDER_FIRST)
	{
		cairo_move_to (cr, 0.5, height - 1);
		cairo_line_to (cr, 0.5, 0.5);
	}
	else
		cairo_move_to (cr, 0.0, 0.5);

	cairo_line_to (cr, width, 0.5);
	cairo_set_source_rgb (cr, hilight.r, hilight.g, hilight.b);
	cairo_stroke (cr);

	/* Effects */
	if (header->style > 0)
	{
		NodokaRGB hilight_header;
		nodoka_shade (fill, &hilight_header,
					  GLASS_HILIGHT * 0.909090909);
		/* glassy header */
		if (header->style == 1)
		{
			cairo_rectangle (cr, 0, 0, width, height);
			nodoka_set_gradient (cr, fill, BULGING_TOP_HILIGHT,
								  BULGING_BOTTOM_HILIGHT, GRADIENT_CENTER,
								  0, height, widget->gradients, FALSE, 1);
			/* glass effect */
			cairo_fill (cr);
		}
		/* raised */
		else if (header->style == 2)
		{
			border = colors->shade[4];
			NodokaRGB shadow_header;
			nodoka_shade (fill, &shadow_header, 0.925);

			if (!widget->gradients)
			{
				cairo_set_source_rgb (cr, shadow_header.r, shadow_header.g,
									  shadow_header.b);
				cairo_rectangle (cr, 0.0, height - 3.0, width, 2.0);
			}
			else
			{
				cairo_pattern_t *pattern;
				pattern = cairo_pattern_create_linear (0.0, height - 4.0, 0.0,
													   height - 1.0);
				cairo_pattern_add_color_stop_rgba (pattern, 0.0,
												   shadow_header.r,
												   shadow_header.g,
												   shadow_header.b, 0.0);
				cairo_pattern_add_color_stop_rgba (pattern, 1.0,
												   shadow_header.r,
												   shadow_header.g,
												   shadow_header.b, 1);
				cairo_set_source (cr, pattern);
				cairo_pattern_destroy (pattern);
				cairo_rectangle (cr, 0.0, height - 4.0, width, 3.0);
			}
			cairo_fill (cr);
		}
	}

	if (widget->focus)
	{
		border.r = colors->spot[1].r * 0.3 + colors->spot[2].r * 0.7;
		border.g = colors->spot[1].g * 0.3 + colors->spot[2].g * 0.7;
		border.b = colors->spot[1].b * 0.3 + colors->spot[2].b * 0.7;
		
		cairo_move_to (cr, 0.0, height - 1.5);
		cairo_line_to (cr, width, height - 1.5);
		cairo_set_source_rgba (cr, border.r, border.g, border.b, 0.5);
		cairo_stroke (cr);
	}

	/* Draw bottom border */
	cairo_move_to (cr, 0.0, height - 0.5);
	cairo_line_to (cr, width, height - 0.5);
	cairo_set_source_rgb (cr, border.r, border.g, border.b);
	cairo_stroke (cr);

	/* Draw resize grip */
	if ((widget->ltr && header->order != NDK_ORDER_LAST) ||
	    (!widget->ltr && header->order != NDK_ORDER_FIRST) || header->resizable)
	{
		SeparatorParameters separator;
		separator.horizontal = FALSE;

		if (widget->ltr)
			nodoka_draw_separator (cr, colors, widget, &separator, width - 0.5, 4.0,
							   1, height - 8.0);
		else
			nodoka_draw_separator (cr, colors, widget, &separator, 0.5, 4.0,
							   1, height - 8.0);
	}
}

/* We can't draw transparent things here, since it will be called on the same
 * surface multiple times, when placed on a handlebox_bin or dockitem_bin */
void
nodoka_draw_toolbar (cairo_t * cr, const NodokaColors * colors,
					const WidgetParameters * widget, const ToolbarParameters * toolbar, 
					int x, int y, int width, int height)
{
	NodokaRGB *fill = (NodokaRGB *) & colors->bg[0];
	NodokaRGB *dark = (NodokaRGB *) & colors->shade[3];

	if (toolbar->horizontal)
		cairo_translate (cr, x, y);
	else
	{
		int tmp = height;
		height = width;
		width = tmp;
		rotate_mirror_translate (cr, M_PI / 2, x, y, FALSE, FALSE);
	}


	cairo_rectangle (cr, 0, 0, width, height);

	/* Raised toolbar */
	if (toolbar->style == 1)
	{
		double alpha = 1;
		nodoka_set_gradient (cr, fill, BULGING_TOP_HILIGHT, BULGING_BOTTOM_HILIGHT,
							  GRADIENT_CENTER, 0, height, widget->gradients,
							  FALSE, alpha);

		cairo_fill (cr);
	}
	/* Lowered toolbar */
	else if (toolbar->style == 2)
	{
		double alpha = 1;
		nodoka_set_gradient (cr, fill, HOLLOW_TOP_HILIGHT, HOLLOW_BOTTOM_HILIGHT,
							  GRADIENT_CENTER, 0, height, widget->gradients,
							  FALSE, alpha);

		cairo_fill (cr);
	}
	/* Gradient toolbar */
	else if (toolbar->style == 3)
	{
		cairo_pattern_t *pattern;
		NodokaRGB lower;
		nodoka_shade (fill, &lower, 0.93);
		pattern = cairo_pattern_create_linear (0, 0, 0, height);
		cairo_pattern_add_color_stop_rgb (pattern, 0.0, fill->r, fill->g,
										  fill->b);
		cairo_pattern_add_color_stop_rgb (pattern, 1.0, lower.r, lower.g,
										  lower.b);
		cairo_set_source (cr, pattern);
		cairo_fill (cr);
		cairo_pattern_destroy (pattern);
	}
	/* Flat toolbar */
	else
	{
		cairo_set_source_rgb (cr, fill->r, fill->g, fill->b);
		cairo_fill (cr);
	}

	cairo_move_to (cr, 0, height - 0.5);
	cairo_line_to (cr, width, height - 0.5);
	cairo_set_source_rgb (cr, dark->r, dark->g, dark->b);
	cairo_stroke (cr);
}

void
nodoka_draw_menuitem (cairo_t * cr,
					  const NodokaColors * colors,
					  const WidgetParameters * widget,
					  int x, int y, int width, int height)
{
	NodokaRGB *fill = (NodokaRGB *) & colors->spot[1];
	NodokaRGB *border = (NodokaRGB *) & colors->spot[2];

	cairo_translate (cr, x, y);
	cairo_set_line_width (cr, 1.0);
	nodoka_rounded_rectangle (cr, 0.5, 0.5, width - 1, height - 1,
								  widget->roundness, widget->corners);
		
	nodoka_set_gradient (cr, fill, DARK_TOP_HILIGHT, DARK_BOTTOM_HILIGHT,
						  GRADIENT_CENTER, 0, height, widget->gradients,
						  FALSE, 1.0);

	cairo_fill_preserve (cr);

	cairo_set_source_rgba (cr, border->r, border->g, border->b, 0.8);
	cairo_stroke (cr);
}

void
nodoka_draw_scrollbar_trough (cairo_t * cr, const NodokaColors * colors,
							  const WidgetParameters * widget,
							  const ScrollBarParameters * scrollbar, int x,
							  int y, int width, int height)
{
	NodokaRGB *bg = (NodokaRGB *) & colors->bg[widget->state_type];
	NodokaRGB *border = (NodokaRGB *) & colors->shade[5];

	cairo_save (cr);

	cairo_set_line_width (cr, 1);

	if (scrollbar->horizontal)
	{
		int tmp = height;

		rotate_mirror_translate (cr, M_PI / 2, x, y, FALSE, FALSE);
		height = width;
		width = tmp;
	}
	else
	{
		cairo_translate (cr, x, y);
	}

	/* Draw fill */
	nodoka_set_gradient (cr, bg, HOLLOW_BOTTOM_HILIGHT, BULGING_TOP_HILIGHT,
					  1.0 - GRADIENT_CENTER, width, 0, widget->gradients, FALSE,
						  1.0);

	nodoka_rounded_rectangle (cr, 0, 0, width, height,
								  widget->roundness, widget->corners);
	cairo_fill (cr);

	/* Draw border */
	nodoka_rounded_rectangle_fast (cr, 0.5, 0.5, width - 1, height - 1,
								  widget->roundness, widget->corners);
	cairo_set_source_rgba (cr, border->r, border->g, border->b, 0.8);
	cairo_stroke (cr);

	cairo_restore (cr);
}

void
nodoka_draw_scrollbar_stepper (cairo_t * cr, const NodokaColors * colors,
							   const WidgetParameters * widget,
							   const ScrollBarParameters * scrollbar,
							   const ScrollBarStepperParameters * stepper,
							   int x, int y, int width, int height)
{
	const NodokaRGB *fill = &colors->bg[widget->state_type];
	NodokaRGB border_normal;
	nodoka_shade (&colors->shade[6], &border_normal, 0.95);

	/* Border */
	border_normal.r = border_normal.r * (0.6) + fill->r * 0.4;
	border_normal.g = border_normal.g * (0.6) + fill->g * 0.4;
	border_normal.b = border_normal.b * (0.6) + fill->b * 0.4;
	
	cairo_save (cr);
	
	if (!(scrollbar->horizontal))
		rotate_mirror_translate (cr, 0, x, y, FALSE, FALSE);
	else
	{
		int tmp = height;

		rotate_mirror_translate (cr, M_PI / 2, x, y, FALSE, FALSE);
		height = width;
		width = tmp;
	}
	cairo_set_line_width (cr, 1.0);


	if (widget->roundness >= 1)
	{
		nodoka_rounded_rectangle_inverted (cr, 1, 1, width -2, height -2, widget->roundness, widget->corners);
	}
	else
		cairo_rectangle (cr, 1, 1, width -2, height -2);

	nodoka_set_gradient (cr, fill, BULGING_TOP_HILIGHT, BULGING_BOTTOM_HILIGHT,
						  GRADIENT_CENTER, width, 0, widget->gradients, FALSE,
						  1.0);
	cairo_fill (cr);

	if (widget->roundness >= 1)
	{	
		nodoka_rounded_rectangle_inverted (cr, 0.5, 0.5, width - 1, height - 1, widget->roundness, widget->corners);
	}
	else
		cairo_rectangle (cr, 0.5, 0.5, width -1, height - 1);

		cairo_set_source_rgb (cr, border_normal.r, border_normal.g, border_normal.b);

	cairo_stroke (cr);

	cairo_restore (cr);
}

void
nodoka_draw_scrollbar_slider (cairo_t * cr, const NodokaColors * colors,
							  const WidgetParameters * widget,
							  const ScrollBarParameters * scrollbar, int x,
							  int y, int width, int height)
{
	if (scrollbar->junction & NDK_JUNCTION_BEGIN)
	{
		if (scrollbar->horizontal)
		{
			x -= 1;
			width += 1;
		}
		else
		{
			y -= 1;
			height += 1;
		}
	}
	
	if (scrollbar->junction & NDK_JUNCTION_END)
	{
		if (scrollbar->horizontal)
			width += 1;
		else
			height += 1;
	}

	NodokaRGB fill;
	NodokaRGB border;

	if (scrollbar->has_color && (!(widget->disabled)))
	{
		NodokaRGB *fill1 = (NodokaRGB *) & scrollbar->color;
		if (widget->prelight)
		{
			nodoka_shade (fill1, &fill, 1.1);
		}
		else
			fill = scrollbar->color;
	}
	else
		fill  = colors->bg[widget->state_type];

	nodoka_shade (&colors->shade[6], &border, 0.95);

	/* Border */
	border.r = border.r * (0.6) + fill.r * 0.4;
	border.g = border.g * (0.6) + fill.g * 0.4;
	border.b = border.b * (0.6) + fill.b * 0.4;

	if (scrollbar->horizontal)
		cairo_translate (cr, x, y);
	else
	{
		int tmp = height;

		rotate_mirror_translate (cr, M_PI / 2, x, y, FALSE, FALSE);
		height = width;
		width = tmp;
	}

	cairo_set_line_width (cr, 1);

	/* Draw background */
	nodoka_rounded_rectangle (cr, 1, 1, width - 2, height - 2, 6, widget->corners);
	nodoka_set_gradient (cr, &fill, BULGING_TOP_HILIGHT, BULGING_BOTTOM_HILIGHT,
						  GRADIENT_CENTER, 0, height, widget->gradients, FALSE,
						  1.0);
	cairo_fill (cr);

	/* Draw border */
	nodoka_rounded_rectangle_fast (cr, 0.5, 0.5, width - 1, height - 1, widget->roundness, widget->corners);
	cairo_set_source_rgb (cr, border.r, border.g, border.b);
	cairo_stroke (cr);


	/* Options */
	NodokaRGB style = colors->shade[5];

	/* Circles */
	if (scrollbar->style % 3 == 1)
	{
		int circ_radius = 1;
		int circ_space = 4;
		int i;
		float circ_y = height / 2.0 - (circ_space / 2.0);

		float circ_x = width / 2.0 - (circ_space);

		cairo_translate (cr, 0.5, 0.5);

		for (i = 0; i < 3; i++)
		{
			cairo_move_to (cr, circ_x, circ_y);

			cairo_arc (cr, circ_x, circ_y, circ_radius, 0, M_PI * 2);
			cairo_close_path (cr);

			cairo_arc (cr, circ_x, circ_y + circ_space, circ_radius, 0,
					   M_PI * 2);
			cairo_close_path (cr);

			cairo_set_source_rgba (cr, style.r, style.g, style.b, 0.5);
			cairo_fill (cr);

			circ_x += circ_space;
		}
	}
	/* Handle */
	if (scrollbar->style % 3 == 2)
	{
		int bar_x = width / 2 - 4;

		cairo_translate (cr, 0.5, 0.5);
		int i;

		for (i = 0; i < 3; i++)
		{
			cairo_move_to (cr, bar_x, 4.5);
			cairo_line_to (cr, bar_x, height - 5.5);
			cairo_set_source_rgba (cr, border.r, border.g, border.b, 0.5);
			cairo_stroke (cr);

			bar_x += 3;
		}
	}
}

void
nodoka_draw_selected_cell (cairo_t * cr,
						   const NodokaColors * colors,
						   const WidgetParameters * widget,
						   int x, int y, int width, int height)
{
	cairo_set_line_cap (cr, CAIRO_LINE_CAP_SQUARE);
	cairo_set_line_join (cr, CAIRO_LINE_JOIN_MITER);

	NodokaRGB fill;
	cairo_save (cr);

	cairo_translate (cr, x, y);

	if (widget->focus)
		fill = colors->base[widget->state_type];
	else
		fill = colors->base[GTK_STATE_ACTIVE];

	nodoka_set_gradient (cr, &fill, BULGING_TOP_HILIGHT,
						  BULGING_BOTTOM_HILIGHT, GRADIENT_CENTER, 0,
						  height, widget->gradients, FALSE, 1.0);
	cairo_rectangle (cr, 0, 0, width, height);
	cairo_fill (cr);

	NodokaRGB border;
	nodoka_shade (&fill, &border, (!widget->gradients ? 0.9 : 0.95));

	cairo_move_to (cr, 0, 0.5);
	cairo_rel_line_to (cr, width, 0);
	cairo_move_to (cr, 0, height - 0.5);
	cairo_rel_line_to (cr, width, 0);

	cairo_set_source_rgb (cr, border.r, border.g, border.b);
	cairo_stroke (cr);
	cairo_restore (cr);
}

void
nodoka_draw_statusbar (cairo_t * cr,
					   const NodokaColors * colors,
					   const WidgetParameters * widget,
					   int x, int y, int width, int height)
{
	NodokaRGB *dark = (NodokaRGB *) & colors->shade[3];

	cairo_set_line_width (cr, 1);
	cairo_translate (cr, x, y + 0.5);
	cairo_move_to (cr, 0, 0);
	cairo_line_to (cr, width, 0);
	cairo_set_source_rgb (cr, dark->r, dark->g, dark->b);
	cairo_stroke (cr);
}

void
nodoka_draw_menu_frame (cairo_t * cr,
						const NodokaColors * colors,
						const WidgetParameters * widget,
						int x, int y, int width, int height)
{
	NodokaRGB *border = (NodokaRGB *) & colors->shade[5];
	cairo_translate (cr, x, y);
	cairo_set_line_width (cr, 1);
	cairo_rectangle (cr, 0.5, 0.5, width - 1, height - 1);
	cairo_set_source_rgb (cr, border->r, border->g, border->b);

	cairo_stroke (cr);

}

void
nodoka_draw_handle (cairo_t * cr, const NodokaColors * colors,
					const WidgetParameters * widget,
					const HandleParameters * handle, int x, int y, int width,
					int height)
{
	NodokaRGB *dark = (NodokaRGB *) & colors->shade[3];
	
	NodokaRGB *bg = (NodokaRGB *) & colors->bg[widget->state_type];
	
	if (handle->type == NDK_HANDLE_SPLITTER)
	{
		cairo_set_source_rgb (cr, bg->r, bg->g, bg->b);
		cairo_rectangle (cr, x, y, width, height);
		cairo_fill (cr);
	}

	int bar_height;
	int i;
	int num_bars, bar_spacing;

	num_bars = 3;
	bar_spacing = 3;
	bar_height = num_bars * bar_spacing;

	if (handle->horizontal)
	{
		int tmp = height;

		rotate_mirror_translate (cr, M_PI / 2, x + 0.5, y + 0.5, FALSE, FALSE);
		height = width;
		width = tmp;
	}
	else
	{
		cairo_translate (cr, x, y);
	}

	int circ_radius = 1;
	int circ_space = 6;

	float circ_y = height / 2.0 - circ_space;
	float circ_x = width / 2.0;

	for (i = 0; i < 3; i++)
	{
		cairo_move_to (cr, circ_x, circ_y);

		cairo_arc (cr, circ_x, circ_y, circ_radius + 0.5, 0, M_PI * 2);
		cairo_close_path (cr);

		cairo_set_source_rgba (cr, dark->r, dark->g, dark->b, 0.4);
		cairo_fill (cr);

		cairo_arc (cr, circ_x, circ_y, circ_radius, 0, M_PI * 2);
		cairo_close_path (cr);

		cairo_set_source_rgba (cr, dark->r, dark->g, dark->b, 0.4);
		cairo_fill (cr);
		circ_y += circ_space;
	}
}

static void
nodoka_draw_normal_arrow (cairo_t * cr, NodokaRGB * color,
						  double x, double y, double width, double height)
{
	const float ARROW_WIDTH = 7.7;
	const int ARROW_HEIGHT = 4.0;

	cairo_set_line_width (cr, 1);

	cairo_move_to (cr, x - ARROW_WIDTH / 2, y -ARROW_HEIGHT / 2);
	cairo_line_to (cr, x, y+ ARROW_HEIGHT / 2);
	cairo_line_to (cr, x + ARROW_WIDTH / 2, y -ARROW_HEIGHT / 2);

	cairo_set_source_rgb (cr, color->r, color->g, color->b);
	cairo_stroke (cr);
}

static void
nodoka_draw_small_arrow (cairo_t * cr, NodokaRGB * color, double x, double y,
						 double width, double height)
{
	const float ARROW_WIDTH = 5.5;
	const int ARROW_HEIGHT = 3;

	cairo_set_line_width (cr, 0.8);

	cairo_move_to (cr, x - ARROW_WIDTH / 2.0, y - ARROW_HEIGHT / 2.0);
	cairo_line_to (cr, x, y + ARROW_HEIGHT / 2.0);
	cairo_line_to (cr, x + ARROW_WIDTH / 2.0, y - ARROW_HEIGHT / 2.0);

	cairo_set_source_rgb (cr, color->r, color->g, color->b);
	cairo_stroke (cr);
}

static void
nodoka_draw_combo_arrow (cairo_t * cr, NodokaRGB * color,
						 double x, double y, double width, double height)
{
	const float ARROW_WIDTH = 7.7;
	const int ARROW_HEIGHT = 4.0;
	const int ARROW_SPACING = 8;

	cairo_set_line_width (cr, 1);

	y -= ARROW_SPACING / 2;

	cairo_move_to (cr, x - ARROW_WIDTH / 2, y + ARROW_HEIGHT / 2);
	cairo_line_to (cr, x, y + -ARROW_HEIGHT / 2);
	cairo_line_to (cr, x + ARROW_WIDTH / 2, y + ARROW_HEIGHT / 2);
	cairo_set_source_rgb (cr, color->r, color->g, color->b);
	cairo_stroke (cr);

	y += ARROW_SPACING;

	cairo_move_to (cr, x - ARROW_WIDTH / 2, y + -ARROW_HEIGHT / 2);
	cairo_line_to (cr, x, y + ARROW_HEIGHT / 2);
	cairo_line_to (cr, x + ARROW_WIDTH / 2, y + -ARROW_HEIGHT / 2);
	cairo_set_source_rgb (cr, color->r, color->g, color->b);
	cairo_stroke (cr);
}

static void
_nodoka_draw_arrow (cairo_t * cr, NodokaRGB * color,
					NodokaDirection dir, NodokaArrowType type,
					double x, double y, double width, double height)
{
	double rotate;

	if (dir == NDK_DIRECTION_LEFT)
		rotate = M_PI * 1.5;
	else if (dir == NDK_DIRECTION_RIGHT)
		rotate = M_PI * 0.5;
	else if (dir == NDK_DIRECTION_UP)
		rotate = M_PI;
	else
		rotate = 0;

	if (type == NDK_ARROW_NORMAL || type == NDK_ARROW_SCROLL)
	{
		rotate_mirror_translate (cr, rotate, x, y, FALSE, FALSE);
		nodoka_draw_normal_arrow (cr, color, 0, 0, width, height);
	}
	else if (type == NDK_ARROW_COMBO)
	{
		cairo_translate (cr, x, y);
		nodoka_draw_combo_arrow (cr, color, 0, 0, width, height);
	}
	else if (type == NDK_ARROW_SPINBUTTON)
	{
		rotate_mirror_translate (cr, rotate, x, y, FALSE, FALSE);
		nodoka_draw_small_arrow (cr, color, 0, 0, width, height);
	}
}

void
nodoka_draw_arrow (cairo_t * cr, const NodokaColors * colors,
				   const WidgetParameters * widget,
				   const ArrowParameters * arrow, int x, int y, int width,
				   int height)
{
	NodokaRGB *color = (NodokaRGB *) & colors->text[widget->state_type];

	gdouble tx, ty;

	if (arrow->direction == NDK_DIRECTION_DOWN
		|| arrow->direction == NDK_DIRECTION_UP)
	{
		tx = x + width / 2;
		ty = (y + height / 2) + 0.5;
	}
	else
	{
		tx = (x + width / 2) + 0.5;
		ty = y + height / 2;
	}

	if ((arrow->type == NDK_ARROW_SCROLL))
	{
		tx += width%2;
		ty += height%2;
	}

	if (widget->disabled)
	{
		_nodoka_draw_arrow (cr, (NodokaRGB *) & colors->shade[0],
							arrow->direction, arrow->type, tx + 0.5, ty + 0.5,
							width, height);
	}

	cairo_identity_matrix (cr);

	_nodoka_draw_arrow (cr, color, arrow->direction, arrow->type, tx, ty, width,
						height);
}

void
nodoka_draw_radiobutton (cairo_t * cr, const NodokaColors * colors,
						 const WidgetParameters * widget,
						 const OptionParameters * status, int x, int y,
						 int width, int height, double trans)
{
	NodokaRGB border;
	NodokaRGB dot;

	int radius = ((width < height) ? width / 2 : height / 2);

	if (widget->state_type == NDK_STATE_INSENSITIVE)
	{
		border = colors->shade[3];
		dot = colors->shade[3];
	}
	else
	{
		border = colors->shade[5];
		if (widget->prelight)
			border = colors->spot[1];
		dot = status->bullet_color;
	}

	cairo_translate (cr, x, y);

	/* Draw backround */
	cairo_set_line_width (cr, 1);
	if (widget->state_type != NDK_STATE_INSENSITIVE)
	{
		NodokaRGB bg = colors->base[0];
		if (widget->prelight)
		{
			bg.r = bg.r * 0.8 + colors->spot[1].r * 0.2;
			bg.g = bg.g * 0.8 + colors->spot[1].g * 0.2;
			bg.b = bg.b * 0.8 + colors->spot[1].b * 0.2;
		}
		cairo_arc (cr, width/2, height/2, radius - 1, 0, M_PI * 2);
		cairo_set_source_rgb (cr, bg.r, bg.g, bg.b);
		cairo_fill (cr);
	}

	if (widget->focus)
	{
		border.r = colors->spot[1].r * 0.7 + colors->spot[2].r * 0.3;
		border.g = colors->spot[1].g * 0.7 + colors->spot[2].g * 0.3;
		border.b = colors->spot[1].b * 0.7 + colors->spot[2].b * 0.3;
		
		cairo_arc (cr, width/2, height/2, radius + 0.5, 0, M_PI * 2);
		cairo_set_source_rgba (cr, border.r, border.g, border.b, 0.3);
		cairo_stroke (cr);
	}

	/* Draw border */
	cairo_arc (cr, width/2, height/2, radius - 0.5, 0, M_PI * 2);
	cairo_set_source_rgb (cr, border.r, border.g, border.b);
	cairo_stroke (cr);

	/* Draw bullet */
	if (status->draw_bullet && (!(status->inconsistent)))
	{
		cairo_arc (cr, width/2, height/2, radius - 3, 0, M_PI * 2);
		cairo_set_source_rgba (cr, dot.r, dot.g, dot.b, trans);
		cairo_fill (cr);
	}

	/* Draw inconsistent */
	if (status->inconsistent)
	{
		cairo_set_line_width (cr, 2);
		cairo_move_to (cr, width/2 - (radius - 2), height/2);
		cairo_line_to (cr, width/2 + (radius - 2), height/2);
		cairo_set_source_rgba (cr, dot.r, dot.g, dot.b, trans);
		cairo_stroke (cr);
	}

	/* Draw shadow */
	if (!(widget->disabled))
	{
		cairo_arc (cr, width/2, height/2, radius - 1, M_PI * 2 / 3.0, M_PI * 16 / 9.0);
		cairo_set_source_rgba (cr, 0.2, 0.2, 0.2, 0.2);
		cairo_stroke (cr);
	}
}

void
nodoka_draw_checkbutton (cairo_t * cr,
						 const NodokaColors * colors,
						 const WidgetParameters * widget,
						 const OptionParameters * status,
						 int x, int y, int width, int height, double trans)
{
	NodokaRGB border;
	NodokaRGB dot;

	if (width < height)
	{
		height = width;
	}
	else
	{
		width = height;
	}
	
	if (widget->state_type == NDK_STATE_INSENSITIVE)
	{
		border = colors->shade[3];
		dot = colors->shade[3];
	}
	else
	{
		border = colors->shade[5];
		if (widget->prelight)
			border = colors->spot[1];
		dot = status->bullet_color;
	}
	if ((widget->focus))
	{
		border.r = colors->spot[1].r * 0.7 + colors->spot[2].r * 0.3;
		border.g = colors->spot[1].g * 0.7 + colors->spot[2].g * 0.3;
		border.b = colors->spot[1].b * 0.7 + colors->spot[2].b * 0.3;
	}
	NodokaRGB shadow;
	nodoka_shade (&border, &shadow, 0.9);

	cairo_translate (cr, x, y);
	cairo_set_line_width (cr, 1);

	if (widget->xthickness > 2 && widget->ythickness > 2)
	{
		cairo_rectangle (cr, 1.5, 1.5, width - 3, height - 3);
		cairo_set_source_rgba (cr, shadow.r, shadow.g, shadow.b, 0.15);
		cairo_stroke (cr);

		/* Draw the rectangle for the checkbox itself */
		cairo_rectangle (cr, 1.5, 1.5, width - 3, height - 3);
	}
	else
	{
		cairo_rectangle (cr, 1.5, 1.5, width - 3, height - 3);
	}


	if (widget->state_type != NDK_STATE_INSENSITIVE)
	{
		NodokaRGB bg = colors->base[0];
		if (widget->prelight)
		{
			bg.r = bg.r * 0.8 + colors->spot[1].r * 0.2;
			bg.g = bg.g * 0.8 + colors->spot[1].g * 0.2;
			bg.b = bg.b * 0.8 + colors->spot[1].b * 0.2;
		}

		cairo_set_source_rgb (cr, bg.r, bg.g, bg.b);
		cairo_fill_preserve (cr);
	}

	cairo_set_source_rgb (cr, border.r, border.g, border.b);
	cairo_stroke (cr);

	/* Shadow */
	if (!(widget->disabled))
	{
		cairo_move_to (cr, 2.5, height - 2);
		cairo_line_to (cr, 2.5, 2.5);
		cairo_line_to (cr, width - 2, 2.5);
		cairo_set_source_rgba (cr, 0.2, 0.2, 0.2, 0.15);
		cairo_stroke (cr);
	}

	if (widget->focus)
	{
		cairo_rectangle (cr, 0.5, 0.5, width - 1, height - 1);
		cairo_set_source_rgba (cr, border.r, border.g, border.b, 0.3);
		cairo_stroke (cr);
	}

	if (status->draw_bullet)
	{
		if (status->inconsistent)	/* Inconsistent */
		{
			cairo_set_line_width (cr, 2.0);
			cairo_move_to (cr, 3, height/2 + 1);
			cairo_line_to (cr, width - 3, height/2 + 1);
			cairo_set_source_rgba (cr, dot.r, dot.g, dot.b, trans);
			cairo_stroke (cr);
		}
		else
		{
			cairo_scale (cr, width / 14.0, height / 14.0);
			cairo_translate (cr, -2, 0);
			cairo_move_to (cr, 5, 8);
			cairo_rel_line_to (cr, 5, 4);
			cairo_rel_curve_to (cr, 1.4, -5, -1, -1, 5.7, -12.5);
			cairo_rel_curve_to (cr, -4, 4, -4, 4, -6.7, 9.3);
			cairo_rel_line_to (cr, -2.3, -2.5);
			cairo_set_source_rgba (cr, dot.r, dot.g, dot.b, trans);
			cairo_fill (cr);
		}
	}

}

void
nodoka_draw_resize_grip (cairo_t * cr,
						 const NodokaColors * colors,
						 const WidgetParameters * widget,
						 const ResizeGripParameters * grip,
						 int x, int y, int width, int height)
{
	NodokaRGB *dark = (NodokaRGB *) & colors->shade[3];
	NodokaRGB hilight;
	nodoka_shade (dark, &hilight, 1.3);
	int lx, ly;
	int bx, ex, by, ey;
	int sx, sy;

	switch (grip->edge)
	{
	case NDK_WINDOW_EDGE_NORTH_WEST:
	{
		by = 0;
		ey = 3;
		sx = x + 10;
		sy = y + 11;
		break;
	}
	case NDK_WINDOW_EDGE_NORTH:
	{
		by = 0;
		ey = 0;
		sx = x + width / 2.0 + 1.5 * 3.5;
		sy = y + 11;
		break;
	}
	case NDK_WINDOW_EDGE_NORTH_EAST:
	{
		by = 0;
		ey = 3;
		sx = x + width;
		sy = y + 11;
		break;
	}
	case NDK_WINDOW_EDGE_WEST:
	{
		by = 0;
		ey = 3;
		sx = x + 10;
		sy = y + height / 2.0 + 1.5 * 3.5;
		break;
	}
	case NDK_WINDOW_EDGE_EAST:
	{
		by = 0;
		ey = 3;
		sx = x + width;
		sy = y + height / 2.0 + 1.5 * 3.5;
		break;
	}
	case NDK_WINDOW_EDGE_SOUTH_WEST:
	{
		by = 0;
		ey = 3;
		sx = x + 10;
		sy = y + height;
		break;
	}
	case NDK_WINDOW_EDGE_SOUTH:
	{
		by = 3;
		ey = 3;
		sx = x + width / 2.0 + 1.5 * 3.5;
		sy = y + height;
		break;
	}
	case NDK_WINDOW_EDGE_SOUTH_EAST:
	{
		by = 0;
		ey = 3;
		sx = x + width;
		sy = y + height;
		break;
	}
	default:
	{
		by = 0;
		ey = 3;
		sx = x + width;
		sy = y + height;
		break;
	}
	}

	cairo_set_line_width (cr, 1);

	for (ly = by; ly <= ey; ly++)
	{
		switch (grip->edge)
		{
		case NDK_WINDOW_EDGE_NORTH_WEST:
		{
			bx = ly;
			ex = 3;
			break;
		}
		case NDK_WINDOW_EDGE_NORTH:
		{
			bx = 0;
			ex = 3;
			break;
		}
		case NDK_WINDOW_EDGE_NORTH_EAST:
		{
			bx = 0;
			ex = 3 - ly;
			break;
		}
		case NDK_WINDOW_EDGE_WEST:
		{
			bx = 3;
			ex = 3;
			break;
		}
		case NDK_WINDOW_EDGE_EAST:
		{
			bx = 0;
			ex = 0;
			break;
		}
		case NDK_WINDOW_EDGE_SOUTH_WEST:
		{
			bx = 3 - ly;
			ex = 3;
			break;
		}
		case NDK_WINDOW_EDGE_SOUTH:
		{
			bx = 0;
			ex = 3;
			break;
		}
		case NDK_WINDOW_EDGE_SOUTH_EAST:
		{
			bx = 0;
			ex = ly;
			break;
		}
		default:
		{
			bx = 0;
			ex = ly;
			break;
		}
		}

		for (lx = bx; lx <= ex; lx++)
		{
			int ny = (3.5 - ly) * 3;
			int nx = lx * 3;

			cairo_set_source_rgb (cr, hilight.r, hilight.g, hilight.b);
			cairo_rectangle (cr, sx - nx - 1, sy - ny - 1, 2, 2);
			cairo_fill (cr);

			cairo_set_source_rgb (cr, dark->r, dark->g, dark->b);
			cairo_rectangle (cr, sx - nx - 1, sy - ny - 1, 1, 1);
			cairo_fill (cr);
		}
	}
}

void nodoka_draw_focus (cairo_t * cr, const NodokaColors * colors, const WidgetParameters * widget,
				const FocusParameters * focus, int x, int y, int width, int height)
{
	NodokaRGB focus_ring;

	focus_ring.r = colors->spot[1].r * 0.3 + colors->spot[2].r * 0.7;
	focus_ring.g = colors->spot[1].g * 0.3 + colors->spot[2].g * 0.7;
	focus_ring.b = colors->spot[1].b * 0.3 + colors->spot[2].b * 0.7;

	cairo_translate (cr, x, y);
	cairo_set_line_width (cr, 1.0);

	/* Draw main focus ring */
	cairo_set_source_rgb (cr, focus_ring.r, focus_ring.g, focus_ring.b);
	nodoka_rounded_rectangle_fast (cr, 0.5, 0.5,
						  width - 1,
						  height - 1, widget->roundness,
						  widget->corners);
	cairo_stroke (cr);

	/* Draw focus overlay fill */
	if (focus->fill)
	{
		cairo_set_source_rgba (cr, focus_ring.r, focus_ring.g, focus_ring.b, 0.05);
		nodoka_rounded_rectangle (cr, 1, 1,
						  width - 2,
						  height - 2, widget->roundness - 1,
						  widget->corners);
		cairo_fill (cr);
	}

	/* Draw secondary focus ring */
	if (focus->inner && !(focus->fill))
		cairo_set_source_rgba (cr, focus_ring.r, focus_ring.g, focus_ring.b, 0.5);
	else
		cairo_set_source_rgba (cr, focus_ring.r, focus_ring.g, focus_ring.b, 0.35);
		

	if (focus->inner)
	{
		nodoka_rounded_rectangle_fast (cr, 1.5, 1.5,
						  width - 3,
						  height - 3, widget->roundness - 1,
						  widget->corners);
	}
	else
	{
		nodoka_rounded_rectangle_fast (cr, -0.5, -0.5, 
						  width + 1,
						  height + 1, widget->roundness + 1,
						  widget->corners);
	}

	cairo_stroke (cr);

	cairo_translate (cr, -x, -y);	
}

G_GNUC_INTERNAL void nodoka_draw_simple_focus (cairo_t * cr,
										 const NodokaColors * colors,
										 const WidgetParameters * widget,
										 const FocusParameters * focus,
										 int x, int y, int width, int height)
{
	NodokaRGB focus_ring;

	focus_ring.r = colors->spot[1].r * 0.3 + colors->spot[2].r * 0.7;
	focus_ring.g = colors->spot[1].g * 0.3 + colors->spot[2].g * 0.7;
	focus_ring.b = colors->spot[1].b * 0.3 + colors->spot[2].b * 0.7;

	cairo_translate (cr, x, y);
	cairo_set_line_width (cr, 1.0);

	/* Draw main focus ring */
	cairo_set_source_rgb (cr, focus_ring.r, focus_ring.g, focus_ring.b);
	nodoka_rounded_rectangle_fast (cr, 0.5, 0.5,
						  width - 1,
						  height - 1, widget->roundness,
						  NDK_CORNER_ALL);
	cairo_stroke (cr);

	/* Draw focus overlay fill */
	if (focus->fill)
	{
		cairo_set_source_rgba (cr, focus_ring.r, focus_ring.g, focus_ring.b, 0.05);
		nodoka_rounded_rectangle (cr, 1, 1,
						  width - 2,
						  height - 2, widget->roundness,
						  NDK_CORNER_ALL);
		cairo_fill (cr);
	}

	cairo_translate (cr, -x, -y);	
}

void nodoka_draw_tooltip (cairo_t * cr, const NodokaColors * colors, const WidgetParameters * widget, 
				int x, int y, int width, int height)
{
	const NodokaRGB * fill = (NodokaRGB *) & colors->bg[widget->state_type];
	NodokaRGB border;
	nodoka_shade (fill, &border, 0.6);
	
	cairo_translate (cr, x, y);
	cairo_set_line_width (cr, 1);

	nodoka_set_gradient (cr, fill, 1.1, 1.0, 0.7, - height / 6, height, 
				widget->gradients, FALSE, 1.0);

	cairo_rectangle (cr, 0, 0, width, height);
	cairo_fill (cr);

	cairo_set_source_rgb (cr, border.r, border.g, border.b);
	cairo_rectangle (cr, 0.5, 0.5, width - 1, height - 1);
	cairo_stroke (cr);
}
