# gr-satellites
GNU Radio decoders for Amateur satellites.

gr-satellites is a GNU Radio out-of-tree module encompassing a collection of
telemetry decoders that supports many different Amateur satellites. This
open-source project started in 2015 with the goal of providing telemetry
decoders for all the satellites that transmit on the Amateur radio bands. It
supports most popular protocols, such as AX.25, the GOMspace NanoCom U482C and
AX100 modems, an important part of the CCSDS stack, the AO-40 protocol used in
the FUNcube satellites, and several ad-hoc protocols used in other satellites.

The decoders don't need a graphical interface to run, so they can be used in an
embedded or remote computer. They are designed to run in real time and
print the telemetry packets to the terminal as they are received. Optionally,
the telemetry packets can be uploaded in real time to the [SatNOGS
database](http://db.satnogs.org/) or any other telemetry server that implements
the SiDS (Simple Downlink Sharing Convention) protocol.

It is also possible to use the decoder with a recording (audio WAV or IQ file),
in case that the telemetry wasn't processed in real time. To do this, one has to
know the time and date at which the recording was started and the recording has
to be played back at normal speed. This allows the decoder to compute the
correct timestamps for the packets when uploading them to the telemetry
server. It also simplifies Doppler correction of the recording with Gpredict if
the recording was not Doppler corrected.

## Branches and releases

There are currently three development branches of gr-satellites:

* `maint-3.7` Compatible with GNU Radio 3.7 only. No future development will be
  done on this branch. Perhaps important changes, such as very popular
  satellites, will be backported from the `maint-3.8` branch.

* `maint-3.8` Compatible with GNU Radio 3.8 only. Future development is being
  doing in this branch, adding support to new satellites as they get
  launched. If contributing to gr-satellites, please send pull requests to
  `maint-3.8` if in doubt.

* `next` Branch where a large refactor of gr-satellites bringing important
  changes is being done. Active development is happening, but there are still
  several months of until the `next` branch is fully usable.

Starting in 2020, `master` is set equal to `maint-3.8` (before this, it was set
equal to `maint-3.7`).

Regarding the numbered releases, the `1.x.y` series is used for stable releases
on the `maint-3.7` branch (and so, releases supporting GNU Radio 3.7), the
`2.x.y` series is used for stable releases on the `maint-3.8` (and so, releases
supporting GNU Radio 3.8), and alpha releases are done on the `next` branch
showcasing some of the latest developments.

## Installation

First you need to install the dependencies (see below).

After this, gr-satellites must be installed as any other GNU Radio out-of-tree
module. The producedure usually boils down to doing the following:

```bash
mkdir build
cd build
cmake ..
make
sudo make install
sudo ldconfig
```

Finally, you need to compile the hierarchical flowgraphs include in
gr-satellites (see below).

## Dependencies

gr-satellites v2 requires GNU Radio version 3.8.

Required dependencies:

  * Phil Karn's KA9Q `libfec`. A fork that builds in modern linux systems can be found
    [here](https://github.com/quiet/libfec).
  * [construct](https://construct.readthedocs.io/en/latest/), at least version 2.9.
  * [requests](https://pypi.org/project/requests/).
  * [swig](http://www.swig.org/)

The following GNUradio out-of-tree modules are only required for the decoder of
one particular satellite. You may install only the ones you're interested in.

  * [beesat-sdr](https://github.com/daniestevez/beesat-sdr) BEESAT and TECHNOSAT decoder and TNC. It is also used for D-STAR One.
  * [gr-lilacsat](https://github.com/bg2bhc/gr-lilacsat) This only needs to be installed
  if you want to submit telemetry to HIT. A complete decoder which does not use gr-lilacsat
  is already included in gr-satellites.
  * [PW-Sat2 FramePayloadDecoder](https://github.com/PW-Sat2/FramePayloadDecoder) This only
  needs to be installed if you want to parse frames from PW-Sat2. See the instructions
  [here](https://destevez.net/2018/12/decoding-pw-sat2-with-gr-satellites/).
  * [gr-equisat_decoder](https://github.com/BrownSpaceEngineering/gr-equisat_decoder) EQUiSat decoder blocks, telemetry parser, and submitter.

If you want to use any of the realtime image decoders, you also need to install
[feh](https://feh.finalrewind.org/).

## Hierarchichal flowgraphs

Some of the decoders use hierarchichal flowgraphs. These are placed in the
folder `apps/hierarchical`. The hierarchical flowgraphs must be compiled and
installed before using any of the flowgraphs which include them.

To compile and install the hierarchical flowgraphs, the script
`compile_hierarchical.sh` in the root folder can be used.

## Usage

The signal is fed to the decoders using a UDP stream. The format used is the
same that [gqrx](http://gqrx.dk/doc/streaming-audio-over-udp) uses. Therefore,
you can use gqrx to feed the signal to the decoders. You will have to set the
proper frequency, mode and bandpass in gqrx for the satellite you want to
receive. This is probably the easiest way to start using the decoders from
gr-satellites. Gqrx supports Doppler correction with Gpredict.

*Note:* The exact frequency setting for optimal decoding may need to tuned to
properly center the signal within the passband.  This is especially true for
SSB signals. One way to do this is by using this the Radio Control panel within
Gpredict. This allows the user to make small adjustments while monitoring signals
in the gqrx passband.

It is also possible to use the frontend streamers from
[gr-frontends](https://github.com/daniestevez/gr-frontends). These allow to
stream data by UDP from different SDR hardware without using a GUI SDR program. It
remains to perform Doppler correction with Gpredict. There are also frontend
streamers to use a conventional receiver connected via soundcard and recordings
(audio WAV and IQ).

Each satellite has its own decoder in the `apps/` folder. You can open the
`.grc` file with `gnuradio-companion` and edit the parameters (they are on the
upper part of the flowgraph). You can also generate and run the corresponding
`.py` script and specify the parameters on the command line. Use the -h flag to
get help on how to specify the parameters. The decoder will printing each
telemetry packet in the terminal as soon as it receives it.

## Satellites supported

  * `sat_1kuns_pf`
    [1KUNS-PF](https://en.wikipedia.org/wiki/1KUNS-PF),
    which transmits
    1k2 or 9k6 GMSK telemetry in the 70cm band. It uses the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. You must use FM mode to receive this satellite (437.300 MHz).
    1KUNS-PF transmits JPEG images from an onboard
    camera. This decoder includes an image decoder which shows the images in
    real time using feh.
  * `sat_3cat_1`
    [3CAT-1](https://nanosatlab.upc.edu/en/missions-and-projects/3cat-1),
    which transmits 9k6 GFSK telemetry in the 70cm band. It uses a Texas
    Intruments CC1101 transceiver with a PN9 scrambler, and a (255,223) Reed-Solomon
    code from the [rscode](http://rscode.sourceforge.net/) library.
    You must use FM mode to receive this satellite (437.250MHz).
  * `sat_3cat2`
    [3CAT-2](https://nanosatlab.upc.edu/en/missions-and-projects/3cat-2) *(inactive)*,
    which
    transmits 9k6 AX.25 BPSK telemetry in the 2m band. You must use wide SSB
    mode to receive this satellite.
  * `aausat_4`
    [AAUSAT-4](http://www.space.aau.dk/aausat4/), which transmits 2k4 or 9k6 GFSK
    telemetry in the 70cm band. It uses the CSP protocol and FEC with an r=1/2, k=7
    convolutional code and a (255,223) Reed-Solomon code. You must use FM mode
    to receive this satellite (437.425MHz).
  * `aisat`
    [AISAT](https://directory.eoportal.org/web/eoportal/satellite-missions/a/aisat),
    which transmits 4k8 AF GMSK telemetry in the 70cm band. It uses a NanoCom
    U482C transceiver with the CSP protocol and FEC with a (255,223)
    Reed-Solomon code. It also uses a CCSDS scrambler. There is no telemetry
    parser yet, as the beacon format is unknown. This satellite has an AIS
    receiver on board. You must use FM mode to receive this satellite.
  * `aistechsat2`
    [AISTECHSAT-2](https://space.skyrocket.de/doc_sdat/aistechsat-1.htm),
    which transmits
    4k8 or 9k6 GMSK telemetry in the 70cm band. It uses the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. You must use FM mode to receive this satellite (436.730MHz).
  * `aistechsat3`
    [AISTECHSAT-3](https://space.skyrocket.de/doc_sdat/aistechsat-1.htm),
    which transmits
    4k8 or 9k6 GMSK telemetry in the 70cm band. It uses the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. The telemetry format is
    [documented](https://www.amsat-ea.org/app/download/11302778/Aistechsat-3+TLM+codes%2C+modulation+and+format+v1.0_AMSAT_EA.pdf)
    but a telemetry decoder has not been implemented yet.
    You must use FM mode to receive this satellite (436.730MHz).
  * `amgu_1`
    [AmGU-1](https://space.skyrocket.de/doc_sdat/amgu-1.htm),
    which transmits 4k8 FSK telemetry in the 70cm band. It uses the Mobitex
    protocol, which includes FEC with a (12,8,3) linear code and CRC-16CCITT for
    error detection.
    The decoder requires the [beesat-sdr](https://github.com/daniestevez/beesat-sdr) OOT module.
    You must use FM mode to receive this satellite (436.250MHz).
  * `ao40_uncoded`
    [AO-40](https://en.wikipedia.org/wiki/OSCAR_40) *(inactive)*,
    which transmitted 400bps BPSK
    telemetry in several bands. This decoder is for the uncoded telemetry, which
    did not use any forward error correction. The specifications of the telemetry
    can be found [in this document](http://www.amsat-dl.org/p3d/tlmspec.txt). AO-40
    is no longer functional, but it is of high historic interest. You must use SSB
    mode to receive this satellite.
  * `ao73`
    [AO-73 (FUNcube)](https://funcube.org.uk/), which transmits 1k2 BPSK
    telemetry in the 2m band. It uses the AO-40 FEC protocol, which includes
    block interleaving, an r=1/2, k=7 convolutional code, CCSDS scrambling and
    two interleaved (160,128) Reed-Solomon codes. You must use SSB mode to
    receive this satellite (145.935MHz).
  * `astrocast`, `astrocast_old`, `astrocast_9k6`
    [Astrocast 0.1](https://space.skyrocket.de/doc_sdat/astrocast-0.htm), which
    transmits 1k2 FSK telemetry in the 70cm band. It uses
    the FX.25 protocol with a (255,223) CCSDS Reed-Solomon code.
    The `astrocast_old` decoder is for the protocol variant used until 2019-02-13, which
    wast not completely compatible with AX.25.
    The `astrocast_9k6` decoder is for the 9k6 FSK protocol used for data download. This
    uses five interleaved (255,223) CCSDS Reed-Solomon codewords following the CCSDS
    standard.
    You must use FM mode
    to receive this satellite (437.150MHz).
  * `at03`
    [QB50 AT03 (PEGASUS)](https://spacedatacenter.at/pegasus/),
    which transmits 9k6 GFSK telemetry in the 70cm band. It uses the TT-64
    protocol, which includes a CRC16-ARC and FEC with a (64,48) Reed-Solomon
    code. Reed-Solomon decoding is done with the
    [rscode](http://rscode.sourceforge.net/) library. You must use FM mode to
    receive this satellite (436.670MHz).
  * `athenoxat-1`
    [ATHENOXAT-1](http://space.skyrocket.de/doc_sdat/athenoxat-1.htm),
    which transmits 4k8 AF GMSK telemetry in the 70cm band. It uses a NanoCom
    U482C transceiver, with the CSP protocol and FEC with a (255,223)
    Reed-Solomon code. It also uses a CCSDS scrambler. There is no telemetry
    parser yet, as the beacon format is unknown. This satellite is on a low
    inclination orbit, so it can only be received near the equator. You must
    use FM mode to receive this satellite (437.485MHz).
  * `atl_1`
    [ATL-1](https://space.skyrocket.de/doc_sdat/atl-1.htm), which
    transmits 1.25k, 2.5k, 5k or 12.5k FSK telemetry in the 70cm band. It uses the AO-40 FEC protocol, which includes
    block interleaving, an r=1/2, k=7 convolutional code, CCSDS scrambling and
    two interleaved (160,128) Reed-Solomon codes. You must use FM mode
    to receive this satellite (437.175MHz).
  * `au02`
    [QB50 AU02 (UNSW-EC0)](http://www.acser.unsw.edu.au/QB50),
    which transmits 4k8 AF GMSK telemetry in the 70cm band. It uses a NanoCom
    U482C transceiver, with the CSP protocol and FEC with an r=1/2, k=7
    convolutional code and a (255,223) Reed-Solomon code. It also uses a CCSDS
    scrambler. You must use FM mode to receive this satellite (436.525MHz).
  * `au03`
    [QB50 AU03 (i-INSPIRE II)](http://sydney.edu.au/inspire-cubesat/project/index.shtml),
    which transmits 4k8 AF GMSK telemetry in the 70cm band. It uses a NanoCom
    U482C transceiver, with the CSP protocol and FEC with an r=1/2, k=7
    convolutional code and a (255,223) Reed-Solomon code. It also uses a CCSDS
    scrambler. You must use FM mode to receive this satellite (436.330MHz).
  * `beesat`
    [BEESAT-1,-2,-4 and -9](http://space.skyrocket.de/doc_sdat/beesat.htm)
    and [TECHNOSAT](https://directory.eoportal.org/web/eoportal/satellite-missions/t/technosat),
    which transmit 4k8 FSK telemetry in the 70cm band. They use the Mobitex-NX
    protocol, which includes FEC with a (12,8,3) linear code and CRC-16CCITT for
    error detection.
    The decoder requires the [beesat-sdr](https://github.com/daniestevez/beesat-sdr) OOT module.
    You must use FM mode to receive these satellites (435.950MHz).
  * `by701`
    [BY70-1](http://space.skyrocket.de/doc_sdat/by70-1.htm) *(inactive)*, which transmits 9k6
    BPSK telemetry in the 70cm band. It uses FEC with an r=1/2, k=7
    convolutional code and a (255,223) Reed-Solomon code (the same as the
    LilacSat-2 9k6 BPSK telemetry). You must use wide SSB mode to receive this
    satellite. It has an optical camera on board and it transmits JPEG images
    together with the telemetry. `by701` includes a complete telemetry decoder
    and image receive software. This satellite launched on 28 December 2016 into a 520x220km
    orbit. The perigee is too low because of a problem in the launch. BY70-1 reentered
    on 18 February 2017. You must use wide SSB mode to receive this satellite.
  * `ca03`
    [QB50 CA03 (Ex-Alta 1)](https://albertasat.ca/amateur-radio-information/),
    which transmits 4k8 GFSK telemetry in the 70cm band. Occasionaly it has been seen
    to transmit in 9k6. It uses the CSP protocol
    and FEC with a (255,223) Reed-Solomon code. It also uses a G3RUH scrambler. The
    transceiver is the GomSpace NanoCom AX100, the same transceiver used in
    GOMX-3. You must use FM mode to receive this satellite (436.705MHz).
  * `cz02`
    [QB50 CZ02 (VZLUSAT-1)](http://vzlusat1.cz/en/),
    which transmits 4k8 AF GMSK telemetry in the 70cm band. It uses a NanoCom
    U482C transceiver, with the CSP protocol and a (255,223) Reed-Solomon code.
    It also uses a CCSDS scrambler. You must use FM mode to receive this satellite (437.240MHz).
  * `delphini1`
    [Delphini-1](https://projects.au.dk/ausat/delphini1/),
    which transmits
    4k8 GMSK telemetry in the 70cm band. It uses the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. You must use FM mode to receive this satellite (437.500MHz).
  * `dsat`
    [D-SAT](https://www.dsat.space/) *(inactive)*,
    which transmits 4k8 AF GMSK telemetry in
    the 70cm band. It uses a NanoCom U482C transceiver with the CSP protocol and
    FEC with a (255,223) Reed-Solomon code. It also uses a CCSDS scrambler. This
    receiver supports sending frames to the D-SAT groundstation software, which
    decodes telemetry. See
    [this post](http://destevez.net/2017/08/d-sat-support-added-to-gr-satellites/)
    for detailed instructions. D-SAT transmits JPEG images from an onboard
    camera. This decoder includes an image decoder which shows the images in
    real time using feh.
  * `dstar_one`
    [D-STAR One](http://www.d-star.one/),
    which transmits 4k8 FSK telemetry in the 70cm band. It uses the Mobitex
    protocol, which includes FEC with a (12,8,3) linear code and CRC-16CCITT for
    error detection.
    The decoder requires the [beesat-sdr](https://github.com/daniestevez/beesat-sdr) OOT module.
    You must use FM mode to receive this satellite (435.700MHz).
  * `duchifat_3`
    [DUCHIFAT-3](https://space.skyrocket.de/doc_sdat/duchifat-3.htm),
    which transmits 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. You must use wide SSB mode to receive this satellite (436.400MHz).
  * `entrysat`
    [EntrySat](https://websites.isae-supaero.fr/entrysat/),
    which transmits 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. You must use wide SSB mode to receive this satellite (436.950MHz).
  * `equisat`
    [EQUiSat](http://brownspace.org/equisat), which transmits 9k6 baud 4FSK
    telemetry in the 70cm band. It uses a double-interleaving encoding and a
    (255,223) Reed-Solomon code. 
    The decoder requires the [gr-equisat_decoder](https://github.com/BrownSpaceEngineering/gr-equisat_decoder) OOT module.
    You must use FM mode to receive this satellite (435.525MHz).
    For more info see [brownspace.org/receiving-equisat](http://brownspace.org/receiving-equisat),
    and see below for how to submit to the telemetry database.
  * `eseo`
    [ESEO](https://www.esa.int/Education/ESEO),
    which transmits 9k6 or 4k8 GFSK telemetry in the 70cm band. It uses a custom protocol
    vaguely similar to AX.25 with some form of G3RUH scrambling and a
    (255,239) Reed-Solomon code. You must use FM mode to receive this satellite (437.000MHz).
  * `facsat_1`,
    [FACSAT-1](https://en.wikipedia.org/wiki/FACSAT-1), which transmits
    9k6 GMSK telemetry in the 70cm band. It uses the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. The telemetry format is unknown. You must use FM mode to
    receive this satellite (437.350MHz).
  * `floripasat_1`,
    [FloripaSat-1](https://floripasat.ufsc.br/), which transmits 1k2 FSK
    telemetry in the 2m band (beacon) and 2k4 FSK telemetry in the 70cm band
    (downlink). It uses the
    [NGHam](https://github.com/skagmo/ngham) protocol and AX.25, but the
    Reed-Solomon of NGHam and the complete AX.25 seem to be not correctly
    implemented. You must use FM mode to receive this satellite (145.900MHz for
    beacon and 436.1MHz for downlink).
  * `fmn1`
    [FMN-1](https://space.skyrocket.de/doc_sdat/fengmaniu-1.htm),
    which transmits 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. You must use wide SSB mode to receive this satellite (435.350MHz).
  * `galassia`
    [GALASSIA](https://eoportal.org/web/eoportal/satellite-missions/content/-/article/galass-1),
    which transmits 4k8 AF GMSK telemetry in the 70cm band. It uses a NanoCom
    U482C transceiver with the the CSP protocol and FEC with a (255,223)
    Reed-Solomon code. It also uses a CCSDS scrambler. There is no telemetry
    parser yet, as the beacon format is unknown. This satellite is on a low
    inclination orbit, so it can only be received near the equator. You must
    use FM mode to receive this satellite.
  * `gomx_1`
    [GOMX-1](https://directory.eoportal.org/web/eoportal/satellite-missions/g/gomx-1),
    which transmits 4k8 AF GMSK telemetry in the 70cm band. It uses a NanoCom
    U482C transceiver with the CSP protocol and FEC with a (255,223)
    Reed-Solomon code. It also uses a CCSDS scrambler. The beacons
    include information from ADS-B beacons transmitted by terrestrial
    aircraft. You must use FM mode to receive this satellite (437.255MHz).
  * `gomx_3`
    [GOMX-3](https://directory.eoportal.org/web/eoportal/satellite-missions/g/gomx-3)
    *(inactive)*,
    which transmits 19k2 GFSK telemetry in the 70cm band. It uses the CSP
    protocol and FEC with a (255,223) Reed-Solomon code. It also uses a G3RUH
    scrambler. The beacons include information from ADS-B beacons transmitted by
    terrestrial aircraft. GOMX-3 reentered on 18 October 2016. You
    must use FM mode to receive this satellite.
  * `gr01`
    [QB50 GR01 (DUTHSat)](http://www.duthsat.gr/) *(inactive)*,
    which transmits 1k2 or 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. For 1k2 telemetry you must use SSB mode, while for 9k6 telemetry you
    must use wide SSB mode.
  * `il01`
    [QB50 IL01 (DUCHIFAT-2)](http://www.h-space-lab.org/php/hoopoe-en.php),
    which transmits 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. You must use wide SSB mode to receive this satellite (437.740MHz).
  * `indus`
    Mystery satellite transmitting on 435.080MHz using 1k2 FSK AX.25 and the
    callsign INDUSR-10 (see [here](https://twitter.com/ea4gpz/status/952593838266298368)).
  * `innosat_2`,
    [INNOSAT-2](https://space.skyrocket.de/doc_sdat/innosat-2.htm), which transmits
    4k8 GMSK telemetry in the 70cm band. It uses the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. You must use FM mode to receive this satellite (437.450MHz).
  * `itasat1`
    [ITASAT 1](http://www.itasat.ita.br/),
    which transmits 1k2 AX.25 BPSK telemetry in the 2m band.
    You must use SSB mode to receive this satellite (145.860MHz).
  * `jy1sat`
    [JY1-Sat (FUNcube-6)](https://amsat-uk.org/tag/jy1sat/),
    which transmits 1k2 BPSK
    telemetry in the 2m band. It uses the AO-40 FEC protocol, which includes
    block interleaving, an r=1/2, k=7 convolutional code, CCSDS scrambling and
    two interleaved (160,128) Reed-Solomon codes.
    Decoding SSDV images is supported. See [this post](https://destevez.net/2019/04/decoding-ssdv-from-jy1sat/).
    You must use SSB mode to receive this satellite (145.840MHz).
  * `k2sat_image`
    [K2SAT](http://www.amsatuk.me.uk/iaru/finished_detail.php?serialnum=552),
    which transmits images using QPSK in the 13cm band. See
    [this post](http://destevez.net/2018/07/k2sat-s-band-image-receiver/).
  * `kr01`
    [QB50 KR01 (LINK)](http://space.skyrocket.de/doc_sdat/link.htm)
    *(inactive)*,
    which transmits 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. Currently it transmits 1k2 telemetry (safe mode perhaps), so you
    must use SSB mode to receive this satellite.
  * `ks_1q`
    [KS-1Q](http://space.skyrocket.de/doc_sdat/cas-2t.htm)
    *(inactive)*, which transmits 20k
    FSK telemetry in the 70cm band. It uses KISS framed CSP packets and FEC with
    an r=1/2, k=7 convolutional code and a (255,223) Reed-Solomon code (the
    protocol is very similar to LilacSat-2). It also uses a CCSDS scrambler.
    You must use FM mode to receive this satellite.
  * `lilacsat1`
    [LilacSat-1](http://lilacsat.hit.edu.cn/wp/?page_id=594) *(inactive)*,
    which transmits 9k6
    BPSK telemetry in the 70cm band. It uses FEC with an r=1/2, k=7
    convolutional code an interleaved telemetry and Codec2 digital voice.
    It has an optical camera on board and it transmits JPEG images
    together with the telemetry. LilacSat-1 reentered
    in March 2019. You must use wide SSB mode to receive this satellite.
  * `lilacsat2`
    [LilacSat-2](http://lilacsat.hit.edu.cn/?page_id=257), which transmits 9k6
    BPSK, 4k8 GFSK and FM subaudio telemetry in the 70cm band. It uses FEC with
    an r=1/2, k=7 convolutional code and a (255,223) Reed-Solomon code. The
    decoders for this satellite are organized a bit different from
    the decoders for other satellites, because LilacSat-2 transmits in several
    different frequencies using several different modes. You can use `lilacsat2`
    as a usual single-frequency single-mode decoder. You can use gqrx or one of
    the frontends from gr-frontends to feed an UDP audio stream  to `lilacsat2`.
    However, you can decode only one frequency and mode using this method. You
    should tune to 437.200MHz in wide SSB mode to receive 9k6 BPSK telemetry, to
    437.200MHz in FM mode to receive FM subaudio telemetry and to 437.225MHz in
    FM mode to receive 4k8 GFSK telemetry. `lilacsat2` will recognise the
    telemetry format automatically. To receive all the frequencies and modes at
    the same time, you need to use an SDR receiver. The receivers
    `lilacsat_fcdpp` and `lilacsat_rtlsdr` can be used with a FUNcube Dongle
    Pro+ and an RTL-SDR respectively. These are complete receivers and
    decoders. They submit telemetry to the SatNOGS database and can use Doppler
    correction with Gpredict, in the same way as the frontends from
    gr-frontends. For `lilacsat_fcdpp` and `lilacsat_rtlsdr`,
    when using Doppler correction with Gpredict, you have to set
    437.200MHz as the downlink frequency in Gpredict.
  * `lucky_7`
    [Lucky-7](http://lucky7satellite.org/),
    which transmits 4k8 GFSK telemetry in the 70cm band. It uses a SiLabs
    Si4463 transceiver with a PN9 scrambler and a CRC-16. You must
    use FM mode to receive this satellite (437.525MHz).
  * `lume1`
    [LUME-1](https://space.skyrocket.de/doc_sdat/lume-1.htm),
    which transmits
    4k8 GMSK telemetry in the 70cm band. It uses the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. You must use FM mode to receive this satellite (437.060MHz).
  * `luojia_1`
    [Luojia-1](https://space.skyrocket.de/doc_sdat/luojia-1.htm),
    which transmits
    4k8 GMSK telemetry in the 70cm band. It uses the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. You must use FM mode to receive this satellite (437.250 MHz).
  * `mysat1`
    [MYSAT 1](https://space.skyrocket.de/doc_sdat/mysat-1.htm),
    which transmits 1k2 or 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. For 1k2 telemetry you must use SSB mode, while for 9k6 telemetry you
    must use wide SSB mode (435.77MHz).
  * `nayif1`
    [Nayif-1 (FUNcube-5)](https://amsat-uk.org/satellites/communications/nayif-1/),
    which transmits 1k2 BPSK
    telemetry in the 2m band. It uses the AO-40 FEC protocol, which includes
    block interleaving, an r=1/2, k=7 convolutional code, CCSDS scrambling and
    two interleaved (160,128) Reed-Solomon codes. You must use SSB mode to
    receive this satellite (145.940MHz).
  * `nusat`
    [ÑuSat-1 and -2](http://space.skyrocket.de/doc_sdat/nusat-1.htm),
    which transmit 40k FSK telemetry in the 70cm band
    (ÑuSat-1 on 436.445, ÑuSat-2 on 437.455).
    They use FEC with a
    (64, 60) Reed-Solomon code and a CRC-8. Since a sample rate of 48kHz is too
    low to work with 40k FSK, the flowgraph is prepared to use an IQ recording
    at 192kHz. Depending on the characteristics of your IQ recording you may
    need to edit the flowgraph. The Reed-Solomon decoder is taken from the
    [rscode](http://rscode.sourceforge.net/) library.
    A sample IQ recording is
    included in [satellite-recordings](https://github.com/daniestevez/satellite-recordings).
  * `ops_sat`
    [OPS-SAT](https://opssat1.esoc.esa.int/),
    which transmits 9k6 AX.25 telemetry in the 70cm band. It uses a G3RUH
    scrambler and the frames are encoded using a CCSDS Reed-Solomon code and a
    CCSDS scrambler. You must use FM mode to receive this satellite (437.200 MHz).
  * `picsat`
    [PicSat](http://picsat.obspm.fr/),
    which transmits 1k2 or 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. For 1k2 telemetry you must use SSB mode, while for 9k6 telemetry you
    must use wide SSB mode (435.525MHz).
  * `pwsat2`
    [PW-Sat2](https://pw-sat.pl/en/home-page/),
    which transmits 1k2, 2k4, 4k8 or 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. Currently the decoder only supports 1k2 or 9k6, since it is not clear
    if the other baudrates will be ever used.
    Telemetry parsing is supported using the external PW-Sat2 software. See
    [here](https://destevez.net/2018/12/decoding-pw-sat2-with-gr-satellites/) for the
    instructions. Uploading to the [PW-Sat2 Telemetry Server](http://radio.pw-sat.pl/) is
    also supported. See
    [here](https://destevez.net/2018/12/uploading-pw-sat2-telemetry-with-gr-satellites/)
    for the instructions.
    For 1k2 telemetry you must use SSB mode, while for 9k6 telemetry you
    must use wide SSB mode (435.275MHz).
  * `qo100`
    [QO-100 (Es'hail 2 AMSAT Phase
    4-A)](https://amsat-dl.org/eshail-2-amsat-phase-4-a),
    which transmits a 400baud BPSK beacon in the 3cm band. The beacon is relayed
    from the control centre in Qatar via a 13cm/3cm linear transponder on the satellite.
    The format of the telemetry is the same as that of the AO-40
    beacon. Uncoded and FEC frames are sent alternatively.
    You must use SSB mode to receive this satellite (10489.800MHz).
  * `reaktor_hello_world`
    [Reaktor Hello World](https://reaktorspace.com/reaktor-hello-world/),
    which transmits 9k6 GFSK telemetry in the 70cm band. It uses a Texas
    Intruments CC1125 transceiver with a PN9 scrambler and a CRC-16. You must
    use FM mode to receive this satellite (437.775MHz).
  * `shaonian_xing`
    [Shaonian Xing (MXSat-1)](https://space.skyrocket.de/doc_sdat/shaonian-xing.htm),
    which transmits 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. You must use wide SSB mode to receive this satellite (436.375MHz).
  * `smog_p`
    [SMOG-P](https://space.skyrocket.de/doc_sdat/smog-p.htm), which
    transmits 1.25k, 2.5k, 5k or 12.5k FSK telemetry in the 70cm band. It uses the AO-40 FEC protocol, which includes
    block interleaving, an r=1/2, k=7 convolutional code, CCSDS scrambling and
    two interleaved (160,128) Reed-Solomon codes. You must use FM mode
    to receive this satellite (437.150MHz).
  * `snet`
    [S-NET A,B,C,D](http://www.raumfahrttechnik.tu-berlin.de/menue/forschung/aktuelle_projekte/s-net/),
    which transmit 1k2 AFSK telemetry in the 70cm band. They use a custom coding
    with BCH FEC and interleaving. You must use FM mode to receive these
    satellites (435.950MHz).
  * `sokrat`
    [Sokrat](https://space.skyrocket.de/doc_sdat/sokrat.htm),
    which transmits 4k8 FSK telemetry in the 70cm band. It uses the Mobitex
    protocol, which includes FEC with a (12,8,3) linear code and CRC-16CCITT for
    error detection.
    The decoder requires the [beesat-sdr](https://github.com/daniestevez/beesat-sdr) OOT module.
    You must use FM mode to receive this satellite (436.000MHz).
  * `spooqy_1`
    [SpooQy-1](https://sites.google.com/site/phylej/projects-1/spooqy-1),
    which transmits
    9k6 GMSK telemetry in the 70cm band. It uses the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. The telemetry format is unknown. You must use FM mode to
    receive these satellites (436.200MHz).
  * `suomi_100`
    [Suomi 100](http://www.suomi100satelliitti.fi/),
    which transmits
    9k6 GMSK telemetry in the 70cm band. It uses the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. You must use FM mode to
    receive this satellite (437.775MHz).
  * `swiatowid`
    [Światowid](https://satrevolution.com/projects/swiatowid/),
    which transmits 9k6 FSK image data in the 70cm band. It uses a custom
    protocol with a (58,48) Reed-Solomon code and a CRC-16CCITT. You must use FM
    mode to receive this satellite (435.500MHz). Światowid also transmits
    standard 1k2 AFSK AX.25 APRS in the same frequency. The APRS data can be
    received with the `generic_1k2_afsk_ax25` decoder.
  * `tanusha3_pm`
    [TANUSHA-3](https://swsu.ru/space/),
    which transmits FM audio, 1k2 AX.25 AFSK telemetry and 1k2 audio frequency
    phase modulation AX.25 telemetry in the 70cm band. This decoder is for the
    phase modulation packets. For the AFSK packets you can use any regular
    packet decoder such as direwolf. You must use FM mode to receive this
    satellite (437.050MHz).
  * `taurus1`
    [Taurus-1](https://space.skyrocket.de/doc_sdat/taurus-1.htm),
    which transmits 9k6
    BPSK telemetry in the 70cm band. It uses FEC with an r=1/2, k=7
    convolutional code an interleaved telemetry and Codec2 digital voice.
    You must use wide SSB mode to receive this satellite
    (435.840MHz).
  * `tw_1a`, `tw_1b`, `tw_1c`
    [TW-1A](http://space.skyrocket.de/doc_sdat/shangkeda-2.htm),
    [TW-1B](http://space.skyrocket.de/doc_sdat/njust-2.htm),
    [TW-1C](http://space.skyrocket.de/doc_sdat/njfa-1.htm), which transmit 4k8
    GFSK telemetry in the 70cm band. They use the CSP protocol and FEC with a
    (255,223) Reed-Solomon code. They also use a G3RUH scrambler. The
    transceiver is the GomSpace NanoCom AX100, the same transceiver used in
    GOMX-3. There is no beacon parser yet, as the beacon format is unknown.
    The only difference between the 3 receivers is that the NORAD ID is set for
    the correct satellite when doing telemetry submission. You must use FM mode
    to receive these satellites. TW-1A, TW-1C (435.645 MHz), TW-1B (437.645 MHz).
  * `ty_2`, `ty_4`, `ty_6`
    [TY-2](https://space.skyrocket.de/doc_sdat/ty-2.htm),
    [TY 4-01](https://space.skyrocket.de/doc_sdat/ty-4-01.htm),
    and [TY-6](http://space.skyrocket.de/doc_sdat/ty-6.htm), which transmit
    9k6 GMSK telemetry in the 70cm band. They use the GomSpace NanoCom AX100
    transceiver in ASM+Golay mode. This uses a CCSDS scrambler and a (255,223)
    Reed-Solomon code. The telemetry format is unknown. The only
    difference between the 2 receivers is that the NORAD ID is set for the
    correct satellite when doing telemetry submission. You must use FM mode to
    receive these satellites. TY-2 (435.350MHz), TY 4-01 (435.925MHz), TY-6 (436.100MHz).
  * `ua01`
    [QB50 UA01 (PolyITAN-2-SAU)](http://space.skyrocket.de/doc_sdat/polyitan-2-sau.htm),
    which transmits 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler and two stages of NRZI coding.
    You must use wide SSB mode to receive this satellite (436.600MHz).
  * `ukube1`
    [UKube-1 (FUNcube-2)](https://amsat-uk.org/satellites/communications/ukube-1/),
    which transmits 1k2 BPSK
    telemetry in the 2m band. It uses the AO-40 FEC protocol, which includes
    block interleaving, an r=1/2, k=7 convolutional code, CCSDS scrambling and
    two interleaved (160,128) Reed-Solomon codes. You must use SSB mode to
    receive this satellite (145.915MHz).
  * `zhou_enlai`
    [Zhou Enlai](https://space.skyrocket.de/doc_sdat/zhou_enlai.htm),
    which transmits 9k6 AX.25 BPSK telemetry in the 70cm band. It uses a G3RUH
    scrambler. You must use wide SSB mode to receive this satellite (437.644MHz).

## Generic decoders

I do not add specific decoders to gr-satellites for satellites using standard
AFSK or FSK AX.25 packet radio, since there are many satellites using these
modes and there are already very good decoders for packet radio such as
[direwolf](https://github.com/wb2osz/direwolf). However, in case someone finds
it useful, I have added `generic_4k8_fsk_ax25`, `generic_9k6_fsk_ax25` and
`generic_19k2_fsk_ax25` generic decoders for 4k8, 9k6 and 19k2 FSK AX.25 packet
radio. Additionally, there is `generic_1k2_afsk_ax25` for 1k2 AFSK AX.25 packet
radio. Note that the performance of the AFSK decoder may not be optimal due to
improvable aspects in the signal processing (designing a good AGC is a
challenge, for instance).

## CCSDS Basic Blue Books
Descriptions for these blocks have been added in [CCSDS_README.md](https://github.com/athatheo/gr-satellites/blob/master/CCSDS_README.md) file.

## Submitting telemetry to SatNOGS

To sumbit telemetry to the [SatNOGS database](http://db.satnogs.org/) (or
another SiDS telemetry server), you have to specify your callsign and
coordinates. The callsign is specified using the `--callsign` parameter and the
latitude and longitude are specified using the `--latitude` and `--longitude`
parameters if you are using the `.py` script. If you are using the `.grc` file
with `gnuradio-companion`, you can set these parameters by editing the boxes on
the upper part of the flowgraph.

The format for the latitude and longitude is of the form `00.00000` or
`-00.00000`. The `-` means South (for latitude) or West (for longitude).

If you want to submit telemetry from a recording, you have to specify the UTC
date and time when the recording was started. This allows the decoder to compute
the proper timestamp for the packets. The format is `YYYY-MM-DD HH:MM:SS` and it
is specified using `--recstart` if using the `.py` script or with the parameter
box on the upper part of the flowgrah if using the `.grc` file with
`gnuradio-companion`.

It is also **very important** that the decoder and the recording streamer are
started simultaneously. This can be achieved by something like
```bash
gr-frontends/wav_48kHz.py -f recording.wav & \
gr-satellites/sat_3cat2.py --recstart="2016-01-01 00:00" --callsign=N0CALL --latitude=0.000 --longitude=0.000
```

## KISS submitter

There are many satellites that use standard packet radio AX.25 and can be
received with any software TNC such as
[Direwolf](https://github.com/wb2osz/direwolf). gr-satellites includes
`kiss_submitter` to perform telemetry submission when using a software TNC.

`kiss_submitter` connects to the software TNC as a KISS TCP client. The frames
received by the software TNC will be submitted by `kiss_submitter`. To use
`kiss_submitter`, you must specify your callsign and coordinates as when
submitting telemetry using any of the decoders. You also need to specify the
NORAD ID of the satellite you are receiving. This can be done by setting using
`--norad` if using the `.py` script or with the parameter if using the `.grc`
file. It is very important that you set the NORAD ID correctly. You can search
the NORAD ID in [celestrak](http://celestrak.com/satcat/search.asp).

You must start the software TNC first and the run the `.py` script or the `.grc`
file for `kiss_submitter`.

## Submitting telemetry to FUNcube

The flowgraphs for the different FUNcube satellites/payloads also support
submitting telemetry to the FUNcube server. To use this, you need to obtain the
"Site Id" (your username) and "Auth code" from your account on the FUNcube
server. These parameters can then be indicated by using the `--site-id` and
`--auth-code` if using the `.py` script or by editing the boxes in the lower
right part of the flowgraph if using the `.grc` file.

## Submitting telemetry to HIT severs (LilacSat, BY70-1, etc.)

It is also possible to use the flowgraphs in gr-satellites to submit telemetry
to the Harbin Institute of Technology servers using `proxy_publish.py` in
`gr-lilacsat/examples/proxy_publish`. To enable this, you must open the
flowgraphs in `gnuradio-companion` and enable the "Socket PDU" block (usually on
the lower right corner of the flowgraph). This block is disabled by default
because when it is enabled the flowgraph won't run unless `proxy_publish.py` is
running. Also see [this information](http://lilacsat.hit.edu.cn/?p=559) about
how to set the proper ports in `proxy_publish.py`.

## Submitting telemetry to BME (SMOG-P, ATL-1, SMOG-1)

To submit telemetry to the [BME server](https://gnd.bme.hu:8080/) you first need
to register an account there. Then modify the "BME Telemetry Forwarder" in the
GRC flowgraph to include your username and password.

## Submitting telemetry from EQUiSat

The EQUiSat flowgraph also supports submitting telemetry to the Brown Space 
Engineering data server. To use this, you need to obtain an API key from the
server. This can be done simply with the following command:

        curl -s -X POST http://api.brownspace.org/equisat/generate-key

The API key returned from this can then be specified using the `--api-key` parameter 
if using the `.py` script or by editing the box in the upper middle part of the 
flowgraph if using the `.grc` file.

You can also specify similar submit metadata as the SatNOGS submitter (above), 
such as your callsign and location. See `python equisat.py --help` for all the options.

## Hints for receiving different modes

### Wide SSB

Some modes (9k6 BPSK, for instance) need to be received using SSB mode, but the
bandwidth of the signal is larger than the usual 3kHz bandwidth of a
conventional SSB receiver. Therefore, an SDR receiver or a heavily modified
conventional SSB receiver is needed (a 9k6 BPKS signal is about 15kHz wide).

The decoders for satellites using these kind of *wide SSB* signals expect the
signal to be centred at an audio frequency of 12kHz. This means that you have to
dial in USB mode to a frequency 12kHz lower than the nominal frequency of the
satellite (+/- Doppler). If your SDR program allows this (gqrx does), the best
idea is to set an SSB audio filter from 0Hz to 24kHz and then tune the signal in
the middle of the passband. Alternatively, you can use the `--bfo` parameter if
using the `.py` file or edit the corresponding parameter in the `.grc` file to
use a frequency different from 12kHz.

If you are using the wide SSB receivers from
[gr-frontends](https://github.com/daniestevez/gr-frontends) you don't need to do
anything special, as these receivers already dial in USB mode to a frequency
12kHz than the nominal and use a 24kHz wide audio filter.

### Receiving FSK and sideband inversion

We are all used to the two SSB modes: USB (which is sideband-preserving) and LSB
(which is sideband-inverting). When receiving FM (or FSK), there is the same
concept. An FM receiver can be sideband-preserving or sideband-inverting. This
makes no difference when receiving analog FM (both sound the same) or AX.25
(which uses a differential protocol).

However, some satellites which use FSK (AAUSAT-4 and GOMX-3, for instance) need
a sideband-preserving FM receiver. If your receiver is sideband-inverting, you
can use set `--invert=-1` while running the `.py` file or edit the corresponding
parameter in the `.grc` file to invert the signal again in the decoder and
recover the original signal with the correct sidebands.

## Other hints

To run the decoder and save the output to a file, it is possible to do something
like

```bash
python2 -u aausat_4.py | tee /tmp/aausat4.log
```

This will both print the beacons in real time and also save all the output to
the text file `/tmp/aausat4.log`.
