/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 *
 * This file is part of GNU Radio
 *
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * gr-baz by Balint Seeber (http://spench.net/contact)
 * Information, documentation & samples: http://wiki.spench.net/wiki/gr-baz
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <baz_block_status.h>
#include <gnuradio/io_signature.h>

#include <stdio.h>

/*
 * Create a new instance of baz_block_status and return
 * a boost shared_ptr.  This is effectively the public constructor.
 */
baz_block_status_sptr
baz_make_block_status (int size, unsigned long work_iterations, unsigned long samples_processed, gr::msg_queue::sptr queue)
{
  return baz_block_status_sptr (new baz_block_status (size, queue, work_iterations, samples_processed));
}

/*
 * Specify constraints on number of input and output streams.
 * This info is used to construct the input and output signatures
 * (2nd & 3rd args to gr::block's constructor).  The input and
 * output signatures are used by the runtime system to
 * check that a valid number and type of inputs and outputs
 * are connected to this block.  In this case, we accept
 * only 1 input and 1 output.
 */
static const int MIN_IN = 1;	// mininum number of input streams
static const int MAX_IN = 1;	// maximum number of input streams
static const int MIN_OUT = 0;	// minimum number of output streams
static const int MAX_OUT = 1;	// maximum number of output streams

/*
 * The private constructor
 */
baz_block_status::baz_block_status (int size, gr::msg_queue::sptr queue, unsigned long work_iterations, unsigned long samples_processed)
  : gr::sync_block ("block_status",
		   gr::io_signature::make (MIN_IN, MAX_IN, size),
		   gr::io_signature::make (MIN_OUT, MAX_OUT, size))
  , d_size(size), d_queue(queue), d_work_iterations(work_iterations), d_samples_processed(samples_processed)
{
  fprintf(stderr, "[%s] Size: %d, work iterations: %lu, samples processed: %lu\n", name().c_str(), size, work_iterations, samples_processed);
}

/*
 * Our virtual destructor.
 */
baz_block_status::~baz_block_status ()
{
}

void baz_block_status::set_size(int size)
{
  d_size = size;
}

int
baz_block_status::work (int noutput_items,
			gr_vector_const_void_star &input_items,
			gr_vector_void_star &output_items)
{
  const char *in = (const char*) input_items[0];
  char* out = NULL;
  if (output_items.size() > 0)
    out = (char*)output_items[0];

  for (int i = 0; i < noutput_items; i++) {
    if (out != NULL)
      memcpy(out + (i * d_size), in + (i * d_size), d_size);

    if (d_samples_processed > 0)
    {
      --d_samples_processed;
      if (d_samples_processed == 0)
      {
fprintf(stderr, "[%s] Status change: samples processed\n", name().c_str()/*, d_work_iterations, d_samples_processed*/);
        if (d_queue)
        {
          gr::message::sptr msg = gr::message::make(0, d_work_iterations, d_samples_processed);
          d_queue->insert_tail(msg);
        }
      }
    }
  }

  if (d_work_iterations > 0)
  {
    --d_work_iterations;
    if (d_work_iterations == 0)
    {
fprintf(stderr, "[%s] Status change: work iterations\n", name().c_str()/*, d_work_iterations, d_samples_processed*/);
      if (d_queue)
      {
        gr::message::sptr msg = gr::message::make(0, d_work_iterations, d_samples_processed);
        d_queue->insert_tail(msg);
      }
    }
  }

  return noutput_items;
}
