/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 *
 * This file is part of GNU Radio
 *
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * gr-baz by Balint Seeber (http://spench.net/contact)
 * Information, documentation & samples: http://wiki.spench.net/wiki/gr-baz
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <baz_native_callback.h>
#include <gr_io_signature.h>

#include <stdio.h>

/*
 * Create a new instance of baz_native_callback_x and return
 * a boost shared_ptr.  This is effectively the public constructor.
 */
baz_native_callback_x_sptr
baz_make_native_callback_x (int size, baz_native_callback_target_sptr target, bool threshold_enable/*=false*/, float threshold_level/*=0.0*/)
{
  return baz_native_callback_x_sptr (new baz_native_callback_x (size, target, threshold_enable, threshold_level));
}

/*
 * Specify constraints on number of input and output streams.
 * This info is used to construct the input and output signatures
 * (2nd & 3rd args to gr_block's constructor).  The input and
 * output signatures are used by the runtime system to
 * check that a valid number and type of inputs and outputs
 * are connected to this block.  In this case, we accept
 * only 1 input and 1 output.
 */
static const int MIN_IN = 1;	// mininum number of input streams
static const int MAX_IN = 1;	// maximum number of input streams
static const int MIN_OUT = 0;	// minimum number of output streams
static const int MAX_OUT = 0;	// maximum number of output streams

/*
 * The private constructor
 */
baz_native_callback_x::baz_native_callback_x (int size, baz_native_callback_target_sptr target, bool threshold_enable, float threshold_level)
  : gr_sync_block ("native_callback_x",
		   gr_make_io_signature (MIN_IN, MAX_IN, size),
		   gr_make_io_signature (MIN_OUT, MAX_OUT, 0))
  , d_size(size), d_target(target), d_threshold_enable(threshold_enable), d_threshold_level(threshold_level)
  , d_triggered(false), d_samples_processed(0)
{
  fprintf(stderr, "[%s] Size: %d, threshold enabled: %s (%.1f)\n", name().c_str(), size, (threshold_enable ? "yes" : "no"), threshold_level);
}

/*
 * Our virtual destructor.
 */
baz_native_callback_x::~baz_native_callback_x ()
{
}

void baz_native_callback_x::set_size(int size)
{
  d_size = size;
}

void baz_native_callback_x::set_target(baz_native_callback_target_sptr target)
{
    d_target = target;
}

void baz_native_callback_x::set_threshold_enable(bool enable)
{
  d_threshold_enable = enable;
}

void baz_native_callback_x::set_threshold_level(float threshold_level)
{
  d_threshold_level = threshold_level;
}

int
baz_native_callback_x::work (int noutput_items,
			gr_vector_const_void_star &input_items,
			gr_vector_void_star &output_items)
{
  const float *in = (const float*) input_items[0];  // FIXME

  for (int i = 0; i < noutput_items; i++) {
    bool trigger = false;

    if ((d_triggered == false) && (in[i] >= d_threshold_level)) {
      trigger = d_triggered = true;
      //fprintf(stderr, "[%s] Triggered\n", name().c_str());
    }
    else if ((d_triggered) && (in[i] < d_threshold_level)) {
      d_triggered = false;
    }

    if ((d_threshold_enable == false) || (trigger)) {
      dynamic_cast<baz_native_callback_target*>(d_target.get())->callback(in[i], d_samples_processed); // FIXME: Could use return value for one-shot/continuous/re-trigger
    }

    ++d_samples_processed;
  }

  return noutput_items;
}
