# -*- coding: utf-8 -*- #
# Copyright 2021 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Functionality related to Cloud Run Integration API clients."""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function
from __future__ import unicode_literals

from frozendict import frozendict
from googlecloudsdk.command_lib.run import exceptions

_INTEGRATION_TYPES = frozenset([
    frozendict({
        'name':
            'custom-domain',
        'resource_name':
            'router',
        'description':
            'Configure a custom domain for Cloud Run services with Google Cloud '
            'Load Balancer.',
        'example_command':
            '$ gcloud run integration types create --type=custom-domain '
            '--parameters=domain=example.com',
        'parameters':
            frozenset([
                frozendict({
                    'name': 'domain',
                    'description':
                        'The domain to configure for your Cloud Run service. This '
                        'must be a domain you can configure DNS for.',
                    'type': 'domain',
                    'required': True,
                }),
                frozendict({
                    'name': 'dns-zone',
                    'description':
                        'The ID of the Cloud DNS Zone already configured for this '
                        'domain. If not specified, manual DNS configuration is '
                        'expected.',
                    'type': 'string',
                }),
                frozendict({
                    'name': 'paths',
                    'description':
                        'The paths at the domain for your Cloud Run service. '
                        'Defaults to "/" if not specified. (e.g. "/foo/*" for '
                        '"example.com/foo/*")',
                    'type': 'path_matcher',
                }),
            ]),
    }),
    frozendict({
        'name':
            'redis',
        'description':
            'Configure a Memorystore-Redis instance for Cloud Run Services.',
        'example_command':
            '$ gcloud run integration types create --type=redis',
        'parameters':
            frozenset([
                frozendict({
                    'name': 'memory-size-gb',
                    'description': 'Memory capacity of Redis instance.',
                    'type': 'int',
                    'default': 1,
                }),
                frozendict({
                    'name': 'tier',
                    'description':
                        'The service tier of the instance. '
                        'Supported options include BASIC for standalone '
                        'instance and STANDARD_HA for highly available '
                        'primary/replica instances.',
                    'type': 'string',
                }),
                frozendict({
                    'name': 'version',
                    'description':
                        'The version of Redis software. If not '
                        'provided, latest supported version will be used. '
                        'Supported values include: REDIS_6_X, REDIS_5_0, '
                        'REDIS_4_0 and REDIS_3_2.',
                    'type': 'string',
                }),
            ]),
    }),
])


def IntegrationTypes(client):
  """Gets the type definitions for Cloud Run Integrations.

  Currently it's just returning some builtin defnitions because the API is
  not implemented yet.

  Args:
    client: GAPIC API client, the api client to use.

  Returns:
    array of integration type.
  """
  del client
  return _INTEGRATION_TYPES


def CheckValidIntegrationType(int_type):
  """Checks if IntegrationType is supported.

  Args:
    int_type: str, integration type to validate.
  Rasies: ArgumentError
  """
  if int_type not in [
      integration['name'] for integration in _INTEGRATION_TYPES
  ]:
    raise exceptions.ArgumentError(
        'Integration type {} is not supported.'.format(int_type))
