# -*- coding: utf-8 -*- #
# Copyright 2021 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Delete trigger command."""

from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals

from googlecloudsdk.api_lib.cloudbuild import cloudbuild_util
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.cloudbuild import resource_args
from googlecloudsdk.command_lib.util.concepts import concept_parsers
from googlecloudsdk.core import log
from googlecloudsdk.core import properties
from googlecloudsdk.core import resources


@base.ReleaseTracks(base.ReleaseTrack.ALPHA, base.ReleaseTrack.BETA)
class Delete(base.DeleteCommand):
  """Delete a build trigger."""

  detailed_help = {
      'DESCRIPTION':
          'Delete a build trigger.',
      'EXAMPLES': ("""
        To delete a build trigger, run:

          $ {command} MY-TRIGGER
      """),
  }

  @staticmethod
  def Args(parser):
    concept_parsers.ConceptParser.ForResource(
        'TRIGGER',
        resource_args.GetTriggerResourceSpec(),
        'Build Trigger.',
        required=True).AddToParser(parser)

  def Run(self, args):
    """Deletes a build trigger.

    Args:
      args: an argparse namespace. All the arguments that were provided to this
        command invocation.

    Returns:
      Some value that we want to have printed later.
    """

    client = cloudbuild_util.GetClientInstance()

    project = properties.VALUES.core.project.GetOrFail
    location = args.region or cloudbuild_util.DEFAULT_REGION
    trigger = args.TRIGGER

    name = resources.REGISTRY.Parse(
        trigger,
        params={
            'projectsId': project,
            'locationsId': location,
            'triggersId': trigger,
        },
        collection='cloudbuild.projects.locations.triggers').RelativeName()

    client.projects_locations_triggers.Delete(
        client.MESSAGES_MODULE.CloudbuildProjectsLocationsTriggersDeleteRequest(
            name=name))

    log.DeletedResource(name)
