# -*- coding: utf-8 -*- #
# Copyright 2020 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""This class will store in-memory instance of ops agent policy."""

from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals

import enum
import json
from googlecloudsdk.calliope import exceptions
from googlecloudsdk.core.resource import resource_property


class OpsAgentPolicy(object):
  """An Ops Agent policy encapsulates the underlying OS Config Guest Policy."""

  class Agent(object):
    """An Ops Agent contains agent type, version, enable_autoupgrade."""

    class Type(str, enum.Enum):
      LOGGING = 'logging'
      METRICS = 'metrics'

    class PackageState(str, enum.Enum):
      INSTALLED = 'installed'
      REMOVED = 'removed'

    def __init__(self,
                 agent_type,
                 version='',
                 package_state=PackageState.INSTALLED,
                 enable_autoupgrade=False):
      """Initialize Agent instance.

      Args:
        agent_type: Type, agent type to be installed.
        version: str, agent version, e.g. 'latest', '5.5.2', '5.x.x'.
        package_state: Optional PackageState. DesiredState for the package.
        enable_autoupgrade: Optional bool. Enable autoupgrade for the package or
          not.
      """
      self.type = agent_type
      self.version = version
      self.package_state = package_state
      self.enable_autoupgrade = enable_autoupgrade

    def __eq__(self, other):
      return self.__dict__ == other.__dict__

    def ToJson(self):
      """Generate JSON with camel-cased key."""

      key_camel_cased_dict = {
          resource_property.ConvertToCamelCase(key): value
          for key, value in self.__dict__.items()
      }
      return json.dumps(key_camel_cased_dict, default=str, sort_keys=True)

  class Assignment(object):
    """The group or groups of VM instances that the policy applies to."""

    class OsType(object):
      """The criteria for selecting VM Instances by OS type."""

      class OsShortName(str, enum.Enum):
        CENTOS = 'centos'
        RHEL = 'rhel'
        DEBIAN = 'debian'

      def __init__(self, short_name, version, architecture=None):
        """Initialize OsType instance.

        Args:
          short_name: str, OS distro name, e.g. 'centos', 'debian'.
          version: str, OS version, e.g. '19.10', '7', '7.8'.
          architecture: Optional str, OS architecture, e.x. 'x86_64'.
        """
        self.short_name = short_name
        self.version = version
        self.architecture = architecture

      def __eq__(self, other):
        return self.__dict__ == other.__dict__

    def __init__(self, group_labels, zones, instances, os_types):
      """Initialize Assignment Instance.

      Args:
        group_labels: list of dict, VM group label matchers.
        zones: list, VM zone matchers.
        instances: list, instance name matchers.
        os_types: OsType, VM OS type matchers.
      """
      self.group_labels = group_labels
      self.zones = zones
      self.instances = instances
      self.os_types = os_types

    def __eq__(self, other):
      return self.__dict__ == other.__dict__

  def __init__(self,
               assignment,
               agents,
               description,
               etag=None,
               name=None,
               update_time=None,
               create_time=None):
    """Initialize an ops agent policy instance.

    Args:
      assignment: Assignment, selection criteria for applying policy to VMs.
      agents: list of Agent, the agent to be installed on VMs.
      description: str, user specified description of the policy.
      etag: Optional str, unique tag for policy, generated by the API.
      name: Optional str, user specified name of the policy.
      update_time: Optional str, update time in RFC3339 format.
      create_time: Optional str, create time in RFC3339 format.
    """
    self.assignment = assignment
    self.agents = agents
    self.description = description
    self.etag = etag
    self.id = name
    self.update_time = update_time
    self.create_time = create_time

  def __eq__(self, other):
    return self.__dict__ == other.__dict__


def CreateOpsAgentPolicy(description, agents, group_labels, os_types, zones,
                         instances):
  """Create Ops Agent Policy.

  Args:
    description: str, ops agent policy description.
    agents: list of dict, fields describing agents from the command line.
    group_labels: list of dict, VM group label matchers.
    os_types: dict, VM OS type matchers.
    zones: list, VM zone matchers.
    instances: list, instance name matchers.

  Returns:
    ops agent policy.
  """
  assignment_os_types = []
  for os_type in os_types:
    try:
      assignment_os_types.append(
          OpsAgentPolicy.Assignment.OsType(
              OpsAgentPolicy.Assignment.OsType.OsShortName(
                  os_type['short-name']), os_type['version'],
              os_type.get('architecture')))
    except KeyError as e:
      raise exceptions.BadArgumentException(
          'os_type',
          'os_type specification %s missing a required key: %s' % (os_type, e))
    except ValueError as e:
      raise exceptions.BadArgumentException(
          'os_type',
          'os_type specification %s contains invalid short-name: %s' %
          (os_type, e))
  assignment = OpsAgentPolicy.Assignment(group_labels, zones, instances,
                                         assignment_os_types)

  ops_agents = []
  for agent in agents:
    try:
      ops_agents.append(
          OpsAgentPolicy.Agent(
              OpsAgentPolicy.Agent.Type(agent['type']),
              agent.get('version', ''),
              OpsAgentPolicy.Agent.PackageState(
                  agent.get('package-state',
                            OpsAgentPolicy.Agent.PackageState.INSTALLED)),
              agent.get('enable-autoupgrade', False)))
    except KeyError as e:
      raise exceptions.BadArgumentException(
          'agent',
          'agent specification %s missing a required key: %s' % (agent, e))
    except ValueError as e:
      raise exceptions.BadArgumentException(
          'agent',
          'agent specification %s contains invalid enum value: %s' % (agent, e))
  return OpsAgentPolicy(assignment, ops_agents, description)
