# Copyright 2017 Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Command to analyze syntax."""

from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.ml.language import flags
from googlecloudsdk.command_lib.ml.language import language_command_util


class AnalyzeSyntax(base.Command):
  """Use Google Cloud Natural Language API to identify linguistic information.

  Syntactic Analysis extracts linguistic information, breaking up the given
  text into a series of sentences and tokens (generally, word boundaries),
  providing further analysis on those tokens.

  {service_account_help}

  {language_help}
  """

  detailed_help = {
      'service_account_help': language_command_util.SERVICE_ACCOUNT_HELP,
      'language_help': language_command_util.LANGUAGE_HELP
  }

  @staticmethod
  def Args(parser):
    flags.AddLanguageFlags(parser)

  def Run(self, args):
    """This is what gets called when the user runs this command.

    Args:
      args: an argparse namespace. All the arguments that were provided to this
        command invocation.

    Raises:
      ContentFileError: if content file can't be found and is not a Google
          Cloud Storage URL.
      ContentError: if content is given but empty.
      googlecloudsdk.api_lib.util.exceptions.HttpException: if the API returns
          an error.

    Returns:
      the result of the analyze syntax command.
    """
    return language_command_util.RunLanguageCommand(
        'extractSyntax',
        content_file=args.content_file,
        content=args.content,
        language=args.language,
        content_type=args.content_type,
        encoding_type=args.encoding_type)

  def Format(self, args):
    return 'json'
