package zson_test

import (
	"bytes"
	"errors"
	"math"
	"net/netip"
	"strings"
	"testing"

	"github.com/brimdata/zed"
	"github.com/brimdata/zed/pkg/nano"
	"github.com/brimdata/zed/zio"
	"github.com/brimdata/zed/zio/zngio"
	"github.com/brimdata/zed/zson"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/x448/float16"
)

func boomerang(t *testing.T, in interface{}, out interface{}) {
	zctx := zed.NewContext()
	arena := zed.NewArena()
	defer arena.Unref()
	rec, err := zson.MarshalZNG(zctx, arena, in)
	require.NoError(t, err)
	var buf bytes.Buffer
	zw := zngio.NewWriter(zio.NopCloser(&buf))
	err = zw.Write(rec)
	require.NoError(t, err)
	require.NoError(t, zw.Close())
	zr := zngio.NewReader(zctx, &buf)
	defer zr.Close()
	val, err := zr.Read()
	require.NoError(t, err)
	err = zson.UnmarshalZNG(zctx, arena, *val, out)
	require.NoError(t, err)
}

func TestMarshalZNG(t *testing.T) {
	arena := zed.NewArena()
	defer arena.Unref()
	type S2 struct {
		Field2 string `zed:"f2"`
		Field3 int
	}
	type S1 struct {
		Field1  string
		Sub1    S2
		PField1 *bool
	}
	rec, err := zson.MarshalZNG(zed.NewContext(), arena, S1{
		Field1: "value1",
		Sub1: S2{
			Field2: "value2",
			Field3: -1,
		},
	})
	require.NoError(t, err)
	require.NotNil(t, rec)
	assert.Equal(t, `{Field1:"value1",Sub1:{f2:"value2",Field3:-1},PField1:null(bool)}`, zson.FormatValue(rec))
}

func TestMarshalMap(t *testing.T) {
	type s struct {
		Name string
		Map  map[string]int
	}
	cases := []s{
		{Name: "nil", Map: nil},
		{Name: "empty", Map: map[string]int{}},
		{Name: "nonempty", Map: map[string]int{"a": 1, "b": 2}},
	}
	for _, c := range cases {
		c := c
		var v s
		t.Run(c.Name, func(t *testing.T) {
			boomerang(t, c, &v)
			assert.Equal(t, c, v)
		})
	}
}

type ZNGThing struct {
	A string `zed:"a"`
	B int
}

type ZNGThings struct {
	Things []ZNGThing
}

func TestMarshalSlice(t *testing.T) {
	arena := zed.NewArena()
	defer arena.Unref()
	m := zson.NewZNGMarshaler()
	m.Decorate(zson.StyleSimple)

	s := []ZNGThing{{"hello", 123}, {"world", 0}}
	r := ZNGThings{s}
	rec, err := m.Marshal(arena, r)
	require.NoError(t, err)
	require.NotNil(t, rec)
	assert.Equal(t, `{Things:[{a:"hello",B:123}(=ZNGThing),{a:"world",B:0}(ZNGThing)]}(=ZNGThings)`, zson.FormatValue(rec))

	empty := []ZNGThing{}
	r2 := ZNGThings{empty}
	rec2, err := m.Marshal(arena, r2)
	require.NoError(t, err)
	require.NotNil(t, rec2)
	assert.Equal(t, "{Things:[]([ZNGThing={a:string,B:int64}])}(=ZNGThings)", zson.FormatValue(rec2))

	rec3, err := m.Marshal(arena, ZNGThings{nil})
	require.NoError(t, err)
	require.NotNil(t, rec3)
	assert.Equal(t, "{Things:null([ZNGThing={a:string,B:int64}])}(=ZNGThings)", zson.FormatValue(rec3))

}

func TestMarshalNilSlice(t *testing.T) {
	type TestNilSlice struct {
		Name  string
		Slice []string
	}
	t1 := TestNilSlice{Name: "test"}
	var t2 TestNilSlice
	boomerang(t, t1, &t2)
	assert.Equal(t, t1, t2)
}

func TestMarshalEmptySlice(t *testing.T) {
	type TestNilSlice struct {
		Name  string
		Slice []string
	}
	t1 := TestNilSlice{Name: "test", Slice: []string{}}
	var t2 TestNilSlice
	boomerang(t, t1, &t2)
	assert.Equal(t, t1, t2)
}

func TestMarshalTime(t *testing.T) {
	type TestTime struct {
		Ts nano.Ts
	}
	t1 := TestTime{Ts: nano.Now()}
	var t2 TestTime
	boomerang(t, t1, &t2)
	assert.Equal(t, t1, t2)
}

type TestIP struct {
	Addr netip.Addr
}

func TestIPType(t *testing.T) {
	zctx := zed.NewContext()
	arena := zed.NewArena()
	defer arena.Unref()

	s := TestIP{Addr: netip.MustParseAddr("192.168.1.1")}
	rec, err := zson.MarshalZNG(zctx, arena, s)
	require.NoError(t, err)
	require.NotNil(t, rec)
	assert.Equal(t, "{Addr:192.168.1.1}", zson.FormatValue(rec))

	var tip TestIP
	err = zson.UnmarshalZNG(zctx, arena, rec, &tip)
	require.NoError(t, err)
	require.Equal(t, s, tip)
}

func TestUnmarshalRecord(t *testing.T) {
	zctx := zed.NewContext()
	arena := zed.NewArena()
	defer arena.Unref()

	type T3 struct {
		T3f1 int32
		T3f2 float32
	}
	type T2 struct {
		T2f1 T3
		T2f2 string
	}
	type T1 struct {
		T1f1 *T2 `zed:"top"`
	}
	v1 := T1{
		T1f1: &T2{T2f1: T3{T3f1: 1, T3f2: 1.0}, T2f2: "t2f2-string1"},
	}
	rec, err := zson.MarshalZNG(zctx, arena, v1)
	require.NoError(t, err)
	require.NotNil(t, rec)

	const expected = `{top:{T2f1:{T3f1:1(int32),T3f2:1.(float32)},T2f2:"t2f2-string1"}}`
	require.Equal(t, expected, zson.FormatValue(rec))

	val := zson.MustParseValue(zctx, arena, expected)
	var v2 T1
	err = zson.UnmarshalZNG(zctx, arena, val, &v2)
	require.NoError(t, err)
	require.Equal(t, v1, v2)

	type T4 struct {
		T4f1 *T2 `zed:"top"`
	}
	var v3 *T4
	err = zson.UnmarshalZNG(zctx, arena, rec, &v3)
	require.NoError(t, err)
	require.NotNil(t, v3)
	require.NotNil(t, v3.T4f1)
	require.Equal(t, *v1.T1f1, *v3.T4f1)
}

func TestUnmarshalNull(t *testing.T) {
	zctx := zed.NewContext()
	arena := zed.NewArena()
	defer arena.Unref()
	t.Run("slice", func(t *testing.T) {
		slice := []int{1}
		require.NoError(t, zson.UnmarshalZNG(zctx, arena, zed.Null, &slice))
		assert.Nil(t, slice)
		slice = []int{1}
		assert.EqualError(t, zson.UnmarshalZNG(zctx, arena, zed.NullInt64, &slice), `unmarshaling type "int64": not an array`)
		slice = []int{1}
		v := zson.MustParseValue(zctx, arena, "null([int64])")
		require.NoError(t, zson.UnmarshalZNG(zctx, arena, v, &slice))
		assert.Nil(t, slice)
		v = zson.MustParseValue(zctx, arena, "null(bytes)")
		buf := []byte("testing")
		require.NoError(t, zson.UnmarshalZNG(zctx, arena, v, &buf))
		assert.Nil(t, buf)
	})
	t.Run("primitive", func(t *testing.T) {
		integer := -1
		require.NoError(t, zson.UnmarshalZNG(zctx, arena, zed.Null, &integer))
		assert.Equal(t, integer, 0)
		intptr := &integer
		require.NoError(t, zson.UnmarshalZNG(zctx, arena, zed.Null, &intptr))
		assert.Nil(t, intptr)
		assert.EqualError(t, zson.UnmarshalZNG(zctx, arena, zed.NullIP, &intptr), "incompatible type translation: zng type ip go type int go kind int")
	})
	t.Run("map", func(t *testing.T) {
		m := map[string]string{"key": "value"}
		require.NoError(t, zson.UnmarshalZNG(zctx, arena, zed.Null, &m))
		assert.Nil(t, m)
		val := zson.MustParseValue(zctx, arena, "null({foo:int64})")
		require.EqualError(t, zson.UnmarshalZNG(zctx, arena, val, &m), "not a map")
		m = map[string]string{"key": "value"}
		val = zson.MustParseValue(zctx, arena, "null(|{string:string}|)")
		require.NoError(t, zson.UnmarshalZNG(zctx, arena, val, &m))
		assert.Nil(t, m)
	})
	t.Run("struct", func(t *testing.T) {
		type testobj struct {
			Val int
		}
		var obj struct {
			Test *testobj `zed:"test"`
		}
		val := zson.MustParseValue(zctx, arena, "{test:null({Val:int64})}")
		require.NoError(t, zson.UnmarshalZNG(zctx, arena, val, &obj))
		require.Nil(t, obj.Test)
		val = zson.MustParseValue(zctx, arena, "{test:null(ip)}")
		require.EqualError(t, zson.UnmarshalZNG(zctx, arena, val, &obj), `cannot unmarshal Zed value "null(ip)" into Go struct`)
		var slice struct {
			Test []string `zed:"test"`
		}
		slice.Test = []string{"1"}
		val = zson.MustParseValue(zctx, arena, "{test:null}")
		require.NoError(t, zson.UnmarshalZNG(zctx, arena, val, &slice))
		require.Nil(t, slice.Test)
	})
}

func TestUnmarshalSlice(t *testing.T) {
	zctx := zed.NewContext()
	arena := zed.NewArena()
	defer arena.Unref()

	type T1 struct {
		T1f1 []bool
	}
	v1 := T1{
		T1f1: []bool{true, false, true},
	}
	rec, err := zson.MarshalZNG(zctx, arena, v1)
	require.NoError(t, err)
	require.NotNil(t, rec)

	var v2 T1
	err = zson.UnmarshalZNG(zctx, arena, rec, &v2)
	require.NoError(t, err)
	require.Equal(t, v1, v2)

	type T2 struct {
		Field1 []*int
	}
	intp := func(x int) *int { return &x }
	v3 := T2{
		Field1: []*int{intp(1), intp(2)},
	}
	rec, err = zson.MarshalZNG(zctx, arena, v3)
	require.NoError(t, err)
	require.NotNil(t, rec)

	var v4 T2
	err = zson.UnmarshalZNG(zctx, arena, rec, &v4)
	require.NoError(t, err)
	require.Equal(t, v1, v2)
}

type testMarshaler string

func (m testMarshaler) MarshalZNG(mc *zson.MarshalZNGContext) (zed.Type, error) {
	return mc.MarshalValue("marshal-" + string(m))
}

func (m *testMarshaler) UnmarshalZNG(mc *zson.UnmarshalZNGContext, val zed.Value) error {
	var s string
	if err := mc.Unmarshal(val, &s); err != nil {
		return err
	}
	ss := strings.Split(s, "-")
	if len(ss) != 2 && ss[0] != "marshal" {
		return errors.New("bad value")
	}
	*m = testMarshaler(ss[1])
	return nil
}

func TestMarshalInterface(t *testing.T) {
	zctx := zed.NewContext()
	arena := zed.NewArena()
	defer arena.Unref()

	type rectype struct {
		M1 *testMarshaler
		M2 testMarshaler
	}
	m1 := testMarshaler("m1")
	r1 := rectype{M1: &m1, M2: testMarshaler("m2")}
	rec, err := zson.MarshalZNG(zed.NewContext(), arena, r1)
	require.NoError(t, err)
	require.NotNil(t, rec)
	assert.Equal(t, `{M1:"marshal-m1",M2:"marshal-m2"}`, zson.FormatValue(rec))

	var r2 rectype
	err = zson.UnmarshalZNG(zctx, arena, rec, &r2)
	require.NoError(t, err)
	assert.Equal(t, "m1", string(*r2.M1))
	assert.Equal(t, "m2", string(r2.M2))
}

func TestMarshalArray(t *testing.T) {
	zctx := zed.NewContext()
	arena := zed.NewArena()
	defer arena.Unref()

	type rectype struct {
		A1 [2]int8
		A2 *[2]string
		A3 [][2]byte
	}
	a2 := &[2]string{"foo", "bar"}
	r1 := rectype{A1: [2]int8{1, 2}, A2: a2} // A3 left as nil
	rec, err := zson.MarshalZNG(zctx, arena, r1)
	require.NoError(t, err)
	require.NotNil(t, rec)
	const expected = `{A1:[1(int8),2(int8)],A2:["foo","bar"],A3:null([bytes])}`
	assert.Equal(t, expected, zson.FormatValue(rec))

	var r2 rectype
	err = zson.UnmarshalZNG(zctx, arena, rec, &r2)
	require.NoError(t, err)
	assert.Equal(t, r1.A1, r2.A1)
	assert.Equal(t, *r2.A2, *r2.A2)
	assert.Len(t, r2.A3, 0)
}

func TestNumbers(t *testing.T) {
	zctx := zed.NewContext()
	arena := zed.NewArena()
	defer arena.Unref()

	type rectype struct {
		I    int
		I8   int8
		I16  int16
		I32  int32
		I64  int64
		U    uint
		UI8  uint8
		UI16 uint16
		UI32 uint32
		UI64 uint64
		F16  float16.Float16
		F32  float32
		F64  float64
	}
	r1 := rectype{
		I:    math.MinInt64,
		I8:   math.MinInt8,
		I16:  math.MinInt16,
		I32:  math.MinInt32,
		I64:  math.MinInt64,
		U:    math.MaxUint64,
		UI8:  math.MaxUint8,
		UI16: math.MaxUint16,
		UI32: math.MaxUint32,
		UI64: math.MaxUint64,
		F16:  float16.Fromfloat32(65504.0),
		F32:  math.MaxFloat32,
		F64:  math.MaxFloat64,
	}
	rec, err := zson.MarshalZNG(zctx, arena, r1)
	require.NoError(t, err)
	require.NotNil(t, rec)
	const expected = "{I:-9223372036854775808,I8:-128(int8),I16:-32768(int16),I32:-2147483648(int32),I64:-9223372036854775808,U:18446744073709551615(uint64),UI8:255(uint8),UI16:65535(uint16),UI32:4294967295(uint32),UI64:18446744073709551615(uint64),F16:65504.(float16),F32:3.4028235e+38(float32),F64:1.7976931348623157e+308}"
	assert.Equal(t, expected, zson.FormatValue(rec))

	var r2 rectype
	err = zson.UnmarshalZNG(zctx, arena, rec, &r2)
	require.NoError(t, err)
	assert.Equal(t, r1, r2)
}

func TestCustomRecord(t *testing.T) {
	arena := zed.NewArena()
	defer arena.Unref()

	vals := []interface{}{
		ZNGThing{"hello", 123},
		99,
	}
	m := zson.NewZNGMarshaler()
	rec, err := m.MarshalCustom(arena, []string{"foo", "bar"}, vals)
	require.NoError(t, err)
	assert.Equal(t, `{foo:{a:"hello",B:123},bar:99}`, zson.FormatValue(rec))

	vals = []interface{}{
		ZNGThing{"hello", 123},
		nil,
	}
	rec, err = m.MarshalCustom(arena, []string{"foo", "bar"}, vals)
	require.NoError(t, err)
	assert.Equal(t, `{foo:{a:"hello",B:123},bar:null}`, zson.FormatValue(rec))
}

type ThingTwo struct {
	C string `zed:"c"`
}

type ThingaMaBob interface {
	Who() string
}

func (t *ZNGThing) Who() string { return t.A }
func (t *ThingTwo) Who() string { return t.C }

func Make(which int) ThingaMaBob {
	if which == 1 {
		return &ZNGThing{A: "It's a thing one"}
	}
	if which == 2 {
		return &ThingTwo{"It's a thing two"}
	}
	return nil
}

type Rolls []int

func TestInterfaceZNGMarshal(t *testing.T) {
	arena := zed.NewArena()
	defer arena.Unref()

	t1 := Make(2)
	m := zson.NewZNGMarshaler()
	m.Decorate(zson.StylePackage)
	zv, err := m.Marshal(arena, t1)
	require.NoError(t, err)
	assert.Equal(t, "zson_test.ThingTwo={c:string}", zson.String(zv.Type()))

	m.Decorate(zson.StyleSimple)
	rolls := Rolls{1, 2, 3}
	zv, err = m.Marshal(arena, rolls)
	require.NoError(t, err)
	assert.Equal(t, "Rolls=[int64]", zson.String(zv.Type()))

	m.Decorate(zson.StyleFull)
	zv, err = m.Marshal(arena, rolls)
	require.NoError(t, err)
	assert.Equal(t, `"github.com/brimdata/zed/zson_test.Rolls"=[int64]`, zson.String(zv.Type()))

	plain := []int32{1, 2, 3}
	zv, err = m.Marshal(arena, plain)
	require.NoError(t, err)
	assert.Equal(t, "[int32]", zson.String(zv.Type()))
}

func TestInterfaceUnmarshal(t *testing.T) {
	arena := zed.NewArena()
	defer arena.Unref()

	t1 := Make(1)
	m := zson.NewZNGMarshaler()
	m.Decorate(zson.StylePackage)
	zv, err := m.Marshal(arena, t1)
	require.NoError(t, err)
	assert.Equal(t, "zson_test.ZNGThing={a:string,B:int64}", zson.String(zv.Type()))

	u := zson.NewZNGUnmarshaler()
	u.Bind(ZNGThing{}, ThingTwo{})
	var thing ThingaMaBob
	require.NoError(t, err)
	err = u.Unmarshal(zv, &thing)
	require.NoError(t, err)
	assert.Equal(t, "It's a thing one", thing.Who())

	var thingI interface{}
	err = u.Unmarshal(zv, &thingI)
	require.NoError(t, err, zson.String(zv))
	actualThing, ok := thingI.(*ZNGThing)
	assert.Equal(t, true, ok)
	assert.Equal(t, t1, actualThing)

	u2 := zson.NewZNGUnmarshaler()
	var genericThing interface{}
	err = u2.Unmarshal(zv, &genericThing)
	require.Error(t, err)
	assert.Equal(t, `unmarshaling records into interface value requires type binding`, err.Error())
}

func TestBindings(t *testing.T) {
	arena := zed.NewArena()
	defer arena.Unref()

	t1 := Make(1)
	m := zson.NewZNGMarshaler()
	m.NamedBindings([]zson.Binding{
		{"SpecialThingOne", &ZNGThing{}},
		{"SpecialThingTwo", &ThingTwo{}},
	})
	zv, err := m.Marshal(arena, t1)
	require.NoError(t, err)
	assert.Equal(t, "SpecialThingOne={a:string,B:int64}", zson.String(zv.Type()))

	u := zson.NewZNGUnmarshaler()
	u.NamedBindings([]zson.Binding{
		{"SpecialThingOne", &ZNGThing{}},
		{"SpecialThingTwo", &ThingTwo{}},
	})
	var thing ThingaMaBob
	require.NoError(t, err)
	err = u.Unmarshal(zv, &thing)
	require.NoError(t, err)
	assert.Equal(t, "It's a thing one", thing.Who())
}

func TestEmptyInterface(t *testing.T) {
	zctx := zed.NewContext()
	arena := zed.NewArena()
	defer arena.Unref()

	zv, err := zson.MarshalZNG(zctx, arena, int8(123))
	require.NoError(t, err)
	assert.Equal(t, "int8", zson.String(zv.Type()))

	var v interface{}
	err = zson.UnmarshalZNG(zctx, arena, zv, &v)
	require.NoError(t, err)
	i, ok := v.(int8)
	assert.Equal(t, true, ok)
	assert.Equal(t, int8(123), i)

	var actual int8
	err = zson.UnmarshalZNG(zctx, arena, zv, &actual)
	require.NoError(t, err)
	assert.Equal(t, int8(123), actual)
}

type CustomInt8 int8

func TestNamedNormal(t *testing.T) {
	arena := zed.NewArena()
	defer arena.Unref()

	t1 := CustomInt8(88)
	m := zson.NewZNGMarshaler()
	m.Decorate(zson.StyleSimple)

	zv, err := m.Marshal(arena, t1)
	require.NoError(t, err)
	assert.Equal(t, "CustomInt8=int8", zson.String(zv.Type()))

	var actual CustomInt8
	u := zson.NewZNGUnmarshaler()
	u.Bind(CustomInt8(0))
	err = u.Unmarshal(zv, &actual)
	require.NoError(t, err)
	assert.Equal(t, t1, actual)

	var actualI interface{}
	err = u.Unmarshal(zv, &actualI)
	require.NoError(t, err)
	cast, ok := actualI.(CustomInt8)
	assert.Equal(t, true, ok)
	assert.Equal(t, t1, cast)
}

type EmbeddedA struct {
	A ThingaMaBob
}

type EmbeddedB struct {
	A interface{}
}

func TestEmbeddedInterface(t *testing.T) {
	arena := zed.NewArena()
	defer arena.Unref()

	t1 := &EmbeddedA{
		A: Make(1),
	}
	m := zson.NewZNGMarshaler()
	m.Decorate(zson.StyleSimple)
	zv, err := m.Marshal(arena, t1)
	require.NoError(t, err)
	assert.Equal(t, "EmbeddedA={A:ZNGThing={a:string,B:int64}}", zson.String(zv.Type()))

	u := zson.NewZNGUnmarshaler()
	u.Bind(ZNGThing{}, ThingTwo{})
	var actual EmbeddedA
	require.NoError(t, err)
	err = u.Unmarshal(zv, &actual)
	require.NoError(t, err)
	assert.Equal(t, "It's a thing one", actual.A.Who())

	var actualB EmbeddedB
	require.NoError(t, err)
	err = u.Unmarshal(zv, &actualB)
	require.NoError(t, err)
	thingB, ok := actualB.A.(*ZNGThing)
	assert.Equal(t, true, ok)
	assert.Equal(t, "It's a thing one", thingB.Who())
}

func TestMultipleZedValues(t *testing.T) {
	arena := zed.NewArena()
	defer arena.Unref()

	bytes := []byte("foo")
	u := zson.NewZNGUnmarshaler()
	u.SetContext(zed.NewContext(), arena)
	var foo zed.Value
	err := u.Unmarshal(arena.New(zed.TypeString, bytes), &foo)
	require.NoError(t, err)
	// clobber bytes slice
	copy(bytes, []byte("bar"))
	var bar zed.Value
	err = u.Unmarshal(arena.New(zed.TypeString, bytes), &bar)
	require.NoError(t, err)
	assert.Equal(t, "foo", string(foo.Bytes()))
	assert.Equal(t, "bar", string(bar.Bytes()))
}

func TestZedValues(t *testing.T) {
	test := func(t *testing.T, name, s string, v interface{}) {
		t.Run(name, func(t *testing.T) {
			zctx := zed.NewContext()
			arena := zed.NewArena()
			defer arena.Unref()
			val := zson.MustParseValue(zctx, arena, s)
			err := zson.UnmarshalZNG(zctx, arena, val, v)
			require.NoError(t, err)
			val, err = zson.MarshalZNG(zctx, arena, v)
			require.NoError(t, err)
			assert.Equal(t, s, zson.FormatValue(val))
		})
	}
	var testptr struct {
		Value *zed.Value `zed:"value"`
	}
	t.Run("pointer", func(t *testing.T) {
		test(t, "string", "{value:\"foo\"}", &testptr)
		test(t, "typed-null", "{value:null(time)}", &testptr)
		test(t, "null", "{value:null}", &testptr)
		test(t, "record", "{value:{foo:1,bar:\"baz\"}}", &testptr)
	})
	var teststruct struct {
		Value zed.Value `zed:"value"`
	}
	t.Run("struct", func(t *testing.T) {
		test(t, "string", "{value:\"foo\"}", &teststruct)
		test(t, "typed-null", "{value:null(time)}", &teststruct)
		test(t, "null", "{value:null}", &teststruct)
		test(t, "record", "{value:{foo:1,bar:\"baz\"}}", &teststruct)
	})
}
