//go:build ignore

package main

import (
	"bytes"
	"fmt"
	"go/format"
	"log"
	"os"
	"strings"

	"github.com/brimdata/zed/vector"
)

func main() {
	var buf bytes.Buffer
	fmt.Fprintln(&buf, "// Code generated by gencomparefuncs.go. DO NOT EDIT.")
	fmt.Fprintln(&buf)
	fmt.Fprintln(&buf, "package expr")
	fmt.Fprintln(&buf, "import (")
	fmt.Fprintln(&buf, `"bytes"`)
	fmt.Fprintln(&buf, `"strings"`)
	fmt.Fprintln(&buf) // Put subsequent imports in a new section.
	fmt.Fprintln(&buf, `"github.com/brimdata/zed/vector"`)
	fmt.Fprintln(&buf, ")")

	var ents strings.Builder
	for _, op := range []string{"==", "!=", "<", "<=", ">", ">="} {
		for _, typ := range []string{"Int", "Uint", "Float", "String", "Bytes"} {
			for lform := vector.Form(0); lform < 3; lform++ {
				for rform := vector.Form(0); rform < 3; rform++ {
					if lform == vector.FormConst && rform == vector.FormConst {
						// no const x const
						continue
					}
					fmt.Fprintln(&buf, genFunc(op, typ, lform, rform))
					fmt.Fprintf(&ents, "%d: %s,\n", vector.CompareOpCode(op, vector.KindFromString(typ), lform, rform), funcName(op, typ, lform, rform))
				}
			}
		}
	}

	fmt.Fprintln(&buf, "var compareFuncs = map[int]func(vector.Any, vector.Any) *vector.Bool{")
	fmt.Fprintln(&buf, ents.String())
	fmt.Fprintln(&buf, "}")

	out, formatErr := format.Source(buf.Bytes())
	if formatErr != nil {
		// Write unformatted source so we can find the error.
		out = buf.Bytes()
	}
	const fileName = "comparefuncs.go"
	if err := os.WriteFile(fileName, out, 0644); err != nil {
		log.Fatal(err)
	}
	if formatErr != nil {
		log.Fatal(fileName, ":", formatErr)
	}
}

func genFunc(op, typ string, lhs, rhs vector.Form) string {
	var s string
	s += "func "
	s += funcName(op, typ, lhs, rhs)
	s += "(lhs, rhs vector.Any) *vector.Bool {\n"
	s += "n := lhs.Len()\n"
	s += "out := vector.NewBoolEmpty(n, nil)\n"
	s += genVarInit("l", typ, lhs)
	s += genVarInit("r", typ, rhs)
	s += strings.TrimLeft(formToLoop(typ, lhs, rhs, op), "\n")
	s += "return out\n"
	s += "}\n"
	return s
}

func genVarInit(which, typ string, form vector.Form) string {
	switch form {
	case vector.FormFlat:
		return fmt.Sprintf("%s := %shs.(*vector.%s)\n", which, which, typ)
	case vector.FormDict:
		s := fmt.Sprintf("%sd := %shs.(*vector.Dict)\n", which, which)
		s += fmt.Sprintf("%s := %sd.Any.(*vector.%s)\n", which, which, typ)
		s += fmt.Sprintf("%sx := %sd.Index\n", which, which)
		return s
	case vector.FormConst:
		return fmt.Sprintf("%sconst, _ := %shs.(*vector.Const).As%s()\n", which, which, typ)
	default:
		panic("genVarInit: bad form")
	}
}

const flat_flat = `
	for k := uint32(0); k < n; k++ {
		if l.Values[k] %s r.Values[k] {
			out.Set(k)
		}
	}
`

const flat_dict = `
	for k := uint32(0); k < n; k++ {
		if l.Values[k] %s r.Values[rx[k]] {
			out.Set(k)
		}
	}
`

const flat_const = `
	for k := uint32(0); k < n; k++ {
		if l.Values[k] %s rconst {
			out.Set(k)
		}
	}
`

const dict_flat = `
	for k := uint32(0); k < n; k++ {
		if l.Values[lx[k]] %s r.Values[k] {
			out.Set(k)
		}
	}
`

const dict_dict = `
	for k := uint32(0); k < n; k++ {
		if l.Values[lx[k]] %s r.Values[rx[k]] {
			out.Set(k)
		}
	}
`

const dict_const = `
	for k := uint32(0); k < n; k++ {
		if l.Values[lx[k]] %s rconst {
			out.Set(k)
		}
	}
`

const const_flat = `
	for k := uint32(0); k < n; k++ {
		if lconst %s r.Values[k] {
			out.Set(k)
		}
	}
`

const const_dict = `
	for k := uint32(0); k < n; k++ {
		if lconst %s r.Values[rx[k]] {
			out.Set(k)
		}
	}
`

const flat_flat_s = `
	for k := uint32(0); k < n; k++ {
		if %s.Compare(l.Value(k), r.Value(k)) %s 0 {
			out.Set(k)
		}
	}
`

const flat_dict_s = `
	for k := uint32(0); k < n; k++ {
		if %s.Compare(l.Value(k), r.Value(uint32(rx[k]))) %s 0 {
			out.Set(k)
		}
	}
`

const flat_const_s = `
	for k := uint32(0); k < n; k++ {
		if %s.Compare(l.Value(k), rconst) %s 0 {
			out.Set(k)
		}
	}
`

const dict_flat_s = `
	for k := uint32(0); k < n; k++ {
		if %s.Compare(l.Value(uint32(lx[k])), r.Value(k)) %s 0 {
			out.Set(k)
		}
	}
`

const dict_dict_s = `
	for k := uint32(0); k < n; k++ {
		if %s.Compare(l.Value(uint32(lx[k])), r.Value(uint32(rx[k]))) %s 0 {
			out.Set(k)
		}
	}
`

const dict_const_s = `
	for k := uint32(0); k < n; k++ {
		if %s.Compare(l.Value(uint32(lx[k])), rconst) %s 0 {
			out.Set(k)
		}
	}
`

const const_flat_s = `
	for k := uint32(0); k < n; k++ {
		if %s.Compare(lconst, r.Value(k)) %s 0 {
			out.Set(k)
		}
	}
`

const const_dict_s = `
	for k := uint32(0); k < n; k++ {
		if %s.Compare(lconst, r.Value(uint32(rx[k]))) %s 0 {
			out.Set(k)
		}
	}
`

func formToLoop(typ string, lform, rform vector.Form, op string) string {
	switch typ {
	case "String", "Bytes":
		return formToLoops(typ, lform, rform, op)
	default:
		return formToLoopv(lform, rform, op)
	}
}

func formToLoops(typ string, lform, rform vector.Form, op string) string {
	pkg := strings.ToLower(typ)
	if pkg == "string" {
		pkg = "strings"
	}
	switch lform {
	case vector.FormFlat:
		switch rform {
		case vector.FormFlat:
			return fmt.Sprintf(flat_flat_s, pkg, op)
		case vector.FormDict:
			return fmt.Sprintf(flat_dict_s, pkg, op)
		case vector.FormConst:
			return fmt.Sprintf(flat_const_s, pkg, op)
		}
	case vector.FormDict:
		switch rform {
		case vector.FormFlat:
			return fmt.Sprintf(dict_flat_s, pkg, op)
		case vector.FormDict:
			return fmt.Sprintf(dict_dict_s, pkg, op)
		case vector.FormConst:
			return fmt.Sprintf(dict_const_s, pkg, op)
		}
	case vector.FormConst:
		switch rform {
		case vector.FormFlat:
			return fmt.Sprintf(const_flat_s, pkg, op)
		case vector.FormDict:
			return fmt.Sprintf(const_dict_s, pkg, op)
		}
	}
	panic("formToLoop: bad logic")
}

func formToLoopv(lform, rform vector.Form, op string) string {
	switch lform {
	case vector.FormFlat:
		switch rform {
		case vector.FormFlat:
			return fmt.Sprintf(flat_flat, op)
		case vector.FormDict:
			return fmt.Sprintf(flat_dict, op)
		case vector.FormConst:
			return fmt.Sprintf(flat_const, op)
		}
	case vector.FormDict:
		switch rform {
		case vector.FormFlat:
			return fmt.Sprintf(dict_flat, op)
		case vector.FormDict:
			return fmt.Sprintf(dict_dict, op)
		case vector.FormConst:
			return fmt.Sprintf(dict_const, op)
		}
	case vector.FormConst:
		switch rform {
		case vector.FormFlat:
			return fmt.Sprintf(const_flat, op)
		case vector.FormDict:
			return fmt.Sprintf(const_dict, op)
		}
	}
	panic("formToLoop: bad logic")
}

func funcName(op, typ string, lhs, rhs vector.Form) string {
	return "cmp_" + opToAlpha(op) + "_" + typ + "_" + lhs.String() + "_" + rhs.String()
}

func opToAlpha(op string) string {
	switch op {
	case "==":
		return "EQ"
	case "!=":
		return "NE"
	case "<":
		return "LT"
	case "<=":
		return "LE"
	case ">":
		return "GT"
	case ">=":
		return "GE"
	}
	panic("bad op")
}
