package test

import (
	"testing"

	"github.com/tfsec/tfsec/internal/app/tfsec/checks"
	"github.com/tfsec/tfsec/internal/app/tfsec/scanner"
)

func Test_AWSALBDropsInvalidHeaders(t *testing.T) {

	var tests = []struct {
		name                  string
		source                string
		mustIncludeResultCode scanner.RuleCode
		mustExcludeResultCode scanner.RuleCode
	}{
		{
			name: "Alb that doesn't drop invalid headers by default fails",
			source: `
resource "aws_alb" "bad_example" {
	name               = "bad_alb"
	internal           = false
	load_balancer_type = "application"
	
	access_logs {
	  bucket  = aws_s3_bucket.lb_logs.bucket
	  prefix  = "test-lb"
	  enabled = true
	}
  }
`,
			mustIncludeResultCode: checks.AWSALBDropsInvalidHeaders,
		},
		{
			name: "lb that doesn't drop invalid headers by default fails",
			source: `
resource "aws_lb" "bad_example" {
	name               = "bad_alb"
	internal           = false
	load_balancer_type = "application"
	
	access_logs {
	  bucket  = aws_s3_bucket.lb_logs.bucket
	  prefix  = "test-lb"
	  enabled = true
	}
  }
`,
			mustIncludeResultCode: checks.AWSALBDropsInvalidHeaders,
		},
		{
			name: "Alb that doesn't drop invalid headers explicitly fails",
			source: `
resource "aws_alb" "bad_example" {
	name               = "bad_alb"
	internal           = false
	load_balancer_type = "application"
	
	access_logs {
	  bucket  = aws_s3_bucket.lb_logs.bucket
	  prefix  = "test-lb"
	  enabled = true
	}
  
	drop_invalid_header_fields = false
  }
`,
			mustIncludeResultCode: checks.AWSALBDropsInvalidHeaders,
		},
		{
			name: "lb that doesn't drop invalid headers fails",
			source: `
resource "aws_lb" "bad_example" {
	name               = "bad_alb"
	internal           = false
	load_balancer_type = "application"
	
	access_logs {
	  bucket  = aws_s3_bucket.lb_logs.bucket
	  prefix  = "test-lb"
	  enabled = true
	}
  
	drop_invalid_header_fields = false
  }
`,
			mustIncludeResultCode: checks.AWSALBDropsInvalidHeaders,
		},
		{
			name: "ALB that drops invalid headers passes check",
			source: `
resource "aws_alb" "good_example" {
	name               = "good_alb"
	internal           = false
	load_balancer_type = "application"
	
	access_logs {
	  bucket  = aws_s3_bucket.lb_logs.bucket
	  prefix  = "test-lb"
	  enabled = true
	}
  
	drop_invalid_header_fields = true
  }
`,
			mustExcludeResultCode: checks.AWSALBDropsInvalidHeaders,
		},
		{
			name: "LB that drops invalid headers passes check",
			source: `
resource "aws_lb" "good_example" {
	name               = "good_alb"
	internal           = false
	load_balancer_type = "application"
	
	access_logs {
	  bucket  = aws_s3_bucket.lb_logs.bucket
	  prefix  = "test-lb"
	  enabled = true
	}
  
	drop_invalid_header_fields = true
  }
`,
			mustExcludeResultCode: checks.AWSALBDropsInvalidHeaders,
		},
	}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			results := scanSource(test.source)
			assertCheckCode(t, test.mustIncludeResultCode, test.mustExcludeResultCode, results)
		})
	}

}
