// Copyright (c) 2021 Tailscale Inc & AUTHORS All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// +build ignore

package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
)

func main() {
	for _, goos := range []string{"windows", "linux", "darwin", "freebsd", "openbsd"} {
		generate(goos)
	}
}

func generate(goos string) {
	var x struct {
		Imports []string
	}
	cmd := exec.Command("go", "list", "-json", "tailscale.com/cmd/tailscaled")
	cmd.Env = append(os.Environ(), "GOOS="+goos, "GOARCH=amd64")
	j, err := cmd.Output()
	if err != nil {
		log.Fatalf("GOOS=%s GOARCH=amd64 %s: %v", goos, cmd, err)
	}
	if err := json.Unmarshal(j, &x); err != nil {
		log.Fatal(err)
	}
	var out bytes.Buffer
	out.WriteString(`// Copyright (c) 2021 Tailscale Inc & AUTHORS All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated by gen_deps.go; DO NOT EDIT.

package integration

import (
	// And depend on a bunch of tailscaled innards, for Go's test caching.
	// Otherwise cmd/go never sees that we depend on these packages'
	// transitive deps when we run "go install tailscaled" in a child
	// process and can cache a prior success when a dependency changes.
`)
	for _, dep := range x.Imports {
		fmt.Fprintf(&out, "\t_ %q\n", dep)
	}
	fmt.Fprintf(&out, ")\n")

	filename := fmt.Sprintf("tailscaled_deps_test_%s.go", goos)
	err = ioutil.WriteFile(filename, out.Bytes(), 0644)
	if err != nil {
		log.Fatal(err)
	}
}
