package excelize

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/xuri/nfp"
)

func TestNumFmt(t *testing.T) {
	for _, item := range [][]string{
		{"123", "general", "123"},
		{"-123", ";general", "-123"},
		{"43528", "y", "19"},
		{"43528", "Y", "19"},
		{"43528", "yy", "19"},
		{"43528", "YY", "19"},
		{"43528", "yyy", "2019"},
		{"43528", "YYY", "2019"},
		{"43528", "yyyy", "2019"},
		{"43528", "YYYY", "2019"},
		{"43528", "yyyyy", "2019"},
		{"43528", "YYYYY", "2019"},
		{"43528", "m", "3"},
		{"43528", "mm", "03"},
		{"43528", "mmm", "Mar"},
		{"43528", "mmmm", "March"},
		{"43528", "mmmmm", "M"},
		{"43528", "mmmmmm", "March"},
		{"43528", "d", "4"},
		{"43528", "dd", "04"},
		{"43528", "ddd", "Mon"},
		{"43528", "dddd", "Monday"},
		{"43528", "h", "0"},
		{"43528", "hh", "00"},
		{"43528", "hhh", "00"},
		{"43543.544872685183", "hhmm", "1304"},
		{"43543.544872685183", "mmhhmmmm", "0313March"},
		{"43543.544872685183", "mm hh mm mm", "03 13 04 03"},
		{"43543.544872685183", "mm hh m m", "03 13 4 3"},
		{"43543.544872685183", "m s", "4 37"},
		{"43528", "[h]", "1044672"},
		{"43528", "[m]", "62680320"},
		{"43528", "s", "0"},
		{"43528", "ss", "00"},
		{"43528", "[s]", "3760819200"},
		{"43543.544872685183", "h:mm:ss AM/PM", "1:04:37 PM"},
		{"43543.544872685183", "AM/PM h:mm:ss", "PM 1:04:37"},
		{"43543.086539351854", "hh:mm:ss AM/PM", "02:04:37 AM"},
		{"43543.086539351854", "AM/PM hh:mm:ss", "AM 02:04:37"},
		{"43543.086539351854", "AM/PM hh:mm:ss a/p", "AM 02:04:37 a"},
		{"0.609375", "[HH]:mm:ss", "14:37:30"},
		{"43528", "YYYY", "2019"},
		{"43528", "", "43528"},
		{"43528.2123", "YYYY-MM-DD hh:mm:ss", "2019-03-04 05:05:43"},
		{"43528.2123", "YYYY-MM-DD hh:mm:ss;YYYY-MM-DD hh:mm:ss", "2019-03-04 05:05:43"},
		{"43528.2123", "M/D/YYYY h:m:s", "3/4/2019 5:5:43"},
		{"43528.003958333335", "m/d/yyyy h:m:s", "3/4/2019 0:5:42"},
		{"43528.003958333335", "M/D/YYYY h:mm:s", "3/4/2019 0:05:42"},
		{"0.64583333333333337", "h:mm:ss am/pm", "3:30:00 pm"},
		{"43528.003958333335", "h:mm", "0:05"},
		{"6.9444444444444444E-5", "h:m", "0:0"},
		{"6.9444444444444444E-5", "h:mm", "0:00"},
		{"6.9444444444444444E-5", "h:m", "0:0"},
		{"0.50070601851851848", "h:m", "12:1"},
		{"0.97952546296296295", "h:m", "23:30"},
		{"43528", "mmmm", "March"},
		{"43528", "dddd", "Monday"},
		{"0", ";;;", "0"},
		{"43528", "[$-409]MM/DD/YYYY", "03/04/2019"},
		{"43528", "[$-409]MM/DD/YYYY am/pm", "03/04/2019 AM"},
		{"43528", "[$-111]MM/DD/YYYY", "43528"},
		{"43528", "[$US-409]MM/DD/YYYY", "US03/04/2019"},
		{"43543.586539351854", "AM/PM h h:mm", "PM 14 2:04"},
		{"text", "AM/PM h h:mm", "text"},
		{"43466.189571759256", "[$-404]aaa;@", "週二"},
		{"43466.189571759256", "[$-404]aaaa;@", "星期二"},
		{"43466.189571759256", "[$-804]aaa;@", "周二"},
		{"43466.189571759256", "[$-804]aaaa;@", "星期二"},
		{"43466.189571759256", "[$-435]aaa;@", "Bi."},
		{"43466.189571759256", "[$-435]aaaa;@", "ULwesibili"},
		{"43466.189571759256", "[$-404]ddd;@", "週二"},
		{"43466.189571759256", "[$-404]dddd;@", "星期二"},
		{"43466.189571759256", "[$-804]ddd;@", "周二"},
		{"43466.189571759256", "[$-804]dddd;@", "星期二"},
		{"43466.189571759256", "[$-435]ddd;@", "Bi."},
		{"43466.189571759256", "[$-435]dddd;@", "ULwesibili"},
		{"44562.189571759256", "[$-36]mmm dd yyyy  h:mm AM/PM d", "Jan. 01 2022  4:32 vm. 1"},
		{"44562.189571759256", "[$-36]mmmm dd yyyy  h:mm AM/PM dd", "Januarie 01 2022  4:32 vm. 01"},
		{"44562.189571759256", "[$-36]mmmmm dd yyyy  h:mm AM/PM ddd", "J 01 2022  4:32 vm. Sa."},
		{"44682.18957170139", "[$-36]mmm dd yyyy  h:mm AM/PM dddd", "Mei 01 2022  4:32 vm. Sondag"},
		{"44682.18957170139", "[$-36]mmmm dd yyyy  h:mm AM/PM aaa", "Mei 01 2022  4:32 vm. So."},
		{"44682.18957170139", "[$-36]mmmmm dd yyyy  h:mm AM/PM aaaa", "M 01 2022  4:32 vm. Sondag"},
		{"44562.189571759256", "[$-436]mmm dd yyyy  h:mm AM/PM d", "Jan. 01 2022  4:32 vm. 1"},
		{"44562.189571759256", "[$-436]mmmm dd yyyy  h:mm AM/PM dd", "Januarie 01 2022  4:32 vm. 01"},
		{"44562.189571759256", "[$-436]mmmmm dd yyyy  h:mm AM/PM ddd", "J 01 2022  4:32 vm. Sa."},
		{"44682.18957170139", "[$-436]mmm dd yyyy  h:mm AM/PM dddd", "Mei 01 2022  4:32 vm. Sondag"},
		{"44682.18957170139", "[$-436]mmmm dd yyyy  h:mm AM/PM aaa", "Mei 01 2022  4:32 vm. So."},
		{"44682.18957170139", "[$-436]mmmmm dd yyyy  h:mm AM/PM aaaa", "M 01 2022  4:32 vm. Sondag"},
		{"44562.189571759256", "[$-1C]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 p.d."},
		{"44562.189571759256", "[$-1C]mmmm dd yyyy  h:mm AM/PM", "janar 01 2022  4:32 p.d."},
		{"44562.189571759256", "[$-1C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 p.d."},
		{"44562.189571759256", "[$-1C]mmmmmm dd yyyy  h:mm AM/PM", "janar 01 2022  4:32 p.d."},
		{"43543.503206018519", "[$-1C]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 m.d."},
		{"43543.503206018519", "[$-1C]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 m.d. mar"},
		{"43543.503206018519", "[$-1C]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 m.d. mar"},
		{"43543.503206018519", "[$-1C]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 m.d. e martë"},
		{"44562.189571759256", "[$-41C]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 p.d."},
		{"44562.189571759256", "[$-41C]mmmm dd yyyy  h:mm AM/PM", "janar 01 2022  4:32 p.d."},
		{"44562.189571759256", "[$-41C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 p.d."},
		{"44562.189571759256", "[$-41C]mmmmmm dd yyyy  h:mm AM/PM", "janar 01 2022  4:32 p.d."},
		{"43543.503206018519", "[$-41C]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 m.d."},
		{"43543.503206018519", "[$-41C]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 m.d. mar"},
		{"43543.503206018519", "[$-41C]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 m.d. mar"},
		{"43543.503206018519", "[$-41C]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 m.d. e martë"},
		{"44562.189571759256", "[$-84]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 vorm."},
		{"44562.189571759256", "[$-84]mmmm dd yyyy  h:mm AM/PM", "Januar 01 2022  4:32 vorm."},
		{"44562.189571759256", "[$-84]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 vorm."},
		{"44562.189571759256", "[$-84]mmmmmm dd yyyy  h:mm AM/PM", "Januar 01 2022  4:32 vorm."},
		{"43543.503206018519", "[$-84]mmm dd yyyy  h:mm AM/PM", "Mär 19 2019  12:04 nam."},
		{"43543.503206018519", "[$-84]mmmm dd yyyy  h:mm AM/PM aaa", "März 19 2019  12:04 nam. Zi."},
		{"43543.503206018519", "[$-84]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 nam. Zi."},
		{"43543.503206018519", "[$-84]mmmmmm dd yyyy  h:mm AM/PM dddd", "März 19 2019  12:04 nam. Ziischtig"},
		{"44562.189571759256", "[$-484]mmm dd yyyy  h:mm AM/PM", "Jän. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-484]mmmm dd yyyy  h:mm AM/PM", "Jänner 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-484]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-484]mmmmmm dd yyyy  h:mm AM/PM", "Jänner 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-484]mmm dd yyyy  h:mm AM/PM", "März 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-484]mmmm dd yyyy  h:mm AM/PM aaa", "März 19 2019  12:04 PM Zi."},
		{"43543.503206018519", "[$-484]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Zi."},
		{"43543.503206018519", "[$-484]mmmmmm dd yyyy  h:mm AM/PM dddd", "März 19 2019  12:04 PM Zischti"},
		{"44562.189571759256", "[$-5E]mmm dd yyyy  h:mm AM/PM", "\u1303\u1295\u12E9 01 2022  4:32 \u1325\u12CB\u1275"},
		{"44562.189571759256", "[$-5E]mmmm dd yyyy  h:mm AM/PM", "\u1303\u1295\u12E9\u12C8\u122A 01 2022  4:32 \u1325\u12CB\u1275"},
		{"44562.189571759256", "[$-5E]mmmmm dd yyyy  h:mm AM/PM", "\u1303 01 2022  4:32 \u1325\u12CB\u1275"},
		{"44562.189571759256", "[$-5E]mmmmmm dd yyyy  h:mm AM/PM", "\u1303\u1295\u12E9\u12C8\u122A 01 2022  4:32 \u1325\u12CB\u1275"},
		{"43543.503206018519", "[$-5E]mmm dd yyyy  h:mm AM/PM", "\u121B\u122D\u127D 19 2019  12:04 \u12A8\u1230\u12D3\u1275"},
		{"43543.503206018519", "[$-5E]mmmm dd yyyy  h:mm AM/PM aaa", "\u121B\u122D\u127D 19 2019  12:04 \u12A8\u1230\u12D3\u1275 \u121B\u12AD\u1230"},
		{"43543.503206018519", "[$-5E]mmmmm dd yyyy  h:mm AM/PM ddd", "\u121B 19 2019  12:04 \u12A8\u1230\u12D3\u1275 \u121B\u12AD\u1230"},
		{"43543.503206018519", "[$-5E]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u121B\u122D\u127D 19 2019  12:04 \u12A8\u1230\u12D3\u1275 \u121B\u12AD\u1230\u129E"},
		{"44562.189571759256", "[$-45E]mmm dd yyyy  h:mm AM/PM", "\u1303\u1295\u12E9 01 2022  4:32 \u1325\u12CB\u1275"},
		{"44562.189571759256", "[$-45E]mmmm dd yyyy  h:mm AM/PM", "\u1303\u1295\u12E9\u12C8\u122A 01 2022  4:32 \u1325\u12CB\u1275"},
		{"44562.189571759256", "[$-45E]mmmmm dd yyyy  h:mm AM/PM", "\u1303 01 2022  4:32 \u1325\u12CB\u1275"},
		{"44562.189571759256", "[$-45E]mmmmmm dd yyyy  h:mm AM/PM", "\u1303\u1295\u12E9\u12C8\u122A 01 2022  4:32 \u1325\u12CB\u1275"},
		{"43543.503206018519", "[$-45E]mmm dd yyyy  h:mm AM/PM", "\u121B\u122D\u127D 19 2019  12:04 \u12A8\u1230\u12D3\u1275"},
		{"43543.503206018519", "[$-45E]mmmm dd yyyy  h:mm AM/PM aaa", "\u121B\u122D\u127D 19 2019  12:04 \u12A8\u1230\u12D3\u1275 \u121B\u12AD\u1230"},
		{"43543.503206018519", "[$-45E]mmmmm dd yyyy  h:mm AM/PM ddd", "\u121B 19 2019  12:04 \u12A8\u1230\u12D3\u1275 \u121B\u12AD\u1230"},
		{"43543.503206018519", "[$-45E]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u121B\u122D\u127D 19 2019  12:04 \u12A8\u1230\u12D3\u1275 \u121B\u12AD\u1230\u129E"},
		{"44562.189571759256", "[$-1]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-1]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-1]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-1]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-1]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-1401]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1401]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1401]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1401]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-1401]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-1401]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-1401]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-1401]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-3C01]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3C01]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3C01]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3C01]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-3C01]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-3C01]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-3C01]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-3C01]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-c01]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-c01]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-c01]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-c01]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-c01]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-c01]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-c01]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-c01]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-801]mmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-801]mmmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-801]mmmmm dd yyyy  h:mm AM/PM", "\u0643 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-801]mmmmmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-801]mmm dd yyyy  h:mm AM/PM", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-801]mmmm dd yyyy  h:mm AM/PM aaa", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-801]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0622 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-801]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-2C01]mmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2C01]mmmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2C01]mmmmm dd yyyy  h:mm AM/PM", "\u0643 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2C01]mmmmmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-2C01]mmm dd yyyy  h:mm AM/PM", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-2C01]mmmm dd yyyy  h:mm AM/PM aaa", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-2C01]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0622 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-2C01]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-3401]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3401]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3401]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3401]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-3401]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-3401]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-3401]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-3401]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-3001]mmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3001]mmmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3001]mmmmm dd yyyy  h:mm AM/PM", "\u0643 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3001]mmmmmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-3001]mmm dd yyyy  h:mm AM/PM", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-3001]mmmm dd yyyy  h:mm AM/PM aaa", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-3001]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0622 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-3001]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-1801]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1801]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1801]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1801]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-1801]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-1801]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-1801]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-1801]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-2001]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2001]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2001]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2001]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-2001]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-2001]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-2001]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-2001]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-4001]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-4001]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-4001]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-4001]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-4001]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-4001]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-4001]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-4001]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-401]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-401]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-401]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-401]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-401]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-401]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-401]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-401]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-2801]mmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2801]mmmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2801]mmmmm dd yyyy  h:mm AM/PM", "\u0643 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2801]mmmmmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-2801]mmm dd yyyy  h:mm AM/PM", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-2801]mmmm dd yyyy  h:mm AM/PM aaa", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-2801]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0622 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-2801]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-1C01]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1C01]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1C01]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-1C01]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-1C01]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-1C01]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-1C01]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-1C01]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-3801]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3801]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3801]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-3801]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-3801]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-3801]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-3801]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-3801]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-2401]mmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2401]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2401]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-2401]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0646\u0627\u064A\u0631 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-2401]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-2401]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-2401]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-2401]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-2B]mmm dd yyyy  h:mm AM/PM", "\u0540\u0576\u057E 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2B]mmmm dd yyyy  h:mm AM/PM", "\u0540\u0578\u0582\u0576\u057E\u0561\u0580 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2B]mmmmm dd yyyy  h:mm AM/PM", "\u0540 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2B]mmmmmm dd yyyy  h:mm AM/PM", "\u0540\u0578\u0582\u0576\u057E\u0561\u0580 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-2B]mmm dd yyyy  h:mm AM/PM", "\u0544\u0580\u057F 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-2B]mmmm dd yyyy  h:mm AM/PM aaa", "\u0544\u0561\u0580\u057F 19 2019  12:04 PM \u0535\u0580\u0584"},
		{"43543.503206018519", "[$-2B]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0544 19 2019  12:04 PM \u0535\u0580\u0584"},
		{"43543.503206018519", "[$-2B]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0544\u0561\u0580\u057F 19 2019  12:04 PM \u0535\u0580\u0565\u0584\u0577\u0561\u0562\u0569\u056B"},
		{"44562.189571759256", "[$-42B]mmm dd yyyy  h:mm AM/PM", "\u0540\u0576\u057E 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42B]mmmm dd yyyy  h:mm AM/PM", "\u0540\u0578\u0582\u0576\u057E\u0561\u0580 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42B]mmmmm dd yyyy  h:mm AM/PM", "\u0540 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42B]mmmmmm dd yyyy  h:mm AM/PM", "\u0540\u0578\u0582\u0576\u057E\u0561\u0580 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-42B]mmm dd yyyy  h:mm AM/PM", "\u0544\u0580\u057F 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-42B]mmmm dd yyyy  h:mm AM/PM aaa", "\u0544\u0561\u0580\u057F 19 2019  12:04 PM \u0535\u0580\u0584"},
		{"43543.503206018519", "[$-42B]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0544 19 2019  12:04 PM \u0535\u0580\u0584"},
		{"43543.503206018519", "[$-42B]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0544\u0561\u0580\u057F 19 2019  12:04 PM \u0535\u0580\u0565\u0584\u0577\u0561\u0562\u0569\u056B"},
		{"44562.189571759256", "[$-4D]mmm dd yyyy  h:mm AM/PM", "\u099C\u09BE\u09A8\u09C1 01 2022  4:32 \u09F0\u09BE\u09A4\u09BF\u09AA\u09C1"},
		{"44562.189571759256", "[$-4D]mmmm dd yyyy  h:mm AM/PM", "\u099C\u09BE\u09A8\u09C1\u09F1\u09BE\u09F0\u09C0 01 2022  4:32 \u09F0\u09BE\u09A4\u09BF\u09AA\u09C1"},
		{"44562.189571759256", "[$-4D]mmmmm dd yyyy  h:mm AM/PM", "\u099C 01 2022  4:32 \u09F0\u09BE\u09A4\u09BF\u09AA\u09C1"},
		{"44562.189571759256", "[$-4D]mmmmmm dd yyyy  h:mm AM/PM", "\u099C\u09BE\u09A8\u09C1\u09F1\u09BE\u09F0\u09C0 01 2022  4:32 \u09F0\u09BE\u09A4\u09BF\u09AA\u09C1"},
		{"43543.503206018519", "[$-4D]mmm dd yyyy  h:mm AM/PM", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 \u0986\u09AC\u09C7\u09B2\u09BF"},
		{"43543.503206018519", "[$-4D]mmmm dd yyyy  h:mm AM/PM aaa", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 \u0986\u09AC\u09C7\u09B2\u09BF \u09AE\u0999\u09CD\u0997\u09B2."},
		{"43543.503206018519", "[$-4D]mmmmm dd yyyy  h:mm AM/PM ddd", "\u09AE 19 2019  12:04 \u0986\u09AC\u09C7\u09B2\u09BF \u09AE\u0999\u09CD\u0997\u09B2."},
		{"43543.503206018519", "[$-4D]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 \u0986\u09AC\u09C7\u09B2\u09BF \u09AE\u0999\u09CD\u0997\u09B2\u09AC\u09BE\u09F0"},
		{"44562.189571759256", "[$-44D]mmm dd yyyy  h:mm AM/PM", "\u099C\u09BE\u09A8\u09C1 01 2022  4:32 \u09F0\u09BE\u09A4\u09BF\u09AA\u09C1"},
		{"44562.189571759256", "[$-44D]mmmm dd yyyy  h:mm AM/PM", "\u099C\u09BE\u09A8\u09C1\u09F1\u09BE\u09F0\u09C0 01 2022  4:32 \u09F0\u09BE\u09A4\u09BF\u09AA\u09C1"},
		{"44562.189571759256", "[$-44D]mmmmm dd yyyy  h:mm AM/PM", "\u099C 01 2022  4:32 \u09F0\u09BE\u09A4\u09BF\u09AA\u09C1"},
		{"44562.189571759256", "[$-44D]mmmmmm dd yyyy  h:mm AM/PM", "\u099C\u09BE\u09A8\u09C1\u09F1\u09BE\u09F0\u09C0 01 2022  4:32 \u09F0\u09BE\u09A4\u09BF\u09AA\u09C1"},
		{"43543.503206018519", "[$-44D]mmm dd yyyy  h:mm AM/PM", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 \u0986\u09AC\u09C7\u09B2\u09BF"},
		{"43543.503206018519", "[$-44D]mmmm dd yyyy  h:mm AM/PM aaa", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 \u0986\u09AC\u09C7\u09B2\u09BF \u09AE\u0999\u09CD\u0997\u09B2."},
		{"43543.503206018519", "[$-44D]mmmmm dd yyyy  h:mm AM/PM ddd", "\u09AE 19 2019  12:04 \u0986\u09AC\u09C7\u09B2\u09BF \u09AE\u0999\u09CD\u0997\u09B2."},
		{"43543.503206018519", "[$-44D]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 \u0986\u09AC\u09C7\u09B2\u09BF \u09AE\u0999\u09CD\u0997\u09B2\u09AC\u09BE\u09F0"},
		{"44562.189571759256", "[$-742C]mmm dd yyyy  h:mm AM/PM", "\u0408\u0430\u043D 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-742C]mmmm dd yyyy  h:mm AM/PM", "j\u0430\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-742C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-742C]mmmmmm dd yyyy  h:mm AM/PM", "j\u0430\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-742C]mmm dd yyyy  h:mm AM/PM", "\u041C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-742C]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0427\u0430"},
		{"43543.503206018519", "[$-742C]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0427\u0430"},
		{"43543.503206018519", "[$-742C]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0447\u04D9\u0440\u0448\u04D9\u043D\u0431\u04D9%A0\u0430\u0445\u0448\u0430\u043C\u044B"},
		{"44562.189571759256", "[$-82C]mmm dd yyyy  h:mm AM/PM", "\u0408\u0430\u043D 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-82C]mmmm dd yyyy  h:mm AM/PM", "j\u0430\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-82C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-82C]mmmmmm dd yyyy  h:mm AM/PM", "j\u0430\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-82C]mmm dd yyyy  h:mm AM/PM", "\u041C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-82C]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0427\u0430"},
		{"43543.503206018519", "[$-82C]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0427\u0430"},
		{"43543.503206018519", "[$-82C]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0447\u04D9\u0440\u0448\u04D9\u043D\u0431\u04D9%A0\u0430\u0445\u0448\u0430\u043C\u044B"},
		{"44562.189571759256", "[$-2C]mmm dd yyyy  h:mm AM/PM", "yan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2C]mmmm dd yyyy  h:mm AM/PM", "yanvar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2C]mmmmm dd yyyy  h:mm AM/PM", "y 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2C]mmmmmm dd yyyy  h:mm AM/PM", "yanvar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-2C]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-2C]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 PM %C7.A."},
		{"43543.503206018519", "[$-2C]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM %C7.A."},
		{"43543.503206018519", "[$-2C]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 PM %E7\u0259r\u015F\u0259nb\u0259%A0ax\u015Fam\u0131"},
		{"44562.189571759256", "[$-782C]mmm dd yyyy  h:mm AM/PM", "yan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-782C]mmmm dd yyyy  h:mm AM/PM", "yanvar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-782C]mmmmm dd yyyy  h:mm AM/PM", "y 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-782C]mmmmmm dd yyyy  h:mm AM/PM", "yanvar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-782C]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-782C]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 PM %C7.A."},
		{"43543.503206018519", "[$-782C]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM %C7.A."},
		{"43543.503206018519", "[$-782C]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 PM %E7\u0259r\u015F\u0259nb\u0259%A0ax\u015Fam\u0131"},
		{"44562.189571759256", "[$-42C]mmm dd yyyy  h:mm AM/PM", "yan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42C]mmmm dd yyyy  h:mm AM/PM", "yanvar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42C]mmmmm dd yyyy  h:mm AM/PM", "y 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42C]mmmmmm dd yyyy  h:mm AM/PM", "yanvar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-42C]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-42C]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 PM %C7.A."},
		{"43543.503206018519", "[$-42C]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM %C7.A."},
		{"43543.503206018519", "[$-42C]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 PM %E7\u0259r\u015F\u0259nb\u0259%A0ax\u015Fam\u0131"},
		{"43543.503206018519", "[$-45]mmm dd yyyy  h:mm AM/PM aaa", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 PM \u09AE\u0999\u09CD\u0997\u09B2."},
		{"43543.503206018519", "[$-45]mmmm dd yyyy  h:mm AM/PM ddd", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 PM \u09AE\u0999\u09CD\u0997\u09B2."},
		{"43543.503206018519", "[$-45]mmmmm dd yyyy  h:mm AM/PM dddd", "\u09AE 19 2019  12:04 PM \u09AE\u0999\u09CD\u0997\u09B2\u09AC\u09BE\u09B0"},
		{"43543.503206018519", "[$-845]mmm dd yyyy  h:mm AM/PM aaa", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 PM \u09AE\u0999\u09CD\u0997\u09B2."},
		{"43543.503206018519", "[$-845]mmmm dd yyyy  h:mm AM/PM ddd", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 PM \u09AE\u0999\u09CD\u0997\u09B2."},
		{"43543.503206018519", "[$-845]mmmmm dd yyyy  h:mm AM/PM dddd", "\u09AE 19 2019  12:04 PM \u09AE\u0999\u09CD\u0997\u09B2\u09AC\u09BE\u09B0"},
		{"43543.503206018519", "[$-445]mmm dd yyyy  h:mm AM/PM aaa", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 PM \u09AE\u0999\u09CD\u0997\u09B2."},
		{"43543.503206018519", "[$-445]mmmm dd yyyy  h:mm AM/PM ddd", "\u09AE\u09BE\u09B0\u09CD\u099A 19 2019  12:04 PM \u09AE\u0999\u09CD\u0997\u09B2."},
		{"43543.503206018519", "[$-445]mmmmm dd yyyy  h:mm AM/PM dddd", "\u09AE 19 2019  12:04 PM \u09AE\u0999\u09CD\u0997\u09B2\u09AC\u09BE\u09B0"},
		{"44562.189571759256", "[$-6D]mmm dd yyyy  h:mm AM/PM", "\u0493\u0438\u043D 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6D]mmmm dd yyyy  h:mm AM/PM", "\u0493\u0438\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6D]mmmmm dd yyyy  h:mm AM/PM", "\u0493 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6D]mmmmmm dd yyyy  h:mm AM/PM", "\u0493\u0438\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-6D]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-6D]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0428\u0448"},
		{"43543.503206018519", "[$-6D]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0428\u0448"},
		{"43543.503206018519", "[$-6D]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0428\u0438\u0448\u04D9\u043C\u0431\u0435"},
		{"44562.189571759256", "[$-46D]mmm dd yyyy  h:mm AM/PM", "\u0493\u0438\u043D 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46D]mmmm dd yyyy  h:mm AM/PM", "\u0493\u0438\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46D]mmmmm dd yyyy  h:mm AM/PM", "\u0493 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46D]mmmmmm dd yyyy  h:mm AM/PM", "\u0493\u0438\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-46D]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-46D]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0428\u0448"},
		{"43543.503206018519", "[$-46D]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0428\u0448"},
		{"43543.503206018519", "[$-46D]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0428\u0438\u0448\u04D9\u043C\u0431\u0435"},
		{"44562.189571759256", "[$-2D]mmm dd yyyy  h:mm AM/PM", "urt. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2D]mmmm dd yyyy  h:mm AM/PM", "urtarrila 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2D]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2D]mmmmmm dd yyyy  h:mm AM/PM", "urtarrila 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-2D]mmm dd yyyy  h:mm AM/PM", "mar. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-2D]mmmm dd yyyy  h:mm AM/PM aaa", "martxoa 19 2019  12:04 PM ar."},
		{"43543.503206018519", "[$-2D]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM ar."},
		{"43543.503206018519", "[$-2D]mmmmmm dd yyyy  h:mm AM/PM dddd", "martxoa 19 2019  12:04 PM asteartea"},
		{"44562.189571759256", "[$-42D]mmm dd yyyy  h:mm AM/PM", "urt. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42D]mmmm dd yyyy  h:mm AM/PM", "urtarrila 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42D]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42D]mmmmmm dd yyyy  h:mm AM/PM", "urtarrila 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-42D]mmm dd yyyy  h:mm AM/PM", "mar. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-42D]mmmm dd yyyy  h:mm AM/PM aaa", "martxoa 19 2019  12:04 PM ar."},
		{"43543.503206018519", "[$-42D]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM ar."},
		{"43543.503206018519", "[$-42D]mmmmmm dd yyyy  h:mm AM/PM dddd", "martxoa 19 2019  12:04 PM asteartea"},
		{"44562.189571759256", "[$-23]mmm dd yyyy  h:mm AM/PM", "\u0441\u0442\u0443\u0434\u0437 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-23]mmmm dd yyyy  h:mm AM/PM", "\u0441\u0442\u0443\u0434\u0437\u0435\u043D\u044C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-23]mmmmm dd yyyy  h:mm AM/PM", "\u0441 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-23]mmmmmm dd yyyy  h:mm AM/PM", "\u0441\u0442\u0443\u0434\u0437\u0435\u043D\u044C 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-23]mmm dd yyyy  h:mm AM/PM", "\u0441\u0430\u043A 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-23]mmmm dd yyyy  h:mm AM/PM aaa", "\u0441\u0430\u043A\u0430\u0432\u0456\u043A 19 2019  12:04 PM \u0430\u045E\u0442"},
		{"43543.503206018519", "[$-23]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0441 19 2019  12:04 PM \u0430\u045E\u0442"},
		{"43543.503206018519", "[$-23]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0441\u0430\u043A\u0430\u0432\u0456\u043A 19 2019  12:04 PM \u0430\u045E\u0442\u043E\u0440\u0430\u043A"},
		{"44562.189571759256", "[$-423]mmm dd yyyy  h:mm AM/PM", "\u0441\u0442\u0443\u0434\u0437 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-423]mmmm dd yyyy  h:mm AM/PM", "\u0441\u0442\u0443\u0434\u0437\u0435\u043D\u044C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-423]mmmmm dd yyyy  h:mm AM/PM", "\u0441 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-423]mmmmmm dd yyyy  h:mm AM/PM", "\u0441\u0442\u0443\u0434\u0437\u0435\u043D\u044C 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-423]mmm dd yyyy  h:mm AM/PM", "\u0441\u0430\u043A 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-423]mmmm dd yyyy  h:mm AM/PM aaa", "\u0441\u0430\u043A\u0430\u0432\u0456\u043A 19 2019  12:04 PM \u0430\u045E\u0442"},
		{"43543.503206018519", "[$-423]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0441 19 2019  12:04 PM \u0430\u045E\u0442"},
		{"43543.503206018519", "[$-423]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0441\u0430\u043A\u0430\u0432\u0456\u043A 19 2019  12:04 PM \u0430\u045E\u0442\u043E\u0440\u0430\u043A"},
		{"44562.189571759256", "[$-641A]mmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-641A]mmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-641A]mmmmm dd yyyy  h:mm AM/PM", "\u0458 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-641A]mmmmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-641A]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-641A]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E"},
		{"43543.503206018519", "[$-641A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0443\u0442\u043E"},
		{"43543.503206018519", "[$-641A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E\u0440\u0430\u043A"},
		{"44562.189571759256", "[$-201A]mmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-201A]mmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-201A]mmmmm dd yyyy  h:mm AM/PM", "\u0458 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-201A]mmmmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-201A]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-201A]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E"},
		{"43543.503206018519", "[$-201A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0443\u0442\u043E"},
		{"43543.503206018519", "[$-201A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E\u0440\u0430\u043A"},
		{"44562.189571759256", "[$-681A]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-681A]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-681A]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-681A]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-681A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-681A]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 PM uto"},
		{"43543.503206018519", "[$-681A]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM uto"},
		{"43543.503206018519", "[$-681A]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 PM utorak"},
		{"44562.189571759256", "[$-781A]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-781A]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-781A]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-781A]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-781A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-781A]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 PM uto"},
		{"43543.503206018519", "[$-781A]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM uto"},
		{"43543.503206018519", "[$-781A]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 PM utorak"},
		{"44562.189571759256", "[$-141A]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-141A]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-141A]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-141A]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-141A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-141A]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 PM uto"},
		{"43543.503206018519", "[$-141A]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM uto"},
		{"43543.503206018519", "[$-141A]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 PM utorak"},
		{"44562.189571759256", "[$-7E]mmm dd yyyy  h:mm AM/PM", "Gen. 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-7E]mmmm dd yyyy  h:mm AM/PM", "Genver 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-7E]mmmmm dd yyyy  h:mm AM/PM", "G 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-7E]mmmmmm dd yyyy  h:mm AM/PM", "Genver 01 2022  4:32 A.M."},
		{"43543.503206018519", "[$-7E]mmm dd yyyy  h:mm AM/PM", "Meur. 19 2019  12:04 G.M."},
		{"43543.503206018519", "[$-7E]mmmm dd yyyy  h:mm AM/PM aaa", "Meurzh 19 2019  12:04 G.M. Meu."},
		{"43543.503206018519", "[$-7E]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 G.M. Meu."},
		{"43543.503206018519", "[$-7E]mmmmmm dd yyyy  h:mm AM/PM dddd", "Meurzh 19 2019  12:04 G.M. Meurzh"},
		{"44562.189571759256", "[$-47E]mmm dd yyyy  h:mm AM/PM", "Gen. 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-47E]mmmm dd yyyy  h:mm AM/PM", "Genver 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-47E]mmmmm dd yyyy  h:mm AM/PM", "G 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-47E]mmmmmm dd yyyy  h:mm AM/PM", "Genver 01 2022  4:32 A.M."},
		{"43543.503206018519", "[$-47E]mmm dd yyyy  h:mm AM/PM", "Meur. 19 2019  12:04 G.M."},
		{"43543.503206018519", "[$-47E]mmmm dd yyyy  h:mm AM/PM aaa", "Meurzh 19 2019  12:04 G.M. Meu."},
		{"43543.503206018519", "[$-47E]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 G.M. Meu."},
		{"43543.503206018519", "[$-47E]mmmmmm dd yyyy  h:mm AM/PM dddd", "Meurzh 19 2019  12:04 G.M. Meurzh"},
		{"44562.189571759256", "[$-2]mmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0443 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2]mmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0443\u0430\u0440\u0438 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2]mmmmm dd yyyy  h:mm AM/PM", "\u044F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-2]mmmmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0443\u0430\u0440\u0438 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-2]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-2]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0432\u0442"},
		{"43543.503206018519", "[$-2]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0432\u0442"},
		{"43543.503206018519", "[$-2]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0432\u0442\u043E\u0440\u043D\u0438\u043A"},
		{"44562.189571759256", "[$-402]mmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0443 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-402]mmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0443\u0430\u0440\u0438 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-402]mmmmm dd yyyy  h:mm AM/PM", "\u044F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-402]mmmmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0443\u0430\u0440\u0438 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-402]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-402]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0432\u0442"},
		{"43543.503206018519", "[$-402]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0432\u0442"},
		{"43543.503206018519", "[$-402]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0432\u0442\u043E\u0440\u043D\u0438\u043A"},
		{"44562.189571759256", "[$-55]mmm dd yyyy  h:mm AM/PM", "\u1007\u1014\u103A 01 2022  4:32 \u1014\u1036\u1014\u1000\u103A"},
		{"44562.189571759256", "[$-55]mmmm dd yyyy  h:mm AM/PM", "\u1007\u1014\u103A\u1014\u101D\u102B\u101B\u102E 01 2022  4:32 \u1014\u1036\u1014\u1000\u103A"},
		{"44562.189571759256", "[$-55]mmmmm dd yyyy  h:mm AM/PM", "\u1007 01 2022  4:32 \u1014\u1036\u1014\u1000\u103A"},
		{"44562.189571759256", "[$-55]mmmmmm dd yyyy  h:mm AM/PM", "\u1007\u1014\u103A\u1014\u101D\u102B\u101B\u102E 01 2022  4:32 \u1014\u1036\u1014\u1000\u103A"},
		{"43543.503206018519", "[$-55]mmm dd yyyy  h:mm AM/PM", "\u1019\u1010\u103A 19 2019  12:04 \u100A\u1014\u1031"},
		{"43543.503206018519", "[$-55]mmmm dd yyyy  h:mm AM/PM aaa", "\u1019\u1010\u103A 19 2019  12:04 \u100A\u1014\u1031 \u1021\u1004\u103A\u1039\u1002\u102B"},
		{"43543.503206018519", "[$-55]mmmmm dd yyyy  h:mm AM/PM ddd", "\u1019 19 2019  12:04 \u100A\u1014\u1031 \u1021\u1004\u103A\u1039\u1002\u102B"},
		{"43543.503206018519", "[$-55]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u1019\u1010\u103A 19 2019  12:04 \u100A\u1014\u1031 \u1021\u1004\u103A\u1039\u1002\u102B"},
		{"44562.189571759256", "[$-455]mmm dd yyyy  h:mm AM/PM", "\u1007\u1014\u103A 01 2022  4:32 \u1014\u1036\u1014\u1000\u103A"},
		{"44562.189571759256", "[$-455]mmmm dd yyyy  h:mm AM/PM", "\u1007\u1014\u103A\u1014\u101D\u102B\u101B\u102E 01 2022  4:32 \u1014\u1036\u1014\u1000\u103A"},
		{"44562.189571759256", "[$-455]mmmmm dd yyyy  h:mm AM/PM", "\u1007 01 2022  4:32 \u1014\u1036\u1014\u1000\u103A"},
		{"44562.189571759256", "[$-455]mmmmmm dd yyyy  h:mm AM/PM", "\u1007\u1014\u103A\u1014\u101D\u102B\u101B\u102E 01 2022  4:32 \u1014\u1036\u1014\u1000\u103A"},
		{"43543.503206018519", "[$-455]mmm dd yyyy  h:mm AM/PM", "\u1019\u1010\u103A 19 2019  12:04 \u100A\u1014\u1031"},
		{"43543.503206018519", "[$-455]mmmm dd yyyy  h:mm AM/PM aaa", "\u1019\u1010\u103A 19 2019  12:04 \u100A\u1014\u1031 \u1021\u1004\u103A\u1039\u1002\u102B"},
		{"43543.503206018519", "[$-455]mmmmm dd yyyy  h:mm AM/PM ddd", "\u1019 19 2019  12:04 \u100A\u1014\u1031 \u1021\u1004\u103A\u1039\u1002\u102B"},
		{"43543.503206018519", "[$-455]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u1019\u1010\u103A 19 2019  12:04 \u100A\u1014\u1031 \u1021\u1004\u103A\u1039\u1002\u102B"},
		{"44562.189571759256", "[$-3]mmm dd yyyy  h:mm AM/PM", "gen. 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-3]mmmm dd yyyy  h:mm AM/PM", "gener 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-3]mmmmm dd yyyy  h:mm AM/PM", "g 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-3]mmmmmm dd yyyy  h:mm AM/PM", "gener 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-3]mmm dd yyyy  h:mm AM/PM", "març 19 2019  12:04 p.%A0m."},
		{"43543.503206018519", "[$-3]mmmm dd yyyy  h:mm AM/PM aaa", "març 19 2019  12:04 p.%A0m. dt."},
		{"43543.503206018519", "[$-3]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 p.%A0m. dt."},
		{"43543.503206018519", "[$-3]mmmmmm dd yyyy  h:mm AM/PM dddd", "març 19 2019  12:04 p.%A0m. dimarts"},
		{"44562.189571759256", "[$-403]mmm dd yyyy  h:mm AM/PM", "gen. 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-403]mmmm dd yyyy  h:mm AM/PM", "gener 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-403]mmmmm dd yyyy  h:mm AM/PM", "g 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-403]mmmmmm dd yyyy  h:mm AM/PM", "gener 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-403]mmm dd yyyy  h:mm AM/PM", "març 19 2019  12:04 p.%A0m."},
		{"43543.503206018519", "[$-403]mmmm dd yyyy  h:mm AM/PM aaa", "març 19 2019  12:04 p.%A0m. dt."},
		{"43543.503206018519", "[$-403]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 p.%A0m. dt."},
		{"43543.503206018519", "[$-403]mmmmmm dd yyyy  h:mm AM/PM dddd", "març 19 2019  12:04 p.%A0m. dimarts"},
		{"44562.189571759256", "[$-45F]mmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-45F]mmmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-45F]mmmmm dd yyyy  h:mm AM/PM", "\u0643 01 2022  4:32 \u0635"},
		{"44562.189571759256", "[$-45F]mmmmmm dd yyyy  h:mm AM/PM", "\u0643\u0627\u0646\u0648\u0646%A0\u0627\u0644\u062B\u0627\u0646\u064A 01 2022  4:32 \u0635"},
		{"43543.503206018519", "[$-45F]mmm dd yyyy  h:mm AM/PM", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645"},
		{"43543.503206018519", "[$-45F]mmmm dd yyyy  h:mm AM/PM aaa", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-45F]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0622 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"43543.503206018519", "[$-45F]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0622\u0630\u0627\u0631 19 2019  12:04 \u0645 \u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621"},
		{"44562.189571759256", "[$-92]mmm dd yyyy  h:mm AM/PM", "\u06A9\u0627\u0646\u0648\u0648\u0646\u06CC%20\u062F\u0648\u0648\u06D5\u0645 01 2022  4:32 \u067E.\u0646"},
		{"44562.189571759256", "[$-92]mmmm dd yyyy  h:mm AM/PM", "\u06A9\u0627\u0646\u0648\u0648\u0646\u06CC%20\u062F\u0648\u0648\u06D5\u0645 01 2022  4:32 \u067E.\u0646"},
		{"44562.189571759256", "[$-92]mmmmm dd yyyy  h:mm AM/PM", "\u06A9 01 2022  4:32 \u067E.\u0646"},
		{"44562.189571759256", "[$-92]mmmmmm dd yyyy  h:mm AM/PM", "\u06A9\u0627\u0646\u0648\u0648\u0646\u06CC%20\u062F\u0648\u0648\u06D5\u0645 01 2022  4:32 \u067E.\u0646"},
		{"43543.503206018519", "[$-92]mmm dd yyyy  h:mm AM/PM", "\u0626\u0627\u0632\u0627\u0631 19 2019  12:04 \u062F.\u0646"},
		{"43543.503206018519", "[$-92]mmmm dd yyyy  h:mm AM/PM aaa", "\u0626\u0627\u0632\u0627\u0631 19 2019  12:04 \u062F.\u0646 \u0633\u06CE\u0634\u06D5\u0645\u0645\u06D5"},
		{"43543.503206018519", "[$-92]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0626 19 2019  12:04 \u062F.\u0646 \u0633\u06CE\u0634\u06D5\u0645\u0645\u06D5"},
		{"43543.503206018519", "[$-92]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0626\u0627\u0632\u0627\u0631 19 2019  12:04 \u062F.\u0646 \u0633\u06CE\u0634\u06D5\u0645\u0645\u06D5"},
		{"44562.189571759256", "[$-7C92]mmm dd yyyy  h:mm AM/PM", "\u06A9\u0627\u0646\u0648\u0648\u0646\u06CC%20\u062F\u0648\u0648\u06D5\u0645 01 2022  4:32 \u067E.\u0646"},
		{"44562.189571759256", "[$-7C92]mmmm dd yyyy  h:mm AM/PM", "\u06A9\u0627\u0646\u0648\u0648\u0646\u06CC%20\u062F\u0648\u0648\u06D5\u0645 01 2022  4:32 \u067E.\u0646"},
		{"44562.189571759256", "[$-7C92]mmmmm dd yyyy  h:mm AM/PM", "\u06A9 01 2022  4:32 \u067E.\u0646"},
		{"44562.189571759256", "[$-7C92]mmmmmm dd yyyy  h:mm AM/PM", "\u06A9\u0627\u0646\u0648\u0648\u0646\u06CC%20\u062F\u0648\u0648\u06D5\u0645 01 2022  4:32 \u067E.\u0646"},
		{"43543.503206018519", "[$-7C92]mmm dd yyyy  h:mm AM/PM", "\u0626\u0627\u0632\u0627\u0631 19 2019  12:04 \u062F.\u0646"},
		{"43543.503206018519", "[$-7C92]mmmm dd yyyy  h:mm AM/PM aaa", "\u0626\u0627\u0632\u0627\u0631 19 2019  12:04 \u062F.\u0646 \u0633\u06CE\u0634\u06D5\u0645\u0645\u06D5"},
		{"43543.503206018519", "[$-7C92]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0626 19 2019  12:04 \u062F.\u0646 \u0633\u06CE\u0634\u06D5\u0645\u0645\u06D5"},
		{"43543.503206018519", "[$-7C92]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0626\u0627\u0632\u0627\u0631 19 2019  12:04 \u062F.\u0646 \u0633\u06CE\u0634\u06D5\u0645\u0645\u06D5"},

		{"44562.189571759256", "[$-492]mmm dd yyyy  h:mm AM/PM", "\u06A9\u0627\u0646\u0648\u0648\u0646\u06CC%20\u062F\u0648\u0648\u06D5\u0645 01 2022  4:32 \u067E.\u0646"},
		{"44562.189571759256", "[$-492]mmmm dd yyyy  h:mm AM/PM", "\u06A9\u0627\u0646\u0648\u0648\u0646\u06CC%20\u062F\u0648\u0648\u06D5\u0645 01 2022  4:32 \u067E.\u0646"},
		{"44562.189571759256", "[$-492]mmmmm dd yyyy  h:mm AM/PM", "\u06A9 01 2022  4:32 \u067E.\u0646"},
		{"44562.189571759256", "[$-492]mmmmmm dd yyyy  h:mm AM/PM", "\u06A9\u0627\u0646\u0648\u0648\u0646\u06CC%20\u062F\u0648\u0648\u06D5\u0645 01 2022  4:32 \u067E.\u0646"},
		{"43543.503206018519", "[$-492]mmm dd yyyy  h:mm AM/PM", "\u0626\u0627\u0632\u0627\u0631 19 2019  12:04 \u062F.\u0646"},
		{"43543.503206018519", "[$-492]mmmm dd yyyy  h:mm AM/PM aaa", "\u0626\u0627\u0632\u0627\u0631 19 2019  12:04 \u062F.\u0646 \u0633\u06CE\u0634\u06D5\u0645\u0645\u06D5"},
		{"43543.503206018519", "[$-492]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0626 19 2019  12:04 \u062F.\u0646 \u0633\u06CE\u0634\u06D5\u0645\u0645\u06D5"},
		{"43543.503206018519", "[$-492]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0626\u0627\u0632\u0627\u0631 19 2019  12:04 \u062F.\u0646 \u0633\u06CE\u0634\u06D5\u0645\u0645\u06D5"},
		{"44562.189571759256", "[$-5C]mmm dd yyyy  h:mm AM/PM", "\u13A4\u13C3\u13B8 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-5C]mmmm dd yyyy  h:mm AM/PM", "\u13A4\u13C3\u13B8\u13D4\u13C5 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-5C]mmmmm dd yyyy  h:mm AM/PM", "\u13A4 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-5C]mmmmmm dd yyyy  h:mm AM/PM", "\u13A4\u13C3\u13B8\u13D4\u13C5 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-5C]mmm dd yyyy  h:mm AM/PM", "\u13A0\u13C5\u13F1 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-5C]mmmm dd yyyy  h:mm AM/PM aaa", "\u13A0\u13C5\u13F1 19 2019  12:04 PM \u13D4\u13B5\u13C1"},
		{"43543.503206018519", "[$-5C]mmmmm dd yyyy  h:mm AM/PM ddd", "\u13A0 19 2019  12:04 PM \u13D4\u13B5\u13C1"},
		{"43543.503206018519", "[$-5C]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u13A0\u13C5\u13F1 19 2019  12:04 PM \u13D4\u13B5\u13C1\u13A2\u13A6"},
		{"44562.189571759256", "[$-7C5C]mmm dd yyyy  h:mm AM/PM", "\u13A4\u13C3\u13B8 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C5C]mmmm dd yyyy  h:mm AM/PM", "\u13A4\u13C3\u13B8\u13D4\u13C5 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C5C]mmmmm dd yyyy  h:mm AM/PM", "\u13A4 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C5C]mmmmmm dd yyyy  h:mm AM/PM", "\u13A4\u13C3\u13B8\u13D4\u13C5 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C5C]mmm dd yyyy  h:mm AM/PM", "\u13A0\u13C5\u13F1 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C5C]mmmm dd yyyy  h:mm AM/PM aaa", "\u13A0\u13C5\u13F1 19 2019  12:04 PM \u13D4\u13B5\u13C1"},
		{"43543.503206018519", "[$-7C5C]mmmmm dd yyyy  h:mm AM/PM ddd", "\u13A0 19 2019  12:04 PM \u13D4\u13B5\u13C1"},
		{"43543.503206018519", "[$-7C5C]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u13A0\u13C5\u13F1 19 2019  12:04 PM \u13D4\u13B5\u13C1\u13A2\u13A6"},
		{"44562.189571759256", "[$-45C]mmm dd yyyy  h:mm AM/PM", "\u13A4\u13C3\u13B8 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-45C]mmmm dd yyyy  h:mm AM/PM", "\u13A4\u13C3\u13B8\u13D4\u13C5 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-45C]mmmmm dd yyyy  h:mm AM/PM", "\u13A4 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-45C]mmmmmm dd yyyy  h:mm AM/PM", "\u13A4\u13C3\u13B8\u13D4\u13C5 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-45C]mmm dd yyyy  h:mm AM/PM", "\u13A0\u13C5\u13F1 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-45C]mmmm dd yyyy  h:mm AM/PM aaa", "\u13A0\u13C5\u13F1 19 2019  12:04 PM \u13D4\u13B5\u13C1"},
		{"43543.503206018519", "[$-45C]mmmmm dd yyyy  h:mm AM/PM ddd", "\u13A0 19 2019  12:04 PM \u13D4\u13B5\u13C1"},
		{"43543.503206018519", "[$-45C]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u13A0\u13C5\u13F1 19 2019  12:04 PM \u13D4\u13B5\u13C1\u13A2\u13A6"},
		{"43543.503206018519", "[$-4]mmm dd yyyy  h:mm AM/PM aaa", "3月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-4]mmmm dd yyyy  h:mm AM/PM ddd", "三月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-4]mmmmm dd yyyy  h:mm AM/PM dddd", "三 19 2019  12:04 下午 星期二"},
		{"43543.503206018519", "[$-7804]mmm dd yyyy  h:mm AM/PM aaa", "3月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-7804]mmmm dd yyyy  h:mm AM/PM ddd", "三月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-7804]mmmmm dd yyyy  h:mm AM/PM dddd", "三 19 2019  12:04 下午 星期二"},
		{"43543.503206018519", "[$-804]mmm dd yyyy  h:mm AM/PM aaa", "3月 19 2019  12:04 下午 周二"},
		{"43543.503206018519", "[$-804]mmmm dd yyyy  h:mm AM/PM ddd", "三月 19 2019  12:04 下午 周二"},
		{"43543.503206018519", "[$-804]mmmmm dd yyyy  h:mm AM/PM dddd", "三 19 2019  12:04 下午 星期二"},
		{"43543.503206018519", "[$-1004]mmm dd yyyy  h:mm AM/PM aaa", "三月 19 2019  12:04 下午 周二"},
		{"43543.503206018519", "[$-1004]mmmm dd yyyy  h:mm AM/PM ddd", "三月 19 2019  12:04 下午 周二"},
		{"43543.503206018519", "[$-1004]mmmmm dd yyyy  h:mm AM/PM dddd", "三 19 2019  12:04 下午 星期二"},
		{"43543.503206018519", "[$-7C04]mmm dd yyyy  h:mm AM/PM aaa", "3月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-7C04]mmmm dd yyyy  h:mm AM/PM ddd", "3月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-7C04]mmmmm dd yyyy  h:mm AM/PM dddd", "3 19 2019  12:04 下午 星期二"},
		{"43543.503206018519", "[$-C04]mmm dd yyyy  h:mm AM/PM aaa", "三月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-C04]mmmm dd yyyy  h:mm AM/PM ddd", "三月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-C04]mmmmm dd yyyy  h:mm AM/PM dddd", "三 19 2019  12:04 下午 星期二"},
		{"43543.503206018519", "[$-1404]mmm dd yyyy  h:mm AM/PM aaa", "3月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-1404]mmmm dd yyyy  h:mm AM/PM ddd", "3月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-1404]mmmmm dd yyyy  h:mm AM/PM dddd", "3 19 2019  12:04 下午 星期二"},
		{"43543.503206018519", "[$-404]mmm dd yyyy  h:mm AM/PM aaa", "3月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-404]mmmm dd yyyy  h:mm AM/PM ddd", "3月 19 2019  12:04 下午 週二"},
		{"43543.503206018519", "[$-404]mmmmm dd yyyy  h:mm AM/PM dddd", "3 19 2019  12:04 下午 星期二"},
		{"43543.503206018519", "[$-9]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-9]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-9]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-1000]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-1000]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-1000]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-C09]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 pm Tue"},
		{"43543.503206018519", "[$-C09]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 pm Tue"},
		{"43543.503206018519", "[$-C09]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 pm Tuesday"},
		{"43543.503206018519", "[$-c09]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 pm Tue"},
		{"43543.503206018519", "[$-c09]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 pm Tue"},
		{"43543.503206018519", "[$-c09]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 pm Tuesday"},
		{"43543.503206018519", "[$-2809]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-2809]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-2809]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-1009]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-1009]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-1009]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-2409]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-2409]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-2409]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-3C09]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-3C09]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-3C09]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-4009]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-4009]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-4009]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-1809]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 pm Tue"},
		{"43543.503206018519", "[$-1809]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 pm Tue"},
		{"43543.503206018519", "[$-1809]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 pm Tuesday"},
		{"43543.503206018519", "[$-2009]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-2009]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-2009]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-4409]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-4409]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-4409]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-1409]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-1409]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-1409]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-3409]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-3409]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-3409]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-4809]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-4809]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-4809]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-1C09]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-1C09]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-1C09]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-2C09]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-2C09]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-2C09]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-4C09]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-4C09]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-4C09]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-809]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 pm Tue"},
		{"43543.503206018519", "[$-809]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 pm Tue"},
		{"43543.503206018519", "[$-809]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 pm Tuesday"},
		{"43543.503206018519", "[$-3009]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-3009]mmmm dd yyyy  h:mm AM/PM ddd", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-3009]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Tuesday"},
		{"43543.503206018519", "[$-25]mmm dd yyyy  h:mm AM/PM aaa", "märts 19 2019  12:04 PM T"},
		{"43543.503206018519", "[$-25]mmmm dd yyyy  h:mm AM/PM ddd", "märts 19 2019  12:04 PM T"},
		{"43543.503206018519", "[$-25]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM teisipäev"},
		{"43543.503206018519", "[$-425]mmm dd yyyy  h:mm AM/PM aaa", "märts 19 2019  12:04 PM T"},
		{"43543.503206018519", "[$-425]mmmm dd yyyy  h:mm AM/PM ddd", "märts 19 2019  12:04 PM T"},
		{"43543.503206018519", "[$-425]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM teisipäev"},
		{"43543.503206018519", "[$-38]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 um sein. týs."},
		{"43543.503206018519", "[$-38]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 um sein. týs."},
		{"43543.503206018519", "[$-38]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 um sein. týsdagur"},
		{"43543.503206018519", "[$-438]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 um sein. týs."},
		{"43543.503206018519", "[$-438]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 um sein. týs."},
		{"43543.503206018519", "[$-438]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 um sein. týsdagur"},
		{"43543.503206018519", "[$-64]mmm dd yyyy  h:mm AM/PM aaa", "Mar 19 2019  12:04 PM Mar"},
		{"43543.503206018519", "[$-64]mmmm dd yyyy  h:mm AM/PM ddd", "Marso 19 2019  12:04 PM Mar"},
		{"43543.503206018519", "[$-64]mmmmm dd yyyy  h:mm AM/PM dddd", "03 19 2019  12:04 PM Martes"},
		{"43543.503206018519", "[$-464]mmm dd yyyy  h:mm AM/PM ddd", "Mar 19 2019  12:04 PM Mar"},
		{"43543.503206018519", "[$-464]mmmm dd yyyy  h:mm AM/PM ddd", "Marso 19 2019  12:04 PM Mar"},
		{"43543.503206018519", "[$-464]mmmmm dd yyyy  h:mm AM/PM dddd", "03 19 2019  12:04 PM Martes"},
		{"43543.503206018519", "[$-B]mmm dd yyyy  h:mm AM/PM aaa", "maalis 19 2019  12:04 ip. ti"},
		{"43543.503206018519", "[$-B]mmmm dd yyyy  h:mm AM/PM ddd", "maaliskuu 19 2019  12:04 ip. ti"},
		{"43543.503206018519", "[$-B]mmmmm dd yyyy  h:mm AM/PM dddd", "03 19 2019  12:04 ip. tiistai"},
		{"43543.503206018519", "[$-40B]mmm dd yyyy  h:mm AM/PM aaa", "maalis 19 2019  12:04 ip. ti"},
		{"43543.503206018519", "[$-40B]mmmm dd yyyy  h:mm AM/PM ddd", "maaliskuu 19 2019  12:04 ip. ti"},
		{"43543.503206018519", "[$-40B]mmmmm dd yyyy  h:mm AM/PM dddd", "03 19 2019  12:04 ip. tiistai"},
		{"44562.189571759256", "[$-C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-80C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-80C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-80C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-80C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-80C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-80C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-2c0C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 mat."},
		{"44562.189571759256", "[$-2c0C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 mat."},
		{"44562.189571759256", "[$-2c0C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 mat."},
		{"43543.503206018519", "[$-2c0C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 soir mar."},
		{"43543.503206018519", "[$-2c0C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 soir mar."},
		{"43543.503206018519", "[$-2c0C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 soir mardi"},
		{"44562.189571759256", "[$-c0C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-c0C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-c0C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-c0C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-c0C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-c0C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-1C0C]mmm dd yyyy  h:mm AM/PM", "Janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1C0C]mmmm dd yyyy  h:mm AM/PM", "Janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1C0C]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-1C0C]mmm dd yyyy  h:mm AM/PM aaa", "Mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-1C0C]mmmm dd yyyy  h:mm AM/PM ddd", "Mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-1C0C]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-240C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-240C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-240C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-240C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-240C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-240C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-300C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-300C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-300C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-300C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-300C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-300C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-40C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-40C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-40C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-40C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-40C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-40C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-3c0C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-3c0C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-3c0C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-3c0C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-3c0C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-3c0C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-140C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-140C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-140C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-140C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-140C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-140C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-340C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-340C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-340C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-340C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-340C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-340C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-380C]mmm dd yyyy  h:mm AM/PM", "jan. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-380C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-380C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-380C]mmm dd yyyy  h:mm AM/PM aaa", "mar. 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-380C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-380C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-180C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-180C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-180C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-180C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-180C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-180C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-200C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-200C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-200C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-200C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-200C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-200C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-280C]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-280C]mmmm dd yyyy  h:mm AM/PM", "janvier 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-280C]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-280C]mmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-280C]mmmm dd yyyy  h:mm AM/PM ddd", "mars 19 2019  12:04 PM mar."},
		{"43543.503206018519", "[$-280C]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-62]m dd yyyy  h:mm AM/PM", "1 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-62]mm dd yyyy  h:mm AM/PM", "01 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-62]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-62]mmmm dd yyyy  h:mm AM/PM", "Jannewaris 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-62]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-62]mmmmmm dd yyyy  h:mm AM/PM", "Jannewaris 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-62]m dd yyyy  h:mm AM/PM", "3 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-62]mm dd yyyy  h:mm AM/PM", "03 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-62]mmm dd yyyy  h:mm AM/PM", "Mrt 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-62]mmmm dd yyyy  h:mm AM/PM aaa", "Maart 19 2019  12:04 PM tii"},
		{"43543.503206018519", "[$-62]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM tii"},
		{"43543.503206018519", "[$-62]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maart 19 2019  12:04 PM tiisdei"},
		{"44562.189571759256", "[$-462]m dd yyyy  h:mm AM/PM", "1 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-462]mm dd yyyy  h:mm AM/PM", "01 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-462]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-462]mmmm dd yyyy  h:mm AM/PM", "Jannewaris 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-462]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-462]mmmmmm dd yyyy  h:mm AM/PM", "Jannewaris 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-462]m dd yyyy  h:mm AM/PM", "3 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-462]mm dd yyyy  h:mm AM/PM", "03 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-462]mmm dd yyyy  h:mm AM/PM", "Mrt 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-462]mmmm dd yyyy  h:mm AM/PM aaa", "Maart 19 2019  12:04 PM tii"},
		{"43543.503206018519", "[$-462]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM tii"},
		{"43543.503206018519", "[$-462]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maart 19 2019  12:04 PM tiisdei"},
		{"44562.189571759256", "[$-67]mmm dd yyyy  h:mm AM/PM", "sii 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-67]mmmm dd yyyy  h:mm AM/PM", "siilo 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-67]mmmmm dd yyyy  h:mm AM/PM", "s 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-67]mmmmmm dd yyyy  h:mm AM/PM", "siilo 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-67]mmm dd yyyy  h:mm AM/PM", "mbo 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-67]mmmm dd yyyy  h:mm AM/PM aaa", "mbooy 19 2019  12:04 PM maw"},
		{"43543.503206018519", "[$-67]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM maw"},
		{"43543.503206018519", "[$-67]mmmmmm dd yyyy  h:mm AM/PM dddd", "mbooy 19 2019  12:04 PM mawbaare"},
		{"44562.189571759256", "[$-7C67]mmm dd yyyy  h:mm AM/PM", "sii 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C67]mmmm dd yyyy  h:mm AM/PM", "siilo 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C67]mmmmm dd yyyy  h:mm AM/PM", "s 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C67]mmmmmm dd yyyy  h:mm AM/PM", "siilo 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C67]mmm dd yyyy  h:mm AM/PM", "mbo 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C67]mmmm dd yyyy  h:mm AM/PM aaa", "mbooy 19 2019  12:04 PM maw"},
		{"43543.503206018519", "[$-7C67]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM maw"},
		{"43543.503206018519", "[$-7C67]mmmmmm dd yyyy  h:mm AM/PM dddd", "mbooy 19 2019  12:04 PM mawbaare"},
		{"44562.189571759256", "[$-467]mmm dd yyyy  h:mm AM/PM", "samw 01 2022  4:32 subaka"},
		{"44562.189571759256", "[$-467]mmmm dd yyyy  h:mm AM/PM", "samwiee 01 2022  4:32 subaka"},
		{"44562.189571759256", "[$-467]mmmmm dd yyyy  h:mm AM/PM", "s 01 2022  4:32 subaka"},
		{"44562.189571759256", "[$-467]mmmmmm dd yyyy  h:mm AM/PM", "samwiee 01 2022  4:32 subaka"},
		{"43543.503206018519", "[$-467]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 kikiiɗe"},
		{"43543.503206018519", "[$-467]mmmm dd yyyy  h:mm AM/PM aaa", "marsa 19 2019  12:04 kikiiɗe tal."},
		{"43543.503206018519", "[$-467]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 kikiiɗe tal."},
		{"43543.503206018519", "[$-467]mmmmmm dd yyyy  h:mm AM/PM dddd", "marsa 19 2019  12:04 kikiiɗe talaata"},
		{"44562.189571759256", "[$-867]mmm dd yyyy  h:mm AM/PM", "samw 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-867]mmmm dd yyyy  h:mm AM/PM", "samwiee 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-867]mmmmm dd yyyy  h:mm AM/PM", "s 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-867]mmmmmm dd yyyy  h:mm AM/PM", "samwiee 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-867]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-867]mmmm dd yyyy  h:mm AM/PM aaa", "marsa 19 2019  12:04 PM tal."},
		{"43543.503206018519", "[$-867]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM tal."},
		{"43543.503206018519", "[$-867]mmmmmm dd yyyy  h:mm AM/PM dddd", "marsa 19 2019  12:04 PM talaata"},
		{"44562.189571759256", "[$-56]mmm dd yyyy  h:mm AM/PM", "Xan. 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-56]mmmm dd yyyy  h:mm AM/PM", "Xaneiro 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-56]mmmmm dd yyyy  h:mm AM/PM", "X 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-56]mmmmmm dd yyyy  h:mm AM/PM", "Xaneiro 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-56]mmm dd yyyy  h:mm AM/PM", "Mar. 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-56]mmmm dd yyyy  h:mm AM/PM", "Marzo 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-56]mmmmm dd yyyy  h:mm AM/PM", "M 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-56]mmmmmm dd yyyy  h:mm AM/PM", "Marzo 19 2019  12:04 p.m."},
		{"44562.189571759256", "[$-56]mmm dd yyyy  h:mm AM/PM", "Xan. 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-56]mmmm dd yyyy  h:mm AM/PM", "Xaneiro 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-56]mmmmm dd yyyy  h:mm AM/PM", "X 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-56]mmmmmm dd yyyy  h:mm AM/PM", "Xaneiro 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-56]mmm dd yyyy  h:mm AM/PM aaa", "Mar. 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-56]mmmm dd yyyy  h:mm AM/PM ddd", "Marzo 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-56]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 p.m. martes"},
		{"43543.503206018519", "[$-56]mmmmmm dd yyyy  h:mm AM/PM", "Marzo 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-456]mmm dd yyyy  h:mm AM/PM aaa", "Mar. 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-456]mmmm dd yyyy  h:mm AM/PM ddd", "Marzo 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-456]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 p.m. martes"},
		{"44562.189571759256", "[$-37]mmm dd yyyy  h:mm AM/PM", "\u10D8\u10D0\u10DC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-37]mmmm dd yyyy  h:mm AM/PM", "\u10D8\u10D0\u10DC\u10D5\u10D0\u10E0\u10D8 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-37]mmmmm dd yyyy  h:mm AM/PM", "\u10D8 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-37]mmmmmm dd yyyy  h:mm AM/PM", "\u10D8\u10D0\u10DC\u10D5\u10D0\u10E0\u10D8 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-37]mmm dd yyyy  h:mm AM/PM", "\u10DB\u10D0\u10E0 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-37]mmmm dd yyyy  h:mm AM/PM aaa", "\u10DB\u10D0\u10E0\u10E2\u10D8 19 2019  12:04 PM \u10E1\u10D0\u10DB\u10E8."},
		{"43543.503206018519", "[$-37]mmmmm dd yyyy  h:mm AM/PM ddd", "\u10DB 19 2019  12:04 PM \u10E1\u10D0\u10DB\u10E8."},
		{"43543.503206018519", "[$-37]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u10DB\u10D0\u10E0\u10E2\u10D8 19 2019  12:04 PM \u10E1\u10D0\u10DB\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8"},
		{"44562.189571759256", "[$-437]mmm dd yyyy  h:mm AM/PM", "\u10D8\u10D0\u10DC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-437]mmmm dd yyyy  h:mm AM/PM", "\u10D8\u10D0\u10DC\u10D5\u10D0\u10E0\u10D8 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-437]mmmmm dd yyyy  h:mm AM/PM", "\u10D8 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-437]mmmmmm dd yyyy  h:mm AM/PM", "\u10D8\u10D0\u10DC\u10D5\u10D0\u10E0\u10D8 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-437]mmm dd yyyy  h:mm AM/PM", "\u10DB\u10D0\u10E0 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-437]mmmm dd yyyy  h:mm AM/PM aaa", "\u10DB\u10D0\u10E0\u10E2\u10D8 19 2019  12:04 PM \u10E1\u10D0\u10DB\u10E8."},
		{"43543.503206018519", "[$-437]mmmmm dd yyyy  h:mm AM/PM ddd", "\u10DB 19 2019  12:04 PM \u10E1\u10D0\u10DB\u10E8."},
		{"43543.503206018519", "[$-437]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u10DB\u10D0\u10E0\u10E2\u10D8 19 2019  12:04 PM \u10E1\u10D0\u10DB\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8"},
		{"43543.503206018519", "[$-7]mmm dd yyyy  h:mm AM/PM aaa", "Mär 19 2019  12:04 PM Di"},
		{"43543.503206018519", "[$-7]mmmm dd yyyy  h:mm AM/PM ddd", "März 19 2019  12:04 PM Di"},
		{"43543.503206018519", "[$-7]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Dienstag"},
		{"44562.189571759256", "[$-C07]mmm dd yyyy  h:mm AM/PM aaa", "Jän 01 2022  4:32 AM Sa"},
		{"44562.189571759256", "[$-C07]mmmm dd yyyy  h:mm AM/PM ddd", "Jänner 01 2022  4:32 AM Sa"},
		{"44562.189571759256", "[$-C07]mmmmm dd yyyy  h:mm AM/PM dddd", "J 01 2022  4:32 AM Samstag"},
		{"43543.503206018519", "[$-407]mmm dd yyyy  h:mm AM/PM aaa", "Mär 19 2019  12:04 PM Di"},
		{"43543.503206018519", "[$-407]mmmm dd yyyy  h:mm AM/PM ddd", "März 19 2019  12:04 PM Di"},
		{"43543.503206018519", "[$-407]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Dienstag"},
		{"43543.503206018519", "[$-1407]mmm dd yyyy  h:mm AM/PM aaa", "Mär 19 2019  12:04 PM Di"},
		{"43543.503206018519", "[$-1407]mmmm dd yyyy  h:mm AM/PM ddd", "März 19 2019  12:04 PM Di"},
		{"43543.503206018519", "[$-1407]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Dienstag"},
		{"43543.503206018519", "[$-807]mmm dd yyyy  h:mm AM/PM aaa", "Mär 19 2019  12:04 PM Di"},
		{"43543.503206018519", "[$-807]mmmm dd yyyy  h:mm AM/PM ddd", "März 19 2019  12:04 PM Di"},
		{"43543.503206018519", "[$-807]mmmmm dd yyyy  h:mm AM/PM dddd", "M 19 2019  12:04 PM Dienstag"},
		{"44562.189571759256", "[$-8]mmm dd yyyy  h:mm AM/PM", "\u0399\u03B1\u03BD 01 2022  4:32 \u03C0\u03BC"},
		{"44562.189571759256", "[$-8]mmmm dd yyyy  h:mm AM/PM", "\u0399\u03B1\u03BD\u03BF\u03C5\u03AC\u03C1\u03B9\u03BF\u03C2 01 2022  4:32 \u03C0\u03BC"},
		{"44562.189571759256", "[$-8]mmmmm dd yyyy  h:mm AM/PM", "\u0399 01 2022  4:32 \u03C0\u03BC"},
		{"44562.189571759256", "[$-8]mmmmmm dd yyyy  h:mm AM/PM", "\u0399\u03B1\u03BD\u03BF\u03C5\u03AC\u03C1\u03B9\u03BF\u03C2 01 2022  4:32 \u03C0\u03BC"},
		{"43543.503206018519", "[$-8]mmm dd yyyy  h:mm AM/PM", "\u039C\u03B1\u03C1 19 2019  12:04 \u03BC\u03BC"},
		{"43543.503206018519", "[$-8]mmmm dd yyyy  h:mm AM/PM aaa", "\u039C\u03AC\u03C1\u03C4\u03B9\u03BF\u03C2 19 2019  12:04 \u03BC\u03BC \u03A4\u03C1\u03B9"},
		{"43543.503206018519", "[$-8]mmmmm dd yyyy  h:mm AM/PM ddd", "\u039C 19 2019  12:04 \u03BC\u03BC \u03A4\u03C1\u03B9"},
		{"43543.503206018519", "[$-8]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u039C\u03AC\u03C1\u03C4\u03B9\u03BF\u03C2 19 2019  12:04 \u03BC\u03BC \u03A4\u03C1\u03AF\u03C4\u03B7"},
		{"44562.189571759256", "[$-408]mmm dd yyyy  h:mm AM/PM", "\u0399\u03B1\u03BD 01 2022  4:32 \u03C0\u03BC"},
		{"44562.189571759256", "[$-408]mmmm dd yyyy  h:mm AM/PM", "\u0399\u03B1\u03BD\u03BF\u03C5\u03AC\u03C1\u03B9\u03BF\u03C2 01 2022  4:32 \u03C0\u03BC"},
		{"44562.189571759256", "[$-408]mmmmm dd yyyy  h:mm AM/PM", "\u0399 01 2022  4:32 \u03C0\u03BC"},
		{"44562.189571759256", "[$-408]mmmmmm dd yyyy  h:mm AM/PM", "\u0399\u03B1\u03BD\u03BF\u03C5\u03AC\u03C1\u03B9\u03BF\u03C2 01 2022  4:32 \u03C0\u03BC"},
		{"43543.503206018519", "[$-408]mmm dd yyyy  h:mm AM/PM", "\u039C\u03B1\u03C1 19 2019  12:04 \u03BC\u03BC"},
		{"43543.503206018519", "[$-408]mmmm dd yyyy  h:mm AM/PM aaa", "\u039C\u03AC\u03C1\u03C4\u03B9\u03BF\u03C2 19 2019  12:04 \u03BC\u03BC \u03A4\u03C1\u03B9"},
		{"43543.503206018519", "[$-408]mmmmm dd yyyy  h:mm AM/PM ddd", "\u039C 19 2019  12:04 \u03BC\u03BC \u03A4\u03C1\u03B9"},
		{"43543.503206018519", "[$-408]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u039C\u03AC\u03C1\u03C4\u03B9\u03BF\u03C2 19 2019  12:04 \u03BC\u03BC \u03A4\u03C1\u03AF\u03C4\u03B7"},
		{"44562.189571759256", "[$-6F]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6F]mmmm dd yyyy  h:mm AM/PM", "januaari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6F]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6F]mmmmmm dd yyyy  h:mm AM/PM", "januaari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-6F]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-6F]mmmm dd yyyy  h:mm AM/PM aaa", "marsi 19 2019  12:04 PM marl."},
		{"43543.503206018519", "[$-6F]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM marl."},
		{"43543.503206018519", "[$-6F]mmmmmm dd yyyy  h:mm AM/PM dddd", "marsi 19 2019  12:04 PM marlunngorneq"},
		{"44562.189571759256", "[$-46F]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46F]mmmm dd yyyy  h:mm AM/PM", "januaari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46F]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46F]mmmmmm dd yyyy  h:mm AM/PM", "januaari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-46F]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-46F]mmmm dd yyyy  h:mm AM/PM aaa", "marsi 19 2019  12:04 PM marl."},
		{"43543.503206018519", "[$-46F]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM marl."},
		{"43543.503206018519", "[$-46F]mmmmmm dd yyyy  h:mm AM/PM dddd", "marsi 19 2019  12:04 PM marlunngorneq"},
		{"44562.189571759256", "[$-74]mmm dd yyyy  h:mm AM/PM", "jteĩ 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-74]mmmm dd yyyy  h:mm AM/PM", "jasyteĩ 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-74]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-74]mmmmmm dd yyyy  h:mm AM/PM", "jasyteĩ 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-74]mmm dd yyyy  h:mm AM/PM", "japy 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-74]mmmm dd yyyy  h:mm AM/PM aaa", "jasyapy 19 2019  12:04 p.m. apy"},
		{"43543.503206018519", "[$-74]mmmmm dd yyyy  h:mm AM/PM ddd", "j 19 2019  12:04 p.m. apy"},
		{"43543.503206018519", "[$-74]mmmmmm dd yyyy  h:mm AM/PM dddd", "jasyapy 19 2019  12:04 p.m. araapy"},
		{"44562.189571759256", "[$-474]mmm dd yyyy  h:mm AM/PM", "jteĩ 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-474]mmmm dd yyyy  h:mm AM/PM", "jasyteĩ 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-474]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-474]mmmmmm dd yyyy  h:mm AM/PM", "jasyteĩ 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-474]mmm dd yyyy  h:mm AM/PM", "japy 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-474]mmmm dd yyyy  h:mm AM/PM aaa", "jasyapy 19 2019  12:04 p.m. apy"},
		{"43543.503206018519", "[$-474]mmmmm dd yyyy  h:mm AM/PM ddd", "j 19 2019  12:04 p.m. apy"},
		{"43543.503206018519", "[$-474]mmmmmm dd yyyy  h:mm AM/PM dddd", "jasyapy 19 2019  12:04 p.m. araapy"},
		{"44562.189571759256", "[$-47]mmm dd yyyy  h:mm AM/PM", "\u0A9C\u0ABE\u0AA8\u0ACD\u0AAF\u0AC1 01 2022  4:32 \u0AAA\u0AC2\u0AB0\u0ACD\u0AB5 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8"},
		{"44562.189571759256", "[$-47]mmmm dd yyyy  h:mm AM/PM", "\u0A9C\u0ABE\u0AA8\u0ACD\u0AAF\u0AC1\u0A86\u0AB0\u0AC0 01 2022  4:32 \u0AAA\u0AC2\u0AB0\u0ACD\u0AB5 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8"},
		{"44562.189571759256", "[$-47]mmmmm dd yyyy  h:mm AM/PM", "\u0A9C 01 2022  4:32 \u0AAA\u0AC2\u0AB0\u0ACD\u0AB5 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8"},
		{"44562.189571759256", "[$-47]mmmmmm dd yyyy  h:mm AM/PM", "\u0A9C\u0ABE\u0AA8\u0ACD\u0AAF\u0AC1\u0A86\u0AB0\u0AC0 01 2022  4:32 \u0AAA\u0AC2\u0AB0\u0ACD\u0AB5 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8"},
		{"43543.503206018519", "[$-47]mmm dd yyyy  h:mm AM/PM", "\u0AAE\u0ABE\u0AB0\u0ACD\u0A9A 19 2019  12:04 \u0A89\u0AA4\u0ACD\u0AA4\u0AB0 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8"},
		{"43543.503206018519", "[$-47]mmmm dd yyyy  h:mm AM/PM aaa", "\u0AAE\u0ABE\u0AB0\u0ACD\u0A9A 19 2019  12:04 \u0A89\u0AA4\u0ACD\u0AA4\u0AB0 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8 \u0AAE\u0A82\u0A97\u0AB3"},
		{"43543.503206018519", "[$-47]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0AAE 19 2019  12:04 \u0A89\u0AA4\u0ACD\u0AA4\u0AB0 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8 \u0AAE\u0A82\u0A97\u0AB3"},
		{"43543.503206018519", "[$-47]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0AAE\u0ABE\u0AB0\u0ACD\u0A9A 19 2019  12:04 \u0A89\u0AA4\u0ACD\u0AA4\u0AB0 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8 \u0AAE\u0A82\u0A97\u0AB3\u0AB5\u0ABE\u0AB0"},
		{"44562.189571759256", "[$-447]mmm dd yyyy  h:mm AM/PM", "\u0A9C\u0ABE\u0AA8\u0ACD\u0AAF\u0AC1 01 2022  4:32 \u0AAA\u0AC2\u0AB0\u0ACD\u0AB5 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8"},
		{"44562.189571759256", "[$-447]mmmm dd yyyy  h:mm AM/PM", "\u0A9C\u0ABE\u0AA8\u0ACD\u0AAF\u0AC1\u0A86\u0AB0\u0AC0 01 2022  4:32 \u0AAA\u0AC2\u0AB0\u0ACD\u0AB5 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8"},
		{"44562.189571759256", "[$-447]mmmmm dd yyyy  h:mm AM/PM", "\u0A9C 01 2022  4:32 \u0AAA\u0AC2\u0AB0\u0ACD\u0AB5 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8"},
		{"44562.189571759256", "[$-447]mmmmmm dd yyyy  h:mm AM/PM", "\u0A9C\u0ABE\u0AA8\u0ACD\u0AAF\u0AC1\u0A86\u0AB0\u0AC0 01 2022  4:32 \u0AAA\u0AC2\u0AB0\u0ACD\u0AB5 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8"},
		{"43543.503206018519", "[$-447]mmm dd yyyy  h:mm AM/PM", "\u0AAE\u0ABE\u0AB0\u0ACD\u0A9A 19 2019  12:04 \u0A89\u0AA4\u0ACD\u0AA4\u0AB0 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8"},
		{"43543.503206018519", "[$-447]mmmm dd yyyy  h:mm AM/PM aaa", "\u0AAE\u0ABE\u0AB0\u0ACD\u0A9A 19 2019  12:04 \u0A89\u0AA4\u0ACD\u0AA4\u0AB0 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8 \u0AAE\u0A82\u0A97\u0AB3"},
		{"43543.503206018519", "[$-447]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0AAE 19 2019  12:04 \u0A89\u0AA4\u0ACD\u0AA4\u0AB0 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8 \u0AAE\u0A82\u0A97\u0AB3"},
		{"43543.503206018519", "[$-447]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0AAE\u0ABE\u0AB0\u0ACD\u0A9A 19 2019  12:04 \u0A89\u0AA4\u0ACD\u0AA4\u0AB0 \u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8 \u0AAE\u0A82\u0A97\u0AB3\u0AB5\u0ABE\u0AB0"},
		{"44562.189571759256", "[$-68]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-68]mmmm dd yyyy  h:mm AM/PM", "Janairu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-68]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-68]mmmmmm dd yyyy  h:mm AM/PM", "Janairu 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-68]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-68]mmmm dd yyyy  h:mm AM/PM aaa", "Maris 19 2019  12:04 PM Tal"},
		{"43543.503206018519", "[$-68]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Tal"},
		{"43543.503206018519", "[$-68]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maris 19 2019  12:04 PM Talata"},
		{"44562.189571759256", "[$-7C68]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C68]mmmm dd yyyy  h:mm AM/PM", "Janairu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C68]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C68]mmmmmm dd yyyy  h:mm AM/PM", "Janairu 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C68]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C68]mmmm dd yyyy  h:mm AM/PM aaa", "Maris 19 2019  12:04 PM Tal"},
		{"43543.503206018519", "[$-7C68]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Tal"},
		{"43543.503206018519", "[$-7C68]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maris 19 2019  12:04 PM Talata"},
		{"44562.189571759256", "[$-468]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-468]mmmm dd yyyy  h:mm AM/PM", "Janairu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-468]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-468]mmmmmm dd yyyy  h:mm AM/PM", "Janairu 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-468]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-468]mmmm dd yyyy  h:mm AM/PM aaa", "Maris 19 2019  12:04 PM Tal"},
		{"43543.503206018519", "[$-468]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Tal"},
		{"43543.503206018519", "[$-468]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maris 19 2019  12:04 PM Talata"},
		{"44562.189571759256", "[$-75]mmm dd yyyy  h:mm AM/PM", "Ian. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-75]mmmm dd yyyy  h:mm AM/PM", "Ianuali 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-75]mmmmm dd yyyy  h:mm AM/PM", "I 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-75]mmmmmm dd yyyy  h:mm AM/PM", "Ianuali 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-75]mmm dd yyyy  h:mm AM/PM", "Mal. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-75]mmmm dd yyyy  h:mm AM/PM aaa", "Malaki 19 2019  12:04 PM P2"},
		{"43543.503206018519", "[$-75]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM P2"},
		{"43543.503206018519", "[$-75]mmmmmm dd yyyy  h:mm AM/PM dddd", "Malaki 19 2019  12:04 PM Poʻalua"},
		{"44562.189571759256", "[$-475]mmm dd yyyy  h:mm AM/PM", "Ian. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-475]mmmm dd yyyy  h:mm AM/PM", "Ianuali 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-475]mmmmm dd yyyy  h:mm AM/PM", "I 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-475]mmmmmm dd yyyy  h:mm AM/PM", "Ianuali 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-475]mmm dd yyyy  h:mm AM/PM", "Mal. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-475]mmmm dd yyyy  h:mm AM/PM aaa", "Malaki 19 2019  12:04 PM P2"},
		{"43543.503206018519", "[$-475]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM P2"},
		{"43543.503206018519", "[$-475]mmmmmm dd yyyy  h:mm AM/PM dddd", "Malaki 19 2019  12:04 PM Poʻalua"},
		{"44562.189571759256", "[$-D]mmm dd yyyy  h:mm AM/PM", "\u05D9\u05E0\u05D5 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-D]mmmm dd yyyy  h:mm AM/PM", "\u05D9\u05E0\u05D5\u05D0\u05E8 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-D]mmmmm dd yyyy  h:mm AM/PM", "\u05D9 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-D]mmmmmm dd yyyy  h:mm AM/PM", "\u05D9\u05E0\u05D5\u05D0\u05E8 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-D]mmm dd yyyy  h:mm AM/PM", "\u05DE\u05E8\u05E5 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-D]mmmm dd yyyy  h:mm AM/PM aaa", "\u05DE\u05E8\u05E5 19 2019  12:04 PM \u05D9\u05D5\u05DD%A0\u05D2"},
		{"43543.503206018519", "[$-D]mmmmm dd yyyy  h:mm AM/PM ddd", "\u05DE 19 2019  12:04 PM \u05D9\u05D5\u05DD%A0\u05D2"},
		{"43543.503206018519", "[$-D]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u05DE\u05E8\u05E5 19 2019  12:04 PM \u05D9\u05D5\u05DD%A0\u05E9\u05DC\u05D9\u05E9\u05D9"},
		{"44562.189571759256", "[$-40D]mmm dd yyyy  h:mm AM/PM", "\u05D9\u05E0\u05D5 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-40D]mmmm dd yyyy  h:mm AM/PM", "\u05D9\u05E0\u05D5\u05D0\u05E8 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-40D]mmmmm dd yyyy  h:mm AM/PM", "\u05D9 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-40D]mmmmmm dd yyyy  h:mm AM/PM", "\u05D9\u05E0\u05D5\u05D0\u05E8 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-40D]mmm dd yyyy  h:mm AM/PM", "\u05DE\u05E8\u05E5 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-40D]mmmm dd yyyy  h:mm AM/PM aaa", "\u05DE\u05E8\u05E5 19 2019  12:04 PM \u05D9\u05D5\u05DD%A0\u05D2"},
		{"43543.503206018519", "[$-40D]mmmmm dd yyyy  h:mm AM/PM ddd", "\u05DE 19 2019  12:04 PM \u05D9\u05D5\u05DD%A0\u05D2"},
		{"43543.503206018519", "[$-40D]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u05DE\u05E8\u05E5 19 2019  12:04 PM \u05D9\u05D5\u05DD%A0\u05E9\u05DC\u05D9\u05E9\u05D9"},
		{"44562.189571759256", "[$-39]mmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-39]mmmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-39]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-39]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"43543.503206018519", "[$-39]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928"},
		{"43543.503206018519", "[$-39]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0902\u0917\u0932."},
		{"43543.503206018519", "[$-39]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0902\u0917\u0932."},
		{"43543.503206018519", "[$-39]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0902\u0917\u0932\u0935\u093E\u0930"},
		{"44562.189571759256", "[$-439]mmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-439]mmmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-439]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-439]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"43543.503206018519", "[$-439]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928"},
		{"43543.503206018519", "[$-439]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0902\u0917\u0932."},
		{"43543.503206018519", "[$-439]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0902\u0917\u0932."},
		{"43543.503206018519", "[$-439]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0902\u0917\u0932\u0935\u093E\u0930"},
		{"44562.189571759256", "[$-E]mmm dd yyyy  h:mm AM/PM", "jan. 01 2022  4:32 de."},
		{"44562.189571759256", "[$-E]mmmm dd yyyy  h:mm AM/PM", "január 01 2022  4:32 de."},
		{"44562.189571759256", "[$-E]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 de."},
		{"44562.189571759256", "[$-E]mmmmmm dd yyyy  h:mm AM/PM", "január 01 2022  4:32 de."},
		{"43543.503206018519", "[$-E]mmm dd yyyy  h:mm AM/PM", "márc. 19 2019  12:04 du."},
		{"43543.503206018519", "[$-E]mmmm dd yyyy  h:mm AM/PM aaa", "március 19 2019  12:04 du. K"},
		{"43543.503206018519", "[$-E]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 du. K"},
		{"43543.503206018519", "[$-E]mmmmmm dd yyyy  h:mm AM/PM dddd", "március 19 2019  12:04 du. kedd"},
		{"44562.189571759256", "[$-40E]mmm dd yyyy  h:mm AM/PM", "jan. 01 2022  4:32 de."},
		{"44562.189571759256", "[$-40E]mmmm dd yyyy  h:mm AM/PM", "január 01 2022  4:32 de."},
		{"44562.189571759256", "[$-40E]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 de."},
		{"44562.189571759256", "[$-40E]mmmmmm dd yyyy  h:mm AM/PM", "január 01 2022  4:32 de."},
		{"43543.503206018519", "[$-40E]mmm dd yyyy  h:mm AM/PM", "márc. 19 2019  12:04 du."},
		{"43543.503206018519", "[$-40E]mmmm dd yyyy  h:mm AM/PM aaa", "március 19 2019  12:04 du. K"},
		{"43543.503206018519", "[$-40E]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 du. K"},
		{"43543.503206018519", "[$-40E]mmmmmm dd yyyy  h:mm AM/PM dddd", "március 19 2019  12:04 du. kedd"},
		{"44562.189571759256", "[$-F]mmm dd yyyy  h:mm AM/PM", "jan. 01 2022  4:32 f.h."},
		{"44562.189571759256", "[$-F]mmmm dd yyyy  h:mm AM/PM", "janúar 01 2022  4:32 f.h."},
		{"44562.189571759256", "[$-F]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 f.h."},
		{"44562.189571759256", "[$-F]mmmmmm dd yyyy  h:mm AM/PM", "janúar 01 2022  4:32 f.h."},
		{"43543.503206018519", "[$-F]mmm dd yyyy  h:mm AM/PM", "mar. 19 2019  12:04 e.h."},
		{"43543.503206018519", "[$-F]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 e.h. þri."},
		{"43543.503206018519", "[$-F]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 e.h. þri."},
		{"43543.503206018519", "[$-F]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 e.h. þriðjudagur"},
		{"44562.189571759256", "[$-40F]mmm dd yyyy  h:mm AM/PM", "jan. 01 2022  4:32 f.h."},
		{"44562.189571759256", "[$-40F]mmmm dd yyyy  h:mm AM/PM", "janúar 01 2022  4:32 f.h."},
		{"44562.189571759256", "[$-40F]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 f.h."},
		{"44562.189571759256", "[$-40F]mmmmmm dd yyyy  h:mm AM/PM", "janúar 01 2022  4:32 f.h."},
		{"43543.503206018519", "[$-40F]mmm dd yyyy  h:mm AM/PM", "mar. 19 2019  12:04 e.h."},
		{"43543.503206018519", "[$-40F]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 e.h. þri."},
		{"43543.503206018519", "[$-40F]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 e.h. þri."},
		{"43543.503206018519", "[$-40F]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 e.h. þriðjudagur"},
		{"44562.189571759256", "[$-70]mmm dd yyyy  h:mm AM/PM", "Jen 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-70]mmmm dd yyyy  h:mm AM/PM", "Jenụwarị 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-70]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-70]mmmmmm dd yyyy  h:mm AM/PM", "Jenụwarị 01 2022  4:32 A.M."},
		{"43543.503206018519", "[$-70]mmm dd yyyy  h:mm AM/PM", "Mac 19 2019  12:04 P.M."},
		{"43543.503206018519", "[$-70]mmmm dd yyyy  h:mm AM/PM aaa", "Machị 19 2019  12:04 P.M. Tiu"},
		{"43543.503206018519", "[$-70]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 P.M. Tiu"},
		{"43543.503206018519", "[$-70]mmmmmm dd yyyy  h:mm AM/PM dddd", "Machị 19 2019  12:04 P.M. Tiuzdee"},
		{"44562.189571759256", "[$-470]mmm dd yyyy  h:mm AM/PM", "Jen 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-470]mmmm dd yyyy  h:mm AM/PM", "Jenụwarị 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-470]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 A.M."},
		{"44562.189571759256", "[$-470]mmmmmm dd yyyy  h:mm AM/PM", "Jenụwarị 01 2022  4:32 A.M."},
		{"43543.503206018519", "[$-470]mmm dd yyyy  h:mm AM/PM", "Mac 19 2019  12:04 P.M."},
		{"43543.503206018519", "[$-470]mmmm dd yyyy  h:mm AM/PM aaa", "Machị 19 2019  12:04 P.M. Tiu"},
		{"43543.503206018519", "[$-470]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 P.M. Tiu"},
		{"43543.503206018519", "[$-470]mmmmmm dd yyyy  h:mm AM/PM dddd", "Machị 19 2019  12:04 P.M. Tiuzdee"},
		{"44562.189571759256", "[$-21]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-21]mmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-21]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-21]mmmmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-21]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-21]mmmm dd yyyy  h:mm AM/PM aaa", "Maret 19 2019  12:04 PM Sel"},
		{"43543.503206018519", "[$-21]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Sel"},
		{"43543.503206018519", "[$-21]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maret 19 2019  12:04 PM Selasa"},
		{"44562.189571759256", "[$-421]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-421]mmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-421]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-421]mmmmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-421]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-421]mmmm dd yyyy  h:mm AM/PM aaa", "Maret 19 2019  12:04 PM Sel"},
		{"43543.503206018519", "[$-421]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Sel"},
		{"43543.503206018519", "[$-421]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maret 19 2019  12:04 PM Selasa"},
		{"44562.189571759256", "[$-5D]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-5D]mmmm dd yyyy  h:mm AM/PM", "Jaannuari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-5D]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-5D]mmmmmm dd yyyy  h:mm AM/PM", "Jaannuari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-5D]mmm dd yyyy  h:mm AM/PM", "Mas 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-5D]mmmm dd yyyy  h:mm AM/PM aaa", "Maatsi 19 2019  12:04 PM Aip"},
		{"43543.503206018519", "[$-5D]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Aip"},
		{"43543.503206018519", "[$-5D]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maatsi 19 2019  12:04 PM Aippiq"},
		{"44562.189571759256", "[$-7C5D]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C5D]mmmm dd yyyy  h:mm AM/PM", "Jaannuari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C5D]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C5D]mmmmmm dd yyyy  h:mm AM/PM", "Jaannuari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C5D]mmm dd yyyy  h:mm AM/PM", "Mas 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C5D]mmmm dd yyyy  h:mm AM/PM aaa", "Maatsi 19 2019  12:04 PM Aip"},
		{"43543.503206018519", "[$-7C5D]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Aip"},
		{"43543.503206018519", "[$-7C5D]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maatsi 19 2019  12:04 PM Aippiq"},
		{"44562.189571759256", "[$-85D]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-85D]mmmm dd yyyy  h:mm AM/PM", "Jaannuari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-85D]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-85D]mmmmmm dd yyyy  h:mm AM/PM", "Jaannuari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-85D]mmm dd yyyy  h:mm AM/PM", "Mas 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-85D]mmmm dd yyyy  h:mm AM/PM aaa", "Maatsi 19 2019  12:04 PM Aip"},
		{"43543.503206018519", "[$-85D]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Aip"},
		{"43543.503206018519", "[$-85D]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maatsi 19 2019  12:04 PM Aippiq"},
		{"44562.189571759256", "[$-785D]mmm dd yyyy  h:mm AM/PM", "\u152E\u14D0\u14C4 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-785D]mmmm dd yyyy  h:mm AM/PM", "\u152E\u14D0\u14C4\u140A\u1546 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-785D]mmmmm dd yyyy  h:mm AM/PM", "\u152E 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-785D]mmmmmm dd yyyy  h:mm AM/PM", "\u152E\u14D0\u14C4\u140A\u1546 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-785D]mmm dd yyyy  h:mm AM/PM", "\u14AB\u1466\u14EF 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-785D]mmmm dd yyyy  h:mm AM/PM aaa", "\u14AB\u1466\u14EF 19 2019  12:04 PM \u140A\u1403\u1449\u1431"},
		{"43543.503206018519", "[$-785D]mmmmm dd yyyy  h:mm AM/PM ddd", "\u14AB 19 2019  12:04 PM \u140A\u1403\u1449\u1431"},
		{"43543.503206018519", "[$-785D]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u14AB\u1466\u14EF 19 2019  12:04 PM \u140A\u1403\u1449\u1431\u1585"},
		{"44562.189571759256", "[$-45D]mmm dd yyyy  h:mm AM/PM", "\u152E\u14D0\u14C4 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-45D]mmmm dd yyyy  h:mm AM/PM", "\u152E\u14D0\u14C4\u140A\u1546 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-45D]mmmmm dd yyyy  h:mm AM/PM", "\u152E 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-45D]mmmmmm dd yyyy  h:mm AM/PM", "\u152E\u14D0\u14C4\u140A\u1546 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-45D]mmm dd yyyy  h:mm AM/PM", "\u14AB\u1466\u14EF 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-45D]mmmm dd yyyy  h:mm AM/PM aaa", "\u14AB\u1466\u14EF 19 2019  12:04 PM \u140A\u1403\u1449\u1431"},
		{"43543.503206018519", "[$-45D]mmmmm dd yyyy  h:mm AM/PM ddd", "\u14AB 19 2019  12:04 PM \u140A\u1403\u1449\u1431"},
		{"43543.503206018519", "[$-45D]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u14AB\u1466\u14EF 19 2019  12:04 PM \u140A\u1403\u1449\u1431\u1585"},
		{"44562.189571759256", "[$-3C]mmm dd yyyy  h:mm AM/PM", "Ean 01 2022  4:32 r.n."},
		{"44593.189571759256", "[$-3C]mmm dd yyyy  h:mm AM/PM", "Feabh 01 2022  4:32 r.n."},
		{"44621.18957170139", "[$-3C]mmm dd yyyy  h:mm AM/PM", "Márta 01 2022  4:32 r.n."},
		{"44652.18957170139", "[$-3C]mmm dd yyyy  h:mm AM/PM", "Aib 01 2022  4:32 r.n."},
		{"44682.18957170139", "[$-3C]mmm dd yyyy  h:mm AM/PM", "Beal 01 2022  4:32 r.n."},
		{"44713.18957170139", "[$-3C]mmm dd yyyy  h:mm AM/PM", "Meith 01 2022  4:32 r.n."},
		{"44743.18957170139", "[$-3C]mmm dd yyyy  h:mm AM/PM", "Iúil 01 2022  4:32 r.n."},
		{"44774.18957170139", "[$-3C]mmm dd yyyy  h:mm AM/PM", "Lún 01 2022  4:32 r.n."},
		{"44805.18957170139", "[$-3C]mmm dd yyyy  h:mm AM/PM", "MFómh 01 2022  4:32 r.n."},
		{"44835.18957170139", "[$-3C]mmm dd yyyy  h:mm AM/PM", "DFómh 01 2022  4:32 r.n."},
		{"44866.18957170139", "[$-3C]mmm dd yyyy  h:mm AM/PM", "Samh 01 2022  4:32 r.n."},
		{"44896.18957170139", "[$-3C]mmm dd yyyy  h:mm AM/PM", "Noll 01 2022  4:32 r.n."},
		{"44562.189571759256", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Eanáir 01 2022  4:32 r.n."},
		{"44593.189571759256", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Feabhra 01 2022  4:32 r.n."},
		{"44621.18957170139", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Márta 01 2022  4:32 r.n."},
		{"44652.18957170139", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Aibreán 01 2022  4:32 r.n."},
		{"44682.18957170139", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Bealtaine 01 2022  4:32 r.n."},
		{"44713.18957170139", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Meitheamh 01 2022  4:32 r.n."},
		{"44743.18957170139", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Iúil 01 2022  4:32 r.n."},
		{"44774.18957170139", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Lúnasa 01 2022  4:32 r.n."},
		{"44805.18957170139", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Meán Fómhair 01 2022  4:32 r.n."},
		{"44835.18957170139", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Deireadh Fómhair 01 2022  4:32 r.n."},
		{"44866.18957170139", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Samhain 01 2022  4:32 r.n."},
		{"44896.18957170139", "[$-3C]mmmm dd yyyy  h:mm AM/PM", "Nollaig 01 2022  4:32 r.n."},
		{"44562.189571759256", "[$-3C]mmmmm dd yyyy  h:mm AM/PM", "E 01 2022  4:32 r.n."},
		{"44593.189571759256", "[$-3C]mmmmm dd yyyy  h:mm AM/PM", "F 01 2022  4:32 r.n."},
		{"44621.18957170139", "[$-3C]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 r.n."},
		{"44652.18957170139", "[$-3C]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 r.n."},
		{"44682.18957170139", "[$-3C]mmmmm dd yyyy  h:mm AM/PM", "B 01 2022  4:32 r.n."},
		{"44713.18957170139", "[$-3C]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 r.n."},
		{"44743.18957170139", "[$-3C]mmmmm dd yyyy  h:mm AM/PM", "I 01 2022  4:32 r.n."},
		{"44774.18957170139", "[$-3C]mmmmm dd yyyy  h:mm AM/PM", "L 01 2022  4:32 r.n."},
		{"44805.18957170139", "[$-3C]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 r.n."},
		{"44835.18957170139", "[$-3C]mmmmm dd yyyy  h:mm AM/PM aaa", "D 01 2022  4:32 r.n. Sath"},
		{"44866.18957170139", "[$-3C]mmmmm dd yyyy  h:mm AM/PM ddd", "S 01 2022  4:32 r.n. Máirt"},
		{"44896.18957170139", "[$-3C]mmmmm dd yyyy  h:mm AM/PM dddd", "N 01 2022  4:32 r.n. Déardaoin"},
		{"44562.189571759256", "[$-83C]mmm dd yyyy  h:mm AM/PM", "Ean 01 2022  4:32 r.n."},
		{"44593.189571759256", "[$-83C]mmm dd yyyy  h:mm AM/PM", "Feabh 01 2022  4:32 r.n."},
		{"44621.18957170139", "[$-83C]mmm dd yyyy  h:mm AM/PM", "Márta 01 2022  4:32 r.n."},
		{"44652.18957170139", "[$-83C]mmm dd yyyy  h:mm AM/PM", "Aib 01 2022  4:32 r.n."},
		{"44682.18957170139", "[$-83C]mmm dd yyyy  h:mm AM/PM", "Beal 01 2022  4:32 r.n."},
		{"44713.18957170139", "[$-83C]mmm dd yyyy  h:mm AM/PM", "Meith 01 2022  4:32 r.n."},
		{"44743.18957170139", "[$-83C]mmm dd yyyy  h:mm AM/PM", "Iúil 01 2022  4:32 r.n."},
		{"44774.18957170139", "[$-83C]mmm dd yyyy  h:mm AM/PM", "Lún 01 2022  4:32 r.n."},
		{"44805.18957170139", "[$-83C]mmm dd yyyy  h:mm AM/PM", "MFómh 01 2022  4:32 r.n."},
		{"44835.18957170139", "[$-83C]mmm dd yyyy  h:mm AM/PM", "DFómh 01 2022  4:32 r.n."},
		{"44866.18957170139", "[$-83C]mmm dd yyyy  h:mm AM/PM", "Samh 01 2022  4:32 r.n."},
		{"44896.18957170139", "[$-83C]mmm dd yyyy  h:mm AM/PM", "Noll 01 2022  4:32 r.n."},
		{"44562.189571759256", "[$-83C]mmmm dd yyyy  h:mm AM/PM", "Eanáir 01 2022  4:32 r.n."},
		{"44593.189571759256", "[$-83C]mmmm dd yyyy  h:mm AM/PM", "Feabhra 01 2022  4:32 r.n."},
		{"44621.18957170139", "[$-83C]mmmm dd yyyy  h:mm AM/PM", "Márta 01 2022  4:32 r.n."},
		{"44652.18957170139", "[$-83C]mmmm dd yyyy  h:mm AM/PM", "Aibreán 01 2022  4:32 r.n."},
		{"44682.18957170139", "[$-83C]mmmm dd yyyy  h:mm AM/PM", "Bealtaine 01 2022  4:32 r.n."},
		{"44713.18957170139", "[$-83C]mmmm dd yyyy  h:mm AM/PM", "Meitheamh 01 2022  4:32 r.n."},
		{"44743.18957170139", "[$-83C]mmmm dd yyyy  h:mm AM/PM", "Iúil 01 2022  4:32 r.n."},
		{"44774.18957170139", "[$-83C]mmmm dd yyyy  h:mm AM/PM", "Lúnasa 01 2022  4:32 r.n."},
		{"44805.18957170139", "[$-83C]mmmm dd yyyy  h:mm AM/PM", "Meán Fómhair 01 2022  4:32 r.n."},
		{"44835.18957170139", "[$-83C]mmmm dd yyyy  h:mm AM/PM aaa", "Deireadh Fómhair 01 2022  4:32 r.n. Sath"},
		{"44866.18957170139", "[$-83C]mmmm dd yyyy  h:mm AM/PM ddd", "Samhain 01 2022  4:32 r.n. Máirt"},
		{"44896.18957170139", "[$-83C]mmmm dd yyyy  h:mm AM/PM dddd", "Nollaig 01 2022  4:32 r.n. Déardaoin"},
		{"43543.503206018519", "[$-10]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 PM mar"},
		{"43543.503206018519", "[$-10]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 PM mar"},
		{"43543.503206018519", "[$-10]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM martedì"},
		{"43543.503206018519", "[$-410]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 PM mar"},
		{"43543.503206018519", "[$-410]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 PM mar"},
		{"43543.503206018519", "[$-410]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM martedì"},
		{"43543.503206018519", "[$-810]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 PM mar"},
		{"43543.503206018519", "[$-810]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 PM mar"},
		{"43543.503206018519", "[$-810]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM martedì"},
		{"43543.503206018519", "[$-11]mmm dd yyyy  h:mm AM/PM aaa", "3月 19 2019  12:04 午後 火"},
		{"43543.503206018519", "[$-11]mmmm dd yyyy  h:mm AM/PM ddd", "3月 19 2019  12:04 午後 火"},
		{"43543.503206018519", "[$-11]mmmmm dd yyyy  h:mm AM/PM dddd", "3 19 2019  12:04 午後 火曜日"},
		{"43543.503206018519", "[$-411]mmm dd yyyy  h:mm AM/PM aaa", "3月 19 2019  12:04 午後 火"},
		{"43543.503206018519", "[$-411]mmmm dd yyyy  h:mm AM/PM ddd", "3月 19 2019  12:04 午後 火"},
		{"43543.503206018519", "[$-411]mmmmm dd yyyy  h:mm AM/PM dddd", "3 19 2019  12:04 午後 火曜日"},
		{"44562.189571759256", "[$-4B]mmm dd yyyy  h:mm AM/PM", "\u0C9C\u0CA8\u0CB5\u0CB0\u0CBF 01 2022  4:32 \u0CAA\u0CC2\u0CB0\u0CCD\u0CB5\u0CBE\u0CB9\u0CCD\u0CA8"},
		{"44562.189571759256", "[$-4B]mmmm dd yyyy  h:mm AM/PM", "\u0C9C\u0CA8\u0CB5\u0CB0\u0CBF 01 2022  4:32 \u0CAA\u0CC2\u0CB0\u0CCD\u0CB5\u0CBE\u0CB9\u0CCD\u0CA8"},
		{"44562.189571759256", "[$-4B]mmmmm dd yyyy  h:mm AM/PM", "\u0C9C 01 2022  4:32 \u0CAA\u0CC2\u0CB0\u0CCD\u0CB5\u0CBE\u0CB9\u0CCD\u0CA8"},
		{"44562.189571759256", "[$-4B]mmmmmm dd yyyy  h:mm AM/PM", "\u0C9C\u0CA8\u0CB5\u0CB0\u0CBF 01 2022  4:32 \u0CAA\u0CC2\u0CB0\u0CCD\u0CB5\u0CBE\u0CB9\u0CCD\u0CA8"},
		{"43543.503206018519", "[$-4B]mmm dd yyyy  h:mm AM/PM", "\u0CAE\u0CBE\u0CB0\u0CCD\u0C9A\u0CCD 19 2019  12:04 \u0C85\u0CAA\u0CB0\u0CBE\u0CB9\u0CCD\u0CA8"},
		{"43543.503206018519", "[$-4B]mmmm dd yyyy  h:mm AM/PM aaa", "\u0CAE\u0CBE\u0CB0\u0CCD\u0C9A\u0CCD 19 2019  12:04 \u0C85\u0CAA\u0CB0\u0CBE\u0CB9\u0CCD\u0CA8 \u0CAE\u0C82\u0C97\u0CB3."},
		{"43543.503206018519", "[$-4B]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0CAE 19 2019  12:04 \u0C85\u0CAA\u0CB0\u0CBE\u0CB9\u0CCD\u0CA8 \u0CAE\u0C82\u0C97\u0CB3."},
		{"43543.503206018519", "[$-4B]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0CAE\u0CBE\u0CB0\u0CCD\u0C9A\u0CCD 19 2019  12:04 \u0C85\u0CAA\u0CB0\u0CBE\u0CB9\u0CCD\u0CA8 \u0CAE\u0C82\u0C97\u0CB3\u0CB5\u0CBE\u0CB0"},
		{"44562.189571759256", "[$-44B]mmm dd yyyy  h:mm AM/PM", "\u0C9C\u0CA8\u0CB5\u0CB0\u0CBF 01 2022  4:32 \u0CAA\u0CC2\u0CB0\u0CCD\u0CB5\u0CBE\u0CB9\u0CCD\u0CA8"},
		{"44562.189571759256", "[$-44B]mmmm dd yyyy  h:mm AM/PM", "\u0C9C\u0CA8\u0CB5\u0CB0\u0CBF 01 2022  4:32 \u0CAA\u0CC2\u0CB0\u0CCD\u0CB5\u0CBE\u0CB9\u0CCD\u0CA8"},
		{"44562.189571759256", "[$-44B]mmmmm dd yyyy  h:mm AM/PM", "\u0C9C 01 2022  4:32 \u0CAA\u0CC2\u0CB0\u0CCD\u0CB5\u0CBE\u0CB9\u0CCD\u0CA8"},
		{"44562.189571759256", "[$-44B]mmmmmm dd yyyy  h:mm AM/PM", "\u0C9C\u0CA8\u0CB5\u0CB0\u0CBF 01 2022  4:32 \u0CAA\u0CC2\u0CB0\u0CCD\u0CB5\u0CBE\u0CB9\u0CCD\u0CA8"},
		{"43543.503206018519", "[$-44B]mmm dd yyyy  h:mm AM/PM", "\u0CAE\u0CBE\u0CB0\u0CCD\u0C9A\u0CCD 19 2019  12:04 \u0C85\u0CAA\u0CB0\u0CBE\u0CB9\u0CCD\u0CA8"},
		{"43543.503206018519", "[$-44B]mmmm dd yyyy  h:mm AM/PM aaa", "\u0CAE\u0CBE\u0CB0\u0CCD\u0C9A\u0CCD 19 2019  12:04 \u0C85\u0CAA\u0CB0\u0CBE\u0CB9\u0CCD\u0CA8 \u0CAE\u0C82\u0C97\u0CB3."},
		{"43543.503206018519", "[$-44B]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0CAE 19 2019  12:04 \u0C85\u0CAA\u0CB0\u0CBE\u0CB9\u0CCD\u0CA8 \u0CAE\u0C82\u0C97\u0CB3."},
		{"43543.503206018519", "[$-44B]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0CAE\u0CBE\u0CB0\u0CCD\u0C9A\u0CCD 19 2019  12:04 \u0C85\u0CAA\u0CB0\u0CBE\u0CB9\u0CCD\u0CA8 \u0CAE\u0C82\u0C97\u0CB3\u0CB5\u0CBE\u0CB0"},
		{"44562.189571759256", "[$-471]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-471]mmmm dd yyyy  h:mm AM/PM", "January 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-471]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-471]mmmmmm dd yyyy  h:mm AM/PM", "January 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-471]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-471]mmmm dd yyyy  h:mm AM/PM aaa", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-471]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-471]mmmmmm dd yyyy  h:mm AM/PM dddd", "March 19 2019  12:04 PM Tuesday"},
		{"44562.189571759256", "[$-60]mmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0624\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-60]mmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0624\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-60]mmmmm dd yyyy  h:mm AM/PM", "\u062C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-60]mmmmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0624\u0631\u06CC 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-60]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0655\u0686 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-60]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0655\u0686 19 2019  12:04 PM \u0628\u06C6\u0645\u0648\u0627\u0631"},
		{"43543.503206018519", "[$-60]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 PM \u0628\u06C6\u0645\u0648\u0627\u0631"},
		{"43543.503206018519", "[$-60]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0655\u0686 19 2019  12:04 PM \u0628\u06C6\u0645\u0648\u0627\u0631"},
		{"44562.189571759256", "[$-460]mmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0624\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-460]mmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0624\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-460]mmmmm dd yyyy  h:mm AM/PM", "\u062C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-460]mmmmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0624\u0631\u06CC 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-460]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0655\u0686 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-460]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0655\u0686 19 2019  12:04 PM \u0628\u06C6\u0645\u0648\u0627\u0631"},
		{"43543.503206018519", "[$-460]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 PM \u0628\u06C6\u0645\u0648\u0627\u0631"},
		{"43543.503206018519", "[$-460]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0655\u0686 19 2019  12:04 PM \u0628\u06C6\u0645\u0648\u0627\u0631"},
		{"44562.189571759256", "[$-860]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-860]mmmm dd yyyy  h:mm AM/PM", "January 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-860]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-860]mmmmmm dd yyyy  h:mm AM/PM", "January 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-860]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-860]mmmm dd yyyy  h:mm AM/PM aaa", "March 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-860]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-860]mmmmmm dd yyyy  h:mm AM/PM dddd", "March 19 2019  12:04 PM Tuesday"},
		{"44562.189571759256", "[$-3F]mmm dd yyyy  h:mm AM/PM", "қаң 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-3F]mmmm dd yyyy  h:mm AM/PM", "Қаңтар 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-3F]mmmmm dd yyyy  h:mm AM/PM", "Қ 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-3F]mmmmmm dd yyyy  h:mm AM/PM", "Қаңтар 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-3F]mmm dd yyyy  h:mm AM/PM", "нау 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-3F]mmmm dd yyyy  h:mm AM/PM aaa", "Наурыз 19 2019  12:04 PM \u0441\u0435\u0439"},
		{"43543.503206018519", "[$-3F]mmmmm dd yyyy  h:mm AM/PM ddd", "Н 19 2019  12:04 PM \u0441\u0435\u0439"},
		{"43543.503206018519", "[$-3F]mmmmmm dd yyyy  h:mm AM/PM dddd", "Наурыз 19 2019  12:04 PM \u0441\u0435\u0439\u0441\u0435\u043D\u0431\u0456"},
		{"44562.189571759256", "[$-43F]mmm dd yyyy  h:mm AM/PM", "қаң 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-43F]mmmm dd yyyy  h:mm AM/PM", "Қаңтар 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-43F]mmmmm dd yyyy  h:mm AM/PM", "Қ 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-43F]mmmmmm dd yyyy  h:mm AM/PM", "Қаңтар 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-43F]mmm dd yyyy  h:mm AM/PM", "нау 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-43F]mmmm dd yyyy  h:mm AM/PM aaa", "Наурыз 19 2019  12:04 PM \u0441\u0435\u0439"},
		{"43543.503206018519", "[$-43F]mmmmm dd yyyy  h:mm AM/PM ddd", "Н 19 2019  12:04 PM \u0441\u0435\u0439"},
		{"43543.503206018519", "[$-43F]mmmmmm dd yyyy  h:mm AM/PM dddd", "Наурыз 19 2019  12:04 PM \u0441\u0435\u0439\u0441\u0435\u043D\u0431\u0456"},
		{"44562.189571759256", "[$-53]mmm dd yyyy  h:mm AM/PM", "\u17E1 01 2022  4:32 \u1796\u17D2\u179A\u17B9\u1780"},
		{"44562.189571759256", "[$-53]mmmm dd yyyy  h:mm AM/PM", "\u1798\u1780\u179A\u17B6 01 2022  4:32 \u1796\u17D2\u179A\u17B9\u1780"},
		{"44562.189571759256", "[$-53]mmmmm dd yyyy  h:mm AM/PM", "\u1798 01 2022  4:32 \u1796\u17D2\u179A\u17B9\u1780"},
		{"44562.189571759256", "[$-53]mmmmmm dd yyyy  h:mm AM/PM", "\u1798\u1780\u179A\u17B6 01 2022  4:32 \u1796\u17D2\u179A\u17B9\u1780"},
		{"43543.503206018519", "[$-53]mmm dd yyyy  h:mm AM/PM", "\u17E3 19 2019  12:04 \u179B\u17D2\u1784\u17B6\u1785"},
		{"43543.503206018519", "[$-53]mmmm dd yyyy  h:mm AM/PM aaa", "\u1798\u17B7\u1793\u17B6 19 2019  12:04 \u179B\u17D2\u1784\u17B6\u1785 \u17A2."},
		{"43543.503206018519", "[$-53]mmmmm dd yyyy  h:mm AM/PM ddd", "\u1798 19 2019  12:04 \u179B\u17D2\u1784\u17B6\u1785 \u17A2."},
		{"43543.503206018519", "[$-53]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u1798\u17B7\u1793\u17B6 19 2019  12:04 \u179B\u17D2\u1784\u17B6\u1785 \u1790\u17D2\u1784\u17C3\u17A2\u1784\u17D2\u1782\u17B6\u179A"},
		{"44562.189571759256", "[$-453]mmm dd yyyy  h:mm AM/PM", "\u17E1 01 2022  4:32 \u1796\u17D2\u179A\u17B9\u1780"},
		{"44562.189571759256", "[$-453]mmmm dd yyyy  h:mm AM/PM", "\u1798\u1780\u179A\u17B6 01 2022  4:32 \u1796\u17D2\u179A\u17B9\u1780"},
		{"44562.189571759256", "[$-453]mmmmm dd yyyy  h:mm AM/PM", "\u1798 01 2022  4:32 \u1796\u17D2\u179A\u17B9\u1780"},
		{"44562.189571759256", "[$-453]mmmmmm dd yyyy  h:mm AM/PM", "\u1798\u1780\u179A\u17B6 01 2022  4:32 \u1796\u17D2\u179A\u17B9\u1780"},
		{"43543.503206018519", "[$-453]mmm dd yyyy  h:mm AM/PM", "\u17E3 19 2019  12:04 \u179B\u17D2\u1784\u17B6\u1785"},
		{"43543.503206018519", "[$-453]mmmm dd yyyy  h:mm AM/PM aaa", "\u1798\u17B7\u1793\u17B6 19 2019  12:04 \u179B\u17D2\u1784\u17B6\u1785 \u17A2."},
		{"43543.503206018519", "[$-453]mmmmm dd yyyy  h:mm AM/PM ddd", "\u1798 19 2019  12:04 \u179B\u17D2\u1784\u17B6\u1785 \u17A2."},
		{"43543.503206018519", "[$-453]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u1798\u17B7\u1793\u17B6 19 2019  12:04 \u179B\u17D2\u1784\u17B6\u1785 \u1790\u17D2\u1784\u17C3\u17A2\u1784\u17D2\u1782\u17B6\u179A"},
		{"44562.189571759256", "[$-86]mmm dd yyyy  h:mm AM/PM", "nab'e 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-86]mmmm dd yyyy  h:mm AM/PM", "nab'e ik' 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-86]mmmmm dd yyyy  h:mm AM/PM", "n 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-86]mmmmmm dd yyyy  h:mm AM/PM", "nab'e ik' 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-86]mmm dd yyyy  h:mm AM/PM", "urox 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-86]mmmm dd yyyy  h:mm AM/PM aaa", "urox ik' 19 2019  12:04 p.m. oxq'"},
		{"43543.503206018519", "[$-86]mmmmm dd yyyy  h:mm AM/PM ddd", "u 19 2019  12:04 p.m. oxq'"},
		{"43543.503206018519", "[$-86]mmmmmm dd yyyy  h:mm AM/PM dddd", "urox ik' 19 2019  12:04 p.m. oxq'ij"},
		{"44562.189571759256", "[$-486]mmm dd yyyy  h:mm AM/PM", "nab'e 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-486]mmmm dd yyyy  h:mm AM/PM", "nab'e ik' 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-486]mmmmm dd yyyy  h:mm AM/PM", "n 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-486]mmmmmm dd yyyy  h:mm AM/PM", "nab'e ik' 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-486]mmm dd yyyy  h:mm AM/PM", "urox 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-486]mmmm dd yyyy  h:mm AM/PM aaa", "urox ik' 19 2019  12:04 p.m. oxq'"},
		{"43543.503206018519", "[$-486]mmmmm dd yyyy  h:mm AM/PM ddd", "u 19 2019  12:04 p.m. oxq'"},
		{"43543.503206018519", "[$-486]mmmmmm dd yyyy  h:mm AM/PM dddd", "urox ik' 19 2019  12:04 p.m. oxq'ij"},
		{"44562.189571759256", "[$-87]mmm dd yyyy  h:mm AM/PM", "mut. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-87]mmmm dd yyyy  h:mm AM/PM", "Mutarama 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-87]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-87]mmmmmm dd yyyy  h:mm AM/PM", "Mutarama 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-87]mmm dd yyyy  h:mm AM/PM", "wer. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-87]mmmm dd yyyy  h:mm AM/PM aaa", "Werurwe 19 2019  12:04 PM kab."},
		{"43543.503206018519", "[$-87]mmmmm dd yyyy  h:mm AM/PM ddd", "W 19 2019  12:04 PM kab."},
		{"43543.503206018519", "[$-87]mmmmmm dd yyyy  h:mm AM/PM dddd", "Werurwe 19 2019  12:04 PM Ku wa kabiri"},
		{"44562.189571759256", "[$-487]mmm dd yyyy  h:mm AM/PM", "mut. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-487]mmmm dd yyyy  h:mm AM/PM", "Mutarama 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-487]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-487]mmmmmm dd yyyy  h:mm AM/PM", "Mutarama 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-487]mmm dd yyyy  h:mm AM/PM", "wer. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-487]mmmm dd yyyy  h:mm AM/PM aaa", "Werurwe 19 2019  12:04 PM kab."},
		{"43543.503206018519", "[$-487]mmmmm dd yyyy  h:mm AM/PM ddd", "W 19 2019  12:04 PM kab."},
		{"43543.503206018519", "[$-487]mmmmmm dd yyyy  h:mm AM/PM dddd", "Werurwe 19 2019  12:04 PM Ku wa kabiri"},
		{"44562.189571759256", "[$-41]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41]mmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41]mmmmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-41]mmm dd yyyy  h:mm AM/PM", "Mac 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-41]mmmm dd yyyy  h:mm AM/PM aaa", "Machi 19 2019  12:04 PM Jnn"},
		{"43543.503206018519", "[$-41]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Jnn"},
		{"43543.503206018519", "[$-41]mmmmmm dd yyyy  h:mm AM/PM dddd", "Machi 19 2019  12:04 PM Jumanne"},
		{"44562.189571759256", "[$-441]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-441]mmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-441]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-441]mmmmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-441]mmm dd yyyy  h:mm AM/PM", "Mac 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-441]mmmm dd yyyy  h:mm AM/PM aaa", "Machi 19 2019  12:04 PM Jnn"},
		{"43543.503206018519", "[$-441]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Jnn"},
		{"43543.503206018519", "[$-441]mmmmmm dd yyyy  h:mm AM/PM dddd", "Machi 19 2019  12:04 PM Jumanne"},
		{"44562.189571759256", "[$-57]mmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-57]mmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-57]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-57]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940 01 2022  4:32 \u092E.\u092A\u0942."},
		{"43543.503206018519", "[$-57]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902."},
		{"43543.503206018519", "[$-57]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933."},
		{"43543.503206018519", "[$-57]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933."},
		{"43543.503206018519", "[$-57]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933\u093E\u0930"},
		{"44562.189571759256", "[$-457]mmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-457]mmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-457]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-457]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940 01 2022  4:32 \u092E.\u092A\u0942."},
		{"43543.503206018519", "[$-457]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902."},
		{"43543.503206018519", "[$-457]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933."},
		{"43543.503206018519", "[$-457]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933."},
		{"43543.503206018519", "[$-457]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933\u093E\u0930"},
		{"43543.503206018519", "[$-12]mmm dd yyyy  h:mm AM/PM aaa", "3 19 2019  12:04 오후 화"},
		{"43543.503206018519", "[$-12]mmmm dd yyyy  h:mm AM/PM ddd", "3월 19 2019  12:04 오후 화"},
		{"43543.503206018519", "[$-12]mmmmm dd yyyy  h:mm AM/PM dddd", "3 19 2019  12:04 오후 화요일"},
		{"43543.503206018519", "[$-412]mmm dd yyyy  h:mm AM/PM aaa", "3 19 2019  12:04 오후 화"},
		{"43543.503206018519", "[$-412]mmmm dd yyyy  h:mm AM/PM ddd", "3월 19 2019  12:04 오후 화"},
		{"43543.503206018519", "[$-412]mmmmm dd yyyy  h:mm AM/PM dddd", "3 19 2019  12:04 오후 화요일"},
		{"44562.189571759256", "[$-40]mmm dd yyyy  h:mm AM/PM", "\u042F\u043D\u0432 01 2022  4:32 \u0442\u04A3"},
		{"44562.189571759256", "[$-40]mmmm dd yyyy  h:mm AM/PM", "\u042F\u043D\u0432\u0430\u0440\u044C 01 2022  4:32 \u0442\u04A3"},
		{"44562.189571759256", "[$-40]mmmmm dd yyyy  h:mm AM/PM", "\u042F 01 2022  4:32 \u0442\u04A3"},
		{"44562.189571759256", "[$-40]mmmmmm dd yyyy  h:mm AM/PM", "\u042F\u043D\u0432\u0430\u0440\u044C 01 2022  4:32 \u0442\u04A3"},
		{"43543.503206018519", "[$-40]mmm dd yyyy  h:mm AM/PM", "\u041C\u0430\u0440 19 2019  12:04 \u0442\u043A"},
		{"43543.503206018519", "[$-40]mmmm dd yyyy  h:mm AM/PM aaa", "\u041C\u0430\u0440\u0442 19 2019  12:04 \u0442\u043A \u0448\u0435\u0439\u0448."},
		{"43543.503206018519", "[$-40]mmmmm dd yyyy  h:mm AM/PM ddd", "\u041C 19 2019  12:04 \u0442\u043A \u0448\u0435\u0439\u0448."},
		{"43543.503206018519", "[$-40]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u041C\u0430\u0440\u0442 19 2019  12:04 \u0442\u043A \u0448\u0435\u0439\u0448\u0435\u043C\u0431\u0438"},
		{"44562.189571759256", "[$-440]mmm dd yyyy  h:mm AM/PM", "\u042F\u043D\u0432 01 2022  4:32 \u0442\u04A3"},
		{"44562.189571759256", "[$-440]mmmm dd yyyy  h:mm AM/PM", "\u042F\u043D\u0432\u0430\u0440\u044C 01 2022  4:32 \u0442\u04A3"},
		{"44562.189571759256", "[$-440]mmmmm dd yyyy  h:mm AM/PM", "\u042F 01 2022  4:32 \u0442\u04A3"},
		{"44562.189571759256", "[$-440]mmmmmm dd yyyy  h:mm AM/PM", "\u042F\u043D\u0432\u0430\u0440\u044C 01 2022  4:32 \u0442\u04A3"},
		{"43543.503206018519", "[$-440]mmm dd yyyy  h:mm AM/PM", "\u041C\u0430\u0440 19 2019  12:04 \u0442\u043A"},
		{"43543.503206018519", "[$-440]mmmm dd yyyy  h:mm AM/PM aaa", "\u041C\u0430\u0440\u0442 19 2019  12:04 \u0442\u043A \u0448\u0435\u0439\u0448."},
		{"43543.503206018519", "[$-440]mmmmm dd yyyy  h:mm AM/PM ddd", "\u041C 19 2019  12:04 \u0442\u043A \u0448\u0435\u0439\u0448."},
		{"43543.503206018519", "[$-440]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u041C\u0430\u0440\u0442 19 2019  12:04 \u0442\u043A \u0448\u0435\u0439\u0448\u0435\u043C\u0431\u0438"},
		{"44562.189571759256", "[$-54]mmm dd yyyy  h:mm AM/PM", "\u0EA1.\u0E81. 01 2022  4:32 \u0E81\u0EC8\u0EAD\u0E99\u0E97\u0EC8\u0EBD\u0E87"},
		{"44562.189571759256", "[$-54]mmmm dd yyyy  h:mm AM/PM", "\u0EA1\u0EB1\u0E87\u0E81\u0EAD\u0E99 01 2022  4:32 \u0E81\u0EC8\u0EAD\u0E99\u0E97\u0EC8\u0EBD\u0E87"},
		{"44562.189571759256", "[$-54]mmmmm dd yyyy  h:mm AM/PM", "\u0EA1 01 2022  4:32 \u0E81\u0EC8\u0EAD\u0E99\u0E97\u0EC8\u0EBD\u0E87"},
		{"44562.189571759256", "[$-54]mmmmmm dd yyyy  h:mm AM/PM", "\u0EA1\u0EB1\u0E87\u0E81\u0EAD\u0E99 01 2022  4:32 \u0E81\u0EC8\u0EAD\u0E99\u0E97\u0EC8\u0EBD\u0E87"},
		{"43543.503206018519", "[$-54]mmm dd yyyy  h:mm AM/PM", "\u0EA1.\u0E99. 19 2019  12:04 \u0EAB\u0EBC\u0EB1\u0E87\u0E97\u0EC8\u0EBD\u0E87"},
		{"43543.503206018519", "[$-54]mmmm dd yyyy  h:mm AM/PM aaa", "\u0EA1\u0EB5\u0E99\u0EB2 19 2019  12:04 \u0EAB\u0EBC\u0EB1\u0E87\u0E97\u0EC8\u0EBD\u0E87 \u0EAD\u0EB1\u0E87\u0E84\u0EB2\u0E99"},
		{"43543.503206018519", "[$-54]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0EA1 19 2019  12:04 \u0EAB\u0EBC\u0EB1\u0E87\u0E97\u0EC8\u0EBD\u0E87 \u0EAD\u0EB1\u0E87\u0E84\u0EB2\u0E99"},
		{"43543.503206018519", "[$-54]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0EA1\u0EB5\u0E99\u0EB2 19 2019  12:04 \u0EAB\u0EBC\u0EB1\u0E87\u0E97\u0EC8\u0EBD\u0E87 \u0EA7\u0EB1\u0E99\u0EAD\u0EB1\u0E87\u0E84\u0EB2\u0E99"},
		{"44562.189571759256", "[$-454]mmm dd yyyy  h:mm AM/PM", "\u0EA1.\u0E81. 01 2022  4:32 \u0E81\u0EC8\u0EAD\u0E99\u0E97\u0EC8\u0EBD\u0E87"},
		{"44562.189571759256", "[$-454]mmmm dd yyyy  h:mm AM/PM", "\u0EA1\u0EB1\u0E87\u0E81\u0EAD\u0E99 01 2022  4:32 \u0E81\u0EC8\u0EAD\u0E99\u0E97\u0EC8\u0EBD\u0E87"},
		{"44562.189571759256", "[$-454]mmmmm dd yyyy  h:mm AM/PM", "\u0EA1 01 2022  4:32 \u0E81\u0EC8\u0EAD\u0E99\u0E97\u0EC8\u0EBD\u0E87"},
		{"44562.189571759256", "[$-454]mmmmmm dd yyyy  h:mm AM/PM", "\u0EA1\u0EB1\u0E87\u0E81\u0EAD\u0E99 01 2022  4:32 \u0E81\u0EC8\u0EAD\u0E99\u0E97\u0EC8\u0EBD\u0E87"},
		{"43543.503206018519", "[$-454]mmm dd yyyy  h:mm AM/PM", "\u0EA1.\u0E99. 19 2019  12:04 \u0EAB\u0EBC\u0EB1\u0E87\u0E97\u0EC8\u0EBD\u0E87"},
		{"43543.503206018519", "[$-454]mmmm dd yyyy  h:mm AM/PM aaa", "\u0EA1\u0EB5\u0E99\u0EB2 19 2019  12:04 \u0EAB\u0EBC\u0EB1\u0E87\u0E97\u0EC8\u0EBD\u0E87 \u0EAD\u0EB1\u0E87\u0E84\u0EB2\u0E99"},
		{"43543.503206018519", "[$-454]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0EA1 19 2019  12:04 \u0EAB\u0EBC\u0EB1\u0E87\u0E97\u0EC8\u0EBD\u0E87 \u0EAD\u0EB1\u0E87\u0E84\u0EB2\u0E99"},
		{"43543.503206018519", "[$-454]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0EA1\u0EB5\u0E99\u0EB2 19 2019  12:04 \u0EAB\u0EBC\u0EB1\u0E87\u0E97\u0EC8\u0EBD\u0E87 \u0EA7\u0EB1\u0E99\u0EAD\u0EB1\u0E87\u0E84\u0EB2\u0E99"},
		{"44562.189571759256", "[$-476]mmm dd yyyy  h:mm AM/PM", "Ian 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-476]mmmm dd yyyy  h:mm AM/PM", "Ianuarius 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-476]mmmmm dd yyyy  h:mm AM/PM", "I 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-476]mmmmmm dd yyyy  h:mm AM/PM", "Ianuarius 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-476]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-476]mmmm dd yyyy  h:mm AM/PM aaa", "Martius 19 2019  12:04 PM Mar"},
		{"43543.503206018519", "[$-476]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Mar"},
		{"43543.503206018519", "[$-476]mmmmmm dd yyyy  h:mm AM/PM dddd", "Martius 19 2019  12:04 PM Martis"},
		{"44562.189571759256", "[$-26]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 priekšp."},
		{"44562.189571759256", "[$-26]mmmm dd yyyy  h:mm AM/PM", "janvāris 01 2022  4:32 priekšp."},
		{"44562.189571759256", "[$-26]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 priekšp."},
		{"44562.189571759256", "[$-26]mmmmmm dd yyyy  h:mm AM/PM", "janvāris 01 2022  4:32 priekšp."},
		{"43543.503206018519", "[$-26]mmm dd yyyy  h:mm AM/PM", "marts 19 2019  12:04 pēcp."},
		{"43543.503206018519", "[$-26]mmmm dd yyyy  h:mm AM/PM aaa", "marts 19 2019  12:04 pēcp. otrd."},
		{"43543.503206018519", "[$-26]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 pēcp. otrd."},
		{"43543.503206018519", "[$-26]mmmmmm dd yyyy  h:mm AM/PM dddd", "marts 19 2019  12:04 pēcp. otrdiena"},
		{"44562.189571759256", "[$-426]mmm dd yyyy  h:mm AM/PM", "janv. 01 2022  4:32 priekšp."},
		{"44562.189571759256", "[$-426]mmmm dd yyyy  h:mm AM/PM", "janvāris 01 2022  4:32 priekšp."},
		{"44562.189571759256", "[$-426]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 priekšp."},
		{"44562.189571759256", "[$-426]mmmmmm dd yyyy  h:mm AM/PM", "janvāris 01 2022  4:32 priekšp."},
		{"43543.503206018519", "[$-426]mmm dd yyyy  h:mm AM/PM", "marts 19 2019  12:04 pēcp."},
		{"43543.503206018519", "[$-426]mmmm dd yyyy  h:mm AM/PM aaa", "marts 19 2019  12:04 pēcp. otrd."},
		{"43543.503206018519", "[$-426]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 pēcp. otrd."},
		{"43543.503206018519", "[$-426]mmmmmm dd yyyy  h:mm AM/PM dddd", "marts 19 2019  12:04 pēcp. otrdiena"},
		{"44562.189571759256", "[$-27]mmm dd yyyy  h:mm AM/PM", "saus. 01 2022  4:32 priešpiet"},
		{"44562.189571759256", "[$-27]mmmm dd yyyy  h:mm AM/PM", "sausis 01 2022  4:32 priešpiet"},
		{"44562.189571759256", "[$-27]mmmmm dd yyyy  h:mm AM/PM", "s 01 2022  4:32 priešpiet"},
		{"44562.189571759256", "[$-27]mmmmmm dd yyyy  h:mm AM/PM", "sausis 01 2022  4:32 priešpiet"},
		{"43543.503206018519", "[$-27]mmm dd yyyy  h:mm AM/PM", "kov. 19 2019  12:04 popiet"},
		{"43543.503206018519", "[$-27]mmmm dd yyyy  h:mm AM/PM aaa", "kovas 19 2019  12:04 popiet an"},
		{"43543.503206018519", "[$-27]mmmmm dd yyyy  h:mm AM/PM ddd", "k 19 2019  12:04 popiet an"},
		{"43543.503206018519", "[$-27]mmmmmm dd yyyy  h:mm AM/PM dddd", "kovas 19 2019  12:04 popiet antradienis"},
		{"44562.189571759256", "[$-427]mmm dd yyyy  h:mm AM/PM", "saus. 01 2022  4:32 priešpiet"},
		{"44562.189571759256", "[$-427]mmmm dd yyyy  h:mm AM/PM", "sausis 01 2022  4:32 priešpiet"},
		{"44562.189571759256", "[$-427]mmmmm dd yyyy  h:mm AM/PM", "s 01 2022  4:32 priešpiet"},
		{"44562.189571759256", "[$-427]mmmmmm dd yyyy  h:mm AM/PM", "sausis 01 2022  4:32 priešpiet"},
		{"43543.503206018519", "[$-427]mmm dd yyyy  h:mm AM/PM", "kov. 19 2019  12:04 popiet"},
		{"43543.503206018519", "[$-427]mmmm dd yyyy  h:mm AM/PM aaa", "kovas 19 2019  12:04 popiet an"},
		{"43543.503206018519", "[$-427]mmmmm dd yyyy  h:mm AM/PM ddd", "k 19 2019  12:04 popiet an"},
		{"43543.503206018519", "[$-427]mmmmmm dd yyyy  h:mm AM/PM dddd", "kovas 19 2019  12:04 popiet antradienis"},
		{"44562.189571759256", "[$-7C2E]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C2E]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C2E]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C2E]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C2E]mmm dd yyyy  h:mm AM/PM", "měr 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C2E]mmmm dd yyyy  h:mm AM/PM aaa", "měrc 19 2019  12:04 PM wa\u0142"},
		{"43543.503206018519", "[$-7C2E]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM wa\u0142"},
		{"43543.503206018519", "[$-7C2E]mmmmmm dd yyyy  h:mm AM/PM dddd", "měrc 19 2019  12:04 PM wa\u0142tora"},
		{"44562.189571759256", "[$-82E]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-82E]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-82E]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-82E]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-82E]mmm dd yyyy  h:mm AM/PM", "měr 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-82E]mmmm dd yyyy  h:mm AM/PM aaa", "měrc 19 2019  12:04 PM wa\u0142"},
		{"43543.503206018519", "[$-82E]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM wa\u0142"},
		{"43543.503206018519", "[$-82E]mmmmmm dd yyyy  h:mm AM/PM dddd", "měrc 19 2019  12:04 PM wa\u0142tora"},
		{"44562.189571759256", "[$-6E]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6E]mmmm dd yyyy  h:mm AM/PM", "Januar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6E]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6E]mmmmmm dd yyyy  h:mm AM/PM", "Januar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-6E]mmm dd yyyy  h:mm AM/PM", "Mäe 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-6E]mmmm dd yyyy  h:mm AM/PM aaa", "Mäerz 19 2019  12:04 PM Dën"},
		{"43543.503206018519", "[$-6E]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Dën"},
		{"43543.503206018519", "[$-6E]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mäerz 19 2019  12:04 PM Dënschdeg"},
		{"44562.189571759256", "[$-46E]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46E]mmmm dd yyyy  h:mm AM/PM", "Januar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46E]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46E]mmmmmm dd yyyy  h:mm AM/PM", "Januar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-46E]mmm dd yyyy  h:mm AM/PM", "Mäe 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-46E]mmmm dd yyyy  h:mm AM/PM aaa", "Mäerz 19 2019  12:04 PM Dën"},
		{"43543.503206018519", "[$-46E]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Dën"},
		{"43543.503206018519", "[$-46E]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mäerz 19 2019  12:04 PM Dënschdeg"},
		{"44562.189571759256", "[$-2F]mmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D. 01 2022  4:32 \u043F\u0440\u0435\u0442\u043F\u043B."},
		{"44562.189571759256", "[$-2F]mmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440\u0438 01 2022  4:32 \u043F\u0440\u0435\u0442\u043F\u043B."},
		{"44562.189571759256", "[$-2F]mmmmm dd yyyy  h:mm AM/PM", "\u0458 01 2022  4:32 \u043F\u0440\u0435\u0442\u043F\u043B."},
		{"44562.189571759256", "[$-2F]mmmmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440\u0438 01 2022  4:32 \u043F\u0440\u0435\u0442\u043F\u043B."},
		{"43543.503206018519", "[$-2F]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440. 19 2019  12:04 \u043F\u043E\u043F\u043B."},
		{"43543.503206018519", "[$-2F]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 \u043F\u043E\u043F\u043B. \u0432\u0442."},
		{"43543.503206018519", "[$-2F]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 \u043F\u043E\u043F\u043B. \u0432\u0442."},
		{"43543.503206018519", "[$-2F]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 \u043F\u043E\u043F\u043B. \u0432\u0442\u043E\u0440\u043D\u0438\u043A"},
		{"44562.189571759256", "[$-42F]mmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D. 01 2022  4:32 \u043F\u0440\u0435\u0442\u043F\u043B."},
		{"44562.189571759256", "[$-42F]mmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440\u0438 01 2022  4:32 \u043F\u0440\u0435\u0442\u043F\u043B."},
		{"44562.189571759256", "[$-42F]mmmmm dd yyyy  h:mm AM/PM", "\u0458 01 2022  4:32 \u043F\u0440\u0435\u0442\u043F\u043B."},
		{"44562.189571759256", "[$-42F]mmmmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440\u0438 01 2022  4:32 \u043F\u0440\u0435\u0442\u043F\u043B."},
		{"43543.503206018519", "[$-42F]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440. 19 2019  12:04 \u043F\u043E\u043F\u043B."},
		{"43543.503206018519", "[$-42F]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 \u043F\u043E\u043F\u043B. \u0432\u0442."},
		{"43543.503206018519", "[$-42F]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 \u043F\u043E\u043F\u043B. \u0432\u0442."},
		{"43543.503206018519", "[$-42F]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 \u043F\u043E\u043F\u043B. \u0432\u0442\u043E\u0440\u043D\u0438\u043A"},
		{"44562.189571759256", "[$-3E]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 PG"},
		{"44562.189571759256", "[$-3E]mmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 PG"},
		{"44562.189571759256", "[$-3E]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 PG"},
		{"44562.189571759256", "[$-3E]mmmmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 PG"},
		{"43543.503206018519", "[$-3E]mmm dd yyyy  h:mm AM/PM", "Mac 19 2019  12:04 PTG"},
		{"43543.503206018519", "[$-3E]mmmm dd yyyy  h:mm AM/PM aaa", "Mac 19 2019  12:04 PTG Sel"},
		{"43543.503206018519", "[$-3E]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PTG Sel"},
		{"43543.503206018519", "[$-3E]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mac 19 2019  12:04 PTG Selasa"},
		{"44562.189571759256", "[$-83E]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 PG"},
		{"44562.189571759256", "[$-83E]mmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 PG"},
		{"44562.189571759256", "[$-83E]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 PG"},
		{"44562.189571759256", "[$-83E]mmmmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 PG"},
		{"43543.503206018519", "[$-83E]mmm dd yyyy  h:mm AM/PM", "Mac 19 2019  12:04 PTG"},
		{"43543.503206018519", "[$-83E]mmmm dd yyyy  h:mm AM/PM aaa", "Mac 19 2019  12:04 PTG Sel"},
		{"43543.503206018519", "[$-83E]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PTG Sel"},
		{"43543.503206018519", "[$-83E]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mac 19 2019  12:04 PTG Selasa"},
		{"44562.189571759256", "[$-43E]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 PG"},
		{"44562.189571759256", "[$-43E]mmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 PG"},
		{"44562.189571759256", "[$-43E]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 PG"},
		{"44562.189571759256", "[$-43E]mmmmmm dd yyyy  h:mm AM/PM", "Januari 01 2022  4:32 PG"},
		{"43543.503206018519", "[$-43E]mmm dd yyyy  h:mm AM/PM", "Mac 19 2019  12:04 PTG"},
		{"43543.503206018519", "[$-43E]mmmm dd yyyy  h:mm AM/PM aaa", "Mac 19 2019  12:04 PTG Sel"},
		{"43543.503206018519", "[$-43E]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PTG Sel"},
		{"43543.503206018519", "[$-43E]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mac 19 2019  12:04 PTG Selasa"},
		{"44562.189571759256", "[$-4C]mmm dd yyyy  h:mm AM/PM", "\u0D1C\u0D28\u0D41 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-4C]mmmm dd yyyy  h:mm AM/PM", "\u0D1C\u0D28\u0D41\u0D35\u0D30\u0D3F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-4C]mmmmm dd yyyy  h:mm AM/PM", "\u0D1C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-4C]mmmmmm dd yyyy  h:mm AM/PM", "\u0D1C\u0D28\u0D41\u0D35\u0D30\u0D3F 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-4C]mmm dd yyyy  h:mm AM/PM", "\u0D2E\u0D3E\u0D7C 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-4C]mmmm dd yyyy  h:mm AM/PM aaa", "\u0D2E\u0D3E\u0D30\u0D4D\u200D\u200C\u0D1A\u0D4D\u0D1A\u0D4D 19 2019  12:04 PM \u0D1A\u0D4A\u0D35\u0D4D\u0D35"},
		{"43543.503206018519", "[$-4C]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0D2E 19 2019  12:04 PM \u0D1A\u0D4A\u0D35\u0D4D\u0D35"},
		{"43543.503206018519", "[$-4C]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0D2E\u0D3E\u0D30\u0D4D\u200D\u200C\u0D1A\u0D4D\u0D1A\u0D4D 19 2019  12:04 PM \u0D1A\u0D4A\u0D35\u0D4D\u0D35\u0D3E\u0D34\u0D4D\u0D1A"},
		{"44562.189571759256", "[$-44C]mmm dd yyyy  h:mm AM/PM", "\u0D1C\u0D28\u0D41 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-44C]mmmm dd yyyy  h:mm AM/PM", "\u0D1C\u0D28\u0D41\u0D35\u0D30\u0D3F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-44C]mmmmm dd yyyy  h:mm AM/PM", "\u0D1C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-44C]mmmmmm dd yyyy  h:mm AM/PM", "\u0D1C\u0D28\u0D41\u0D35\u0D30\u0D3F 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-44C]mmm dd yyyy  h:mm AM/PM", "\u0D2E\u0D3E\u0D7C 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-44C]mmmm dd yyyy  h:mm AM/PM aaa", "\u0D2E\u0D3E\u0D30\u0D4D\u200D\u200C\u0D1A\u0D4D\u0D1A\u0D4D 19 2019  12:04 PM \u0D1A\u0D4A\u0D35\u0D4D\u0D35"},
		{"43543.503206018519", "[$-44C]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0D2E 19 2019  12:04 PM \u0D1A\u0D4A\u0D35\u0D4D\u0D35"},
		{"43543.503206018519", "[$-44C]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0D2E\u0D3E\u0D30\u0D4D\u200D\u200C\u0D1A\u0D4D\u0D1A\u0D4D 19 2019  12:04 PM \u0D1A\u0D4A\u0D35\u0D4D\u0D35\u0D3E\u0D34\u0D4D\u0D1A"},
		{"44562.189571759256", "[$-3A]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-3A]mmmm dd yyyy  h:mm AM/PM", "Jannar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-3A]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-3A]mmmmmm dd yyyy  h:mm AM/PM", "Jannar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-3A]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-3A]mmmm dd yyyy  h:mm AM/PM aaa", "Marzu 19 2019  12:04 PM Tli"},
		{"43543.503206018519", "[$-3A]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Tli"},
		{"43543.503206018519", "[$-3A]mmmmmm dd yyyy  h:mm AM/PM dddd", "Marzu 19 2019  12:04 PM It-Tlieta"},
		{"44562.189571759256", "[$-43A]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-43A]mmmm dd yyyy  h:mm AM/PM", "Jannar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-43A]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-43A]mmmmmm dd yyyy  h:mm AM/PM", "Jannar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-43A]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-43A]mmmm dd yyyy  h:mm AM/PM aaa", "Marzu 19 2019  12:04 PM Tli"},
		{"43543.503206018519", "[$-43A]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Tli"},
		{"43543.503206018519", "[$-43A]mmmmmm dd yyyy  h:mm AM/PM dddd", "Marzu 19 2019  12:04 PM It-Tlieta"},
		{"44562.189571759256", "[$-81]mmm dd yyyy  h:mm AM/PM", "Kohi 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-81]mmmm dd yyyy  h:mm AM/PM", "Kohitātea 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-81]mmmmm dd yyyy  h:mm AM/PM", "K 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-81]mmmmmm dd yyyy  h:mm AM/PM", "Kohitātea 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-81]mmm dd yyyy  h:mm AM/PM", "Pou 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-81]mmmm dd yyyy  h:mm AM/PM aaa", "Poutūterangi 19 2019  12:04 p.m. Tū"},
		{"43543.503206018519", "[$-81]mmmmm dd yyyy  h:mm AM/PM ddd", "P 19 2019  12:04 p.m. Tū"},
		{"43543.503206018519", "[$-81]mmmmmm dd yyyy  h:mm AM/PM dddd", "Poutūterangi 19 2019  12:04 p.m. Rātū"},
		{"44562.189571759256", "[$-481]mmm dd yyyy  h:mm AM/PM", "Kohi 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-481]mmmm dd yyyy  h:mm AM/PM", "Kohitātea 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-481]mmmmm dd yyyy  h:mm AM/PM", "K 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-481]mmmmmm dd yyyy  h:mm AM/PM", "Kohitātea 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-481]mmm dd yyyy  h:mm AM/PM", "Pou 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-481]mmmm dd yyyy  h:mm AM/PM aaa", "Poutūterangi 19 2019  12:04 p.m. Tū"},
		{"43543.503206018519", "[$-481]mmmmm dd yyyy  h:mm AM/PM ddd", "P 19 2019  12:04 p.m. Tū"},
		{"43543.503206018519", "[$-481]mmmmmm dd yyyy  h:mm AM/PM dddd", "Poutūterangi 19 2019  12:04 p.m. Rātū"},
		{"44562.189571759256", "[$-7A]mmm dd yyyy  h:mm AM/PM", "Kiñe Tripantu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7A]mmmm dd yyyy  h:mm AM/PM", "Kiñe Tripantu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7A]mmmmm dd yyyy  h:mm AM/PM", "K 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7A]mmmmmm dd yyyy  h:mm AM/PM", "Kiñe Tripantu 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7A]mmm dd yyyy  h:mm AM/PM", "Kila 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7A]mmmm dd yyyy  h:mm AM/PM aaa", "Kila 19 2019  12:04 PM Kila"},
		{"43543.503206018519", "[$-7A]mmmmm dd yyyy  h:mm AM/PM ddd", "K 19 2019  12:04 PM Kila"},
		{"43543.503206018519", "[$-7A]mmmmmm dd yyyy  h:mm AM/PM dddd", "Kila 19 2019  12:04 PM Kila Ante"},
		{"44562.189571759256", "[$-47A]mmm dd yyyy  h:mm AM/PM", "Kiñe Tripantu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-47A]mmmm dd yyyy  h:mm AM/PM", "Kiñe Tripantu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-47A]mmmmm dd yyyy  h:mm AM/PM", "K 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-47A]mmmmmm dd yyyy  h:mm AM/PM", "Kiñe Tripantu 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-47A]mmm dd yyyy  h:mm AM/PM", "Kila 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-47A]mmmm dd yyyy  h:mm AM/PM aaa", "Kila 19 2019  12:04 PM Kila"},
		{"43543.503206018519", "[$-47A]mmmmm dd yyyy  h:mm AM/PM ddd", "K 19 2019  12:04 PM Kila"},
		{"43543.503206018519", "[$-47A]mmmmmm dd yyyy  h:mm AM/PM dddd", "Kila 19 2019  12:04 PM Kila Ante"},
		{"44562.189571759256", "[$-4E]mmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947. 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-4E]mmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-4E]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-4E]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940 01 2022  4:32 \u092E.\u092A\u0942."},
		{"43543.503206018519", "[$-4E]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902."},
		{"43543.503206018519", "[$-4E]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933."},
		{"43543.503206018519", "[$-4E]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933."},
		{"43543.503206018519", "[$-4E]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933\u0935\u093E\u0930"},
		{"44562.189571759256", "[$-44E]mmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947. 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-44E]mmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-44E]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092E.\u092A\u0942."},
		{"44562.189571759256", "[$-44E]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940 01 2022  4:32 \u092E.\u092A\u0942."},
		{"43543.503206018519", "[$-44E]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902."},
		{"43543.503206018519", "[$-44E]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933."},
		{"43543.503206018519", "[$-44E]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933."},
		{"43543.503206018519", "[$-44E]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902. \u092E\u0902\u0917\u0933\u0935\u093E\u0930"},
		{"43543.503206018519", "[$-44E]mmmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902."},
		{"43543.503206018519", "[$-44E]mmmmm dd yyyy  h:mm AM/PM", "\u092E 19 2019  12:04 \u092E.\u0928\u0902."},
		{"43543.503206018519", "[$-44E]mmmmmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E.\u0928\u0902."},
		{"44562.189571759256", "[$-7C]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C]mmmm dd yyyy  h:mm AM/PM", "Tsothohrkó:Wa 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C]mmmmm dd yyyy  h:mm AM/PM", "T 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C]mmmmmm dd yyyy  h:mm AM/PM", "Tsothohrkó:Wa 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C]mmmm dd yyyy  h:mm AM/PM aaa", "Enniskó:Wa 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-7C]mmmmm dd yyyy  h:mm AM/PM ddd", "E 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-7C]mmmmmm dd yyyy  h:mm AM/PM dddd", "Enniskó:Wa 19 2019  12:04 PM Ratironhia'kehronòn:ke"},
		{"44562.189571759256", "[$-47C]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-47C]mmmm dd yyyy  h:mm AM/PM", "Tsothohrkó:Wa 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-47C]mmmmm dd yyyy  h:mm AM/PM", "T 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-47C]mmmmmm dd yyyy  h:mm AM/PM", "Tsothohrkó:Wa 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-47C]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-47C]mmmm dd yyyy  h:mm AM/PM aaa", "Enniskó:Wa 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-47C]mmmmm dd yyyy  h:mm AM/PM ddd", "E 19 2019  12:04 PM Tue"},
		{"43543.503206018519", "[$-47C]mmmmmm dd yyyy  h:mm AM/PM dddd", "Enniskó:Wa 19 2019  12:04 PM Ratironhia'kehronòn:ke"},
		{"44562.189571759256", "[$-50]mmm dd yyyy  h:mm AM/PM", "1-р сар 01 2022  4:32 \u04AF.\u04E9."},
		{"44562.189571759256", "[$-50]mmmm dd yyyy  h:mm AM/PM", "\u041D\u044D\u0433\u0434\u04AF\u0433\u044D\u044D\u0440 \u0441\u0430\u0440 01 2022  4:32 \u04AF.\u04E9."},
		{"44562.189571759256", "[$-50]mmmmm dd yyyy  h:mm AM/PM", "\u041D 01 2022  4:32 \u04AF.\u04E9."},
		{"44562.189571759256", "[$-50]mmmmmm dd yyyy  h:mm AM/PM", "\u041D\u044D\u0433\u0434\u04AF\u0433\u044D\u044D\u0440 \u0441\u0430\u0440 01 2022  4:32 \u04AF.\u04E9."},
		{"43543.503206018519", "[$-50]mmm dd yyyy  h:mm AM/PM", "3-р сар 19 2019  12:04 \u04AF.\u0445."},
		{"43543.503206018519", "[$-50]mmmm dd yyyy  h:mm AM/PM aaa", "\u0413\u0443\u0440\u0430\u0432\u0434\u0443\u0433\u0430\u0430\u0440 \u0441\u0430\u0440 19 2019  12:04 \u04AF.\u0445. \u041C\u044F"},
		{"43543.503206018519", "[$-50]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0413 19 2019  12:04 \u04AF.\u0445. \u041C\u044F"},
		{"43543.503206018519", "[$-50]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0413\u0443\u0440\u0430\u0432\u0434\u0443\u0433\u0430\u0430\u0440 \u0441\u0430\u0440 19 2019  12:04 \u04AF.\u0445. \u043C\u044F\u0433\u043C\u0430\u0440"},
		{"44562.189571759256", "[$-7850]mmm dd yyyy  h:mm AM/PM", "1-р сар 01 2022  4:32 \u04AF.\u04E9."},
		{"44562.189571759256", "[$-7850]mmmm dd yyyy  h:mm AM/PM", "\u041D\u044D\u0433\u0434\u04AF\u0433\u044D\u044D\u0440 \u0441\u0430\u0440 01 2022  4:32 \u04AF.\u04E9."},
		{"44562.189571759256", "[$-7850]mmmmm dd yyyy  h:mm AM/PM", "\u041D 01 2022  4:32 \u04AF.\u04E9."},
		{"44562.189571759256", "[$-7850]mmmmmm dd yyyy  h:mm AM/PM", "\u041D\u044D\u0433\u0434\u04AF\u0433\u044D\u044D\u0440 \u0441\u0430\u0440 01 2022  4:32 \u04AF.\u04E9."},
		{"43543.503206018519", "[$-7850]mmm dd yyyy  h:mm AM/PM", "3-р сар 19 2019  12:04 \u04AF.\u0445."},
		{"43543.503206018519", "[$-7850]mmmm dd yyyy  h:mm AM/PM aaa", "\u0413\u0443\u0440\u0430\u0432\u0434\u0443\u0433\u0430\u0430\u0440 \u0441\u0430\u0440 19 2019  12:04 \u04AF.\u0445. \u041C\u044F"},
		{"43543.503206018519", "[$-7850]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0413 19 2019  12:04 \u04AF.\u0445. \u041C\u044F"},
		{"43543.503206018519", "[$-7850]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0413\u0443\u0440\u0430\u0432\u0434\u0443\u0433\u0430\u0430\u0440 \u0441\u0430\u0440 19 2019  12:04 \u04AF.\u0445. \u043C\u044F\u0433\u043C\u0430\u0440"},
		{"44562.189571759256", "[$-450]mmm dd yyyy  h:mm AM/PM", "1-р сар 01 2022  4:32 \u04AF.\u04E9."},
		{"44562.189571759256", "[$-450]mmmm dd yyyy  h:mm AM/PM", "\u041D\u044D\u0433\u0434\u04AF\u0433\u044D\u044D\u0440 \u0441\u0430\u0440 01 2022  4:32 \u04AF.\u04E9."},
		{"44562.189571759256", "[$-450]mmmmm dd yyyy  h:mm AM/PM", "\u041D 01 2022  4:32 \u04AF.\u04E9."},
		{"44562.189571759256", "[$-450]mmmmmm dd yyyy  h:mm AM/PM", "\u041D\u044D\u0433\u0434\u04AF\u0433\u044D\u044D\u0440 \u0441\u0430\u0440 01 2022  4:32 \u04AF.\u04E9."},
		{"43543.503206018519", "[$-450]mmm dd yyyy  h:mm AM/PM", "3-р сар 19 2019  12:04 \u04AF.\u0445."},
		{"43543.503206018519", "[$-450]mmmm dd yyyy  h:mm AM/PM aaa", "\u0413\u0443\u0440\u0430\u0432\u0434\u0443\u0433\u0430\u0430\u0440 \u0441\u0430\u0440 19 2019  12:04 \u04AF.\u0445. \u041C\u044F"},
		{"43543.503206018519", "[$-450]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0413 19 2019  12:04 \u04AF.\u0445. \u041C\u044F"},
		{"43543.503206018519", "[$-450]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0413\u0443\u0440\u0430\u0432\u0434\u0443\u0433\u0430\u0430\u0440 \u0441\u0430\u0440 19 2019  12:04 \u04AF.\u0445. \u043C\u044F\u0433\u043C\u0430\u0440"},
		{"44562.189571759256", "[$-7C50]mmm dd yyyy  h:mm AM/PM", "M01 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-7C50]mmm dd yyyy  h:mm AM/PM", "M12 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C50]mmmm dd yyyy  h:mm AM/PM", "M01 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-7C50]mmmm dd yyyy  h:mm AM/PM aaa", "M12 01 2022  4:32 AM \u182D\u1820\u1837\u1820\u182D\u202F\u1824\u1828%20\u1833\u1825\u1837\u182A\u1821\u1828"},
		{"44562.189571759256", "[$-7C50]mmmmm dd yyyy  h:mm AM/PM ddd", "M 01 2022  4:32 AM \u182D\u1820\u1837\u1820\u182D\u202F\u1824\u1828%20\u1835\u1822\u1837\u182D\u1824\u182D\u1820\u1828"},
		{"44896.18957170139", "[$-7C50]mmmmm dd yyyy  h:mm AM/PM dddd", "M 01 2022  4:32 AM \u182D\u1820\u1837\u1820\u182D\u202F\u1824\u1828%20\u1833\u1825\u1837\u182A\u1821\u1828"},
		{"44562.189571759256", "[$-850]mmm dd yyyy  h:mm AM/PM", "M01 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-850]mmm dd yyyy  h:mm AM/PM", "M12 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-850]mmmm dd yyyy  h:mm AM/PM", "M01 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-850]mmmm dd yyyy  h:mm AM/PM aaa", "M12 01 2022  4:32 AM \u182D\u1820\u1837\u1820\u182D\u202F\u1824\u1828%20\u1833\u1825\u1837\u182A\u1821\u1828"},
		{"44562.189571759256", "[$-850]mmmmm dd yyyy  h:mm AM/PM ddd", "M 01 2022  4:32 AM \u182D\u1820\u1837\u1820\u182D\u202F\u1824\u1828%20\u1835\u1822\u1837\u182D\u1824\u182D\u1820\u1828"},
		{"44896.18957170139", "[$-850]mmmmm dd yyyy  h:mm AM/PM dddd", "M 01 2022  4:32 AM \u182D\u1820\u1837\u1820\u182D\u202F\u1824\u1828%20\u1833\u1825\u1837\u182A\u1821\u1828"},
		{"44562.189571759256", "[$-C50]mmm dd yyyy  h:mm AM/PM", "M01 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-C50]mmm dd yyyy  h:mm AM/PM", "M12 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-C50]mmmm dd yyyy  h:mm AM/PM", "M01 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-C50]mmmm dd yyyy  h:mm AM/PM aaa", "M12 01 2022  4:32 AM \u182B\u1826\u1837\u182A\u1826"},
		{"44562.189571759256", "[$-C50]mmmmm dd yyyy  h:mm AM/PM ddd", "M 01 2022  4:32 AM \u182A\u1822\u182E\u182A\u1820"},
		{"44896.18957170139", "[$-C50]mmmmm dd yyyy  h:mm AM/PM dddd", "M 01 2022  4:32 AM \u182B\u1826\u1837\u182A\u1826"},
		{"44562.189571759256", "[$-61]mmm dd yyyy  h:mm AM/PM", "\u091C\u0928 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-61]mmmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-61]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-61]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"43543.503206018519", "[$-61]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928"},
		{"43543.503206018519", "[$-61]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0919\u094D\u0917\u0932"},
		{"43543.503206018519", "[$-61]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0919\u094D\u0917\u0932"},
		{"43543.503206018519", "[$-61]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0919\u094D\u0917\u0932\u0935\u093E\u0930"},
		{"44562.189571759256", "[$-861]mmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-861]mmmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-861]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-861]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"43543.503206018519", "[$-861]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928"},
		{"43543.503206018519", "[$-861]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0919\u094D\u0917\u0932"},
		{"43543.503206018519", "[$-861]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0919\u094D\u0917\u0932"},
		{"43543.503206018519", "[$-861]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0919\u094D\u0917\u0932\u092C\u093E\u0930"},
		{"44562.189571759256", "[$-461]mmm dd yyyy  h:mm AM/PM", "\u091C\u0928 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-461]mmmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-461]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"44562.189571759256", "[$-461]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u0928\u0935\u0930\u0940 01 2022  4:32 \u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928"},
		{"43543.503206018519", "[$-461]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928"},
		{"43543.503206018519", "[$-461]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0919\u094D\u0917\u0932"},
		{"43543.503206018519", "[$-461]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0919\u094D\u0917\u0932"},
		{"43543.503206018519", "[$-461]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u0905\u092A\u0930\u093E\u0939\u094D\u0928 \u092E\u0919\u094D\u0917\u0932\u0935\u093E\u0930"},
		{"44562.189571759256", "[$-14]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-14]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-14]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-14]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-14]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-14]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 p.m. tir."},
		{"43543.503206018519", "[$-14]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 p.m. tir."},
		{"43543.503206018519", "[$-14]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 p.m. tirsdag"},
		{"44562.189571759256", "[$-7C14]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-7C14]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-7C14]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-7C14]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-7C14]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-7C14]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 p.m. tir"},
		{"43543.503206018519", "[$-7C14]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 p.m. tir"},
		{"43543.503206018519", "[$-7C14]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 p.m. tirsdag"},
		{"44562.189571759256", "[$-414]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-414]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-414]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-414]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-414]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-414]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 p.m. tir"},
		{"43543.503206018519", "[$-414]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 p.m. tir"},
		{"43543.503206018519", "[$-414]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 p.m. tirsdag"},
		{"44562.189571759256", "[$-7814]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 f.m."},
		{"44562.189571759256", "[$-7814]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 f.m."},
		{"44562.189571759256", "[$-7814]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 f.m."},
		{"44562.189571759256", "[$-7814]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 f.m."},
		{"43543.503206018519", "[$-7814]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 e.m."},
		{"43543.503206018519", "[$-7814]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 e.m. tys"},
		{"43543.503206018519", "[$-7814]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 e.m. tys"},
		{"43543.503206018519", "[$-7814]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 e.m. tysdag"},
		{"44562.189571759256", "[$-814]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 f.m."},
		{"44562.189571759256", "[$-814]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 f.m."},
		{"44562.189571759256", "[$-814]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 f.m."},
		{"44562.189571759256", "[$-814]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 f.m."},
		{"43543.503206018519", "[$-814]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 e.m."},
		{"43543.503206018519", "[$-814]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 e.m. tys"},
		{"43543.503206018519", "[$-814]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 e.m. tys"},
		{"43543.503206018519", "[$-814]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 e.m. tysdag"},
		{"44562.189571759256", "[$-82]mmm dd yyyy  h:mm AM/PM", "gen. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-82]mmmm dd yyyy  h:mm AM/PM", "genièr 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-82]mmmmm dd yyyy  h:mm AM/PM", "g 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-82]mmmmmm dd yyyy  h:mm AM/PM", "genièr 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-82]mmm dd yyyy  h:mm AM/PM", "març 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-82]mmmm dd yyyy  h:mm AM/PM aaa", "març 19 2019  12:04 PM dma."},
		{"43543.503206018519", "[$-82]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM dma."},
		{"43543.503206018519", "[$-82]mmmmmm dd yyyy  h:mm AM/PM dddd", "març 19 2019  12:04 PM dimarts"},
		{"44562.189571759256", "[$-482]mmm dd yyyy  h:mm AM/PM", "gen. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-482]mmmm dd yyyy  h:mm AM/PM", "genièr 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-482]mmmmm dd yyyy  h:mm AM/PM", "g 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-482]mmmmmm dd yyyy  h:mm AM/PM", "genièr 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-482]mmm dd yyyy  h:mm AM/PM", "març 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-482]mmmm dd yyyy  h:mm AM/PM aaa", "març 19 2019  12:04 PM dma."},
		{"43543.503206018519", "[$-482]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM dma."},
		{"43543.503206018519", "[$-482]mmmmmm dd yyyy  h:mm AM/PM dddd", "març 19 2019  12:04 PM dimarts"},
		{"44562.189571759256", "[$-48]mmm dd yyyy  h:mm AM/PM", "\u0B1C\u0B3E\u0B28\u0B41\u0B5F\u0B3E\u0B30\u0B40 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-48]mmmm dd yyyy  h:mm AM/PM", "\u0B1C\u0B3E\u0B28\u0B41\u0B5F\u0B3E\u0B30\u0B40 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-48]mmmmm dd yyyy  h:mm AM/PM", "\u0B1C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-48]mmmmmm dd yyyy  h:mm AM/PM", "\u0B1C\u0B3E\u0B28\u0B41\u0B5F\u0B3E\u0B30\u0B40 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-48]mmm dd yyyy  h:mm AM/PM", "\u0B2E\u0B3E\u0B30\u0B4D\u0B1A\u0B4D\u0B1A 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-48]mmmm dd yyyy  h:mm AM/PM aaa", "\u0B2E\u0B3E\u0B30\u0B4D\u0B1A\u0B4D\u0B1A 19 2019  12:04 PM \u0B2E\u0B19\u0B4D\u0B17\u0B33."},
		{"43543.503206018519", "[$-48]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0B2E 19 2019  12:04 PM \u0B2E\u0B19\u0B4D\u0B17\u0B33."},
		{"43543.503206018519", "[$-48]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0B2E\u0B3E\u0B30\u0B4D\u0B1A\u0B4D\u0B1A 19 2019  12:04 PM \u0B2E\u0B19\u0B4D\u0B17\u0B33\u0B2C\u0B3E\u0B30"},
		{"44562.189571759256", "[$-448]mmm dd yyyy  h:mm AM/PM", "\u0B1C\u0B3E\u0B28\u0B41\u0B5F\u0B3E\u0B30\u0B40 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-448]mmmm dd yyyy  h:mm AM/PM", "\u0B1C\u0B3E\u0B28\u0B41\u0B5F\u0B3E\u0B30\u0B40 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-448]mmmmm dd yyyy  h:mm AM/PM", "\u0B1C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-448]mmmmmm dd yyyy  h:mm AM/PM", "\u0B1C\u0B3E\u0B28\u0B41\u0B5F\u0B3E\u0B30\u0B40 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-448]mmm dd yyyy  h:mm AM/PM", "\u0B2E\u0B3E\u0B30\u0B4D\u0B1A\u0B4D\u0B1A 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-448]mmmm dd yyyy  h:mm AM/PM aaa", "\u0B2E\u0B3E\u0B30\u0B4D\u0B1A\u0B4D\u0B1A 19 2019  12:04 PM \u0B2E\u0B19\u0B4D\u0B17\u0B33."},
		{"43543.503206018519", "[$-448]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0B2E 19 2019  12:04 PM \u0B2E\u0B19\u0B4D\u0B17\u0B33."},
		{"43543.503206018519", "[$-448]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0B2E\u0B3E\u0B30\u0B4D\u0B1A\u0B4D\u0B1A 19 2019  12:04 PM \u0B2E\u0B19\u0B4D\u0B17\u0B33\u0B2C\u0B3E\u0B30"},
		{"44562.189571759256", "[$-72]mmm dd yyyy  h:mm AM/PM", "Ama 01 2022  4:32 WD"},
		{"44562.189571759256", "[$-72]mmmm dd yyyy  h:mm AM/PM", "Amajjii 01 2022  4:32 WD"},
		{"44562.189571759256", "[$-72]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 WD"},
		{"44562.189571759256", "[$-72]mmmmmm dd yyyy  h:mm AM/PM", "Amajjii 01 2022  4:32 WD"},
		{"43543.503206018519", "[$-72]mmm dd yyyy  h:mm AM/PM", "Bit 19 2019  12:04 WB"},
		{"43543.503206018519", "[$-72]mmmm dd yyyy  h:mm AM/PM aaa", "Bitooteessa 19 2019  12:04 WB Qib"},
		{"43543.503206018519", "[$-72]mmmmm dd yyyy  h:mm AM/PM ddd", "B 19 2019  12:04 WB Qib"},
		{"43543.503206018519", "[$-72]mmmmmm dd yyyy  h:mm AM/PM dddd", "Bitooteessa 19 2019  12:04 WB Qibxata"},
		{"44562.189571759256", "[$-472]mmm dd yyyy  h:mm AM/PM", "Ama 01 2022  4:32 WD"},
		{"44562.189571759256", "[$-472]mmmm dd yyyy  h:mm AM/PM", "Amajjii 01 2022  4:32 WD"},
		{"44562.189571759256", "[$-472]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 WD"},
		{"44562.189571759256", "[$-472]mmmmmm dd yyyy  h:mm AM/PM", "Amajjii 01 2022  4:32 WD"},
		{"43543.503206018519", "[$-472]mmm dd yyyy  h:mm AM/PM", "Bit 19 2019  12:04 WB"},
		{"43543.503206018519", "[$-472]mmmm dd yyyy  h:mm AM/PM aaa", "Bitooteessa 19 2019  12:04 WB Qib"},
		{"43543.503206018519", "[$-472]mmmmm dd yyyy  h:mm AM/PM ddd", "B 19 2019  12:04 WB Qib"},
		{"43543.503206018519", "[$-472]mmmmmm dd yyyy  h:mm AM/PM dddd", "Bitooteessa 19 2019  12:04 WB Qibxata"},
		{"44562.189571759256", "[$-63]mmm dd yyyy  h:mm AM/PM", "\u0633\u0644\u0648\u0627\u063A\u0647 01 2022  4:32 \u063A.\u0645."},
		{"44562.189571759256", "[$-63]mmmm dd yyyy  h:mm AM/PM", "\u0633\u0644\u0648\u0627\u063A\u0647 01 2022  4:32 \u063A.\u0645."},
		{"44562.189571759256", "[$-63]mmmmm dd yyyy  h:mm AM/PM", "\u0633\u0644\u0648\u0627\u063A\u0647 01 2022  4:32 \u063A.\u0645."},
		{"44562.189571759256", "[$-63]mmmmmm dd yyyy  h:mm AM/PM", "\u0633\u0644\u0648\u0627\u063A\u0647 01 2022  4:32 \u063A.\u0645."},
		{"44713.188888888886", "[$-63]mmm dd yyyy  h:mm AM/PM", "\u0686\u0646\u06AB\u0627 \u069A 01 2022  4:32 \u063A.\u0645."},
		{"44713.188888888886", "[$-63]mmmm dd yyyy  h:mm AM/PM", "\u0686\u0646\u06AB\u0627 \u069A\u0632\u0645\u0631\u0649 01 2022  4:32 \u063A.\u0645."},
		{"44713.188888888886", "[$-63]mmmmm dd yyyy  h:mm AM/PM", "\u0686\u0646\u06AB\u0627 \u069A\u0632\u0645\u0631\u0649 01 2022  4:32 \u063A.\u0645."},
		{"44713.188888888886", "[$-63]mmmmmm dd yyyy  h:mm AM/PM", "\u0686\u0646\u06AB\u0627 \u069A\u0632\u0645\u0631\u0649 01 2022  4:32 \u063A.\u0645."},
		{"43543.503206018519", "[$-63]mmm dd yyyy  h:mm AM/PM", "\u0648\u0631\u0649 19 2019  12:04 \u063A.\u0648."},
		{"43543.503206018519", "[$-63]mmmm dd yyyy  h:mm AM/PM aaa", "\u0648\u0631\u0649 19 2019  12:04 \u063A.\u0648. \u062F\u0631\u06D0\u0646\u06CD"},
		{"43543.503206018519", "[$-63]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0648\u0631\u0649 19 2019  12:04 \u063A.\u0648. \u062F\u0631\u06D0\u0646\u06CD"},
		{"43543.503206018519", "[$-63]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0648\u0631\u0649 19 2019  12:04 \u063A.\u0648. \u062F\u0631\u06D0\u0646\u06CD"},
		{"44562.189571759256", "[$-463]mmm dd yyyy  h:mm AM/PM", "\u0633\u0644\u0648\u0627\u063A\u0647 01 2022  4:32 \u063A.\u0645."},
		{"44562.189571759256", "[$-463]mmmm dd yyyy  h:mm AM/PM", "\u0633\u0644\u0648\u0627\u063A\u0647 01 2022  4:32 \u063A.\u0645."},
		{"44562.189571759256", "[$-463]mmmmm dd yyyy  h:mm AM/PM", "\u0633\u0644\u0648\u0627\u063A\u0647 01 2022  4:32 \u063A.\u0645."},
		{"44562.189571759256", "[$-463]mmmmmm dd yyyy  h:mm AM/PM", "\u0633\u0644\u0648\u0627\u063A\u0647 01 2022  4:32 \u063A.\u0645."},
		{"44713.188888888886", "[$-463]mmm dd yyyy  h:mm AM/PM", "\u0686\u0646\u06AB\u0627 \u069A 01 2022  4:32 \u063A.\u0645."},
		{"44713.188888888886", "[$-463]mmmm dd yyyy  h:mm AM/PM", "\u0686\u0646\u06AB\u0627 \u069A\u0632\u0645\u0631\u0649 01 2022  4:32 \u063A.\u0645."},
		{"44713.188888888886", "[$-463]mmmmm dd yyyy  h:mm AM/PM", "\u0686\u0646\u06AB\u0627 \u069A\u0632\u0645\u0631\u0649 01 2022  4:32 \u063A.\u0645."},
		{"44713.188888888886", "[$-463]mmmmmm dd yyyy  h:mm AM/PM", "\u0686\u0646\u06AB\u0627 \u069A\u0632\u0645\u0631\u0649 01 2022  4:32 \u063A.\u0645."},
		{"43543.503206018519", "[$-463]mmm dd yyyy  h:mm AM/PM", "\u0648\u0631\u0649 19 2019  12:04 \u063A.\u0648."},
		{"43543.503206018519", "[$-463]mmmm dd yyyy  h:mm AM/PM aaa", "\u0648\u0631\u0649 19 2019  12:04 \u063A.\u0648. \u062F\u0631\u06D0\u0646\u06CD"},
		{"43543.503206018519", "[$-463]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0648\u0631\u0649 19 2019  12:04 \u063A.\u0648. \u062F\u0631\u06D0\u0646\u06CD"},
		{"43543.503206018519", "[$-463]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0648\u0631\u0649 19 2019  12:04 \u063A.\u0648. \u062F\u0631\u06D0\u0646\u06CD"},
		{"44562.189571759256", "[$-29]mmm dd yyyy  h:mm AM/PM", "\u0698\u0627\u0646\u0648\u064A\u0647 01 2022  4:32 \u0642.\u0638"},
		{"44562.189571759256", "[$-29]mmmm dd yyyy  h:mm AM/PM", "\u0698\u0627\u0646\u0648\u064A\u0647 01 2022  4:32 \u0642.\u0638"},
		{"44562.189571759256", "[$-29]mmmmm dd yyyy  h:mm AM/PM", "\u0698 01 2022  4:32 \u0642.\u0638"},
		{"44562.189571759256", "[$-29]mmmmmm dd yyyy  h:mm AM/PM", "\u0698\u0627\u0646\u0648\u064A\u0647 01 2022  4:32 \u0642.\u0638"},
		{"43543.503206018519", "[$-29]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0628.\u0638"},
		{"43543.503206018519", "[$-29]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0628.\u0638 \u0633\u0647%A0\u0634\u0646\u0628\u0647"},
		{"43543.503206018519", "[$-29]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0628.\u0638 \u0633\u0647%A0\u0634\u0646\u0628\u0647"},
		{"43543.503206018519", "[$-29]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0628.\u0638 \u0633\u0647%A0\u0634\u0646\u0628\u0647"},
		{"44562.189571759256", "[$-429]mmm dd yyyy  h:mm AM/PM", "\u0698\u0627\u0646\u0648\u064A\u0647 01 2022  4:32 \u0642.\u0638"},
		{"44562.189571759256", "[$-429]mmmm dd yyyy  h:mm AM/PM", "\u0698\u0627\u0646\u0648\u064A\u0647 01 2022  4:32 \u0642.\u0638"},
		{"44562.189571759256", "[$-429]mmmmm dd yyyy  h:mm AM/PM", "\u0698 01 2022  4:32 \u0642.\u0638"},
		{"44562.189571759256", "[$-429]mmmmmm dd yyyy  h:mm AM/PM", "\u0698\u0627\u0646\u0648\u064A\u0647 01 2022  4:32 \u0642.\u0638"},
		{"43543.503206018519", "[$-429]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0628.\u0638"},
		{"43543.503206018519", "[$-429]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0628.\u0638 \u0633\u0647%A0\u0634\u0646\u0628\u0647"},
		{"43543.503206018519", "[$-429]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0628.\u0638 \u0633\u0647%A0\u0634\u0646\u0628\u0647"},
		{"43543.503206018519", "[$-429]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0633 19 2019  12:04 \u0628.\u0638 \u0633\u0647%A0\u0634\u0646\u0628\u0647"},
		{"44562.189571759256", "[$-15]mmm dd yyyy  h:mm AM/PM", "sty 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-15]mmmm dd yyyy  h:mm AM/PM", "styczeń 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-15]mmmmm dd yyyy  h:mm AM/PM", "s 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-15]mmmmmm dd yyyy  h:mm AM/PM", "styczeń 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-15]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-15]mmmm dd yyyy  h:mm AM/PM aaa", "marzec 19 2019  12:04 PM wt."},
		{"43543.503206018519", "[$-15]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM wt."},
		{"43543.503206018519", "[$-15]mmmmmm dd yyyy  h:mm AM/PM dddd", "marzec 19 2019  12:04 PM wtorek"},
		{"44562.189571759256", "[$-415]mmm dd yyyy  h:mm AM/PM", "sty 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-415]mmmm dd yyyy  h:mm AM/PM", "styczeń 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-415]mmmmm dd yyyy  h:mm AM/PM", "s 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-415]mmmmmm dd yyyy  h:mm AM/PM", "styczeń 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-415]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-415]mmmm dd yyyy  h:mm AM/PM aaa", "marzec 19 2019  12:04 PM wt."},
		{"43543.503206018519", "[$-415]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM wt."},
		{"43543.503206018519", "[$-415]mmmmmm dd yyyy  h:mm AM/PM dddd", "marzec 19 2019  12:04 PM wtorek"},
		{"44562.189571759256", "[$-16]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-16]mmmm dd yyyy  h:mm AM/PM", "janeiro 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-16]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-16]mmmmmm dd yyyy  h:mm AM/PM", "janeiro 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-16]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-16]mmmm dd yyyy  h:mm AM/PM aaa", "março 19 2019  12:04 PM ter"},
		{"43543.503206018519", "[$-16]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM ter"},
		{"43543.503206018519", "[$-16]mmmmmm dd yyyy  h:mm AM/PM dddd", "março 19 2019  12:04 PM terça-feira"},
		{"44562.189571759256", "[$-416]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-416]mmmm dd yyyy  h:mm AM/PM", "janeiro 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-416]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-416]mmmmmm dd yyyy  h:mm AM/PM", "janeiro 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-416]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-416]mmmm dd yyyy  h:mm AM/PM aaa", "março 19 2019  12:04 PM ter"},
		{"43543.503206018519", "[$-416]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM ter"},
		{"43543.503206018519", "[$-416]mmmmmm dd yyyy  h:mm AM/PM dddd", "março 19 2019  12:04 PM terça-feira"},
		{"44562.189571759256", "[$-816]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-816]mmmm dd yyyy  h:mm AM/PM", "janeiro 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-816]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-816]mmmmmm dd yyyy  h:mm AM/PM", "janeiro 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-816]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-816]mmmm dd yyyy  h:mm AM/PM aaa", "março 19 2019  12:04 PM ter"},
		{"43543.503206018519", "[$-816]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM ter"},
		{"43543.503206018519", "[$-816]mmmmmm dd yyyy  h:mm AM/PM dddd", "março 19 2019  12:04 PM terça-feira"},
		{"44562.189571759256", "[$-46]mmm dd yyyy  h:mm AM/PM", "\u0A1C\u0A28\u0A35\u0A30\u0A40 01 2022  4:32 \u0A38\u0A35\u0A47\u0A30"},
		{"44562.189571759256", "[$-46]mmmm dd yyyy  h:mm AM/PM", "\u0A1C\u0A28\u0A35\u0A30\u0A40 01 2022  4:32 \u0A38\u0A35\u0A47\u0A30"},
		{"44562.189571759256", "[$-46]mmmmm dd yyyy  h:mm AM/PM", "\u0A1C 01 2022  4:32 \u0A38\u0A35\u0A47\u0A30"},
		{"44562.189571759256", "[$-46]mmmmmm dd yyyy  h:mm AM/PM", "\u0A1C\u0A28\u0A35\u0A30\u0A40 01 2022  4:32 \u0A38\u0A35\u0A47\u0A30"},
		{"43543.503206018519", "[$-46]mmm dd yyyy  h:mm AM/PM", "\u0A2E\u0A3E\u0A30\u0A1A 19 2019  12:04 \u0A38\u0A3C\u0A3E\u0A2E"},
		{"43543.503206018519", "[$-46]mmmm dd yyyy  h:mm AM/PM aaa", "\u0A2E\u0A3E\u0A30\u0A1A 19 2019  12:04 \u0A38\u0A3C\u0A3E\u0A2E \u0A2E\u0A70\u0A17\u0A32."},
		{"43543.503206018519", "[$-46]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0A2E 19 2019  12:04 \u0A38\u0A3C\u0A3E\u0A2E \u0A2E\u0A70\u0A17\u0A32."},
		{"43543.503206018519", "[$-46]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0A2E\u0A3E\u0A30\u0A1A 19 2019  12:04 \u0A38\u0A3C\u0A3E\u0A2E \u0A2E\u0A70\u0A17\u0A32\u0A35\u0A3E\u0A30"},
		{"44562.189571759256", "[$-7C46]mmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C46]mmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C46]mmmmm dd yyyy  h:mm AM/PM", "\u062C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C46]mmmmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C46]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C46]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0628\u062F\u06BE"},
		{"43543.503206018519", "[$-7C46]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 PM \u0628\u062F\u06BE"},
		{"43543.503206018519", "[$-7C46]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0628\u062F\u06BE"},
		{"44562.189571759256", "[$-446]mmm dd yyyy  h:mm AM/PM", "\u0A1C\u0A28\u0A35\u0A30\u0A40 01 2022  4:32 \u0A38\u0A35\u0A47\u0A30"},
		{"44562.189571759256", "[$-446]mmmm dd yyyy  h:mm AM/PM", "\u0A1C\u0A28\u0A35\u0A30\u0A40 01 2022  4:32 \u0A38\u0A35\u0A47\u0A30"},
		{"44562.189571759256", "[$-446]mmmmm dd yyyy  h:mm AM/PM", "\u0A1C 01 2022  4:32 \u0A38\u0A35\u0A47\u0A30"},
		{"44562.189571759256", "[$-446]mmmmmm dd yyyy  h:mm AM/PM", "\u0A1C\u0A28\u0A35\u0A30\u0A40 01 2022  4:32 \u0A38\u0A35\u0A47\u0A30"},
		{"43543.503206018519", "[$-446]mmm dd yyyy  h:mm AM/PM", "\u0A2E\u0A3E\u0A30\u0A1A 19 2019  12:04 \u0A38\u0A3C\u0A3E\u0A2E"},
		{"43543.503206018519", "[$-446]mmmm dd yyyy  h:mm AM/PM aaa", "\u0A2E\u0A3E\u0A30\u0A1A 19 2019  12:04 \u0A38\u0A3C\u0A3E\u0A2E \u0A2E\u0A70\u0A17\u0A32."},
		{"43543.503206018519", "[$-446]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0A2E 19 2019  12:04 \u0A38\u0A3C\u0A3E\u0A2E \u0A2E\u0A70\u0A17\u0A32."},
		{"43543.503206018519", "[$-446]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0A2E\u0A3E\u0A30\u0A1A 19 2019  12:04 \u0A38\u0A3C\u0A3E\u0A2E \u0A2E\u0A70\u0A17\u0A32\u0A35\u0A3E\u0A30"},
		{"44562.189571759256", "[$-846]mmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-846]mmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-846]mmmmm dd yyyy  h:mm AM/PM", "\u062C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-846]mmmmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-846]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-846]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0628\u062F\u06BE"},
		{"43543.503206018519", "[$-846]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 PM \u0628\u062F\u06BE"},
		{"43543.503206018519", "[$-846]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0628\u062F\u06BE"},
		{"44562.189571759256", "[$-6B]mmm dd yyyy  h:mm AM/PM", "Qul 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-6B]mmmm dd yyyy  h:mm AM/PM", "Qulla puquy 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-6B]mmmmm dd yyyy  h:mm AM/PM", "Q 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-6B]mmmmmm dd yyyy  h:mm AM/PM", "Qulla puquy 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-6B]mmm dd yyyy  h:mm AM/PM", "Pau 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-6B]mmmm dd yyyy  h:mm AM/PM aaa", "Pauqar waray 19 2019  12:04 p.m. ati"},
		{"43543.503206018519", "[$-6B]mmmmm dd yyyy  h:mm AM/PM ddd", "P 19 2019  12:04 p.m. ati"},
		{"43543.503206018519", "[$-6B]mmmmmm dd yyyy  h:mm AM/PM dddd", "Pauqar waray 19 2019  12:04 p.m. atipachaw"},
		{"44562.189571759256", "[$-46B]mmm dd yyyy  h:mm AM/PM", "Qul 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-46B]mmmm dd yyyy  h:mm AM/PM", "Qulla puquy 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-46B]mmmmm dd yyyy  h:mm AM/PM", "Q 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-46B]mmmmmm dd yyyy  h:mm AM/PM", "Qulla puquy 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-46B]mmm dd yyyy  h:mm AM/PM", "Pau 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-46B]mmmm dd yyyy  h:mm AM/PM aaa", "Pauqar waray 19 2019  12:04 p.m. ati"},
		{"43543.503206018519", "[$-46B]mmmmm dd yyyy  h:mm AM/PM ddd", "P 19 2019  12:04 p.m. ati"},
		{"43543.503206018519", "[$-46B]mmmmmm dd yyyy  h:mm AM/PM dddd", "Pauqar waray 19 2019  12:04 p.m. atipachaw"},
		{"44562.189571759256", "[$-86B]mmm dd yyyy  h:mm AM/PM", "kull 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-86B]mmmm dd yyyy  h:mm AM/PM", "kulla 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-86B]mmmmm dd yyyy  h:mm AM/PM", "k 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-86B]mmmmmm dd yyyy  h:mm AM/PM", "kulla 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-86B]mmm dd yyyy  h:mm AM/PM", "paw 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-86B]mmmm dd yyyy  h:mm AM/PM aaa", "pawkar 19 2019  12:04 PM wan"},
		{"43543.503206018519", "[$-86B]mmmmm dd yyyy  h:mm AM/PM ddd", "p 19 2019  12:04 PM wan"},
		{"43543.503206018519", "[$-86B]mmmmmm dd yyyy  h:mm AM/PM dddd", "pawkar 19 2019  12:04 PM wanra"},
		{"44562.189571759256", "[$-C6B]mmm dd yyyy  h:mm AM/PM", "Qul 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-C6B]mmmm dd yyyy  h:mm AM/PM", "Qulla puquy 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-C6B]mmmmm dd yyyy  h:mm AM/PM", "Q 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-C6B]mmmmmm dd yyyy  h:mm AM/PM", "Qulla puquy 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-C6B]mmm dd yyyy  h:mm AM/PM", "Pau 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-C6B]mmmm dd yyyy  h:mm AM/PM aaa", "Pauqar waray 19 2019  12:04 p.m. Mar"},
		{"43543.503206018519", "[$-C6B]mmmmm dd yyyy  h:mm AM/PM ddd", "P 19 2019  12:04 p.m. Mar"},
		{"43543.503206018519", "[$-C6B]mmmmmm dd yyyy  h:mm AM/PM dddd", "Pauqar waray 19 2019  12:04 p.m. Martes"},
		{"44562.189571759256", "[$-18]mmm dd yyyy  h:mm AM/PM", "ian. 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-18]mmmm dd yyyy  h:mm AM/PM", "ianuarie 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-18]mmmmm dd yyyy  h:mm AM/PM", "i 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-18]mmmmmm dd yyyy  h:mm AM/PM", "ianuarie 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-18]mmm dd yyyy  h:mm AM/PM", "mar. 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-18]mmmm dd yyyy  h:mm AM/PM aaa", "martie 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-18]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-18]mmmmmm dd yyyy  h:mm AM/PM dddd", "martie 19 2019  12:04 p.m. marți"},
		{"44562.189571759256", "[$-818]mmm dd yyyy  h:mm AM/PM", "ian. 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-818]mmmm dd yyyy  h:mm AM/PM", "ianuarie 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-818]mmmmm dd yyyy  h:mm AM/PM", "i 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-818]mmmmmm dd yyyy  h:mm AM/PM", "ianuarie 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-818]mmm dd yyyy  h:mm AM/PM", "mar. 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-818]mmmm dd yyyy  h:mm AM/PM aaa", "martie 19 2019  12:04 p.m. Mar"},
		{"43543.503206018519", "[$-818]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 p.m. Mar"},
		{"43543.503206018519", "[$-818]mmmmmm dd yyyy  h:mm AM/PM dddd", "martie 19 2019  12:04 p.m. marți"},
		{"44562.189571759256", "[$-418]mmm dd yyyy  h:mm AM/PM", "ian. 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-418]mmmm dd yyyy  h:mm AM/PM", "ianuarie 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-418]mmmmm dd yyyy  h:mm AM/PM", "i 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-418]mmmmmm dd yyyy  h:mm AM/PM", "ianuarie 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-418]mmm dd yyyy  h:mm AM/PM", "mar. 19 2019  12:04 p.m."},
		{"43543.503206018519", "[$-418]mmmm dd yyyy  h:mm AM/PM aaa", "martie 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-418]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-418]mmmmmm dd yyyy  h:mm AM/PM dddd", "martie 19 2019  12:04 p.m. marți"},
		{"44562.189571759256", "[$-17]mmm dd yyyy  h:mm AM/PM", "schan. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-17]mmmm dd yyyy  h:mm AM/PM", "schaner 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-17]mmmmm dd yyyy  h:mm AM/PM", "s 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-17]mmmmmm dd yyyy  h:mm AM/PM", "schaner 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-17]mmm dd yyyy  h:mm AM/PM", "mars 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-17]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM ma"},
		{"43543.503206018519", "[$-17]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM ma"},
		{"43543.503206018519", "[$-17]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-417]mmm dd yyyy  h:mm AM/PM", "schan. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-417]mmmm dd yyyy  h:mm AM/PM", "schaner 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-417]mmmmm dd yyyy  h:mm AM/PM", "s 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-417]mmmmmm dd yyyy  h:mm AM/PM", "schaner 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-417]mmm dd yyyy  h:mm AM/PM", "mars 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-417]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM ma"},
		{"43543.503206018519", "[$-417]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM ma"},
		{"43543.503206018519", "[$-417]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 PM mardi"},
		{"44562.189571759256", "[$-19]mmm dd yyyy  h:mm AM/PM", "янв. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-19]mmmm dd yyyy  h:mm AM/PM", "январь 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-19]mmmmm dd yyyy  h:mm AM/PM", "я 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-19]mmm dd yyyy  h:mm AM/PM aaa", "март 19 2019  12:04 PM \u0412\u0442"},
		{"43543.503206018519", "[$-19]mmmm dd yyyy  h:mm AM/PM ddd", "март 19 2019  12:04 PM \u0412\u0442"},
		{"43543.503206018519", "[$-19]mmmmm dd yyyy  h:mm AM/PM dddd", "м 19 2019  12:04 PM \u0432\u0442\u043E\u0440\u043D\u0438\u043A"},
		{"43543.503206018519", "[$-819]mmm dd yyyy  h:mm AM/PM aaa", "март 19 2019  12:04 PM \u0412\u0442"},
		{"43543.503206018519", "[$-819]mmmm dd yyyy  h:mm AM/PM ddd", "март 19 2019  12:04 PM \u0412\u0442"},
		{"43543.503206018519", "[$-819]mmmmm dd yyyy  h:mm AM/PM dddd", "м 19 2019  12:04 PM \u0432\u0442\u043E\u0440\u043D\u0438\u043A"},
		{"43543.503206018519", "[$-419]mmm dd yyyy  h:mm AM/PM aaa", "март 19 2019  12:04 PM \u0412\u0442"},
		{"43543.503206018519", "[$-419]mmmm dd yyyy  h:mm AM/PM ddd", "март 19 2019  12:04 PM \u0412\u0442"},
		{"43543.503206018519", "[$-419]mmmmm dd yyyy  h:mm AM/PM dddd", "м 19 2019  12:04 PM \u0432\u0442\u043E\u0440\u043D\u0438\u043A"},
		{"44562.189571759256", "[$-85]mmm dd yyyy  h:mm AM/PM", "\u0422\u0445\u0441 01 2022  4:32 \u041A\u0418"},
		{"44562.189571759256", "[$-85]mmmm dd yyyy  h:mm AM/PM", "\u0422\u043E\u0445\u0441\u0443\u043D\u043D\u044C\u0443 01 2022  4:32 \u041A\u0418"},
		{"44562.189571759256", "[$-85]mmmmm dd yyyy  h:mm AM/PM", "\u0422 01 2022  4:32 \u041A\u0418"},
		{"44562.189571759256", "[$-85]mmmmmm dd yyyy  h:mm AM/PM", "\u0422\u043E\u0445\u0441\u0443\u043D\u043D\u044C\u0443 01 2022  4:32 \u041A\u0418"},
		{"43543.503206018519", "[$-85]mmm dd yyyy  h:mm AM/PM", "\u041A\u043B\u043D 19 2019  12:04 \u041A\u041A"},
		{"43543.503206018519", "[$-85]mmmm dd yyyy  h:mm AM/PM aaa", "\u041A\u0443\u043B\u0443\u043D \u0442\u0443\u0442\u0430\u0440 19 2019  12:04 \u041A\u041A \u043E\u043F"},
		{"43543.503206018519", "[$-85]mmmmm dd yyyy  h:mm AM/PM ddd", "\u041A 19 2019  12:04 \u041A\u041A \u043E\u043F"},
		{"43543.503206018519", "[$-85]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u041A\u0443\u043B\u0443\u043D \u0442\u0443\u0442\u0430\u0440 19 2019  12:04 \u041A\u041A \u041E\u043F\u0442\u0443\u043E\u0440\u0443\u043D\u043D\u044C\u0443\u043A"},
		{"44562.189571759256", "[$-485]mmm dd yyyy  h:mm AM/PM", "\u0422\u0445\u0441 01 2022  4:32 \u041A\u0418"},
		{"44562.189571759256", "[$-485]mmmm dd yyyy  h:mm AM/PM", "\u0422\u043E\u0445\u0441\u0443\u043D\u043D\u044C\u0443 01 2022  4:32 \u041A\u0418"},
		{"44562.189571759256", "[$-485]mmmmm dd yyyy  h:mm AM/PM", "\u0422 01 2022  4:32 \u041A\u0418"},
		{"44562.189571759256", "[$-485]mmmmmm dd yyyy  h:mm AM/PM", "\u0422\u043E\u0445\u0441\u0443\u043D\u043D\u044C\u0443 01 2022  4:32 \u041A\u0418"},
		{"43543.503206018519", "[$-485]mmm dd yyyy  h:mm AM/PM", "\u041A\u043B\u043D 19 2019  12:04 \u041A\u041A"},
		{"43543.503206018519", "[$-485]mmmm dd yyyy  h:mm AM/PM aaa", "\u041A\u0443\u043B\u0443\u043D \u0442\u0443\u0442\u0430\u0440 19 2019  12:04 \u041A\u041A \u043E\u043F"},
		{"43543.503206018519", "[$-485]mmmmm dd yyyy  h:mm AM/PM ddd", "\u041A 19 2019  12:04 \u041A\u041A \u043E\u043F"},
		{"43543.503206018519", "[$-485]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u041A\u0443\u043B\u0443\u043D \u0442\u0443\u0442\u0430\u0440 19 2019  12:04 \u041A\u041A \u041E\u043F\u0442\u0443\u043E\u0440\u0443\u043D\u043D\u044C\u0443\u043A"},
		{"44562.189571759256", "[$-703B]mmm dd yyyy  h:mm AM/PM", "uđiv 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-703B]mmmm dd yyyy  h:mm AM/PM", "uđđâivemáánu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-703B]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-703B]mmmmmm dd yyyy  h:mm AM/PM", "uđđâivemáánu 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-703B]mmm dd yyyy  h:mm AM/PM", "njuh 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-703B]mmmm dd yyyy  h:mm AM/PM aaa", "njuhčâmáánu 19 2019  12:04 PM maj"},
		{"43543.503206018519", "[$-703B]mmmmm dd yyyy  h:mm AM/PM ddd", "n 19 2019  12:04 PM maj"},
		{"43543.503206018519", "[$-703B]mmmmmm dd yyyy  h:mm AM/PM dddd", "njuhčâmáánu 19 2019  12:04 PM majebargâ"},
		{"44562.189571759256", "[$-243B]mmm dd yyyy  h:mm AM/PM", "uđiv 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-243B]mmmm dd yyyy  h:mm AM/PM", "uđđâivemáánu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-243B]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-243B]mmmmmm dd yyyy  h:mm AM/PM", "uđđâivemáánu 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-243B]mmm dd yyyy  h:mm AM/PM", "njuh 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-243B]mmmm dd yyyy  h:mm AM/PM aaa", "njuhčâmáánu 19 2019  12:04 PM maj"},
		{"43543.503206018519", "[$-243B]mmmmm dd yyyy  h:mm AM/PM ddd", "n 19 2019  12:04 PM maj"},
		{"43543.503206018519", "[$-243B]mmmmmm dd yyyy  h:mm AM/PM dddd", "njuhčâmáánu 19 2019  12:04 PM majebargâ"},
		{"44562.189571759256", "[$-7C3B]mmm dd yyyy  h:mm AM/PM", "ådåj 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C3B]mmmm dd yyyy  h:mm AM/PM", "ådåjakmánno 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C3B]mmmmm dd yyyy  h:mm AM/PM", "å 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C3B]mmmmmm dd yyyy  h:mm AM/PM", "ådåjakmánno 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C3B]mmm dd yyyy  h:mm AM/PM", "snju 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C3B]mmmm dd yyyy  h:mm AM/PM aaa", "sjnjuktjamánno 19 2019  12:04 PM dis"},
		{"43543.503206018519", "[$-7C3B]mmmmm dd yyyy  h:mm AM/PM ddd", "s 19 2019  12:04 PM dis"},
		{"43543.503206018519", "[$-7C3B]mmmmmm dd yyyy  h:mm AM/PM dddd", "sjnjuktjamánno 19 2019  12:04 PM dijstahka"},
		{"44562.189571759256", "[$-103B]mmm dd yyyy  h:mm AM/PM", "ådåj 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-103B]mmmm dd yyyy  h:mm AM/PM", "ådåjakmánno 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-103B]mmmmm dd yyyy  h:mm AM/PM", "å 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-103B]mmmmmm dd yyyy  h:mm AM/PM", "ådåjakmánno 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-103B]mmm dd yyyy  h:mm AM/PM", "snju 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-103B]mmmm dd yyyy  h:mm AM/PM aaa", "sjnjuktjamánno 19 2019  12:04 PM dis"},
		{"43543.503206018519", "[$-103B]mmmmm dd yyyy  h:mm AM/PM ddd", "s 19 2019  12:04 PM dis"},
		{"43543.503206018519", "[$-103B]mmmmmm dd yyyy  h:mm AM/PM dddd", "sjnjuktjamánno 19 2019  12:04 PM dijstahka"},
		{"44562.189571759256", "[$-143B]mmm dd yyyy  h:mm AM/PM", "ådåj 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-143B]mmmm dd yyyy  h:mm AM/PM", "ådåjakmánno 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-143B]mmmmm dd yyyy  h:mm AM/PM", "å 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-143B]mmmmmm dd yyyy  h:mm AM/PM", "ådåjakmánno 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-143B]mmm dd yyyy  h:mm AM/PM", "snju 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-143B]mmmm dd yyyy  h:mm AM/PM aaa", "sjnjuktjamánno 19 2019  12:04 PM dis"},
		{"43543.503206018519", "[$-143B]mmmmm dd yyyy  h:mm AM/PM ddd", "s 19 2019  12:04 PM dis"},
		{"43543.503206018519", "[$-143B]mmmmmm dd yyyy  h:mm AM/PM dddd", "sjnjuktjamánno 19 2019  12:04 PM dijstahka"},
		{"44562.189571759256", "[$-3B]mmm dd yyyy  h:mm AM/PM", "ođđj 01 2022  4:32 i.b."},
		{"44562.189571759256", "[$-3B]mmmm dd yyyy  h:mm AM/PM", "ođđajagemánnu 01 2022  4:32 i.b."},
		{"44562.189571759256", "[$-3B]mmmmm dd yyyy  h:mm AM/PM", "o 01 2022  4:32 i.b."},
		{"44562.189571759256", "[$-3B]mmmmmm dd yyyy  h:mm AM/PM", "ođđajagemánnu 01 2022  4:32 i.b."},
		{"43543.503206018519", "[$-3B]mmm dd yyyy  h:mm AM/PM", "njuk 19 2019  12:04 e.b."},
		{"43543.503206018519", "[$-3B]mmmm dd yyyy  h:mm AM/PM aaa", "njukčamánnu 19 2019  12:04 e.b. maŋ"},
		{"43543.503206018519", "[$-3B]mmmmm dd yyyy  h:mm AM/PM ddd", "n 19 2019  12:04 e.b. maŋ"},
		{"43543.503206018519", "[$-3B]mmmmmm dd yyyy  h:mm AM/PM dddd", "njukčamánnu 19 2019  12:04 e.b. maŋŋebárga"},
		{"44562.189571759256", "[$-C3B]mmm dd yyyy  h:mm AM/PM", "ođđj 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-C3B]mmmm dd yyyy  h:mm AM/PM", "ođđajagemánu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-C3B]mmmmm dd yyyy  h:mm AM/PM", "o 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-C3B]mmmmmm dd yyyy  h:mm AM/PM", "ođđajagemánu 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-C3B]mmm dd yyyy  h:mm AM/PM", "njuk 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-C3B]mmmm dd yyyy  h:mm AM/PM aaa", "njukčamánnu 19 2019  12:04 PM di"},
		{"43543.503206018519", "[$-C3B]mmmmm dd yyyy  h:mm AM/PM ddd", "n 19 2019  12:04 PM di"},
		{"43543.503206018519", "[$-C3B]mmmmmm dd yyyy  h:mm AM/PM dddd", "njukčamánnu 19 2019  12:04 PM maŋŋebárga"},
		{"44562.189571759256", "[$-43B]mmm dd yyyy  h:mm AM/PM", "ođđj 01 2022  4:32 i.b."},
		{"44562.189571759256", "[$-43B]mmmm dd yyyy  h:mm AM/PM", "ođđajagemánnu 01 2022  4:32 i.b."},
		{"44562.189571759256", "[$-43B]mmmmm dd yyyy  h:mm AM/PM", "o 01 2022  4:32 i.b."},
		{"44562.189571759256", "[$-43B]mmmmmm dd yyyy  h:mm AM/PM", "ođđajagemánnu 01 2022  4:32 i.b."},
		{"43543.503206018519", "[$-43B]mmm dd yyyy  h:mm AM/PM", "njuk 19 2019  12:04 e.b."},
		{"43543.503206018519", "[$-43B]mmmm dd yyyy  h:mm AM/PM aaa", "njukčamánnu 19 2019  12:04 e.b. maŋ"},
		{"43543.503206018519", "[$-43B]mmmmm dd yyyy  h:mm AM/PM ddd", "n 19 2019  12:04 e.b. maŋ"},
		{"43543.503206018519", "[$-43B]mmmmmm dd yyyy  h:mm AM/PM dddd", "njukčamánnu 19 2019  12:04 e.b. maŋŋebárga"},
		{"44562.189571759256", "[$-83B]mmm dd yyyy  h:mm AM/PM", "ođđj 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-83B]mmmm dd yyyy  h:mm AM/PM", "ođđajagemánnu 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-83B]mmmmm dd yyyy  h:mm AM/PM", "o 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-83B]mmmmmm dd yyyy  h:mm AM/PM", "ođđajagemánnu 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-83B]mmm dd yyyy  h:mm AM/PM", "njuk 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-83B]mmmm dd yyyy  h:mm AM/PM aaa", "njukčamánnu 19 2019  12:04 PM dis"},
		{"43543.503206018519", "[$-83B]mmmmm dd yyyy  h:mm AM/PM ddd", "n 19 2019  12:04 PM dis"},
		{"43543.503206018519", "[$-83B]mmmmmm dd yyyy  h:mm AM/PM dddd", "njukčamánnu 19 2019  12:04 PM disdat"},
		{"44562.189571759256", "[$-743B]mmm dd yyyy  h:mm AM/PM", "ođđee´jjmään 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-743B]mmmm dd yyyy  h:mm AM/PM", "ođđee´jjmään 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-743B]mmmmm dd yyyy  h:mm AM/PM", "o 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-743B]mmmmmm dd yyyy  h:mm AM/PM", "ođđee´jjmään 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-743B]mmm dd yyyy  h:mm AM/PM", "pâ´zzlâšttam-mään 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-743B]mmmm dd yyyy  h:mm AM/PM aaa", "pâ´zzlâšttam-mään 19 2019  12:04 PM m%E2"},
		{"43543.503206018519", "[$-743B]mmmmm dd yyyy  h:mm AM/PM ddd", "p 19 2019  12:04 PM m%E2"},
		{"43543.503206018519", "[$-743B]mmmmmm dd yyyy  h:mm AM/PM dddd", "pâ´zzlâšttam-mään 19 2019  12:04 PM m%E2%E2ibargg"},
		{"44562.189571759256", "[$-203B]mmm dd yyyy  h:mm AM/PM", "ođđee´jjmään 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-203B]mmmm dd yyyy  h:mm AM/PM", "ođđee´jjmään 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-203B]mmmmm dd yyyy  h:mm AM/PM", "o 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-203B]mmmmmm dd yyyy  h:mm AM/PM", "ođđee´jjmään 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-203B]mmm dd yyyy  h:mm AM/PM", "pâ´zzlâšttam-mään 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-203B]mmmm dd yyyy  h:mm AM/PM aaa", "pâ´zzlâšttam-mään 19 2019  12:04 PM m%E2"},
		{"43543.503206018519", "[$-203B]mmmmm dd yyyy  h:mm AM/PM ddd", "p 19 2019  12:04 PM m%E2"},
		{"43543.503206018519", "[$-203B]mmmmmm dd yyyy  h:mm AM/PM dddd", "pâ´zzlâšttam-mään 19 2019  12:04 PM m%E2%E2ibargg"},
		{"44562.189571759256", "[$-783B]mmm dd yyyy  h:mm AM/PM", "tsïen 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-783B]mmmm dd yyyy  h:mm AM/PM", "tsïengele 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-783B]mmmmm dd yyyy  h:mm AM/PM", "t 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-783B]mmmmmm dd yyyy  h:mm AM/PM", "tsïengele 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-783B]mmm dd yyyy  h:mm AM/PM", "njok 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-783B]mmmm dd yyyy  h:mm AM/PM aaa", "njoktje 19 2019  12:04 PM d%E6j"},
		{"43543.503206018519", "[$-783B]mmmmm dd yyyy  h:mm AM/PM ddd", "n 19 2019  12:04 PM d%E6j"},
		{"43543.503206018519", "[$-783B]mmmmmm dd yyyy  h:mm AM/PM dddd", "njoktje 19 2019  12:04 PM d%E6jsta"},
		{"44562.189571759256", "[$-183B]mmm dd yyyy  h:mm AM/PM", "tsïen 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-183B]mmmm dd yyyy  h:mm AM/PM", "tsïengele 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-183B]mmmmm dd yyyy  h:mm AM/PM", "t 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-183B]mmmmmm dd yyyy  h:mm AM/PM", "tsïengele 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-183B]mmm dd yyyy  h:mm AM/PM", "njok 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-183B]mmmm dd yyyy  h:mm AM/PM aaa", "njoktje 19 2019  12:04 PM d%E6j"},
		{"43543.503206018519", "[$-183B]mmmmm dd yyyy  h:mm AM/PM ddd", "n 19 2019  12:04 PM d%E6j"},
		{"43543.503206018519", "[$-183B]mmmmmm dd yyyy  h:mm AM/PM dddd", "njoktje 19 2019  12:04 PM d%E6jsta"},
		{"44562.189571759256", "[$-1C3B]mmm dd yyyy  h:mm AM/PM", "tsïen 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1C3B]mmmm dd yyyy  h:mm AM/PM", "tsïengele 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1C3B]mmmmm dd yyyy  h:mm AM/PM", "t 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1C3B]mmmmmm dd yyyy  h:mm AM/PM", "tsïengele 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-1C3B]mmm dd yyyy  h:mm AM/PM", "njok 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-1C3B]mmmm dd yyyy  h:mm AM/PM aaa", "njoktje 19 2019  12:04 PM d%E6j"},
		{"43543.503206018519", "[$-1C3B]mmmmm dd yyyy  h:mm AM/PM ddd", "n 19 2019  12:04 PM d%E6j"},
		{"43543.503206018519", "[$-1C3B]mmmmmm dd yyyy  h:mm AM/PM dddd", "njoktje 19 2019  12:04 PM d%E6jsta"},
		{"44562.189571759256", "[$-4F]mmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u094D\u092F\u0941\u0905\u0930\u0940 01 2022  4:32 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u0942\u0930\u094D\u0935"},
		{"44562.189571759256", "[$-4F]mmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u094D\u092F\u0941\u0905\u0930\u0940 01 2022  4:32 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u0942\u0930\u094D\u0935"},
		{"44562.189571759256", "[$-4F]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u0942\u0930\u094D\u0935"},
		{"44562.189571759256", "[$-4F]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u094D\u092F\u0941\u0905\u0930\u0940 01 2022  4:32 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u0942\u0930\u094D\u0935"},
		{"43543.503206018519", "[$-4F]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u091A\u094D\u092F\u093E\u0924"},
		{"43543.503206018519", "[$-4F]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u091A\u094D\u092F\u093E\u0924 \u092E\u0919\u094D\u0917"},
		{"43543.503206018519", "[$-4F]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u091A\u094D\u092F\u093E\u0924 \u092E\u0919\u094D\u0917"},
		{"43543.503206018519", "[$-4F]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u091A\u094D\u092F\u093E\u0924 \u092E\u0902\u0917\u0932\u0935\u093E\u0938\u0930\u0903"},
		{"44562.189571759256", "[$-44F]mmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u094D\u092F\u0941\u0905\u0930\u0940 01 2022  4:32 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u0942\u0930\u094D\u0935"},
		{"44562.189571759256", "[$-44F]mmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u094D\u092F\u0941\u0905\u0930\u0940 01 2022  4:32 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u0942\u0930\u094D\u0935"},
		{"44562.189571759256", "[$-44F]mmmmm dd yyyy  h:mm AM/PM", "\u091C 01 2022  4:32 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u0942\u0930\u094D\u0935"},
		{"44562.189571759256", "[$-44F]mmmmmm dd yyyy  h:mm AM/PM", "\u091C\u093E\u0928\u094D\u092F\u0941\u0905\u0930\u0940 01 2022  4:32 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u0942\u0930\u094D\u0935"},
		{"43543.503206018519", "[$-44F]mmm dd yyyy  h:mm AM/PM", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u091A\u094D\u092F\u093E\u0924"},
		{"43543.503206018519", "[$-44F]mmmm dd yyyy  h:mm AM/PM aaa", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u091A\u094D\u092F\u093E\u0924 \u092E\u0919\u094D\u0917"},
		{"43543.503206018519", "[$-44F]mmmmm dd yyyy  h:mm AM/PM ddd", "\u092E 19 2019  12:04 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u091A\u094D\u092F\u093E\u0924 \u092E\u0919\u094D\u0917"},
		{"43543.503206018519", "[$-44F]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u092E\u093E\u0930\u094D\u091A 19 2019  12:04 \u092E\u0927\u094D\u092F\u093E\u0928\u092A\u091A\u094D\u092F\u093E\u0924 \u092E\u0902\u0917\u0932\u0935\u093E\u0938\u0930\u0903"},
		{"44562.189571759256", "[$-91]mmm dd yyyy  h:mm AM/PM", "Faoi 01 2022  4:32 m"},
		{"44562.189571759256", "[$-91]mmmm dd yyyy  h:mm AM/PM", "Am Faoilleach 01 2022  4:32 m"},
		{"44562.189571759256", "[$-91]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 m"},
		{"44562.189571759256", "[$-91]mmmmmm dd yyyy  h:mm AM/PM", "Am Faoilleach 01 2022  4:32 m"},
		{"43543.503206018519", "[$-91]mmm dd yyyy  h:mm AM/PM", "Màrt 19 2019  12:04 f"},
		{"43543.503206018519", "[$-91]mmmm dd yyyy  h:mm AM/PM aaa", "Am Màrt 19 2019  12:04 f DiM"},
		{"43543.503206018519", "[$-91]mmmmm dd yyyy  h:mm AM/PM ddd", "A 19 2019  12:04 f DiM"},
		{"43543.503206018519", "[$-91]mmmmmm dd yyyy  h:mm AM/PM dddd", "Am Màrt 19 2019  12:04 f DiMàirt"},
		{"44562.189571759256", "[$-491]mmm dd yyyy  h:mm AM/PM", "Faoi 01 2022  4:32 m"},
		{"44562.189571759256", "[$-491]mmmm dd yyyy  h:mm AM/PM", "Am Faoilleach 01 2022  4:32 m"},
		{"44562.189571759256", "[$-491]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 m"},
		{"44562.189571759256", "[$-491]mmmmmm dd yyyy  h:mm AM/PM", "Am Faoilleach 01 2022  4:32 m"},
		{"43543.503206018519", "[$-491]mmm dd yyyy  h:mm AM/PM", "Màrt 19 2019  12:04 f"},
		{"43543.503206018519", "[$-491]mmmm dd yyyy  h:mm AM/PM aaa", "Am Màrt 19 2019  12:04 f DiM"},
		{"43543.503206018519", "[$-491]mmmmm dd yyyy  h:mm AM/PM ddd", "A 19 2019  12:04 f DiM"},
		{"43543.503206018519", "[$-491]mmmmmm dd yyyy  h:mm AM/PM dddd", "Am Màrt 19 2019  12:04 f DiMàirt"},
		{"44562.189571759256", "[$-6C1A]mmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6C1A]mmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6C1A]mmmmm dd yyyy  h:mm AM/PM", "\u0458 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6C1A]mmmmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-6C1A]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-6C1A]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442."},
		{"43543.503206018519", "[$-6C1A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0443\u0442."},
		{"43543.503206018519", "[$-6C1A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E\u0440\u0430\u043A"},
		{"44562.189571759256", "[$-1C1A]mmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1C1A]mmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1C1A]mmmmm dd yyyy  h:mm AM/PM", "\u0458 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1C1A]mmmmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-1C1A]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-1C1A]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E"},
		{"43543.503206018519", "[$-1C1A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0443\u0442\u043E"},
		{"43543.503206018519", "[$-1C1A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E\u0440\u0430\u043A"},
		{"44562.189571759256", "[$-301A]mmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-301A]mmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-301A]mmmmm dd yyyy  h:mm AM/PM", "\u0458 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-301A]mmmmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-301A]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-301A]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E"},
		{"43543.503206018519", "[$-301A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0443\u0442\u043E"},
		{"43543.503206018519", "[$-301A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E\u0440\u0430\u043A"},
		{"44562.189571759256", "[$-281A]mmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-281A]mmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-281A]mmmmm dd yyyy  h:mm AM/PM", "\u0458 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-281A]mmmmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-281A]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-281A]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442."},
		{"43543.503206018519", "[$-281A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0443\u0442."},
		{"43543.503206018519", "[$-281A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E\u0440\u0430\u043A"},
		{"44562.189571759256", "[$-C1A]mmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-C1A]mmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-C1A]mmmmm dd yyyy  h:mm AM/PM", "\u0458 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-C1A]mmmmmm dd yyyy  h:mm AM/PM", "\u0458\u0430\u043D\u0443\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-C1A]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-C1A]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442."},
		{"43543.503206018519", "[$-C1A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0443\u0442."},
		{"43543.503206018519", "[$-C1A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0443\u0442\u043E\u0440\u0430\u043A"},
		{"44562.189571759256", "[$-701A]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 pre podne"},
		{"44562.189571759256", "[$-701A]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 pre podne"},
		{"44562.189571759256", "[$-701A]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 pre podne"},
		{"44562.189571759256", "[$-701A]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 pre podne"},
		{"43543.503206018519", "[$-701A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 po podne"},
		{"43543.503206018519", "[$-701A]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 po podne uto"},
		{"43543.503206018519", "[$-701A]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 po podne uto"},
		{"43543.503206018519", "[$-701A]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 po podne utorak"},
		{"44562.189571759256", "[$-7C1A]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 pre podne"},
		{"44562.189571759256", "[$-7C1A]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 pre podne"},
		{"44562.189571759256", "[$-7C1A]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 pre podne"},
		{"44562.189571759256", "[$-7C1A]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 pre podne"},
		{"43543.503206018519", "[$-7C1A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 po podne"},
		{"43543.503206018519", "[$-7C1A]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 po podne uto"},
		{"43543.503206018519", "[$-7C1A]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 po podne uto"},
		{"43543.503206018519", "[$-7C1A]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 po podne utorak"},
		{"44562.189571759256", "[$-181A]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 prije podne"},
		{"44562.189571759256", "[$-181A]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 prije podne"},
		{"44562.189571759256", "[$-181A]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 prije podne"},
		{"44562.189571759256", "[$-181A]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 prije podne"},
		{"43543.503206018519", "[$-181A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 po podne"},
		{"43543.503206018519", "[$-181A]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 po podne uto"},
		{"43543.503206018519", "[$-181A]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 po podne uto"},
		{"43543.503206018519", "[$-181A]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 po podne utorak"},
		{"44562.189571759256", "[$-2C1A]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 prije podne"},
		{"44562.189571759256", "[$-2C1A]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 prije podne"},
		{"44562.189571759256", "[$-2C1A]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 prije podne"},
		{"44562.189571759256", "[$-2C1A]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 prije podne"},
		{"43543.503206018519", "[$-2C1A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 po podne"},
		{"43543.503206018519", "[$-2C1A]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 po podne uto"},
		{"43543.503206018519", "[$-2C1A]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 po podne uto"},
		{"43543.503206018519", "[$-2C1A]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 po podne utorak"},
		{"44562.189571759256", "[$-241A]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 pre podne"},
		{"44562.189571759256", "[$-241A]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 pre podne"},
		{"44562.189571759256", "[$-241A]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 pre podne"},
		{"44562.189571759256", "[$-241A]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 pre podne"},
		{"43543.503206018519", "[$-241A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 po podne"},
		{"43543.503206018519", "[$-241A]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 po podne uto"},
		{"43543.503206018519", "[$-241A]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 po podne uto"},
		{"43543.503206018519", "[$-241A]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 po podne utorak"},
		{"44562.189571759256", "[$-81A]mmm dd yyyy  h:mm AM/PM", "jan. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-81A]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-81A]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-81A]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-81A]mmm dd yyyy  h:mm AM/PM", "mart 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-81A]mmmm dd yyyy  h:mm AM/PM aaa", "mart 19 2019  12:04 PM uto."},
		{"43543.503206018519", "[$-81A]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM uto."},
		{"43543.503206018519", "[$-81A]mmmmmm dd yyyy  h:mm AM/PM dddd", "mart 19 2019  12:04 PM utorak"},
		{"44562.189571759256", "[$-6C]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6C]mmmm dd yyyy  h:mm AM/PM", "Janaware 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6C]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-6C]mmmmmm dd yyyy  h:mm AM/PM", "Janaware 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-6C]mmm dd yyyy  h:mm AM/PM", "Matš 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-6C]mmmm dd yyyy  h:mm AM/PM aaa", "Matšhe 19 2019  12:04 PM Lbb"},
		{"43543.503206018519", "[$-6C]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Lbb"},
		{"43543.503206018519", "[$-6C]mmmmmm dd yyyy  h:mm AM/PM dddd", "Matšhe 19 2019  12:04 PM Labobedi"},
		{"44562.189571759256", "[$-46C]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46C]mmmm dd yyyy  h:mm AM/PM", "Janaware 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46C]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-46C]mmmmmm dd yyyy  h:mm AM/PM", "Janaware 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-46C]mmm dd yyyy  h:mm AM/PM", "Matš 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-46C]mmmm dd yyyy  h:mm AM/PM aaa", "Matšhe 19 2019  12:04 PM Lbb"},
		{"43543.503206018519", "[$-46C]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Lbb"},
		{"43543.503206018519", "[$-46C]mmmmmm dd yyyy  h:mm AM/PM dddd", "Matšhe 19 2019  12:04 PM Labobedi"},
		{"44562.189571759256", "[$-32]mmm dd yyyy  h:mm AM/PM", "Fer. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-32]mmmm dd yyyy  h:mm AM/PM", "Ferikgong 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-32]mmmmm dd yyyy  h:mm AM/PM", "F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-32]mmmmmm dd yyyy  h:mm AM/PM", "Ferikgong 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-32]mmm dd yyyy  h:mm AM/PM", "Mop. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-32]mmmm dd yyyy  h:mm AM/PM aaa", "Mopitlwe 19 2019  12:04 PM Lab."},
		{"43543.503206018519", "[$-32]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Lab."},
		{"43543.503206018519", "[$-32]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mopitlwe 19 2019  12:04 PM Labobedi"},
		{"44562.189571759256", "[$-832]mmm dd yyyy  h:mm AM/PM", "Fer. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-832]mmmm dd yyyy  h:mm AM/PM", "Ferikgong 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-832]mmmmm dd yyyy  h:mm AM/PM", "F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-832]mmmmmm dd yyyy  h:mm AM/PM", "Ferikgong 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-832]mmm dd yyyy  h:mm AM/PM", "Mop. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-832]mmmm dd yyyy  h:mm AM/PM aaa", "Mopitlwe 19 2019  12:04 PM Lab."},
		{"43543.503206018519", "[$-832]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Lab."},
		{"43543.503206018519", "[$-832]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mopitlwe 19 2019  12:04 PM Labobedi"},
		{"44562.189571759256", "[$-432]mmm dd yyyy  h:mm AM/PM", "Fer. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-432]mmmm dd yyyy  h:mm AM/PM", "Ferikgong 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-432]mmmmm dd yyyy  h:mm AM/PM", "F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-432]mmmmmm dd yyyy  h:mm AM/PM", "Ferikgong 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-432]mmm dd yyyy  h:mm AM/PM", "Mop. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-432]mmmm dd yyyy  h:mm AM/PM aaa", "Mopitlwe 19 2019  12:04 PM Lab."},
		{"43543.503206018519", "[$-432]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Lab."},
		{"43543.503206018519", "[$-432]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mopitlwe 19 2019  12:04 PM Labobedi"},
		{"44562.189571759256", "[$-59]mmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u064A 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-59]mmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u064A 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-59]mmmmm dd yyyy  h:mm AM/PM", "\u062C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-59]mmmmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u064A 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-59]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-59]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0627\u0631"},
		{"43543.503206018519", "[$-59]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 PM \u0627\u0631"},
		{"43543.503206018519", "[$-59]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0627\u0631\u0628\u0639"},
		{"44562.189571759256", "[$-7C59]mmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u064A 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C59]mmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u064A 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C59]mmmmm dd yyyy  h:mm AM/PM", "\u062C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C59]mmmmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u064A 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C59]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C59]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0627\u0631"},
		{"43543.503206018519", "[$-7C59]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 PM \u0627\u0631"},
		{"43543.503206018519", "[$-7C59]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0627\u0631\u0628\u0639"},
		{"44562.189571759256", "[$-859]mmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u064A 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-859]mmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u064A 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-859]mmmmm dd yyyy  h:mm AM/PM", "\u062C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-859]mmmmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u064A 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-859]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-859]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0627\u0631"},
		{"43543.503206018519", "[$-859]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 PM \u0627\u0631"},
		{"43543.503206018519", "[$-859]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0627\u0631\u0628\u0639"},
		{"44562.189571759256", "[$-5B]mmm dd yyyy  h:mm AM/PM", "\u0DA2\u0DB1. 01 2022  4:32 \u0DB4\u0DD9.\u0DC0."},
		{"44562.189571759256", "[$-5B]mmmm dd yyyy  h:mm AM/PM", "\u0DA2\u0DB1\u0DC0\u0DCF\u0DBB\u0DD2 01 2022  4:32 \u0DB4\u0DD9.\u0DC0."},
		{"44562.189571759256", "[$-5B]mmmmm dd yyyy  h:mm AM/PM", "\u0DA2 01 2022  4:32 \u0DB4\u0DD9.\u0DC0."},
		{"44562.189571759256", "[$-5B]mmmmmm dd yyyy  h:mm AM/PM", "\u0DA2\u0DB1\u0DC0\u0DCF\u0DBB\u0DD2 01 2022  4:32 \u0DB4\u0DD9.\u0DC0."},
		{"43543.503206018519", "[$-5B]mmm dd yyyy  h:mm AM/PM", "\u0DB8\u0DCF\u0DBB\u0DCA\u0DAD\u0DD4. 19 2019  12:04 \u0DB4.\u0DC0."},
		{"43543.503206018519", "[$-5B]mmmm dd yyyy  h:mm AM/PM aaa", "\u0DB8\u0DCF\u0DBB\u0DCA\u0DAD\u0DD4 19 2019  12:04 \u0DB4.\u0DC0. \u0627\u0631"},
		{"43543.503206018519", "[$-5B]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0DB8 19 2019  12:04 \u0DB4.\u0DC0. \u0627\u0631"},
		{"43543.503206018519", "[$-5B]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0DB8\u0DCF\u0DBB\u0DCA\u0DAD\u0DD4 19 2019  12:04 \u0DB4.\u0DC0. \u0627\u0631\u0628\u0639"},
		{"44562.189571759256", "[$-45B]mmm dd yyyy  h:mm AM/PM", "\u0DA2\u0DB1. 01 2022  4:32 \u0DB4\u0DD9.\u0DC0."},
		{"44562.189571759256", "[$-45B]mmmm dd yyyy  h:mm AM/PM", "\u0DA2\u0DB1\u0DC0\u0DCF\u0DBB\u0DD2 01 2022  4:32 \u0DB4\u0DD9.\u0DC0."},
		{"44562.189571759256", "[$-45B]mmmmm dd yyyy  h:mm AM/PM", "\u0DA2 01 2022  4:32 \u0DB4\u0DD9.\u0DC0."},
		{"44562.189571759256", "[$-45B]mmmmmm dd yyyy  h:mm AM/PM", "\u0DA2\u0DB1\u0DC0\u0DCF\u0DBB\u0DD2 01 2022  4:32 \u0DB4\u0DD9.\u0DC0."},
		{"43543.503206018519", "[$-45B]mmm dd yyyy  h:mm AM/PM", "\u0DB8\u0DCF\u0DBB\u0DCA\u0DAD\u0DD4. 19 2019  12:04 \u0DB4.\u0DC0."},
		{"43543.503206018519", "[$-45B]mmmm dd yyyy  h:mm AM/PM aaa", "\u0DB8\u0DCF\u0DBB\u0DCA\u0DAD\u0DD4 19 2019  12:04 \u0DB4.\u0DC0. \u0627\u0631"},
		{"43543.503206018519", "[$-45B]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0DB8 19 2019  12:04 \u0DB4.\u0DC0. \u0627\u0631"},
		{"43543.503206018519", "[$-45B]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0DB8\u0DCF\u0DBB\u0DCA\u0DAD\u0DD4 19 2019  12:04 \u0DB4.\u0DC0. \u0627\u0631\u0628\u0639"},
		{"44562.189571759256", "[$-1B]mmm dd yyyy  h:mm AM/PM", "1 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1B]mmmm dd yyyy  h:mm AM/PM", "január 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1B]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1B]mmmmmm dd yyyy  h:mm AM/PM", "január 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-1B]mmm dd yyyy  h:mm AM/PM", "3 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-1B]mmmm dd yyyy  h:mm AM/PM aaa", "marec 19 2019  12:04 PM ut"},
		{"43543.503206018519", "[$-1B]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM ut"},
		{"43543.503206018519", "[$-1B]mmmmmm dd yyyy  h:mm AM/PM dddd", "marec 19 2019  12:04 PM utorok"},
		{"44562.189571759256", "[$-41B]mmm dd yyyy  h:mm AM/PM", "1 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41B]mmmm dd yyyy  h:mm AM/PM", "január 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41B]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41B]mmmmmm dd yyyy  h:mm AM/PM", "január 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-41B]mmm dd yyyy  h:mm AM/PM", "3 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-41B]mmmm dd yyyy  h:mm AM/PM aaa", "marec 19 2019  12:04 PM ut"},
		{"43543.503206018519", "[$-41B]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM ut"},
		{"43543.503206018519", "[$-41B]mmmmmm dd yyyy  h:mm AM/PM dddd", "marec 19 2019  12:04 PM utorok"},
		{"44562.189571759256", "[$-24]mmm dd yyyy  h:mm AM/PM", "jan. 01 2022  4:32 dop."},
		{"44562.189571759256", "[$-24]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 dop."},
		{"44562.189571759256", "[$-24]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 dop."},
		{"44562.189571759256", "[$-24]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 dop."},
		{"43543.503206018519", "[$-24]mmm dd yyyy  h:mm AM/PM", "mar. 19 2019  12:04 pop."},
		{"43543.503206018519", "[$-24]mmmm dd yyyy  h:mm AM/PM aaa", "marec 19 2019  12:04 pop. tor."},
		{"43543.503206018519", "[$-24]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 pop. tor."},
		{"43543.503206018519", "[$-24]mmmmmm dd yyyy  h:mm AM/PM dddd", "marec 19 2019  12:04 pop. torek"},
		{"44562.189571759256", "[$-424]mmm dd yyyy  h:mm AM/PM", "jan. 01 2022  4:32 dop."},
		{"44562.189571759256", "[$-424]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 dop."},
		{"44562.189571759256", "[$-424]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 dop."},
		{"44562.189571759256", "[$-424]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 dop."},
		{"43543.503206018519", "[$-424]mmm dd yyyy  h:mm AM/PM", "mar. 19 2019  12:04 pop."},
		{"43543.503206018519", "[$-424]mmmm dd yyyy  h:mm AM/PM aaa", "marec 19 2019  12:04 pop. tor."},
		{"43543.503206018519", "[$-424]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 pop. tor."},
		{"43543.503206018519", "[$-424]mmmmmm dd yyyy  h:mm AM/PM dddd", "marec 19 2019  12:04 pop. torek"},
		{"44562.189571759256", "[$-77]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 GH"},
		{"44562.189571759256", "[$-77]mmmm dd yyyy  h:mm AM/PM", "Jannaayo 01 2022  4:32 GH"},
		{"44562.189571759256", "[$-77]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 GH"},
		{"44562.189571759256", "[$-77]mmmmmm dd yyyy  h:mm AM/PM", "Jannaayo 01 2022  4:32 GH"},
		{"43543.503206018519", "[$-77]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 GD"},
		{"43543.503206018519", "[$-77]mmmm dd yyyy  h:mm AM/PM aaa", "Maarso 19 2019  12:04 GD Tldo"},
		{"43543.503206018519", "[$-77]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 GD Tldo"},
		{"43543.503206018519", "[$-77]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maarso 19 2019  12:04 GD Talaado"},
		{"44562.189571759256", "[$-477]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 GH"},
		{"44562.189571759256", "[$-477]mmmm dd yyyy  h:mm AM/PM", "Jannaayo 01 2022  4:32 GH"},
		{"44562.189571759256", "[$-477]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 GH"},
		{"44562.189571759256", "[$-477]mmmmmm dd yyyy  h:mm AM/PM", "Jannaayo 01 2022  4:32 GH"},
		{"43543.503206018519", "[$-477]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 GD"},
		{"43543.503206018519", "[$-477]mmmm dd yyyy  h:mm AM/PM aaa", "Maarso 19 2019  12:04 GD Tldo"},
		{"43543.503206018519", "[$-477]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 GD Tldo"},
		{"43543.503206018519", "[$-477]mmmmmm dd yyyy  h:mm AM/PM dddd", "Maarso 19 2019  12:04 GD Talaado"},
		{"44562.189571759256", "[$-30]mmm dd yyyy  h:mm AM/PM", "Phe 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-30]mmmm dd yyyy  h:mm AM/PM", "Phesekgong 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-30]mmmmm dd yyyy  h:mm AM/PM", "P 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-30]mmmmmm dd yyyy  h:mm AM/PM", "Phesekgong 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-30]mmm dd yyyy  h:mm AM/PM", "Ube 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-30]mmmm dd yyyy  h:mm AM/PM aaa", "Hlakubele 19 2019  12:04 PM Bed"},
		{"43543.503206018519", "[$-30]mmmmm dd yyyy  h:mm AM/PM ddd", "H 19 2019  12:04 PM Bed"},
		{"43543.503206018519", "[$-30]mmmmmm dd yyyy  h:mm AM/PM dddd", "Hlakubele 19 2019  12:04 PM Labobedi"},
		{"44562.189571759256", "[$-430]mmm dd yyyy  h:mm AM/PM", "Phe 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-430]mmmm dd yyyy  h:mm AM/PM", "Phesekgong 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-430]mmmmm dd yyyy  h:mm AM/PM", "P 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-430]mmmmmm dd yyyy  h:mm AM/PM", "Phesekgong 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-430]mmm dd yyyy  h:mm AM/PM", "Ube 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-430]mmmm dd yyyy  h:mm AM/PM aaa", "Hlakubele 19 2019  12:04 PM Bed"},
		{"43543.503206018519", "[$-430]mmmmm dd yyyy  h:mm AM/PM ddd", "H 19 2019  12:04 PM Bed"},
		{"43543.503206018519", "[$-430]mmmmmm dd yyyy  h:mm AM/PM dddd", "Hlakubele 19 2019  12:04 PM Labobedi"},
		{"44562.189571759256", "[$-A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a. m."},
		{"43543.503206018519", "[$-A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 p. m."},
		{"43543.503206018519", "[$-A]mmmm dd yyyy  h:mm AM/PM", "marzo 19 2019  12:04 p. m."},
		{"43543.503206018519", "[$-A]mmmmm dd yyyy  h:mm AM/PM", "m 19 2019  12:04 p. m."},
		{"44562.189571759256", "[$-A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a. m."},
		{"43543.503206018519", "[$-A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 p. m."},
		{"43543.503206018519", "[$-A]mmmm dd yyyy  h:mm AM/PM", "marzo 19 2019  12:04 p. m."},
		{"43543.503206018519", "[$-A]mmmmm dd yyyy  h:mm AM/PM", "m 19 2019  12:04 p. m."},
		{"44562.189571759256", "[$-A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a. m."},
		{"43543.503206018519", "[$-A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p. m. ma."},
		{"43543.503206018519", "[$-A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p. m. ma."},
		{"43543.503206018519", "[$-A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p. m. martes"},
		{"44562.189571759256", "[$-2C0A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-2C0A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-2C0A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-2C0A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-2C0A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-2C0A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"44562.189571759256", "[$-200A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-200A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-200A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-200A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-200A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-200A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"44562.189571759256", "[$-400A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-400A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-400A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-400A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-400A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-400A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"44562.189571759256", "[$-340A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-340A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-340A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-340A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-340A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-340A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"44562.189571759256", "[$-240A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-240A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-240A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-240A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-240A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-240A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"44562.189571759256", "[$-140A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-140A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-140A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-140A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-140A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-140A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"44562.189571759256", "[$-5C0A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-5C0A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a.m."},
		{"44562.189571759256", "[$-5C0A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a.m."},
		{"43543.503206018519", "[$-5C0A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-5C0A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-5C0A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.m. martes"},
		{"43543.503206018519", "[$-1C0A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-1C0A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-1C0A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"43543.503206018519", "[$-300A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-300A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-300A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"43543.503206018519", "[$-440A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-440A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-440A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"43543.503206018519", "[$-100A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-100A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-100A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"43543.503206018519", "[$-480A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-480A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-480A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"43543.503206018519", "[$-580A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-580A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.m. mar."},
		{"43543.503206018519", "[$-580A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.m. martes"},
		{"44562.189571759256", "[$-80A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-80A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-80A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a. m."},
		{"43543.503206018519", "[$-80A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 p. m."},
		{"43543.503206018519", "[$-80A]mmmm dd yyyy  h:mm AM/PM", "marzo 19 2019  12:04 p. m."},
		{"43543.503206018519", "[$-80A]mmmmm dd yyyy  h:mm AM/PM", "m 19 2019  12:04 p. m."},
		{"44562.189571759256", "[$-80A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-80A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-80A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a. m."},
		{"43543.503206018519", "[$-80A]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 p. m."},
		{"43543.503206018519", "[$-80A]mmmm dd yyyy  h:mm AM/PM", "marzo 19 2019  12:04 p. m."},
		{"43543.503206018519", "[$-80A]mmmmm dd yyyy  h:mm AM/PM", "m 19 2019  12:04 p. m."},
		{"44562.189571759256", "[$-80A]mmm dd yyyy  h:mm AM/PM", "ene 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-80A]mmmm dd yyyy  h:mm AM/PM", "enero 01 2022  4:32 a. m."},
		{"44562.189571759256", "[$-80A]mmmmm dd yyyy  h:mm AM/PM", "e 01 2022  4:32 a. m."},
		{"43543.503206018519", "[$-80A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p. m. mar."},
		{"43543.503206018519", "[$-80A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p. m. mar."},
		{"43543.503206018519", "[$-80A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p. m. martes"},
		{"43543.503206018519", "[$-4C0A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-4C0A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-4C0A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"43543.503206018519", "[$-180A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-180A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-180A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"43543.503206018519", "[$-3C0A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-3C0A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-3C0A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"44562.189571759256", "[$-280A]mmm dd yyyy  h:mm AM/PM", "Ene. 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-280A]mmmm dd yyyy  h:mm AM/PM", "Enero 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-280A]mmmmm dd yyyy  h:mm AM/PM", "E 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-280A]mmmmmm dd yyyy  h:mm AM/PM", "Enero 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-280A]mmm dd yyyy  h:mm AM/PM", "Mar. 19 2019  12:04 p.%A0m."},
		{"43543.503206018519", "[$-280A]mmmm dd yyyy  h:mm AM/PM aaa", "Marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-280A]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-280A]mmmmmm dd yyyy  h:mm AM/PM dddd", "Marzo 19 2019  12:04 p.%A0m. martes"},
		{"43543.503206018519", "[$-500A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-500A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-500A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 p.%A0m. martes"},
		{"43543.503206018519", "[$-40A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 PM ma."},
		{"43543.503206018519", "[$-40A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 PM ma."},
		{"43543.503206018519", "[$-40A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM martes"},
		{"43543.503206018519", "[$-C0A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 PM ma."},
		{"43543.503206018519", "[$-C0A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 PM ma."},
		{"43543.503206018519", "[$-C0A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM martes"},
		{"43543.503206018519", "[$-540A]mmm dd yyyy  h:mm AM/PM aaa", "mar 19 2019  12:04 PM mar"},
		{"43543.503206018519", "[$-540A]mmmm dd yyyy  h:mm AM/PM ddd", "marzo 19 2019  12:04 PM mar"},
		{"43543.503206018519", "[$-540A]mmmmm dd yyyy  h:mm AM/PM dddd", "m 19 2019  12:04 PM martes"},
		{"44562.189571759256", "[$-380A]mmm dd yyyy  h:mm AM/PM", "Ene. 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-380A]mmmm dd yyyy  h:mm AM/PM", "Enero 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-380A]mmmmm dd yyyy  h:mm AM/PM", "E 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-380A]mmmmmm dd yyyy  h:mm AM/PM", "Enero 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-380A]mmm dd yyyy  h:mm AM/PM", "Mar. 19 2019  12:04 p.%A0m."},
		{"43543.503206018519", "[$-380A]mmmm dd yyyy  h:mm AM/PM aaa", "Marzo 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-380A]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 p.%A0m. mar."},
		{"43543.503206018519", "[$-380A]mmmmmm dd yyyy  h:mm AM/PM dddd", "Marzo 19 2019  12:04 p.%A0m. martes"},
		{"44562.189571759256", "[$-1D]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1D]mmmm dd yyyy  h:mm AM/PM", "januari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1D]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1D]mmmmmm dd yyyy  h:mm AM/PM", "januari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-1D]mmm dd yyyy  h:mm AM/PM", "mar 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-1D]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM tis"},
		{"43543.503206018519", "[$-1D]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM tis"},
		{"43543.503206018519", "[$-1D]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 PM tisdag"},
		{"44562.189571759256", "[$-81D]mmm dd yyyy  h:mm AM/PM", "jan. 01 2022  4:32 fm"},
		{"44562.189571759256", "[$-81D]mmmm dd yyyy  h:mm AM/PM", "januari 01 2022  4:32 fm"},
		{"44562.189571759256", "[$-81D]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 fm"},
		{"44562.189571759256", "[$-81D]mmmmmm dd yyyy  h:mm AM/PM", "januari 01 2022  4:32 fm"},
		{"43543.503206018519", "[$-81D]mmm dd yyyy  h:mm AM/PM", "mars 19 2019  12:04 em"},
		{"43543.503206018519", "[$-81D]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 em tis"},
		{"43543.503206018519", "[$-81D]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 em tis"},
		{"43543.503206018519", "[$-81D]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 em tisdag"},
		{"44562.189571759256", "[$-41D]mmm dd yyyy  h:mm AM/PM", "jan. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41D]mmmm dd yyyy  h:mm AM/PM", "januari 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41D]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41D]mmmmmm dd yyyy  h:mm AM/PM", "januari 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-41D]mmm dd yyyy  h:mm AM/PM", "mars 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-41D]mmmm dd yyyy  h:mm AM/PM aaa", "mars 19 2019  12:04 PM tis"},
		{"43543.503206018519", "[$-41D]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 PM tis"},
		{"43543.503206018519", "[$-41D]mmmmmm dd yyyy  h:mm AM/PM dddd", "mars 19 2019  12:04 PM tisdag"},
		{"44562.189571759256", "[$-5A]mmm dd yyyy  h:mm AM/PM", "\u071F\u0722%A0\u070F\u0712 01 2022  4:32 \u0729.\u071B"},
		{"44562.189571759256", "[$-5A]mmmm dd yyyy  h:mm AM/PM", "\u071F\u0722\u0718\u0722%A0\u0710\u071A\u072A\u071D 01 2022  4:32 \u0729.\u071B"},
		{"44562.189571759256", "[$-5A]mmmmm dd yyyy  h:mm AM/PM", "\u071F 01 2022  4:32 \u0729.\u071B"},
		{"44562.189571759256", "[$-5A]mmmmmm dd yyyy  h:mm AM/PM", "\u071F\u0722\u0718\u0722%A0\u0710\u071A\u072A\u071D 01 2022  4:32 \u0729.\u071B"},
		{"43543.503206018519", "[$-5A]mmm dd yyyy  h:mm AM/PM", "\u0710\u0715\u072A 19 2019  12:04 \u0712.\u071B"},
		{"43543.503206018519", "[$-5A]mmmm dd yyyy  h:mm AM/PM aaa", "\u0710\u0715\u072A 19 2019  12:04 \u0712.\u071B \u070F\u0713%A0\u070F\u0712\u072B"},
		{"43543.503206018519", "[$-5A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0710 19 2019  12:04 \u0712.\u071B \u070F\u0713%A0\u070F\u0712\u072B"},
		{"43543.503206018519", "[$-5A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0710\u0715\u072A 19 2019  12:04 \u0712.\u071B \u072C\u0720\u072C\u0710%A0\u0712\u072B\u0712\u0710"},
		{"44562.189571759256", "[$-45A]mmm dd yyyy  h:mm AM/PM", "\u071F\u0722%A0\u070F\u0712 01 2022  4:32 \u0729.\u071B"},
		{"44562.189571759256", "[$-45A]mmmm dd yyyy  h:mm AM/PM", "\u071F\u0722\u0718\u0722%A0\u0710\u071A\u072A\u071D 01 2022  4:32 \u0729.\u071B"},
		{"44562.189571759256", "[$-45A]mmmmm dd yyyy  h:mm AM/PM", "\u071F 01 2022  4:32 \u0729.\u071B"},
		{"44562.189571759256", "[$-45A]mmmmmm dd yyyy  h:mm AM/PM", "\u071F\u0722\u0718\u0722%A0\u0710\u071A\u072A\u071D 01 2022  4:32 \u0729.\u071B"},
		{"43543.503206018519", "[$-45A]mmm dd yyyy  h:mm AM/PM", "\u0710\u0715\u072A 19 2019  12:04 \u0712.\u071B"},
		{"43543.503206018519", "[$-45A]mmmm dd yyyy  h:mm AM/PM aaa", "\u0710\u0715\u072A 19 2019  12:04 \u0712.\u071B \u070F\u0713%A0\u070F\u0712\u072B"},
		{"43543.503206018519", "[$-45A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0710 19 2019  12:04 \u0712.\u071B \u070F\u0713%A0\u070F\u0712\u072B"},
		{"43543.503206018519", "[$-45A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0710\u0715\u072A 19 2019  12:04 \u0712.\u071B \u072C\u0720\u072C\u0710%A0\u0712\u072B\u0712\u0710"},
		{"44562.189571759256", "[$-28]mmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-28]mmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-28]mmmmm dd yyyy  h:mm AM/PM", "\u044F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-28]mmmmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-28]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-28]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0441\u0448\u0431"},
		{"43543.503206018519", "[$-28]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0441\u0448\u0431"},
		{"43543.503206018519", "[$-28]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0441\u0435\u0448\u0430\u043D\u0431\u0435"},
		{"44562.189571759256", "[$-7C28]mmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C28]mmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C28]mmmmm dd yyyy  h:mm AM/PM", "\u044F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C28]mmmmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C28]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C28]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0441\u0448\u0431"},
		{"43543.503206018519", "[$-7C28]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0441\u0448\u0431"},
		{"43543.503206018519", "[$-7C28]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0441\u0435\u0448\u0430\u043D\u0431\u0435"},
		{"44562.189571759256", "[$-428]mmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-428]mmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-428]mmmmm dd yyyy  h:mm AM/PM", "\u044F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-428]mmmmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-428]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-428]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0441\u0448\u0431"},
		{"43543.503206018519", "[$-428]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0441\u0448\u0431"},
		{"43543.503206018519", "[$-428]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0441\u0435\u0448\u0430\u043D\u0431\u0435"},
		{"44562.189571759256", "[$-5F]mmm dd yyyy  h:mm AM/PM", "Yen 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-5F]mmmm dd yyyy  h:mm AM/PM", "Yennayer 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-5F]mmmmm dd yyyy  h:mm AM/PM", "Y 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-5F]mmmmmm dd yyyy  h:mm AM/PM", "Yennayer 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-5F]mmm dd yyyy  h:mm AM/PM", "Megh 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-5F]mmmm dd yyyy  h:mm AM/PM aaa", "Meghres 19 2019  12:04 PM ttl"},
		{"43543.503206018519", "[$-5F]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM ttl"},
		{"43543.503206018519", "[$-5F]mmmmmm dd yyyy  h:mm AM/PM dddd", "Meghres 19 2019  12:04 PM ttlata"},
		{"44562.189571759256", "[$-7C5F]mmm dd yyyy  h:mm AM/PM", "Yen 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C5F]mmmm dd yyyy  h:mm AM/PM", "Yennayer 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C5F]mmmmm dd yyyy  h:mm AM/PM", "Y 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-7C5F]mmmmmm dd yyyy  h:mm AM/PM", "Yennayer 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-7C5F]mmm dd yyyy  h:mm AM/PM", "Megh 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-7C5F]mmmm dd yyyy  h:mm AM/PM aaa", "Meghres 19 2019  12:04 PM ttl"},
		{"43543.503206018519", "[$-7C5F]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM ttl"},
		{"43543.503206018519", "[$-7C5F]mmmmmm dd yyyy  h:mm AM/PM dddd", "Meghres 19 2019  12:04 PM ttlata"},
		{"44562.189571759256", "[$-85F]mmm dd yyyy  h:mm AM/PM", "Yen 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-85F]mmmm dd yyyy  h:mm AM/PM", "Yennayer 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-85F]mmmmm dd yyyy  h:mm AM/PM", "Y 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-85F]mmmmmm dd yyyy  h:mm AM/PM", "Yennayer 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-85F]mmm dd yyyy  h:mm AM/PM", "Megh 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-85F]mmmm dd yyyy  h:mm AM/PM aaa", "Meghres 19 2019  12:04 PM ttl"},
		{"43543.503206018519", "[$-85F]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM ttl"},
		{"43543.503206018519", "[$-85F]mmmmmm dd yyyy  h:mm AM/PM dddd", "Meghres 19 2019  12:04 PM ttlata"},
		{"44562.189571759256", "[$-49]mmm dd yyyy  h:mm AM/PM", "\u0B9C\u0BA9\u0BB5\u0BB0\u0BBF 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"44562.189571759256", "[$-49]mmmm dd yyyy  h:mm AM/PM", "\u0B9C\u0BA9\u0BB5\u0BB0\u0BBF 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"44562.189571759256", "[$-49]mmmmm dd yyyy  h:mm AM/PM", "\u0B9C 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"44562.189571759256", "[$-49]mmmmmm dd yyyy  h:mm AM/PM", "\u0B9C\u0BA9\u0BB5\u0BB0\u0BBF 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"43543.503206018519", "[$-49]mmm dd yyyy  h:mm AM/PM", "\u0BAE\u0BBE\u0BB0\u0BCD\u0B9A\u0BCD 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8"},
		{"43543.503206018519", "[$-49]mmmm dd yyyy  h:mm AM/PM aaa", "\u0BAE\u0BBE\u0BB0\u0BCD\u0B9A\u0BCD 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8 \u0B9A\u0BC6\u0BB5\u0BCD\u0BB5\u0BBE\u0BAF\u0BCD"},
		{"43543.503206018519", "[$-49]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0BAE 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8 \u0B9A\u0BC6\u0BB5\u0BCD\u0BB5\u0BBE\u0BAF\u0BCD"},
		{"43543.503206018519", "[$-49]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0BAE\u0BBE\u0BB0\u0BCD\u0B9A\u0BCD 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8 \u0B9A\u0BC6\u0BB5\u0BCD\u0BB5\u0BBE\u0BAF\u0BCD\u0B95\u0BCD\u0B95\u0BBF\u0BB4\u0BAE\u0BC8"},
		{"44562.189571759256", "[$-449]mmm dd yyyy  h:mm AM/PM", "\u0B9C\u0BA9\u0BB5\u0BB0\u0BBF 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"44562.189571759256", "[$-449]mmmm dd yyyy  h:mm AM/PM", "\u0B9C\u0BA9\u0BB5\u0BB0\u0BBF 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"44562.189571759256", "[$-449]mmmmm dd yyyy  h:mm AM/PM", "\u0B9C 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"44562.189571759256", "[$-449]mmmmmm dd yyyy  h:mm AM/PM", "\u0B9C\u0BA9\u0BB5\u0BB0\u0BBF 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"43543.503206018519", "[$-449]mmm dd yyyy  h:mm AM/PM", "\u0BAE\u0BBE\u0BB0\u0BCD\u0B9A\u0BCD 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8"},
		{"43543.503206018519", "[$-449]mmmm dd yyyy  h:mm AM/PM aaa", "\u0BAE\u0BBE\u0BB0\u0BCD\u0B9A\u0BCD 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8 \u0B9A\u0BC6\u0BB5\u0BCD\u0BB5\u0BBE\u0BAF\u0BCD"},
		{"43543.503206018519", "[$-449]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0BAE 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8 \u0B9A\u0BC6\u0BB5\u0BCD\u0BB5\u0BBE\u0BAF\u0BCD"},
		{"43543.503206018519", "[$-449]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0BAE\u0BBE\u0BB0\u0BCD\u0B9A\u0BCD 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8 \u0B9A\u0BC6\u0BB5\u0BCD\u0BB5\u0BBE\u0BAF\u0BCD\u0B95\u0BCD\u0B95\u0BBF\u0BB4\u0BAE\u0BC8"},
		{"44562.189571759256", "[$-849]mmm dd yyyy  h:mm AM/PM", "\u0B9C\u0BA9. 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"44562.189571759256", "[$-849]mmmm dd yyyy  h:mm AM/PM", "\u0B9C\u0BA9\u0BB5\u0BB0\u0BBF 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"44562.189571759256", "[$-849]mmmmm dd yyyy  h:mm AM/PM", "\u0B9C 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"44562.189571759256", "[$-849]mmmmmm dd yyyy  h:mm AM/PM", "\u0B9C\u0BA9\u0BB5\u0BB0\u0BBF 01 2022  4:32 \u0B95\u0BBE\u0BB2\u0BC8"},
		{"43543.503206018519", "[$-849]mmm dd yyyy  h:mm AM/PM", "\u0BAE\u0BBE\u0BB0\u0BCD. 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8"},
		{"43543.503206018519", "[$-849]mmmm dd yyyy  h:mm AM/PM aaa", "\u0BAE\u0BBE\u0BB0\u0BCD\u0B9A\u0BCD 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8 \u0B9A\u0BC6\u0BB5\u0BCD."},
		{"43543.503206018519", "[$-849]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0BAE 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8 \u0B9A\u0BC6\u0BB5\u0BCD."},
		{"43543.503206018519", "[$-849]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0BAE\u0BBE\u0BB0\u0BCD\u0B9A\u0BCD 19 2019  12:04 \u0BAE\u0BBE\u0BB2\u0BC8 \u0B9A\u0BC6\u0BB5\u0BCD\u0BB5\u0BBE\u0BAF\u0BCD"},
		{"44562.189571759256", "[$-44]mmm dd yyyy  h:mm AM/PM", "\u0433\u044B\u0439\u043D. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-44]mmmm dd yyyy  h:mm AM/PM", "\u0433\u044B\u0439\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-44]mmmmm dd yyyy  h:mm AM/PM", "\u0433 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-44]mmmmmm dd yyyy  h:mm AM/PM", "\u0433\u044B\u0439\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-44]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-44]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0441\u0438\u0448."},
		{"43543.503206018519", "[$-44]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0441\u0438\u0448."},
		{"43543.503206018519", "[$-44]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0441\u0438\u0448\u04D9\u043C\u0431\u0435"},
		{"44562.189571759256", "[$-444]mmm dd yyyy  h:mm AM/PM", "\u0433\u044B\u0439\u043D. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-444]mmmm dd yyyy  h:mm AM/PM", "\u0433\u044B\u0439\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-444]mmmmm dd yyyy  h:mm AM/PM", "\u0433 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-444]mmmmmm dd yyyy  h:mm AM/PM", "\u0433\u044B\u0439\u043D\u0432\u0430\u0440 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-444]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440. 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-444]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0441\u0438\u0448."},
		{"43543.503206018519", "[$-444]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 PM \u0441\u0438\u0448."},
		{"43543.503206018519", "[$-444]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 PM \u0441\u0438\u0448\u04D9\u043C\u0431\u0435"},
		{"44562.189571759256", "[$-4A]mmm dd yyyy  h:mm AM/PM", "\u0C1C\u0C28 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-4A]mmmm dd yyyy  h:mm AM/PM", "\u0C1C\u0C28\u0C35\u0C30\u0C3F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-4A]mmmmm dd yyyy  h:mm AM/PM", "\u0C1C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-4A]mmmmmm dd yyyy  h:mm AM/PM", "\u0C1C\u0C28\u0C35\u0C30\u0C3F 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-4A]mmm dd yyyy  h:mm AM/PM", "\u0C2E\u0C3E\u0C30\u0C4D\u0C1A\u0C3F 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-4A]mmmm dd yyyy  h:mm AM/PM aaa", "\u0C2E\u0C3E\u0C30\u0C4D\u0C1A\u0C3F 19 2019  12:04 PM \u0C2E\u0C02\u0C17\u0C33"},
		{"43543.503206018519", "[$-4A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0C2E 19 2019  12:04 PM \u0C2E\u0C02\u0C17\u0C33"},
		{"43543.503206018519", "[$-4A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0C2E\u0C3E\u0C30\u0C4D\u0C1A\u0C3F 19 2019  12:04 PM \u0C2E\u0C02\u0C17\u0C33\u0C35\u0C3E\u0C30\u0C02"},
		{"44562.189571759256", "[$-44A]mmm dd yyyy  h:mm AM/PM", "\u0C1C\u0C28 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-44A]mmmm dd yyyy  h:mm AM/PM", "\u0C1C\u0C28\u0C35\u0C30\u0C3F 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-44A]mmmmm dd yyyy  h:mm AM/PM", "\u0C1C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-44A]mmmmmm dd yyyy  h:mm AM/PM", "\u0C1C\u0C28\u0C35\u0C30\u0C3F 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-44A]mmm dd yyyy  h:mm AM/PM", "\u0C2E\u0C3E\u0C30\u0C4D\u0C1A\u0C3F 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-44A]mmmm dd yyyy  h:mm AM/PM aaa", "\u0C2E\u0C3E\u0C30\u0C4D\u0C1A\u0C3F 19 2019  12:04 PM \u0C2E\u0C02\u0C17\u0C33"},
		{"43543.503206018519", "[$-44A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0C2E 19 2019  12:04 PM \u0C2E\u0C02\u0C17\u0C33"},
		{"43543.503206018519", "[$-44A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0C2E\u0C3E\u0C30\u0C4D\u0C1A\u0C3F 19 2019  12:04 PM \u0C2E\u0C02\u0C17\u0C33\u0C35\u0C3E\u0C30\u0C02"},
		{"44562.189571759256", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e21.\u0e04. 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e01.\u0e18. 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e21.\u0e04. 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e40.\u0e22. 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e1e.\u0e04. 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e21.\u0e22. 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e01.\u0e04. 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e2a.\u0e04. 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e01.\u0e22. 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e15.\u0e04. 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e1e.\u0e22. 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-1E]mmm dd yyyy  h:mm AM/PM", "\u0e18.\u0e04. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e21\u0e01\u0e23\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e01\u0e38\u0e21\u0e20\u0e32\u0e1e\u0e31\u0e19\u0e18\u0e4c 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e21\u0e35\u0e19\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e40\u0e21\u0e29\u0e32\u0e22\u0e19 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e1e\u0e24\u0e29\u0e20\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e21\u0e34\u0e16\u0e38\u0e19\u0e32\u0e22\u0e19 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e01\u0e23\u0e01\u0e0e\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e2a\u0e34\u0e07\u0e2b\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e01\u0e31\u0e19\u0e22\u0e32\u0e22\u0e19 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e15\u0e38\u0e25\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e1e\u0e24\u0e28\u0e08\u0e34\u0e01\u0e32\u0e22\u0e19 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-1E]mmmm dd yyyy  h:mm AM/PM", "\u0e18\u0e31\u0e19\u0e27\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-1E]mmmmm dd yyyy  h:mm AM/PM", "\u0e21 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-1E]mmmmm dd yyyy  h:mm AM/PM", "\u0e01 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-1E]mmmmm dd yyyy  h:mm AM/PM", "\u0e21 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-1E]mmmmm dd yyyy  h:mm AM/PM", "\u0e40 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-1E]mmmmm dd yyyy  h:mm AM/PM", "\u0e1e 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-1E]mmmmm dd yyyy  h:mm AM/PM", "\u0e21 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-1E]mmmmm dd yyyy  h:mm AM/PM", "\u0e01 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-1E]mmmmm dd yyyy  h:mm AM/PM", "\u0e2a 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-1E]mmmmm dd yyyy  h:mm AM/PM", "\u0e01 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-1E]mmmmm dd yyyy  h:mm AM/PM aaa", "\u0e15 01 2022  4:32 AM \u0E2A."},
		{"44866.18957170139", "[$-1E]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0e1e 01 2022  4:32 AM \u0E2D."},
		{"44896.18957170139", "[$-1E]mmmmm dd yyyy  h:mm AM/PM dddd", "\u0e18 01 2022  4:32 AM \u0E1E\u0E24\u0E2B\u0E31\u0E2A\u0E1A\u0E14\u0E35"},
		{"44562.189571759256", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e21.\u0e04. 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e01.\u0e18. 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e21.\u0e04. 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e40.\u0e22. 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e1e.\u0e04. 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e21.\u0e22. 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e01.\u0e04. 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e2a.\u0e04. 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e01.\u0e22. 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e15.\u0e04. 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e1e.\u0e22. 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-41E]mmm dd yyyy  h:mm AM/PM", "\u0e18.\u0e04. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e21\u0e01\u0e23\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e01\u0e38\u0e21\u0e20\u0e32\u0e1e\u0e31\u0e19\u0e18\u0e4c 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e21\u0e35\u0e19\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e40\u0e21\u0e29\u0e32\u0e22\u0e19 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e1e\u0e24\u0e29\u0e20\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e21\u0e34\u0e16\u0e38\u0e19\u0e32\u0e22\u0e19 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e01\u0e23\u0e01\u0e0e\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e2a\u0e34\u0e07\u0e2b\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e01\u0e31\u0e19\u0e22\u0e32\u0e22\u0e19 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e15\u0e38\u0e25\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e1e\u0e24\u0e28\u0e08\u0e34\u0e01\u0e32\u0e22\u0e19 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-41E]mmmm dd yyyy  h:mm AM/PM", "\u0e18\u0e31\u0e19\u0e27\u0e32\u0e04\u0e21 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-41E]mmmmm dd yyyy  h:mm AM/PM", "\u0e21 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-41E]mmmmm dd yyyy  h:mm AM/PM", "\u0e01 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-41E]mmmmm dd yyyy  h:mm AM/PM", "\u0e21 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-41E]mmmmm dd yyyy  h:mm AM/PM", "\u0e40 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-41E]mmmmm dd yyyy  h:mm AM/PM", "\u0e1e 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-41E]mmmmm dd yyyy  h:mm AM/PM", "\u0e21 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-41E]mmmmm dd yyyy  h:mm AM/PM", "\u0e01 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-41E]mmmmm dd yyyy  h:mm AM/PM", "\u0e2a 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-41E]mmmmm dd yyyy  h:mm AM/PM", "\u0e01 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-41E]mmmmm dd yyyy  h:mm AM/PM aaa", "\u0e15 01 2022  4:32 AM \u0E2A."},
		{"44866.18957170139", "[$-41E]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0e1e 01 2022  4:32 AM \u0E2D."},
		{"44896.18957170139", "[$-41E]mmmmm dd yyyy  h:mm AM/PM dddd", "\u0e18 01 2022  4:32 AM \u0E1E\u0E24\u0E2B\u0E31\u0E2A\u0E1A\u0E14\u0E35"},
		{"100", "[$-411]ge\"年\"m\"月\"d\"日\";@", "1900年4月9日"},
		{"43709", "[$-411]ge\"年\"m\"月\"d\"日\";@", "R1年9月1日"},
		{"43709", "[$-411]gge\"年\"m\"月\"d\"日\";@", "\u4EE41年9月1日"},
		{"43709", "[$-411]ggge\"年\"m\"月\"d\"日\";@", "\u4EE4\u548C1年9月1日"},
		{"43709", "[$-411]gee\"年\"m\"月\"d\"日\";@", "R01年9月1日"},
		{"43709", "[$-411]ggee\"年\"m\"月\"d\"日\";@", "\u4EE401年9月1日"},
		{"43709", "[$-411]gggee\"年\"m\"月\"d\"日\";@", "\u4EE4\u548C01年9月1日"},
		{"43709", "[$-ja-JP-x-gannen]ge\"年\"m\"月\"d\"日\";@", "R1年9月1日"},
		{"43709", "[$-ja-JP-x-gannen]gge\"年\"m\"月\"d\"日\";@", "\u4EE41年9月1日"},
		{"43709", "[$-ja-JP-x-gannen]ggge\"年\"m\"月\"d\"日\";@", "\u4EE4\u548C1年9月1日"},
		{"43709", "[$-ja-JP-x-gannen]gee\"年\"m\"月\"d\"日\";@", "R01年9月1日"},
		{"43709", "[$-ja-JP-x-gannen]ggee\"年\"m\"月\"d\"日\";@", "\u4EE401年9月1日"},
		{"43709", "[$-ja-JP-x-gannen]gggee\"年\"m\"月\"d\"日\";@", "\u4EE4\u548C01年9月1日"},
		{"43709", "[$-ja-JP-x-gannen,80]ge\"年\"m\"月\"d\"日\";@", "R1年9月1日"},
		{"43709", "[$-ja-JP-x-gannen,80]gge\"年\"m\"月\"d\"日\";@", "\u4EE4\u5143年9月1日"},
		{"43709", "[$-ja-JP-x-gannen,80]ggge\"年\"m\"月\"d\"日\";@", "\u4EE4\u548C\u5143年9月1日"},
		{"43709", "[$-ja-JP-x-gannen,80]gee\"年\"m\"月\"d\"日\";@", "R01年9月1日"},
		{"43709", "[$-ja-JP-x-gannen,80]ggee\"年\"m\"月\"d\"日\";@", "\u4EE4\u5143年9月1日"},
		{"43709", "[$-ja-JP-x-gannen,80]gggee\"年\"m\"月\"d\"日\";@", "\u4EE4\u548C\u5143年9月1日"},
		{"43466.189571759256", "[$-411]ge\"年\"m\"月\"d\"日\";@", "H31年1月1日"},
		{"43466.189571759256", "[$-411]gge\"年\"m\"月\"d\"日\";@", "\u5E7331年1月1日"},
		{"43466.189571759256", "[$-411]ggge\"年\"m\"月\"d\"日\";@", "\u5E73\u621031年1月1日"},
		{"44896.18957170139", "[$-411]ge\"年\"m\"月\"d\"日\";@", "R4年12月1日"},
		{"44896.18957170139", "[$-411]gge\"年\"m\"月\"d\"日\";@", "\u4EE44年12月1日"},
		{"44896.18957170139", "[$-411]ggge\"年\"m\"月\"d\"日\";@", "\u4EE4\u548C4年12月1日"},
		{"44562.189571759256", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f21 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44593.189571759256", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f22 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44621.18957170139", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f23 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44652.18957170139", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f24 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44682.18957170139", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f25 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44713.18957170139", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f26 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44743.18957170139", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f27 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44774.18957170139", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f28 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44805.18957170139", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f29 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44835.18957170139", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f21\u0f20 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44866.18957170139", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f21\u0f21 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44896.18957170139", "[$-51]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f21\u0f22 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44562.189571759256", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f51\u0f44\u0f0b\u0f54\u0f7c\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44593.189571759256", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f42\u0f49\u0f72\u0f66\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44621.18957170139", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f42\u0f66\u0f74\u0f58\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44652.18957170139", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f56\u0f5e\u0f72\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44682.18957170139", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f63\u0f94\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44713.18957170139", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f51\u0fb2\u0f74\u0f42\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44743.18957170139", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f56\u0f51\u0f74\u0f53\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44774.18957170139", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f56\u0f62\u0f92\u0fb1\u0f51\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44805.18957170139", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f51\u0f42\u0f74\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44835.18957170139", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f66\u0fa4\u0fb1\u0f72\u0f0b\u0f5f\u0fb3\u0f0b\u0f56\u0f45\u0f74\u0f0b\u0f54\u0f0d 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44866.18957170139", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f56\u0f45\u0f74\u0f0b\u0f42\u0f45\u0f72\u0f42\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44896.18957170139", "[$-51]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f56\u0f45\u0f74\u0f0b\u0f42\u0f49\u0f72\u0f66\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44562.189571759256", "[$-51]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44593.189571759256", "[$-51]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44621.18957170139", "[$-51]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44652.18957170139", "[$-51]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44682.18957170139", "[$-51]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44713.18957170139", "[$-51]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44743.18957170139", "[$-51]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44774.18957170139", "[$-51]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44805.18957170139", "[$-51]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44835.18957170139", "[$-51]mmmmm dd yyyy  h:mm AM/PM aaa", "\u0f66 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b \u0F66\u0FA4\u0F7A\u0F53\u0F0B\u0F54\u0F0D"},
		{"44866.18957170139", "[$-51]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b \u0F58\u0F72\u0F42\u0F0B\u0F51\u0F58\u0F62\u0F0D"},
		{"44896.18957170139", "[$-51]mmmmm dd yyyy  h:mm AM/PM dddd", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b \u0F42\u0F5F\u0F60\u0F0B\u0F55\u0F74\u0F62\u0F0B\u0F56\u0F74\u0F0D"},
		{"44562.189571759256", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f21 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44593.189571759256", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f22 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44621.18957170139", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f23 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44652.18957170139", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f24 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44682.18957170139", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f25 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44713.18957170139", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f26 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44743.18957170139", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f27 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44774.18957170139", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f28 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44805.18957170139", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f29 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44835.18957170139", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f21\u0f20 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44866.18957170139", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f21\u0f21 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44896.18957170139", "[$-451]mmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f21\u0f22 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44562.189571759256", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f51\u0f44\u0f0b\u0f54\u0f7c\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44593.189571759256", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f42\u0f49\u0f72\u0f66\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44621.18957170139", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f42\u0f66\u0f74\u0f58\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44652.18957170139", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f56\u0f5e\u0f72\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44682.18957170139", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f63\u0f94\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44713.18957170139", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f51\u0fb2\u0f74\u0f42\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44743.18957170139", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f56\u0f51\u0f74\u0f53\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44774.18957170139", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f56\u0f62\u0f92\u0fb1\u0f51\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44805.18957170139", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f51\u0f42\u0f74\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44835.18957170139", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f66\u0fa4\u0fb1\u0f72\u0f0b\u0f5f\u0fb3\u0f0b\u0f56\u0f45\u0f74\u0f0b\u0f54\u0f0d 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44866.18957170139", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f56\u0f45\u0f74\u0f0b\u0f42\u0f45\u0f72\u0f42\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44896.18957170139", "[$-451]mmmm dd yyyy  h:mm AM/PM", "\u0f5f\u0fb3\u0f0b\u0f56\u0f0b\u0f56\u0f45\u0f74\u0f0b\u0f42\u0f49\u0f72\u0f66\u0f0b\u0f54\u0f0b 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44562.189571759256", "[$-451]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44593.189571759256", "[$-451]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44621.18957170139", "[$-451]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44652.18957170139", "[$-451]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44682.18957170139", "[$-451]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44713.18957170139", "[$-451]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44743.18957170139", "[$-451]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44774.18957170139", "[$-451]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44805.18957170139", "[$-451]mmmmm dd yyyy  h:mm AM/PM", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b"},
		{"44835.18957170139", "[$-451]mmmmm dd yyyy  h:mm AM/PM aaa", "\u0f66 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b \u0F66\u0FA4\u0F7A\u0F53\u0F0B\u0F54\u0F0D"},
		{"44866.18957170139", "[$-451]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b \u0F58\u0F72\u0F42\u0F0B\u0F51\u0F58\u0F62\u0F0D"},
		{"44896.18957170139", "[$-451]mmmmm dd yyyy  h:mm AM/PM dddd", "\u0f5f 01 2022  4:32 \u0f66\u0f94\u0f0b\u0f51\u0fb2\u0f7c\u0f0b \u0F42\u0F5F\u0F60\u0F0B\u0F55\u0F74\u0F62\u0F0B\u0F56\u0F74\u0F0D"},
		{"44562.189571759256", "[$-73]mmm dd yyyy  h:mm AM/PM", "\u1325\u122A 01 2022  4:32 \u1295\u1309\u1206"},
		{"44562.189571759256", "[$-73]mmmm dd yyyy  h:mm AM/PM", "\u1325\u122A 01 2022  4:32 \u1295\u1309\u1206"},
		{"44562.189571759256", "[$-73]mmmmm dd yyyy  h:mm AM/PM", "\u1325 01 2022  4:32 \u1295\u1309\u1206"},
		{"44562.189571759256", "[$-73]mmmmmm dd yyyy  h:mm AM/PM", "\u1325\u122A 01 2022  4:32 \u1295\u1309\u1206"},
		{"43543.503206018519", "[$-73]mmm dd yyyy  h:mm AM/PM", "\u1218\u130B 19 2019  12:04 \u12F5\u1215\u122A%20\u1250\u1275\u122A"},
		{"43543.503206018519", "[$-73]mmmm dd yyyy  h:mm AM/PM aaa", "\u1218\u130B\u1262\u1275 19 2019  12:04 \u12F5\u1215\u122A%20\u1250\u1275\u122A \u1230\u1209"},
		{"43543.503206018519", "[$-73]mmmmm dd yyyy  h:mm AM/PM ddd", "\u1218 19 2019  12:04 \u12F5\u1215\u122A%20\u1250\u1275\u122A \u1230\u1209"},
		{"43543.503206018519", "[$-73]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u1218\u130B\u1262\u1275 19 2019  12:04 \u12F5\u1215\u122A%20\u1250\u1275\u122A \u1220\u1209\u1235"},
		{"44562.189571759256", "[$-873]mmm dd yyyy  h:mm AM/PM", "\u1325\u122A 01 2022  4:32 \u1295\u1309\u1206%20\u1230\u12D3\u1270"},
		{"44562.189571759256", "[$-873]mmmm dd yyyy  h:mm AM/PM", "\u1325\u122A 01 2022  4:32 \u1295\u1309\u1206%20\u1230\u12D3\u1270"},
		{"44562.189571759256", "[$-873]mmmmm dd yyyy  h:mm AM/PM", "\u1325 01 2022  4:32 \u1295\u1309\u1206%20\u1230\u12D3\u1270"},
		{"44562.189571759256", "[$-873]mmmmmm dd yyyy  h:mm AM/PM", "\u1325\u122A 01 2022  4:32 \u1295\u1309\u1206%20\u1230\u12D3\u1270"},
		{"43543.503206018519", "[$-873]mmm dd yyyy  h:mm AM/PM", "\u1218\u130B 19 2019  12:04 \u12F5\u1215\u122D%20\u1230\u12D3\u1275"},
		{"43543.503206018519", "[$-873]mmmm dd yyyy  h:mm AM/PM aaa", "\u1218\u130B\u1262\u1275 19 2019  12:04 \u12F5\u1215\u122D%20\u1230\u12D3\u1275 \u1230\u1209"},
		{"43543.503206018519", "[$-873]mmmmm dd yyyy  h:mm AM/PM ddd", "\u1218 19 2019  12:04 \u12F5\u1215\u122D%20\u1230\u12D3\u1275 \u1230\u1209"},
		{"43543.503206018519", "[$-873]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u1218\u130B\u1262\u1275 19 2019  12:04 \u12F5\u1215\u122D%20\u1230\u12D3\u1275 \u1220\u1209\u1235"},
		{"44562.189571759256", "[$-473]mmm dd yyyy  h:mm AM/PM", "\u1325\u122A 01 2022  4:32 \u1295\u1309\u1206"},
		{"44562.189571759256", "[$-473]mmmm dd yyyy  h:mm AM/PM", "\u1325\u122A 01 2022  4:32 \u1295\u1309\u1206"},
		{"44562.189571759256", "[$-473]mmmmm dd yyyy  h:mm AM/PM", "\u1325 01 2022  4:32 \u1295\u1309\u1206"},
		{"44562.189571759256", "[$-473]mmmmmm dd yyyy  h:mm AM/PM", "\u1325\u122A 01 2022  4:32 \u1295\u1309\u1206"},
		{"43543.503206018519", "[$-473]mmm dd yyyy  h:mm AM/PM", "\u1218\u130B 19 2019  12:04 \u12F5\u1215\u122A%20\u1250\u1275\u122A"},
		{"43543.503206018519", "[$-473]mmmm dd yyyy  h:mm AM/PM aaa", "\u1218\u130B\u1262\u1275 19 2019  12:04 \u12F5\u1215\u122A%20\u1250\u1275\u122A \u1230\u1209"},
		{"43543.503206018519", "[$-473]mmmmm dd yyyy  h:mm AM/PM ddd", "\u1218 19 2019  12:04 \u12F5\u1215\u122A%20\u1250\u1275\u122A \u1230\u1209"},
		{"43543.503206018519", "[$-473]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u1218\u130B\u1262\u1275 19 2019  12:04 \u12F5\u1215\u122A%20\u1250\u1275\u122A \u1220\u1209\u1235"},
		{"44562.189571759256", "[$-31]mmm dd yyyy  h:mm AM/PM", "Sun 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-31]mmmm dd yyyy  h:mm AM/PM", "Sunguti 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-31]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-31]mmmmmm dd yyyy  h:mm AM/PM", "Sunguti 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-31]mmm dd yyyy  h:mm AM/PM", "Kul 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-31]mmmm dd yyyy  h:mm AM/PM aaa", "Nyenyankulu 19 2019  12:04 PM Bir"},
		{"43543.503206018519", "[$-31]mmmmm dd yyyy  h:mm AM/PM ddd", "N 19 2019  12:04 PM Bir"},
		{"43543.503206018519", "[$-31]mmmmmm dd yyyy  h:mm AM/PM dddd", "Nyenyankulu 19 2019  12:04 PM Ravumbirhi"},
		{"44562.189571759256", "[$-431]mmm dd yyyy  h:mm AM/PM", "Sun 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-431]mmmm dd yyyy  h:mm AM/PM", "Sunguti 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-431]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-431]mmmmmm dd yyyy  h:mm AM/PM", "Sunguti 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-431]mmm dd yyyy  h:mm AM/PM", "Kul 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-431]mmmm dd yyyy  h:mm AM/PM aaa", "Nyenyankulu 19 2019  12:04 PM Bir"},
		{"43543.503206018519", "[$-431]mmmmm dd yyyy  h:mm AM/PM ddd", "N 19 2019  12:04 PM Bir"},
		{"43543.503206018519", "[$-431]mmmmmm dd yyyy  h:mm AM/PM dddd", "Nyenyankulu 19 2019  12:04 PM Ravumbirhi"},
		{"44562.189571759256", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Oca 01 2022  4:32 \u00F6\u00F6"},
		{"44593.189571759256", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Şub 01 2022  4:32 \u00F6\u00F6"},
		{"44621.18957170139", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Mar 01 2022  4:32 \u00F6\u00F6"},
		{"44652.18957170139", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Nis 01 2022  4:32 \u00F6\u00F6"},
		{"44682.18957170139", "[$-1F]mmm dd yyyy  h:mm AM/PM", "May 01 2022  4:32 \u00F6\u00F6"},
		{"44713.18957170139", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Haz 01 2022  4:32 \u00F6\u00F6"},
		{"44743.18957170139", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Tem 01 2022  4:32 \u00F6\u00F6"},
		{"44774.18957170139", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Ağu 01 2022  4:32 \u00F6\u00F6"},
		{"44805.18957170139", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Eyl 01 2022  4:32 \u00F6\u00F6"},
		{"44835.18957170139", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Eki 01 2022  4:32 \u00F6\u00F6"},
		{"44866.18957170139", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Kas 01 2022  4:32 \u00F6\u00F6"},
		{"44896.18957170139", "[$-1F]mmm dd yyyy  h:mm AM/PM", "Ara 01 2022  4:32 \u00F6\u00F6"},
		{"44562.189571759256", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Ocak 01 2022  4:32 \u00F6\u00F6"},
		{"44593.189571759256", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Şubat 01 2022  4:32 \u00F6\u00F6"},
		{"44621.18957170139", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Mart 01 2022  4:32 \u00F6\u00F6"},
		{"44652.18957170139", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Nisan 01 2022  4:32 \u00F6\u00F6"},
		{"44682.18957170139", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Mayıs 01 2022  4:32 \u00F6\u00F6"},
		{"44713.18957170139", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Haziran 01 2022  4:32 \u00F6\u00F6"},
		{"44743.18957170139", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Temmuz 01 2022  4:32 \u00F6\u00F6"},
		{"44774.18957170139", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Ağustos 01 2022  4:32 \u00F6\u00F6"},
		{"44805.18957170139", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Eylül 01 2022  4:32 \u00F6\u00F6"},
		{"44835.18957170139", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Ekim 01 2022  4:32 \u00F6\u00F6"},
		{"44866.18957170139", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Kasım 01 2022  4:32 \u00F6\u00F6"},
		{"44896.18957170139", "[$-1F]mmmm dd yyyy  h:mm AM/PM", "Aralık 01 2022  4:32 \u00F6\u00F6"},
		{"44562.189571759256", "[$-1F]mmmmm dd yyyy  h:mm AM/PM", "O 01 2022  4:32 \u00F6\u00F6"},
		{"44593.189571759256", "[$-1F]mmmmm dd yyyy  h:mm AM/PM", "Ş 01 2022  4:32 \u00F6\u00F6"},
		{"44621.18957170139", "[$-1F]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 \u00F6\u00F6"},
		{"44652.18957170139", "[$-1F]mmmmm dd yyyy  h:mm AM/PM", "N 01 2022  4:32 \u00F6\u00F6"},
		{"44682.18957170139", "[$-1F]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 \u00F6\u00F6"},
		{"44713.18957170139", "[$-1F]mmmmm dd yyyy  h:mm AM/PM", "H 01 2022  4:32 \u00F6\u00F6"},
		{"44743.18957170139", "[$-1F]mmmmm dd yyyy  h:mm AM/PM", "T 01 2022  4:32 \u00F6\u00F6"},
		{"44774.18957170139", "[$-1F]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 \u00F6\u00F6"},
		{"44805.18957170139", "[$-1F]mmmmm dd yyyy  h:mm AM/PM", "E 01 2022  4:32 \u00F6\u00F6"},
		{"44835.18957170139", "[$-1F]mmmmm dd yyyy  h:mm AM/PM aaa", "E 01 2022  4:32 \u00F6\u00F6 Cmt"},
		{"44866.18957170139", "[$-1F]mmmmm dd yyyy  h:mm AM/PM ddd", "K 01 2022  4:32 \u00F6\u00F6 Sal"},
		{"44896.18957170139", "[$-1F]mmmmm dd yyyy  h:mm AM/PM dddd", "A 01 2022  4:32 \u00F6\u00F6 Perşembe"},
		{"44562.189571759256", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Oca 01 2022  4:32 \u00F6\u00F6"},
		{"44593.189571759256", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Şub 01 2022  4:32 \u00F6\u00F6"},
		{"44621.18957170139", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Mar 01 2022  4:32 \u00F6\u00F6"},
		{"44652.18957170139", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Nis 01 2022  4:32 \u00F6\u00F6"},
		{"44682.18957170139", "[$-41F]mmm dd yyyy  h:mm AM/PM", "May 01 2022  4:32 \u00F6\u00F6"},
		{"44713.18957170139", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Haz 01 2022  4:32 \u00F6\u00F6"},
		{"44743.18957170139", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Tem 01 2022  4:32 \u00F6\u00F6"},
		{"44774.18957170139", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Ağu 01 2022  4:32 \u00F6\u00F6"},
		{"44805.18957170139", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Eyl 01 2022  4:32 \u00F6\u00F6"},
		{"44835.18957170139", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Eki 01 2022  4:32 \u00F6\u00F6"},
		{"44866.18957170139", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Kas 01 2022  4:32 \u00F6\u00F6"},
		{"44896.18957170139", "[$-41F]mmm dd yyyy  h:mm AM/PM", "Ara 01 2022  4:32 \u00F6\u00F6"},
		{"44562.189571759256", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Ocak 01 2022  4:32 \u00F6\u00F6"},
		{"44593.189571759256", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Şubat 01 2022  4:32 \u00F6\u00F6"},
		{"44621.18957170139", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Mart 01 2022  4:32 \u00F6\u00F6"},
		{"44652.18957170139", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Nisan 01 2022  4:32 \u00F6\u00F6"},
		{"44682.18957170139", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Mayıs 01 2022  4:32 \u00F6\u00F6"},
		{"44713.18957170139", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Haziran 01 2022  4:32 \u00F6\u00F6"},
		{"44743.18957170139", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Temmuz 01 2022  4:32 \u00F6\u00F6"},
		{"44774.18957170139", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Ağustos 01 2022  4:32 \u00F6\u00F6"},
		{"44805.18957170139", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Eylül 01 2022  4:32 \u00F6\u00F6"},
		{"44835.18957170139", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Ekim 01 2022  4:32 \u00F6\u00F6"},
		{"44866.18957170139", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Kasım 01 2022  4:32 \u00F6\u00F6"},
		{"44896.18957170139", "[$-41F]mmmm dd yyyy  h:mm AM/PM", "Aralık 01 2022  4:32 \u00F6\u00F6"},
		{"44562.189571759256", "[$-41F]mmmmm dd yyyy  h:mm AM/PM", "O 01 2022  4:32 \u00F6\u00F6"},
		{"44593.189571759256", "[$-41F]mmmmm dd yyyy  h:mm AM/PM", "Ş 01 2022  4:32 \u00F6\u00F6"},
		{"44621.18957170139", "[$-41F]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 \u00F6\u00F6"},
		{"44652.18957170139", "[$-41F]mmmmm dd yyyy  h:mm AM/PM", "N 01 2022  4:32 \u00F6\u00F6"},
		{"44682.18957170139", "[$-41F]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 \u00F6\u00F6"},
		{"44713.18957170139", "[$-41F]mmmmm dd yyyy  h:mm AM/PM", "H 01 2022  4:32 \u00F6\u00F6"},
		{"44743.18957170139", "[$-41F]mmmmm dd yyyy  h:mm AM/PM", "T 01 2022  4:32 \u00F6\u00F6"},
		{"44774.18957170139", "[$-41F]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 \u00F6\u00F6"},
		{"44805.18957170139", "[$-41F]mmmmm dd yyyy  h:mm AM/PM", "E 01 2022  4:32 \u00F6\u00F6"},
		{"44835.18957170139", "[$-41F]mmmmm dd yyyy  h:mm AM/PM aaa", "E 01 2022  4:32 \u00F6\u00F6 Cmt"},
		{"44866.18957170139", "[$-41F]mmmmm dd yyyy  h:mm AM/PM ddd", "K 01 2022  4:32 \u00F6\u00F6 Sal"},
		{"44896.18957170139", "[$-41F]mmmmm dd yyyy  h:mm AM/PM dddd", "A 01 2022  4:32 \u00F6\u00F6 Perşembe"},
		{"44562.189571759256", "[$-42]mmm dd yyyy  h:mm AM/PM", "Ýan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42]mmmm dd yyyy  h:mm AM/PM", "Ýanwar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42]mmmmm dd yyyy  h:mm AM/PM", "Ý 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-42]mmmmmm dd yyyy  h:mm AM/PM", "Ýanwar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-42]mmm dd yyyy  h:mm AM/PM", "Mart 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-42]mmmm dd yyyy  h:mm AM/PM aaa", "Mart 19 2019  12:04 PM Sb"},
		{"43543.503206018519", "[$-42]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Sb"},
		{"43543.503206018519", "[$-42]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mart 19 2019  12:04 PM Sişenbe"},
		{"44562.189571759256", "[$-442]mmm dd yyyy  h:mm AM/PM", "Ýan 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-442]mmmm dd yyyy  h:mm AM/PM", "Ýanwar 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-442]mmmmm dd yyyy  h:mm AM/PM", "Ý 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-442]mmmmmm dd yyyy  h:mm AM/PM", "Ýanwar 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-442]mmm dd yyyy  h:mm AM/PM", "Mart 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-442]mmmm dd yyyy  h:mm AM/PM aaa", "Mart 19 2019  12:04 PM Sb"},
		{"43543.503206018519", "[$-442]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 PM Sb"},
		{"43543.503206018519", "[$-442]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mart 19 2019  12:04 PM Sişenbe"},
		{"44562.189571759256", "[$-22]mmm dd yyyy  h:mm AM/PM", "\u0421\u0456\u0447 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-22]mmmm dd yyyy  h:mm AM/PM", "\u0441\u0456\u0447\u0435\u043D\u044C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-22]mmmmm dd yyyy  h:mm AM/PM", "\u0441 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-22]mmmmmm dd yyyy  h:mm AM/PM", "\u0441\u0456\u0447\u0435\u043D\u044C 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-22]mmm dd yyyy  h:mm AM/PM", "\u0411\u0435\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-22]mmmm dd yyyy  h:mm AM/PM aaa", "\u0431\u0435\u0440\u0435\u0437\u0435\u043D\u044C 19 2019  12:04 PM \u0412\u0442"},
		{"43543.503206018519", "[$-22]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0431 19 2019  12:04 PM \u0412\u0442"},
		{"43543.503206018519", "[$-22]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0431\u0435\u0440\u0435\u0437\u0435\u043D\u044C 19 2019  12:04 PM \u0432\u0456\u0432\u0442\u043E\u0440\u043E\u043A"},
		{"44562.189571759256", "[$-422]mmm dd yyyy  h:mm AM/PM", "\u0421\u0456\u0447 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-422]mmmm dd yyyy  h:mm AM/PM", "\u0441\u0456\u0447\u0435\u043D\u044C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-422]mmmmm dd yyyy  h:mm AM/PM", "\u0441 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-422]mmmmmm dd yyyy  h:mm AM/PM", "\u0441\u0456\u0447\u0435\u043D\u044C 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-422]mmm dd yyyy  h:mm AM/PM", "\u0411\u0435\u0440 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-422]mmmm dd yyyy  h:mm AM/PM aaa", "\u0431\u0435\u0440\u0435\u0437\u0435\u043D\u044C 19 2019  12:04 PM \u0412\u0442"},
		{"43543.503206018519", "[$-422]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0431 19 2019  12:04 PM \u0412\u0442"},
		{"43543.503206018519", "[$-422]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0431\u0435\u0440\u0435\u0437\u0435\u043D\u044C 19 2019  12:04 PM \u0432\u0456\u0432\u0442\u043E\u0440\u043E\u043A"},
		{"44562.189571759256", "[$-2E]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 dopołdnja"},
		{"44562.189571759256", "[$-2E]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 dopołdnja"},
		{"44562.189571759256", "[$-2E]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 dopołdnja"},
		{"44562.189571759256", "[$-2E]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 dopołdnja"},
		{"43543.503206018519", "[$-2E]mmm dd yyyy  h:mm AM/PM", "měr 19 2019  12:04 popołdnju"},
		{"43543.503206018519", "[$-2E]mmmm dd yyyy  h:mm AM/PM aaa", "měrc 19 2019  12:04 popołdnju wut"},
		{"43543.503206018519", "[$-2E]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 popołdnju wut"},
		{"43543.503206018519", "[$-2E]mmmmmm dd yyyy  h:mm AM/PM dddd", "měrc 19 2019  12:04 popołdnju wutora"},
		{"44562.189571759256", "[$-42E]mmm dd yyyy  h:mm AM/PM", "jan 01 2022  4:32 dopołdnja"},
		{"44562.189571759256", "[$-42E]mmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 dopołdnja"},
		{"44562.189571759256", "[$-42E]mmmmm dd yyyy  h:mm AM/PM", "j 01 2022  4:32 dopołdnja"},
		{"44562.189571759256", "[$-42E]mmmmmm dd yyyy  h:mm AM/PM", "januar 01 2022  4:32 dopołdnja"},
		{"43543.503206018519", "[$-42E]mmm dd yyyy  h:mm AM/PM", "měr 19 2019  12:04 popołdnju"},
		{"43543.503206018519", "[$-42E]mmmm dd yyyy  h:mm AM/PM aaa", "měrc 19 2019  12:04 popołdnju wut"},
		{"43543.503206018519", "[$-42E]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 popołdnju wut"},
		{"43543.503206018519", "[$-42E]mmmmmm dd yyyy  h:mm AM/PM dddd", "měrc 19 2019  12:04 popołdnju wutora"},
		{"44562.189571759256", "[$-20]mmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-20]mmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-20]mmmmm dd yyyy  h:mm AM/PM", "\u062C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-20]mmmmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-20]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-20]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0645\u0646\u06AF\u0644"},
		{"43543.503206018519", "[$-20]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 PM \u0645\u0646\u06AF\u0644"},
		{"43543.503206018519", "[$-20]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0645\u0646\u06AF\u0644"},
		{"44562.189571759256", "[$-820]mmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 \u062F\u0646"},
		{"44562.189571759256", "[$-820]mmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 \u062F\u0646"},
		{"44562.189571759256", "[$-820]mmmmm dd yyyy  h:mm AM/PM", "\u062C 01 2022  4:32 \u062F\u0646"},
		{"44562.189571759256", "[$-820]mmmmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 \u062F\u0646"},
		{"43543.503206018519", "[$-820]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0686 19 2019  12:04 \u0631\u0627\u062A"},
		{"43543.503206018519", "[$-820]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0686 19 2019  12:04 \u0631\u0627\u062A \u0645\u0646\u06AF\u0644"},
		{"43543.503206018519", "[$-820]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0631\u0627\u062A \u0645\u0646\u06AF\u0644"},
		{"43543.503206018519", "[$-820]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0686 19 2019  12:04 \u0631\u0627\u062A \u0645\u0646\u06AF\u0644"},
		{"44562.189571759256", "[$-420]mmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-420]mmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-420]mmmmm dd yyyy  h:mm AM/PM", "\u062C 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-420]mmmmmm dd yyyy  h:mm AM/PM", "\u062C\u0646\u0648\u0631\u06CC 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-420]mmm dd yyyy  h:mm AM/PM", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-420]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0645\u0646\u06AF\u0644"},
		{"43543.503206018519", "[$-420]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 PM \u0645\u0646\u06AF\u0644"},
		{"43543.503206018519", "[$-420]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u0686 19 2019  12:04 PM \u0645\u0646\u06AF\u0644"},
		{"44562.189571759256", "[$-80]mmm dd yyyy  h:mm AM/PM", "1-\u0626\u0627\u064A 01 2022  4:32 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0628\u06C7\u0631\u06C7\u0646"},
		{"44562.189571759256", "[$-80]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0627\u0646\u06CB\u0627\u0631 01 2022  4:32 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0628\u06C7\u0631\u06C7\u0646"},
		{"44562.189571759256", "[$-80]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0628\u06C7\u0631\u06C7\u0646"},
		{"44562.189571759256", "[$-80]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0627\u0646\u06CB\u0627\u0631 01 2022  4:32 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0628\u06C7\u0631\u06C7\u0646"},
		{"43543.503206018519", "[$-80]mmm dd yyyy  h:mm AM/PM", "3-\u0626\u0627\u064A 19 2019  12:04 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0643\u06D0\u064A\u0649\u0646"},
		{"43543.503206018519", "[$-80]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u062A 19 2019  12:04 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0643\u06D0\u064A\u0649\u0646 \u0633\u06D5"},
		{"43543.503206018519", "[$-80]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0643\u06D0\u064A\u0649\u0646 \u0633\u06D5"},
		{"43543.503206018519", "[$-80]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u062A 19 2019  12:04 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0643\u06D0\u064A\u0649\u0646 \u0633\u06D5\u064A\u0634\u06D5\u0646\u0628\u06D5"},
		{"44562.189571759256", "[$-480]mmm dd yyyy  h:mm AM/PM", "1-\u0626\u0627\u064A 01 2022  4:32 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0628\u06C7\u0631\u06C7\u0646"},
		{"44562.189571759256", "[$-480]mmmm dd yyyy  h:mm AM/PM", "\u064A\u0627\u0646\u06CB\u0627\u0631 01 2022  4:32 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0628\u06C7\u0631\u06C7\u0646"},
		{"44562.189571759256", "[$-480]mmmmm dd yyyy  h:mm AM/PM", "\u064A 01 2022  4:32 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0628\u06C7\u0631\u06C7\u0646"},
		{"44562.189571759256", "[$-480]mmmmmm dd yyyy  h:mm AM/PM", "\u064A\u0627\u0646\u06CB\u0627\u0631 01 2022  4:32 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0628\u06C7\u0631\u06C7\u0646"},
		{"43543.503206018519", "[$-480]mmm dd yyyy  h:mm AM/PM", "3-\u0626\u0627\u064A 19 2019  12:04 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0643\u06D0\u064A\u0649\u0646"},
		{"43543.503206018519", "[$-480]mmmm dd yyyy  h:mm AM/PM aaa", "\u0645\u0627\u0631\u062A 19 2019  12:04 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0643\u06D0\u064A\u0649\u0646 \u0633\u06D5"},
		{"43543.503206018519", "[$-480]mmmmm dd yyyy  h:mm AM/PM ddd", "\u0645 19 2019  12:04 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0643\u06D0\u064A\u0649\u0646 \u0633\u06D5"},
		{"43543.503206018519", "[$-480]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u0645\u0627\u0631\u062A 19 2019  12:04 \u0686\u06C8\u0634\u062A\u0649\u0646%20\u0643\u06D0\u064A\u0649\u0646 \u0633\u06D5\u064A\u0634\u06D5\u0646\u0628\u06D5"},
		{"44562.189571759256", "[$-7843]mmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432 01 2022  4:32 \u0422\u041E"},
		{"44562.189571759256", "[$-7843]mmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432\u0430\u0440 01 2022  4:32 \u0422\u041E"},
		{"44562.189571759256", "[$-7843]mmmmm dd yyyy  h:mm AM/PM", "\u044F 01 2022  4:32 \u0422\u041E"},
		{"44562.189571759256", "[$-7843]mmmmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432\u0430\u0440 01 2022  4:32 \u0422\u041E"},
		{"43543.503206018519", "[$-7843]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 \u0422\u041A"},
		{"43543.503206018519", "[$-7843]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 \u0422\u041A \u0441\u0435\u0448"},
		{"43543.503206018519", "[$-7843]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 \u0422\u041A \u0441\u0435\u0448"},
		{"43543.503206018519", "[$-7843]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 \u0422\u041A \u0441\u0435\u0448\u0430\u043D\u0431\u0430"},
		{"44562.189571759256", "[$-843]mmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432 01 2022  4:32 \u0422\u041E"},
		{"44562.189571759256", "[$-843]mmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432\u0430\u0440 01 2022  4:32 \u0422\u041E"},
		{"44562.189571759256", "[$-843]mmmmm dd yyyy  h:mm AM/PM", "\u044F 01 2022  4:32 \u0422\u041E"},
		{"44562.189571759256", "[$-843]mmmmmm dd yyyy  h:mm AM/PM", "\u044F\u043D\u0432\u0430\u0440 01 2022  4:32 \u0422\u041E"},
		{"43543.503206018519", "[$-843]mmm dd yyyy  h:mm AM/PM", "\u043C\u0430\u0440 19 2019  12:04 \u0422\u041A"},
		{"43543.503206018519", "[$-843]mmmm dd yyyy  h:mm AM/PM aaa", "\u043C\u0430\u0440\u0442 19 2019  12:04 \u0422\u041A \u0441\u0435\u0448"},
		{"43543.503206018519", "[$-843]mmmmm dd yyyy  h:mm AM/PM ddd", "\u043C 19 2019  12:04 \u0422\u041A \u0441\u0435\u0448"},
		{"43543.503206018519", "[$-843]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u043C\u0430\u0440\u0442 19 2019  12:04 \u0422\u041A \u0441\u0435\u0448\u0430\u043D\u0431\u0430"},
		{"44562.189571759256", "[$-43]mmm dd yyyy  h:mm AM/PM", "Yan 01 2022  4:32 TO"},
		{"44562.189571759256", "[$-43]mmmm dd yyyy  h:mm AM/PM", "Yanvar 01 2022  4:32 TO"},
		{"44562.189571759256", "[$-43]mmmmm dd yyyy  h:mm AM/PM", "Y 01 2022  4:32 TO"},
		{"44562.189571759256", "[$-43]mmmmmm dd yyyy  h:mm AM/PM", "Yanvar 01 2022  4:32 TO"},
		{"43543.503206018519", "[$-43]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 TK"},
		{"43543.503206018519", "[$-43]mmmm dd yyyy  h:mm AM/PM aaa", "Mart 19 2019  12:04 TK Sesh"},
		{"43543.503206018519", "[$-43]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 TK Sesh"},
		{"43543.503206018519", "[$-43]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mart 19 2019  12:04 TK seshanba"},
		{"44562.189571759256", "[$-7C43]mmm dd yyyy  h:mm AM/PM", "Yan 01 2022  4:32 TO"},
		{"44562.189571759256", "[$-7C43]mmmm dd yyyy  h:mm AM/PM", "Yanvar 01 2022  4:32 TO"},
		{"44562.189571759256", "[$-7C43]mmmmm dd yyyy  h:mm AM/PM", "Y 01 2022  4:32 TO"},
		{"44562.189571759256", "[$-7C43]mmmmmm dd yyyy  h:mm AM/PM", "Yanvar 01 2022  4:32 TO"},
		{"43543.503206018519", "[$-7C43]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 TK"},
		{"43543.503206018519", "[$-7C43]mmmm dd yyyy  h:mm AM/PM aaa", "Mart 19 2019  12:04 TK Sesh"},
		{"43543.503206018519", "[$-7C43]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 TK Sesh"},
		{"43543.503206018519", "[$-7C43]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mart 19 2019  12:04 TK seshanba"},
		{"44562.189571759256", "[$-443]mmm dd yyyy  h:mm AM/PM", "Yan 01 2022  4:32 TO"},
		{"44562.189571759256", "[$-443]mmmm dd yyyy  h:mm AM/PM", "Yanvar 01 2022  4:32 TO"},
		{"44562.189571759256", "[$-443]mmmmm dd yyyy  h:mm AM/PM", "Y 01 2022  4:32 TO"},
		{"44562.189571759256", "[$-443]mmmmmm dd yyyy  h:mm AM/PM", "Yanvar 01 2022  4:32 TO"},
		{"43543.503206018519", "[$-443]mmm dd yyyy  h:mm AM/PM", "Mar 19 2019  12:04 TK"},
		{"43543.503206018519", "[$-443]mmmm dd yyyy  h:mm AM/PM aaa", "Mart 19 2019  12:04 TK Sesh"},
		{"43543.503206018519", "[$-443]mmmmm dd yyyy  h:mm AM/PM ddd", "M 19 2019  12:04 TK Sesh"},
		{"43543.503206018519", "[$-443]mmmmmm dd yyyy  h:mm AM/PM dddd", "Mart 19 2019  12:04 TK seshanba"},
		{"44562.189571759256", "[$-803]mmm dd yyyy  h:mm AM/PM", "gen. 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-803]mmmm dd yyyy  h:mm AM/PM", "gener 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-803]mmmmm dd yyyy  h:mm AM/PM", "g 01 2022  4:32 a.%A0m."},
		{"44562.189571759256", "[$-803]mmmmmm dd yyyy  h:mm AM/PM", "gener 01 2022  4:32 a.%A0m."},
		{"43543.503206018519", "[$-803]mmm dd yyyy  h:mm AM/PM", "març 19 2019  12:04 p.%A0m."},
		{"43543.503206018519", "[$-803]mmmm dd yyyy  h:mm AM/PM aaa", "març 19 2019  12:04 p.%A0m. dt."},
		{"43543.503206018519", "[$-803]mmmmm dd yyyy  h:mm AM/PM ddd", "m 19 2019  12:04 p.%A0m. dt."},
		{"43543.503206018519", "[$-803]mmmmmm dd yyyy  h:mm AM/PM dddd", "març 19 2019  12:04 p.%A0m. dimarts"},
		{"44562.189571759256", "[$-33]mmm dd yyyy  h:mm AM/PM", "Pha 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-33]mmmm dd yyyy  h:mm AM/PM", "Phando 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-33]mmmmm dd yyyy  h:mm AM/PM", "P 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-33]mmmmmm dd yyyy  h:mm AM/PM", "Phando 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-33]mmm dd yyyy  h:mm AM/PM", "Ṱhf 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-33]mmmm dd yyyy  h:mm AM/PM aaa", "Ṱhafamuhwe 19 2019  12:04 PM Vhi"},
		{"43543.503206018519", "[$-33]mmmmm dd yyyy  h:mm AM/PM ddd", "Ṱ 19 2019  12:04 PM Vhi"},
		{"43543.503206018519", "[$-33]mmmmmm dd yyyy  h:mm AM/PM dddd", "Ṱhafamuhwe 19 2019  12:04 PM Ḽavhuvhili"},
		{"44562.189571759256", "[$-433]mmm dd yyyy  h:mm AM/PM", "Pha 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-433]mmmm dd yyyy  h:mm AM/PM", "Phando 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-433]mmmmm dd yyyy  h:mm AM/PM", "P 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-433]mmmmmm dd yyyy  h:mm AM/PM", "Phando 01 2022  4:32 AM"},
		{"43543.503206018519", "[$-433]mmm dd yyyy  h:mm AM/PM", "Ṱhf 19 2019  12:04 PM"},
		{"43543.503206018519", "[$-433]mmmm dd yyyy  h:mm AM/PM aaa", "Ṱhafamuhwe 19 2019  12:04 PM Vhi"},
		{"43543.503206018519", "[$-433]mmmmm dd yyyy  h:mm AM/PM ddd", "Ṱ 19 2019  12:04 PM Vhi"},
		{"43543.503206018519", "[$-433]mmmmmm dd yyyy  h:mm AM/PM dddd", "Ṱhafamuhwe 19 2019  12:04 PM Ḽavhuvhili"},
		{"44562.189571759256", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 1 01 2022  4:32 SA"},
		{"44593.189571759256", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 2 01 2022  4:32 SA"},
		{"44621.18957170139", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 3 01 2022  4:32 SA"},
		{"44652.18957170139", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 4 01 2022  4:32 SA"},
		{"44682.18957170139", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 5 01 2022  4:32 SA"},
		{"44713.18957170139", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 6 01 2022  4:32 SA"},
		{"44743.18957170139", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 7 01 2022  4:32 SA"},
		{"44774.18957170139", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 8 01 2022  4:32 SA"},
		{"44805.18957170139", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 9 01 2022  4:32 SA"},
		{"44835.18957170139", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 10 01 2022  4:32 SA"},
		{"44866.18957170139", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 11 01 2022  4:32 SA"},
		{"44896.18957170139", "[$-2A]mmm dd yyyy  h:mm AM/PM", "Thg 12 01 2022  4:32 SA"},
		{"44562.189571759256", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 1 01 2022  4:32 SA"},
		{"44593.189571759256", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 2 01 2022  4:32 SA"},
		{"44621.18957170139", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 3 01 2022  4:32 SA"},
		{"44652.18957170139", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 4 01 2022  4:32 SA"},
		{"44682.18957170139", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 5 01 2022  4:32 SA"},
		{"44713.18957170139", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 6 01 2022  4:32 SA"},
		{"44743.18957170139", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 7 01 2022  4:32 SA"},
		{"44774.18957170139", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 8 01 2022  4:32 SA"},
		{"44805.18957170139", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 9 01 2022  4:32 SA"},
		{"44835.18957170139", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 10 01 2022  4:32 SA"},
		{"44866.18957170139", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 11 01 2022  4:32 SA"},
		{"44896.18957170139", "[$-2A]mmmm dd yyyy  h:mm AM/PM", "Tháng 12 01 2022  4:32 SA"},
		{"44562.189571759256", "[$-2A]mmmmm dd yyyy  h:mm AM/PM", "T 1 01 2022  4:32 SA"},
		{"44593.189571759256", "[$-2A]mmmmm dd yyyy  h:mm AM/PM", "T 2 01 2022  4:32 SA"},
		{"44621.18957170139", "[$-2A]mmmmm dd yyyy  h:mm AM/PM", "T 3 01 2022  4:32 SA"},
		{"44652.18957170139", "[$-2A]mmmmm dd yyyy  h:mm AM/PM", "T 4 01 2022  4:32 SA"},
		{"44682.18957170139", "[$-2A]mmmmm dd yyyy  h:mm AM/PM", "T 5 01 2022  4:32 SA"},
		{"44713.18957170139", "[$-2A]mmmmm dd yyyy  h:mm AM/PM", "T 6 01 2022  4:32 SA"},
		{"44743.18957170139", "[$-2A]mmmmm dd yyyy  h:mm AM/PM", "T 7 01 2022  4:32 SA"},
		{"44774.18957170139", "[$-2A]mmmmm dd yyyy  h:mm AM/PM", "T 8 01 2022  4:32 SA"},
		{"44805.18957170139", "[$-2A]mmmmm dd yyyy  h:mm AM/PM", "T 9 01 2022  4:32 SA"},
		{"44835.18957170139", "[$-2A]mmmmm dd yyyy  h:mm AM/PM aaa", "T 10 01 2022  4:32 SA T7"},
		{"44866.18957170139", "[$-2A]mmmmm dd yyyy  h:mm AM/PM ddd", "T 11 01 2022  4:32 SA T3"},
		{"44896.18957170139", "[$-2A]mmmmm dd yyyy  h:mm AM/PM dddd", "T 12 01 2022  4:32 SA Th\u1EE9%20N\u0103m"},
		{"44562.189571759256", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 1 01 2022  4:32 SA"},
		{"44593.189571759256", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 2 01 2022  4:32 SA"},
		{"44621.18957170139", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 3 01 2022  4:32 SA"},
		{"44652.18957170139", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 4 01 2022  4:32 SA"},
		{"44682.18957170139", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 5 01 2022  4:32 SA"},
		{"44713.18957170139", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 6 01 2022  4:32 SA"},
		{"44743.18957170139", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 7 01 2022  4:32 SA"},
		{"44774.18957170139", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 8 01 2022  4:32 SA"},
		{"44805.18957170139", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 9 01 2022  4:32 SA"},
		{"44835.18957170139", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 10 01 2022  4:32 SA"},
		{"44866.18957170139", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 11 01 2022  4:32 SA"},
		{"44896.18957170139", "[$-42A]mmm dd yyyy  h:mm AM/PM", "Thg 12 01 2022  4:32 SA"},
		{"44562.189571759256", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 1 01 2022  4:32 SA"},
		{"44593.189571759256", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 2 01 2022  4:32 SA"},
		{"44621.18957170139", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 3 01 2022  4:32 SA"},
		{"44652.18957170139", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 4 01 2022  4:32 SA"},
		{"44682.18957170139", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 5 01 2022  4:32 SA"},
		{"44713.18957170139", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 6 01 2022  4:32 SA"},
		{"44743.18957170139", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 7 01 2022  4:32 SA"},
		{"44774.18957170139", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 8 01 2022  4:32 SA"},
		{"44805.18957170139", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 9 01 2022  4:32 SA"},
		{"44835.18957170139", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 10 01 2022  4:32 SA"},
		{"44866.18957170139", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 11 01 2022  4:32 SA"},
		{"44896.18957170139", "[$-42A]mmmm dd yyyy  h:mm AM/PM", "Tháng 12 01 2022  4:32 SA"},
		{"44562.189571759256", "[$-42A]mmmmm dd yyyy  h:mm AM/PM", "T 1 01 2022  4:32 SA"},
		{"44593.189571759256", "[$-42A]mmmmm dd yyyy  h:mm AM/PM", "T 2 01 2022  4:32 SA"},
		{"44621.18957170139", "[$-42A]mmmmm dd yyyy  h:mm AM/PM", "T 3 01 2022  4:32 SA"},
		{"44652.18957170139", "[$-42A]mmmmm dd yyyy  h:mm AM/PM", "T 4 01 2022  4:32 SA"},
		{"44682.18957170139", "[$-42A]mmmmm dd yyyy  h:mm AM/PM", "T 5 01 2022  4:32 SA"},
		{"44713.18957170139", "[$-42A]mmmmm dd yyyy  h:mm AM/PM", "T 6 01 2022  4:32 SA"},
		{"44743.18957170139", "[$-42A]mmmmm dd yyyy  h:mm AM/PM", "T 7 01 2022  4:32 SA"},
		{"44774.18957170139", "[$-42A]mmmmm dd yyyy  h:mm AM/PM", "T 8 01 2022  4:32 SA"},
		{"44805.18957170139", "[$-42A]mmmmm dd yyyy  h:mm AM/PM", "T 9 01 2022  4:32 SA"},
		{"44835.18957170139", "[$-42A]mmmmm dd yyyy  h:mm AM/PM aaa", "T 10 01 2022  4:32 SA T7"},
		{"44866.18957170139", "[$-42A]mmmmm dd yyyy  h:mm AM/PM ddd", "T 11 01 2022  4:32 SA T3"},
		{"44896.18957170139", "[$-42A]mmmmm dd yyyy  h:mm AM/PM dddd", "T 12 01 2022  4:32 SA Th\u1EE9%20N\u0103m"},
		{"44562.189571759256", "[$-52]mmm dd yyyy  h:mm AM/PM", "Ion 01 2022  4:32 yb"},
		{"44593.189571759256", "[$-52]mmm dd yyyy  h:mm AM/PM", "Chwef 01 2022  4:32 yb"},
		{"44621.18957170139", "[$-52]mmm dd yyyy  h:mm AM/PM", "Maw 01 2022  4:32 yb"},
		{"44652.18957170139", "[$-52]mmm dd yyyy  h:mm AM/PM", "Ebr 01 2022  4:32 yb"},
		{"44682.18957170139", "[$-52]mmm dd yyyy  h:mm AM/PM", "Mai 01 2022  4:32 yb"},
		{"44713.18957170139", "[$-52]mmm dd yyyy  h:mm AM/PM", "Meh 01 2022  4:32 yb"},
		{"44743.18957170139", "[$-52]mmm dd yyyy  h:mm AM/PM", "Gorff 01 2022  4:32 yb"},
		{"44774.18957170139", "[$-52]mmm dd yyyy  h:mm AM/PM", "Awst 01 2022  4:32 yb"},
		{"44805.18957170139", "[$-52]mmm dd yyyy  h:mm AM/PM", "Medi 01 2022  4:32 yb"},
		{"44835.18957170139", "[$-52]mmm dd yyyy  h:mm AM/PM", "Hyd 01 2022  4:32 yb"},
		{"44866.18957170139", "[$-52]mmm dd yyyy  h:mm AM/PM", "Tach 01 2022  4:32 yb"},
		{"44896.18957170139", "[$-52]mmm dd yyyy  h:mm AM/PM", "Rhag 01 2022  4:32 yb"},
		{"44562.189571759256", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Ionawr 01 2022  4:32 yb"},
		{"44593.189571759256", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Chwefror 01 2022  4:32 yb"},
		{"44621.18957170139", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Mawrth 01 2022  4:32 yb"},
		{"44652.18957170139", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Ebrill 01 2022  4:32 yb"},
		{"44682.18957170139", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Mai 01 2022  4:32 yb"},
		{"44713.18957170139", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Mehefin 01 2022  4:32 yb"},
		{"44743.18957170139", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Gorffennaf 01 2022  4:32 yb"},
		{"44774.18957170139", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Awst 01 2022  4:32 yb"},
		{"44805.18957170139", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Medi 01 2022  4:32 yb"},
		{"44835.18957170139", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Hydref 01 2022  4:32 yb"},
		{"44866.18957170139", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Tachwedd 01 2022  4:32 yb"},
		{"44896.18957170139", "[$-52]mmmm dd yyyy  h:mm AM/PM", "Rhagfyr 01 2022  4:32 yb"},
		{"44562.189571759256", "[$-52]mmmmm dd yyyy  h:mm AM/PM", "I 01 2022  4:32 yb"},
		{"44593.189571759256", "[$-52]mmmmm dd yyyy  h:mm AM/PM", "C 01 2022  4:32 yb"},
		{"44621.18957170139", "[$-52]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 yb"},
		{"44652.18957170139", "[$-52]mmmmm dd yyyy  h:mm AM/PM", "E 01 2022  4:32 yb"},
		{"44682.18957170139", "[$-52]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 yb"},
		{"44713.18957170139", "[$-52]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 yb"},
		{"44743.18957170139", "[$-52]mmmmm dd yyyy  h:mm AM/PM", "G 01 2022  4:32 yb"},
		{"44774.18957170139", "[$-52]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 yb"},
		{"44805.18957170139", "[$-52]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 yb"},
		{"44835.18957170139", "[$-52]mmmmm dd yyyy  h:mm AM/PM aaa", "H 01 2022  4:32 yb Sad"},
		{"44866.18957170139", "[$-52]mmmmm dd yyyy  h:mm AM/PM ddd", "T 01 2022  4:32 yb Maw"},
		{"44896.18957170139", "[$-52]mmmmm dd yyyy  h:mm AM/PM dddd", "R 01 2022  4:32 yb Dydd Iau"},
		{"44562.189571759256", "[$-452]mmm dd yyyy  h:mm AM/PM", "Ion 01 2022  4:32 yb"},
		{"44593.189571759256", "[$-452]mmm dd yyyy  h:mm AM/PM", "Chwef 01 2022  4:32 yb"},
		{"44621.18957170139", "[$-452]mmm dd yyyy  h:mm AM/PM", "Maw 01 2022  4:32 yb"},
		{"44652.18957170139", "[$-452]mmm dd yyyy  h:mm AM/PM", "Ebr 01 2022  4:32 yb"},
		{"44682.18957170139", "[$-452]mmm dd yyyy  h:mm AM/PM", "Mai 01 2022  4:32 yb"},
		{"44713.18957170139", "[$-452]mmm dd yyyy  h:mm AM/PM", "Meh 01 2022  4:32 yb"},
		{"44743.18957170139", "[$-452]mmm dd yyyy  h:mm AM/PM", "Gorff 01 2022  4:32 yb"},
		{"44774.18957170139", "[$-452]mmm dd yyyy  h:mm AM/PM", "Awst 01 2022  4:32 yb"},
		{"44805.18957170139", "[$-452]mmm dd yyyy  h:mm AM/PM", "Medi 01 2022  4:32 yb"},
		{"44835.18957170139", "[$-452]mmm dd yyyy  h:mm AM/PM", "Hyd 01 2022  4:32 yb"},
		{"44866.18957170139", "[$-452]mmm dd yyyy  h:mm AM/PM", "Tach 01 2022  4:32 yb"},
		{"44896.18957170139", "[$-452]mmm dd yyyy  h:mm AM/PM", "Rhag 01 2022  4:32 yb"},
		{"44562.189571759256", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Ionawr 01 2022  4:32 yb"},
		{"44593.189571759256", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Chwefror 01 2022  4:32 yb"},
		{"44621.18957170139", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Mawrth 01 2022  4:32 yb"},
		{"44652.18957170139", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Ebrill 01 2022  4:32 yb"},
		{"44682.18957170139", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Mai 01 2022  4:32 yb"},
		{"44713.18957170139", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Mehefin 01 2022  4:32 yb"},
		{"44743.18957170139", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Gorffennaf 01 2022  4:32 yb"},
		{"44774.18957170139", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Awst 01 2022  4:32 yb"},
		{"44805.18957170139", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Medi 01 2022  4:32 yb"},
		{"44835.18957170139", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Hydref 01 2022  4:32 yb"},
		{"44866.18957170139", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Tachwedd 01 2022  4:32 yb"},
		{"44896.18957170139", "[$-452]mmmm dd yyyy  h:mm AM/PM", "Rhagfyr 01 2022  4:32 yb"},
		{"44562.189571759256", "[$-452]mmmmm dd yyyy  h:mm AM/PM", "I 01 2022  4:32 yb"},
		{"44593.189571759256", "[$-452]mmmmm dd yyyy  h:mm AM/PM", "C 01 2022  4:32 yb"},
		{"44621.18957170139", "[$-452]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 yb"},
		{"44652.18957170139", "[$-452]mmmmm dd yyyy  h:mm AM/PM", "E 01 2022  4:32 yb"},
		{"44682.18957170139", "[$-452]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 yb"},
		{"44713.18957170139", "[$-452]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 yb"},
		{"44743.18957170139", "[$-452]mmmmm dd yyyy  h:mm AM/PM", "G 01 2022  4:32 yb"},
		{"44774.18957170139", "[$-452]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 yb"},
		{"44805.18957170139", "[$-452]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 yb"},
		{"44835.18957170139", "[$-452]mmmmm dd yyyy  h:mm AM/PM aaa", "H 01 2022  4:32 yb Sad"},
		{"44866.18957170139", "[$-452]mmmmm dd yyyy  h:mm AM/PM ddd", "T 01 2022  4:32 yb Maw"},
		{"44896.18957170139", "[$-452]mmmmm dd yyyy  h:mm AM/PM dddd", "R 01 2022  4:32 yb Dydd Iau"},
		{"44562.189571759256", "[$-88]mmm dd yyyy  h:mm AM/PM", "Sam. 01 2022  4:32 Sub"},
		{"44593.189571759256", "[$-88]mmm dd yyyy  h:mm AM/PM", "Few. 01 2022  4:32 Sub"},
		{"44621.18957170139", "[$-88]mmm dd yyyy  h:mm AM/PM", "Maa 01 2022  4:32 Sub"},
		{"44652.18957170139", "[$-88]mmm dd yyyy  h:mm AM/PM", "Awr. 01 2022  4:32 Sub"},
		{"44682.18957170139", "[$-88]mmm dd yyyy  h:mm AM/PM", "Me 01 2022  4:32 Sub"},
		{"44713.18957170139", "[$-88]mmm dd yyyy  h:mm AM/PM", "Suw 01 2022  4:32 Sub"},
		{"44743.18957170139", "[$-88]mmm dd yyyy  h:mm AM/PM", "Sul. 01 2022  4:32 Sub"},
		{"44774.18957170139", "[$-88]mmm dd yyyy  h:mm AM/PM", "Ut 01 2022  4:32 Sub"},
		{"44805.18957170139", "[$-88]mmm dd yyyy  h:mm AM/PM", "Sept. 01 2022  4:32 Sub"},
		{"44835.18957170139", "[$-88]mmm dd yyyy  h:mm AM/PM", "Okt. 01 2022  4:32 Sub"},
		{"44866.18957170139", "[$-88]mmm dd yyyy  h:mm AM/PM", "Now. 01 2022  4:32 Sub"},
		{"44896.18957170139", "[$-88]mmm dd yyyy  h:mm AM/PM", "Des. 01 2022  4:32 Sub"},
		{"44562.189571759256", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Samwiye 01 2022  4:32 Sub"},
		{"44593.189571759256", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Fewriye 01 2022  4:32 Sub"},
		{"44621.18957170139", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Maars 01 2022  4:32 Sub"},
		{"44652.18957170139", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Awril 01 2022  4:32 Sub"},
		{"44682.18957170139", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Me 01 2022  4:32 Sub"},
		{"44713.18957170139", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Suwe 01 2022  4:32 Sub"},
		{"44743.18957170139", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Sullet 01 2022  4:32 Sub"},
		{"44774.18957170139", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Ut 01 2022  4:32 Sub"},
		{"44805.18957170139", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Septàmbar 01 2022  4:32 Sub"},
		{"44835.18957170139", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Oktoobar 01 2022  4:32 Sub"},
		{"44866.18957170139", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Noowàmbar 01 2022  4:32 Sub"},
		{"44896.18957170139", "[$-88]mmmm dd yyyy  h:mm AM/PM", "Desàmbar 01 2022  4:32 Sub"},
		{"44562.189571759256", "[$-88]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 Sub"},
		{"44593.189571759256", "[$-88]mmmmm dd yyyy  h:mm AM/PM", "F 01 2022  4:32 Sub"},
		{"44621.18957170139", "[$-88]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 Sub"},
		{"44652.18957170139", "[$-88]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 Sub"},
		{"44682.18957170139", "[$-88]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 Sub"},
		{"44713.18957170139", "[$-88]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 Sub"},
		{"44743.18957170139", "[$-88]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 Sub"},
		{"44774.18957170139", "[$-88]mmmmm dd yyyy  h:mm AM/PM", "U 01 2022  4:32 Sub"},
		{"44805.18957170139", "[$-88]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 Sub"},
		{"44835.18957170139", "[$-88]mmmmm dd yyyy  h:mm AM/PM aaa", "O 01 2022  4:32 Sub Gaa."},
		{"44866.18957170139", "[$-88]mmmmm dd yyyy  h:mm AM/PM ddd", "N 01 2022  4:32 Sub Tal."},
		{"44896.18957170139", "[$-88]mmmmm dd yyyy  h:mm AM/PM dddd", "D 01 2022  4:32 Sub Alxames"},
		{"44562.189571759256", "[$-488]mmm dd yyyy  h:mm AM/PM", "Sam. 01 2022  4:32 Sub"},
		{"44593.189571759256", "[$-488]mmm dd yyyy  h:mm AM/PM", "Few. 01 2022  4:32 Sub"},
		{"44621.18957170139", "[$-488]mmm dd yyyy  h:mm AM/PM", "Maa 01 2022  4:32 Sub"},
		{"44652.18957170139", "[$-488]mmm dd yyyy  h:mm AM/PM", "Awr. 01 2022  4:32 Sub"},
		{"44682.18957170139", "[$-488]mmm dd yyyy  h:mm AM/PM", "Me 01 2022  4:32 Sub"},
		{"44713.18957170139", "[$-488]mmm dd yyyy  h:mm AM/PM", "Suw 01 2022  4:32 Sub"},
		{"44743.18957170139", "[$-488]mmm dd yyyy  h:mm AM/PM", "Sul. 01 2022  4:32 Sub"},
		{"44774.18957170139", "[$-488]mmm dd yyyy  h:mm AM/PM", "Ut 01 2022  4:32 Sub"},
		{"44805.18957170139", "[$-488]mmm dd yyyy  h:mm AM/PM", "Sept. 01 2022  4:32 Sub"},
		{"44835.18957170139", "[$-488]mmm dd yyyy  h:mm AM/PM", "Okt. 01 2022  4:32 Sub"},
		{"44866.18957170139", "[$-488]mmm dd yyyy  h:mm AM/PM", "Now. 01 2022  4:32 Sub"},
		{"44896.18957170139", "[$-488]mmm dd yyyy  h:mm AM/PM", "Des. 01 2022  4:32 Sub"},
		{"44562.189571759256", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Samwiye 01 2022  4:32 Sub"},
		{"44593.189571759256", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Fewriye 01 2022  4:32 Sub"},
		{"44621.18957170139", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Maars 01 2022  4:32 Sub"},
		{"44652.18957170139", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Awril 01 2022  4:32 Sub"},
		{"44682.18957170139", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Me 01 2022  4:32 Sub"},
		{"44713.18957170139", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Suwe 01 2022  4:32 Sub"},
		{"44743.18957170139", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Sullet 01 2022  4:32 Sub"},
		{"44774.18957170139", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Ut 01 2022  4:32 Sub"},
		{"44805.18957170139", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Septàmbar 01 2022  4:32 Sub"},
		{"44835.18957170139", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Oktoobar 01 2022  4:32 Sub"},
		{"44866.18957170139", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Noowàmbar 01 2022  4:32 Sub"},
		{"44896.18957170139", "[$-488]mmmm dd yyyy  h:mm AM/PM", "Desàmbar 01 2022  4:32 Sub"},
		{"44562.189571759256", "[$-488]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 Sub"},
		{"44593.189571759256", "[$-488]mmmmm dd yyyy  h:mm AM/PM", "F 01 2022  4:32 Sub"},
		{"44621.18957170139", "[$-488]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 Sub"},
		{"44652.18957170139", "[$-488]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 Sub"},
		{"44682.18957170139", "[$-488]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 Sub"},
		{"44713.18957170139", "[$-488]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 Sub"},
		{"44743.18957170139", "[$-488]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 Sub"},
		{"44774.18957170139", "[$-488]mmmmm dd yyyy  h:mm AM/PM", "U 01 2022  4:32 Sub"},
		{"44805.18957170139", "[$-488]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 Sub"},
		{"44835.18957170139", "[$-488]mmmmm dd yyyy  h:mm AM/PM aaa", "O 01 2022  4:32 Sub Gaa."},
		{"44866.18957170139", "[$-488]mmmmm dd yyyy  h:mm AM/PM ddd", "N 01 2022  4:32 Sub Tal."},
		{"44896.18957170139", "[$-488]mmmmm dd yyyy  h:mm AM/PM dddd", "D 01 2022  4:32 Sub Alxames"},
		{"44562.189571759256", "[$-34]mmm dd yyyy  h:mm AM/PM", "uJan. 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-34]mmm dd yyyy  h:mm AM/PM", "uFeb. 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-34]mmm dd yyyy  h:mm AM/PM", "uMat. 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-34]mmm dd yyyy  h:mm AM/PM", "uEpr. 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-34]mmm dd yyyy  h:mm AM/PM", "uMey. 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-34]mmm dd yyyy  h:mm AM/PM", "uJun. 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-34]mmm dd yyyy  h:mm AM/PM", "uJul. 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-34]mmm dd yyyy  h:mm AM/PM", "uAg. 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-34]mmm dd yyyy  h:mm AM/PM", "uSep. 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-34]mmm dd yyyy  h:mm AM/PM", "uOkt. 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-34]mmm dd yyyy  h:mm AM/PM", "uNov. 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-34]mmm dd yyyy  h:mm AM/PM", "uDis. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uJanuwari 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uFebuwari 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uMatshi 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uAprili 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uMeyi 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uJuni 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uJulayi 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uAgasti 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uSeptemba 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uOktobha 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uNovemba 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-34]mmmm dd yyyy  h:mm AM/PM", "uDisemba 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-34]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-34]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-34]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-34]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-34]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-34]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-34]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-34]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-34]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-34]mmmmm dd yyyy  h:mm AM/PM aaa", "u 01 2022  4:32 AM uMgq."},
		{"44866.18957170139", "[$-34]mmmmm dd yyyy  h:mm AM/PM ddd", "u 01 2022  4:32 AM uLwesib."},
		{"44896.18957170139", "[$-34]mmmmm dd yyyy  h:mm AM/PM dddd", "u 01 2022  4:32 AM Lwesine"},
		{"44562.189571759256", "[$-434]mmm dd yyyy  h:mm AM/PM", "uJan. 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-434]mmm dd yyyy  h:mm AM/PM", "uFeb. 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-434]mmm dd yyyy  h:mm AM/PM", "uMat. 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-434]mmm dd yyyy  h:mm AM/PM", "uEpr. 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-434]mmm dd yyyy  h:mm AM/PM", "uMey. 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-434]mmm dd yyyy  h:mm AM/PM", "uJun. 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-434]mmm dd yyyy  h:mm AM/PM", "uJul. 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-434]mmm dd yyyy  h:mm AM/PM", "uAg. 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-434]mmm dd yyyy  h:mm AM/PM", "uSep. 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-434]mmm dd yyyy  h:mm AM/PM", "uOkt. 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-434]mmm dd yyyy  h:mm AM/PM", "uNov. 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-434]mmm dd yyyy  h:mm AM/PM", "uDis. 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uJanuwari 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uFebuwari 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uMatshi 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uAprili 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uMeyi 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uJuni 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uJulayi 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uAgasti 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uSeptemba 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uOktobha 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uNovemba 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-434]mmmm dd yyyy  h:mm AM/PM", "uDisemba 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-434]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-434]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-434]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-434]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-434]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-434]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-434]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-434]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-434]mmmmm dd yyyy  h:mm AM/PM", "u 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-434]mmmmm dd yyyy  h:mm AM/PM aaa", "u 01 2022  4:32 AM uMgq."},
		{"44866.18957170139", "[$-434]mmmmm dd yyyy  h:mm AM/PM ddd", "u 01 2022  4:32 AM uLwesib."},
		{"44896.18957170139", "[$-434]mmmmm dd yyyy  h:mm AM/PM dddd", "u 01 2022  4:32 AM Lwesine"},
		{"44562.189571759256", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua2cd\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44593.189571759256", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua44d\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44621.18957170139", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua315\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44652.18957170139", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua1d6\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44682.18957170139", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua26c\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44713.18957170139", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua0d8\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44743.18957170139", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua3c3\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44774.18957170139", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua246\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44805.18957170139", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua22c\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44835.18957170139", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua2b0\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44866.18957170139", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua2b0\ua2aa\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44896.18957170139", "[$-78]mmm dd yyyy  h:mm AM/PM", "\ua2b0\ua44b\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44562.189571759256", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua2cd\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44593.189571759256", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua44d\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44621.18957170139", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua315\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44652.18957170139", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua1d6\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44682.18957170139", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua26c\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44713.18957170139", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua0d8\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44743.18957170139", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua3c3\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44774.18957170139", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua246\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44805.18957170139", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua22c\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44835.18957170139", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua2b0\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44866.18957170139", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua2b0\ua2aa\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44896.18957170139", "[$-78]mmmm dd yyyy  h:mm AM/PM", "\ua2b0\ua44b\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44562.189571759256", "[$-78]mmmmm dd yyyy  h:mm AM/PM", "\ua2cd 01 2022  4:32 \ua3b8\ua111"},
		{"44593.189571759256", "[$-78]mmmmm dd yyyy  h:mm AM/PM", "\ua44d 01 2022  4:32 \ua3b8\ua111"},
		{"44621.18957170139", "[$-78]mmmmm dd yyyy  h:mm AM/PM", "\ua315 01 2022  4:32 \ua3b8\ua111"},
		{"44652.18957170139", "[$-78]mmmmm dd yyyy  h:mm AM/PM", "\ua1d6 01 2022  4:32 \ua3b8\ua111"},
		{"44682.18957170139", "[$-78]mmmmm dd yyyy  h:mm AM/PM", "\ua26c 01 2022  4:32 \ua3b8\ua111"},
		{"44713.18957170139", "[$-78]mmmmm dd yyyy  h:mm AM/PM", "\ua0d8 01 2022  4:32 \ua3b8\ua111"},
		{"44743.18957170139", "[$-78]mmmmm dd yyyy  h:mm AM/PM", "\ua3c3 01 2022  4:32 \ua3b8\ua111"},
		{"44774.18957170139", "[$-78]mmmmm dd yyyy  h:mm AM/PM", "\ua246 01 2022  4:32 \ua3b8\ua111"},
		{"44805.18957170139", "[$-78]mmmmm dd yyyy  h:mm AM/PM", "\ua22c 01 2022  4:32 \ua3b8\ua111"},
		{"44835.18957170139", "[$-78]mmmmm dd yyyy  h:mm AM/PM aaa", "\ua2b0 01 2022  4:32 \ua3b8\ua111 \uA18F\uA0D8"},
		{"44866.18957170139", "[$-78]mmmmm dd yyyy  h:mm AM/PM ddd", "\ua2b0 01 2022  4:32 \ua3b8\ua111 \uA18F\uA44D"},
		{"44896.18957170139", "[$-78]mmmmm dd yyyy  h:mm AM/PM dddd", "\ua2b0 01 2022  4:32 \ua3b8\ua111 \uA18F\uA282\uA1D6"},
		{"44562.189571759256", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua2cd\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44593.189571759256", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua44d\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44621.18957170139", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua315\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44652.18957170139", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua1d6\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44682.18957170139", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua26c\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44713.18957170139", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua0d8\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44743.18957170139", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua3c3\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44774.18957170139", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua246\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44805.18957170139", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua22c\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44835.18957170139", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua2b0\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44866.18957170139", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua2b0\ua2aa\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44896.18957170139", "[$-478]mmm dd yyyy  h:mm AM/PM", "\ua2b0\ua44b\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44562.189571759256", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua2cd\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44593.189571759256", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua44d\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44621.18957170139", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua315\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44652.18957170139", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua1d6\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44682.18957170139", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua26c\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44713.18957170139", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua0d8\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44743.18957170139", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua3c3\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44774.18957170139", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua246\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44805.18957170139", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua22c\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44835.18957170139", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua2b0\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44866.18957170139", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua2b0\ua2aa\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44896.18957170139", "[$-478]mmmm dd yyyy  h:mm AM/PM", "\ua2b0\ua44b\ua1aa 01 2022  4:32 \ua3b8\ua111"},
		{"44562.189571759256", "[$-478]mmmmm dd yyyy  h:mm AM/PM", "\ua2cd 01 2022  4:32 \ua3b8\ua111"},
		{"44593.189571759256", "[$-478]mmmmm dd yyyy  h:mm AM/PM", "\ua44d 01 2022  4:32 \ua3b8\ua111"},
		{"44621.18957170139", "[$-478]mmmmm dd yyyy  h:mm AM/PM", "\ua315 01 2022  4:32 \ua3b8\ua111"},
		{"44652.18957170139", "[$-478]mmmmm dd yyyy  h:mm AM/PM", "\ua1d6 01 2022  4:32 \ua3b8\ua111"},
		{"44682.18957170139", "[$-478]mmmmm dd yyyy  h:mm AM/PM", "\ua26c 01 2022  4:32 \ua3b8\ua111"},
		{"44713.18957170139", "[$-478]mmmmm dd yyyy  h:mm AM/PM", "\ua0d8 01 2022  4:32 \ua3b8\ua111"},
		{"44743.18957170139", "[$-478]mmmmm dd yyyy  h:mm AM/PM", "\ua3c3 01 2022  4:32 \ua3b8\ua111"},
		{"44774.18957170139", "[$-478]mmmmm dd yyyy  h:mm AM/PM", "\ua246 01 2022  4:32 \ua3b8\ua111"},
		{"44805.18957170139", "[$-478]mmmmm dd yyyy  h:mm AM/PM", "\ua22c 01 2022  4:32 \ua3b8\ua111"},
		{"44835.18957170139", "[$-478]mmmmm dd yyyy  h:mm AM/PM aaa", "\ua2b0 01 2022  4:32 \ua3b8\ua111 \uA18F\uA0D8"},
		{"44866.18957170139", "[$-478]mmmmm dd yyyy  h:mm AM/PM ddd", "\ua2b0 01 2022  4:32 \ua3b8\ua111 \uA18F\uA44D"},
		{"44896.18957170139", "[$-478]mmmmm dd yyyy  h:mm AM/PM dddd", "\ua2b0 01 2022  4:32 \ua3b8\ua111 \uA18F\uA282\uA1D6"},
		{"44562.189571759256", "[$-43D]mmm dd yyyy  h:mm AM/PM", "\u05D9\u05D0\u05B7\u05E0 01 2022  4:32 \u05E4\u05BF\u05D0\u05B7\u05E8\u05DE\u05D9\u05D8\u05D0\u05B8\u05D2"},
		{"44562.189571759256", "[$-43D]mmmm dd yyyy  h:mm AM/PM", "\u05D9\u05D0\u05B7\u05E0\u05D5\u05D0\u05B7\u05E8 01 2022  4:32 \u05E4\u05BF\u05D0\u05B7\u05E8\u05DE\u05D9\u05D8\u05D0\u05B8\u05D2"},
		{"44562.189571759256", "[$-43D]mmmmm dd yyyy  h:mm AM/PM", "\u05D9 01 2022  4:32 \u05E4\u05BF\u05D0\u05B7\u05E8\u05DE\u05D9\u05D8\u05D0\u05B8\u05D2"},
		{"44562.189571759256", "[$-43D]mmmmmm dd yyyy  h:mm AM/PM", "\u05D9\u05D0\u05B7\u05E0\u05D5\u05D0\u05B7\u05E8 01 2022  4:32 \u05E4\u05BF\u05D0\u05B7\u05E8\u05DE\u05D9\u05D8\u05D0\u05B8\u05D2"},
		{"43543.503206018519", "[$-43D]mmm dd yyyy  h:mm AM/PM", "\u05DE\u05E2\u05E8\u05E5 19 2019  12:04 \u05E0\u05D0\u05B8\u05DB\u05DE\u05D9\u05D8\u05D0\u05B8\u05D2"},
		{"43543.503206018519", "[$-43D]mmmm dd yyyy  h:mm AM/PM aaa", "\u05DE\u05E2\u05E8\u05E5 19 2019  12:04 \u05E0\u05D0\u05B8\u05DB\u05DE\u05D9\u05D8\u05D0\u05B8\u05D2 \u05D9\u05D5\u05DD%A0\u05D2"},
		{"43543.503206018519", "[$-43D]mmmmm dd yyyy  h:mm AM/PM ddd", "\u05DE 19 2019  12:04 \u05E0\u05D0\u05B8\u05DB\u05DE\u05D9\u05D8\u05D0\u05B8\u05D2 \u05D9\u05D5\u05DD%A0\u05D2"},
		{"43543.503206018519", "[$-43D]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u05DE\u05E2\u05E8\u05E5 19 2019  12:04 \u05E0\u05D0\u05B8\u05DB\u05DE\u05D9\u05D8\u05D0\u05B8\u05D2 \u05D3\u05D9\u05E0\u05E1\u05D8\u05D9\u05E7"},
		{"44562.189571759256", "[$-6A]mmm dd yyyy  h:mm AM/PM", "\u1E62\u1EB9\u0301 01 2022  4:32 %C0%E1r\u1ECD\u0300"},
		{"44562.189571759256", "[$-6A]mmmm dd yyyy  h:mm AM/PM", "\u1E62\u1EB9\u0301r\u1EB9\u0301 01 2022  4:32 %C0%E1r\u1ECD\u0300"},
		{"44562.189571759256", "[$-6A]mmmmm dd yyyy  h:mm AM/PM", "\u1E62 01 2022  4:32 %C0%E1r\u1ECD\u0300"},
		{"44562.189571759256", "[$-6A]mmmmmm dd yyyy  h:mm AM/PM", "\u1E62\u1EB9\u0301r\u1EB9\u0301 01 2022  4:32 %C0%E1r\u1ECD\u0300"},
		{"43543.503206018519", "[$-6A]mmm dd yyyy  h:mm AM/PM", "\u1EB8r 19 2019  12:04 \u1ECC\u0300s%E1n"},
		{"43543.503206018519", "[$-6A]mmmm dd yyyy  h:mm AM/PM aaa", "\u1EB8r\u1EB9\u0300n%E0 19 2019  12:04 \u1ECC\u0300s%E1n %CC\u1E63g"},
		{"43543.503206018519", "[$-6A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u1EB8 19 2019  12:04 \u1ECC\u0300s%E1n %CC\u1E63g"},
		{"43543.503206018519", "[$-6A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u1EB8r\u1EB9\u0300n%E0 19 2019  12:04 \u1ECC\u0300s%E1n \u1ECCj\u1ECD\u0301%20%CCs\u1EB9\u0301gun"},
		{"44562.189571759256", "[$-46A]mmm dd yyyy  h:mm AM/PM", "\u1E62\u1EB9\u0301 01 2022  4:32 %C0%E1r\u1ECD\u0300"},
		{"44562.189571759256", "[$-46A]mmmm dd yyyy  h:mm AM/PM", "\u1E62\u1EB9\u0301r\u1EB9\u0301 01 2022  4:32 %C0%E1r\u1ECD\u0300"},
		{"44562.189571759256", "[$-46A]mmmmm dd yyyy  h:mm AM/PM", "\u1E62 01 2022  4:32 %C0%E1r\u1ECD\u0300"},
		{"44562.189571759256", "[$-46A]mmmmmm dd yyyy  h:mm AM/PM", "\u1E62\u1EB9\u0301r\u1EB9\u0301 01 2022  4:32 %C0%E1r\u1ECD\u0300"},
		{"43543.503206018519", "[$-46A]mmm dd yyyy  h:mm AM/PM", "\u1EB8r 19 2019  12:04 \u1ECC\u0300s%E1n"},
		{"43543.503206018519", "[$-46A]mmmm dd yyyy  h:mm AM/PM aaa", "\u1EB8r\u1EB9\u0300n%E0 19 2019  12:04 \u1ECC\u0300s%E1n %CC\u1E63g"},
		{"43543.503206018519", "[$-46A]mmmmm dd yyyy  h:mm AM/PM ddd", "\u1EB8 19 2019  12:04 \u1ECC\u0300s%E1n %CC\u1E63g"},
		{"43543.503206018519", "[$-46A]mmmmmm dd yyyy  h:mm AM/PM dddd", "\u1EB8r\u1EB9\u0300n%E0 19 2019  12:04 \u1ECC\u0300s%E1n \u1ECCj\u1ECD\u0301%20%CCs\u1EB9\u0301gun"},
		{"44562.189571759256", "[$-35]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-35]mmm dd yyyy  h:mm AM/PM", "Feb 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-35]mmm dd yyyy  h:mm AM/PM", "Mas 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-35]mmm dd yyyy  h:mm AM/PM", "Eph 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-35]mmm dd yyyy  h:mm AM/PM", "Mey 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-35]mmm dd yyyy  h:mm AM/PM", "Jun 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-35]mmm dd yyyy  h:mm AM/PM", "Jul 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-35]mmm dd yyyy  h:mm AM/PM", "Agas 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-35]mmm dd yyyy  h:mm AM/PM", "Sep 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-35]mmm dd yyyy  h:mm AM/PM", "Okt 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-35]mmm dd yyyy  h:mm AM/PM", "Nov 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-35]mmm dd yyyy  h:mm AM/PM", "Dis 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Januwari 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Febhuwari 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Mashi 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Ephreli 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Meyi 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Juni 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Julayi 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Agasti 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Septemba 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Okthoba 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Novemba 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-35]mmmm dd yyyy  h:mm AM/PM", "Disemba 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-35]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-35]mmmmm dd yyyy  h:mm AM/PM", "F 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-35]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-35]mmmmm dd yyyy  h:mm AM/PM", "E 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-35]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-35]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-35]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-35]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-35]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-35]mmmmm dd yyyy  h:mm AM/PM aaa", "O 01 2022  4:32 AM Mgq."},
		{"44866.18957170139", "[$-35]mmmmm dd yyyy  h:mm AM/PM ddd", "N 01 2022  4:32 AM Bi."},
		{"44896.18957170139", "[$-35]mmmmm dd yyyy  h:mm AM/PM dddd", "D 01 2022  4:32 AM ULwesine"},
		{"44562.189571759256", "[$-435]mmm dd yyyy  h:mm AM/PM", "Jan 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-435]mmm dd yyyy  h:mm AM/PM", "Feb 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-435]mmm dd yyyy  h:mm AM/PM", "Mas 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-435]mmm dd yyyy  h:mm AM/PM", "Eph 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-435]mmm dd yyyy  h:mm AM/PM", "Mey 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-435]mmm dd yyyy  h:mm AM/PM", "Jun 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-435]mmm dd yyyy  h:mm AM/PM", "Jul 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-435]mmm dd yyyy  h:mm AM/PM", "Agas 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-435]mmm dd yyyy  h:mm AM/PM", "Sep 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-435]mmm dd yyyy  h:mm AM/PM", "Okt 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-435]mmm dd yyyy  h:mm AM/PM", "Nov 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-435]mmm dd yyyy  h:mm AM/PM", "Dis 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Januwari 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Febhuwari 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Mashi 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Ephreli 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Meyi 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Juni 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Julayi 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Agasti 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Septemba 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Okthoba 01 2022  4:32 AM"},
		{"44866.18957170139", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Novemba 01 2022  4:32 AM"},
		{"44896.18957170139", "[$-435]mmmm dd yyyy  h:mm AM/PM", "Disemba 01 2022  4:32 AM"},
		{"44562.189571759256", "[$-435]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44593.189571759256", "[$-435]mmmmm dd yyyy  h:mm AM/PM", "F 01 2022  4:32 AM"},
		{"44621.18957170139", "[$-435]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 AM"},
		{"44652.18957170139", "[$-435]mmmmm dd yyyy  h:mm AM/PM", "E 01 2022  4:32 AM"},
		{"44682.18957170139", "[$-435]mmmmm dd yyyy  h:mm AM/PM", "M 01 2022  4:32 AM"},
		{"44713.18957170139", "[$-435]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44743.18957170139", "[$-435]mmmmm dd yyyy  h:mm AM/PM", "J 01 2022  4:32 AM"},
		{"44774.18957170139", "[$-435]mmmmm dd yyyy  h:mm AM/PM", "A 01 2022  4:32 AM"},
		{"44805.18957170139", "[$-435]mmmmm dd yyyy  h:mm AM/PM", "S 01 2022  4:32 AM"},
		{"44835.18957170139", "[$-435]mmmmm dd yyyy  h:mm AM/PM aaa", "O 01 2022  4:32 AM Mgq."},
		{"44866.18957170139", "[$-435]mmmmm dd yyyy  h:mm AM/PM ddd", "N 01 2022  4:32 AM Bi."},
		{"44896.18957170139", "[$-435]mmmmm dd yyyy  h:mm AM/PM dddd", "D 01 2022  4:32 AM ULwesine"},
		{"43543.503206018519", "[$-F800]dddd, mmmm dd, yyyy", "Tuesday, March 19, 2019"},
		{"43543.503206018519", "[$-F400]h:mm:ss AM/PM", "12:04:37 PM"},
		{"text_", "General", "text_"},
		{"text_", "\"=====\"@@@\"--\"@\"----\"", "=====text_text_text_--text_----"},
		{"0.0450685976001E+21", "0_);[Red]\\(0\\)", "45068597600100000000"},
		{"8.0450685976001E+21", "0_);[Red]\\(0\\)", "8045068597600100000000"},
		{"8.0450685976001E-21", "0_);[Red]\\(0\\)", "0"},
		{"8.04506", "0_);[Red]\\(0\\)", "8"},
		{"-0.0450685976001E+21", "0_);[Red]\\(0\\)", "(45068597600100000000)"},
		{"-8.0450685976001E+21", "0_);[Red]\\(0\\)", "(8045068597600100000000)"},
		{"-8.0450685976001E-21", "0_);[Red]\\(0\\)", "(0)"},
		{"-8.04506", "0_);[Red]\\(0\\)", "(8)"},
		{"1234.5678", "0", "1235"},
		{"1234.5678", "0.00", "1234.57"},
		{"1234.5678", "#,##0", "1,235"},
		{"1234.5678", "#,##0.00", "1,234.57"},
		{"1234.5678", "0%", "123457%"},
		{"1234.5678", "#,##0 ;(#,##0)", "1,235 "},
		{"1234.5678", "#,##0 ;[red](#,##0)", "1,235 "},
		{"1234.5678", "#,##0.00;(#,##0.00)", "1,234.57"},
		{"1234.5678", "#,##0.00;[red](#,##0.00)", "1,234.57"},
		{"-1234.5678", "0.00", "-1234.57"},
		{"-1234.5678", "0.00;-0.00", "-1234.57"},
		{"-1234.5678", "0.00%%", "-12345678.00%%"},
		{"2.1", "mmss.0000", "2400.000"},
		{"0.007", "[h]:mm:ss.0", "0:10:04.8"},
		{"0.007", "[h]:mm:ss.00", "0:10:04.80"},
		{"0.007", "[h]:mm:ss.000", "0:10:04.800"},
		{"0.007", "[h]:mm:ss.0000", "0:10:04.800"},
		{"123", "[h]:mm,:ss.0", "2952:00,:00.0"},
		{"123", "yy-.dd", "00-.02"},
		{"123", "[DBNum1][$-804]yyyy\"年\"m\"月\";@", "\u4e00\u4e5d\u25cb\u25cb\u5e74\u4e94\u6708"},
		{"123", "[DBNum2][$-804]yyyy\"年\"m\"月\";@", "\u58f9\u7396\u96f6\u96f6\u5e74\u4f0d\u6708"},
		{"123", "[DBNum3][$-804]yyyy\"年\"m\"月\";@", "\uff11\uff19\uff10\uff10\u5e74\uff15\u6708"},
		{"1234567890", "[DBNum1][$-804]0.00", "\u4e00\u4e8c\u4e09\u56db\u4e94\u516d\u4e03\u516b\u4e5d\u25cb.\u25cb\u25cb"},
		{"1234567890", "[DBNum2][$-804]0.00", "\u58f9\u8d30\u53c1\u8086\u4f0d\u9646\u67d2\u634c\u7396\u96f6.\u96f6\u96f6"},
		{"1234567890", "[DBNum3][$-804]0.00", "\uff11\uff12\uff13\uff14\uff15\uff16\uff17\uff18\uff19\uff10.\uff10\uff10"},
		{"1234.5678", "0.00###", "1234.5678"},
		{"1234.5678", "00000.00###", "01234.5678"},
		{"-1234.5678", "00000.00###;;", ""},
		{"1234.5678", "0.00000", "1234.56780"},
		{"8.8888666665555487", "0.00000", "8.88887"},
		{"8.8888666665555493e+19", "#,000.00", "88,888,666,665,555,500,000.00"},
		{"8.8888666665555493e+19", "0.00000", "88888666665555500000.00000"},
		{"37947.7500001", "0.00000000E+00", "3.79477500E+04"},
		{"1.234E-16", "0.00000000000000000000", "0.00000000000000012340"},
		{"1.234E-16", "0.000000000000000000", "0.000000000000000123"},
		{"1.234E-16", "0.000000000000000000%", "0.000000000000012340%"},
		{"1.234E-16", "0.000000000000000000%%%%", "0.000000000000012340%"},
		{"1234.5678", "[$$-409]#,##0.00", "$1,234.57"},
		// Unsupported number format
		{"37947.7500001", "0.00000000E+000", "37947.7500001"},
		{"123", "[$x.-unknown]#,##0.00", "123"},
		{"123", "[$x.-unknown]MM/DD/YYYY", "123"},
		{"123", "[DBNum4][$-804]yyyy\"年\"m\"月\";@", "123"},
		// Invalid number format
		{"123", "x0.00s", "123"},
		{"123", "[h]:m00m:ss", "123"},
		{"123", "yy-00dd", "123"},
		{"123", "yy-##dd", "123"},
		{"123", "xx[h]:mm,:ss.0xx", "xx2952:00,:00.0xx"},
		{"-123", "x0.00s", "-123"},
		{"-1234.5678", ";E+;", "-1234.5678"},
		{"1234.5678", "E+;", "1234.5678"},
		{"1234.5678", "00000.00###s", "1234.5678"},
		{"1234.5678", "0.0xxx00", "1234.5678"},
		{"-1234.5678", "00000.00###;s;", "-1234.5678"},
	} {
		result := format(item[0], item[1], false, CellTypeNumber, nil)
		assert.Equal(t, item[2], result, item)
	}
	// Test format number with specified date and time format code
	for _, item := range [][]string{
		{"43543.503206018519", "[$-F800]dddd, mmmm dd, yyyy", "2019年3月19日"},
		{"43543.503206018519", "[$-F400]h:mm:ss AM/PM", "12:04:37"},
	} {
		result := format(item[0], item[1], false, CellTypeNumber, &Options{
			ShortDatePattern: "yyyy/m/d",
			LongDatePattern:  "yyyy\"年\"M\"月\"d\"日\"",
			LongTimePattern:  "H:mm:ss",
		})
		assert.Equal(t, item[2], result, item)
	}
	// Test format number with string data type cell value
	for _, cellType := range []CellType{CellTypeSharedString, CellTypeInlineString} {
		for _, item := range [][]string{
			{"1234.5678", "General", "1234.5678"},
			{"1234.5678", "yyyy\"年\"m\"月\"d\"日\";@", "1234.5678"},
			{"1234.5678", "h\"时\"mm\"分\"ss\"秒\";@", "1234.5678"},
			{"1234.5678", "\"¥\"#,##0.00_);\\(\"¥\"#,##0.00\\)", "1234.5678"},
			{"1234.5678", "0_);[Red]\\(0\\)", "1234.5678"},
			{"1234.5678", "\"text\"@", "text1234.5678"},
		} {
			result := format(item[0], item[1], false, cellType, nil)
			assert.Equal(t, item[2], result, item)
		}
	}
	nf := numberFormat{}
	changeNumFmtCode, err := nf.currencyLanguageHandler(nfp.Token{Parts: []nfp.Part{{}}})
	assert.Equal(t, ErrUnsupportedNumberFormat, err)
	assert.False(t, changeNumFmtCode)
}
