// Copyright 2016-2020, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package model

import (
	"fmt"

	"github.com/hashicorp/hcl/v2"
	"github.com/hashicorp/hcl/v2/hclsyntax"
	"github.com/pulumi/pulumi/pkg/v3/codegen/hcl2/syntax"
)

// PromiseType represents eventual values that do not carry additional information.
type PromiseType struct {
	// ElementType is the element type of the promise.
	ElementType Type
}

// NewPromiseType creates a new promise type with the given element type after replacing any promise types within
// the element type with their respective element types.
func NewPromiseType(elementType Type) *PromiseType {
	return &PromiseType{ElementType: ResolvePromises(elementType)}
}

// SyntaxNode returns the syntax node for the type. This is always syntax.None.
func (*PromiseType) SyntaxNode() hclsyntax.Node {
	return syntax.None
}

// Traverse attempts to traverse the promise type with the given traverser. The result type of traverse(promise(T))
// is promise(traverse(T)).
func (t *PromiseType) Traverse(traverser hcl.Traverser) (Traversable, hcl.Diagnostics) {
	element, diagnostics := t.ElementType.Traverse(traverser)
	return NewPromiseType(element.(Type)), diagnostics
}

// Equals returns true if this type has the same identity as the given type.
func (t *PromiseType) Equals(other Type) bool {
	return t.equals(other, nil)
}

func (t *PromiseType) equals(other Type, seen map[Type]struct{}) bool {
	if t == other {
		return true
	}
	otherPromise, ok := other.(*PromiseType)
	return ok && t.ElementType.equals(otherPromise.ElementType, seen)
}

// AssignableFrom returns true if this type is assignable from the indicated source type. A promise(T) is assignable
// from values of type promise(U) and U, where T is assignable from U.
func (t *PromiseType) AssignableFrom(src Type) bool {
	return assignableFrom(t, src, func() bool {
		if src, ok := src.(*PromiseType); ok {
			return t.ElementType.AssignableFrom(src.ElementType)
		}
		return t.ElementType.AssignableFrom(src)
	})
}

// ConversionFrom returns the kind of conversion (if any) that is possible from the source type to this type. An
// promise(T) is convertible from a type U or promise(U) if U is convertible to T. If the conversion from U to T is
// unsafe, the entire conversion is unsafe. Otherwise, the conversion is safe.
func (t *PromiseType) ConversionFrom(src Type) ConversionKind {
	kind, _ := t.conversionFrom(src, false, nil)
	return kind
}

func (t *PromiseType) conversionFrom(
	src Type, unifying bool, seen map[Type]struct{}) (ConversionKind, hcl.Diagnostics) {
	return conversionFrom(t, src, unifying, seen, func() (ConversionKind, hcl.Diagnostics) {
		if src, ok := src.(*PromiseType); ok {
			return t.ElementType.conversionFrom(src.ElementType, unifying, seen)
		}
		return t.ElementType.conversionFrom(src, unifying, seen)
	})
}

func (t *PromiseType) String() string {
	return t.string(nil)
}

func (t *PromiseType) string(seen map[Type]struct{}) string {
	return fmt.Sprintf("promise(%s)", t.ElementType.string(seen))
}

func (t *PromiseType) unify(other Type) (Type, ConversionKind) {
	return unify(t, other, func() (Type, ConversionKind) {
		switch other := other.(type) {
		case *PromiseType:
			// If the other type is a promise type, unify based on the element type.
			elementType, conversionKind := t.ElementType.unify(other.ElementType)
			return NewPromiseType(elementType), conversionKind
		case *OutputType:
			// If the other type is an output type, prefer the optional type, but unify the element type.
			elementType, conversionKind := t.unify(other.ElementType)
			return NewOutputType(elementType), conversionKind
		default:
			// Prefer the promise type.
			kind, _ := t.conversionFrom(other, true, nil)
			return t, kind
		}
	})
}

func (t *PromiseType) isType() {}
