// Copyright 2016-2021, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package python

import (
	"strings"
	"testing"

	"github.com/blang/semver"
	"github.com/hashicorp/hcl/v2"

	"github.com/pulumi/pulumi/pkg/v3/codegen/hcl2"
	"github.com/pulumi/pulumi/pkg/v3/codegen/hcl2/syntax"
	"github.com/pulumi/pulumi/pkg/v3/codegen/internal/utils"
)

func parseAndBindProgram(t *testing.T, text, name string, options ...hcl2.BindOption) (*hcl2.Program, hcl.Diagnostics) {
	parser := syntax.NewParser()
	err := parser.ParseFile(strings.NewReader(text), name)
	if err != nil {
		t.Fatalf("could not read %v: %v", name, err)
	}
	if parser.Diagnostics.HasErrors() {
		t.Fatalf("failed to parse files: %v", parser.Diagnostics)
	}

	options = append(options, hcl2.PluginHost(utils.NewHost(testdataPath)))

	program, diags, err := hcl2.BindProgram(parser.Files, options...)
	if err != nil {
		t.Fatalf("could not bind program: %v", err)
	}
	return program, diags
}

func TestMakeSafeEnumName(t *testing.T) {
	tests := []struct {
		input    string
		expected string
		wantErr  bool
	}{
		{"red", "RED", false},
		{"snake_cased_name", "SNAKE_CASED_NAME", false},
		{"+", "", true},
		{"*", "ASTERISK", false},
		{"0", "ZERO", false},
		{"8.3", "TYPE_NAME_8_3", false},
		{"11", "TYPE_NAME_11", false},
		{"Microsoft-Windows-Shell-Startup", "MICROSOFT_WINDOWS_SHELL_STARTUP", false},
		{"Microsoft.Batch", "MICROSOFT_BATCH", false},
		{"readonly", "READONLY", false},
		{"SystemAssigned, UserAssigned", "SYSTEM_ASSIGNED_USER_ASSIGNED", false},
		{"Dev(NoSLA)_Standard_D11_v2", "DEV_NO_SL_A_STANDARD_D11_V2", false},
		{"Standard_E8as_v4+1TB_PS", "STANDARD_E8AS_V4_1_T_B_PS", false},
		{"Plants'R'Us", "PLANTS_R_US", false},
		{"Pulumi Planters Inc.", "PULUMI_PLANTERS_INC_", false},
		{"ZeroPointOne", "ZERO_POINT_ONE", false},
	}
	for _, tt := range tests {
		t.Run(tt.input, func(t *testing.T) {
			got, err := makeSafeEnumName(tt.input, "TypeName")
			if (err != nil) != tt.wantErr {
				t.Errorf("makeSafeEnumName() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.expected {
				t.Errorf("makeSafeEnumName() got = %v, want %v", got, tt.expected)
			}
		})
	}
}

func TestVersionConversion(t *testing.T) {
	tests := []struct {
		input    string
		expected string
	}{
		{"0.0.1", "0.0.1"},
		{"0.0.1-alpha", "0.0.1a"},
		{"2.3.4-dev", "2.3.4dev"},
		{"2.3.4-beta", "2.3.4b"},
		{"2.3.4-rc", "2.3.4rc"},
		{"3.13.0-alpha.1631222709+d49fd02f.dirty", "3.13.0a"},
	}
	for _, tt := range tests {
		ver := semver.MustParse(tt.input)
		got := pythonVersion(&ver)
		if got != tt.expected {
			t.Errorf("Unexpected version number: got = %q, want %q", got, tt.expected)
		}
	}
}
