// Copyright 2016-2018, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package b64

import (
	"encoding/base64"

	"github.com/pulumi/pulumi/pkg/v2/secrets"
	"github.com/pulumi/pulumi/sdk/v2/go/common/resource/config"
)

const Type = "b64"

// NewBase64SecretsManager returns a secrets manager that just base64 encodes instead of encrypting. Useful for testing.
func NewBase64SecretsManager() secrets.Manager {
	return &manager{}
}

type manager struct{}

func (m *manager) Type() string                         { return Type }
func (m *manager) State() interface{}                   { return map[string]string{} }
func (m *manager) Encrypter() (config.Encrypter, error) { return &base64Crypter{}, nil }
func (m *manager) Decrypter() (config.Decrypter, error) { return &base64Crypter{}, nil }

type base64Crypter struct{}

func (c *base64Crypter) EncryptValue(s string) (string, error) {
	return base64.StdEncoding.EncodeToString([]byte(s)), nil
}
func (c *base64Crypter) DecryptValue(s string) (string, error) {
	b, err := base64.StdEncoding.DecodeString(s)
	if err != nil {
		return "", err
	}
	return string(b), nil
}
