// Copyright 2018, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"github.com/pkg/errors"
	"github.com/spf13/cobra"

	"github.com/pulumi/pulumi/pkg/v2/backend/display"
	"github.com/pulumi/pulumi/sdk/v2/go/common/resource/config"
	"github.com/pulumi/pulumi/sdk/v2/go/common/util/cmdutil"
)

// TO-DO: Remove as part of Pulumi v3.0.0
func newHistoryCmd() *cobra.Command {
	var stack string
	var jsonOut bool
	var showSecrets bool
	var pageSize int
	var page int
	var cmd = &cobra.Command{
		Use:        "history",
		Aliases:    []string{"hist"},
		SuggestFor: []string{"updates"},
		Hidden:     true,
		Short:      "[DEPRECATED] Display history for a stack",
		Long: "Display history for a stack.\n\n" +
			"This command displays data about previous updates for a stack.\n\n" +
			"This command is now DEPRECATED, please use `pulumi stack history`.\n" +
			"The command will be removed in a future release",
		Args: cmdutil.NoArgs,
		Run: cmdutil.RunFunc(func(cmd *cobra.Command, args []string) error {
			opts := display.Options{
				Color: cmdutil.GetGlobalColorization(),
			}
			s, err := requireStack(stack, false /*offerNew */, opts, false /*setCurrent*/)
			if err != nil {
				return err
			}

			b := s.Backend()
			updates, err := b.GetHistory(commandContext(), s.Ref(), pageSize, page)
			if err != nil {
				return errors.Wrap(err, "getting history")
			}
			var decrypter config.Decrypter
			if showSecrets {
				crypter, err := getStackDecrypter(s)
				if err != nil {
					return errors.Wrap(err, "decrypting secrets")
				}
				decrypter = crypter
			}

			if jsonOut {
				return displayUpdatesJSON(updates, decrypter)
			}

			return displayUpdatesConsole(updates, page, opts, false)
		}),
	}
	cmd.PersistentFlags().StringVarP(
		&stack, "stack", "s", "",
		"Choose a stack other than the currently selected one")
	cmd.Flags().BoolVar(
		&showSecrets, "show-secrets", false,
		"Show secret values when listing config instead of displaying blinded values")
	cmd.PersistentFlags().BoolVarP(
		&jsonOut, "json", "j", false, "Emit output as JSON")
	cmd.PersistentFlags().IntVar(
		&pageSize, "page-size", 0, "Used with 'page' to control number of results returned")
	cmd.PersistentFlags().IntVar(
		&page, "page", 0, "Used with 'page-size' to paginate results")
	return cmd
}
