package bmtree

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestNewPath_compare(t *testing.T) {

	ta := require.New(t)

	ta.True(NewPath(0, 0, 7) < NewPath(0, 1, 7))
	ta.True(NewPath(0, 1, 7) < NewPath(0, 2, 7))
	ta.True(NewPath(12, 0, 7) < NewPath(12, 1, 7))
	ta.True(NewPath(12, 1, 7) < NewPath(12, 2, 7))

	ta.True(NewPath(7, 2, 3) > NewPath(6, 3, 3))
}

func TestNewPath(t *testing.T) {

	ta := require.New(t)

	cases := []struct {
		input        string
		from, height int32
		wantbm       string
	}{
		// a: 0110  0001
		{"", 0, 0, "00000000000000000000000000000000" + "00000000000000000000000000000000"},
		{"", 0, 1, "00000000000000000000000000000000" + "00000000000000000000000000000000"},
		{"", 0, 32, "00000000000000000000000000000000" + "00000000000000000000000000000000"},
		{"a", 0, 0, "00000000000000000000000000000000" + "00000000000000000000000000000000"},
		{"a", 0, 1, "00000000000000000000000000000000" + "00000000000000000000000000000001"},
		{"a", 0, 2, "00000000000000000000000000000001" + "00000000000000000000000000000011"},
		{"a", 0, 5, "00000000000000000000000000001100" + "00000000000000000000000000011111"},
		{"a", 0, 6, "00000000000000000000000000011000" + "00000000000000000000000000111111"},
		{"a", 0, 7, "00000000000000000000000000110000" + "00000000000000000000000001111111"},
		{"a", 0, 8, "00000000000000000000000001100001" + "00000000000000000000000011111111"},
		{"a", 2, 0, "00000000000000000000000000000000" + "00000000000000000000000000000000"},
		{"a", 2, 1, "00000000000000000000000000000001" + "00000000000000000000000000000001"},
		{"a", 2, 2, "00000000000000000000000000000010" + "00000000000000000000000000000011"},
		{"a", 2, 5, "00000000000000000000000000010000" + "00000000000000000000000000011111"},
		{"a", 2, 6, "00000000000000000000000000100001" + "00000000000000000000000000111111"},
		{"a", 2, 8, "00000000000000000000000010000100" + "00000000000000000000000011111100"},

		// "to" over string end
		{"abc", 10, 32, "10001001100011000000000000000000" + "11111111111111000000000000000000"},
		{"abc", 23, 8, "00000000000000000000000010000000" + "00000000000000000000000010000000"},
		{"abcdefgh", 31, 32, "00110010101100110011001110110100" + "11111111111111111111111111111111"},
		{"abcdefgh", 32, 32, "01100101011001100110011101101000" + "11111111111111111111111111111111"},
		{"abcdefgh", 33, 32, "11001010110011001100111011010000" + "11111111111111111111111111111110"},
	}

	for i, c := range cases {
		gotbm64 := PathOf(c.input, c.from, c.height)
		gotlen := PathLen(gotbm64)
		gotbm := fmt.Sprintf("%064b", gotbm64)

		ta.Equal(c.wantbm, gotbm, "%d-th: case: %+v", i+1, c)
		ta.Equal(gotbm64, NewPath(gotbm64>>32, gotlen, c.height))
	}
}

func TestPathsOf(t *testing.T) {

	ta := require.New(t)

	got := PathsOf([]string{"abc", "abc", "abd"}, 10, 32, false)
	gots := toStr(got)

	want := []string{
		"10001001100011000000000000000000" + "11111111111111000000000000000000",
		"10001001100011000000000000000000" + "11111111111111000000000000000000",
		"10001001100100000000000000000000" + "11111111111111000000000000000000",
	}
	ta.Equal(want, gots)

	got = PathsOf([]string{"abc", "abc", "abd"}, 10, 32, true)
	gots = toStr(got)

	want = []string{
		"10001001100011000000000000000000" + "11111111111111000000000000000000",
		"10001001100100000000000000000000" + "11111111111111000000000000000000",
	}
	ta.Equal(want, gots)
}

func toStr(vbs []uint64) []string {
	gots := make([]string, len(vbs))
	for i, g := range vbs {
		gots[i] = fmt.Sprintf("%064b", g)
	}
	return gots
}
