---
title: Processing and Exporting Data
weight: 4
linkTitle: Exporting data
---

Once you've instrumented your code, you need to get the data out in order to do anything useful with it. This page will cover the basics of the process and export pipeline.

## Sampling

Sampling is a process that restricts the amount of traces that are generated by a system. The exact sampler you should use depends on your specific needs, but in general you should make a decision at the start of a trace, and allow the sampling decision to propagate to other services.

A sampler needs to be set on the tracer provider when its configured, as follows:

```go
provider := sdktrace.NewTracerProvider(
	sdktrace.WithSampler(sdktrace.AlwaysSample()),
)
```

`AlwaysSample` and `NeverSample` are fairly self-explanatory. Always means that every trace will be sampled, the converse holds as true for Never. When you're getting started, or in a development environment, you'll almost always want to use `AlwaysSample`.

Other samplers include:

* `TraceIDRatioBased`, which will sample a fraction of traces, based on the fraction given to the sampler. Thus, if you set this to .5, half of traces will be sampled.
* `ParentBased`, which behaves differently based on the incoming sampling decision. In general, this will sample spans that have parents that were sampled, and will not sample spans whose parents were _not_ sampled.

When you're in production, you should consider using the `TraceIDRatioBased` sampler with the `ParentBased` sampler.

## Resources

Resources are a special type of attribute that apply to all spans generated by a process. These should be used to represent underlying metadata about a process that's non-ephemeral - for example, the hostname of a process, or its instance ID.

Resources should be assigned to a tracer provider at its initialization, and are created much like attributes:

```go
resources := resource.NewWithAttributes(
	semconv.SchemaURL,
	semconv.ServiceNameKey.String("myService"),
	semconv.ServiceVersionKey.String("1.0.0"),
	semconv.ServiceInstanceIDKey.String("abcdef12345"),
)

provider := sdktrace.NewTracerProvider(
	...
	sdktrace.WithResource(resources),
)
```

Note the use of the `semconv` package to provide conventional names for resource attributes.  This helps ensure that consumers of telemetry produced with these semantic conventions can
easily discover relevant attributes and understand their meaning.

Resources can also be detected automatically through `resource.Detector` implementations.  These `Detector`s may discover information about the currently running process, the operating
system it is running on, the cloud provider hosting that operating system instance, or any number of other resource attributes.

```go
resources := resource.New(context.Background(),
	resource.WithFromEnv(), // pull attributes from OTEL_RESOURCE_ATTRIBUTES and OTEL_SERVICE_NAME environment variables
	resource.WithProcess(), // This option configures a set of Detectors that discover process information
	resource.WithDetectors(thirdparty.Detector{}), // Bring your own external Detector implementation
	resource.WithAttributes(attribute.String("foo", "bar")), // Or specify resource attributes directly
)
```

## OTLP Exporter

OpenTelemetry Protocol (OTLP) export is available in the `go.opentelemetry.io/otel/exporters/otlp/otlptrace` and `go.opentelemetry.io/otel/exporters/otlp/otlpmetrics` packages.

Please find more documentation on [GitHub](https://github.com/open-telemetry/opentelemetry-go/tree/main/exporters/otlp)

## Jaeger Exporter

Jaeger export is available in the `go.opentelemetry.io/otel/exporters/jaeger` package.

Please find more documentation on [GitHub](https://github.com/open-telemetry/opentelemetry-go/tree/main/exporters/jaeger)

## Prometheus Exporter

Prometheus export is available in the `go.opentelemetry.io/otel/exporters/prometheus` package.

Please find more documentation on [GitHub](https://github.com/open-telemetry/opentelemetry-go/tree/main/exporters/prometheus)
