function global:Start-Utf8Process {
    param(
        [string] $FileName,
        [string] $Arguments
    )

    $Process = New-Object System.Diagnostics.Process
    $StartInfo = $Process.StartInfo
    $StartInfo.StandardErrorEncoding = $StartInfo.StandardOutputEncoding = [System.Text.Encoding]::UTF8
    $StartInfo.RedirectStandardError = $StartInfo.RedirectStandardInput = $StartInfo.RedirectStandardOutput = $true
    $StartInfo.FileName = $Filename
    $StartInfo.Arguments = $Arguments
    $StartInfo.UseShellExecute = $false
    if ($PWD.Provider.Name -eq 'FileSystem') {
        $StartInfo.WorkingDirectory = $PWD.ProviderPath
    }
    $StartInfo.CreateNoWindow = $true
    $Process.Start() | Out-Null
    $Process.WaitForExit() | Out-Null
    $stderr = $Process.StandardError.ReadToEnd().Trim()
    if ($stderr -ne '') {
        $Host.UI.WriteErrorLine($stderr)
    }
    return $Process.StandardOutput.ReadToEnd()
}

$env:POWERLINE_COMMAND = "oh-my-posh"
$env:CONDA_PROMPT_MODIFIER = $false
$env:SHELL_VERSION = $PSVersionTable.PSVersion.ToString()

# specific module support (disabled by default)
if ($null -eq $env:POSH_GIT_ENABLED) {
    $env:POSH_GIT_ENABLED = $false
}

# used to detect empty hit
$global:OMP_LASTHISTORYID = -1

if (Test-Path "::CONFIG::") {
    $env:POSH_THEME = (Resolve-Path -Path "::CONFIG::").ProviderPath
}

# set secondary prompt`
Set-PSReadLineOption -ContinuationPrompt (@(Start-Utf8Process "::OMP::" "print secondary --config=""$Env:POSH_THEME""") -join "`n")

function global:Set-PoshContext {}

function global:Get-PoshContext {
    $cleanPWD = $PWD.ProviderPath
    $cleanPSWD = $PWD.ToString()
    $cleanPWD = $cleanPWD.TrimEnd('\')
    $cleanPSWD = $cleanPSWD.TrimEnd('\')
    return $cleanPWD, $cleanPSWD
}

function global:Initialize-ModuleSupport {
    if ($env:POSH_GIT_ENABLED -eq $true) {
        $global:GitStatus = Get-GitStatus
        $env:POSH_GIT_STATUS = Write-GitStatus -Status $global:GitStatus
    }
}

Set-Item -Force -Path Function:prompt -Value {
    #store if the last command was successful
    $lastCommandSuccess = $?
    #store the last exit code for restore
    $realLASTEXITCODE = $global:LASTEXITCODE
    $omp = "::OMP::"
    $cleanPWD, $cleanPSWD = Get-PoshContext
    if ($global:POSH_TRANSIENT -eq $true) {
        @(Start-Utf8Process $omp "print transient --error=$global:OMP_ERRORCODE --pwd=""$cleanPWD"" --pswd=""$cleanPSWD"" --execution-time=$global:OMP_EXECUTIONTIME --config=""$Env:POSH_THEME"" --shell-version=""$env:SHELL_VERSION""") -join "`n"
        $global:POSH_TRANSIENT = $false
        return
    }
    if (Test-Path variable:/PSDebugContext) {
        @(Start-Utf8Process $omp "print debug --pwd=""$cleanPWD"" --pswd=""$cleanPSWD"" --config=""$Env:POSH_THEME""") -join "`n"
        return
    }
    $global:OMP_ERRORCODE = 0
    Initialize-ModuleSupport
    if ($lastCommandSuccess -eq $false) {
        # native app exit code
        if ($realLASTEXITCODE -is [int] -and $realLASTEXITCODE -ne 0) {
            $global:OMP_ERRORCODE = $realLASTEXITCODE
        }
        else {
            $global:OMP_ERRORCODE = 1
        }
    }

    # read stack count from current stack(if invoked from profile=right value, otherwise use the global variable set in Set-PoshPrompt(stack scoped to module))
    $stackCount = (Get-Location -stack).Count
    try {
        if ($global:OMP_GLOBAL_SESSIONSTATE -ne $null) {
            $stackCount = ($global:OMP_GLOBAL_SESSIONSTATE).path.locationstack('').count
        }
    }
    catch {}

    $global:OMP_EXECUTIONTIME = -1
    $history = Get-History -ErrorAction Ignore -Count 1
    if ($null -ne $history -and $null -ne $history.EndExecutionTime -and $null -ne $history.StartExecutionTime -and $global:OMP_LASTHISTORYID -ne $history.Id) {
        $global:OMP_EXECUTIONTIME = ($history.EndExecutionTime - $history.StartExecutionTime).TotalMilliseconds
        $global:OMP_LASTHISTORYID = $history.Id
    }
    Set-PoshContext
    $terminalWidth = $Host.UI.RawUI.WindowSize.Width
    $standardOut = @(Start-Utf8Process $omp "print primary --error=$global:OMP_ERRORCODE --pwd=""$cleanPWD"" --pswd=""$cleanPSWD"" --execution-time=$global:OMP_EXECUTIONTIME --stack-count=$stackCount --config=""$Env:POSH_THEME"" --shell-version=""$env:SHELL_VERSION"" --terminal-width=$terminalWidth")
    # make sure PSReadLine knows we have a multiline prompt
    $extraLines = ($standardOut | Measure-Object -Line).Lines - 1
    if ($extraLines -gt 0) {
        Set-PSReadlineOption -ExtraPromptLineCount $extraLines
    }
    # the output can be multiline, joining these ensures proper rendering by adding line breaks with `n
    $standardOut -join "`n"
    $global:LASTEXITCODE = $realLASTEXITCODE
}

<#
.SYNOPSIS
    Exports the current oh-my-posh theme
.DESCRIPTION
    By default the config is exported in json to the clipboard
.EXAMPLE
    Export-PoshTheme
    Current theme exported in json to clipboard
.EXAMPLE
    Export-PoshTheme -Format toml
    Current theme exported in toml to clipboard
.EXAMPLE
    Export-PoshTheme c:\temp\theme.toml toml
    Current theme exported in toml to c:\temp\theme.toml
.EXAMPLE
    Export-PoshTheme ~\theme.toml toml
    Current theme exported in toml to your home\theme.toml
#>
function global:Export-PoshTheme {
    param(
        [Parameter(Mandatory = $false)]
        [string]
        # The file path where the theme will be exported. If not provided, the config is copied to the clipboard by default.
        $FilePath,
        [Parameter(Mandatory = $false)]
        [ValidateSet('json', 'yaml', 'toml')]
        [string]
        # The format of the theme
        $Format = 'json'
    )

    $omp = "::OMP::"
    $configString = @(Start-Utf8Process $omp "config export --config=""$Env:POSH_THEME"" --format=$Format")
    # if no path, copy to clipboard by default
    if ($FilePath -ne "") {
        #https://stackoverflow.com/questions/3038337/powershell-resolve-path-that-might-not-exist
        $FilePath = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath($FilePath)
        [IO.File]::WriteAllLines($FilePath, $configString)
    }
    else {
        Set-Clipboard $configString
        Write-Output "Theme copied to clipboard"
    }
}

function global:Enable-PoshTooltips {
    Set-PSReadlineKeyHandler -Key SpaceBar -ScriptBlock {
        [Microsoft.PowerShell.PSConsoleReadLine]::Insert(' ')
        $position = $host.UI.RawUI.CursorPosition
        $omp = "::OMP::"
        $cleanPWD, $cleanPSWD = Get-PoshContext
        $command = $null
        $cursor = $null
        [Microsoft.PowerShell.PSConsoleReadLine]::GetBufferState([ref]$command, [ref]$cursor)
        $command = ($command -split " ")[0]
        $standardOut = @(Start-Utf8Process $omp "print tooltip --pwd=""$cleanPWD"" --pswd=""$cleanPSWD"" --config=""$Env:POSH_THEME"" --command=""$command"" --shell-version=""$env:SHELL_VERSION""")
        Write-Host $standardOut -NoNewline
        $host.UI.RawUI.CursorPosition = $position
    }
}

function global:Enable-PoshTransientPrompt {
    Set-PSReadlineKeyHandler -Key Enter -ScriptBlock {
        $global:POSH_TRANSIENT = $true
        $previousOutputEncoding = [Console]::OutputEncoding
        [Console]::OutputEncoding = [Text.Encoding]::UTF8
        try {
            [Microsoft.PowerShell.PSConsoleReadLine]::InvokePrompt()
        }
        finally {
            [Microsoft.PowerShell.PSConsoleReadLine]::AcceptLine()
            [Console]::OutputEncoding = $previousOutputEncoding
        }
    }
}

function global:Enable-PoshLineError {
    $omp = "::OMP::"
    $validLine = @(Start-Utf8Process $omp "print valid --config=""$Env:POSH_THEME""") -join "`n"
    $errorLine = @(Start-Utf8Process $omp "print error --config=""$Env:POSH_THEME""") -join "`n"
    Set-PSReadLineOption -PromptText $validLine, $errorLine
}

<#
 .SYNOPSIS
     Returns an ansi formatted hyperlink
     if name not set, uri is used as the name of the hyperlink
 .EXAMPLE
     Get-Hyperlink
 #>
function global:Get-Hyperlink {
    param(
        [Parameter(Mandatory, ValuefromPipeline = $True)]
        [string]$uri,
        [Parameter(ValuefromPipeline = $True)]
        [string]$name
    )
    $esc = [char]27
    if ("" -eq $name) {
        $name = $uri
    }
    if ($null -ne $env:WSL_DISTRO_NAME) {
        # wsl conversion if needed
        $uri = &wslpath -m $uri
    }
    return "$esc]8;;file://$uri$esc\$name$esc]8;;$esc\"
}

function global:Get-PoshThemes() {
    param(
        [Parameter(Mandatory = $false, HelpMessage = "The themes folder")]
        [string]
        $Path = $env:POSH_THEMES_PATH,
        [switch]
        [Parameter(Mandatory = $false, HelpMessage = "List themes path")]
        $List
    )

    if ($Path -eq "") {
        do {
            $temp = Read-Host 'Please enter the themes path'
        }
        while (-not (Test-Path -Path $temp))
        $Path = (Resolve-Path -Path $temp).ProviderPath
    }

    $logo = @'
   __  _____ _      ___  ___       ______         _      __
  / / |  _  | |     |  \/  |       | ___ \       | |     \ \
 / /  | | | | |__   | .  . |_   _  | |_/ /__  ___| |__    \ \
< <   | | | | '_ \  | |\/| | | | | |  __/ _ \/ __| '_ \    > >
 \ \  \ \_/ / | | | | |  | | |_| | | | | (_) \__ \ | | |  / /
  \_\  \___/|_| |_| \_|  |_/\__, | \_|  \___/|___/_| |_| /_/
                             __/ |
                            |___/
'@
    Write-Host $logo
    $themes = Get-ChildItem -Path "$Path\*" -Include '*.omp.json' | Sort-Object Name
    if ($List -eq $true) {
        $themes | Select-Object @{ Name = 'hyperlink'; Expression = { Get-Hyperlink -uri $_.fullname } } | Format-Table -HideTableHeaders
    }
    else {
        $omp = "::OMP::"
        $themes | ForEach-Object -Process {
            Write-Host "Theme: $(Get-Hyperlink -uri $_.fullname -name $_.BaseName.Replace('.omp', ''))`n"
            @(Start-Utf8Process $omp "print primary --config=""$($_.FullName)"" --pwd=""$PWD"" --shell pwsh")
            Write-Host "`n"
        }
    }
    Write-Host @"

Themes location: $(Get-Hyperlink -uri "$Path")

To change your theme, adjust the init script in $PROFILE.
Example:
  oh-my-posh init pwsh --config $Path/jandedobbeleer.omp.json | Invoke-Expression

"@
}
