package main

import (
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	mock "github.com/stretchr/testify/mock"
)

// MockedTracer is an autogenerated mock type for the tracer type
type MockedTracer struct {
	mock.Mock
}

// close provides a mock function with given fields:
func (_m *MockedTracer) close() {
	_m.Called()
}

// error provides a mock function with given fields: message
func (_m *MockedTracer) error(message string) {
	_m.Called(message)
}

// init provides a mock function with given fields: home
func (_m *MockedTracer) init(home string) {
	_m.Called(home)
}

// trace provides a mock function with given fields: start, function, args
func (_m *MockedTracer) trace(start time.Time, function string, args ...string) {
	_va := make([]interface{}, len(args))
	for _i := range args {
		_va[_i] = args[_i]
	}
	var _ca []interface{}
	_ca = append(_ca, start, function)
	_ca = append(_ca, _va...)
	_m.Called(_ca...)
}

func TestNormalHostName(t *testing.T) {
	hostName := "hello"
	assert.Equal(t, hostName, cleanHostName(hostName))
}

func TestHostNameWithLocal(t *testing.T) {
	hostName := "hello.local"
	assert.Equal(t, "hello", cleanHostName(hostName))
}

func TestHostNameWithLan(t *testing.T) {
	hostName := "hello.lan"
	cleanHostName := cleanHostName(hostName)
	assert.Equal(t, "hello", cleanHostName)
}

func TestWindowsPathWithDriveLetter(t *testing.T) {
	cases := []struct {
		Case     string
		CWD      string
		Expected string
	}{
		{Case: "C drive", CWD: `C:\Windows\`, Expected: `C:\Windows\`},
		{Case: "C drive lower case", CWD: `c:\Windows\`, Expected: `C:\Windows\`},
		{Case: "P drive lower case", CWD: `p:\some\`, Expected: `P:\some\`},
		{Case: "some drive lower case", CWD: `some:\some\`, Expected: `some:\some\`},
		{Case: "drive ending in c:", CWD: `src:\source\`, Expected: `src:\source\`},
		{Case: "registry drive", CWD: `HKLM:\SOFTWARE\magnetic:test\`, Expected: `HKLM:\SOFTWARE\magnetic:test\`},
	}
	for _, tc := range cases {
		tracer := &MockedTracer{}
		tracer.On("trace", mock.Anything, mock.Anything, mock.Anything)
		env := &environment{
			tracer: tracer,
			args: &args{
				PWD: &tc.CWD,
			},
		}
		assert.Equal(t, env.getcwd(), tc.Expected)
	}
}
