package generator

import (
	"bytes"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"text/template"

	"k8s.io/gengo/types"
	"k8s.io/klog/v2"
)

const (
	generatedTextWarning = "<!--- THIS FILE IS AUTOGENERATED!!! DO NOT EDIT!!! -->\n"
)

type Generator interface {
	Run() error
}

func NewGenerator(packageName string, requestedTypesList []string, headerTextFile string, outputFileName string, templateDirectory string) (Generator, error) {
	if packageName == "" {
		return nil, errors.New("a package name must be specified")
	}

	headerText, err := loadHeaderText(headerTextFile)
	if err != nil {
		return nil, fmt.Errorf("error loading header text: %v", err)
	}

	if err := checkTemplateDir(templateDirectory); err != nil {
		return nil, fmt.Errorf("invalid template directory: %v", err)
	}

	return &generator{
		packageName:       packageName,
		requestedTypes:    newStringSet(requestedTypesList),
		headerText:        headerText,
		outputFileName:    outputFileName,
		templateDirectory: templateDirectory,
	}, nil
}

// checkTemplateDir checks whether the template directory given exists and can be read
func checkTemplateDir(dir string) error {
	if dir == "" {
		return nil
	}
	path, err := filepath.Abs(dir)
	if err != nil {
		return err
	}

	if fi, err := os.Stat(path); err != nil {
		return fmt.Errorf("cannot read directory %q: %v", path, err)
	} else if !fi.IsDir() {
		return fmt.Errorf("path %q is not a directory", path)
	}
	return nil
}

// loadHeaderText loads the header text from the file if a filename was given
func loadHeaderText(fileName string) ([]byte, error) {
	if fileName == "" {
		return []byte{}, nil
	}

	headerText, err := ioutil.ReadFile(fileName)
	if err != nil {
		return nil, fmt.Errorf("error reading file: %v", err)
	}
	return headerText, nil
}

type generator struct {
	packageName       string
	requestedTypes    stringSet
	headerText        []byte
	outputFileName    string
	templateDirectory string
}

// Run runs the generation logic for the generator
func (g *generator) Run() error {
	typesToRender, err := g.loadTypesAndReferences()
	if err != nil {
		return fmt.Errorf("unable to load types: %v", err)
	}

	for typ := range typesToRender {
		klog.Infof("Rendering reference for type: %s", typ.Name.Name)
	}

	if err := g.renderOutput(typesToRender); err != nil {
		return fmt.Errorf("error rendering output: %v", err)
	}

	return nil
}

// loadTypes loads the package in the generator and returns a map
// of types and the types that reference them.
func (g *generator) loadTypesAndReferences() (map[*types.Type][]*types.Type, error) {
	pkg, err := loadPackage(g.packageName)
	if err != nil {
		return nil, fmt.Errorf("could not load package: %v", err)
	}

	typeReferences := findTypeReferences(pkg.Types)
	pkgTypeSet := newTypeSetFromStringMap(pkg.Types)

	if !g.requestedTypes.isEmpty() {
		typeReferences = filterToRequestedTypes(typeReferences, g.requestedTypes)
	}

	return filterToPackageTypes(typeReferences, pkgTypeSet), nil
}

func (g *generator) renderOutput(typesToRender map[*types.Type][]*types.Type) error {
	typeList := createTypeList(typesToRender)

	t, err := g.buildTemplate(typesToRender, typeList)
	if err != nil {
		return fmt.Errorf("error building template: %v", err)
	}

	// Create a buffer and render everything into that before writing out
	b := bytes.NewBuffer(append(g.headerText, []byte(generatedTextWarning)...))
	if err := t.ExecuteTemplate(b, "package", map[string]interface{}{
		"types": typeList,
	}); err != nil {
		return fmt.Errorf("error executing template: %v", err)
	}

	if err := g.writeOutput(b.Bytes()); err != nil {
		return fmt.Errorf("error writing output: %v", err)
	}
	return nil
}

func (g *generator) buildTemplate(typesToRender map[*types.Type][]*types.Type, typeList []*types.Type) (*template.Template, error) {
	knownTypes := newTypeSetFromList(typeList)
	t := template.New("").Funcs(map[string]interface{}{
		"aliasDisplayName": aliasDisplayNameFunc(knownTypes),
		"backtick":         backtick,
		"dereference":      tryDereference,
		"fieldEmbedded":    fieldEmbedded,
		"fieldName":        fieldName,
		"isOptionalMember": isOptionalMember,
		"linkForType":      linkForTypeFunc(knownTypes),
		"renderCommentsBR": renderCommentsBR,
		"renderCommentsLF": renderCommentsLF,
		"sortedTypes":      sortTypes,
		"typeDisplayName":  typeDisplayNameFunc(knownTypes),
		"typeReferences":   typeReferencesFunc(typesToRender, knownTypes),
		"visibleTypes":     visibleTypes,
	})

	var err error
	t, err = loadTemplatesInto(t, g.templateDirectory)
	if err != nil {
		return nil, fmt.Errorf("error loading templates: %v", err)
	}

	return t, nil
}

func (g *generator) writeOutput(content []byte) error {
	if g.outputFileName == "" {
		_, err := os.Stdout.Write(content)
		if err != nil {
			return fmt.Errorf("error writing output to stdout: %v", err)
		}
		klog.Infof("Rendered output written to stdout")
		return nil
	}

	if err := ioutil.WriteFile(g.outputFileName, content, 0600); err != nil {
		return fmt.Errorf("could not write file %q: %v", g.outputFileName, err)
	}
	klog.Infof("Rendered output written to %q", g.outputFileName)

	return nil
}
