// Copyright 2020-2022 The NATS Authors
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package cli

import (
	"context"
	"encoding/json"
	"fmt"
	"math"
	"os"
	"os/exec"
	"os/signal"
	"path/filepath"
	"sort"
	"strconv"
	"strings"
	"sync"
	"syscall"
	"time"

	"github.com/AlecAivazis/survey/v2"
	"github.com/choria-io/fisk"
	"github.com/dustin/go-humanize"
	"github.com/emicklei/dot"
	"github.com/google/go-cmp/cmp"
	"github.com/gosuri/uiprogress"
	"github.com/nats-io/jsm.go"
	"github.com/nats-io/jsm.go/api"
	"github.com/nats-io/nats.go"
	"github.com/xlab/tablewriter"
)

type streamCmd struct {
	stream           string
	force            bool
	json             bool
	msgID            int64
	retentionPolicyS string
	inputFile        string
	outFile          string
	filterSubject    string
	showAll          bool

	destination           string
	subjects              []string
	ack                   bool
	storage               string
	maxMsgLimit           int64
	maxMsgPerSubjectLimit int64
	maxBytesLimitString   string
	maxBytesLimit         int64
	maxAgeLimit           string
	maxMsgSizeString      string
	maxMsgSize            int64
	maxConsumers          int
	reportSortConsumers   bool
	reportSortMsgs        bool
	reportSortName        bool
	reportSortReverse     bool
	reportSortStorage     bool
	reportSort            string
	reportRaw             bool
	reportLimitCluster    string
	reportLeaderDistrib   bool
	discardPolicy         string
	validateOnly          bool
	backupDirectory       string
	showProgress          bool
	healthCheck           bool
	snapShotConsumers     bool
	dupeWindow            string
	replicas              int64
	placementCluster      string
	placementTags         []string
	peerName              string
	sources               []string
	mirror                string
	interactive           bool
	purgeKeep             uint64
	purgeSubject          string
	purgeSequence         uint64
	description           string
	repubSource           string
	repubDest             string
	repubHeadersOnly      bool
	allowRollup           bool
	allowRollupSet        bool
	denyDelete            bool
	denyDeleteSet         bool
	denyPurge             bool
	denyPurgeSet          bool
	allowDirect           bool
	allowDirectSet        bool
	allowMirrorDirect     bool
	allowMirrorDirectSet  bool
	discardPerSubj        bool
	discardPerSubjSet     bool
	showStateOnly         bool

	fServer    string
	fCluster   string
	fEmpty     bool
	fIdle      time.Duration
	fCreated   time.Duration
	fConsumers int
	fInvert    bool

	listNames    bool
	vwStartId    int
	vwStartDelta time.Duration
	vwPageSize   int
	vwRaw        bool
	vwSubject    string

	dryRun         bool
	selectedStream *jsm.Stream
	nc             *nats.Conn
	mgr            *jsm.Manager
}

type streamStat struct {
	Name      string
	Consumers int
	Msgs      int64
	Bytes     uint64
	Storage   string
	Template  string
	Cluster   *api.ClusterInfo
	LostBytes uint64
	LostMsgs  int
	Deleted   int
	Mirror    *api.StreamSourceInfo
	Sources   []*api.StreamSourceInfo
	Placement *api.Placement
}

func configureStreamCommand(app commandHost) {
	c := &streamCmd{msgID: -1}

	addCreateFlags := func(f *fisk.CmdClause, edit bool) {
		f.Flag("subjects", "Subjects that are consumed by the Stream").Default().StringsVar(&c.subjects)
		f.Flag("description", "Sets a contextual description for the stream").StringVar(&c.description)
		if !edit {
			f.Flag("storage", "Storage backend to use (file, memory)").EnumVar(&c.storage, "file", "f", "memory", "m")
		}
		f.Flag("replicas", "When clustered, how many replicas of the data to create").Int64Var(&c.replicas)
		f.Flag("tag", "Place the stream on servers that has specific tags (pass multiple times)").StringsVar(&c.placementTags)
		f.Flag("tags", "Backward compatibility only, use --tag").Hidden().StringsVar(&c.placementTags)
		f.Flag("cluster", "Place the stream on a specific cluster").StringVar(&c.placementCluster)
		f.Flag("ack", "Acknowledge publishes").Default("true").BoolVar(&c.ack)
		if !edit {
			f.Flag("retention", "Defines a retention policy (limits, interest, work)").EnumVar(&c.retentionPolicyS, "limits", "interest", "workq", "work")
		}
		f.Flag("discard", "Defines the discard policy (new, old)").EnumVar(&c.discardPolicy, "new", "old")
		f.Flag("discard-per-subject", "Sets the 'new' discard policy and applies it to every subject in the stream").IsSetByUser(&c.discardPerSubjSet).BoolVar(&c.discardPerSubj)
		f.Flag("max-age", "Maximum age of messages to keep").Default("").StringVar(&c.maxAgeLimit)
		f.Flag("max-bytes", "Maximum bytes to keep").PlaceHolder("BYTES").StringVar(&c.maxBytesLimitString)
		f.Flag("max-consumers", "Maximum number of consumers to allow").Default("-1").IntVar(&c.maxConsumers)
		f.Flag("max-msg-size", "Maximum size any 1 message may be").PlaceHolder("BYTES").StringVar(&c.maxMsgSizeString)
		f.Flag("max-msgs", "Maximum amount of messages to keep").Default("0").Int64Var(&c.maxMsgLimit)
		f.Flag("max-msgs-per-subject", "Maximum amount of messages to keep per subject").Default("0").Int64Var(&c.maxMsgPerSubjectLimit)
		f.Flag("dupe-window", "Duration of the duplicate message tracking window").Default("").StringVar(&c.dupeWindow)
		f.Flag("mirror", "Completely mirror another stream").StringVar(&c.mirror)
		f.Flag("source", "Source data from other Streams, merging into this one").PlaceHolder("STREAM").StringsVar(&c.sources)
		f.Flag("allow-rollup", "Allows roll-ups to be done by publishing messages with special headers").IsSetByUser(&c.allowRollupSet).BoolVar(&c.allowRollup)
		f.Flag("deny-delete", "Deny messages from being deleted via the API").IsSetByUser(&c.denyDeleteSet).BoolVar(&c.denyDelete)
		f.Flag("deny-purge", "Deny entire stream or subject purges via the API").IsSetByUser(&c.denyPurgeSet).BoolVar(&c.denyPurge)
		f.Flag("allow-direct", "Allows fast, direct, access to stream data via the direct get API").IsSetByUser(&c.allowDirectSet).BoolVar(&c.allowDirect)
		f.Flag("allow-mirror-direct", "Allows fast, direct, access to stream data via the direct get API on mirrors").IsSetByUser(&c.allowMirrorDirectSet).BoolVar(&c.allowMirrorDirect)

		f.Flag("json", "Produce JSON output").Short('j').UnNegatableBoolVar(&c.json)

		f.PreAction(c.parseLimitStrings)
	}

	str := app.Command("stream", "JetStream Stream management").Alias("str").Alias("st").Alias("ms").Alias("s")
	str.Flag("all", "When listing or selecting streams show all streams including system ones").Short('a').UnNegatableBoolVar(&c.showAll)
	addCheat("stream", str)

	strAdd := str.Command("add", "Create a new Stream").Alias("create").Alias("new").Action(c.addAction)
	strAdd.Arg("stream", "Stream name").StringVar(&c.stream)
	strAdd.Flag("config", "JSON file to read configuration from").ExistingFileVar(&c.inputFile)
	strAdd.Flag("validate", "Only validates the configuration against the official Schema").UnNegatableBoolVar(&c.validateOnly)
	strAdd.Flag("output", "Save configuration instead of creating").PlaceHolder("FILE").StringVar(&c.outFile)
	addCreateFlags(strAdd, false)
	strAdd.Flag("republish-source", "Republish messages to --republish-destination").PlaceHolder("SOURCE").StringVar(&c.repubSource)
	strAdd.Flag("republish-destination", "Republish destination for messages in --republish-source").PlaceHolder("DEST").StringVar(&c.repubDest)
	strAdd.Flag("republish-headers", "Republish only message headers, no bodies").UnNegatableBoolVar(&c.repubHeadersOnly)

	strLs := str.Command("ls", "List all known Streams").Alias("list").Alias("l").Action(c.lsAction)
	strLs.Flag("subject", "Limit the list to streams with matching subjects").StringVar(&c.filterSubject)
	strLs.Flag("names", "Show just the stream names").Short('n').UnNegatableBoolVar(&c.listNames)
	strLs.Flag("json", "Produce JSON output").Short('j').UnNegatableBoolVar(&c.json)

	strReport := str.Command("report", "Reports on Stream statistics").Action(c.reportAction)
	strReport.Flag("subject", "Limit the report to streams with matching subjects").StringVar(&c.filterSubject)
	strReport.Flag("cluster", "Limit report to streams within a specific cluster").StringVar(&c.reportLimitCluster)
	strReport.Flag("consumers", "Sort by number of Consumers").Short('o').UnNegatableBoolVar(&c.reportSortConsumers)
	strReport.Flag("messages", "Sort by number of Messages").Short('m').UnNegatableBoolVar(&c.reportSortMsgs)
	strReport.Flag("name", "Sort by Stream name").Short('n').UnNegatableBoolVar(&c.reportSortName)
	strReport.Flag("storage", "Sort by Storage type").Short('t').UnNegatableBoolVar(&c.reportSortStorage)
	strReport.Flag("raw", "Show un-formatted numbers").Short('r').UnNegatableBoolVar(&c.reportRaw)
	strReport.Flag("dot", "Produce a GraphViz graph of replication topology").StringVar(&c.outFile)
	strReport.Flag("leaders", "Show details about RAFT leaders").Short('l').UnNegatableBoolVar(&c.reportLeaderDistrib)

	strFind := str.Command("find", "Finds streams matching certain criteria").Alias("query").Action(c.findAction)
	strFind.Flag("server-name", "Display streams present on a regular expression matched server").StringVar(&c.fServer)
	strFind.Flag("cluster", "Display streams present on a regular expression matched cluster").StringVar(&c.fCluster)
	strFind.Flag("empty", "Display streams with no messages").UnNegatableBoolVar(&c.fEmpty)
	strFind.Flag("idle", "Display streams with no new messages or consumer deliveries for a period").PlaceHolder("DURATION").DurationVar(&c.fIdle)
	strFind.Flag("created", "Display streams created longer ago than duration").PlaceHolder("DURATION").DurationVar(&c.fCreated)
	strFind.Flag("consumers", "Display streams with fewer consumers than threshold").PlaceHolder("THRESHOLD").Default("-1").IntVar(&c.fConsumers)
	strFind.Flag("subject", "Filters Streams by those with interest matching a subject or wildcard").StringVar(&c.filterSubject)
	strFind.Flag("names", "Show just the stream names").Short('n').UnNegatableBoolVar(&c.listNames)
	strFind.Flag("invert", "Invert the check - before becomes after, with becomes without").BoolVar(&c.fInvert)

	strInfo := str.Command("info", "Stream information").Alias("nfo").Alias("i").Action(c.infoAction)
	strInfo.Arg("stream", "Stream to retrieve information for").StringVar(&c.stream)
	strInfo.Flag("json", "Produce JSON output").Short('j').UnNegatableBoolVar(&c.json)
	strInfo.Flag("state", "Shows only the stream state").UnNegatableBoolVar(&c.showStateOnly)

	strState := str.Command("state", "Stream state").Action(c.stateAction)
	strState.Arg("stream", "Stream to retrieve state information for").StringVar(&c.stream)
	strState.Flag("json", "Produce JSON output").Short('j').UnNegatableBoolVar(&c.json)

	strSubs := str.Command("subjects", "Query subjects held in a stream").Alias("subj").Action(c.subjectsAction)
	strSubs.Arg("stream", "Stream name").StringVar(&c.stream)
	strSubs.Arg("filter", "Limit the subjects to those matching a filter").Default(">").StringVar(&c.filterSubject)
	strSubs.Flag("json", "Produce JSON output").Short('j').UnNegatableBoolVar(&c.json)
	strSubs.Flag("sort", "Adjusts the sorting order (name, messages)").Default("messages").EnumVar(&c.reportSort, "name", "subjects", "messages", "count")
	strSubs.Flag("reverse", "Reverse sort servers").Short('R').UnNegatableBoolVar(&c.reportSortReverse)
	strSubs.Flag("names", "SList only subject names").BoolVar(&c.listNames)

	strEdit := str.Command("edit", "Edits an existing stream").Alias("update").Action(c.editAction)
	strEdit.Arg("stream", "Stream to retrieve edit").StringVar(&c.stream)
	strEdit.Flag("config", "JSON file to read configuration from").ExistingFileVar(&c.inputFile)
	strEdit.Flag("force", "Force edit without prompting").Short('f').UnNegatableBoolVar(&c.force)
	strEdit.Flag("interactive", "Edit the configuring using your editor").Short('i').BoolVar(&c.interactive)
	strEdit.Flag("dry-run", "Only shows differences, do not edit the stream").UnNegatableBoolVar(&c.dryRun)
	addCreateFlags(strEdit, true)

	strRm := str.Command("rm", "Removes a Stream").Alias("delete").Alias("del").Action(c.rmAction)
	strRm.Arg("stream", "Stream name").StringVar(&c.stream)
	strRm.Flag("force", "Force removal without prompting").Short('f').UnNegatableBoolVar(&c.force)

	strPurge := str.Command("purge", "Purge a Stream without deleting it").Action(c.purgeAction)
	strPurge.Arg("stream", "Stream name").StringVar(&c.stream)
	strPurge.Flag("json", "Produce JSON output").Short('j').UnNegatableBoolVar(&c.json)
	strPurge.Flag("force", "Force removal without prompting").Short('f').UnNegatableBoolVar(&c.force)
	strPurge.Flag("subject", "Limits the purge to a specific subject").PlaceHolder("SUBJECT").StringVar(&c.purgeSubject)
	strPurge.Flag("seq", "Purge up to but not including a specific message sequence").PlaceHolder("SEQUENCE").Uint64Var(&c.purgeSequence)
	strPurge.Flag("keep", "Keeps a certain number of messages after the purge").PlaceHolder("MESSAGES").Uint64Var(&c.purgeKeep)

	strCopy := str.Command("copy", "Creates a new Stream based on the configuration of another, does not copy data").Alias("cp").Action(c.cpAction)
	strCopy.Arg("source", "Source Stream to copy").Required().StringVar(&c.stream)
	strCopy.Arg("destination", "New Stream to create").Required().StringVar(&c.destination)
	addCreateFlags(strCopy, false)

	strRmMsg := str.Command("rmm", "Securely removes an individual message from a Stream").Action(c.rmMsgAction)
	strRmMsg.Arg("stream", "Stream name").StringVar(&c.stream)
	strRmMsg.Arg("id", "Message Sequence to remove").Int64Var(&c.msgID)
	strRmMsg.Flag("force", "Force removal without prompting").Short('f').UnNegatableBoolVar(&c.force)

	strView := str.Command("view", "View messages in a stream").Action(c.viewAction)
	strView.Arg("stream", "Stream name").StringVar(&c.stream)
	strView.Arg("size", "Page size").Default("10").IntVar(&c.vwPageSize)
	strView.Flag("id", "Start at a specific message Sequence").IntVar(&c.vwStartId)
	strView.Flag("since", "Delivers messages received since a duration like 1d3h5m2s").DurationVar(&c.vwStartDelta)
	strView.Flag("raw", "Show the raw data received").UnNegatableBoolVar(&c.vwRaw)
	strView.Flag("subject", "Filter the stream using a subject").StringVar(&c.vwSubject)

	strGet := str.Command("get", "Retrieves a specific message from a Stream").Action(c.getAction)
	strGet.Arg("stream", "Stream name").StringVar(&c.stream)
	strGet.Arg("id", "Message Sequence to retrieve").Int64Var(&c.msgID)
	strGet.Flag("last-for", "Retrieves the message for a specific subject").Short('S').PlaceHolder("SUBJECT").StringVar(&c.filterSubject)
	strGet.Flag("json", "Produce JSON output").Short('j').UnNegatableBoolVar(&c.json)

	strBackup := str.Command("backup", "Creates a backup of a Stream over the NATS network").Alias("snapshot").Action(c.backupAction)
	strBackup.Arg("stream", "Stream to backup").Required().StringVar(&c.stream)
	strBackup.Arg("target", "Directory to create the backup in").Required().StringVar(&c.backupDirectory)
	strBackup.Flag("progress", "Enables or disables progress reporting using a progress bar").Default("true").BoolVar(&c.showProgress)
	strBackup.Flag("check", "Checks the Stream for health prior to backup").UnNegatableBoolVar(&c.healthCheck)
	strBackup.Flag("consumers", "Enable or disable consumer backups").Default("true").BoolVar(&c.snapShotConsumers)

	strRestore := str.Command("restore", "Restore a Stream over the NATS network").Action(c.restoreAction)
	strRestore.Arg("file", "The directory holding the backup to restore").Required().ExistingDirVar(&c.backupDirectory)
	strRestore.Flag("progress", "Enables or disables progress reporting using a progress bar").Default("true").BoolVar(&c.showProgress)
	strRestore.Flag("config", "Load a different configuration when restoring the stream").ExistingFileVar(&c.inputFile)
	strRestore.Flag("cluster", "Place the stream in a specific cluster").StringVar(&c.placementCluster)
	strRestore.Flag("tag", "Place the stream on servers that has specific tags (pass multiple times)").StringsVar(&c.placementTags)

	strSeal := str.Command("seal", "Seals a stream preventing further updates").Action(c.sealAction)
	strSeal.Arg("stream", "The name of the Stream to seal").Required().StringVar(&c.stream)
	strSeal.Flag("force", "Force sealing without prompting").Short('f').UnNegatableBoolVar(&c.force)

	strCluster := str.Command("cluster", "Manages a clustered Stream").Alias("c")
	strClusterDown := strCluster.Command("step-down", "Force a new leader election by standing down the current leader").Alias("stepdown").Alias("sd").Alias("elect").Alias("down").Alias("d").Action(c.leaderStandDown)
	strClusterDown.Arg("stream", "Stream to act on").StringVar(&c.stream)

	strClusterRemovePeer := strCluster.Command("peer-remove", "Removes a peer from the Stream cluster").Alias("pr").Action(c.removePeer)
	strClusterRemovePeer.Arg("stream", "The stream to act on").StringVar(&c.stream)
	strClusterRemovePeer.Arg("peer", "The name of the peer to remove").StringVar(&c.peerName)
}

func init() {
	registerCommand("stream", 16, configureStreamCommand)
}

func (c *streamCmd) subjectsAction(_ *fisk.ParseContext) (err error) {
	asked := c.connectAndAskStream()

	subs, err := c.mgr.StreamContainedSubjects(c.stream, c.filterSubject)
	if err != nil {
		return err
	}

	if c.json {
		printJSON(subs)
		return nil
	}

	if asked {
		fmt.Println()
	}

	if len(subs) == 0 {
		fmt.Printf("No subjects found matching %s\n", c.filterSubject)
		return nil
	}

	var longest int
	var most uint64
	var names []string

	for s, c := range subs {
		names = append(names, s)
		if len(s) > longest {
			longest = len(s)
		}
		if c > most {
			most = c
		}
	}

	cols := 1
	format := fmt.Sprintf("  %%%ds: %%s\n", longest)
	countWidth := len(humanize.Comma(int64(most)))
	switch {
	case longest+countWidth < 20:
		cols = 3
		format = fmt.Sprintf("  %%20s: %%%ds %%20s: %%%ds %%20s: %%%ds\n", countWidth, countWidth, countWidth)
	case longest+countWidth < 30:
		cols = 2
		format = fmt.Sprintf("  %%30s: %%%ds %%30s: %%%ds\n", countWidth, countWidth)
	}

	sort.Slice(names, func(i, j int) bool {
		if c.reportSort == "name" || c.reportSort == "subjects" {
			return c.boolReverse(names[i] < names[j])
		} else {
			return c.boolReverse(subs[names[i]] < subs[names[j]])
		}
	})

	if c.listNames {
		for _, n := range names {
			fmt.Println(n)
		}
		return
	}

	sliceGroups(names, cols, func(g []string) {
		if cols == 1 {
			fmt.Printf(format, g[0], humanize.Comma(int64(subs[g[0]])))
		} else if cols == 2 {
			fmt.Printf(format, g[0], humanize.Comma(int64(subs[g[0]])), g[1], humanize.Comma(int64(subs[g[1]])))
		} else {
			fmt.Printf(format, g[0], humanize.Comma(int64(subs[g[0]])), g[1], humanize.Comma(int64(subs[g[1]])), g[2], humanize.Comma(int64(subs[g[2]])))
		}
	})

	return nil
}

func (c *streamCmd) parseLimitStrings(_ *fisk.ParseContext) (err error) {
	if c.maxBytesLimitString != "" {
		c.maxBytesLimit, err = parseStringAsBytes(c.maxBytesLimitString)
		if err != nil {
			return err
		}
	}

	if c.maxMsgSizeString != "" {
		c.maxMsgSize, err = parseStringAsBytes(c.maxMsgSizeString)
		if err != nil {
			return err
		}
	}

	return nil
}

func (c *streamCmd) findAction(_ *fisk.ParseContext) (err error) {
	c.nc, c.mgr, err = prepareHelper("", natsOpts()...)
	if err != nil {
		return fmt.Errorf("setup failed: %v", err)
	}

	opts := []jsm.StreamQueryOpt{}
	if c.fServer != "" {
		opts = append(opts, jsm.StreamQueryServerName(c.fServer))
	}
	if c.fCluster != "" {
		opts = append(opts, jsm.StreamQueryClusterName(c.fCluster))
	}
	if c.fEmpty {
		opts = append(opts, jsm.StreamQueryWithoutMessages())
	}
	if c.fIdle > 0 {
		opts = append(opts, jsm.StreamQueryIdleLongerThan(c.fIdle))
	}
	if c.fCreated > 0 {
		opts = append(opts, jsm.StreamQueryOlderThan(c.fCreated))
	}
	if c.fConsumers >= 0 {
		opts = append(opts, jsm.StreamQueryFewerConsumersThan(uint(c.fConsumers)))
	}
	if c.fInvert {
		opts = append(opts, jsm.StreamQueryInvert())
	}
	if c.filterSubject != "" {
		opts = append(opts, jsm.StreamQuerySubjectWildcard(c.filterSubject))
	}

	found, err := c.mgr.QueryStreams(opts...)
	if err != nil {
		return err
	}

	out := ""
	switch {
	case c.json:
		out, err = toJSON(found)
	case c.listNames:
		out = c.renderStreamsAsList(found)
	default:
		out, err = c.renderStreamsAsTable(found)
	}
	if err != nil {
		return err
	}

	fmt.Println(out)

	return nil
}

func (c *streamCmd) loadStream(stream string) (*jsm.Stream, error) {
	if c.selectedStream != nil && c.selectedStream.Name() == stream {
		return c.selectedStream, nil
	}

	return c.mgr.LoadStream(stream)
}

func (c *streamCmd) leaderStandDown(_ *fisk.ParseContext) error {
	c.connectAndAskStream()

	stream, err := c.loadStream(c.stream)
	if err != nil {
		return err
	}

	info, err := stream.LatestInformation()
	if err != nil {
		return err
	}

	if info.Cluster == nil {
		return fmt.Errorf("stream %q is not clustered", stream.Name())
	}

	leader := info.Cluster.Leader
	if leader == "" {
		return fmt.Errorf("stream has no current leader")
	}

	log.Printf("Requesting leader step down of %q in a %d peer RAFT group", leader, len(info.Cluster.Replicas)+1)
	err = stream.LeaderStepDown()
	if err != nil {
		return err
	}

	ctr := 0
	start := time.Now()
	for range time.NewTicker(500 * time.Millisecond).C {
		if ctr == 10 {
			return fmt.Errorf("stream did not elect a new leader in time")
		}
		ctr++

		info, err = stream.Information()
		if err != nil {
			log.Printf("Failed to retrieve Stream State: %s", err)
			continue
		}

		if info.Cluster.Leader == "" {
			log.Printf("No leader elected")
			continue
		}

		if info.Cluster.Leader != leader {
			log.Printf("New leader elected %q", info.Cluster.Leader)
			break
		}
	}

	if info.Cluster.Leader == leader {
		log.Printf("Leader did not change after %s", time.Since(start).Round(time.Millisecond))
	}

	fmt.Println()
	return c.showStream(stream)
}

func (c *streamCmd) removePeer(_ *fisk.ParseContext) error {
	c.connectAndAskStream()

	stream, err := c.loadStream(c.stream)
	if err != nil {
		return err
	}

	info, err := stream.Information()
	if err != nil {
		return err
	}

	if info.Cluster == nil {
		return fmt.Errorf("stream %q is not clustered", stream.Name())
	}

	if c.peerName == "" {
		peerNames := []string{info.Cluster.Leader}
		for _, r := range info.Cluster.Replicas {
			peerNames = append(peerNames, r.Name)
		}

		err = askOne(&survey.Select{
			Message: "Select a Peer",
			Options: peerNames,
		}, &c.peerName)
		if err != nil {
			return err
		}
	}

	log.Printf("Removing peer %q", c.peerName)

	err = stream.RemoveRAFTPeer(c.peerName)
	if err != nil {
		return err
	}

	log.Printf("Requested removal of peer %q", c.peerName)

	return nil
}

func (c *streamCmd) viewAction(_ *fisk.ParseContext) error {
	if c.vwPageSize > 25 {
		c.vwPageSize = 25
	}

	c.connectAndAskStream()

	str, err := c.loadStream(c.stream)
	if err != nil {
		return err
	}

	if str.Retention() == api.WorkQueuePolicy {
		return fmt.Errorf("work queue stream contents can not be viewed")
	}

	pops := []jsm.PagerOption{
		jsm.PagerSize(c.vwPageSize),
	}

	switch {
	case c.vwStartDelta > 0:
		pops = append(pops, jsm.PagerStartDelta(c.vwStartDelta))
	case c.vwStartId > 0:
		pops = append(pops, jsm.PagerStartId(c.vwStartId))
	}

	if c.vwSubject != "" {
		pops = append(pops, jsm.PagerFilterSubject(c.vwSubject))
	}

	pgr, err := str.PageContents(pops...)
	if err != nil {
		return err
	}
	defer pgr.Close()

	ctx, cancel := context.WithCancel(ctx)
	sigs := make(chan os.Signal, 1)
	signal.Notify(sigs, syscall.SIGINT, syscall.SIGTERM, syscall.SIGQUIT)

	go func() {
		select {
		case <-ctx.Done():
			return
		case <-sigs:
			cancel()
		}
	}()

	for {
		msg, last, err := pgr.NextMsg(ctx)
		if err != nil && last {
			log.Println("Reached apparent end of data")
			return nil
		}
		if err != nil {
			return err
		}

		switch {
		case c.vwRaw:
			fmt.Println(string(msg.Data))
		default:
			meta, err := jsm.ParseJSMsgMetadata(msg)
			if err == nil {
				fmt.Printf("[%d] Subject: %s Received: %s\n", meta.StreamSequence(), msg.Subject, meta.TimeStamp().Format(time.RFC3339))
			} else {
				fmt.Printf("Subject: %s Reply: %s\n", msg.Subject, msg.Reply)
			}

			if len(msg.Header) > 0 {
				fmt.Println()
				for k, vs := range msg.Header {
					for _, v := range vs {
						fmt.Printf("  %s: %s\n", k, v)
					}
				}
			}

			fmt.Println()
			if len(msg.Data) == 0 {
				fmt.Println("nil body")
			} else {
				fmt.Println(string(msg.Data))
				if !strings.HasSuffix(string(msg.Data), "\n") {
					fmt.Println()
				}
			}

		}

		if last {
			next := false
			askOne(&survey.Confirm{Message: "Next Page?", Default: true}, &next)
			if !next {
				return nil
			}
		}
	}
}

func (c *streamCmd) sealAction(_ *fisk.ParseContext) error {
	c.connectAndAskStream()

	if !c.force {
		ok, err := askConfirmation(fmt.Sprintf("Really seal Stream %s, sealed streams can not be unsealed or modified", c.stream), false)
		fisk.FatalIfError(err, "could not obtain confirmation")

		if !ok {
			return nil
		}
	}

	stream, err := c.loadStream(c.stream)
	fisk.FatalIfError(err, "could not seal Stream")

	stream.Seal()
	fisk.FatalIfError(err, "could not seal Stream")

	return c.showStream(stream)
}

func (c *streamCmd) restoreAction(_ *fisk.ParseContext) error {
	_, mgr, err := prepareHelper("", natsOpts()...)
	fisk.FatalIfError(err, "setup failed")

	var bm api.JSApiStreamRestoreRequest
	bmj, err := os.ReadFile(filepath.Join(c.backupDirectory, "backup.json"))
	fisk.FatalIfError(err, "restore failed")
	err = json.Unmarshal(bmj, &bm)
	fisk.FatalIfError(err, "restore failed")

	var cfg *api.StreamConfig

	known, err := mgr.IsKnownStream(bm.Config.Name)
	fisk.FatalIfError(err, "Could not check if the stream already exist")
	if known {
		fisk.Fatalf("Stream %q already exist", bm.Config.Name)
	}

	var progress *uiprogress.Bar
	var bps uint64

	cb := func(p jsm.RestoreProgress) {
		bps = p.BytesPerSecond()

		if progress == nil {
			progress = uiprogress.AddBar(p.ChunksToSend()).AppendCompleted().PrependFunc(func(b *uiprogress.Bar) string {
				return humanize.IBytes(bps) + "/s"
			})
			progress.Width = progressWidth()
		}

		progress.Set(int(p.ChunksSent()))
	}

	var opts []jsm.SnapshotOption

	if c.showProgress {
		uiprogress.Start()
		opts = append(opts, jsm.RestoreNotify(cb))
	} else {
		opts = append(opts, jsm.SnapshotDebug())
	}

	if c.inputFile != "" {
		cfg, err := c.loadConfigFile(c.inputFile)
		if err != nil {
			return err
		}

		// we need to confirm this new config has the same stream
		// name as the snapshot else the server state can get confused
		// see https://github.com/nats-io/nats-server/issues/2850
		if bm.Config.Name != cfg.Name {
			return fmt.Errorf("stream names may not be changed during restore")
		}
	} else {
		cfg = &bm.Config
	}

	if c.placementCluster != "" || len(c.placementTags) > 0 {
		cfg.Placement = &api.Placement{
			Cluster: c.placementCluster,
			Tags:    c.placementTags,
		}
	}

	opts = append(opts, jsm.RestoreConfiguration(*cfg))

	fmt.Printf("Starting restore of Stream %q from file %q\n\n", bm.Config.Name, c.backupDirectory)

	fp, _, err := mgr.RestoreSnapshotFromDirectory(ctx, bm.Config.Name, c.backupDirectory, opts...)
	fisk.FatalIfError(err, "restore failed")
	if c.showProgress {
		progress.Set(int(fp.ChunksSent()))
		uiprogress.Stop()
	}

	fmt.Println()
	fmt.Printf("Restored stream %q in %v\n", bm.Config.Name, fp.EndTime().Sub(fp.StartTime()).Round(time.Second))
	fmt.Println()

	stream, err := mgr.LoadStream(bm.Config.Name)
	fisk.FatalIfError(err, "could not request Stream info")
	err = c.showStream(stream)
	fisk.FatalIfError(err, "could not show stream")

	return nil
}

func backupStream(stream *jsm.Stream, showProgress bool, consumers bool, check bool, target string) error {
	first := true
	inprogress := true
	pmu := sync.Mutex{}
	var bar *uiprogress.Bar
	var bps uint64
	var progress *uiprogress.Progress
	expected := 1
	timedOut := false

	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	timeout := time.AfterFunc(5*time.Second, func() {
		cancel()
		timedOut = true
	})

	var received uint32

	cb := func(p jsm.SnapshotProgress) {
		if bar == nil && showProgress {
			if p.BytesExpected() > 0 {
				expected = int(p.BytesExpected())
			}
			bar = progress.AddBar(expected).AppendCompleted().PrependFunc(func(b *uiprogress.Bar) string {
				return humanize.IBytes(bps) + "/s"
			})
			bar.Width = progressWidth()
		}

		if first {
			fmt.Printf("Starting backup of Stream %q with %s\n", stream.Name(), humanize.IBytes(p.BytesExpected()))
			if showProgress {
				fmt.Println()
			}

			if p.HealthCheck() {
				fmt.Printf("Health Check was requested, this can take a long time without progress reports\n\n")
			}

			first = false
		}

		if p.ChunksReceived() != received {
			timeout.Reset(5 * time.Second)
			received = p.ChunksReceived()
		}

		bps = p.BytesPerSecond()

		if showProgress {
			bar.Set(int(p.BytesReceived()))
		}

		if p.Finished() {
			pmu.Lock()
			if inprogress {
				if showProgress {
					progress.Stop()
				}

				inprogress = false
			}
			pmu.Unlock()
		}
	}

	var opts []jsm.SnapshotOption

	if consumers {
		opts = append(opts, jsm.SnapshotConsumers())
	}

	if showProgress {
		progress = uiprogress.New()
		progress.Start()
	}

	opts = append(opts, jsm.SnapshotNotify(cb))

	if check {
		opts = append(opts, jsm.SnapshotHealthCheck())
	}

	fp, err := stream.SnapshotToDirectory(ctx, target, opts...)
	if err != nil {
		return err
	}

	pmu.Lock()
	if showProgress && inprogress {
		bar.Set(int(fp.BytesReceived()))
		uiprogress.Stop()
		inprogress = false
	}
	pmu.Unlock()

	fmt.Println()

	if timedOut {
		return fmt.Errorf("backup timed out after receiving no data for a long period")
	}

	fmt.Printf("Received %s compressed data in %s chunks for stream %q in %v, %s uncompressed \n", humanize.IBytes(fp.BytesReceived()), humanize.Comma(int64(fp.ChunksReceived())), stream.Name(), fp.EndTime().Sub(fp.StartTime()).Round(time.Millisecond), humanize.IBytes(fp.UncompressedBytesReceived()))

	return nil
}

func (c *streamCmd) backupAction(_ *fisk.ParseContext) error {
	var err error

	c.nc, c.mgr, err = prepareHelper("", natsOpts()...)
	fisk.FatalIfError(err, "setup failed")

	stream, err := c.loadStream(c.stream)
	if err != nil {
		return err
	}

	err = backupStream(stream, c.showProgress, c.snapShotConsumers, c.healthCheck, c.backupDirectory)
	fisk.FatalIfError(err, "snapshot failed")

	return nil
}

func (c *streamCmd) reportAction(_ *fisk.ParseContext) error {
	_, mgr, err := prepareHelper("", natsOpts()...)
	fisk.FatalIfError(err, "setup failed")

	if !c.json {
		fmt.Print("Obtaining Stream stats\n\n")
	}

	stats := []streamStat{}
	leaders := make(map[string]*raftLeader)
	showReplication := false
	var filter *jsm.StreamNamesFilter

	if c.filterSubject != "" {
		filter = &jsm.StreamNamesFilter{Subject: c.filterSubject}
	}

	dg := dot.NewGraph(dot.Directed)
	dg.Label("Stream Replication Structure")

	err = mgr.EachStream(filter, func(stream *jsm.Stream) {
		info, err := stream.LatestInformation()
		fisk.FatalIfError(err, "could not get stream info for %s", stream.Name())

		if info.Cluster != nil {
			if c.reportLimitCluster != "" && info.Cluster.Name != c.reportLimitCluster {
				return
			}

			if info.Cluster.Leader != "" {
				_, ok := leaders[info.Cluster.Leader]
				if !ok {
					leaders[info.Cluster.Leader] = &raftLeader{name: info.Cluster.Leader, cluster: info.Cluster.Name}
				}
				leaders[info.Cluster.Leader].groups++
			}
		}

		deleted := info.State.NumDeleted
		// backward compat with servers that predate the num_deleted response
		if len(info.State.Deleted) > 0 {
			deleted = len(info.State.Deleted)
		}
		s := streamStat{
			Name:      info.Config.Name,
			Consumers: info.State.Consumers,
			Msgs:      int64(info.State.Msgs),
			Bytes:     info.State.Bytes,
			Storage:   info.Config.Storage.String(),
			Template:  info.Config.Template,
			Cluster:   info.Cluster,
			Deleted:   deleted,
			Mirror:    info.Mirror,
			Sources:   info.Sources,
			Placement: info.Config.Placement,
		}
		if info.State.Lost != nil {
			s.LostBytes = info.State.Lost.Bytes
			s.LostMsgs = len(info.State.Lost.Msgs)
		}

		if len(info.Config.Sources) > 0 {
			showReplication = true
			node, ok := dg.FindNodeById(info.Config.Name)
			if !ok {
				node = dg.Node(info.Config.Name)
			}
			for _, source := range info.Config.Sources {
				snode, ok := dg.FindNodeById(source.Name)
				if !ok {
					snode = dg.Node(source.Name)
				}
				edge := dg.Edge(snode, node).Attr("color", "green")
				if source.FilterSubject != "" {
					edge.Label(source.FilterSubject)
				}
			}
		}

		if info.Config.Mirror != nil {
			showReplication = true
			node, ok := dg.FindNodeById(info.Config.Name)
			if !ok {
				node = dg.Node(info.Config.Name)
			}
			mnode, ok := dg.FindNodeById(info.Config.Mirror.Name)
			if !ok {
				mnode = dg.Node(info.Config.Mirror.Name)
			}
			dg.Edge(mnode, node).Attr("color", "blue").Label("Mirror")
		}

		stats = append(stats, s)
	})
	if err != nil {
		return err
	}

	if len(stats) == 0 {
		if !c.json {
			fmt.Println("No Streams defined")
		}
		return nil
	}

	if c.reportSortConsumers {
		sort.Slice(stats, func(i, j int) bool { return stats[i].Consumers < stats[j].Consumers })
	} else if c.reportSortMsgs {
		sort.Slice(stats, func(i, j int) bool { return stats[i].Msgs < stats[j].Msgs })
	} else if c.reportSortName {
		sort.Slice(stats, func(i, j int) bool { return stats[i].Name < stats[j].Name })
	} else if c.reportSortStorage {
		sort.Slice(stats, func(i, j int) bool { return stats[i].Storage < stats[j].Storage })
	} else {
		sort.Slice(stats, func(i, j int) bool { return stats[i].Bytes < stats[j].Bytes })
	}

	c.renderStreams(stats)

	if showReplication {
		c.renderReplication(stats)

		if c.outFile != "" {
			os.WriteFile(c.outFile, []byte(dg.String()), 0644)
		}
	}

	if c.reportLeaderDistrib && len(leaders) > 0 {
		renderRaftLeaders(leaders, "Streams")
	}

	return nil
}

func (c *streamCmd) renderReplication(stats []streamStat) {
	table := newTableWriter("Replication Report")
	table.AddHeaders("Stream", "Kind", "API Prefix", "Source Stream", "Active", "Lag", "Error")

	for _, s := range stats {
		if len(s.Sources) == 0 && s.Mirror == nil {
			continue
		}

		if s.Mirror != nil {
			apierr := ""
			if s.Mirror.Error != nil {
				apierr = s.Mirror.Error.Error()
			}

			eApiPrefix := ""
			if s.Mirror.External != nil {
				eApiPrefix = s.Mirror.External.ApiPrefix
			}

			if c.reportRaw {
				table.AddRow(s.Name, "Mirror", eApiPrefix, s.Mirror.Name, s.Mirror.Active, s.Mirror.Lag, apierr)
			} else {
				table.AddRow(s.Name, "Mirror", eApiPrefix, s.Mirror.Name, humanizeDuration(s.Mirror.Active), humanize.Comma(int64(s.Mirror.Lag)), apierr)
			}
		}

		for _, source := range s.Sources {
			apierr := ""
			if source != nil && source.Error != nil {
				apierr = source.Error.Error()
			}

			eApiPrefix := ""
			if source.External != nil {
				eApiPrefix = source.External.ApiPrefix
			}

			if c.reportRaw {
				table.AddRow(s.Name, "Source", eApiPrefix, source.Name, source.Active, source.Lag, apierr)
			} else {
				table.AddRow(s.Name, "Source", eApiPrefix, source.Name, humanizeDuration(source.Active), humanize.Comma(int64(source.Lag)), apierr)
			}

		}
	}
	fmt.Println(table.Render())
}

func (c *streamCmd) renderStreams(stats []streamStat) {
	table := newTableWriter("Stream Report")
	table.AddHeaders("Stream", "Storage", "Placement", "Consumers", "Messages", "Bytes", "Lost", "Deleted", "Replicas")

	for _, s := range stats {
		lost := "0"
		placement := ""
		if s.Placement != nil {
			if s.Placement.Cluster != "" {
				placement = fmt.Sprintf("cluster: %s ", s.Placement.Cluster)
			}
			if len(s.Placement.Tags) > 0 {
				placement = fmt.Sprintf("%stags: %s", placement, strings.Join(s.Placement.Tags, ", "))
			}
		}

		if c.reportRaw {
			if s.LostMsgs > 0 {
				lost = fmt.Sprintf("%d (%d)", s.LostMsgs, s.LostBytes)
			}
			table.AddRow(s.Name, s.Storage, placement, s.Consumers, s.Msgs, s.Bytes, lost, s.Deleted, renderCluster(s.Cluster))
		} else {
			if s.LostMsgs > 0 {
				lost = fmt.Sprintf("%s (%s)", humanize.Comma(int64(s.LostMsgs)), humanize.IBytes(s.LostBytes))
			}
			table.AddRow(s.Name, s.Storage, placement, s.Consumers, humanize.Comma(s.Msgs), humanize.IBytes(s.Bytes), lost, s.Deleted, renderCluster(s.Cluster))
		}
	}

	fmt.Println(table.Render())
}

func (c *streamCmd) loadConfigFile(file string) (*api.StreamConfig, error) {
	f, err := os.ReadFile(file)
	if err != nil {
		return nil, err
	}

	var cfg api.StreamConfig

	// there is a chance that this is a `nats s info --json` output
	// which is a StreamInfo, so we detect if this is one of those
	// by checking if there's a config key then extract that, else
	// we try loading it as a StreamConfig

	var nfo map[string]any
	err = json.Unmarshal(f, &nfo)
	if err != nil {
		return nil, err
	}

	_, ok := nfo["config"]
	if ok {
		var nfo api.StreamInfo
		err = json.Unmarshal(f, &nfo)
		if err != nil {
			return nil, err
		}
		cfg = nfo.Config
	} else {
		err = json.Unmarshal(f, &cfg)
		if err != nil {
			return nil, err
		}
	}

	if cfg.Name != c.stream && c.stream != "" {
		cfg.Name = c.stream
	}

	return &cfg, nil
}

func (c *streamCmd) copyAndEditStream(cfg api.StreamConfig, pc *fisk.ParseContext) (api.StreamConfig, error) {
	var err error

	if c.inputFile != "" {
		cfg, err := c.loadConfigFile(c.inputFile)
		if err != nil {
			return api.StreamConfig{}, err
		}

		if cfg.Name == "" {
			cfg.Name = c.stream
		}

		return *cfg, nil
	}

	cfg.NoAck = !c.ack

	if c.discardPolicy != "" {
		cfg.Discard = c.discardPolicyFromString()
	}

	if len(c.subjects) > 0 {
		cfg.Subjects = splitCLISubjects(c.subjects)
	}

	if c.storage != "" {
		cfg.Storage = c.storeTypeFromString(c.storage)
	}

	if c.retentionPolicyS != "" {
		cfg.Retention = c.retentionPolicyFromString()
	}

	if c.maxBytesLimit != 0 {
		cfg.MaxBytes = c.maxBytesLimit
	}

	if c.maxMsgLimit != 0 {
		cfg.MaxMsgs = c.maxMsgLimit
	}

	if c.maxMsgPerSubjectLimit != 0 {
		cfg.MaxMsgsPer = c.maxMsgPerSubjectLimit
	}

	if c.maxAgeLimit != "" {
		cfg.MaxAge, err = parseDurationString(c.maxAgeLimit)
		if err != nil {
			return api.StreamConfig{}, fmt.Errorf("invalid maximum age limit format: %v", err)
		}
	}

	if c.maxMsgSize != 0 {
		cfg.MaxMsgSize = int32(c.maxMsgSize)
	}

	if c.maxConsumers != -1 {
		cfg.MaxConsumers = c.maxConsumers
	}

	if c.dupeWindow != "" {
		dw, err := parseDurationString(c.dupeWindow)
		if err != nil {
			return api.StreamConfig{}, fmt.Errorf("invalid duplicate window: %v", err)
		}
		cfg.Duplicates = dw
	}

	if c.replicas != 0 {
		cfg.Replicas = int(c.replicas)
	}

	if cfg.Placement == nil {
		cfg.Placement = &api.Placement{}
	}

	if cfg.Placement == nil {
		cfg.Placement = &api.Placement{}
	}

	if c.placementCluster != "" {
		cfg.Placement.Cluster = c.placementCluster
	}

	if len(c.placementTags) > 0 {
		cfg.Placement.Tags = c.placementTags
	}

	if cfg.Placement.Cluster == "" && len(cfg.Placement.Tags) == 0 {
		cfg.Placement = nil
	}

	if len(c.sources) > 0 || c.mirror != "" {
		return cfg, fmt.Errorf("cannot edit mirrors or sources using the CLI, use --config instead")
	}

	if c.description != "" {
		cfg.Description = c.description
	}

	if c.allowRollupSet {
		cfg.RollupAllowed = c.allowRollup
	}

	if c.denyPurgeSet {
		cfg.DenyPurge = c.denyPurge
	}

	if c.denyDeleteSet {
		cfg.DenyDelete = c.denyDelete
	}

	if c.allowDirectSet {
		cfg.AllowDirect = c.allowDirect
	}

	if c.allowMirrorDirectSet {
		cfg.MirrorDirect = c.allowMirrorDirectSet
	}

	if c.discardPerSubjSet {
		cfg.DiscardNewPer = c.discardPerSubj
	}

	return cfg, nil
}

func (c *streamCmd) interactiveEdit(cfg api.StreamConfig) (api.StreamConfig, error) {
	editor := os.Getenv("EDITOR")
	if editor == "" {
		return api.StreamConfig{}, fmt.Errorf("set EDITOR environment variable to your chosen editor")
	}

	cj, err := json.MarshalIndent(cfg, "", "  ")
	if err != nil {
		return api.StreamConfig{}, fmt.Errorf("could not create temporary file: %s", err)
	}

	tfile, err := os.CreateTemp("", "")
	if err != nil {
		return api.StreamConfig{}, fmt.Errorf("could not create temporary file: %s", err)
	}
	defer os.Remove(tfile.Name())

	_, err = fmt.Fprint(tfile, string(cj))
	if err != nil {
		return api.StreamConfig{}, fmt.Errorf("could not create temporary file: %s", err)
	}

	tfile.Close()

	cmd := exec.Command(editor, tfile.Name())
	cmd.Stdin = os.Stdin
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	err = cmd.Run()
	if err != nil {
		return api.StreamConfig{}, fmt.Errorf("could not create temporary file: %s", err)
	}

	ncfg, err := c.loadConfigFile(tfile.Name())
	if err != nil {
		return api.StreamConfig{}, fmt.Errorf("could not create temporary file: %s", err)
	}

	return *ncfg, nil
}

func (c *streamCmd) editAction(pc *fisk.ParseContext) error {
	c.connectAndAskStream()

	sourceStream, err := c.loadStream(c.stream)
	fisk.FatalIfError(err, "could not request Stream %s configuration", c.stream)

	// lazy deep copy
	input := sourceStream.Configuration()
	ij, err := json.Marshal(input)
	if err != nil {
		return err
	}
	var cfg api.StreamConfig
	err = json.Unmarshal(ij, &cfg)
	if err != nil {
		return err
	}

	if c.interactive {
		cfg, err = c.interactiveEdit(cfg)
		fisk.FatalIfError(err, "could not create new configuration for Stream %s", c.stream)
	} else {
		cfg, err = c.copyAndEditStream(cfg, pc)
		fisk.FatalIfError(err, "could not create new configuration for Stream %s", c.stream)
	}

	// sorts strings to subject lists that only differ in ordering is considered equal
	sorter := cmp.Transformer("Sort", func(in []string) []string {
		out := append([]string(nil), in...)
		sort.Strings(out)
		return out
	})

	diff := cmp.Diff(sourceStream.Configuration(), cfg, sorter)
	if diff == "" {
		if !c.dryRun {
			fmt.Println("No difference in configuration")
		}

		return nil
	}

	fmt.Printf("Differences (-old +new):\n%s", diff)
	if c.dryRun {
		os.Exit(1)
	}

	if !c.force {
		ok, err := askConfirmation(fmt.Sprintf("Really edit Stream %s", c.stream), false)
		fisk.FatalIfError(err, "could not obtain confirmation")

		if !ok {
			return nil
		}
	}

	err = sourceStream.UpdateConfiguration(cfg)
	fisk.FatalIfError(err, "could not edit Stream %s", c.stream)

	if !c.json {
		fmt.Printf("Stream %s was updated\n\n", c.stream)
	}

	c.showStream(sourceStream)

	return nil
}

func (c *streamCmd) cpAction(pc *fisk.ParseContext) error {
	if c.stream == c.destination {
		fisk.Fatalf("source and destination Stream names cannot be the same")
	}

	c.connectAndAskStream()

	sourceStream, err := c.loadStream(c.stream)
	fisk.FatalIfError(err, "could not request Stream %s configuration", c.stream)

	// lazy deep copy
	input := sourceStream.Configuration()
	ij, err := json.Marshal(input)
	if err != nil {
		return err
	}
	var cfg api.StreamConfig
	err = json.Unmarshal(ij, &cfg)
	if err != nil {
		return err
	}

	cfg, err = c.copyAndEditStream(cfg, pc)
	fisk.FatalIfError(err, "could not copy Stream %s", c.stream)

	cfg.Name = c.destination

	newStream, err := c.mgr.NewStreamFromDefault(cfg.Name, cfg)
	fisk.FatalIfError(err, "could not create Stream")

	if !c.json {
		fmt.Printf("Stream %s was created\n\n", c.stream)
	}

	c.showStream(newStream)

	return nil
}

func (c *streamCmd) showStreamConfig(cfg api.StreamConfig) {
	if cfg.Description != "" {
		fmt.Printf("          Description: %s\n", cfg.Description)
	}
	if len(cfg.Subjects) > 0 {
		fmt.Printf("             Subjects: %s\n", strings.Join(cfg.Subjects, ", "))
	}
	fmt.Printf("             Replicas: %d\n", cfg.Replicas)
	if cfg.Sealed {
		fmt.Printf("               Sealed: true\n")
	}
	fmt.Printf("              Storage: %s\n", cfg.Storage.String())
	if cfg.Placement != nil {
		if cfg.Placement.Cluster != "" {
			fmt.Printf("    Placement Cluster: %s\n", cfg.Placement.Cluster)
		}
		if len(cfg.Placement.Tags) > 0 {
			fmt.Printf("       Placement Tags: %s\n", strings.Join(cfg.Placement.Tags, ", "))
		}
	}
	if cfg.RePublish != nil {
		if cfg.RePublish.HeadersOnly {
			fmt.Printf(" Republishing Headers: %s to %s", cfg.RePublish.Source, cfg.RePublish.Destination)
		} else {
			fmt.Printf("         Republishing: %s to %s", cfg.RePublish.Source, cfg.RePublish.Destination)
		}
	}

	fmt.Println()
	fmt.Println("Options:")
	fmt.Println()

	fmt.Printf("            Retention: %s\n", cfg.Retention.String())
	fmt.Printf("     Acknowledgements: %v\n", !cfg.NoAck)
	dnp := cfg.Discard.String()
	if cfg.DiscardNewPer {
		dnp = "New Per Subject"
	}
	fmt.Printf("       Discard Policy: %s\n", dnp)
	fmt.Printf("     Duplicate Window: %v\n", cfg.Duplicates)
	if cfg.AllowDirect {
		fmt.Printf("           Direct Get: %t\n", cfg.AllowDirect)
	}
	if cfg.MirrorDirect {
		fmt.Printf("    Mirror Direct Get: %t\n", cfg.MirrorDirect)
	}
	fmt.Printf("    Allows Msg Delete: %v\n", !cfg.DenyDelete)
	fmt.Printf("         Allows Purge: %v\n", !cfg.DenyPurge)
	fmt.Printf("       Allows Rollups: %v\n", cfg.RollupAllowed)

	fmt.Println()
	fmt.Println("Limits:")
	fmt.Println()

	if cfg.MaxMsgs == -1 {
		fmt.Println("     Maximum Messages: unlimited")
	} else {
		fmt.Printf("     Maximum Messages: %s\n", humanize.Comma(cfg.MaxMsgs))
	}
	if cfg.MaxMsgsPer <= 0 {
		fmt.Println("  Maximum Per Subject: unlimited")
	} else {
		fmt.Printf("  Maximum Per Subject: %s\n", humanize.Comma(cfg.MaxMsgsPer))
	}
	if cfg.MaxBytes == -1 {
		fmt.Println("        Maximum Bytes: unlimited")
	} else {
		fmt.Printf("        Maximum Bytes: %s\n", humanize.IBytes(uint64(cfg.MaxBytes)))
	}
	if cfg.MaxAge <= 0 {
		fmt.Println("          Maximum Age: unlimited")
	} else {
		fmt.Printf("          Maximum Age: %s\n", humanizeDuration(cfg.MaxAge))
	}
	if cfg.MaxMsgSize == -1 {
		fmt.Println(" Maximum Message Size: unlimited")
	} else {
		fmt.Printf(" Maximum Message Size: %s\n", humanize.IBytes(uint64(cfg.MaxMsgSize)))
	}
	if cfg.MaxConsumers == -1 {
		fmt.Println("    Maximum Consumers: unlimited")
	} else {
		fmt.Printf("    Maximum Consumers: %s\n", humanize.Comma(int64(cfg.MaxConsumers)))
	}
	if cfg.Template != "" {
		fmt.Printf("  Managed by Template: %s\n", cfg.Template)
	}

	if cfg.Mirror != nil || len(cfg.Sources) > 0 {
		fmt.Println()
		fmt.Println("Replication:")
		fmt.Println()
	}

	if cfg.Mirror != nil {
		fmt.Printf("               Mirror: %s\n", c.renderSource(cfg.Mirror))
	}

	if len(cfg.Sources) > 0 {
		fmt.Printf("              Sources: ")
		sort.Slice(cfg.Sources, func(i, j int) bool {
			return cfg.Sources[i].Name < cfg.Sources[j].Name
		})

		for i, source := range cfg.Sources {
			if i == 0 {
				fmt.Println(c.renderSource(source))
			} else {
				fmt.Printf("                       %s\n", c.renderSource(source))
			}
		}
	}

	fmt.Println()
}

func (c *streamCmd) renderSource(s *api.StreamSource) string {
	parts := []string{s.Name}
	if s.OptStartSeq > 0 {
		parts = append(parts, fmt.Sprintf("Start Seq: %s", humanize.Comma(int64(s.OptStartSeq))))
	}

	if s.OptStartTime != nil {
		parts = append(parts, fmt.Sprintf("Start Time: %v", s.OptStartTime))
	}
	if s.FilterSubject != "" {
		parts = append(parts, fmt.Sprintf("Subject: %s", s.FilterSubject))
	}
	if s.External != nil {
		if s.External.ApiPrefix != "" {
			parts = append(parts, fmt.Sprintf("API Prefix: %s", s.External.ApiPrefix))
		}
		if s.External.DeliverPrefix != "" {
			parts = append(parts, fmt.Sprintf("Delivery Prefix: %s", s.External.DeliverPrefix))
		}
	}

	return strings.Join(parts, ", ")
}

func (c *streamCmd) showStream(stream *jsm.Stream) error {
	info, err := stream.LatestInformation()
	if err != nil {
		return err
	}

	c.showStreamInfo(info)

	return nil
}

func (c *streamCmd) showStreamInfo(info *api.StreamInfo) {
	if c.json {
		err := printJSON(info)
		fisk.FatalIfError(err, "could not display info")
		return
	}

	if !c.showStateOnly {
		fmt.Printf("Information for Stream %s created %s\n", c.stream, info.Created.Local().Format("2006-01-02 15:04:05"))
		fmt.Println()
		c.showStreamConfig(info.Config)
		fmt.Println()
	} else {
		fmt.Printf("State for Stream %s created %s\n", c.stream, info.Created.Local().Format("2006-01-02 15:04:05"))
		fmt.Println()
	}

	if info.Cluster != nil && info.Cluster.Name != "" {
		fmt.Println("Cluster Information:")
		fmt.Println()
		fmt.Printf("                 Name: %s\n", info.Cluster.Name)
		fmt.Printf("               Leader: %s\n", info.Cluster.Leader)
		for _, r := range info.Cluster.Replicas {
			state := []string{r.Name}

			if r.Current {
				state = append(state, "current")
			} else {
				state = append(state, "outdated")
			}

			if r.Offline {
				state = append(state, "OFFLINE")
			}

			if r.Active > 0 && r.Active < math.MaxInt64 {
				state = append(state, fmt.Sprintf("seen %s ago", humanizeDuration(r.Active)))
			} else {
				state = append(state, "not seen")
			}

			switch {
			case r.Lag > 1:
				state = append(state, fmt.Sprintf("%s operations behind", humanize.Comma(int64(r.Lag))))
			case r.Lag == 1:
				state = append(state, fmt.Sprintf("%s operation behind", humanize.Comma(int64(r.Lag))))
			}

			fmt.Printf("              Replica: %s\n", strings.Join(state, ", "))

		}
		fmt.Println()
	}

	showSource := func(s *api.StreamSourceInfo) {
		fmt.Printf("          Stream Name: %s\n", s.Name)
		fmt.Printf("                  Lag: %s\n", humanize.Comma(int64(s.Lag)))
		if s.Active > 0 && s.Active < math.MaxInt64 {
			fmt.Printf("            Last Seen: %v\n", humanizeDuration(s.Active))
		} else {
			fmt.Printf("            Last Seen: never\n")
		}
		if s.External != nil {
			fmt.Printf("      Ext. API Prefix: %s\n", s.External.ApiPrefix)
			if s.External.DeliverPrefix != "" {
				fmt.Printf(" Ext. Delivery Prefix: %s\n", s.External.DeliverPrefix)
			}
		}
		if s.Error != nil {
			fmt.Printf("                Error: %s\n", s.Error.Description)
		}
	}
	if info.Mirror != nil {
		fmt.Println("Mirror Information:")
		fmt.Println()
		showSource(info.Mirror)
		fmt.Println()
	}

	if len(info.Sources) > 0 {
		fmt.Println("Source Information:")
		fmt.Println()
		for _, s := range info.Sources {
			showSource(s)
			fmt.Println()
		}
	}

	fmt.Println("State:")
	fmt.Println()
	fmt.Printf("             Messages: %s\n", humanize.Comma(int64(info.State.Msgs)))
	fmt.Printf("                Bytes: %s\n", humanize.IBytes(info.State.Bytes))
	if info.State.Lost != nil && len(info.State.Lost.Msgs) > 0 {
		fmt.Printf("        Lost Messages: %s (%s)\n", humanize.Comma(int64(len(info.State.Lost.Msgs))), humanize.IBytes(info.State.Lost.Bytes))
	}

	if info.State.FirstTime.Equal(time.Unix(0, 0)) || info.State.LastTime.IsZero() {
		fmt.Printf("             FirstSeq: %s\n", humanize.Comma(int64(info.State.FirstSeq)))
	} else {
		fmt.Printf("             FirstSeq: %s @ %s UTC\n", humanize.Comma(int64(info.State.FirstSeq)), info.State.FirstTime.Format("2006-01-02T15:04:05"))
	}

	if info.State.LastTime.Equal(time.Unix(0, 0)) || info.State.LastTime.IsZero() {
		fmt.Printf("              LastSeq: %s\n", humanize.Comma(int64(info.State.LastSeq)))
	} else {
		fmt.Printf("              LastSeq: %s @ %s UTC\n", humanize.Comma(int64(info.State.LastSeq)), info.State.LastTime.Format("2006-01-02T15:04:05"))
	}

	if len(info.State.Deleted) > 0 { // backwards compat with older servers
		fmt.Printf("     Deleted Messages: %s\n", humanize.Comma(int64(len(info.State.Deleted))))
	} else if info.State.NumDeleted > 0 {
		fmt.Printf("     Deleted Messages: %s\n", humanize.Comma(int64(info.State.NumDeleted)))
	}

	fmt.Printf("     Active Consumers: %s\n", humanize.Comma(int64(info.State.Consumers)))

	if info.State.NumSubjects > 0 { // available from 2.8
		fmt.Printf("   Number of Subjects: %s\n", humanize.Comma(int64(info.State.NumSubjects)))
	}

	if len(info.Alternates) > 0 {
		fmt.Printf("           Alternates: ")
		lName := 0
		lCluster := 0
		for _, s := range info.Alternates {
			if len(s.Name) > lName {
				lName = len(s.Name)
			}
			if len(s.Cluster) > lCluster {
				lCluster = len(s.Cluster)
			}
		}

		for i, s := range info.Alternates {
			msg := fmt.Sprintf("%s%s: Cluster: %s%s", strings.Repeat(" ", lName-len(s.Name)), s.Name, strings.Repeat(" ", lCluster-len(s.Cluster)), s.Cluster)
			if s.Domain != "" {
				msg = fmt.Sprintf("%s Domain: %s", msg, s.Domain)
			}

			if i == 0 {
				fmt.Println(msg)
			} else {
				fmt.Printf("                       %s\n", msg)
			}
		}
	}
}

func (c *streamCmd) stateAction(pc *fisk.ParseContext) error {
	c.showStateOnly = true
	return c.infoAction(pc)
}

func (c *streamCmd) infoAction(_ *fisk.ParseContext) error {
	c.connectAndAskStream()

	stream, err := c.loadStream(c.stream)
	fisk.FatalIfError(err, "could not request Stream info")
	err = c.showStream(stream)
	fisk.FatalIfError(err, "could not show stream")

	fmt.Println()

	return nil
}

func (c *streamCmd) discardPolicyFromString() api.DiscardPolicy {
	switch strings.ToLower(c.discardPolicy) {
	case "new":
		return api.DiscardNew
	case "old":
		return api.DiscardOld
	default:
		fisk.Fatalf("invalid discard policy %s", c.discardPolicy)
		return api.DiscardOld // unreachable
	}
}

func (c *streamCmd) storeTypeFromString(s string) api.StorageType {
	switch s {
	case "file", "f":
		return api.FileStorage
	case "memory", "m":
		return api.MemoryStorage
	default:
		fisk.Fatalf("invalid storage type %s", c.storage)
		return api.MemoryStorage // unreachable
	}
}

func (c *streamCmd) retentionPolicyFromString() api.RetentionPolicy {
	switch strings.ToLower(c.retentionPolicyS) {
	case "limits":
		return api.LimitsPolicy
	case "interest":
		return api.InterestPolicy
	case "work queue", "workq", "work":
		return api.WorkQueuePolicy
	default:
		fisk.Fatalf("invalid retention policy %s", c.retentionPolicyS)
		return api.LimitsPolicy // unreachable
	}
}

func (c *streamCmd) prepareConfig(pc *fisk.ParseContext, requireSize bool) api.StreamConfig {
	var err error

	if c.inputFile != "" {
		cfg, err := c.loadConfigFile(c.inputFile)
		fisk.FatalIfError(err, "invalid input")

		if c.stream != "" {
			cfg.Name = c.stream
		}

		if c.stream == "" {
			c.stream = cfg.Name
		}

		if len(c.subjects) > 0 {
			cfg.Subjects = c.subjects
		}

		return *cfg
	}

	if c.stream == "" {
		err = askOne(&survey.Input{
			Message: "Stream Name",
		}, &c.stream, survey.WithValidator(survey.Required))
		fisk.FatalIfError(err, "invalid input")
	}

	if c.mirror == "" && len(c.sources) == 0 {
		if len(c.subjects) == 0 {
			subjects := ""
			err = askOne(&survey.Input{
				Message: "Subjects",
				Help:    "Streams consume messages from subjects, this is a space or comma separated list that can include wildcards. Settable using --subjects",
			}, &subjects, survey.WithValidator(survey.Required))
			fisk.FatalIfError(err, "invalid input")

			c.subjects = splitString(subjects)
		}

		c.subjects = splitCLISubjects(c.subjects)
	}

	if c.mirror != "" && len(c.subjects) > 0 {
		fisk.Fatalf("mirrors cannot listen for messages on subjects")
	}

	if c.storage == "" {
		err = askOne(&survey.Select{
			Message: "Storage",
			Options: []string{"file", "memory"},
			Help:    "Streams are stored on the server, this can be one of many backends and all are usable in clustering mode. Settable using --storage",
		}, &c.storage, survey.WithValidator(survey.Required))
		fisk.FatalIfError(err, "invalid input")
	}

	storage := c.storeTypeFromString(c.storage)

	if c.replicas == 0 {
		c.replicas, err = askOneInt("Replication", "1", "When clustered, defines how many replicas of the data to store.  Settable using --replicas.")
		fisk.FatalIfError(err, "invalid input")
	}
	if c.replicas <= 0 {
		fisk.Fatalf("replicas should be >= 1")
	}

	if c.retentionPolicyS == "" {
		err = askOne(&survey.Select{
			Message: "Retention Policy",
			Options: []string{"Limits", "Interest", "Work Queue"},
			Help:    "Messages are retained either based on limits like size and age (Limits), as long as there are Consumers (Interest) or until any worker processed them (Work Queue)",
			Default: "Limits",
		}, &c.retentionPolicyS, survey.WithValidator(survey.Required))
		fisk.FatalIfError(err, "invalid input")
	}

	if c.discardPolicy == "" {
		err = askOne(&survey.Select{
			Message: "Discard Policy",
			Options: []string{"New", "Old"},
			Help:    "Once the Stream reach it's limits of size or messages the New policy will prevent further messages from being added while Old will delete old messages.",
			Default: "Old",
		}, &c.discardPolicy, survey.WithValidator(survey.Required))
		fisk.FatalIfError(err, "invalid input")
	}

	if c.maxMsgLimit == 0 {
		c.maxMsgLimit, err = askOneInt("Stream Messages Limit", "-1", "Defines the amount of messages to keep in the store for this Stream, when exceeded oldest messages are removed, -1 for unlimited. Settable using --max-msgs")
		fisk.FatalIfError(err, "invalid input")
		if c.maxMsgLimit <= 0 {
			c.maxMsgLimit = -1
		}
	}

	if c.maxMsgPerSubjectLimit == 0 && len(c.subjects) > 0 && (len(c.subjects) > 0 || strings.Contains(c.subjects[0], "*") || strings.Contains(c.subjects[0], ">")) {
		c.maxMsgPerSubjectLimit, err = askOneInt("Per Subject Messages Limit", "-1", "Defines the amount of messages to keep in the store for this Stream per unique subject, when exceeded oldest messages are removed, -1 for unlimited. Settable using --max-msgs-per-subject")
		fisk.FatalIfError(err, "invalid input")
		if c.maxMsgPerSubjectLimit <= 0 {
			c.maxMsgPerSubjectLimit = -1
		}
	}

	var maxAge time.Duration
	if c.maxBytesLimit == 0 {
		reqd := ""
		defltSize := "-1"
		if requireSize {
			reqd = "MaxBytes is required per Account Settings"
			defltSize = "256MB"
		}

		c.maxBytesLimit, err = askOneBytes("Total Stream Size", defltSize, "Defines the combined size of all messages in a Stream, when exceeded messages are removed or new ones are rejected, -1 for unlimited. Settable using --max-bytes", reqd)
		fisk.FatalIfError(err, "invalid input")
	}

	if c.maxBytesLimit <= 0 {
		c.maxBytesLimit = -1
	}

	if c.maxAgeLimit == "" {
		err = askOne(&survey.Input{
			Message: "Message TTL",
			Default: "-1",
			Help:    "Defines the oldest messages that can be stored in the Stream, any messages older than this period will be removed, -1 for unlimited. Supports units (s)econds, (m)inutes, (h)ours, (y)ears, (M)onths, (d)ays. Settable using --max-age",
		}, &c.maxAgeLimit)
		fisk.FatalIfError(err, "invalid input")
	}

	if c.maxAgeLimit != "-1" {
		maxAge, err = parseDurationString(c.maxAgeLimit)
		fisk.FatalIfError(err, "invalid maximum age limit format")
	}

	if c.maxMsgSize == 0 {
		c.maxMsgSize, err = askOneBytes("Max Message Size", "-1", "Defines the maximum size any single message may be to be accepted by the Stream. Settable using --max-msg-size", "")
		fisk.FatalIfError(err, "invalid input")
	}

	if c.maxMsgSize == 0 {
		c.maxMsgSize = -1
	}

	var dupeWindow time.Duration
	if c.dupeWindow == "" && c.mirror == "" {
		defaultDW := (2 * time.Minute).String()
		if maxAge > 0 && maxAge < 2*time.Minute {
			defaultDW = maxAge.String()
		}
		err = askOne(&survey.Input{
			Message: "Duplicate tracking time window",
			Default: defaultDW,
			Help:    "Duplicate messages are identified by the Msg-Id headers and tracked within a window of this size. Supports units (s)econds, (m)inutes, (h)ours, (y)ears, (M)onths, (d)ays. Settable using --dupe-window.",
		}, &c.dupeWindow)
		fisk.FatalIfError(err, "invalid input")
	}

	if c.dupeWindow != "" {
		dupeWindow, err = parseDurationString(c.dupeWindow)
		fisk.FatalIfError(err, "invalid duplicate window format")
	}

	if !c.allowRollupSet {
		c.allowRollup, err = askConfirmation("Allow message Roll-ups", false)
		fisk.FatalIfError(err, "invalid input")
	}

	if !c.denyDeleteSet {
		allow, err := askConfirmation("Allow message deletion", true)
		fisk.FatalIfError(err, "invalid input")
		c.denyDelete = !allow
	}

	if !c.denyPurgeSet {
		allow, err := askConfirmation("Allow purging subjects or the entire stream", true)
		fisk.FatalIfError(err, "invalid input")
		c.denyPurge = !allow
	}

	cfg := api.StreamConfig{
		Name:          c.stream,
		Description:   c.description,
		Subjects:      c.subjects,
		MaxMsgs:       c.maxMsgLimit,
		MaxMsgsPer:    c.maxMsgPerSubjectLimit,
		MaxBytes:      c.maxBytesLimit,
		MaxMsgSize:    int32(c.maxMsgSize),
		Duplicates:    dupeWindow,
		MaxAge:        maxAge,
		Storage:       storage,
		NoAck:         !c.ack,
		Retention:     c.retentionPolicyFromString(),
		Discard:       c.discardPolicyFromString(),
		MaxConsumers:  c.maxConsumers,
		Replicas:      int(c.replicas),
		RollupAllowed: c.allowRollup,
		DenyPurge:     c.denyPurge,
		DenyDelete:    c.denyDelete,
		AllowDirect:   c.allowDirect,
		MirrorDirect:  c.allowMirrorDirectSet,
		DiscardNewPer: c.discardPerSubj,
	}

	if c.placementCluster != "" || len(c.placementTags) > 0 {
		cfg.Placement = &api.Placement{
			Cluster: c.placementCluster,
			Tags:    c.placementTags,
		}
	}

	if c.mirror != "" {
		if isJsonString(c.mirror) {
			cfg.Mirror, err = c.parseStreamSource(c.mirror)
			fisk.FatalIfError(err, "invalid mirror")
		} else {
			cfg.Mirror = c.askMirror()
		}
	}

	for _, source := range c.sources {
		if isJsonString(source) {
			ss, err := c.parseStreamSource(source)
			fisk.FatalIfError(err, "invalid source")
			cfg.Sources = append(cfg.Sources, ss)
		} else {
			ss := c.askSource(source, fmt.Sprintf("%s Source", source))
			cfg.Sources = append(cfg.Sources, ss)
		}
	}

	if c.repubSource != "" && c.repubDest != "" {
		cfg.RePublish = &api.RePublish{
			Source:      c.repubSource,
			Destination: c.repubDest,
			HeadersOnly: c.repubHeadersOnly,
		}
	}

	return cfg
}

func (c *streamCmd) askMirror() *api.StreamSource {
	mirror := &api.StreamSource{Name: c.mirror}
	ok, err := askConfirmation("Adjust mirror start", false)
	fisk.FatalIfError(err, "Could not request mirror details")
	if ok {
		a, err := askOneInt("Mirror Start Sequence", "0", "Start mirroring at a specific sequence")
		fisk.FatalIfError(err, "Invalid sequence")
		mirror.OptStartSeq = uint64(a)

		if mirror.OptStartSeq == 0 {
			ts := ""
			err = askOne(&survey.Input{
				Message: "Mirror Start Time (YYYY:MM:DD HH:MM:SS)",
				Help:    "Start replicating as a specific time stamp in UTC time",
			}, &ts)
			fisk.FatalIfError(err, "could not request start time")
			if ts != "" {
				t, err := time.Parse("2006:01:02 15:04:05", ts)
				fisk.FatalIfError(err, "invalid time format")
				mirror.OptStartTime = &t
			}
		}

		err = askOne(&survey.Input{
			Message: "Filter mirror by subject",
			Help:    "Only replicate data matching this subject",
		}, &mirror.FilterSubject)
		fisk.FatalIfError(err, "could not request filter")
	}

	ok, err = askConfirmation("Import mirror from a different JetStream domain", false)
	fisk.FatalIfError(err, "Could not request mirror details")
	if ok {
		mirror.External = &api.ExternalStream{}
		domainName := ""
		err = askOne(&survey.Input{
			Message: "Foreign JetStream domain name",
			Help:    "The domain name from where to import the JetStream API",
		}, &domainName, survey.WithValidator(survey.Required))
		fisk.FatalIfError(err, "Could not request mirror details")
		mirror.External.ApiPrefix = fmt.Sprintf("$JS.%s.API", domainName)

		err = askOne(&survey.Input{
			Message: "Delivery prefix",
			Help:    "Optional prefix of the delivery subject",
		}, &mirror.External.DeliverPrefix)
		fisk.FatalIfError(err, "Could not request mirror details")
		return mirror
	}

	ok, err = askConfirmation("Import mirror from a different account", false)
	fisk.FatalIfError(err, "Could not request mirror details")
	if !ok {
		return mirror
	}

	mirror.External = &api.ExternalStream{}
	err = askOne(&survey.Input{
		Message: "Foreign account API prefix",
		Help:    "The prefix where the foreign account JetStream API has been imported",
	}, &mirror.External.ApiPrefix, survey.WithValidator(survey.Required))
	fisk.FatalIfError(err, "Could not request mirror details")

	err = askOne(&survey.Input{
		Message: "Foreign account delivery prefix",
		Help:    "The prefix where the foreign account JetStream delivery subjects has been imported",
	}, &mirror.External.DeliverPrefix, survey.WithValidator(survey.Required))
	fisk.FatalIfError(err, "Could not request mirror details")

	return mirror
}

func (c *streamCmd) askSource(name string, prefix string) *api.StreamSource {
	cfg := &api.StreamSource{Name: name}

	ok, err := askConfirmation(fmt.Sprintf("Adjust source %q start", name), false)
	fisk.FatalIfError(err, "Could not request source details")
	if ok {
		a, err := askOneInt(fmt.Sprintf("%s Start Sequence", prefix), "0", "Start mirroring at a specific sequence")
		fisk.FatalIfError(err, "Invalid sequence")
		cfg.OptStartSeq = uint64(a)

		ts := ""
		err = askOne(&survey.Input{
			Message: fmt.Sprintf("%s UTC Time Stamp (YYYY:MM:DD HH:MM:SS)", prefix),
			Help:    "Start replicating as a specific time stamp",
		}, &ts)
		fisk.FatalIfError(err, "could not request start time")
		if ts != "" {
			t, err := time.Parse("2006:01:02 15:04:05", ts)
			fisk.FatalIfError(err, "invalid time format")
			cfg.OptStartTime = &t
		}
	}

	err = askOne(&survey.Input{
		Message: fmt.Sprintf("%s Filter source by subject", prefix),
		Help:    "Only replicate data matching this subject",
	}, &cfg.FilterSubject)
	fisk.FatalIfError(err, "could not request filter")

	ok, err = askConfirmation(fmt.Sprintf("Import %q from a different JetStream domain", name), false)
	fisk.FatalIfError(err, "Could not request source details")
	if ok {
		cfg.External = &api.ExternalStream{}
		domainName := ""
		err = askOne(&survey.Input{
			Message: fmt.Sprintf("%s foreign JetStream domain name", prefix),
			Help:    "The domain name from where to import the JetStream API",
		}, &domainName, survey.WithValidator(survey.Required))
		fisk.FatalIfError(err, "Could not request source details")
		cfg.External.ApiPrefix = fmt.Sprintf("$JS.%s.API", domainName)

		err = askOne(&survey.Input{
			Message: fmt.Sprintf("%s foreign JetStream domain delivery prefix", prefix),
			Help:    "Optional prefix of the delivery subject",
		}, &cfg.External.DeliverPrefix)
		fisk.FatalIfError(err, "Could not request source details")
		return cfg
	}

	ok, err = askConfirmation(fmt.Sprintf("Import %q from a different account", name), false)
	fisk.FatalIfError(err, "Could not request source details")
	if !ok {
		return cfg
	}

	cfg.External = &api.ExternalStream{}
	err = askOne(&survey.Input{
		Message: fmt.Sprintf("%s foreign account API prefix", prefix),
		Help:    "The prefix where the foreign account JetStream API has been imported",
	}, &cfg.External.ApiPrefix, survey.WithValidator(survey.Required))
	fisk.FatalIfError(err, "Could not request source details")

	err = askOne(&survey.Input{
		Message: fmt.Sprintf("%s foreign account delivery prefix", prefix),
		Help:    "The prefix where the foreign account JetStream delivery subjects has been imported",
	}, &cfg.External.DeliverPrefix, survey.WithValidator(survey.Required))
	fisk.FatalIfError(err, "Could not request source details")

	return cfg
}

func (c *streamCmd) parseStreamSource(source string) (*api.StreamSource, error) {
	ss := &api.StreamSource{}
	if isJsonString(source) {
		err := json.Unmarshal([]byte(source), ss)
		if err != nil {
			return nil, err
		}

		if ss.Name == "" {
			return nil, fmt.Errorf("name is required")
		}
	} else {
		ss.Name = source
	}

	return ss, nil
}

func (c *streamCmd) validateCfg(cfg *api.StreamConfig) (bool, []byte, []string, error) {
	if os.Getenv("NOVALIDATE") != "" {
		return true, nil, nil, nil
	}

	j, err := json.MarshalIndent(cfg, "", "  ")
	if err != nil {
		return false, nil, nil, err
	}

	if !cfg.NoAck {
		for _, subject := range cfg.Subjects {
			if subject == ">" {
				return false, j, []string{"subjects cannot be '>' when acknowledgement is enabled"}, nil
			}
		}
	}

	valid, errs := cfg.Validate(new(SchemaValidator))

	return valid, j, errs, nil
}

func (c *streamCmd) addAction(pc *fisk.ParseContext) (err error) {
	_, mgr, err := prepareHelper("", natsOpts()...)
	fisk.FatalIfError(err, "could not create Stream")

	requireSize, _ := mgr.IsStreamMaxBytesRequired()

	cfg := c.prepareConfig(pc, requireSize)

	switch {
	case c.validateOnly:
		valid, j, errs, err := c.validateCfg(&cfg)
		if err != nil {
			return err
		}

		fmt.Println(string(j))
		fmt.Println()
		if !valid {
			fisk.Fatalf("Validation Failed: %s", strings.Join(errs, "\n\t"))
		}

		fmt.Printf("Configuration is a valid Stream matching %s\n", cfg.SchemaType())
		return nil

	case c.outFile != "":
		valid, j, errs, err := c.validateCfg(&cfg)
		fisk.FatalIfError(err, "Could not validate configuration")

		if !valid {
			fisk.Fatalf("Validation Failed: %s", strings.Join(errs, "\n\t"))
		}

		return os.WriteFile(c.outFile, j, 0644)
	}

	str, err := mgr.NewStreamFromDefault(c.stream, cfg)
	fisk.FatalIfError(err, "could not create Stream")

	fmt.Printf("Stream %s was created\n\n", c.stream)

	c.showStream(str)

	return nil
}

func (c *streamCmd) rmAction(_ *fisk.ParseContext) (err error) {
	if c.force {
		if c.stream == "" {
			return fmt.Errorf("--force requires a stream name")
		}

		c.nc, c.mgr, err = prepareHelper("", natsOpts()...)
		fisk.FatalIfError(err, "setup failed")

		err = c.mgr.DeleteStream(c.stream)
		if err != nil {
			if err == context.DeadlineExceeded {
				fmt.Println("Delete failed due to timeout, the stream might not exist or be in an unmanageable state")
			}
		}

		return err
	}

	c.connectAndAskStream()

	ok, err := askConfirmation(fmt.Sprintf("Really delete Stream %s", c.stream), false)
	fisk.FatalIfError(err, "could not obtain confirmation")

	if !ok {
		return nil
	}

	stream, err := c.loadStream(c.stream)
	fisk.FatalIfError(err, "could not remove Stream")

	err = stream.Delete()
	fisk.FatalIfError(err, "could not remove Stream")

	return nil
}

func (c *streamCmd) purgeAction(_ *fisk.ParseContext) (err error) {
	c.connectAndAskStream()

	if !c.force {
		ok, err := askConfirmation(fmt.Sprintf("Really purge Stream %s", c.stream), false)
		fisk.FatalIfError(err, "could not obtain confirmation")

		if !ok {
			return nil
		}
	}

	stream, err := c.loadStream(c.stream)
	fisk.FatalIfError(err, "could not purge Stream")

	var req *api.JSApiStreamPurgeRequest
	if c.purgeKeep > 0 || c.purgeSubject != "" || c.purgeSequence > 0 {
		if c.purgeSequence > 0 && c.purgeKeep > 0 {
			return fmt.Errorf("sequence and keep cannot be combined when purghing")
		}

		req = &api.JSApiStreamPurgeRequest{
			Sequence: c.purgeSequence,
			Subject:  c.purgeSubject,
			Keep:     c.purgeKeep,
		}
	}

	err = stream.Purge(req)
	fisk.FatalIfError(err, "could not purge Stream")

	stream.Reset()

	c.showStream(stream)

	return nil
}

func (c *streamCmd) lsAction(_ *fisk.ParseContext) error {
	_, mgr, err := prepareHelper("", natsOpts()...)
	fisk.FatalIfError(err, "setup failed")

	var streams []*jsm.Stream
	var names []string

	skipped := false

	var filter *jsm.StreamNamesFilter
	if c.filterSubject != "" {
		filter = &jsm.StreamNamesFilter{Subject: c.filterSubject}
	}

	err = mgr.EachStream(filter, func(s *jsm.Stream) {
		if !c.showAll && s.IsInternal() {
			skipped = true
			return
		}

		streams = append(streams, s)
		names = append(names, s.Name())
	})
	if err != nil {
		return fmt.Errorf("could not list streams: %s", err)
	}

	if c.json {
		err = printJSON(names)
		fisk.FatalIfError(err, "could not display Streams")
		return nil
	}

	if c.listNames {
		fmt.Println(c.renderStreamsAsList(streams))
		return nil
	}

	if len(streams) == 0 && skipped {
		fmt.Println("No Streams defined, pass -a to include system streams")
		return nil
	} else if len(streams) == 0 {
		fmt.Println("No Streams defined")
		return nil
	}

	out, err := c.renderStreamsAsTable(streams)
	if err != nil {
		return err
	}

	fmt.Println(out)

	return nil
}

func (c *streamCmd) renderStreamsAsList(streams []*jsm.Stream) string {
	names := make([]string, len(streams))
	for i, s := range streams {
		names[i] = s.Name()
	}

	sort.Strings(names)

	return strings.Join(names, "\n")
}

func (c *streamCmd) renderStreamsAsTable(streams []*jsm.Stream) (string, error) {
	sort.Slice(streams, func(i, j int) bool {
		info, _ := streams[i].LatestInformation()
		jnfo, _ := streams[j].LatestInformation()

		return info.State.Bytes < jnfo.State.Bytes
	})

	var table *tablewriter.Table
	if c.filterSubject == "" {
		table = newTableWriter("Streams")
	} else {
		table = newTableWriter(fmt.Sprintf("Streams matching %s", c.filterSubject))
	}
	table.AddHeaders("Name", "Description", "Created", "Messages", "Size", "Last Message")
	for _, s := range streams {
		nfo, _ := s.LatestInformation()
		table.AddRow(s.Name(), s.Description(), nfo.Created.Local().Format("2006-01-02 15:04:05"), humanize.Comma(int64(nfo.State.Msgs)), humanize.IBytes(nfo.State.Bytes), humanizeDuration(time.Since(nfo.State.LastTime)))
	}

	return table.Render(), nil
}

func (c *streamCmd) rmMsgAction(_ *fisk.ParseContext) (err error) {
	c.connectAndAskStream()

	if c.msgID == -1 {
		id := ""
		err = askOne(&survey.Input{
			Message: "Message Sequence to remove",
		}, &id, survey.WithValidator(survey.Required))
		fisk.FatalIfError(err, "invalid input")

		idint, err := strconv.Atoi(id)
		fisk.FatalIfError(err, "invalid number")

		if idint <= 0 {
			return fmt.Errorf("positive message ID required")
		}
		c.msgID = int64(idint)
	}

	stream, err := c.loadStream(c.stream)
	fisk.FatalIfError(err, "could not load Stream %s", c.stream)

	if !c.force {
		ok, err := askConfirmation(fmt.Sprintf("Really remove message %d from Stream %s", c.msgID, c.stream), false)
		fisk.FatalIfError(err, "could not obtain confirmation")

		if !ok {
			return nil
		}
	}

	return stream.DeleteMessage(uint64(c.msgID))
}

func (c *streamCmd) getAction(_ *fisk.ParseContext) (err error) {
	c.connectAndAskStream()

	if c.msgID == -1 && c.filterSubject == "" {
		id := ""
		err = askOne(&survey.Input{
			Message: "Message Sequence to retrieve",
			Default: "-1",
		}, &id, survey.WithValidator(survey.Required))
		fisk.FatalIfError(err, "invalid input")

		idint, err := strconv.Atoi(id)
		fisk.FatalIfError(err, "invalid number")

		c.msgID = int64(idint)

		if c.msgID == -1 {
			err = askOne(&survey.Input{
				Message: "Subject to retrieve last message for",
			}, &c.filterSubject)
			fisk.FatalIfError(err, "invalid subject")
		}
	}

	stream, err := c.loadStream(c.stream)
	fisk.FatalIfError(err, "could not load Stream %s", c.stream)

	var item *api.StoredMsg
	if c.msgID > -1 {
		item, err = stream.ReadMessage(uint64(c.msgID))
	} else if c.filterSubject != "" {
		item, err = stream.ReadLastMessageForSubject(c.filterSubject)
	} else {
		return fmt.Errorf("no ID or subject specified")
	}
	fisk.FatalIfError(err, "could not retrieve %s#%d", c.stream, c.msgID)

	if c.json {
		printJSON(item)
		return nil
	}

	fmt.Printf("Item: %s#%d received %v on Subject %s\n\n", c.stream, item.Sequence, item.Time, item.Subject)

	if len(item.Header) > 0 {
		fmt.Println("Headers:")
		hdrs, err := decodeHeadersMsg(item.Header)
		if err == nil {
			for k, vals := range hdrs {
				for _, val := range vals {
					fmt.Printf("  %s: %s\n", k, val)
				}
			}
		}
		fmt.Println()
	}

	fmt.Println(string(item.Data))
	fmt.Println()
	return nil
}

func (c *streamCmd) connectAndAskStream() bool {
	var err error

	shouldAsk := c.stream == ""
	c.nc, c.mgr, err = prepareHelper("", natsOpts()...)
	fisk.FatalIfError(err, "setup failed")

	c.stream, c.selectedStream, err = selectStream(c.mgr, c.stream, c.force, c.showAll)
	fisk.FatalIfError(err, "could not pick a Stream to operate on")

	return shouldAsk
}

func (c *streamCmd) boolReverse(v bool) bool {
	if c.reportSortReverse {
		return !v
	}

	return v
}
