package shell

import (
	"regexp"
	"strconv"
	"strings"

	"github.com/lmorg/murex/lang"
	"github.com/lmorg/murex/lang/types"
	"github.com/lmorg/murex/shell/autocomplete"
	"github.com/lmorg/murex/shell/hintsummary"
	"github.com/lmorg/murex/shell/history"
	"github.com/lmorg/murex/utils/ansi"
	"github.com/lmorg/murex/utils/ansi/codes"
	"github.com/lmorg/murex/utils/dedup"
	"github.com/lmorg/murex/utils/parser"
	"github.com/lmorg/murex/utils/readline"
)

func errCallback(err error) {
	s := err.Error()
	if ansi.IsAllowed() {
		s = codes.Reset + codes.FgRed + s
	}
	Prompt.ForceHintTextUpdate(s)
}

var rxHistTag = regexp.MustCompile(`^[-_a-zA-Z0-9]+$`)

func tabCompletion(line []rune, pos int, dtc readline.DelayedTabContext) (string, []string, map[string]string, readline.TabDisplayType) {
	var prefix string

	if pos < 0 {
		return "", []string{}, nil, readline.TabDisplayGrid
	}

	if len(line) > pos-1 {
		line = line[:pos]
	}

	pt, _ := parse(line)

	act := autocomplete.AutoCompleteT{
		Definitions:       make(map[string]string),
		ErrCallback:       errCallback,
		DelayedTabContext: dtc,
		ParsedTokens:      pt,
	}

	rows, err := lang.ShellProcess.Config.Get("shell", "max-suggestions", types.Integer)
	if err != nil {
		rows = 8
	}
	Prompt.MaxTabCompleterRows = rows.(int)

	switch {
	case pt.Variable != "":
		if pt.VarLoc < len(line) {
			prefix = strings.TrimSpace(string(line[pt.VarLoc:]))
		}
		prefix = pt.Variable + prefix
		act.Items = autocomplete.MatchVars(prefix)

	case pt.Comment && pt.FuncName == "^":
		for h := Prompt.History.Len() - 1; h > -1; h-- {
			line, _ := Prompt.History.GetLine(h)
			linePt, _ := parser.Parse([]rune(line), 0)
			if linePt.Comment && rxHistTag.MatchString(linePt.CommentMsg) && strings.HasPrefix(linePt.CommentMsg, pt.CommentMsg) {
				suggestion := linePt.CommentMsg[len(pt.CommentMsg):]
				act.Items = append(act.Items, suggestion)
				act.Definitions[suggestion] = line
				prefix = pt.CommentMsg
			}
		}

	case pt.FuncName == "^":
		autocompleteHistoryHat(&act)

	case pt.ExpectFunc:
		if len(line) == 0 {
			Prompt.ForceHintTextUpdate("Tip: press [ctrl]+r to recall previously used command lines")
		}
		go autocomplete.UpdateGlobalExeList()

		if pt.Loc < len(line) {
			prefix = strings.TrimSpace(string(line[pt.Loc:]))
		}

		switch pt.PipeToken {
		case parser.PipeTokenPosix:
			autocomplete.MatchFunction(prefix, &act)
		case parser.PipeTokenArrow:
			act.TabDisplayType = readline.TabDisplayList
			globalExes := autocomplete.GlobalExes.Get()

			if lang.MethodStdout.Exists(pt.LastFuncName, types.Any) {
				// match everything
				dump := lang.MethodStdout.Dump()

				if len(prefix) == 0 {
					for dt := range dump {
						act.Items = append(act.Items, dump[dt]...)
						for i := range dump[dt] {
							act.Definitions[dump[dt][i]] = string(hintsummary.Get(dump[dt][i], (*globalExes)[dump[dt][i]]))
						}
					}

				} else {

					for dt := range dump {
						for i := range dump[dt] {
							if strings.HasPrefix(dump[dt][i], prefix) {
								act.Items = append(act.Items, dump[dt][i][len(prefix):])
								act.Definitions[dump[dt][i][len(prefix):]] = string(hintsummary.Get(dump[dt][i], (*globalExes)[dump[dt][i]]))
							}
						}
					}
				}

			} else {
				// match type
				outTypes := lang.MethodStdout.Types(pt.LastFuncName)
				outTypes = append(outTypes, types.Any)
				for i := range outTypes {
					inTypes := lang.MethodStdin.Get(outTypes[i])
					if len(prefix) == 0 {
						act.Items = append(act.Items, inTypes...)
						for j := range inTypes {
							act.Definitions[inTypes[j]] = string(hintsummary.Get(inTypes[j], (*globalExes)[inTypes[j]]))
						}
						continue
					}
					for j := range inTypes {
						if strings.HasPrefix(inTypes[j], prefix) {
							act.Items = append(act.Items, inTypes[j][len(prefix):])
							act.Definitions[inTypes[j][len(prefix):]] = string(hintsummary.Get(inTypes[j], (*globalExes)[inTypes[j]]))
						}
					}
				}
			}

			// If `->` returns no results then fall back to returning everything
			if len(act.Items) == 0 {
				autocompleteFunctions(&act, prefix)
			}

		default:
			autocompleteFunctions(&act, prefix)
		}

	default:
		autocomplete.InitExeFlags(pt.FuncName)
		if !pt.ExpectParam && len(act.ParsedTokens.Parameters) > 0 {
			prefix = pt.Parameters[len(pt.Parameters)-1]
		}

		autocomplete.MatchFlags(&act)
	}

	Prompt.MinTabItemLength = act.MinTabItemLength
	width := readline.GetTermWidth()
	switch {
	case width >= 200:
		Prompt.MaxTabItemLength = width / 5
	case width >= 150:
		Prompt.MaxTabItemLength = width / 4
	case width >= 100:
		Prompt.MaxTabItemLength = width / 3
	case width >= 70:
		Prompt.MaxTabItemLength = width / 2
	}

	var i int
	if act.DoNotSort {
		i = len(act.Items)
	} else {
		i = dedup.SortAndDedupString(act.Items)
	}
	if !act.DoNotEscape {
		autocomplete.FormatSuggestions(&act)
	}

	return prefix, act.Items[:i], act.Definitions, act.TabDisplayType
}

func autocompleteFunctions(act *autocomplete.AutoCompleteT, prefix string) {
	act.TabDisplayType = readline.TabDisplayGrid

	autocomplete.MatchFunction(prefix, act)
}

func autocompleteHistoryLine(prefix string) ([]string, map[string]string) {
	var (
		items       []string
		definitions = make(map[string]string)
	)

	dump := Prompt.History.Dump().([]history.Item)

	for i := len(dump) - 1; i > -1; i-- {
		if len(dump[i].Block) <= len(prefix) {
			continue
		}

		if !strings.HasPrefix(dump[i].Block, prefix) {
			continue
		}

		item := dump[i].Block[len(prefix):]

		if definitions[item] != "" {
			continue
		}

		dateTime := dump[i].DateTime.Format("02-Jan-06 15:04")
		items = append(items, item)
		definitions[item] = dateTime
	}

	return items, definitions
}

func autocompleteHistoryHat(act *autocomplete.AutoCompleteT) {
	size := Prompt.History.Len()
	act.Items = make([]string, size)
	act.Definitions = make(map[string]string, size)
	dump := Prompt.History.Dump().([]history.Item)

	j := len(dump)
	for i := range dump {
		j--
		s := strconv.Itoa(dump[i].Index)
		act.Definitions[s] = dump[i].Block
		act.Items[j] = s
	}

	act.TabDisplayType = readline.TabDisplayList
	act.DoNotSort = true
}
