/*
Copyright 2019 The Kubernetes Authors All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package images

import (
	"sort"
	"testing"

	"github.com/google/go-cmp/cmp"
	"k8s.io/minikube/pkg/version"
)

func TestKubeadmImages(t *testing.T) {
	tests := []struct {
		version string
		mirror  string
		invalid bool
		want    []string
	}{
		{"invalid", "", true, nil},
		{"v0.0.1", "", true, nil}, // too old
		{"v2.0.0", "", true, nil}, // too new
		{"v1.17.0", "", false, []string{
			"k8s.gcr.io/kube-proxy:v1.17.0",
			"k8s.gcr.io/kube-scheduler:v1.17.0",
			"k8s.gcr.io/kube-controller-manager:v1.17.0",
			"k8s.gcr.io/kube-apiserver:v1.17.0",
			"k8s.gcr.io/coredns:1.6.5",
			"k8s.gcr.io/etcd:3.4.3-0",
			"k8s.gcr.io/pause:3.1",
			"gcr.io/k8s-minikube/storage-provisioner:" + version.GetStorageProvisionerVersion(),
		}},
		{"v1.16.1", "mirror.k8s.io", false, []string{
			"mirror.k8s.io/kube-proxy:v1.16.1",
			"mirror.k8s.io/kube-scheduler:v1.16.1",
			"mirror.k8s.io/kube-controller-manager:v1.16.1",
			"mirror.k8s.io/kube-apiserver:v1.16.1",
			"mirror.k8s.io/coredns:1.6.2",
			"mirror.k8s.io/etcd:3.3.15-0",
			"mirror.k8s.io/pause:3.1",
			"mirror.k8s.io/k8s-minikube/storage-provisioner:" + version.GetStorageProvisionerVersion(),
		}},
		{"v1.15.0", "", false, []string{
			"k8s.gcr.io/kube-proxy:v1.15.0",
			"k8s.gcr.io/kube-scheduler:v1.15.0",
			"k8s.gcr.io/kube-controller-manager:v1.15.0",
			"k8s.gcr.io/kube-apiserver:v1.15.0",
			"k8s.gcr.io/coredns:1.3.1",
			"k8s.gcr.io/etcd:3.3.10",
			"k8s.gcr.io/pause:3.1",
			"gcr.io/k8s-minikube/storage-provisioner:" + version.GetStorageProvisionerVersion(),
		}},
		{"v1.14.0", "", false, []string{
			"k8s.gcr.io/kube-proxy:v1.14.0",
			"k8s.gcr.io/kube-scheduler:v1.14.0",
			"k8s.gcr.io/kube-controller-manager:v1.14.0",
			"k8s.gcr.io/kube-apiserver:v1.14.0",
			"k8s.gcr.io/coredns:1.3.1",
			"k8s.gcr.io/etcd:3.3.10",
			"k8s.gcr.io/pause:3.1",
			"gcr.io/k8s-minikube/storage-provisioner:" + version.GetStorageProvisionerVersion(),
		}},
		{"v1.13.0", "", false, []string{
			"k8s.gcr.io/kube-proxy:v1.13.0",
			"k8s.gcr.io/kube-scheduler:v1.13.0",
			"k8s.gcr.io/kube-controller-manager:v1.13.0",
			"k8s.gcr.io/kube-apiserver:v1.13.0",
			"k8s.gcr.io/coredns:1.2.6",
			"k8s.gcr.io/etcd:3.2.24",
			"k8s.gcr.io/pause:3.1",
			"gcr.io/k8s-minikube/storage-provisioner:" + version.GetStorageProvisionerVersion(),
		}},
		{"v1.12.0", "", false, []string{
			"k8s.gcr.io/kube-proxy:v1.12.0",
			"k8s.gcr.io/kube-scheduler:v1.12.0",
			"k8s.gcr.io/kube-controller-manager:v1.12.0",
			"k8s.gcr.io/kube-apiserver:v1.12.0",
			"k8s.gcr.io/coredns:1.2.2",
			"k8s.gcr.io/etcd:3.2.24",
			"k8s.gcr.io/pause:3.1",
			"gcr.io/k8s-minikube/storage-provisioner:" + version.GetStorageProvisionerVersion(),
		}},
		{"v1.11.0", "", true, nil},
		{"v1.10.0", "", true, nil},
	}
	for _, tc := range tests {
		got, err := Kubeadm(tc.mirror, tc.version)
		if err == nil && tc.invalid {
			t.Fatalf("expected err (%s): %v", tc.version, got)
		}
		if err != nil && !tc.invalid {
			t.Fatalf("unexpected err (%s): %v", tc.version, err)
		}
		sort.Strings(got)
		sort.Strings(tc.want)
		if diff := cmp.Diff(tc.want, got); diff != "" {
			t.Errorf("%s images mismatch (-want +got):\n%s", tc.version, diff)
		}
	}
}
