// Copyright (c) 2018 SAP SE or an SAP affiliate company. All rights reserved. This file is licensed under the Apache Software License, v. 2 except as noted otherwise in the LICENSE file
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package utils

import (
	"net"
	"regexp"
	"strings"
	"time"

	"k8s.io/apimachinery/pkg/api/resource"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// ValueExists returns true or false, depending on whether the given string <value>
// is part of the given []string list <list>.
func ValueExists(value string, list []string) bool {
	for _, v := range list {
		if v == value {
			return true
		}
	}
	return false
}

// MergeMaps takes two maps <a>, <b> and merges them. If <b> defines a value with a key
// already existing in the <a> map, the <a> value for that key will be overwritten.
func MergeMaps(a, b map[string]interface{}) map[string]interface{} {
	var values = map[string]interface{}{}

	for i, v := range b {
		existing, ok := a[i]
		values[i] = v

		switch elem := v.(type) {
		case map[string]interface{}:
			if ok {
				if extMap, ok := existing.(map[string]interface{}); ok {
					values[i] = MergeMaps(extMap, elem)
				}
			}
		default:
			values[i] = v
		}
	}

	for i, v := range a {
		if _, ok := values[i]; !ok {
			values[i] = v
		}
	}

	return values
}

// MergeStringMaps merges the content of the newMaps with the oldMap. If a key already exists then
// it gets overwritten by the last value with the same key.
func MergeStringMaps(oldMap map[string]string, newMaps ...map[string]string) map[string]string {
	var out map[string]string

	if oldMap != nil {
		out = make(map[string]string)
	}
	for k, v := range oldMap {
		out[k] = v
	}

	for _, newMap := range newMaps {
		if newMap != nil && out == nil {
			out = make(map[string]string)
		}

		for k, v := range newMap {
			out[k] = v
		}
	}

	return out
}

// TimeElapsed takes a <timestamp> and a <duration> checks whether the elapsed time until now is less than the <duration>.
// If yes, it returns true, otherwise it returns false.
func TimeElapsed(timestamp *metav1.Time, duration time.Duration) bool {
	if timestamp == nil {
		return true
	}

	var (
		end = metav1.NewTime(timestamp.Time.UTC().Add(duration))
		now = metav1.NewTime(time.Now().UTC())
	)
	return !now.Before(&end)
}

// FindFreePort finds a free port on the host machine and returns it.
func FindFreePort() (int, error) {
	l, err := net.Listen("tcp", "127.0.0.1:0")
	if err != nil {
		return 0, err
	}
	defer l.Close()
	return l.Addr().(*net.TCPAddr).Port, nil
}

// TestEmail validates the provided <email> against a regular expression and returns whether it matches.
func TestEmail(email string) bool {
	match, _ := regexp.MatchString(`^[^@]+@(?:[a-zA-Z-0-9]+\.)+[a-zA-Z]{2,}$`, email)
	return match
}

// IsTrue returns true if the passed bool pointer is not nil and true.
func IsTrue(value *bool) bool {
	return value != nil && *value
}

// IDForKeyWithOptionalValue returns an identifier for the given key + optional value.
func IDForKeyWithOptionalValue(key string, value *string) string {
	v := ""
	if value != nil {
		v = "=" + *value
	}
	return key + v
}

// QuantityPtr returns a Quantity pointer to its argument.
func QuantityPtr(q resource.Quantity) *resource.Quantity {
	return &q
}

// DurationPtr returns a time.Duration pointer to its argument.
func DurationPtr(d time.Duration) *time.Duration {
	return &d
}

// Indent indents the given string with the given number of spaces.
func Indent(str string, spaces int) string {
	return strings.ReplaceAll(str, "\n", "\n"+strings.Repeat(" ", spaces))
}

// ShallowCopyMapStringInterface creates a shallow copy of the given map.
func ShallowCopyMapStringInterface(values map[string]interface{}) map[string]interface{} {
	copiedValues := make(map[string]interface{}, len(values))
	for k, v := range values {
		copiedValues[k] = v
	}
	return copiedValues
}

// IifString returns onTrue if the condition is true, and onFalse otherwise.
// It is similar to the ternary operator (?:) and the IIF function (see https://en.wikipedia.org/wiki/IIf) in other languages.
func IifString(condition bool, onTrue, onFalse string) string {
	if condition {
		return onTrue
	}
	return onFalse
}
