# Container image vulnerability adaptor interface proposal

## Rationale

source #287 

### Big picture

* Kubescape team is planning to create controls which take into account image vulnerabilities, example: looking for public internet facing workloads with critical vulnerabilities. These are seriously effecting the security health of a cluster and therefore we think it is important to cover it. We think that most container registries are/will support image scanning like Harbor and therefore the ability to get information from them is important.
* There are information in the image repository which is important for existing controls as well. They are incomplete without it, example see this issue: Non-root containers check is broken #19 . These are not necessarily image vulnerability related. Can be information in the image manifest (like the issue before), but it can be the image BOM related.

### Relation to this proposal

There are multiple changes and design decisions needs to be made before Kubescape will support the before outlined controls. However, a focal point the whole picutre is the ability to access vulnerability databases of container images. We anticipate that most container image repositories will support image vulnerability scanning, some major players are already do. Since there is no a single API available which all of these data sources support it is important to create an adaption layer within Kubescape so different datasources can serve Kubescape's goals.

## High level design of Kubescape

### Layers

* Controls and Rules: that actual control logic implementation, the "tests" themselves. Implemented in rego
* OPA engine: the [OPA](https://github.com/open-policy-agent/opa) rego interpreter 
* Rules processor: Kubescape component, it enumerates and runs the controls while also preparing the all the input data that the controls need for running
* Data sources: set of different modules providing data to the Rules processor so it can run the controls with them. Examples: Kubernetes objects, cloud vendor API objects and adding in this proposal the vulnerability infomration
* Cloud Image Vulnerability adaption interface: the subject of this proposal, it gives a common interface for different registry/vulnerability vendors to adapt to.
* CIV adaptors: specific implementation of the CIV interface, example Harbor adaption
```
 -----------------------
| Controls/Rules (rego) |
 -----------------------
            |
 -----------------------
|      OPA engine       |
 -----------------------
            |
 -----------------------
|    Rules processor    |
 ----------------------- 
            |
 -----------------------
|     Data sources      |
 -----------------------              
            |
 =======================
| CIV adaption interface|    <- Adding this layer in this proposal
 ======================= 
            |
 -----------------------
| Specific CIV adaptors |    <- will be implemented based on this proposal
 -----------------------      

        

```

## Functionalities to cover

The interface needs to cover the following functionalities:

* Authentication against the information source (abstracted login)
* Triggering image scan (if applicable, the source might store vulnerabilities for images but cannot scan alone)
* Reading image scan status (with last scan date and etc.)
* Getting vulnerability information for a given image
* Getting image information
  * Image manifests
  * Image BOMs (bill of material)

## Go API proposal

```

/*type ContainerImageRegistryCredentials struct {
	   map[string]string
	Password string
	Tag        string
	Hash       string
}*/

type ContainerImageIdentifier struct {
	Registry   string
	Repository string
	Tag        string
	Hash       string
}

type ContainerImageScanStatus struct {
	ImageID         ContainerImageIdentifier
	IsScanAvailable bool
	IsBomAvailable  bool
	LastScanDate    time.Time
}

type ContainerImageVulnerabilityReport struct {
	ImageID ContainerImageIdentifier
	// TBD
}

type ContainerImageInformation struct {
	ImageID       ContainerImageIdentifier
	Bom           []string
	ImageManifest Manifest // will use here Docker package definition
}

type IContainerImageVulnerabilityAdaptor interface {
	// Credentials are coming from user input (CLI or configuration file) and they are abstracted at string to string map level
	// so and example use would be like registry: "simpledockerregistry:80" and credentials like {"username":"joedoe","password":"abcd1234"}
	Login(registry string, credentials map[string]string) error

	// For "help" purposes
	DescribeAdaptor() string

	GetImagesScanStatus(imageIDs []ContainerImageIdentifier) ([]ContainerImageScanStatus, error)

	GetImagesVulnerabilties(imageIDs []ContainerImageIdentifier) ([]ContainerImageVulnerabilityReport, error)

	GetImagesInformation(imageIDs []ContainerImageIdentifier) ([]ContainerImageInformation, error)
}
```



# Integration

# Input

The objects received from the interface will be converted to an Imetadata compatible objects as following

```
{
    "apiVersion": "image.vulnscan.com/v1",
    "kind": "VulnScan",
    "metadata": {
        "name": "nginx:latest"
    },
    "data": {
        // returned by the adaptor API (structure like our backend gives for an image 
    }
}
```


# Output

The rego results will be a combination of the k8s artifact and the list of relevant CVEs for the control

```
{
    "apiVersion": "result.vulnscan.com/v1",
    "kind": "Pod",
    "metadata": {
        "name": "nginx"
    },
    "relatedObjects": [
        {
            "apiVersion": "v1",
            "kind": "Pod",
            "metadata": {
                "name": "nginx"
            },
            "spec": {
                // podSpec
            },
        },
        {
            "apiVersion": "container.vulnscan.com/v1",
            "kind": "VulnScan",
            "metadata": {
                "name": "nginx:latest",
            },
            "data": {
                
                // returned by the adaptor API (structure like our backend gives for an image  
            }
        }
    ]
}
```