package v1

import (
	"encoding/json"

	"github.com/armosec/kubescape/core/pkg/containerscan"
	"github.com/armosec/kubescape/core/pkg/registryadaptors/registryvulnerabilities"
)

type ArmoCivAdaptorMock struct {
	resultList *registryvulnerabilities.ContainerImageVulnerabilityReport
}

func NewArmoAdaptorMock() (*ArmoCivAdaptorMock, error) {
	scanDetailsResult := struct {
		Total struct {
			Value    int    `json:"value"`
			Relation string `json:"relation"`
		} `json:"total"`
		Response containerscan.VulnerabilitiesList `json:"response"`
		Cursor   string                            `json:"cursor"`
	}{}

	if err := json.Unmarshal([]byte(minikubeReponseMock), &scanDetailsResult); err != nil {
		return nil, err
	}

	vulnerabilities := responseObjectToVulnerabilities(scanDetailsResult.Response)

	resultImageVulnerabilityReport := registryvulnerabilities.ContainerImageVulnerabilityReport{
		ImageID:         registryvulnerabilities.ContainerImageIdentifier{Tag: vulnerabilities[0].Name},
		Vulnerabilities: vulnerabilities,
	}
	return &ArmoCivAdaptorMock{resultList: &resultImageVulnerabilityReport}, nil
}

func (armoCivAdaptorMock *ArmoCivAdaptorMock) Login() error {
	return nil
}
func (armoCivAdaptorMock *ArmoCivAdaptorMock) GetImagesVulnerabilities(imageIDs []registryvulnerabilities.ContainerImageIdentifier) ([]registryvulnerabilities.ContainerImageVulnerabilityReport, error) {
	return []registryvulnerabilities.ContainerImageVulnerabilityReport{*armoCivAdaptorMock.resultList}, nil
}

func (armoCivAdaptorMock *ArmoCivAdaptorMock) GetImageVulnerability(imageID *registryvulnerabilities.ContainerImageIdentifier) (*registryvulnerabilities.ContainerImageVulnerabilityReport, error) {
	return armoCivAdaptorMock.resultList, nil
}

func (armoCivAdaptorMock *ArmoCivAdaptorMock) DescribeAdaptor() string {
	// TODO
	return ""
}

func (armoCivAdaptorMock *ArmoCivAdaptorMock) GetImagesInformation(imageIDs []registryvulnerabilities.ContainerImageIdentifier) ([]registryvulnerabilities.ContainerImageInformation, error) {
	// TODO
	return []registryvulnerabilities.ContainerImageInformation{}, nil
}

func (armoCivAdaptorMock *ArmoCivAdaptorMock) GetImagesScanStatus(imageIDs []registryvulnerabilities.ContainerImageIdentifier) ([]registryvulnerabilities.ContainerImageScanStatus, error) {
	// TODO
	return []registryvulnerabilities.ContainerImageScanStatus{}, nil
}

//==============================================================================================================================
//==============================================================================================================================
//==============================================================================================================================

var minikubeReponseMock = `{"total":{"value":73,"relation":"eq"},"response":[{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2005-2541","https://security-tracker.debian.org/tracker/CVE-2005-2541"],"name":"CVE-2005-2541","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"tar","packageVersion":"1.30+dfsg-6","link":"https://nvd.nist.gov/vuln/detail/CVE-2005-2541","description":"Tar 1.15.1 does not properly warn the user when extracting setuid or setgid files, which may allow local users or remote attackers to gain privileges.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2007-5686","https://security-tracker.debian.org/tracker/CVE-2007-5686"],"name":"CVE-2007-5686","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"login","packageVersion":"1:4.5-1.1","link":"https://nvd.nist.gov/vuln/detail/CVE-2007-5686","description":"initscripts in rPath Linux 1 sets insecure permissions for the /var/log/btmp file, which allows local users to obtain sensitive information regarding authentication attempts.  NOTE: because sshd detects the insecure permissions and does not log certain events, this also prevents sshd from logging failed authentication attempts by remote attackers.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2007-6755","https://security-tracker.debian.org/tracker/CVE-2007-6755"],"name":"CVE-2007-6755","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libssl1.1","packageVersion":"1.1.1d-0+deb10u6","link":"https://nvd.nist.gov/vuln/detail/CVE-2007-6755","description":"The NIST SP 800-90A default statement of the Dual Elliptic Curve Deterministic Random Bit Generation (Dual_EC_DRBG) algorithm contains point Q constants with a possible relationship to certain \"skeleton key\" values, which might allow context-dependent attackers to defeat cryptographic protection mechanisms by leveraging knowledge of those values.  NOTE: this is a preliminary CVE for Dual_EC_DRBG; future research may provide additional details about point Q and associated attacks, and could potentially lead to a RECAST or REJECT of this CVE.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2010-0928","https://security-tracker.debian.org/tracker/CVE-2010-0928"],"name":"CVE-2010-0928","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libssl1.1","packageVersion":"1.1.1d-0+deb10u6","link":"https://nvd.nist.gov/vuln/detail/CVE-2010-0928","description":"OpenSSL 0.9.8i on the Gaisler Research LEON3 SoC on the Xilinx Virtex-II Pro FPGA uses a Fixed Width Exponentiation (FWE) algorithm for certain signature calculations, and does not verify the signature before providing it to a caller, which makes it easier for physically proximate attackers to determine the private key via a modified supply voltage for the microprocessor, related to a \"fault-based attack.\"","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2010-4756","https://security-tracker.debian.org/tracker/CVE-2010-4756"],"name":"CVE-2010-4756","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libc-bin","packageVersion":"2.28-10","link":"https://nvd.nist.gov/vuln/detail/CVE-2010-4756","description":"The glob implementation in the GNU C Library (aka glibc or libc6) allows remote authenticated users to cause a denial of service (CPU and memory consumption) via crafted glob expressions that do not match any pathnames, as demonstrated by glob expressions in STAT commands to an FTP daemon, a different vulnerability than CVE-2010-2632.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2011-3374","https://security-tracker.debian.org/tracker/CVE-2011-3374"],"name":"CVE-2011-3374","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"apt","packageVersion":"1.8.2.3","link":"https://nvd.nist.gov/vuln/detail/CVE-2011-3374","description":"It was found that apt-key in apt, all versions, do not correctly validate gpg keys with the master keyring, leading to a potential man-in-the-middle attack.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2011-3389","https://security-tracker.debian.org/tracker/CVE-2011-3389"],"name":"CVE-2011-3389","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libgnutls30","packageVersion":"3.6.7-4+deb10u7","link":"https://nvd.nist.gov/vuln/detail/CVE-2011-3389","description":"The SSL protocol, as used in certain configurations in Microsoft Windows and Microsoft Internet Explorer, Mozilla Firefox, Google Chrome, Opera, and other products, encrypts data by using CBC mode with chained initialization vectors, which allows man-in-the-middle attackers to obtain plaintext HTTP headers via a blockwise chosen-boundary attack (BCBA) on an HTTPS session, in conjunction with JavaScript code that uses (1) the HTML5 WebSocket API, (2) the Java URLConnection API, or (3) the Silverlight WebClient API, aka a \"BEAST\" attack.","severity":"Medium","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2011-4116","https://security-tracker.debian.org/tracker/CVE-2011-4116"],"name":"CVE-2011-4116","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"perl-base","packageVersion":"5.28.1-6+deb10u1","link":"https://nvd.nist.gov/vuln/detail/CVE-2011-4116","description":"_is_safe in the File::Temp module for Perl does not properly handle symlinks.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2012-2663","https://security-tracker.debian.org/tracker/CVE-2012-2663"],"name":"CVE-2012-2663","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"iptables","packageVersion":"1.8.5-3~bpo10+1","link":"https://nvd.nist.gov/vuln/detail/CVE-2012-2663","description":"extensions/libxt_tcp.c in iptables through 1.4.21 does not match TCP SYN+FIN packets in --syn rules, which might allow remote attackers to bypass intended firewall restrictions via crafted packets.  NOTE: the CVE-2012-6638 fix makes this issue less relevant.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2013-4235","https://security-tracker.debian.org/tracker/CVE-2013-4235"],"name":"CVE-2013-4235","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"login","packageVersion":"1:4.5-1.1","link":"https://nvd.nist.gov/vuln/detail/CVE-2013-4235","description":"shadow: TOCTOU (time-of-check time-of-use) race condition when copying and removing directory trees","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2013-4392","https://security-tracker.debian.org/tracker/CVE-2013-4392"],"name":"CVE-2013-4392","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libsystemd0","packageVersion":"241-7~deb10u7","link":"https://nvd.nist.gov/vuln/detail/CVE-2013-4392","description":"systemd, when updating file permissions, allows local users to change the permissions and SELinux security contexts for arbitrary files via a symlink attack on unspecified files.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2016-10228","https://security-tracker.debian.org/tracker/CVE-2016-10228"],"name":"CVE-2016-10228","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libc-bin","packageVersion":"2.28-10","link":"https://nvd.nist.gov/vuln/detail/CVE-2016-10228","description":"The iconv program in the GNU C Library (aka glibc or libc6) 2.31 and earlier, when invoked with multiple suffixes in the destination encoding (TRANSLATE or IGNORE) along with the -c option, enters an infinite loop when processing invalid multi-byte input sequences, leading to a denial of service.","severity":"Low","metadata":null,"fixedIn":[{"name":"wont-fix","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2016-2781","https://security-tracker.debian.org/tracker/CVE-2016-2781"],"name":"CVE-2016-2781","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"coreutils","packageVersion":"8.30-3","link":"https://nvd.nist.gov/vuln/detail/CVE-2016-2781","description":"chroot in GNU coreutils, when used with --userspec, allows local users to escape to the parent session via a crafted TIOCSTI ioctl call, which pushes characters to the terminal's input buffer.","severity":"Low","metadata":null,"fixedIn":[{"name":"wont-fix","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2017-11164","https://security-tracker.debian.org/tracker/CVE-2017-11164"],"name":"CVE-2017-11164","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libpcre3","packageVersion":"2:8.39-12","link":"https://nvd.nist.gov/vuln/detail/CVE-2017-11164","description":"In PCRE 8.41, the OP_KETRMAX feature in the match function in pcre_exec.c allows stack exhaustion (uncontrolled recursion) when processing a crafted regular expression.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2017-16231","https://security-tracker.debian.org/tracker/CVE-2017-16231"],"name":"CVE-2017-16231","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libpcre3","packageVersion":"2:8.39-12","link":"https://nvd.nist.gov/vuln/detail/CVE-2017-16231","description":"** DISPUTED ** In PCRE 8.41, after compiling, a pcretest load test PoC produces a crash overflow in the function match() in pcre_exec.c because of a self-recursive call. NOTE: third parties dispute the relevance of this report, noting that there are options that can be used to limit the amount of stack that is used.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2017-18018","https://security-tracker.debian.org/tracker/CVE-2017-18018"],"name":"CVE-2017-18018","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"coreutils","packageVersion":"8.30-3","link":"https://nvd.nist.gov/vuln/detail/CVE-2017-18018","description":"In GNU Coreutils through 8.29, chown-core.c in chown and chgrp does not prevent replacement of a plain file with a symlink during use of the POSIX \"-R -L\" options, which allows local users to modify the ownership of arbitrary files by leveraging a race condition.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2017-7245","https://security-tracker.debian.org/tracker/CVE-2017-7245"],"name":"CVE-2017-7245","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libpcre3","packageVersion":"2:8.39-12","link":"https://nvd.nist.gov/vuln/detail/CVE-2017-7245","description":"Stack-based buffer overflow in the pcre32_copy_substring function in pcre_get.c in libpcre1 in PCRE 8.40 allows remote attackers to cause a denial of service (WRITE of size 4) or possibly have unspecified other impact via a crafted file.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2017-7246","https://security-tracker.debian.org/tracker/CVE-2017-7246"],"name":"CVE-2017-7246","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libpcre3","packageVersion":"2:8.39-12","link":"https://nvd.nist.gov/vuln/detail/CVE-2017-7246","description":"Stack-based buffer overflow in the pcre32_copy_substring function in pcre_get.c in libpcre1 in PCRE 8.40 allows remote attackers to cause a denial of service (WRITE of size 268) or possibly have unspecified other impact via a crafted file.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2018-1000654","https://security-tracker.debian.org/tracker/CVE-2018-1000654"],"name":"CVE-2018-1000654","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libtasn1-6","packageVersion":"4.13-3","link":"https://nvd.nist.gov/vuln/detail/CVE-2018-1000654","description":"GNU Libtasn1-4.13 libtasn1-4.13 version libtasn1-4.13, libtasn1-4.12 contains a DoS, specifically CPU usage will reach 100% when running asn1Paser against the POC due to an issue in _asn1_expand_object_id(p_tree), after a long time, the program will be killed. This attack appears to be exploitable via parsing a crafted file.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2018-12886","https://security-tracker.debian.org/tracker/CVE-2018-12886"],"name":"CVE-2018-12886","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"gcc-8-base","packageVersion":"8.3.0-6","link":"https://nvd.nist.gov/vuln/detail/CVE-2018-12886","description":"stack_protect_prologue in cfgexpand.c and stack_protect_epilogue in function.c in GNU Compiler Collection (GCC) 4.1 through 8 (under certain circumstances) generate instruction sequences when targeting ARM targets that spill the address of the stack protector guard, which allows an attacker to bypass the protection of -fstack-protector, -fstack-protector-all, -fstack-protector-strong, and -fstack-protector-explicit against stack overflow by controlling what the stack canary is compared against.","severity":"High","metadata":null,"fixedIn":[{"name":"wont-fix","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2018-20796","https://security-tracker.debian.org/tracker/CVE-2018-20796"],"name":"CVE-2018-20796","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libc-bin","packageVersion":"2.28-10","link":"https://nvd.nist.gov/vuln/detail/CVE-2018-20796","description":"In the GNU C Library (aka glibc or libc6) through 2.29, check_dst_limits_calc_pos_1 in posix/regexec.c has Uncontrolled Recursion, as demonstrated by '(\\227|)(\\\\1\\\\1|t1|\\\\\\2537)+' in grep.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2018-6829","https://security-tracker.debian.org/tracker/CVE-2018-6829"],"name":"CVE-2018-6829","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libgcrypt20","packageVersion":"1.8.4-5+deb10u1","link":"https://nvd.nist.gov/vuln/detail/CVE-2018-6829","description":"cipher/elgamal.c in Libgcrypt through 1.8.2, when used to encrypt messages directly, improperly encodes plaintexts, which allows attackers to obtain sensitive information by reading ciphertext data (i.e., it does not have semantic security in face of a ciphertext-only attack). The Decisional Diffie-Hellman (DDH) assumption does not hold for Libgcrypt's ElGamal implementation.","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2018-7169","https://security-tracker.debian.org/tracker/CVE-2018-7169"],"name":"CVE-2018-7169","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"login","packageVersion":"1:4.5-1.1","link":"https://nvd.nist.gov/vuln/detail/CVE-2018-7169","description":"An issue was discovered in shadow 4.5. newgidmap (in shadow-utils) is setuid and allows an unprivileged user to be placed in a user namespace where setgroups(2) is permitted. This allows an attacker to remove themselves from a supplementary group, which may allow access to certain filesystem paths if the administrator has used \"group blacklisting\" (e.g., chmod g-rwx) to restrict access to paths. This flaw effectively reverts a security feature in the kernel (in particular, the /proc/self/setgroups knob) to prevent this sort of privilege escalation.","severity":"Low","metadata":null,"fixedIn":[{"name":"wont-fix","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2019-1010022","https://security-tracker.debian.org/tracker/CVE-2019-1010022"],"name":"CVE-2019-1010022","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libc-bin","packageVersion":"2.28-10","link":"https://nvd.nist.gov/vuln/detail/CVE-2019-1010022","description":"** DISPUTED ** GNU Libc current is affected by: Mitigation bypass. The impact is: Attacker may bypass stack guard protection. The component is: nptl. The attack vector is: Exploit stack buffer overflow vulnerability and use this bypass vulnerability to bypass stack guard. NOTE: Upstream comments indicate \"this is being treated as a non-security bug and no real threat.\"","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}},{"designators":{"designatorType":"Attributes","attributes":{"cluster":"minikube","containerName":"kube-proxy","customerGUID":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","kind":"daemonset","name":"kube-proxy","namespace":"kube-system"}},"context":[{"attribute":"customerGUID","value":"XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX","source":"designators.attributes"},{"attribute":"cluster","value":"minikube","source":"designators.attributes"},{"attribute":"namespace","value":"kube-system","source":"designators.attributes"},{"attribute":"kind","value":"daemonset","source":"designators.attributes"},{"attribute":"name","value":"kube-proxy","source":"designators.attributes"},{"attribute":"containerName","value":"kube-proxy","source":"designators.attributes"}],"wlid":"wlid://cluster-minikube/namespace-kube-system/daemonset-kube-proxy","containersScanID":"XXXXXXXXXXXXXXXXXX","layers":[{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""},{"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","parentLayerHash":""}],"timestamp":1643522422,"isFixed":0,"layerHash":"sha256:48b90c7688a2c85d7081a437ecba5cb706fbaa98b09def0b206dbbe39e3af558","links":["https://nvd.nist.gov/vuln/detail/CVE-2019-1010023","https://security-tracker.debian.org/tracker/CVE-2019-1010023"],"name":"CVE-2019-1010023","imageHash":"sha256:132b7fc61d18b3ec4a35348f6c915b429f4757d92196e2ea8cd937aea3db41df","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","packageName":"libc-bin","packageVersion":"2.28-10","link":"https://nvd.nist.gov/vuln/detail/CVE-2019-1010023","description":"** DISPUTED ** GNU Libc current is affected by: Re-mapping current loaded library with malicious ELF file. The impact is: In worst case attacker may evaluate privileges. The component is: libld. The attack vector is: Attacker sends 2 ELF files to victim and asks to run ldd on it. ldd execute code. NOTE: Upstream comments indicate \"this is being treated as a non-security bug and no real threat.\"","severity":"Negligible","metadata":null,"fixedIn":[{"name":"not-fixed","imageTag":"k8s.gcr.io/kube-proxy@sha256:561d6cb95c32333db13ea847396167e903d97cf6e08dd937906c3dd0108580b7","version":""}],"relevant":"No signature profile to compare","urgent":0,"neglected":0,"healthStatus":"","categories":{"isRce":false}}],"cursor":""}`
