package parquet

import (
	"testing"
)

func TestInt32PlainDecoder(t *testing.T) {
	testValuesDecoder(t, &int32PlainDecoder{}, []decoderTestCase{
		{
			data: []byte{
				0x00, 0x00, 0x00, 0x80,
				0xFF, 0xFF, 0xFF, 0x7F,
				0x00, 0x00, 0x00, 0x00,
				0x9C, 0xFF, 0xFF, 0xFF,
				0xEA, 0x00, 0x00, 0x00,
			},
			decoded: []interface{}{
				int32(-2147483648),
				int32(2147483647),
				int32(0),
				int32(-100),
				int32(234),
			},
		},
	})
}

func TestInt32DeltaBianryPackedDecoder(t *testing.T) {
	testValuesDecoder(t, &int32DeltaBinaryPackedDecoder{}, []decoderTestCase{
		{
			data: []byte{
				0x80, 0x01, 0x08, 0x05, 0xFF, 0xFF, 0xFF, 0xFF, 0x0F, 0xFD,
				0xFF, 0xFF, 0xFF, 0x0F, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0xFE, 0xFF, 0xFF, 0x7F, 0x00, 0x00, 0x00, 0x00,
				0x9B, 0xFF, 0xFF, 0x7F, 0x4D, 0x01, 0x00, 0x80, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			decoded: []interface{}{int32(-2147483648), int32(2147483647), int32(0), int32(-100), int32(234)},
		},

		{
			data: []byte{
				0x80, 0x01, 0x08, 0x07, 0x86, 0xB5, 0x18, 0x03, 0x02, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x05, 0x00, 0x00,
			},
			decoded: []interface{}{
				int32(200003), int32(200001), int32(200002), int32(200003),
				int32(200002), int32(200001), int32(200000),
			},
		},

		{
			data: []byte{
				0x80, 0x01, 0x08, 0x14, 0x02, 0x02, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00,
			},
			decoded: []interface{}{
				int32(1), int32(2), int32(3), int32(4), int32(5), int32(6),
				int32(7), int32(8), int32(9), int32(10), int32(11), int32(12),
				int32(13), int32(14), int32(15), int32(16), int32(17),
				int32(18), int32(19), int32(20),
			},
		},

		{
			data: []byte{
				0x80, 0x01, 0x08, 0xC8, 0x01, 0xF4, 0x01, 0x41, 0x07, 0x07,
				0x06, 0x07, 0x06, 0x06, 0x07, 0x06, 0x95, 0x93, 0xEB, 0xF4,
				0x90, 0x91, 0x2A, 0x95, 0x1D, 0x60, 0x67, 0x08, 0x66, 0x30,
				0x26, 0xD5, 0x63, 0x42, 0x04, 0x1D, 0x52, 0x1C, 0x4E, 0xC9,
				0x42, 0x92, 0x61, 0x3E, 0x24, 0x48, 0xCE, 0xA2, 0xC3, 0x8E,
				0x87, 0xC8, 0x47, 0x37, 0x80, 0x7E, 0xA3, 0x58, 0xAA, 0x53,
				0x62, 0xA4, 0x30, 0x3B, 0xC3, 0xEE, 0x11, 0x39, 0x61, 0x8C,
				0x62, 0x68, 0x19, 0xEB, 0x67, 0x9B, 0x0E, 0x8B, 0x66, 0x2C,
				0x46, 0x22, 0xE1, 0x7F, 0xD4, 0x9B, 0xF5, 0x4A, 0x2E, 0x2B,
				0xC8, 0xB0, 0x57, 0xB4, 0xA3, 0xCC, 0x0C, 0xB3, 0xAA, 0x8C,
				0x28, 0x21, 0x4D, 0x71, 0x32, 0x4A, 0xFC, 0x12, 0x5F, 0x9B,
				0xF1, 0x01, 0xFB, 0xA1, 0x0F, 0x0B, 0xA7, 0x53, 0x07, 0xD2,
				0x45, 0x07, 0x07, 0x06, 0x06, 0x06, 0x06, 0x07, 0x06, 0xA6,
				0xD1, 0x25, 0x43, 0x1B, 0x8C, 0x62, 0x24, 0x0E, 0x47, 0xF8,
				0x71, 0xC0, 0x6A, 0x11, 0x4A, 0x4A, 0x75, 0x9A, 0xA8, 0x6A,
				0x96, 0x55, 0x46, 0x22, 0x0C, 0x8D, 0x2C, 0x60, 0x58, 0x1F,
				0x23, 0xAF, 0xA1, 0x32, 0x08, 0x46, 0x75, 0x76, 0x79, 0x3A,
				0x2A, 0x9A, 0x03, 0xDD, 0xF5, 0x80, 0xFE, 0xD0, 0xD4, 0xA6,
				0x7B, 0xE3, 0x7D, 0x6D, 0xB7, 0xD2, 0x99, 0x0E, 0x8B, 0x66,
				0x2C, 0x46, 0x22,
			},
			decoded: []interface{}{
				int32(122), int32(110), int32(116), int32(129), int32(135),
				int32(117), int32(134), int32(137), int32(125), int32(113),
				int32(139), int32(106), int32(132), int32(105), int32(137),
				int32(129), int32(120), int32(125), int32(134), int32(116),
				int32(102), int32(137), int32(136), int32(110), int32(118),
				int32(113), int32(108), int32(112), int32(101), int32(104),
				int32(121), int32(112), int32(110), int32(113), int32(112),
				int32(115), int32(133), int32(134), int32(115), int32(126),
				int32(128), int32(102), int32(103), int32(130), int32(114),
				int32(136), int32(103), int32(110), int32(108), int32(110),
				int32(126), int32(134), int32(130), int32(134), int32(113),
				int32(121), int32(112), int32(138), int32(111), int32(137),
				int32(119), int32(103), int32(109), int32(100), int32(137),
				int32(138), int32(138), int32(127), int32(100), int32(110),
				int32(108), int32(129), int32(134), int32(115), int32(126),
				int32(133), int32(125), int32(136), int32(127), int32(130),
				int32(105), int32(105), int32(135), int32(109), int32(129),
				int32(123), int32(112), int32(126), int32(111), int32(124),
				int32(135), int32(104), int32(121), int32(136), int32(133),
				int32(105), int32(117), int32(119), int32(111), int32(129),
				int32(120), int32(130), int32(118), int32(120), int32(107),
				int32(107), int32(100), int32(136), int32(122), int32(124),
				int32(100), int32(130), int32(106), int32(104), int32(116),
				int32(108), int32(135), int32(103), int32(114), int32(112),
				int32(119), int32(101), int32(112), int32(127), int32(135),
				int32(121), int32(117), int32(116), int32(135), int32(138),
				int32(138), int32(126), int32(116), int32(133), int32(101),
				int32(101), int32(115), int32(116), int32(109), int32(102),
				int32(133), int32(129), int32(108), int32(121), int32(139),
				int32(121), int32(106), int32(112), int32(119), int32(123),
				int32(107), int32(114), int32(132), int32(119), int32(127),
				int32(117), int32(100), int32(131), int32(129), int32(129),
				int32(116), int32(113), int32(111), int32(129), int32(101),
				int32(101), int32(126), int32(117), int32(122), int32(137),
				int32(134), int32(131), int32(113), int32(131), int32(121),
				int32(109), int32(104), int32(127), int32(132), int32(131),
				int32(134), int32(102), int32(119), int32(113), int32(139),
				int32(104), int32(127), int32(107), int32(124), int32(109),
				int32(101), int32(124), int32(119), int32(119), int32(139),
				int32(127), int32(119), int32(139), int32(114), int32(108),
			}}})
}

func TestInt32DeltaBianryPackedErrors(t *testing.T) {
	tests := [][]byte{
		// invalid number of miniblocks
		[]byte("\x01\x0200000"),

		// invalid miniblock bit width value
		[]byte("0\a\xb50000000000"),

		// invalid number of miniblocks
		[]byte("0\x00"),
	}

	d := &int32DeltaBinaryPackedDecoder{}

	for i, data := range tests {
		if err := d.init(data); err != nil {
			t.Logf("test %d: error in init: %s", i, err)
			continue
		}

		if err := d.decode(make([]int32, 1000)); err == nil || err == errNED {
			t.Errorf("test %d, error is expected", i)
		} else {
			t.Logf("test %d: error in decode: %s", i, err)
		}
	}
}
