/*
Copyright 2019 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package gcemodel

import (
	"k8s.io/kops/pkg/apis/kops"
	"k8s.io/kops/pkg/model"
	"k8s.io/kops/upup/pkg/fi/cloudup/gce"
	"k8s.io/kops/upup/pkg/fi/cloudup/gcetasks"
)

type GCEModelContext struct {
	*model.KopsModelContext
}

// LinkToNetwork returns the GCE Network object the cluster is located in
func (c *GCEModelContext) LinkToNetwork() *gcetasks.Network {
	name := c.Cluster.Spec.NetworkID
	if name == "" {
		name = c.SafeClusterName()
	}

	return &gcetasks.Network{Name: s(name)}
}

// NameForIPAliasRange returns the name for the secondary IP range attached to a subnet
func (c *GCEModelContext) NameForIPAliasRange(key string) string {
	// We include the cluster name so we could share a subnet...
	// but there's a 5 IP alias range limit per subnet anwyay, so
	// this is rather pointless and in practice we just use a
	// separate subnet per cluster
	return c.SafeObjectName(key)
}

// LinkToSubnet returns a link to the GCE subnet object
func (c *GCEModelContext) LinkToSubnet(subnet *kops.ClusterSubnetSpec) *gcetasks.Subnet {
	name := subnet.ProviderID
	if name == "" {
		name = c.SafeObjectName(subnet.Name)
	}

	return &gcetasks.Subnet{Name: s(name)}
}

// SafeObjectName returns the object name and cluster name escaped for GCE
func (c *GCEModelContext) SafeObjectName(name string) string {
	return gce.SafeObjectName(name, c.Cluster.ObjectMeta.Name)
}

// SafeClusterName returns the cluster name escaped for use as a GCE resource name
func (c *GCEModelContext) SafeClusterName() string {
	return gce.SafeClusterName(c.Cluster.ObjectMeta.Name)
}

// GCETagForRole returns the (network) tag for GCE instances in the given instance group role.
func (c *GCEModelContext) GCETagForRole(role kops.InstanceGroupRole) string {
	return gce.TagForRole(c.Cluster.ObjectMeta.Name, role)
}

func (c *GCEModelContext) LinkToTargetPool(id string) *gcetasks.TargetPool {
	return &gcetasks.TargetPool{Name: s(c.NameForTargetPool(id))}
}

func (c *GCEModelContext) NameForTargetPool(id string) string {
	return c.SafeObjectName(id)
}

func (c *GCEModelContext) NameForForwardingRule(id string) string {
	return c.SafeObjectName(id)
}

func (c *GCEModelContext) NameForIPAddress(id string) string {
	return c.SafeObjectName(id)
}

func (c *GCEModelContext) NameForFirewallRule(id string) string {
	return c.SafeObjectName(id)
}

func (c *GCEModelContext) NetworkingIsIPAlias() bool {
	return c.Cluster.Spec.Networking != nil && c.Cluster.Spec.Networking.GCE != nil
}

func (c *GCEModelContext) NetworkingIsGCERoutes() bool {
	return c.Cluster.Spec.Networking != nil && c.Cluster.Spec.Networking.Kubenet != nil
}
