package types

import (
	"encoding/json"
	"fmt"
	"github.com/go-git/go-git/v5/plumbing/transport"
	"github.com/kluctl/kluctl/v2/pkg/yaml"
	"net/url"
	"regexp"
	"strconv"
	"strings"
)

// +kubebuilder:validation:Type=string
type GitUrl struct {
	url.URL `json:"-"`
}

func ParseGitUrl(u string) (*GitUrl, error) {
	// we re-use go-git's parsing capabilities (especially in regard to SCP urls)
	ep, err := transport.NewEndpoint(u)
	if err != nil {
		return nil, err
	}

	if ep.Protocol == "file" {
		return nil, fmt.Errorf("file:// protocol is not supported: %s", u)
	}

	u2 := ep.String()

	// and we also rely on the standard lib to treat escaping properly
	u3, err := url.Parse(u2)
	if err != nil {
		return nil, err
	}

	return &GitUrl{
		URL: *u3,
	}, nil
}

func ParseGitUrlMust(u string) *GitUrl {
	u2, err := ParseGitUrl(u)
	if err != nil {
		panic(err)
	}
	return u2
}

func (in *GitUrl) DeepCopyInto(out *GitUrl) {
	out.URL = in.URL
	if out.URL.User != nil {
		out.URL.User = &*in.URL.User
	}
}

func (u *GitUrl) UnmarshalJSON(b []byte) error {
	var s string
	err := yaml.ReadYamlBytes(b, &s)
	if err != nil {
		return err
	}
	u2, err := ParseGitUrl(s)
	if err != nil {
		return err
	}
	*u = *u2
	return err
}

func (u GitUrl) MarshalJSON() ([]byte, error) {
	return json.Marshal(u.String())
}

func (u *GitUrl) IsSsh() bool {
	return u.Scheme == "ssh" || u.Scheme == "git" || u.Scheme == "git+ssh"
}

func (u *GitUrl) NormalizePort() string {
	port := u.Port()
	if port == "" {
		return ""
	}

	defaultPort := ""
	switch u.Scheme {
	case "http":
		defaultPort = "80"
	case "https":
		defaultPort = "443"
	case "git":
		defaultPort = "22"
	case "git+ssh":
		defaultPort = "22"
	case "ssh":
		defaultPort = "22"
	case "ftp":
		defaultPort = "21"
	case "rsync":
		defaultPort = "873"
	case "file":
		break
	default:
		return port
	}

	if defaultPort == "" || port == defaultPort {
		return ""
	}
	return port
}

func (u *GitUrl) Normalize() *GitUrl {
	path := strings.ToLower(u.Path)
	path = strings.TrimSuffix(path, ".git")
	path = strings.TrimSuffix(path, "/")

	hostname := strings.ToLower(u.Hostname())
	port := u.NormalizePort()

	if path != "" && hostname != "" && !strings.HasPrefix(path, "/") {
		path = "/" + path
	}

	u2 := *u
	u2.Path = path
	u2.Host = hostname
	if port != "" {
		u2.Host += ":" + port
	}
	return &u2
}

func (u *GitUrl) RepoKey() GitRepoKey {
	u2 := u.Normalize()
	path := strings.TrimPrefix(u2.Path, "/")
	return GitRepoKey{
		Host: u2.Host,
		Path: path,
	}
}

// +kubebuilder:validation:Type=string
type GitRepoKey struct {
	Host string `json:"-"`
	Path string `json:"-"`
}

var hostNameRegex = regexp.MustCompile(`^(([a-zA-Z]|[a-zA-Z][a-zA-Z0-9\-]*[a-zA-Z0-9])\.)*([A-Za-z]|[A-Za-z][A-Za-z0-9\-]*[A-Za-z0-9])$`)

func ParseGitRepoKey(s string) (GitRepoKey, error) {
	if s == "" {
		return GitRepoKey{}, nil
	}

	s2 := strings.SplitN(s, "/", 2)
	if len(s2) != 2 {
		return GitRepoKey{}, fmt.Errorf("invalid git repo key: %s", s)
	}

	var host, port string
	if strings.Contains(s2[0], ":") {
		s3 := strings.SplitN(s2[0], ":", 2)
		host = s3[0]
		port = s3[1]
	} else {
		host = s2[0]
	}

	if !hostNameRegex.MatchString(host) {
		return GitRepoKey{}, fmt.Errorf("invalid git repo key: %s", s)
	}

	if port != "" {
		if _, err := strconv.ParseInt(port, 10, 32); err != nil {
			return GitRepoKey{}, fmt.Errorf("invalid git repo key: %s", s)
		}
	}

	return GitRepoKey{
		Host: s2[0],
		Path: s2[1],
	}, nil
}

func (u GitRepoKey) String() string {
	if u.Host == "" && u.Path == "" {
		return ""
	}
	return fmt.Sprintf("%s/%s", u.Host, u.Path)
}

func (u *GitRepoKey) UnmarshalJSON(b []byte) error {
	var s string
	err := yaml.ReadYamlBytes(b, &s)
	if err != nil {
		return err
	}
	if s == "" {
		u.Host = ""
		u.Path = ""
		return nil
	}
	x, err := ParseGitRepoKey(s)
	if err != nil {
		return err
	}
	*u = x
	return nil
}

func (u GitRepoKey) MarshalJSON() ([]byte, error) {
	b, err := json.Marshal(u.String())
	if err != nil {
		return nil, err
	}

	return b, err
}
