//go:build integ
// +build integ

// Copyright Istio Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package revisions

import (
	"context"
	"fmt"
	"testing"
	"time"

	v1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	"istio.io/istio/pkg/config/protocol"
	"istio.io/istio/pkg/test/framework"
	"istio.io/istio/pkg/test/framework/components/crd"
	"istio.io/istio/pkg/test/framework/components/echo"
	"istio.io/istio/pkg/test/framework/components/echo/check"
	"istio.io/istio/pkg/test/framework/components/echo/deployment"
	"istio.io/istio/pkg/test/framework/components/echo/echotest"
	"istio.io/istio/pkg/test/framework/components/echo/match"
	"istio.io/istio/pkg/test/framework/components/istio"
	"istio.io/istio/pkg/test/framework/components/namespace"
	"istio.io/istio/pkg/test/framework/label"
	"istio.io/istio/pkg/test/framework/resource"
	"istio.io/istio/pkg/test/util/retry"
)

// TestMain defines the entrypoint for pilot tests using a standard Istio installation.
// If a test requires a custom install it should go into its own package, otherwise it should go
// here to reuse a single install across tests.
func TestMain(m *testing.M) {
	// nolint: staticcheck
	framework.
		NewSuite(m).
		RequireMultiPrimary().
		// Requires two CPs with specific names to be configured.
		Label(label.CustomSetup).
		Setup(istio.Setup(nil, func(_ resource.Context, cfg *istio.Config) {
			cfg.ControlPlaneValues = `
revision: stable
`
		})).
		Setup(istio.Setup(nil, func(_ resource.Context, cfg *istio.Config) {
			cfg.ControlPlaneValues = `
profile: empty
revision: canary
components:
  pilot:
    enabled: true
`
		})).
		Run()
}

// TestMultiRevision Sets up a simple client -> server call, where the client and server
// belong to different control planes.
func TestMultiRevision(t *testing.T) {
	framework.NewTest(t).
		Run(func(t framework.TestContext) {
			stable := namespace.NewOrFail(t, namespace.Config{
				Prefix:   "stable",
				Inject:   true,
				Revision: "stable",
			})
			canary := namespace.NewOrFail(t, namespace.Config{
				Prefix:   "canary",
				Inject:   true,
				Revision: "canary",
			})

			echos := deployment.New(t).
				WithClusters(t.Clusters()...).
				WithConfig(echo.Config{
					Service:   "client",
					Namespace: stable,
					Ports:     []echo.Port{},
				}).
				WithConfig(echo.Config{
					Service:   "server",
					Namespace: canary,
					Ports: []echo.Port{
						{
							Name:         "http",
							Protocol:     protocol.HTTP,
							WorkloadPort: 8090,
						},
					},
				}).
				WithConfig(echo.Config{
					Service:    "vm",
					Namespace:  canary,
					DeployAsVM: true,
					Ports:      []echo.Port{},
				}).
				BuildOrFail(t)

			echotest.New(t, echos).
				ConditionallyTo(echotest.ReachableDestinations).
				ToMatch(match.ServiceName(echo.NamespacedName{Name: "server", Namespace: canary})).
				Run(func(t framework.TestContext, from echo.Instance, to echo.Target) {
					retry.UntilSuccessOrFail(t, func() error {
						result, err := from.Call(echo.CallOptions{
							To: to,
							Port: echo.Port{
								Name: "http",
							},
							Retry: echo.Retry{
								NoRetry: true,
							},
							Check: check.And(
								check.OK(),
								check.ReachedTargetClusters(t),
							),
						})
						return check.And(
							check.NoError(),
							check.OK()).Check(result, err)
					}, retry.Delay(time.Millisecond*100))
				})
		})
}

func TestMultiRevisionRouteStatusHandling(t *testing.T) {
	framework.NewTest(t).
		Run(func(t framework.TestContext) {
			if err := crd.DeployGatewayAPI(t); err != nil {
				t.Fatal(err)
			}
			stable := namespace.NewOrFail(t, namespace.Config{
				Prefix:   "stable",
				Inject:   true,
				Revision: "stable",
			})
			canary := namespace.NewOrFail(t, namespace.Config{
				Prefix:   "canary",
				Inject:   true,
				Revision: "canary",
			})

			_ = deployment.New(t).
				WithClusters(t.Clusters()...).
				WithConfig(echo.Config{
					Service:   "client",
					Namespace: stable,
					Ports:     []echo.Port{},
				}).
				WithConfig(echo.Config{
					Service:   "server",
					Namespace: canary,
					Ports: []echo.Port{
						{
							Name:         "http",
							Protocol:     protocol.HTTP,
							WorkloadPort: 8090,
						},
					},
				}).
				BuildOrFail(t)

			t.ConfigIstio().YAML(canary.Name(), fmt.Sprintf(`
apiVersion: gateway.networking.k8s.io/v1
kind: Gateway
metadata:
  name: test-gateway
  namespace: %s
spec:
  gatewayClassName: istio
  listeners:
  - name: http
    hostname: "test.example.com"
    port: 80
    protocol: HTTP
---
apiVersion: gateway.networking.k8s.io/v1
kind: HTTPRoute
metadata:
  name: test-route
  namespace: %s
spec:
  parentRefs:
  - name: test-gateway
  hostnames:
  - "test.example.com"
  rules:
  - matches:
    - path:
        type: PathPrefix
        value: /
    backendRefs:
    - name: server
      port: 8090
`, canary.Name(), canary.Name())).ApplyOrFail(t)

			var parentsFound bool
			checkStatus := func() (completed bool, err error) {
				route, err := t.Clusters().Default().GatewayAPI().GatewayV1beta1().HTTPRoutes(canary.Name()).Get(context.Background(), "test-route", v1.GetOptions{})
				if err != nil {
					return false, err
				}

				// give the status handler time to properly calculate status
				if len(route.Status.Parents) > 0 && !parentsFound {
					parentsFound = true
					return false, nil
				} else if len(route.Status.Parents) == 0 && parentsFound {
					// the status previously had parents, they should not get removed
					return true, fmt.Errorf("httproute status was incorrectly overwritten")
				}

				return false, nil
			}

			var err error
			for attempts := 0; attempts <= 10; attempts++ {
				complete, localErr := checkStatus()
				if complete {
					err = localErr
					break
				}
			}

			if err != nil {
				t.Fatal(err)
			}
		})
}
