import { z } from "zod";

import { ProjectBotsSchema } from "@app/db/schemas";
import { readLimit, writeLimit } from "@app/server/config/rateLimiter";
import { verifyAuth } from "@app/server/plugins/auth/verify-auth";
import { AuthMode } from "@app/services/auth/auth-type";

export const registerProjectBotRouter = async (server: FastifyZodProvider) => {
  server.route({
    method: "GET",
    url: "/:projectId",
    config: {
      rateLimit: readLimit
    },
    schema: {
      params: z.object({
        projectId: z.string().trim()
      }),
      response: {
        200: z.object({
          bot: ProjectBotsSchema.omit({
            iv: true,
            encryptedPrivateKey: true,
            tag: true,
            algorithm: true,
            keyEncoding: true
          })
        })
      }
    },
    onRequest: verifyAuth([AuthMode.JWT]),
    handler: async (req) => {
      const bot = await server.services.projectBot.findBotByProjectId({
        actor: req.permission.type,
        actorId: req.permission.id,
        actorOrgId: req.permission.orgId,
        actorAuthMethod: req.permission.authMethod,
        projectId: req.params.projectId
      });
      return { bot };
    }
  });

  server.route({
    method: "PATCH",
    url: "/:botId/active",
    config: {
      rateLimit: writeLimit
    },
    schema: {
      body: z.object({
        isActive: z.boolean(),
        botKey: z
          .object({
            nonce: z.string().trim().optional(),
            encryptedKey: z.string().trim().optional()
          })
          .optional()
      }),
      params: z.object({
        botId: z.string().trim()
      }),
      response: {
        200: z.object({
          bot: ProjectBotsSchema.omit({
            iv: true,
            encryptedPrivateKey: true,
            tag: true,
            algorithm: true,
            keyEncoding: true
          })
        })
      }
    },
    onRequest: verifyAuth([AuthMode.JWT]),
    handler: async (req) => {
      const bot = await server.services.projectBot.setBotActiveState({
        actor: req.permission.type,
        actorId: req.permission.id,
        actorOrgId: req.permission.orgId,
        actorAuthMethod: req.permission.authMethod,
        botId: req.params.botId,
        botKey: req.body.botKey,
        isActive: req.body.isActive
      });
      return { bot };
    }
  });
};
