// Licensed to Elasticsearch B.V. under one or more contributor
// license agreements. See the NOTICE file distributed with
// this work for additional information regarding copyright
// ownership. Elasticsearch B.V. licenses this file to you under
// the Apache License, Version 2.0 (the "License"); you may
// not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package hcl1

import (
	"testing"
)

const sample = `provider "google" {
	version = "2.5.0"
	project = "instrumenta"
	region = "europe-west2"


  }

  resource "google_container_cluster" "primary" {
	name     = "my-gke-cluster"
	location = "us-central1"

	# We can't create a cluster with no node pool defined, but we want to only use
	# separately managed node pools. So we create the smallest possible default
	# node pool and immediately delete it.
	remove_default_node_pool = true
	initial_node_count = 1

	# Setting an empty username and password explicitly disables basic auth
	master_auth {
	  username = ""
	  password = ""
	}
  }

  resource "google_container_node_pool" "primary_preemptible_nodes" {
	name       = "my-node-pool"
	location   = "us-central1"
	cluster    = "${google_container_cluster.primary.name}"
	node_count = 1

	node_config {
	  preemptible  = true
	  machine_type = "n1-standard-1"

	  metadata = {
		disable-legacy-endpoints = "true"
	  }

	  oauth_scopes = [
		"https://www.googleapis.com/auth/logging.write",
		"https://www.googleapis.com/auth/monitoring",
	  ]
	}
  }

  # The following outputs allow authentication and connectivity to the GKE Cluster
  # by using certificate-based authentication.
  output "client_certificate" {
	value = "${google_container_cluster.primary.master_auth.0.client_certificate}"
  }

  output "client_key" {
	value = "${google_container_cluster.primary.master_auth.0.client_key}"
  }

  output "cluster_ca_certificate" {
	value = "${google_container_cluster.primary.master_auth.0.cluster_ca_certificate}"
  }`

func TestHcl1Parser(t *testing.T) {
	var input interface{}
	parser := &Parser{}
	sampleFileBytes := []byte(sample)
	if err := parser.Unmarshal(sampleFileBytes, &input); err != nil {
		t.Fatalf("parser should not have thrown an error: %v", err)
	}

	if input == nil {
		t.Error("there should be information parsed but its nil")
	}

	inputMap := input.(map[string]interface{})
	if len(inputMap["resource"].([]map[string]interface{})) <= 0 {
		t.Error("there should be resources defined in the parsed file, but none found")
	}
}
