package v3

// VulnerabilityMetadata represents all vulnerability data that is not necessary to perform package-to-vulnerability matching.
type VulnerabilityMetadata struct {
	ID           string   // The identifier of the vulnerability or advisory
	Namespace    string   // Where this entry is valid within
	DataSource   string   // A URL where the data was sourced from
	RecordSource string   // The source of the vulnerability information (relative to the immediate upstream in the enterprise feedgroup)
	Severity     string   // How severe the vulnerability is (valid values are defined by upstream sources currently)
	URLs         []string // URLs to get more information about the vulnerability or advisory
	Description  string   // Description of the vulnerability
	Cvss         []Cvss   // Common Vulnerability Scoring System values
}

// Cvss contains select Common Vulnerability Scoring System fields for a vulnerability.
type Cvss struct {
	// VendorMetadata captures non-standard CVSS fields that vendors can sometimes
	// include when providing CVSS information.  This vendor-specific metadata type
	// allows to capture that data for persisting into the database
	VendorMetadata interface{}
	Metrics        CvssMetrics
	Vector         string // A textual representation of the metric values used to determine the score
	Version        string // The version of the CVSS spec, for example 2.0, 3.0, or 3.1
}

// CvssMetrics are the quantitative values that make up a CVSS score.
type CvssMetrics struct {
	// BaseScore ranges from 0 - 10 and defines qualities intrinsic to the severity of a vulnerability.
	BaseScore float64
	// ExploitabilityScore is a pointer to avoid having a 0 value by default.
	// It is an indicator of how easy it may be for an attacker to exploit
	// a vulnerability
	ExploitabilityScore *float64
	// ImpactScore represents the effects of an exploited vulnerability
	// relative to compromise in confidentiality, integrity, and availability.
	// It is an optional parameter, so that is why it is a pointer instead of
	// a regular field
	ImpactScore *float64
}

func NewCvssMetrics(baseScore, exploitabilityScore, impactScore float64) CvssMetrics {
	return CvssMetrics{
		BaseScore:           baseScore,
		ExploitabilityScore: &exploitabilityScore,
		ImpactScore:         &impactScore,
	}
}
