package vulnerability

import (
	"fmt"

	grypeDB "github.com/anchore/grype/grype/db/v3"
	"github.com/anchore/grype/grype/version"
	"github.com/anchore/syft/syft/pkg"
)

type Reference struct {
	ID        string
	Namespace string
}

type Vulnerability struct {
	Constraint             version.Constraint
	CPEs                   []pkg.CPE
	ID                     string
	Namespace              string
	Fix                    Fix
	Advisories             []Advisory
	RelatedVulnerabilities []Reference
}

func NewVulnerability(vuln grypeDB.Vulnerability) (*Vulnerability, error) {
	format := version.ParseFormat(vuln.VersionFormat)

	constraint, err := version.GetConstraint(vuln.VersionConstraint, format)
	if err != nil {
		return nil, fmt.Errorf("failed to parse constraint='%s' format='%s': %w", vuln.VersionConstraint, format, err)
	}

	advisories := make([]Advisory, len(vuln.Advisories))
	for idx, advisory := range vuln.Advisories {
		advisories[idx] = Advisory{
			ID:   advisory.ID,
			Link: advisory.Link,
		}
	}

	var relatedVulnerabilities []Reference
	for _, r := range vuln.RelatedVulnerabilities {
		relatedVulnerabilities = append(relatedVulnerabilities, Reference{
			ID:        r.ID,
			Namespace: r.Namespace,
		})
	}

	return &Vulnerability{
		Constraint: constraint,
		ID:         vuln.ID,
		CPEs:       make([]pkg.CPE, 0),
		Namespace:  vuln.Namespace,
		Fix: Fix{
			Versions: vuln.Fix.Versions,
			State:    vuln.Fix.State,
		},
		Advisories:             advisories,
		RelatedVulnerabilities: relatedVulnerabilities,
	}, nil
}

func (v Vulnerability) String() string {
	return fmt.Sprintf("Vuln(id=%s constraint='%s')", v.ID, v.Constraint)
}

func (v *Vulnerability) hash() string {
	return fmt.Sprintf("%s|%s|%+v", v.ID, v.Constraint, v.CPEs)
}
