package vulnerability

import (
	"testing"

	"github.com/stretchr/testify/assert"

	"github.com/go-test/deep"

	"github.com/anchore/grype/grype/pkg"
	"github.com/anchore/grype/grype/version"
	"github.com/anchore/syft/syft/distro"
	syftPkg "github.com/anchore/syft/syft/pkg"
)

func TestGetByDistro(t *testing.T) {
	provider := NewProviderFromStore(newMockStore())

	d, err := distro.NewDistro(distro.Debian, "8", "")
	if err != nil {
		t.Fatalf("failed to create distro: %+v", err)
	}

	p := pkg.Package{
		Name: "neutron",
	}

	actual, err := provider.GetByDistro(&d, p)
	if err != nil {
		t.Fatalf("failed to get by distro: %+v", err)
	}

	expected := []Vulnerability{
		{
			Constraint: version.MustGetConstraint("< 2014.1.3-6", version.DebFormat),
			ID:         "CVE-2014-fake-1",
			Namespace:  "debian:8",
			CPEs:       []syftPkg.CPE{},
			Advisories: []Advisory{},
		},
		{
			Constraint: version.MustGetConstraint("< 2013.0.2-1", version.DebFormat),
			ID:         "CVE-2013-fake-2",
			Namespace:  "debian:8",
			CPEs:       []syftPkg.CPE{},
			Advisories: []Advisory{},
		},
	}

	assert.Len(t, actual, len(expected))

	for idx, vuln := range actual {
		for _, d := range deep.Equal(expected[idx], vuln) {
			t.Errorf("diff: %+v", d)
		}
	}
}

func TestGetByDistro_nilDistro(t *testing.T) {
	provider := NewProviderFromStore(newMockStore())

	p := pkg.Package{
		Name: "neutron",
	}

	vulnerabilities, err := provider.GetByDistro(nil, p)

	assert.Empty(t, vulnerabilities)
	assert.NoError(t, err)
}

func must(c syftPkg.CPE, e error) syftPkg.CPE {
	if e != nil {
		panic(e)
	}
	return c
}

func TestGetByCPE(t *testing.T) {

	tests := []struct {
		name     string
		cpe      syftPkg.CPE
		expected []Vulnerability
		err      bool
	}{
		{
			name: "match from name and target SW",
			cpe:  must(syftPkg.NewCPE("cpe:2.3:*:activerecord:activerecord:*:*:*:*:*:ruby:*:*")),
			expected: []Vulnerability{
				{
					Constraint: version.MustGetConstraint("< 3.7.4", version.UnknownFormat),
					ID:         "CVE-2014-fake-4",
					CPEs: []syftPkg.CPE{
						must(syftPkg.NewCPE("cpe:2.3:*:activerecord:activerecord:*:*:something:*:*:ruby:*:*")),
					},
					Namespace:  "nvd",
					Advisories: []Advisory{},
				},
			},
		},

		{
			name: "match from vendor & name",
			cpe:  must(syftPkg.NewCPE("cpe:2.3:*:activerecord:activerecord:*:*:*:*:*:*:*:*")),
			expected: []Vulnerability{
				{
					Constraint: version.MustGetConstraint("< 3.7.6", version.UnknownFormat),
					ID:         "CVE-2014-fake-3",
					CPEs: []syftPkg.CPE{
						must(syftPkg.NewCPE("cpe:2.3:*:activerecord:activerecord:*:*:*:*:*:rails:*:*")),
					},
					Namespace:  "nvd",
					Advisories: []Advisory{},
				},
				{
					Constraint: version.MustGetConstraint("< 3.7.4", version.UnknownFormat),
					ID:         "CVE-2014-fake-4",
					CPEs: []syftPkg.CPE{
						must(syftPkg.NewCPE("cpe:2.3:*:activerecord:activerecord:*:*:something:*:*:ruby:*:*")),
					},
					Namespace:  "nvd",
					Advisories: []Advisory{},
				},
			},
		},

		{
			name: "dont allow any name",
			cpe:  must(syftPkg.NewCPE("cpe:2.3:*:couldntgetthisrightcouldyou:*:*:*:*:*:*:*:*:*")),
			err:  true,
		},
	}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {

			provider := NewProviderFromStore(newMockStore())

			actual, err := provider.GetByCPE(test.cpe)
			if err != nil && !test.err {
				t.Fatalf("expected no err, got: %+v", err)
			} else if err == nil && test.err {
				t.Fatalf("expected an err, got none")
			}

			assert.Len(t, actual, len(test.expected))

			for idx, vuln := range actual {
				for _, d := range deep.Equal(test.expected[idx], vuln) {
					t.Errorf("diff: %+v", d)
				}
			}
		})
	}

}
