package resource

import (
	"strings"
	"time"

	"github.com/aelsabbahy/goss/system"
	"github.com/aelsabbahy/goss/util"
)

type DNS struct {
	Title       string  `json:"title,omitempty" yaml:"title,omitempty"`
	Meta        meta    `json:"meta,omitempty" yaml:"meta,omitempty"`
	Host        string  `json:"-" yaml:"-"`
	Resolveable matcher `json:"resolveable,omitempty" yaml:"resolveable,omitempty"`
	Resolvable  matcher `json:"resolvable" yaml:"resolvable"`
	Addrs       matcher `json:"addrs,omitempty" yaml:"addrs,omitempty"`
	Timeout     int     `json:"timeout" yaml:"timeout"`
	Server      string  `json:"server,omitempty" yaml:"server,omitempty"`
	Skip        bool    `json:"skip,omitempty" yaml:"skip,omitempty"`
}

func (d *DNS) ID() string      { return d.Host }
func (d *DNS) SetID(id string) { d.Host = id }

func (d *DNS) GetTitle() string { return d.Title }
func (d *DNS) GetMeta() meta    { return d.Meta }

func (d *DNS) Validate(sys *system.System) []TestResult {
	skip := false
	if d.Timeout == 0 {
		d.Timeout = 500
	}
	if d.Skip {
		skip = true
	}

	sysDNS := sys.NewDNS(d.Host, sys, util.Config{Timeout: time.Duration(d.Timeout) * time.Millisecond, Server: d.Server})

	var results []TestResult
	// Backwards copatibility hack for now
	if d.Resolvable == nil {
		d.Resolvable = d.Resolveable
	}
	results = append(results, ValidateValue(d, "resolvable", d.Resolvable, sysDNS.Resolvable, skip))
	if shouldSkip(results) {
		skip = true
	}
	if d.Addrs != nil {
		results = append(results, ValidateValue(d, "addrs", d.Addrs, sysDNS.Addrs, skip))
	}
	return results
}

func NewDNS(sysDNS system.DNS, config util.Config) (*DNS, error) {
	var host string
	if sysDNS.Qtype() != "" {
		host = strings.Join([]string{sysDNS.Qtype(), sysDNS.Host()}, ":")
	} else {
		host = sysDNS.Host()
	}

	resolvable, err := sysDNS.Resolvable()
	server := sysDNS.Server()

	d := &DNS{
		Host:       host,
		Resolvable: resolvable,
		Timeout:    config.TimeOutMilliSeconds(),
		Server:     server,
	}
	if !contains(config.IgnoreList, "addrs") {
		addrs, _ := sysDNS.Addrs()
		d.Addrs = addrs
	}
	return d, err
}
