package main

import (
	"fmt"
	"path"

	"github.com/goreleaser/goreleaser/pkg/config"
)

// processEquinoxio create a fake goreleaser config for equinox.io
// and use a similar template.
func processRaw(repo string, exe string, nametpl string) ([]byte, error) {
	if repo == "" {
		return nil, fmt.Errorf("must have GitHub owner/repo")
	}
	if exe == "" {
		exe = path.Base(repo)
	}
	if nametpl == "" {
		nametpl = "{{ .Binary }}_v{{ .Version }}_{{ .Os }}_{{ .Arch }}"
	}

	// translate golang template to shell string
	name, err := makeName("NAME=", nametpl)
	if err != nil {
		return nil, err
	}

	project := config.Project{}
	project.Release.GitHub.Owner = path.Dir(repo)
	project.Release.GitHub.Name = path.Base(repo)
	project.Builds = []config.Build{
		{Binary: exe},
	}
	project.Archive.NameTemplate = name
	return makeShell(shellRaw, &project)
}

const shellRaw = `#!/bin/sh
set -e
# Code generated by godownloader on {{ timestamp }}. DO NOT EDIT.
#

usage() {
  this=$1
  cat <<EOF

$this: download binaries for {{ $.Release.GitHub.Owner }}/{{ $.Release.GitHub.Name }}

Usage: $this [-b bindir] [-d] [tag]
  -b sets bindir or installation directory, Defaults to ./bin
  -d turns on debug logging
  [tag] is a tag from
  https://github.com/{{ $.Release.GitHub.Owner }}/{{ $.Release.GitHub.Name }}/releases
  If tag is missing, then the latest release will be used.

Generated by godownloader
 https://github.com/goreleaser/godownloader

EOF
  exit 2
}
parse_args() {
  #BINDIR is ./bin unless set be ENV
  # over-ridden by flag below

  BINDIR=${BINDIR:-./bin}
  while getopts "b:dh?x" arg; do
    case "$arg" in
      b) BINDIR="$OPTARG" ;;
      d) log_set_priority 10 ;;
      h | \?) usage "$0" ;;
      x) set -x ;;
    esac
  done
  shift $((OPTIND - 1))
  TAG=$1
}
tag_to_version() {
  if [ -z "${TAG}" ]; then
    log_info "checking GitHub for latest tag"
  else
    log_info "checking GitHub for tag '${TAG}'"
  fi
  REALTAG=$(github_release "$OWNER/$REPO" "${TAG}") && true
  if test -z "$REALTAG"; then
    log_crit "unable to find '${TAG}' - use 'latest' or see https://github.com/${PREFIX}/releases for details"
    exit 1
  fi
  # if version starts with 'v', remove it
  TAG="$REALTAG"
  VERSION=${TAG#v}
}
adjust_binary() {
  if [ "$OS" = "windows" ]; then
    NAME="${NAME}.exe"
    BINARY="${BINARY}.exe"
  fi
}
# wrap all destructive operations into a function
# to prevent curl|bash network truncation and disaster
execute() {
  TMPDIR=$(mktemp -d)
  log_info "downloading from ${TARBALL_URL}"
  http_download "${TMPDIR}/${NAME}" "$TARBALL_URL"
  test ! -d "${BINDIR}" && install -d "${BINDIR}"
  install "${TMPDIR}/${NAME}" "${BINDIR}/${BINARY}"
  log_info "installed ${BINDIR}/${BINARY}"
}
` + shellfn + `
OWNER={{ .Release.GitHub.Owner }}
REPO="{{ .Release.GitHub.Name }}"
BINARY={{ (index .Builds 0).Binary }}
BINDIR=${BINDIR:-./bin}
PREFIX="$OWNER/$REPO"
# use in logging routines
log_prefix() {
        echo "$PREFIX"
}
OS=$(uname_os)
ARCH=$(uname_arch)
GITHUB_DOWNLOAD=https://github.com/${OWNER}/${REPO}/releases/download

# make sure we are on a platform that makes sense
uname_os_check "$OS"
uname_arch_check "$ARCH"

# parse_args, show usage and exit if necessary
parse_args "$@"

# setup version from tag
tag_to_version

log_info "found version ${VERSION} for ${TAG}/${OS}/${ARCH}"

{{ .Archive.NameTemplate }}

# adjust binary name based on OS
adjust_binary

# compute URL to download
TARBALL_URL=${GITHUB_DOWNLOAD}/${TAG}/${NAME}

# do it
execute
`
