/*
 Copyright 2020 The GoPlus Authors (goplus.org)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
*/

package parser

import (
	"bytes"
	"fmt"
	"reflect"
	"strings"
	"testing"

	"github.com/goplus/gop/ast"
	"github.com/goplus/gop/ast/asttest"
	"github.com/goplus/gop/token"
	"github.com/qiniu/x/log"
)

func init() {
	log.SetFlags(log.Ldefault &^ log.LstdFlags)
	log.SetOutputLevel(log.Ldebug)
}

// -----------------------------------------------------------------------------

var fsTest1 = asttest.NewSingleFileFS("/foo", "bar.gop", `package bar; import "io"
func New() (*Bar, error) {
	return nil, io.EOF
}

bar, err := New()
if err != nil {
	log.Println(err)
}`)

func TestParseBarPackage(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTest1, "/foo", nil, 0)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar := pkgs["bar"]
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	if !checkFSTest(file, []string{`"io"`}, []string{"init", "New"}) {
		t.Error("file check file")
	}
}

// -----------------------------------------------------------------------------

var fsTest2 = asttest.NewSingleFileFS("/foo", "bar.gop", `import "io"
func New() (*Bar, error) {
	return nil, io.EOF
}

bar, err := New()
if err != nil {
	log.Println(err)
}
`)

func TestParseNoPackageAndGlobalCode(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTest2, "/foo", nil, 0)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestParseNoPackageAndGlobalCode failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	if !checkFSTest(file, []string{`"io"`}, []string{"New", "main"}) {
		t.Error("file check file")
	}
}

// -----------------------------------------------------------------------------

const script3 = `package bar; import "io"
func New() (*Bar, error) {
	return nil, io.EOF
}

bar, err := New()
if err != nil {
	log.Println(err)
}`

func TestParse(t *testing.T) {
	fset := token.NewFileSet()
	local = asttest.NewSingleFileFS("/foo", "bar.gop", script3)
	defer func() {
		local = localFS{}
	}()

	// test parse file
	pkgs, err := Parse(fset, "/foo/bar.gop", nil, 0)
	if err != nil {
		t.Error("parse err!", err)
		return
	}
	pkg, ok := pkgs["bar"]
	if ok != true {
		t.Error("pkg not found")
		return
	}
	file, ok := pkg.Files["/foo/bar.gop"]
	if ok != true {
		t.Error("file not found")
		return
	}
	if !checkFSTest(file, []string{`"io"`}, []string{"init", "New"}) {
		t.Error("file check file")
	}
}

func doTestParse(t *testing.T, src interface{}) {
	fset := token.NewFileSet()
	pkgs, err := Parse(fset, "/foo/bar.gop", src, 0)
	if err != nil {
		t.Error("parse err!", err)
		return
	}
	pkg, ok := pkgs["bar"]
	if ok != true {
		t.Error("pkg not found")
		return
	}
	file, ok := pkg.Files["/foo/bar.gop"]
	if ok != true {
		t.Error("file not found")
		return
	}
	if !checkFSTest(file, []string{`"io"`}, []string{"init", "New"}) {
		t.Error("file check file")
	}
}

func TestParse2(t *testing.T) {
	doTestParse(t, script3)
	doTestParse(t, bytes.NewBufferString(script3))
	doTestParse(t, strings.NewReader(script3))
}

func checkFSTest(file *ast.File, targetImport []string, targetFuncDecl []string) (pass bool) {
	var foundImport, foundFunc int
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					for _, ti := range targetImport {
						if ti == v.Path.Value {
							foundImport++
						}
					}
				}
			}
		case *ast.FuncDecl:
			fmt.Println(d.Name.Name)
			for _, tfd := range targetFuncDecl {
				if tfd == d.Name.Name {
					foundFunc++
				}
			}
		}
	}
	if foundFunc != len(targetFuncDecl) || foundImport != len(targetImport) {
		fmt.Println("invalid check", foundImport, foundFunc)
		return false
	}
	return true
}

// -----------------------------------------------------------------------------

var fsTestMapLit = asttest.NewSingleFileFS("/foo", "bar.gop", `
	x := {"Hello": 1, "xsw": 3.4}
	println("x:", x)
`)

func TestMapLit(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestMapLit, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestMap failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestSliceLit = asttest.NewSingleFileFS("/foo", "bar.gop", `
	x := [1, 3.4, 5.6]
	println("x:", x)
`)

func TestSliceLit(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestSliceLit, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestMap failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestSliceLit2 = asttest.NewSingleFileFS("/foo", "bar.gop", `
	x := [5.6]
	println("x:", x)
`)

func TestSliceLit2(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestSliceLit2, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestMap failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestSliceLit3 = asttest.NewSingleFileFS("/foo", "bar.gop", `
	x := []
	println("x:", x)
`)

func TestSliceLit3(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestSliceLit3, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestMap failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestListComprehension = asttest.NewSingleFileFS("/foo", "bar.gop", `
	x := [x*x for x <- [1, 2, 3, 4], x > 2]
	println("x:", x)
`)

func TestListComprehension(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestListComprehension, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestMap failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestMapComprehension = asttest.NewSingleFileFS("/foo", "bar.gop", `
	x := {v: k*k for k, v <- [3, 5, 7, 11]}
	println("x:", x)
`)

func TestMapComprehension(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestMapComprehension, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestMap failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestMapComprehensionCond = asttest.NewSingleFileFS("/foo", "bar.gop", `
	x := {v: k*k for k, v <- [3, 5, 7, 11], k % 2 == 1}
	println("x:", x)

	x = map[int]int{}
	for k, v := range []int{3, 5, 7, 11} {
		if k % 2 == 1 {
			x[v] = k*k
		}
	}
	println("x:", x)
`)

func TestMapComprehensionCond(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestMapComprehensionCond, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestMap failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestListComprehensionEx = asttest.NewSingleFileFS("/foo", "bar.gop", `
	arr := [1, 2, 3, 4, 5, 6]
	x := [[a, b] for a <- arr, a < b for b <- arr, b > 2]
	println("x:", x)
`)

func TestListComprehensionEx(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestListComprehensionEx, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestMap failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestFor = asttest.NewSingleFileFS("/foo", "bar.gop", `
	sum := 0
	for x <- [1, 3, 5, 7, 11], x > 3 {
		sum += x
	}
	println("sum(5,7,11):", sum)

	sum = 0
	for i, x <- [1, 3, 5, 7, 11] {
		sum += x
	}
	println("sum(1,3,5,7,11):", sum)
`)

func TestFor(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestFor, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestMap failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestMake = asttest.NewSingleFileFS("/foo", "bar.gop", `
	a := make([]int, 0, 4)
	a = append(a, [1, 2, 3]...)

	b := make(map[string]interface{ Error() string })
	b := make(chan bool)
`)

func TestMake(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestMake, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestMap failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestXOR = asttest.NewSingleFileFS("/foo", "bar.gop", `
	println(^uint32(1))
`)

func TestXOR(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestXOR, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestTILDE failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------

var fsTestRational = asttest.NewSingleFileFS("/foo", "bar.gop", `
	println(5/7r + 3.4r)
`)

func TestRational(t *testing.T) {
	fset := token.NewFileSet()
	pkgs, err := ParseFSDir(fset, fsTestRational, "/foo", nil, Trace)
	if err != nil || len(pkgs) != 1 {
		t.Fatal("ParseFSDir failed:", err, len(pkgs))
	}
	bar, isMain := pkgs["main"]
	if !isMain {
		t.Fatal("TestTILDE failed: not main")
	}
	file := bar.Files["/foo/bar.gop"]
	fmt.Println("Pkg:", file.Name)
	for _, decl := range file.Decls {
		fmt.Println("decl:", reflect.TypeOf(decl))
		switch d := decl.(type) {
		case *ast.GenDecl:
			for _, spec := range d.Specs {
				switch v := spec.(type) {
				case *ast.ImportSpec:
					fmt.Println(" - import:", v.Path.Value)
				}
			}
		case *ast.FuncDecl:
			fmt.Println(" - func:", d.Name.Name)
		}
	}
}

// -----------------------------------------------------------------------------
