// Copyright 2019 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package generator

// microgenConfig represents a single microgen target.
type microgenConfig struct {
	// inputDirectoryPath is the path to the input (.proto, etc) files, relative
	// to googleapisDir.
	inputDirectoryPath string

	// importPath is the path that this library should be imported as.
	importPath string

	// pkg is the name that should be used in the package declaration.
	pkg string

	// gRPCServiceConfigPath is the path to the grpc service config for this
	// target, relative to googleapisDir/inputDirectoryPath.
	gRPCServiceConfigPath string

	// apiServiceConfigPath is the path to the gapic service config for this
	// target, relative to googleapisDir/inputDirectoryPath.
	apiServiceConfigPath string

	// releaseLevel is the release level of this target. Values incl ga,
	// beta, alpha.
	releaseLevel string

	// stopGeneration is used to stop generating a given client. This might be
	// useful if a client needs to be deprecated, but retained in the repo
	// metadata.
	stopGeneration bool

	// disableMetadata is used to toggle generation of the gapic_metadata.json
	// file for the client library.
	disableMetadata bool

	// transports is a list of transports to generate a client for. Acceptable
	// values are 'grpc' and 'rest'
	transports []string

	// googleapisDiscovery indicates if the protos reside in googleapis-discovery
	// or not. Default is false, and will be looked up in googleapis.
	googleapisDiscovery bool
}

var microgenGapicConfigs = []*microgenConfig{
	// Cloud APIs
	{
		inputDirectoryPath:   "google/cloud/compute/v1",
		pkg:                  "compute",
		importPath:           "cloud.google.com/go/compute/apiv1",
		apiServiceConfigPath: "compute_v1.yaml",
		transports:           []string{"rest"},
		// TODO: Change to "ga" when ready.
		releaseLevel:        "beta",
		googleapisDiscovery: true,
	},
	{
		inputDirectoryPath:    "google/cloud/texttospeech/v1",
		pkg:                   "texttospeech",
		importPath:            "cloud.google.com/go/texttospeech/apiv1",
		gRPCServiceConfigPath: "texttospeech_grpc_service_config.json",
		apiServiceConfigPath:  "texttospeech_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/asset/v1",
		pkg:                   "asset",
		importPath:            "cloud.google.com/go/asset/apiv1",
		gRPCServiceConfigPath: "cloudasset_grpc_service_config.json",
		apiServiceConfigPath:  "cloudasset_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/billing/v1",
		pkg:                   "billing",
		importPath:            "cloud.google.com/go/billing/apiv1",
		gRPCServiceConfigPath: "cloud_billing_grpc_service_config.json",
		apiServiceConfigPath:  "cloudbilling.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/language/v1",
		pkg:                   "language",
		importPath:            "cloud.google.com/go/language/apiv1",
		gRPCServiceConfigPath: "language_grpc_service_config.json",
		apiServiceConfigPath:  "language_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/language/v1beta2",
		pkg:                   "language",
		importPath:            "cloud.google.com/go/language/apiv1beta2",
		gRPCServiceConfigPath: "language_grpc_service_config.json",
		apiServiceConfigPath:  "language_v1beta2.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/memcache/v1",
		pkg:                   "memcache",
		importPath:            "cloud.google.com/go/memcache/apiv1",
		gRPCServiceConfigPath: "memcache_grpc_service_config.json",
		apiServiceConfigPath:  "memcache_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/memcache/v1beta2",
		pkg:                   "memcache",
		importPath:            "cloud.google.com/go/memcache/apiv1beta2",
		gRPCServiceConfigPath: "memcache_grpc_service_config.json",
		apiServiceConfigPath:  "memcache_v1beta2.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/phishingprotection/v1beta1",
		pkg:                   "phishingprotection",
		importPath:            "cloud.google.com/go/phishingprotection/apiv1beta1",
		gRPCServiceConfigPath: "phishingprotection_grpc_service_config.json",
		apiServiceConfigPath:  "phishingprotection_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/translate/v3",
		pkg:                   "translate",
		importPath:            "cloud.google.com/go/translate/apiv3",
		gRPCServiceConfigPath: "translate_grpc_service_config.json",
		apiServiceConfigPath:  "translate_v3.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/scheduler/v1",
		pkg:                   "scheduler",
		importPath:            "cloud.google.com/go/scheduler/apiv1",
		gRPCServiceConfigPath: "cloudscheduler_grpc_service_config.json",
		apiServiceConfigPath:  "cloudscheduler_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/scheduler/v1beta1",
		pkg:                   "scheduler",
		importPath:            "cloud.google.com/go/scheduler/apiv1beta1",
		gRPCServiceConfigPath: "cloudscheduler_grpc_service_config.json",
		apiServiceConfigPath:  "cloudscheduler_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/speech/v1",
		pkg:                   "speech",
		importPath:            "cloud.google.com/go/speech/apiv1",
		gRPCServiceConfigPath: "speech_grpc_service_config.json",
		apiServiceConfigPath:  "speech_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/speech/v1p1beta1",
		pkg:                   "speech",
		importPath:            "cloud.google.com/go/speech/apiv1p1beta1",
		gRPCServiceConfigPath: "speech_grpc_service_config.json",
		apiServiceConfigPath:  "speech_v1p1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/bigquery/connection/v1beta1",
		pkg:                   "connection",
		importPath:            "cloud.google.com/go/bigquery/connection/apiv1beta1",
		gRPCServiceConfigPath: "bigqueryconnection_grpc_service_config.json",
		apiServiceConfigPath:  "bigqueryconnection_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/bigquery/connection/v1",
		pkg:                   "connection",
		importPath:            "cloud.google.com/go/bigquery/connection/apiv1",
		gRPCServiceConfigPath: "bigqueryconnection_grpc_service_config.json",
		apiServiceConfigPath:  "bigqueryconnection_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/bigquery/datatransfer/v1",
		pkg:                   "datatransfer",
		importPath:            "cloud.google.com/go/bigquery/datatransfer/apiv1",
		gRPCServiceConfigPath: "bigquerydatatransfer_grpc_service_config.json",
		apiServiceConfigPath:  "bigquerydatatransfer_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/bigquery/migration/v2alpha",
		pkg:                   "migration",
		importPath:            "cloud.google.com/go/bigquery/migration/apiv2alpha",
		gRPCServiceConfigPath: "bigquerymigration_grpc_service_config.json",
		apiServiceConfigPath:  "bigquerymigration_v2alpha.yaml",
		releaseLevel:          "alpha",
	},
	{
		inputDirectoryPath:    "google/cloud/bigquery/reservation/v1beta1",
		pkg:                   "reservation",
		importPath:            "cloud.google.com/go/bigquery/reservation/apiv1beta1",
		gRPCServiceConfigPath: "bigqueryreservation_grpc_service_config.json",
		apiServiceConfigPath:  "bigqueryreservation_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/bigquery/reservation/v1",
		pkg:                   "reservation",
		importPath:            "cloud.google.com/go/bigquery/reservation/apiv1",
		gRPCServiceConfigPath: "bigqueryreservation_grpc_service_config.json",
		apiServiceConfigPath:  "bigqueryreservation_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/bigquery/storage/v1beta1",
		pkg:                   "storage",
		importPath:            "cloud.google.com/go/bigquery/storage/apiv1beta1",
		gRPCServiceConfigPath: "bigquerystorage_grpc_service_config.json",
		apiServiceConfigPath:  "bigquerystorage_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/bigquery/storage/v1beta2",
		pkg:                   "storage",
		importPath:            "cloud.google.com/go/bigquery/storage/apiv1beta2",
		gRPCServiceConfigPath: "bigquerystorage_grpc_service_config.json",
		apiServiceConfigPath:  "bigquerystorage_v1beta2.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/bigquery/storage/v1",
		pkg:                   "storage",
		importPath:            "cloud.google.com/go/bigquery/storage/apiv1",
		gRPCServiceConfigPath: "bigquerystorage_grpc_service_config.json",
		apiServiceConfigPath:  "bigquerystorage_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/iot/v1",
		pkg:                   "iot",
		importPath:            "cloud.google.com/go/iot/apiv1",
		gRPCServiceConfigPath: "cloudiot_grpc_service_config.json",
		apiServiceConfigPath:  "cloudiot_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/recommender/v1beta1",
		pkg:                   "recommender",
		importPath:            "cloud.google.com/go/recommender/apiv1beta1",
		gRPCServiceConfigPath: "recommender_grpc_service_config.json",
		apiServiceConfigPath:  "recommender_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/tasks/v2",
		pkg:                   "cloudtasks",
		importPath:            "cloud.google.com/go/cloudtasks/apiv2",
		gRPCServiceConfigPath: "cloudtasks_grpc_service_config.json",
		apiServiceConfigPath:  "cloudtasks_v2.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/tasks/v2beta2",
		pkg:                   "cloudtasks",
		importPath:            "cloud.google.com/go/cloudtasks/apiv2beta2",
		gRPCServiceConfigPath: "cloudtasks_grpc_service_config.json",
		apiServiceConfigPath:  "cloudtasks_v2beta2.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/tasks/v2beta3",
		pkg:                   "cloudtasks",
		importPath:            "cloud.google.com/go/cloudtasks/apiv2beta3",
		gRPCServiceConfigPath: "cloudtasks_grpc_service_config.json",
		apiServiceConfigPath:  "cloudtasks_v2beta3.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/videointelligence/v1",
		pkg:                   "videointelligence",
		importPath:            "cloud.google.com/go/videointelligence/apiv1",
		gRPCServiceConfigPath: "videointelligence_grpc_service_config.json",
		apiServiceConfigPath:  "videointelligence_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/vision/v1",
		pkg:                   "vision",
		importPath:            "cloud.google.com/go/vision/apiv1",
		gRPCServiceConfigPath: "vision_grpc_service_config.json",
		apiServiceConfigPath:  "vision_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/webrisk/v1",
		pkg:                   "webrisk",
		importPath:            "cloud.google.com/go/webrisk/apiv1",
		gRPCServiceConfigPath: "webrisk_grpc_service_config.json",
		apiServiceConfigPath:  "webrisk_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/webrisk/v1beta1",
		pkg:                   "webrisk",
		importPath:            "cloud.google.com/go/webrisk/apiv1beta1",
		gRPCServiceConfigPath: "webrisk_grpc_service_config.json",
		apiServiceConfigPath:  "webrisk_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/secretmanager/v1",
		pkg:                   "secretmanager",
		importPath:            "cloud.google.com/go/secretmanager/apiv1",
		gRPCServiceConfigPath: "secretmanager_grpc_service_config.json",
		apiServiceConfigPath:  "secretmanager_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/secrets/v1beta1",
		pkg:                   "secretmanager",
		importPath:            "cloud.google.com/go/secretmanager/apiv1beta1",
		gRPCServiceConfigPath: "secretmanager_grpc_service_config.json",
		apiServiceConfigPath:  "secretmanager_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/osconfig/v1",
		pkg:                   "osconfig",
		importPath:            "cloud.google.com/go/osconfig/apiv1",
		gRPCServiceConfigPath: "osconfig_grpc_service_config.json",
		apiServiceConfigPath:  "osconfig_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/osconfig/v1alpha",
		pkg:                   "osconfig",
		importPath:            "cloud.google.com/go/osconfig/apiv1alpha",
		gRPCServiceConfigPath: "osconfig_grpc_service_config.json",
		apiServiceConfigPath:  "osconfig_v1alpha.yaml",
		releaseLevel:          "alpha",
	},
	{
		inputDirectoryPath:    "google/cloud/osconfig/v1beta",
		pkg:                   "osconfig",
		importPath:            "cloud.google.com/go/osconfig/apiv1beta",
		gRPCServiceConfigPath: "osconfig_grpc_service_config.json",
		apiServiceConfigPath:  "osconfig_v1beta.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/osconfig/agentendpoint/v1",
		pkg:                   "agentendpoint",
		importPath:            "cloud.google.com/go/osconfig/agentendpoint/apiv1",
		gRPCServiceConfigPath: "agentendpoint_grpc_service_config.json",
		apiServiceConfigPath:  "osconfig_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/osconfig/agentendpoint/v1beta",
		pkg:                   "agentendpoint",
		importPath:            "cloud.google.com/go/osconfig/agentendpoint/apiv1beta",
		gRPCServiceConfigPath: "agentendpoint_grpc_service_config.json",
		apiServiceConfigPath:  "osconfig_v1beta.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/datacatalog/v1",
		pkg:                   "datacatalog",
		importPath:            "cloud.google.com/go/datacatalog/apiv1",
		gRPCServiceConfigPath: "datacatalog_grpc_service_config.json",
		apiServiceConfigPath:  "datacatalog_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/datacatalog/v1beta1",
		pkg:                   "datacatalog",
		importPath:            "cloud.google.com/go/datacatalog/apiv1beta1",
		gRPCServiceConfigPath: "datacatalog_grpc_service_config.json",
		apiServiceConfigPath:  "datacatalog_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/dataproc/v1",
		pkg:                   "dataproc",
		importPath:            "cloud.google.com/go/dataproc/apiv1",
		gRPCServiceConfigPath: "dataproc_grpc_service_config.json",
		apiServiceConfigPath:  "dataproc_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/kms/v1",
		pkg:                   "kms",
		importPath:            "cloud.google.com/go/kms/apiv1",
		gRPCServiceConfigPath: "cloudkms_grpc_service_config.json",
		apiServiceConfigPath:  "cloudkms_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/oslogin/v1",
		pkg:                   "oslogin",
		importPath:            "cloud.google.com/go/oslogin/apiv1",
		gRPCServiceConfigPath: "oslogin_grpc_service_config.json",
		apiServiceConfigPath:  "oslogin_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/oslogin/v1beta",
		pkg:                   "oslogin",
		importPath:            "cloud.google.com/go/oslogin/apiv1beta",
		gRPCServiceConfigPath: "oslogin_grpc_service_config.json",
		apiServiceConfigPath:  "oslogin_v1beta.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/recaptchaenterprise/v1",
		pkg:                   "recaptchaenterprise",
		importPath:            "cloud.google.com/go/recaptchaenterprise/apiv1",
		gRPCServiceConfigPath: "recaptchaenterprise_grpc_service_config.json",
		apiServiceConfigPath:  "recaptchaenterprise_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/recaptchaenterprise/v1beta1",
		pkg:                   "recaptchaenterprise",
		importPath:            "cloud.google.com/go/recaptchaenterprise/apiv1beta1",
		gRPCServiceConfigPath: "recaptchaenterprise_grpc_service_config.json",
		apiServiceConfigPath:  "recaptchaenterprise_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/redis/v1",
		pkg:                   "redis",
		importPath:            "cloud.google.com/go/redis/apiv1",
		gRPCServiceConfigPath: "redis_grpc_service_config.json",
		apiServiceConfigPath:  "redis_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/redis/v1beta1",
		pkg:                   "redis",
		importPath:            "cloud.google.com/go/redis/apiv1beta1",
		gRPCServiceConfigPath: "redis_grpc_service_config.json",
		apiServiceConfigPath:  "redis_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/devtools/clouddebugger/v2",
		pkg:                   "debugger",
		importPath:            "cloud.google.com/go/debugger/apiv2",
		gRPCServiceConfigPath: "clouddebugger_grpc_service_config.json",
		apiServiceConfigPath:  "clouddebugger_v2.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/devtools/clouderrorreporting/v1beta1",
		pkg:                   "errorreporting",
		importPath:            "cloud.google.com/go/errorreporting/apiv1beta1",
		gRPCServiceConfigPath: "errorreporting_grpc_service_config.json",
		apiServiceConfigPath:  "clouderrorreporting_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/devtools/cloudtrace/v1",
		pkg:                   "trace",
		importPath:            "cloud.google.com/go/trace/apiv1",
		gRPCServiceConfigPath: "cloudtrace_grpc_service_config.json",
		apiServiceConfigPath:  "cloudtrace_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/devtools/cloudtrace/v2",
		pkg:                   "trace",
		importPath:            "cloud.google.com/go/trace/apiv2",
		gRPCServiceConfigPath: "cloudtrace_grpc_service_config.json",
		apiServiceConfigPath:  "cloudtrace_v2.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/privacy/dlp/v2",
		pkg:                   "dlp",
		importPath:            "cloud.google.com/go/dlp/apiv2",
		gRPCServiceConfigPath: "dlp_grpc_service_config.json",
		apiServiceConfigPath:  "dlp_v2.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/datastore/admin/v1",
		pkg:                   "admin",
		importPath:            "cloud.google.com/go/datastore/admin/apiv1",
		gRPCServiceConfigPath: "datastore_admin_grpc_service_config.json",
		apiServiceConfigPath:  "datastore_v1.yaml",
		releaseLevel:          "alpha",
	},
	{
		inputDirectoryPath:    "google/spanner/admin/database/v1",
		pkg:                   "database",
		importPath:            "cloud.google.com/go/spanner/admin/database/apiv1",
		gRPCServiceConfigPath: "spanner_admin_database_grpc_service_config.json",
		apiServiceConfigPath:  "spanner_admin_database.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/spanner/admin/instance/v1",
		pkg:                   "instance",
		importPath:            "cloud.google.com/go/spanner/admin/instance/apiv1",
		gRPCServiceConfigPath: "spanner_admin_instance_grpc_service_config.json",
		apiServiceConfigPath:  "spanner_admin_instance.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/spanner/v1",
		pkg:                   "spanner",
		importPath:            "cloud.google.com/go/spanner/apiv1",
		gRPCServiceConfigPath: "spanner_grpc_service_config.json",
		apiServiceConfigPath:  "spanner.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/securitycenter/settings/v1beta1",
		pkg:                   "settings",
		importPath:            "cloud.google.com/go/securitycenter/settings/apiv1beta1",
		gRPCServiceConfigPath: "securitycenter_settings_grpc_service_config.json",
		apiServiceConfigPath:  "securitycenter_settings.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/securitycenter/v1",
		pkg:                   "securitycenter",
		importPath:            "cloud.google.com/go/securitycenter/apiv1",
		gRPCServiceConfigPath: "securitycenter_grpc_service_config.json",
		apiServiceConfigPath:  "securitycenter_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/securitycenter/v1beta1",
		pkg:                   "securitycenter",
		importPath:            "cloud.google.com/go/securitycenter/apiv1beta1",
		gRPCServiceConfigPath: "securitycenter_grpc_service_config.json",
		apiServiceConfigPath:  "securitycenter_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/securitycenter/v1p1beta1",
		pkg:                   "securitycenter",
		importPath:            "cloud.google.com/go/securitycenter/apiv1p1beta1",
		gRPCServiceConfigPath: "securitycenter_grpc_service_config.json",
		apiServiceConfigPath:  "securitycenter_v1p1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/firestore/admin/v1",
		pkg:                   "apiv1",
		importPath:            "cloud.google.com/go/firestore/apiv1/admin",
		gRPCServiceConfigPath: "firestore_admin_grpc_service_config.json",
		apiServiceConfigPath:  "firestore_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/firestore/v1",
		pkg:                   "firestore",
		importPath:            "cloud.google.com/go/firestore/apiv1",
		gRPCServiceConfigPath: "firestore_grpc_service_config.json",
		apiServiceConfigPath:  "firestore_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/devtools/cloudbuild/v1",
		pkg:                   "cloudbuild",
		importPath:            "cloud.google.com/go/cloudbuild/apiv1/v2",
		gRPCServiceConfigPath: "cloudbuild_grpc_service_config.json",
		apiServiceConfigPath:  "cloudbuild_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/dialogflow/cx/v3beta1",
		pkg:                   "cx",
		importPath:            "cloud.google.com/go/dialogflow/cx/apiv3beta1",
		gRPCServiceConfigPath: "dialogflow_grpc_service_config.json",
		apiServiceConfigPath:  "dialogflow_v3beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/dialogflow/cx/v3",
		pkg:                   "cx",
		importPath:            "cloud.google.com/go/dialogflow/cx/apiv3",
		gRPCServiceConfigPath: "dialogflow_grpc_service_config.json",
		apiServiceConfigPath:  "dialogflow_v3.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/dialogflow/v2",
		pkg:                   "dialogflow",
		importPath:            "cloud.google.com/go/dialogflow/apiv2",
		gRPCServiceConfigPath: "dialogflow_grpc_service_config.json",
		apiServiceConfigPath:  "dialogflow_v2.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/iam/credentials/v1",
		pkg:                   "credentials",
		importPath:            "cloud.google.com/go/iam/credentials/apiv1",
		gRPCServiceConfigPath: "iamcredentials_grpc_service_config.json",
		apiServiceConfigPath:  "iamcredentials_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/longrunning",
		pkg:                   "longrunning",
		importPath:            "cloud.google.com/go/longrunning/autogen",
		gRPCServiceConfigPath: "longrunning_grpc_service_config.json",
		apiServiceConfigPath:  "longrunning.yaml",
		releaseLevel:          "alpha",
	},
	{
		inputDirectoryPath:    "google/devtools/containeranalysis/v1beta1",
		pkg:                   "containeranalysis",
		importPath:            "cloud.google.com/go/containeranalysis/apiv1beta1",
		gRPCServiceConfigPath: "containeranalysis_grpc_service_config.json",
		apiServiceConfigPath:  "containeranalysis_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		// The grafeas v1beta1 client must be generated in the same package as containeranalysis v1beta1,
		// but the proto is in a sub-directory of the containeranalysis v1beta1 protos.
		inputDirectoryPath:    "google/devtools/containeranalysis/v1beta1/grafeas",
		pkg:                   "containeranalysis",
		importPath:            "cloud.google.com/go/containeranalysis/apiv1beta1",
		gRPCServiceConfigPath: "../containeranalysis_grpc_service_config.json",
		apiServiceConfigPath:  "../containeranalysis_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/recommender/v1",
		pkg:                   "recommender",
		importPath:            "cloud.google.com/go/recommender/apiv1",
		gRPCServiceConfigPath: "recommender_grpc_service_config.json",
		apiServiceConfigPath:  "recommender_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/videointelligence/v1beta2",
		pkg:                   "videointelligence",
		importPath:            "cloud.google.com/go/videointelligence/apiv1beta2",
		gRPCServiceConfigPath: "videointelligence_grpc_service_config.json",
		apiServiceConfigPath:  "../videointelligence_v1beta2.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/asset/v1p2beta1",
		pkg:                   "asset",
		importPath:            "cloud.google.com/go/asset/apiv1p2beta1",
		gRPCServiceConfigPath: "cloudasset_grpc_service_config.json",
		apiServiceConfigPath:  "cloudasset_v1p2beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/asset/v1p5beta1",
		pkg:                   "asset",
		importPath:            "cloud.google.com/go/asset/apiv1p5beta1",
		gRPCServiceConfigPath: "cloudasset_grpc_service_config.json",
		apiServiceConfigPath:  "cloudasset_v1p5beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/monitoring/v3",
		pkg:                   "monitoring",
		importPath:            "cloud.google.com/go/monitoring/apiv3/v2",
		gRPCServiceConfigPath: "monitoring_grpc_service_config.json",
		apiServiceConfigPath:  "monitoring.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/vision/v1p1beta1",
		pkg:                   "vision",
		importPath:            "cloud.google.com/go/vision/apiv1p1beta1",
		gRPCServiceConfigPath: "vision_grpc_service_config.json",
		apiServiceConfigPath:  "vision_v1p1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/logging/v2",
		pkg:                   "logging",
		importPath:            "cloud.google.com/go/logging/apiv2",
		gRPCServiceConfigPath: "logging_grpc_service_config.json",
		apiServiceConfigPath:  "logging.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/talent/v4",
		pkg:                   "talent",
		importPath:            "cloud.google.com/go/talent/apiv4",
		gRPCServiceConfigPath: "talent_grpc_service_config.json",
		apiServiceConfigPath:  "jobs_v4.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/talent/v4beta1",
		pkg:                   "talent",
		importPath:            "cloud.google.com/go/talent/apiv4beta1",
		gRPCServiceConfigPath: "talent_grpc_service_config.json",
		apiServiceConfigPath:  "jobs_v4beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/pubsub/v1",
		pkg:                   "pubsub",
		importPath:            "cloud.google.com/go/pubsub/apiv1",
		gRPCServiceConfigPath: "pubsub_grpc_service_config.json",
		apiServiceConfigPath:  "pubsub_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/pubsublite/v1",
		pkg:                   "pubsublite",
		importPath:            "cloud.google.com/go/pubsublite/apiv1",
		gRPCServiceConfigPath: "pubsublite_grpc_service_config.json",
		apiServiceConfigPath:  "pubsublite_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/automl/v1",
		pkg:                   "automl",
		importPath:            "cloud.google.com/go/automl/apiv1",
		gRPCServiceConfigPath: "automl_grpc_service_config.json",
		apiServiceConfigPath:  "automl_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/automl/v1beta1",
		pkg:                   "automl",
		importPath:            "cloud.google.com/go/automl/apiv1beta1",
		gRPCServiceConfigPath: "automl_grpc_service_config.json",
		apiServiceConfigPath:  "automl_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/container/v1",
		pkg:                   "container",
		importPath:            "cloud.google.com/go/container/apiv1",
		gRPCServiceConfigPath: "container_grpc_service_config.json",
		apiServiceConfigPath:  "container_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/servicedirectory/v1",
		pkg:                   "servicedirectory",
		importPath:            "cloud.google.com/go/servicedirectory/apiv1",
		gRPCServiceConfigPath: "servicedirectory_grpc_service_config.json",
		apiServiceConfigPath:  "servicedirectory_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/servicedirectory/v1beta1",
		pkg:                   "servicedirectory",
		importPath:            "cloud.google.com/go/servicedirectory/apiv1beta1",
		gRPCServiceConfigPath: "servicedirectory_grpc_service_config.json",
		apiServiceConfigPath:  "servicedirectory_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/gaming/v1",
		pkg:                   "gaming",
		importPath:            "cloud.google.com/go/gaming/apiv1",
		gRPCServiceConfigPath: "gaming_grpc_service_config.json",
		apiServiceConfigPath:  "gameservices_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/gaming/v1beta",
		pkg:                   "gaming",
		importPath:            "cloud.google.com/go/gaming/apiv1beta",
		gRPCServiceConfigPath: "gaming_grpc_service_config.json",
		apiServiceConfigPath:  "gameservices_v1beta.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/policytroubleshooter/v1",
		pkg:                   "policytroubleshooter",
		importPath:            "cloud.google.com/go/policytroubleshooter/apiv1",
		gRPCServiceConfigPath: "checker_grpc_service_config.json",
		apiServiceConfigPath:  "policytroubleshooter_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/monitoring/dashboard/v1",
		pkg:                   "dashboard",
		importPath:            "cloud.google.com/go/monitoring/dashboard/apiv1",
		gRPCServiceConfigPath: "dashboards_grpc_service_config.json",
		apiServiceConfigPath:  "monitoring.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/functions/v1",
		pkg:                   "functions",
		importPath:            "cloud.google.com/go/functions/apiv1",
		gRPCServiceConfigPath: "functions_grpc_service_config.json",
		apiServiceConfigPath:  "cloudfunctions_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/networkconnectivity/v1",
		pkg:                   "networkconnectivity",
		importPath:            "cloud.google.com/go/networkconnectivity/apiv1",
		gRPCServiceConfigPath: "networkconnectivity_v1_grpc_service_config.json",
		apiServiceConfigPath:  "networkconnectivity_v1.yaml",
		// GA after 2021/10/11
		releaseLevel: "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/networkconnectivity/v1alpha1",
		pkg:                   "networkconnectivity",
		importPath:            "cloud.google.com/go/networkconnectivity/apiv1alpha1",
		gRPCServiceConfigPath: "networkconnectivity_grpc_service_config.json",
		apiServiceConfigPath:  "networkconnectivity_v1alpha1.yaml",
		releaseLevel:          "alpha",
	},
	{
		inputDirectoryPath:    "google/cloud/notebooks/v1beta1",
		pkg:                   "notebooks",
		importPath:            "cloud.google.com/go/notebooks/apiv1beta1",
		gRPCServiceConfigPath: "notebooks_grpc_service_config.json",
		apiServiceConfigPath:  "notebooks_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/billing/budgets/v1",
		pkg:                   "budgets",
		importPath:            "cloud.google.com/go/billing/budgets/apiv1",
		gRPCServiceConfigPath: "billingbudgets_grpc_service_config.json",
		apiServiceConfigPath:  "billingbudgets.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/billing/budgets/v1beta1",
		pkg:                   "budgets",
		importPath:            "cloud.google.com/go/billing/budgets/apiv1beta1",
		gRPCServiceConfigPath: "billingbudgets_grpc_service_config.json",
		apiServiceConfigPath:  "billingbudgets.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/video/transcoder/v1",
		pkg:                   "transcoder",
		importPath:            "cloud.google.com/go/video/transcoder/apiv1",
		gRPCServiceConfigPath: "transcoder_grpc_service_config.json",
		apiServiceConfigPath:  "transcoder_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/video/transcoder/v1beta1",
		pkg:                   "transcoder",
		importPath:            "cloud.google.com/go/video/transcoder/apiv1beta1",
		gRPCServiceConfigPath: "transcoder_grpc_service_config.json",
		apiServiceConfigPath:  "transcoder_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/workflows/v1beta",
		pkg:                   "workflows",
		importPath:            "cloud.google.com/go/workflows/apiv1beta",
		gRPCServiceConfigPath: "workflows_grpc_service_config.json",
		apiServiceConfigPath:  "workflows_v1beta.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/workflows/executions/v1",
		pkg:                   "executions",
		importPath:            "cloud.google.com/go/workflows/executions/apiv1",
		gRPCServiceConfigPath: "executions_grpc_service_config.json",
		apiServiceConfigPath:  "workflowexecutions_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/workflows/executions/v1beta",
		pkg:                   "executions",
		importPath:            "cloud.google.com/go/workflows/executions/apiv1beta",
		gRPCServiceConfigPath: "executions_grpc_service_config.json",
		apiServiceConfigPath:  "workflowexecutions_v1beta.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/security/privateca/v1beta1",
		pkg:                   "privateca",
		importPath:            "cloud.google.com/go/security/privateca/apiv1beta1",
		gRPCServiceConfigPath: "privateca_grpc_service_config.json",
		apiServiceConfigPath:  "privateca_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/security/privateca/v1",
		pkg:                   "privateca",
		importPath:            "cloud.google.com/go/security/privateca/apiv1",
		gRPCServiceConfigPath: "privateca_grpc_service_config.json",
		apiServiceConfigPath:  "privateca_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/assuredworkloads/v1beta1",
		pkg:                   "assuredworkloads",
		importPath:            "cloud.google.com/go/assuredworkloads/apiv1beta1",
		gRPCServiceConfigPath: "assuredworkloads_grpc_service_config.json",
		apiServiceConfigPath:  "assuredworkloads_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/accessapproval/v1",
		pkg:                   "accessapproval",
		importPath:            "cloud.google.com/go/accessapproval/apiv1",
		gRPCServiceConfigPath: "accessapproval_grpc_service_config.json",
		apiServiceConfigPath:  "accessapproval_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/websecurityscanner/v1",
		pkg:                   "websecurityscanner",
		importPath:            "cloud.google.com/go/websecurityscanner/apiv1",
		gRPCServiceConfigPath: "websecurityscanner_grpc_service_config.json",
		apiServiceConfigPath:  "websecurityscanner_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/managedidentities/v1",
		pkg:                   "managedidentities",
		importPath:            "cloud.google.com/go/managedidentities/apiv1",
		gRPCServiceConfigPath: "managedidentities_grpc_service_config.json",
		apiServiceConfigPath:  "managedidentities_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/devtools/artifactregistry/v1beta2",
		pkg:                   "artifactregistry",
		importPath:            "cloud.google.com/go/artifactregistry/apiv1beta2",
		gRPCServiceConfigPath: "artifactregistry_grpc_service_config.json",
		apiServiceConfigPath:  "artifactregistry_v1beta2.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/retail/v2",
		pkg:                   "retail",
		importPath:            "cloud.google.com/go/retail/apiv2",
		gRPCServiceConfigPath: "retail_grpc_service_config.json",
		apiServiceConfigPath:  "retail_v2.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:   "google/appengine/v1",
		pkg:                  "appengine",
		importPath:           "cloud.google.com/go/appengine/apiv1",
		apiServiceConfigPath: "appengine_v1.yaml",
		releaseLevel:         "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/binaryauthorization/v1beta1",
		pkg:                   "binaryauthorization",
		importPath:            "cloud.google.com/go/binaryauthorization/apiv1beta1",
		gRPCServiceConfigPath: "binaryauthorization_grpc_service_config.json",
		apiServiceConfigPath:  "binaryauthorization_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/channel/v1",
		pkg:                   "channel",
		importPath:            "cloud.google.com/go/channel/apiv1",
		gRPCServiceConfigPath: "cloudchannel_grpc_service_config.json",
		apiServiceConfigPath:  "cloudchannel_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:   "google/cloud/resourcemanager/v2",
		pkg:                  "resourcemanager",
		importPath:           "cloud.google.com/go/resourcemanager/apiv2",
		apiServiceConfigPath: "cloudresourcemanager_v2.yaml",
		releaseLevel:         "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/resourcemanager/v3",
		pkg:                   "resourcemanager",
		importPath:            "cloud.google.com/go/resourcemanager/apiv3",
		gRPCServiceConfigPath: "cloudresourcemanager_v3_grpc_service_config.json",
		apiServiceConfigPath:  "cloudresourcemanager_v3.yaml",
		// GA after 2021/10/11
		releaseLevel: "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/datalabeling/v1beta1",
		pkg:                   "datalabeling",
		importPath:            "cloud.google.com/go/datalabeling/apiv1beta1",
		gRPCServiceConfigPath: "datalabeling_grpc_service_config.json",
		apiServiceConfigPath:  "datalabeling_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/dataqna/v1alpha",
		pkg:                   "dataqna",
		importPath:            "cloud.google.com/go/dataqna/apiv1alpha",
		gRPCServiceConfigPath: "dataqna_grpc_service_config.json",
		apiServiceConfigPath:  "dataqna_v1alpha.yaml",
		releaseLevel:          "alpha",
	},
	{
		inputDirectoryPath:    "google/cloud/documentai/v1",
		pkg:                   "documentai",
		importPath:            "cloud.google.com/go/documentai/apiv1",
		gRPCServiceConfigPath: "documentai_v1_grpc_service_config.json",
		apiServiceConfigPath:  "documentai_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/documentai/v1beta3",
		pkg:                   "documentai",
		importPath:            "cloud.google.com/go/documentai/apiv1beta3",
		gRPCServiceConfigPath: "documentai_v1beta3_grpc_service_config.json",
		apiServiceConfigPath:  "documentai_v1beta3.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/api/servicemanagement/v1",
		pkg:                   "servicemanagement",
		importPath:            "cloud.google.com/go/servicemanagement/apiv1",
		gRPCServiceConfigPath: "servicemanagement_grpc_service_config.json",
		apiServiceConfigPath:  "servicemanagement_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/domains/v1beta1",
		pkg:                   "domains",
		importPath:            "cloud.google.com/go/domains/apiv1beta1",
		gRPCServiceConfigPath: "domains_grpc_service_config.json",
		apiServiceConfigPath:  "domains_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/mediatranslation/v1beta1",
		pkg:                   "mediatranslation",
		importPath:            "cloud.google.com/go/mediatranslation/apiv1beta1",
		gRPCServiceConfigPath: "mediatranslation_grpc_service_config.json",
		apiServiceConfigPath:  "mediatranslation_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:   "google/api/servicecontrol/v1",
		pkg:                  "servicecontrol",
		importPath:           "cloud.google.com/go/servicecontrol/apiv1",
		apiServiceConfigPath: "servicecontrol.yaml",
		releaseLevel:         "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/orgpolicy/v2",
		pkg:                   "orgpolicy",
		importPath:            "cloud.google.com/go/orgpolicy/apiv2",
		gRPCServiceConfigPath: "orgpolicy_grpc_service_config.json",
		apiServiceConfigPath:  "orgpolicy_v2.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/recommendationengine/v1beta1",
		pkg:                   "recommendationengine",
		importPath:            "cloud.google.com/go/recommendationengine/apiv1beta1",
		gRPCServiceConfigPath: "recommendationengine_grpc_service_config.json",
		apiServiceConfigPath:  "recommendationengine_v1beta1.yaml",
		releaseLevel:          "beta",
		stopGeneration:        true,
	},
	{
		inputDirectoryPath:    "google/cloud/gkehub/v1beta1",
		pkg:                   "gkehub",
		importPath:            "cloud.google.com/go/gkehub/apiv1beta1",
		gRPCServiceConfigPath: "membership_grpc_service_config.json",
		apiServiceConfigPath:  "gkehub_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/apigateway/v1",
		pkg:                   "apigateway",
		importPath:            "cloud.google.com/go/apigateway/apiv1",
		gRPCServiceConfigPath: "apigateway_grpc_service_config.json",
		apiServiceConfigPath:  "apigateway_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/metastore/v1",
		pkg:                   "metastore",
		importPath:            "cloud.google.com/go/metastore/apiv1",
		gRPCServiceConfigPath: "metastore_grpc_service_config.json",
		apiServiceConfigPath:  "metastore_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/metastore/v1alpha",
		pkg:                   "metastore",
		importPath:            "cloud.google.com/go/metastore/apiv1alpha",
		gRPCServiceConfigPath: "metastore_grpc_service_config.json",
		apiServiceConfigPath:  "metastore_v1alpha.yaml",
		releaseLevel:          "alpha",
	},
	{
		inputDirectoryPath:    "google/cloud/metastore/v1beta",
		pkg:                   "metastore",
		importPath:            "cloud.google.com/go/metastore/apiv1beta",
		gRPCServiceConfigPath: "metastore_grpc_service_config.json",
		apiServiceConfigPath:  "metastore_v1beta.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/resourcesettings/v1",
		pkg:                   "resourcesettings",
		importPath:            "cloud.google.com/go/resourcesettings/apiv1",
		gRPCServiceConfigPath: "resourcesettings_grpc_service_config.json",
		apiServiceConfigPath:  "resourcesettings_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/clouddms/v1",
		pkg:                   "clouddms",
		importPath:            "cloud.google.com/go/clouddms/apiv1",
		gRPCServiceConfigPath: "library_grpc_service_config.json",
		apiServiceConfigPath:  "datamigration_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/essentialcontacts/v1",
		pkg:                   "essentialcontacts",
		importPath:            "cloud.google.com/go/essentialcontacts/apiv1",
		gRPCServiceConfigPath: "essentialcontacts_v1_grpc_service_config.json",
		apiServiceConfigPath:  "essentialcontacts_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/api/serviceusage/v1",
		pkg:                   "serviceusage",
		importPath:            "cloud.google.com/go/serviceusage/apiv1",
		gRPCServiceConfigPath: "serviceusage_grpc_service_config.json",
		apiServiceConfigPath:  "serviceusage_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/shell/v1",
		pkg:                   "shell",
		importPath:            "cloud.google.com/go/shell/apiv1",
		gRPCServiceConfigPath: "cloudshell_grpc_service_config.json",
		apiServiceConfigPath:  "cloudshell_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/vpcaccess/v1",
		pkg:                   "vpcaccess",
		importPath:            "cloud.google.com/go/vpcaccess/apiv1",
		gRPCServiceConfigPath: "vpcaccess_grpc_service_config.json",
		apiServiceConfigPath:  "vpcaccess_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/privatecatalog/v1beta1",
		pkg:                   "privatecatalog",
		importPath:            "cloud.google.com/go/privatecatalog/apiv1beta1",
		gRPCServiceConfigPath: "cloudprivatecatalog_grpc_service_config.json",
		apiServiceConfigPath:  "cloudprivatecatalog_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/tpu/v1",
		pkg:                   "tpu",
		importPath:            "cloud.google.com/go/tpu/apiv1",
		gRPCServiceConfigPath: "tpu_grpc_service_config.json",
		apiServiceConfigPath:  "tpu_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/apigeeconnect/v1",
		pkg:                   "apigeeconnect",
		importPath:            "cloud.google.com/go/apigeeconnect/apiv1",
		gRPCServiceConfigPath: "connection_grpc_service_config.json",
		apiServiceConfigPath:  "apigeeconnect_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/lifesciences/v2beta",
		pkg:                   "lifesciences",
		importPath:            "cloud.google.com/go/lifesciences/apiv2beta",
		gRPCServiceConfigPath: "lifesciences_grpc_service_config.json",
		apiServiceConfigPath:  "lifesciences_v2beta.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/aiplatform/v1",
		pkg:                   "aiplatform",
		importPath:            "cloud.google.com/go/aiplatform/apiv1",
		gRPCServiceConfigPath: "aiplatform_grpc_service_config.json",
		apiServiceConfigPath:  "aiplatform_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/gkeconnect/gateway/v1beta1",
		pkg:                   "gateway",
		importPath:            "cloud.google.com/go/gkeconnect/gateway/apiv1beta1",
		gRPCServiceConfigPath: "connectgw_grpc_service_config.json",
		apiServiceConfigPath:  "connectgateway_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/datastream/v1alpha1",
		pkg:                   "datastream",
		importPath:            "cloud.google.com/go/datastream/apiv1alpha1",
		gRPCServiceConfigPath: "datastream_grpc_service_config.json",
		apiServiceConfigPath:  "datastream_v1alpha1.yaml",
		releaseLevel:          "alpha",
	},
	{
		inputDirectoryPath:    "google/dataflow/v1beta3",
		pkg:                   "dataflow",
		importPath:            "cloud.google.com/go/dataflow/apiv1beta3",
		gRPCServiceConfigPath: "dataflow_grpc_service_config.json",
		apiServiceConfigPath:  "dataflow_v1beta3.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/eventarc/v1",
		pkg:                   "eventarc",
		importPath:            "cloud.google.com/go/eventarc/apiv1",
		gRPCServiceConfigPath: "eventarc_grpc_service_config.json",
		apiServiceConfigPath:  "eventarc_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/networkmanagement/v1",
		pkg:                   "networkmanagement",
		importPath:            "cloud.google.com/go/networkmanagement/apiv1",
		gRPCServiceConfigPath: "networkmanagement_grpc_service_config.json",
		apiServiceConfigPath:  "networkmanagement_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/iap/v1",
		pkg:                   "iap",
		importPath:            "cloud.google.com/go/iap/apiv1",
		gRPCServiceConfigPath: "iap_grpc_service_config.json",
		apiServiceConfigPath:  "iap_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/datafusion/v1",
		pkg:                   "datafusion",
		importPath:            "cloud.google.com/go/datafusion/apiv1",
		gRPCServiceConfigPath: "datafusion_grpc_service_config.json",
		apiServiceConfigPath:  "datafusion_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/storagetransfer/v1",
		pkg:                   "storagetransfer",
		importPath:            "cloud.google.com/go/storagetransfer/apiv1",
		gRPCServiceConfigPath: "storagetransfer_grpc_service_config.json",
		apiServiceConfigPath:  "storagetransfer_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/cloud/networksecurity/v1beta1",
		pkg:                   "networksecurity",
		importPath:            "cloud.google.com/go/networksecurity/apiv1beta1",
		gRPCServiceConfigPath: "networksecurity_v1beta1_grpc_service_config.json",
		apiServiceConfigPath:  "networksecurity_v1beta1.yaml",
		releaseLevel:          "beta",
	},
	{
		inputDirectoryPath:    "google/monitoring/metricsscope/v1",
		pkg:                   "metricsscope",
		importPath:            "cloud.google.com/go/monitoring/metricsscope/apiv1",
		gRPCServiceConfigPath: "metricsscopes_grpc_service_config.json",
		apiServiceConfigPath:  "monitoring.yaml",
		// GA after 2021/10/11
		releaseLevel: "beta",
	},
	{
		inputDirectoryPath:    "google/identity/accesscontextmanager/v1",
		pkg:                   "accesscontextmanager",
		importPath:            "cloud.google.com/go/accesscontextmanager/apiv1",
		gRPCServiceConfigPath: "accesscontextmanager_grpc_service_config.json",
		apiServiceConfigPath:  "accesscontextmanager_v1.yaml",
		// GA after 2021/10/11
		releaseLevel: "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/contactcenterinsights/v1",
		pkg:                   "contactcenterinsights",
		importPath:            "cloud.google.com/go/contactcenterinsights/apiv1",
		gRPCServiceConfigPath: "contactcenterinsights_grpc_service_config.json",
		apiServiceConfigPath:  "contactcenterinsights_v1.yaml",
		// GA after 2021/10/11
		releaseLevel: "beta",
	},
	{
		inputDirectoryPath:    "google/cloud/orchestration/airflow/service/v1",
		pkg:                   "service",
		importPath:            "cloud.google.com/go/orchestration/airflow/service/apiv1",
		gRPCServiceConfigPath: "composer_grpc_service_config.json",
		apiServiceConfigPath:  "composer_v1.yaml",
		// GA after 2021/10/27
		releaseLevel: "beta",
	},

	// Non-Cloud APIs
	{
		inputDirectoryPath:    "google/analytics/admin/v1alpha",
		pkg:                   "admin",
		importPath:            "cloud.google.com/go/analytics/admin/apiv1alpha",
		gRPCServiceConfigPath: "admin_grpc_service_config.json",
		apiServiceConfigPath:  "analyticsadmin_v1alpha.yaml",
		releaseLevel:          "alpha",
	},
	{
		inputDirectoryPath:    "google/area120/tables/v1alpha1",
		pkg:                   "tables",
		importPath:            "cloud.google.com/go/area120/tables/apiv1alpha1",
		gRPCServiceConfigPath: "tables_grpc_service_config.json",
		apiServiceConfigPath:  "area120tables_v1alpha1.yaml",
		releaseLevel:          "alpha",
	},
	{
		inputDirectoryPath:    "google/cloud/gsuiteaddons/v1",
		pkg:                   "gsuiteaddons",
		importPath:            "cloud.google.com/go/gsuiteaddons/apiv1",
		gRPCServiceConfigPath: "gsuiteaddons_grpc_service_config.json",
		apiServiceConfigPath:  "gsuiteaddons_v1.yaml",
		releaseLevel:          "ga",
	},
	{
		inputDirectoryPath:    "google/storage/v2",
		pkg:                   "storage",
		importPath:            "cloud.google.com/go/storage/internal/apiv2",
		gRPCServiceConfigPath: "storage_grpc_service_config.json",
		apiServiceConfigPath:  "storage_v2.yaml",
		releaseLevel:          "alpha",
	},
}
