// Copyright 2019 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package generator

// MicrogenConfig represents a single microgen target.
type MicrogenConfig struct {
	// InputDirectoryPath is the path to the input (.proto, etc) files, relative
	// to googleapisDir.
	InputDirectoryPath string

	// ImportPath is the path that this library should be imported as.
	ImportPath string

	// Pkg is the name that should be used in the package declaration.
	Pkg string

	// GRPCServiceConfigPath is the path to the grpc service config for this
	// target, relative to googleapisDir/inputDirectoryPath.
	GRPCServiceConfigPath string

	// ApiServiceConfigPath is the path to the gapic service config for this
	// target, relative to googleapisDir/inputDirectoryPath.
	ApiServiceConfigPath string

	// ReleaseLevel is the release level of this target. Values incl ga,
	// beta, alpha.
	ReleaseLevel string

	// StopGeneration is used to stop generating a given client. This might be
	// useful if a client needs to be deprecated, but retained in the repo
	// metadata.
	StopGeneration bool

	// DisableMetadata is used to toggle generation of the gapic_metadata.json
	// file for the client library.
	DisableMetadata bool

	// Transports is a list of Transports to generate a client for. Acceptable
	// values are 'grpc' and 'rest'
	Transports []string

	// StubsDir indicates that the protobuf/gRPC stubs should be generated
	// in the GAPIC module by replacing the go_package option with the value of
	// ImportPath plus the specified suffix separated by a "/", and using the
	// same Pkg value.
	StubsDir string
}

var MicrogenGapicConfigs = []*MicrogenConfig{
	// Cloud APIs
	{
		InputDirectoryPath:   "google/cloud/compute/v1",
		Pkg:                  "compute",
		ImportPath:           "cloud.google.com/go/compute/apiv1",
		ApiServiceConfigPath: "compute_v1.yaml",
		Transports:           []string{"rest"},
		ReleaseLevel:         "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/texttospeech/v1",
		Pkg:                   "texttospeech",
		ImportPath:            "cloud.google.com/go/texttospeech/apiv1",
		GRPCServiceConfigPath: "texttospeech_grpc_service_config.json",
		ApiServiceConfigPath:  "texttospeech_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/asset/v1",
		Pkg:                   "asset",
		ImportPath:            "cloud.google.com/go/asset/apiv1",
		GRPCServiceConfigPath: "cloudasset_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudasset_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/billing/v1",
		Pkg:                   "billing",
		ImportPath:            "cloud.google.com/go/billing/apiv1",
		GRPCServiceConfigPath: "cloud_billing_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudbilling.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/language/v1",
		Pkg:                   "language",
		ImportPath:            "cloud.google.com/go/language/apiv1",
		GRPCServiceConfigPath: "language_grpc_service_config.json",
		ApiServiceConfigPath:  "language_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/language/v1beta2",
		Pkg:                   "language",
		ImportPath:            "cloud.google.com/go/language/apiv1beta2",
		GRPCServiceConfigPath: "language_grpc_service_config.json",
		ApiServiceConfigPath:  "language_v1beta2.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/memcache/v1",
		Pkg:                   "memcache",
		ImportPath:            "cloud.google.com/go/memcache/apiv1",
		GRPCServiceConfigPath: "memcache_grpc_service_config.json",
		ApiServiceConfigPath:  "memcache_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/memcache/v1beta2",
		Pkg:                   "memcache",
		ImportPath:            "cloud.google.com/go/memcache/apiv1beta2",
		GRPCServiceConfigPath: "memcache_grpc_service_config.json",
		ApiServiceConfigPath:  "memcache_v1beta2.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/phishingprotection/v1beta1",
		Pkg:                   "phishingprotection",
		ImportPath:            "cloud.google.com/go/phishingprotection/apiv1beta1",
		GRPCServiceConfigPath: "phishingprotection_grpc_service_config.json",
		ApiServiceConfigPath:  "phishingprotection_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/translate/v3",
		Pkg:                   "translate",
		ImportPath:            "cloud.google.com/go/translate/apiv3",
		GRPCServiceConfigPath: "translate_grpc_service_config.json",
		ApiServiceConfigPath:  "translate_v3.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/scheduler/v1",
		Pkg:                   "scheduler",
		ImportPath:            "cloud.google.com/go/scheduler/apiv1",
		GRPCServiceConfigPath: "cloudscheduler_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudscheduler_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/scheduler/v1beta1",
		Pkg:                   "scheduler",
		ImportPath:            "cloud.google.com/go/scheduler/apiv1beta1",
		GRPCServiceConfigPath: "cloudscheduler_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudscheduler_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/speech/v1",
		Pkg:                   "speech",
		ImportPath:            "cloud.google.com/go/speech/apiv1",
		GRPCServiceConfigPath: "speech_grpc_service_config.json",
		ApiServiceConfigPath:  "speech_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/speech/v1p1beta1",
		Pkg:                   "speech",
		ImportPath:            "cloud.google.com/go/speech/apiv1p1beta1",
		GRPCServiceConfigPath: "speech_grpc_service_config.json",
		ApiServiceConfigPath:  "speech_v1p1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/connection/v1beta1",
		Pkg:                   "connection",
		ImportPath:            "cloud.google.com/go/bigquery/connection/apiv1beta1",
		GRPCServiceConfigPath: "bigqueryconnection_grpc_service_config.json",
		ApiServiceConfigPath:  "bigqueryconnection_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/connection/v1",
		Pkg:                   "connection",
		ImportPath:            "cloud.google.com/go/bigquery/connection/apiv1",
		GRPCServiceConfigPath: "bigqueryconnection_grpc_service_config.json",
		ApiServiceConfigPath:  "bigqueryconnection_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/dataexchange/v1beta1",
		Pkg:                   "dataexchange",
		ImportPath:            "cloud.google.com/go/bigquery/dataexchange/apiv1beta1",
		GRPCServiceConfigPath: "analyticshub_grpc_service_config.json",
		ApiServiceConfigPath:  "analyticshub_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/datatransfer/v1",
		Pkg:                   "datatransfer",
		ImportPath:            "cloud.google.com/go/bigquery/datatransfer/apiv1",
		GRPCServiceConfigPath: "bigquerydatatransfer_grpc_service_config.json",
		ApiServiceConfigPath:  "bigquerydatatransfer_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/migration/v2alpha",
		Pkg:                   "migration",
		ImportPath:            "cloud.google.com/go/bigquery/migration/apiv2alpha",
		GRPCServiceConfigPath: "bigquerymigration_grpc_service_config.json",
		ApiServiceConfigPath:  "bigquerymigration_v2alpha.yaml",
		ReleaseLevel:          "alpha",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/migration/v2",
		Pkg:                   "migration",
		ImportPath:            "cloud.google.com/go/bigquery/migration/apiv2",
		GRPCServiceConfigPath: "bigquerymigration_grpc_service_config.json",
		ApiServiceConfigPath:  "bigquerymigration_v2.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/reservation/v1beta1",
		Pkg:                   "reservation",
		ImportPath:            "cloud.google.com/go/bigquery/reservation/apiv1beta1",
		GRPCServiceConfigPath: "bigqueryreservation_grpc_service_config.json",
		ApiServiceConfigPath:  "bigqueryreservation_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/reservation/v1",
		Pkg:                   "reservation",
		ImportPath:            "cloud.google.com/go/bigquery/reservation/apiv1",
		GRPCServiceConfigPath: "bigqueryreservation_grpc_service_config.json",
		ApiServiceConfigPath:  "bigqueryreservation_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/storage/v1beta1",
		Pkg:                   "storage",
		ImportPath:            "cloud.google.com/go/bigquery/storage/apiv1beta1",
		GRPCServiceConfigPath: "bigquerystorage_grpc_service_config.json",
		ApiServiceConfigPath:  "bigquerystorage_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/storage/v1beta2",
		Pkg:                   "storage",
		ImportPath:            "cloud.google.com/go/bigquery/storage/apiv1beta2",
		GRPCServiceConfigPath: "bigquerystorage_grpc_service_config.json",
		ApiServiceConfigPath:  "bigquerystorage_v1beta2.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/bigquery/storage/v1",
		Pkg:                   "storage",
		ImportPath:            "cloud.google.com/go/bigquery/storage/apiv1",
		GRPCServiceConfigPath: "bigquerystorage_grpc_service_config.json",
		ApiServiceConfigPath:  "bigquerystorage_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/iot/v1",
		Pkg:                   "iot",
		ImportPath:            "cloud.google.com/go/iot/apiv1",
		GRPCServiceConfigPath: "cloudiot_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudiot_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/recommender/v1beta1",
		Pkg:                   "recommender",
		ImportPath:            "cloud.google.com/go/recommender/apiv1beta1",
		GRPCServiceConfigPath: "recommender_grpc_service_config.json",
		ApiServiceConfigPath:  "recommender_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/tasks/v2",
		Pkg:                   "cloudtasks",
		ImportPath:            "cloud.google.com/go/cloudtasks/apiv2",
		GRPCServiceConfigPath: "cloudtasks_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudtasks_v2.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/tasks/v2beta2",
		Pkg:                   "cloudtasks",
		ImportPath:            "cloud.google.com/go/cloudtasks/apiv2beta2",
		GRPCServiceConfigPath: "cloudtasks_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudtasks_v2beta2.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/tasks/v2beta3",
		Pkg:                   "cloudtasks",
		ImportPath:            "cloud.google.com/go/cloudtasks/apiv2beta3",
		GRPCServiceConfigPath: "cloudtasks_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudtasks_v2beta3.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/videointelligence/v1",
		Pkg:                   "videointelligence",
		ImportPath:            "cloud.google.com/go/videointelligence/apiv1",
		GRPCServiceConfigPath: "videointelligence_grpc_service_config.json",
		ApiServiceConfigPath:  "videointelligence_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/vision/v1",
		Pkg:                   "vision",
		ImportPath:            "cloud.google.com/go/vision/v2/apiv1",
		GRPCServiceConfigPath: "vision_grpc_service_config.json",
		ApiServiceConfigPath:  "vision_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/webrisk/v1",
		Pkg:                   "webrisk",
		ImportPath:            "cloud.google.com/go/webrisk/apiv1",
		GRPCServiceConfigPath: "webrisk_grpc_service_config.json",
		ApiServiceConfigPath:  "webrisk_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/webrisk/v1beta1",
		Pkg:                   "webrisk",
		ImportPath:            "cloud.google.com/go/webrisk/apiv1beta1",
		GRPCServiceConfigPath: "webrisk_grpc_service_config.json",
		ApiServiceConfigPath:  "webrisk_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/secretmanager/v1",
		Pkg:                   "secretmanager",
		ImportPath:            "cloud.google.com/go/secretmanager/apiv1",
		GRPCServiceConfigPath: "secretmanager_grpc_service_config.json",
		ApiServiceConfigPath:  "secretmanager_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/secrets/v1beta1",
		Pkg:                   "secretmanager",
		ImportPath:            "cloud.google.com/go/secretmanager/apiv1beta1",
		GRPCServiceConfigPath: "secretmanager_grpc_service_config.json",
		ApiServiceConfigPath:  "secretmanager_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/osconfig/v1",
		Pkg:                   "osconfig",
		ImportPath:            "cloud.google.com/go/osconfig/apiv1",
		GRPCServiceConfigPath: "osconfig_grpc_service_config.json",
		ApiServiceConfigPath:  "osconfig_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/osconfig/v1alpha",
		Pkg:                   "osconfig",
		ImportPath:            "cloud.google.com/go/osconfig/apiv1alpha",
		GRPCServiceConfigPath: "osconfig_grpc_service_config.json",
		ApiServiceConfigPath:  "osconfig_v1alpha.yaml",
		ReleaseLevel:          "alpha",
	},
	{
		InputDirectoryPath:    "google/cloud/osconfig/v1beta",
		Pkg:                   "osconfig",
		ImportPath:            "cloud.google.com/go/osconfig/apiv1beta",
		GRPCServiceConfigPath: "osconfig_grpc_service_config.json",
		ApiServiceConfigPath:  "osconfig_v1beta.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/osconfig/agentendpoint/v1",
		Pkg:                   "agentendpoint",
		ImportPath:            "cloud.google.com/go/osconfig/agentendpoint/apiv1",
		GRPCServiceConfigPath: "agentendpoint_grpc_service_config.json",
		ApiServiceConfigPath:  "osconfig_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/osconfig/agentendpoint/v1beta",
		Pkg:                   "agentendpoint",
		ImportPath:            "cloud.google.com/go/osconfig/agentendpoint/apiv1beta",
		GRPCServiceConfigPath: "agentendpoint_grpc_service_config.json",
		ApiServiceConfigPath:  "osconfig_v1beta.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/datacatalog/v1",
		Pkg:                   "datacatalog",
		ImportPath:            "cloud.google.com/go/datacatalog/apiv1",
		GRPCServiceConfigPath: "datacatalog_grpc_service_config.json",
		ApiServiceConfigPath:  "datacatalog_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/datacatalog/v1beta1",
		Pkg:                   "datacatalog",
		ImportPath:            "cloud.google.com/go/datacatalog/apiv1beta1",
		GRPCServiceConfigPath: "datacatalog_grpc_service_config.json",
		ApiServiceConfigPath:  "datacatalog_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/dataproc/v1",
		Pkg:                   "dataproc",
		ImportPath:            "cloud.google.com/go/dataproc/apiv1",
		GRPCServiceConfigPath: "dataproc_grpc_service_config.json",
		ApiServiceConfigPath:  "dataproc_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/kms/v1",
		Pkg:                   "kms",
		ImportPath:            "cloud.google.com/go/kms/apiv1",
		GRPCServiceConfigPath: "cloudkms_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudkms_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/oslogin/v1",
		Pkg:                   "oslogin",
		ImportPath:            "cloud.google.com/go/oslogin/apiv1",
		GRPCServiceConfigPath: "oslogin_grpc_service_config.json",
		ApiServiceConfigPath:  "oslogin_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/oslogin/v1beta",
		Pkg:                   "oslogin",
		ImportPath:            "cloud.google.com/go/oslogin/apiv1beta",
		GRPCServiceConfigPath: "oslogin_grpc_service_config.json",
		ApiServiceConfigPath:  "oslogin_v1beta.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/recaptchaenterprise/v1",
		Pkg:                   "recaptchaenterprise",
		ImportPath:            "cloud.google.com/go/recaptchaenterprise/v2/apiv1",
		GRPCServiceConfigPath: "recaptchaenterprise_grpc_service_config.json",
		ApiServiceConfigPath:  "recaptchaenterprise_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/recaptchaenterprise/v1beta1",
		Pkg:                   "recaptchaenterprise",
		ImportPath:            "cloud.google.com/go/recaptchaenterprise/apiv1beta1",
		GRPCServiceConfigPath: "recaptchaenterprise_grpc_service_config.json",
		ApiServiceConfigPath:  "recaptchaenterprise_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/redis/v1",
		Pkg:                   "redis",
		ImportPath:            "cloud.google.com/go/redis/apiv1",
		GRPCServiceConfigPath: "redis_grpc_service_config.json",
		ApiServiceConfigPath:  "redis_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/redis/v1beta1",
		Pkg:                   "redis",
		ImportPath:            "cloud.google.com/go/redis/apiv1beta1",
		GRPCServiceConfigPath: "redis_grpc_service_config.json",
		ApiServiceConfigPath:  "redis_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/devtools/clouddebugger/v2",
		Pkg:                   "debugger",
		ImportPath:            "cloud.google.com/go/debugger/apiv2",
		GRPCServiceConfigPath: "clouddebugger_grpc_service_config.json",
		ApiServiceConfigPath:  "clouddebugger_v2.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/devtools/clouderrorreporting/v1beta1",
		Pkg:                   "errorreporting",
		ImportPath:            "cloud.google.com/go/errorreporting/apiv1beta1",
		GRPCServiceConfigPath: "errorreporting_grpc_service_config.json",
		ApiServiceConfigPath:  "clouderrorreporting_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/devtools/cloudtrace/v1",
		Pkg:                   "trace",
		ImportPath:            "cloud.google.com/go/trace/apiv1",
		GRPCServiceConfigPath: "cloudtrace_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudtrace_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/devtools/cloudtrace/v2",
		Pkg:                   "trace",
		ImportPath:            "cloud.google.com/go/trace/apiv2",
		GRPCServiceConfigPath: "cloudtrace_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudtrace_v2.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/privacy/dlp/v2",
		Pkg:                   "dlp",
		ImportPath:            "cloud.google.com/go/dlp/apiv2",
		GRPCServiceConfigPath: "dlp_grpc_service_config.json",
		ApiServiceConfigPath:  "dlp_v2.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/datastore/admin/v1",
		Pkg:                   "admin",
		ImportPath:            "cloud.google.com/go/datastore/admin/apiv1",
		GRPCServiceConfigPath: "datastore_admin_grpc_service_config.json",
		ApiServiceConfigPath:  "datastore_v1.yaml",
		ReleaseLevel:          "alpha",
	},
	{
		InputDirectoryPath:    "google/spanner/admin/database/v1",
		Pkg:                   "database",
		ImportPath:            "cloud.google.com/go/spanner/admin/database/apiv1",
		GRPCServiceConfigPath: "spanner_admin_database_grpc_service_config.json",
		ApiServiceConfigPath:  "spanner_admin_database.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/spanner/admin/instance/v1",
		Pkg:                   "instance",
		ImportPath:            "cloud.google.com/go/spanner/admin/instance/apiv1",
		GRPCServiceConfigPath: "spanner_admin_instance_grpc_service_config.json",
		ApiServiceConfigPath:  "spanner_admin_instance.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/spanner/v1",
		Pkg:                   "spanner",
		ImportPath:            "cloud.google.com/go/spanner/apiv1",
		GRPCServiceConfigPath: "spanner_grpc_service_config.json",
		ApiServiceConfigPath:  "spanner.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/securitycenter/settings/v1beta1",
		Pkg:                   "settings",
		ImportPath:            "cloud.google.com/go/securitycenter/settings/apiv1beta1",
		GRPCServiceConfigPath: "securitycenter_settings_grpc_service_config.json",
		ApiServiceConfigPath:  "securitycenter_settings.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/securitycenter/v1",
		Pkg:                   "securitycenter",
		ImportPath:            "cloud.google.com/go/securitycenter/apiv1",
		GRPCServiceConfigPath: "securitycenter_grpc_service_config.json",
		ApiServiceConfigPath:  "securitycenter_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/securitycenter/v1beta1",
		Pkg:                   "securitycenter",
		ImportPath:            "cloud.google.com/go/securitycenter/apiv1beta1",
		GRPCServiceConfigPath: "securitycenter_grpc_service_config.json",
		ApiServiceConfigPath:  "securitycenter_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/securitycenter/v1p1beta1",
		Pkg:                   "securitycenter",
		ImportPath:            "cloud.google.com/go/securitycenter/apiv1p1beta1",
		GRPCServiceConfigPath: "securitycenter_grpc_service_config.json",
		ApiServiceConfigPath:  "securitycenter_v1p1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/firestore/admin/v1",
		Pkg:                   "apiv1",
		ImportPath:            "cloud.google.com/go/firestore/apiv1/admin",
		GRPCServiceConfigPath: "firestore_admin_grpc_service_config.json",
		ApiServiceConfigPath:  "firestore_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/firestore/v1",
		Pkg:                   "firestore",
		ImportPath:            "cloud.google.com/go/firestore/apiv1",
		GRPCServiceConfigPath: "firestore_grpc_service_config.json",
		ApiServiceConfigPath:  "firestore_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/devtools/cloudbuild/v1",
		Pkg:                   "cloudbuild",
		ImportPath:            "cloud.google.com/go/cloudbuild/apiv1/v2",
		GRPCServiceConfigPath: "cloudbuild_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudbuild_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/dialogflow/cx/v3beta1",
		Pkg:                   "cx",
		ImportPath:            "cloud.google.com/go/dialogflow/cx/apiv3beta1",
		GRPCServiceConfigPath: "dialogflow_grpc_service_config.json",
		ApiServiceConfigPath:  "dialogflow_v3beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/dialogflow/cx/v3",
		Pkg:                   "cx",
		ImportPath:            "cloud.google.com/go/dialogflow/cx/apiv3",
		GRPCServiceConfigPath: "dialogflow_grpc_service_config.json",
		ApiServiceConfigPath:  "dialogflow_v3.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/dialogflow/v2",
		Pkg:                   "dialogflow",
		ImportPath:            "cloud.google.com/go/dialogflow/apiv2",
		GRPCServiceConfigPath: "dialogflow_grpc_service_config.json",
		ApiServiceConfigPath:  "dialogflow_v2.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/iam/credentials/v1",
		Pkg:                   "credentials",
		ImportPath:            "cloud.google.com/go/iam/credentials/apiv1",
		GRPCServiceConfigPath: "iamcredentials_grpc_service_config.json",
		ApiServiceConfigPath:  "iamcredentials_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/longrunning",
		Pkg:                   "longrunning",
		ImportPath:            "cloud.google.com/go/longrunning/autogen",
		GRPCServiceConfigPath: "longrunning_grpc_service_config.json",
		ApiServiceConfigPath:  "longrunning.yaml",
		ReleaseLevel:          "alpha",
		Transports:            []string{"grpc", "rest"},
	},
	{
		InputDirectoryPath:    "google/devtools/containeranalysis/v1beta1",
		Pkg:                   "containeranalysis",
		ImportPath:            "cloud.google.com/go/containeranalysis/apiv1beta1",
		GRPCServiceConfigPath: "containeranalysis_grpc_service_config.json",
		ApiServiceConfigPath:  "containeranalysis_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		// The grafeas v1beta1 client must be generated in the same package as containeranalysis v1beta1,
		// but the proto is in a sub-directory of the containeranalysis v1beta1 protos.
		InputDirectoryPath:    "google/devtools/containeranalysis/v1beta1/grafeas",
		Pkg:                   "containeranalysis",
		ImportPath:            "cloud.google.com/go/containeranalysis/apiv1beta1",
		GRPCServiceConfigPath: "../containeranalysis_grpc_service_config.json",
		ApiServiceConfigPath:  "../containeranalysis_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/recommender/v1",
		Pkg:                   "recommender",
		ImportPath:            "cloud.google.com/go/recommender/apiv1",
		GRPCServiceConfigPath: "recommender_grpc_service_config.json",
		ApiServiceConfigPath:  "recommender_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/videointelligence/v1beta2",
		Pkg:                   "videointelligence",
		ImportPath:            "cloud.google.com/go/videointelligence/apiv1beta2",
		GRPCServiceConfigPath: "videointelligence_grpc_service_config.json",
		ApiServiceConfigPath:  "../videointelligence_v1beta2.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/asset/v1p2beta1",
		Pkg:                   "asset",
		ImportPath:            "cloud.google.com/go/asset/apiv1p2beta1",
		GRPCServiceConfigPath: "cloudasset_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudasset_v1p2beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/asset/v1p5beta1",
		Pkg:                   "asset",
		ImportPath:            "cloud.google.com/go/asset/apiv1p5beta1",
		GRPCServiceConfigPath: "cloudasset_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudasset_v1p5beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/monitoring/v3",
		Pkg:                   "monitoring",
		ImportPath:            "cloud.google.com/go/monitoring/apiv3/v2",
		GRPCServiceConfigPath: "monitoring_grpc_service_config.json",
		ApiServiceConfigPath:  "monitoring.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/vision/v1p1beta1",
		Pkg:                   "vision",
		ImportPath:            "cloud.google.com/go/vision/apiv1p1beta1",
		GRPCServiceConfigPath: "vision_grpc_service_config.json",
		ApiServiceConfigPath:  "vision_v1p1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/logging/v2",
		Pkg:                   "logging",
		ImportPath:            "cloud.google.com/go/logging/apiv2",
		GRPCServiceConfigPath: "logging_grpc_service_config.json",
		ApiServiceConfigPath:  "logging_v2.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/talent/v4",
		Pkg:                   "talent",
		ImportPath:            "cloud.google.com/go/talent/apiv4",
		GRPCServiceConfigPath: "talent_grpc_service_config.json",
		ApiServiceConfigPath:  "jobs_v4.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/talent/v4beta1",
		Pkg:                   "talent",
		ImportPath:            "cloud.google.com/go/talent/apiv4beta1",
		GRPCServiceConfigPath: "talent_grpc_service_config.json",
		ApiServiceConfigPath:  "jobs_v4beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/pubsub/v1",
		Pkg:                   "pubsub",
		ImportPath:            "cloud.google.com/go/pubsub/apiv1",
		GRPCServiceConfigPath: "pubsub_grpc_service_config.json",
		ApiServiceConfigPath:  "pubsub_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/pubsublite/v1",
		Pkg:                   "pubsublite",
		ImportPath:            "cloud.google.com/go/pubsublite/apiv1",
		GRPCServiceConfigPath: "pubsublite_grpc_service_config.json",
		ApiServiceConfigPath:  "pubsublite_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/automl/v1",
		Pkg:                   "automl",
		ImportPath:            "cloud.google.com/go/automl/apiv1",
		GRPCServiceConfigPath: "automl_grpc_service_config.json",
		ApiServiceConfigPath:  "automl_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/automl/v1beta1",
		Pkg:                   "automl",
		ImportPath:            "cloud.google.com/go/automl/apiv1beta1",
		GRPCServiceConfigPath: "automl_grpc_service_config.json",
		ApiServiceConfigPath:  "automl_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/container/v1",
		Pkg:                   "container",
		ImportPath:            "cloud.google.com/go/container/apiv1",
		GRPCServiceConfigPath: "container_grpc_service_config.json",
		ApiServiceConfigPath:  "container_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/servicedirectory/v1",
		Pkg:                   "servicedirectory",
		ImportPath:            "cloud.google.com/go/servicedirectory/apiv1",
		GRPCServiceConfigPath: "servicedirectory_grpc_service_config.json",
		ApiServiceConfigPath:  "servicedirectory_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/servicedirectory/v1beta1",
		Pkg:                   "servicedirectory",
		ImportPath:            "cloud.google.com/go/servicedirectory/apiv1beta1",
		GRPCServiceConfigPath: "servicedirectory_grpc_service_config.json",
		ApiServiceConfigPath:  "servicedirectory_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/gaming/v1",
		Pkg:                   "gaming",
		ImportPath:            "cloud.google.com/go/gaming/apiv1",
		GRPCServiceConfigPath: "gaming_grpc_service_config.json",
		ApiServiceConfigPath:  "gameservices_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/gaming/v1beta",
		Pkg:                   "gaming",
		ImportPath:            "cloud.google.com/go/gaming/apiv1beta",
		GRPCServiceConfigPath: "gaming_grpc_service_config.json",
		ApiServiceConfigPath:  "gameservices_v1beta.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/policytroubleshooter/v1",
		Pkg:                   "policytroubleshooter",
		ImportPath:            "cloud.google.com/go/policytroubleshooter/apiv1",
		GRPCServiceConfigPath: "checker_grpc_service_config.json",
		ApiServiceConfigPath:  "policytroubleshooter_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/monitoring/dashboard/v1",
		Pkg:                   "dashboard",
		ImportPath:            "cloud.google.com/go/monitoring/dashboard/apiv1",
		GRPCServiceConfigPath: "dashboards_grpc_service_config.json",
		ApiServiceConfigPath:  "monitoring.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/functions/v1",
		Pkg:                   "functions",
		ImportPath:            "cloud.google.com/go/functions/apiv1",
		GRPCServiceConfigPath: "functions_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudfunctions_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/networkconnectivity/v1",
		Pkg:                   "networkconnectivity",
		ImportPath:            "cloud.google.com/go/networkconnectivity/apiv1",
		GRPCServiceConfigPath: "networkconnectivity_v1_grpc_service_config.json",
		ApiServiceConfigPath:  "networkconnectivity_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/networkconnectivity/v1alpha1",
		Pkg:                   "networkconnectivity",
		ImportPath:            "cloud.google.com/go/networkconnectivity/apiv1alpha1",
		GRPCServiceConfigPath: "networkconnectivity_grpc_service_config.json",
		ApiServiceConfigPath:  "networkconnectivity_v1alpha1.yaml",
		ReleaseLevel:          "alpha",
	},
	{
		InputDirectoryPath:    "google/cloud/notebooks/v1beta1",
		Pkg:                   "notebooks",
		ImportPath:            "cloud.google.com/go/notebooks/apiv1beta1",
		GRPCServiceConfigPath: "notebooks_grpc_service_config.json",
		ApiServiceConfigPath:  "notebooks_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/billing/budgets/v1",
		Pkg:                   "budgets",
		ImportPath:            "cloud.google.com/go/billing/budgets/apiv1",
		GRPCServiceConfigPath: "billingbudgets_grpc_service_config.json",
		ApiServiceConfigPath:  "billingbudgets.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/billing/budgets/v1beta1",
		Pkg:                   "budgets",
		ImportPath:            "cloud.google.com/go/billing/budgets/apiv1beta1",
		GRPCServiceConfigPath: "billingbudgets_grpc_service_config.json",
		ApiServiceConfigPath:  "billingbudgets.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/video/transcoder/v1",
		Pkg:                   "transcoder",
		ImportPath:            "cloud.google.com/go/video/transcoder/apiv1",
		GRPCServiceConfigPath: "transcoder_grpc_service_config.json",
		ApiServiceConfigPath:  "transcoder_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/workflows/v1beta",
		Pkg:                   "workflows",
		ImportPath:            "cloud.google.com/go/workflows/apiv1beta",
		GRPCServiceConfigPath: "workflows_grpc_service_config.json",
		ApiServiceConfigPath:  "workflows_v1beta.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/workflows/executions/v1",
		Pkg:                   "executions",
		ImportPath:            "cloud.google.com/go/workflows/executions/apiv1",
		GRPCServiceConfigPath: "executions_grpc_service_config.json",
		ApiServiceConfigPath:  "workflowexecutions_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/workflows/executions/v1beta",
		Pkg:                   "executions",
		ImportPath:            "cloud.google.com/go/workflows/executions/apiv1beta",
		GRPCServiceConfigPath: "executions_grpc_service_config.json",
		ApiServiceConfigPath:  "workflowexecutions_v1beta.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/security/privateca/v1beta1",
		Pkg:                   "privateca",
		ImportPath:            "cloud.google.com/go/security/privateca/apiv1beta1",
		GRPCServiceConfigPath: "privateca_grpc_service_config.json",
		ApiServiceConfigPath:  "privateca_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/security/privateca/v1",
		Pkg:                   "privateca",
		ImportPath:            "cloud.google.com/go/security/privateca/apiv1",
		GRPCServiceConfigPath: "privateca_grpc_service_config.json",
		ApiServiceConfigPath:  "privateca_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/assuredworkloads/v1beta1",
		Pkg:                   "assuredworkloads",
		ImportPath:            "cloud.google.com/go/assuredworkloads/apiv1beta1",
		GRPCServiceConfigPath: "assuredworkloads_grpc_service_config.json",
		ApiServiceConfigPath:  "assuredworkloads.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/accessapproval/v1",
		Pkg:                   "accessapproval",
		ImportPath:            "cloud.google.com/go/accessapproval/apiv1",
		GRPCServiceConfigPath: "accessapproval_grpc_service_config.json",
		ApiServiceConfigPath:  "accessapproval_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/websecurityscanner/v1",
		Pkg:                   "websecurityscanner",
		ImportPath:            "cloud.google.com/go/websecurityscanner/apiv1",
		GRPCServiceConfigPath: "websecurityscanner_grpc_service_config.json",
		ApiServiceConfigPath:  "websecurityscanner_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/managedidentities/v1",
		Pkg:                   "managedidentities",
		ImportPath:            "cloud.google.com/go/managedidentities/apiv1",
		GRPCServiceConfigPath: "managedidentities_grpc_service_config.json",
		ApiServiceConfigPath:  "managedidentities_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/devtools/artifactregistry/v1beta2",
		Pkg:                   "artifactregistry",
		ImportPath:            "cloud.google.com/go/artifactregistry/apiv1beta2",
		GRPCServiceConfigPath: "artifactregistry_grpc_service_config.json",
		ApiServiceConfigPath:  "artifactregistry_v1beta2.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/retail/v2",
		Pkg:                   "retail",
		ImportPath:            "cloud.google.com/go/retail/apiv2",
		GRPCServiceConfigPath: "retail_grpc_service_config.json",
		ApiServiceConfigPath:  "retail_v2.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:   "google/appengine/v1",
		Pkg:                  "appengine",
		ImportPath:           "cloud.google.com/go/appengine/apiv1",
		ApiServiceConfigPath: "appengine_v1.yaml",
		ReleaseLevel:         "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/binaryauthorization/v1beta1",
		Pkg:                   "binaryauthorization",
		ImportPath:            "cloud.google.com/go/binaryauthorization/apiv1beta1",
		GRPCServiceConfigPath: "binaryauthorization_grpc_service_config.json",
		ApiServiceConfigPath:  "binaryauthorization_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/channel/v1",
		Pkg:                   "channel",
		ImportPath:            "cloud.google.com/go/channel/apiv1",
		GRPCServiceConfigPath: "cloudchannel_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudchannel_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:   "google/cloud/resourcemanager/v2",
		Pkg:                  "resourcemanager",
		ImportPath:           "cloud.google.com/go/resourcemanager/apiv2",
		ApiServiceConfigPath: "cloudresourcemanager_v2.yaml",
		ReleaseLevel:         "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/resourcemanager/v3",
		Pkg:                   "resourcemanager",
		ImportPath:            "cloud.google.com/go/resourcemanager/apiv3",
		GRPCServiceConfigPath: "cloudresourcemanager_v3_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudresourcemanager_v3.yaml",
		// GA after 2021/10/11
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/datalabeling/v1beta1",
		Pkg:                   "datalabeling",
		ImportPath:            "cloud.google.com/go/datalabeling/apiv1beta1",
		GRPCServiceConfigPath: "datalabeling_grpc_service_config.json",
		ApiServiceConfigPath:  "datalabeling_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/dataqna/v1alpha",
		Pkg:                   "dataqna",
		ImportPath:            "cloud.google.com/go/dataqna/apiv1alpha",
		GRPCServiceConfigPath: "dataqna_grpc_service_config.json",
		ApiServiceConfigPath:  "dataqna_v1alpha.yaml",
		ReleaseLevel:          "alpha",
	},
	{
		InputDirectoryPath:    "google/cloud/documentai/v1",
		Pkg:                   "documentai",
		ImportPath:            "cloud.google.com/go/documentai/apiv1",
		GRPCServiceConfigPath: "documentai_v1_grpc_service_config.json",
		ApiServiceConfigPath:  "documentai_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/documentai/v1beta3",
		Pkg:                   "documentai",
		ImportPath:            "cloud.google.com/go/documentai/apiv1beta3",
		GRPCServiceConfigPath: "documentai_v1beta3_grpc_service_config.json",
		ApiServiceConfigPath:  "documentai_v1beta3.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/api/servicemanagement/v1",
		Pkg:                   "servicemanagement",
		ImportPath:            "cloud.google.com/go/servicemanagement/apiv1",
		GRPCServiceConfigPath: "servicemanagement_grpc_service_config.json",
		ApiServiceConfigPath:  "servicemanagement_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/domains/v1beta1",
		Pkg:                   "domains",
		ImportPath:            "cloud.google.com/go/domains/apiv1beta1",
		GRPCServiceConfigPath: "domains_grpc_service_config.json",
		ApiServiceConfigPath:  "domains_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/mediatranslation/v1beta1",
		Pkg:                   "mediatranslation",
		ImportPath:            "cloud.google.com/go/mediatranslation/apiv1beta1",
		GRPCServiceConfigPath: "mediatranslation_grpc_service_config.json",
		ApiServiceConfigPath:  "mediatranslation_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:   "google/api/servicecontrol/v1",
		Pkg:                  "servicecontrol",
		ImportPath:           "cloud.google.com/go/servicecontrol/apiv1",
		ApiServiceConfigPath: "servicecontrol.yaml",
		ReleaseLevel:         "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/orgpolicy/v2",
		Pkg:                   "orgpolicy",
		ImportPath:            "cloud.google.com/go/orgpolicy/apiv2",
		GRPCServiceConfigPath: "orgpolicy_grpc_service_config.json",
		ApiServiceConfigPath:  "orgpolicy_v2.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/recommendationengine/v1beta1",
		Pkg:                   "recommendationengine",
		ImportPath:            "cloud.google.com/go/recommendationengine/apiv1beta1",
		GRPCServiceConfigPath: "recommendationengine_grpc_service_config.json",
		ApiServiceConfigPath:  "recommendationengine_v1beta1.yaml",
		ReleaseLevel:          "beta",
		StopGeneration:        true,
	},
	{
		InputDirectoryPath:    "google/cloud/gkehub/v1beta1",
		Pkg:                   "gkehub",
		ImportPath:            "cloud.google.com/go/gkehub/apiv1beta1",
		GRPCServiceConfigPath: "membership_grpc_service_config.json",
		ApiServiceConfigPath:  "gkehub_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/apigateway/v1",
		Pkg:                   "apigateway",
		ImportPath:            "cloud.google.com/go/apigateway/apiv1",
		GRPCServiceConfigPath: "apigateway_grpc_service_config.json",
		ApiServiceConfigPath:  "apigateway_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/metastore/v1",
		Pkg:                   "metastore",
		ImportPath:            "cloud.google.com/go/metastore/apiv1",
		GRPCServiceConfigPath: "metastore_grpc_service_config.json",
		ApiServiceConfigPath:  "metastore_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/metastore/v1alpha",
		Pkg:                   "metastore",
		ImportPath:            "cloud.google.com/go/metastore/apiv1alpha",
		GRPCServiceConfigPath: "metastore_grpc_service_config.json",
		ApiServiceConfigPath:  "metastore_v1alpha.yaml",
		ReleaseLevel:          "alpha",
	},
	{
		InputDirectoryPath:    "google/cloud/metastore/v1beta",
		Pkg:                   "metastore",
		ImportPath:            "cloud.google.com/go/metastore/apiv1beta",
		GRPCServiceConfigPath: "metastore_grpc_service_config.json",
		ApiServiceConfigPath:  "metastore_v1beta.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/resourcesettings/v1",
		Pkg:                   "resourcesettings",
		ImportPath:            "cloud.google.com/go/resourcesettings/apiv1",
		GRPCServiceConfigPath: "resourcesettings_grpc_service_config.json",
		ApiServiceConfigPath:  "resourcesettings_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/clouddms/v1",
		Pkg:                   "clouddms",
		ImportPath:            "cloud.google.com/go/clouddms/apiv1",
		GRPCServiceConfigPath: "library_grpc_service_config.json",
		ApiServiceConfigPath:  "datamigration_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/essentialcontacts/v1",
		Pkg:                   "essentialcontacts",
		ImportPath:            "cloud.google.com/go/essentialcontacts/apiv1",
		GRPCServiceConfigPath: "essentialcontacts_v1_grpc_service_config.json",
		ApiServiceConfigPath:  "essentialcontacts_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/api/serviceusage/v1",
		Pkg:                   "serviceusage",
		ImportPath:            "cloud.google.com/go/serviceusage/apiv1",
		GRPCServiceConfigPath: "serviceusage_grpc_service_config.json",
		ApiServiceConfigPath:  "serviceusage_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/shell/v1",
		Pkg:                   "shell",
		ImportPath:            "cloud.google.com/go/shell/apiv1",
		GRPCServiceConfigPath: "cloudshell_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudshell_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/vpcaccess/v1",
		Pkg:                   "vpcaccess",
		ImportPath:            "cloud.google.com/go/vpcaccess/apiv1",
		GRPCServiceConfigPath: "vpcaccess_grpc_service_config.json",
		ApiServiceConfigPath:  "vpcaccess_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/privatecatalog/v1beta1",
		Pkg:                   "privatecatalog",
		ImportPath:            "cloud.google.com/go/privatecatalog/apiv1beta1",
		GRPCServiceConfigPath: "cloudprivatecatalog_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudprivatecatalog_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/tpu/v1",
		Pkg:                   "tpu",
		ImportPath:            "cloud.google.com/go/tpu/apiv1",
		GRPCServiceConfigPath: "tpu_grpc_service_config.json",
		ApiServiceConfigPath:  "tpu_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/apigeeconnect/v1",
		Pkg:                   "apigeeconnect",
		ImportPath:            "cloud.google.com/go/apigeeconnect/apiv1",
		GRPCServiceConfigPath: "connection_grpc_service_config.json",
		ApiServiceConfigPath:  "apigeeconnect_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/lifesciences/v2beta",
		Pkg:                   "lifesciences",
		ImportPath:            "cloud.google.com/go/lifesciences/apiv2beta",
		GRPCServiceConfigPath: "lifesciences_grpc_service_config.json",
		ApiServiceConfigPath:  "lifesciences_v2beta.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/aiplatform/v1",
		Pkg:                   "aiplatform",
		ImportPath:            "cloud.google.com/go/aiplatform/apiv1",
		GRPCServiceConfigPath: "aiplatform_grpc_service_config.json",
		ApiServiceConfigPath:  "aiplatform_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/aiplatform/v1beta1",
		Pkg:                   "aiplatform",
		ImportPath:            "cloud.google.com/go/aiplatform/apiv1beta1",
		GRPCServiceConfigPath: "aiplatform_grpc_service_config.json",
		ApiServiceConfigPath:  "aiplatform_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/gkeconnect/gateway/v1beta1",
		Pkg:                   "gateway",
		ImportPath:            "cloud.google.com/go/gkeconnect/gateway/apiv1beta1",
		GRPCServiceConfigPath: "connectgw_grpc_service_config.json",
		ApiServiceConfigPath:  "connectgateway_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/datastream/v1alpha1",
		Pkg:                   "datastream",
		ImportPath:            "cloud.google.com/go/datastream/apiv1alpha1",
		GRPCServiceConfigPath: "datastream_grpc_service_config.json",
		ApiServiceConfigPath:  "datastream_v1alpha1.yaml",
		ReleaseLevel:          "alpha",
	},
	{
		InputDirectoryPath:    "google/dataflow/v1beta3",
		Pkg:                   "dataflow",
		ImportPath:            "cloud.google.com/go/dataflow/apiv1beta3",
		GRPCServiceConfigPath: "dataflow_grpc_service_config.json",
		ApiServiceConfigPath:  "dataflow_v1beta3.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/eventarc/v1",
		Pkg:                   "eventarc",
		ImportPath:            "cloud.google.com/go/eventarc/apiv1",
		GRPCServiceConfigPath: "eventarc_grpc_service_config.json",
		ApiServiceConfigPath:  "eventarc_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/networkmanagement/v1",
		Pkg:                   "networkmanagement",
		ImportPath:            "cloud.google.com/go/networkmanagement/apiv1",
		GRPCServiceConfigPath: "networkmanagement_grpc_service_config.json",
		ApiServiceConfigPath:  "networkmanagement_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/iap/v1",
		Pkg:                   "iap",
		ImportPath:            "cloud.google.com/go/iap/apiv1",
		GRPCServiceConfigPath: "iap_grpc_service_config.json",
		ApiServiceConfigPath:  "iap_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/datafusion/v1",
		Pkg:                   "datafusion",
		ImportPath:            "cloud.google.com/go/datafusion/apiv1",
		GRPCServiceConfigPath: "datafusion_grpc_service_config.json",
		ApiServiceConfigPath:  "datafusion_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/storagetransfer/v1",
		Pkg:                   "storagetransfer",
		ImportPath:            "cloud.google.com/go/storagetransfer/apiv1",
		GRPCServiceConfigPath: "storagetransfer_grpc_service_config.json",
		ApiServiceConfigPath:  "storagetransfer_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/networksecurity/v1beta1",
		Pkg:                   "networksecurity",
		ImportPath:            "cloud.google.com/go/networksecurity/apiv1beta1",
		GRPCServiceConfigPath: "networksecurity_v1beta1_grpc_service_config.json",
		ApiServiceConfigPath:  "networksecurity_v1beta1.yaml",
		ReleaseLevel:          "beta",
	},
	{
		InputDirectoryPath:    "google/monitoring/metricsscope/v1",
		Pkg:                   "metricsscope",
		ImportPath:            "cloud.google.com/go/monitoring/metricsscope/apiv1",
		GRPCServiceConfigPath: "metricsscopes_grpc_service_config.json",
		ApiServiceConfigPath:  "monitoring.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/identity/accesscontextmanager/v1",
		Pkg:                   "accesscontextmanager",
		ImportPath:            "cloud.google.com/go/accesscontextmanager/apiv1",
		GRPCServiceConfigPath: "accesscontextmanager_grpc_service_config.json",
		ApiServiceConfigPath:  "accesscontextmanager_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/contactcenterinsights/v1",
		Pkg:                   "contactcenterinsights",
		ImportPath:            "cloud.google.com/go/contactcenterinsights/apiv1",
		GRPCServiceConfigPath: "contactcenterinsights_grpc_service_config.json",
		ApiServiceConfigPath:  "contactcenterinsights_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/orchestration/airflow/service/v1",
		Pkg:                   "service",
		ImportPath:            "cloud.google.com/go/orchestration/airflow/service/apiv1",
		GRPCServiceConfigPath: "composer_grpc_service_config.json",
		ApiServiceConfigPath:  "composer_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/filestore/v1",
		Pkg:                   "filestore",
		ImportPath:            "cloud.google.com/go/filestore/apiv1",
		GRPCServiceConfigPath: "file_grpc_service_config.json",
		ApiServiceConfigPath:  "file_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/deploy/v1",
		Pkg:                   "deploy",
		ImportPath:            "cloud.google.com/go/deploy/apiv1",
		GRPCServiceConfigPath: "clouddeploy_grpc_service_config.json",
		ApiServiceConfigPath:  "clouddeploy_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath:    "google/cloud/ids/v1",
		Pkg:                   "ids",
		ImportPath:            "cloud.google.com/go/ids/apiv1",
		GRPCServiceConfigPath: "ids_grpc_service_config.json",
		ApiServiceConfigPath:  "ids_v1.yaml",
		// GA after 2021/12/15
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/vmmigration/v1",
		Pkg:                   "vmmigration",
		ImportPath:            "cloud.google.com/go/vmmigration/apiv1",
		GRPCServiceConfigPath: "vmmigration_v1_grpc_service_config.json",
		ApiServiceConfigPath:  "vmmigration_v1.yaml",
		// GA after 2021/12/15
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/eventarc/publishing/v1",
		Pkg:                   "publishing",
		ImportPath:            "cloud.google.com/go/eventarc/publishing/apiv1",
		GRPCServiceConfigPath: "eventarcpublishing_grpc_service_config.json",
		ApiServiceConfigPath:  "eventarcpublishing_v1.yaml",
		// GA after 2022/02/13
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/video/livestream/v1",
		Pkg:                   "livestream",
		ImportPath:            "cloud.google.com/go/video/livestream/apiv1",
		GRPCServiceConfigPath: "livestream_grpc_service_config.json",
		ApiServiceConfigPath:  "livestream_v1.yaml",
		// GA after 2022/02/25
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/dataplex/v1",
		Pkg:                   "dataplex",
		ImportPath:            "cloud.google.com/go/dataplex/apiv1",
		GRPCServiceConfigPath: "dataplex_grpc_service_config.json",
		ApiServiceConfigPath:  "dataplex_v1.yaml",
		// GA after 2022/02/28
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/video/stitcher/v1",
		Pkg:                   "stitcher",
		ImportPath:            "cloud.google.com/go/video/stitcher/apiv1",
		GRPCServiceConfigPath: "videostitcher_grpc_service_config.json",
		ApiServiceConfigPath:  "videostitcher_v1.yaml",
		// GA after 2022/04/02
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/certificatemanager/v1",
		Pkg:                   "certificatemanager",
		ImportPath:            "cloud.google.com/go/certificatemanager/apiv1",
		GRPCServiceConfigPath: "certificatemanager_grpc_service_config.json",
		ApiServiceConfigPath:  "certificatemanager_v1.yaml",
		// GA after 2022/04/04
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/optimization/v1",
		Pkg:                   "optimization",
		ImportPath:            "cloud.google.com/go/optimization/apiv1",
		GRPCServiceConfigPath: "cloudoptimization_grpc_service_config.json",
		ApiServiceConfigPath:  "cloudoptimization_v1.yaml",
		// GA after 2022/04/16
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/run/v2",
		Pkg:                   "run",
		ImportPath:            "cloud.google.com/go/run/apiv2",
		GRPCServiceConfigPath: "run_grpc_service_config.json",
		ApiServiceConfigPath:  "run_v2.yaml",
		// GA after 2022/04/30
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/assuredworkloads/v1",
		Pkg:                   "assuredworkloads",
		ImportPath:            "cloud.google.com/go/assuredworkloads/apiv1",
		GRPCServiceConfigPath: "assuredworkloads_grpc_service_config.json",
		ApiServiceConfigPath:  "assuredworkloads_v1.yaml",
		// GA after 2022/06/04
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/binaryauthorization/v1",
		Pkg:                   "binaryauthorization",
		ImportPath:            "cloud.google.com/go/binaryauthorization/apiv1",
		GRPCServiceConfigPath: "binaryauthorization_grpc_service_config.json",
		ApiServiceConfigPath:  "binaryauthorization_v1.yaml",
		// GA after 2022/06/05
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/datastream/v1",
		Pkg:                   "datastream",
		ImportPath:            "cloud.google.com/go/datastream/apiv1",
		GRPCServiceConfigPath: "datastream_grpc_service_config.json",
		ApiServiceConfigPath:  "datastream_v1.yaml",
		// GA after 2022/06/05
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/notebooks/v1",
		Pkg:                   "notebooks",
		ImportPath:            "cloud.google.com/go/notebooks/apiv1",
		GRPCServiceConfigPath: "notebooks_grpc_service_config.json",
		ApiServiceConfigPath:  "notebooks_v1.yaml",
		// GA after 2022/06/05
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/workflows/v1",
		Pkg:                   "workflows",
		ImportPath:            "cloud.google.com/go/workflows/apiv1",
		GRPCServiceConfigPath: "workflows_grpc_service_config.json",
		ApiServiceConfigPath:  "workflows_v1.yaml",
		// GA after 2022/06/05
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/gkebackup/v1",
		Pkg:                   "gkebackup",
		ImportPath:            "cloud.google.com/go/gkebackup/apiv1",
		GRPCServiceConfigPath: "gkebackup_grpc_service_config.json",
		ApiServiceConfigPath:  "gkebackup_v1.yaml",
		// GA after 2022/06/16
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/gkemulticloud/v1",
		Pkg:                   "gkemulticloud",
		ImportPath:            "cloud.google.com/go/gkemulticloud/apiv1",
		GRPCServiceConfigPath: "gkemulticloud_grpc_service_config.json",
		ApiServiceConfigPath:  "gkemulticloud_v1.yaml",
		// GA after 2022/06/16
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/retail/v2alpha",
		Pkg:                   "retail",
		ImportPath:            "cloud.google.com/go/retail/apiv2alpha",
		GRPCServiceConfigPath: "retail_grpc_service_config.json",
		ApiServiceConfigPath:  "retail_v2alpha.yaml",
		// GA after 2022/06/23
		ReleaseLevel: "alpha",
	},
	{
		InputDirectoryPath:    "google/cloud/retail/v2beta",
		Pkg:                   "retail",
		ImportPath:            "cloud.google.com/go/retail/apiv2beta",
		GRPCServiceConfigPath: "retail_grpc_service_config.json",
		ApiServiceConfigPath:  "retail_v2beta.yaml",
		// GA after 2022/06/23
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/batch/v1",
		Pkg:                   "batch",
		ImportPath:            "cloud.google.com/go/batch/apiv1",
		GRPCServiceConfigPath: "batch_grpc_service_config.json",
		ApiServiceConfigPath:  "batch_v1.yaml",
		// GA after 2022/07/10
		ReleaseLevel: "beta",
	},
	{
		InputDirectoryPath:    "google/cloud/baremetalsolution/v2",
		Pkg:                   "baremetalsolution",
		ImportPath:            "cloud.google.com/go/baremetalsolution/apiv2",
		GRPCServiceConfigPath: "baremetalsolution_grpc_service_config.json",
		ApiServiceConfigPath:  "baremetalsolution_v2.yaml",
		// GA after 2022/07/10
		ReleaseLevel: "beta",
	},

	// Non-Cloud APIs
	{
		InputDirectoryPath:    "google/analytics/admin/v1alpha",
		Pkg:                   "admin",
		ImportPath:            "cloud.google.com/go/analytics/admin/apiv1alpha",
		GRPCServiceConfigPath: "admin_grpc_service_config.json",
		ApiServiceConfigPath:  "analyticsadmin_v1alpha.yaml",
		ReleaseLevel:          "alpha",
	},
	{
		InputDirectoryPath:    "google/area120/tables/v1alpha1",
		Pkg:                   "tables",
		ImportPath:            "cloud.google.com/go/area120/tables/apiv1alpha1",
		GRPCServiceConfigPath: "tables_grpc_service_config.json",
		ApiServiceConfigPath:  "area120tables_v1alpha1.yaml",
		ReleaseLevel:          "alpha",
	},
	{
		InputDirectoryPath:    "google/cloud/gsuiteaddons/v1",
		Pkg:                   "gsuiteaddons",
		ImportPath:            "cloud.google.com/go/gsuiteaddons/apiv1",
		GRPCServiceConfigPath: "gsuiteaddons_grpc_service_config.json",
		ApiServiceConfigPath:  "gsuiteaddons_v1.yaml",
		ReleaseLevel:          "ga",
	},
	{
		InputDirectoryPath: "google/storage/v2",
		Pkg:                "storage",
		ImportPath:         "cloud.google.com/go/storage/internal/apiv2",
		// The retry configuration is explicitly excluded in order to disable
		// GAPIC-level retries and allow the veneer layer to handle retries.
		ApiServiceConfigPath: "storage_v2.yaml",
		ReleaseLevel:         "alpha",
	},
}
